/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstdio.h" /* <stdio.h> */
#include "_pstring.h" /* <string.h> */
#include <stdlib.h>

#include "defs.h"
#include "cfg.h"
#include "zchars.h"
#include "zcontext.h"
#include "zcharset.h"
#include "zerror.h"
#include "zfile.h"
#include "zstdio.h"

#include "error.h"
#include "values.h"
#include "words.h"

#include "configur.h"
#include "index.h"
#include "indexjob.h"

#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) &&\
    defined( UKRAINIAN_SUPPORT ) && defined( UKRAINIAN_INTERFACE )
#define EFORMATS zErrorUkrainianFormats
#elif defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) &&\
    defined( RUSSIAN_INTERFACE )
#define EFORMATS zErrorRussianFormats
#else
#define EFORMATS NULL
#endif

static void printUsage( struct zcontext_t *cnt );
static void exitProgram( struct zcontext_t *cnt, int exitCode);
static void printError( struct zcontext_t *cnt, unsigned zint_t errorCode, const char *name);
static void memoryFail( struct zcontext_t *cnt, const char *prog);
static void readIndexObjects( struct zstrcoll_t *sc );

static struct zstrcoll_t _list_;

void main( int argc, char **argv)
{
  Boolean doUsage = False, success = True, printConfig = False, doReadStdin = False;
  char *confFile = NULL, *indexFile = NULL;
  struct zstrcoll_t *toIndexList = &_list_;
  unsigned int flags = 0;
  struct zcontext_t ixingContext;
  struct fludata_t fluData;
  struct zstrbuf_t strBuf;
  const char *verboseString = NULL;
  char *ptr;
  int argx;

/* 樠㥬 ࠧ த ⥪ */
  zContextInit( &ixingContext, printError, EFORMATS, memoryFail, 0);
  setVerboseValue( &ixingContext, NULL);
  zSetFlags( ixingContext.ioFlags, ZCONTEXT_IOFLAG_AUTOFLUSH);
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) && defined( REMOTE_CHARSET )
  zSetRemoteCharset( &ixingContext, ARGS_CHARSET, REMOTE_CHARSET);
#endif
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
  zInitRussians();
#endif
  /* XXX: 㦭 - errorFormat */
  initFludata( &ixingContext, &fluData);

  (void) zStringCollectionInit( &ixingContext, &_list_, 0, 5, zcfDontFreeContent);
  (void) zStringBufferInit( &ixingContext, &strBuf, NULL, 0, 256);

/* ࠥ 㬥 */
  for( argx = 1; argx < argc; argx++)
  {
    if( argv[argx][0] == '-' )
      switch( argv[argx][1] )
      {
        case 'c':
          GET_ARGUMENT( confFile );
          break;

        case 'd':
          doReadStdin = True;
          break;

        case 'f':
	  GET_ARGUMENT( indexFile );
          break;

        case 'o':
          zSetFlags( flags, pifOverwrite);
          break;

        case 'p':
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
	  GET_ARGUMENT( ptr );
          if( !doUsage )
          {
            int charset = zCharsetType( ptr );
	    zSetRemoteCharset( &ixingContext, ARGS_CHARSET, charset);
          }
#endif
          break;

        case 'r':
          zSetFlags( flags, pifReindexAll);
          break;

        case 'u':
          zSetFlags( flags, pifUpdate);
          break;

        case 'v':
          ZGET_ARG(ptr);
          if( ptr != NULL ) verboseString = ptr;
          break;

        case 'w':
          printConfig = True;
          break;

	case 'z':
          toIndexList = NULL;
          break;

        case '?':
        default:
          doUsage = True;
          break;
      }
    else if( toIndexList != NULL )
      zStringCollectionAdd( toIndexList, argv[argx], zcfDontAllocMemory);
  }

  if( doUsage || argc <= 1 ) printUsage( &ixingContext );

  success = configure( &ixingContext, confFile, &strBuf);
  if( verboseString != NULL ) setVerboseValue( &ixingContext, verboseString);

  if( success )
    if( printConfig )
      printConfigure( &ixingContext );
    else
    {
      if( success && toIndexList != NULL )
      {
        if( doReadStdin ) readIndexObjects( toIndexList );
        if( toIndexList->count == 0 && (doReadStdin || indexJobs == NULL || indexJobs->next == NULL) )
	{
          printError( &ixingContext, errProgParams, "Please, specify directories or files to index.");
          success = False;
        }
      }

      if( success && E_TempDir != NULL )
        success = zCheckTempDir( &ixingContext, E_TempDir);

      if( success )
      {
        if( indexFile == NULL )
          indexFile = (E_IndexFile == NULL) ? DEFAULT_INDEX_FILE_NAME : E_IndexFile;
        success = performIndex( &ixingContext, indexFile, toIndexList, flags);
      }
    }

  exitProgram( &ixingContext, success ? 0 : 1);
}

static void readIndexObjects( struct zstrcoll_t *sc )
{
  char line[1024];

  zStringCollectionFree( sc );

  while( fgets( line, sizeof( line ), stdin) != NULL )
  {
    int l = strlen( line );
    if( l > 0 && line[l-1] == '\n' ) line[l-1] = '\0';
    if( *line == '\0' ) continue;
    zStringCollectionAdd( sc, line, 0);
  }
}

static void printUsage( struct zcontext_t *cnt )
{
  zprintf( cnt, "usage: flindex [-c file] [-f file] "
#if defined( HAVE_LSTAT )
		"[-l yes|no] "
#endif
                "[-o] "
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
                "[-p charset] "
#endif
                "[-r] [-v num] [-w] [-z | dir file ...]\n\n"
		"options: (defaults are in brackets)\n"
		"       -c : configuration file to use for indexing\n"
                "       -d : read the index objects from STDIN\n"
		"       -f : index file to create [" DEFAULT_INDEX_FILE_NAME "]\n"
                "       -o : owerwrite the existing index file\n"
                "       -p : print out all messages in the desired charset [%s]\n"
                "       -r : reindex content\n"
                "       -u : update content\n"
		"       -v : verbosity level (0 to 3) [%d]\n"
                "       -w : write the configuration on the screen and exit\n"
                "       -z : create an empty index\n\n"
		"version: " VERSION "\n"
		"   docs: http://www.sbnet.ru/soft/fluids/\n",
#ifdef REMOTE_CHARSET
    zCharsetName( REMOTE_CHARSET, ZCHARSET_NAME_SHORT),
#else
    zCharsetName( LOCAL_CHARSET, ZCHARSET_NAME_SHORT),
#endif
    DEFAULT_VERBOSE_LEVEL);

  exitProgram( cnt, 1);
}

static void exitProgram( struct zcontext_t *cnt, int exitCode)
{
  zStringCollectionFree( &_list_ );
  freeConfMemory( cnt );
  zContextFree( cnt );
  exit( exitCode );
}

static void printError( struct zcontext_t *cnt, unsigned zint_t errorCode, const char *name)
{
  char line[1024+1024+100];
  if( _zErrorCode( errorCode ) == zecNone ) return;
  (void) fluGetErrorString( cnt, line, sizeof( line ), errorCode, name);
  zprintf( cnt, "%s: %s\n", zCheckFlags( errorCode, zefWarning) ? "Warning" : "Error", line);
}

static void memoryFail( struct zcontext_t *cnt, const char *prog)
{
  cnt->printError( cnt, zerNoMemory, prog);
  exitProgram( cnt, -1);
}
