/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */

#include "zcontext.h"
#include "zalloc.h"
#include "zchars.h"
#include "zcharset.h"
#include "zcoll.h"
#include "zconfig.h"
#include "zstring.h"
#include "zstdio.h"
#include "zstdlib.h"

#include "cfg.h"
#include "configur.h"
#include "indexjob.h"

/***************************************************************************/
/*                                                                         */
/*  Index jobs                                                             */
/*                                                                         */
/***************************************************************************/

struct indexjob_t *indexJobs = NULL;
static struct indexjob_t *lastJob = NULL;

void initIndexJob( struct zcontext_t *cnt, struct indexjob_t *ij, struct zstrbuf_t *sb)
{
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
  int i;
#endif

  ZEROFILL( ij, sizeof( struct indexjob_t ));

  ij->context = cnt;
  ij->defaults = indexJobs;
  ij->sb = sb;

  (void) zStringCollectionInit( cnt, &ij->objects, 0, 10, zcfSorted | zcfCheckDuplicate);
  (void) zParamCollectionInit( cnt, &ij->ctypes, 0, 20, zcfCheckDuplicate, 1, NULL, NULL);
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
  (void) zParamCollectionInit( cnt, &ij->charsets, 0, 5, zcfCheckDuplicate, 1, NULL, NULL);
  (void) zParamCollectionInit( cnt, &ij->accessCharsets, 0, 5, zcfFirstInsert, 1, NULL, NULL);
  for( i = 0; i < CMETHOD_LAST; i++) ij->charsetMethods[i] = CMETHOD_NONE;
  ij->torTable = NULL;
#endif

  zReplaceInit( cnt, &ij->replaces, sb);

  (void) zStringCollectionInit( cnt, &ij->ignoreFileNames, 0, 5, 0);
  (void) zStringCollectionInit( cnt, &ij->ignoreDirNames, 0, 5, 0);
  (void) zStringCollectionInit( cnt, &ij->ignoreFullPathFiles, 0, 5, 0);
  (void) zStringCollectionInit( cnt, &ij->ignoreDirWithFile, 0, 5, 0);
  (void) zStringCollectionInit( cnt, &ij->ignoreFileURLs, 0, 5, 0);
}

void addIndexJob( struct indexjob_t *ij )
{
  if( lastJob == NULL )
    indexJobs = lastJob = ij;
  else
  {
    lastJob->next = ij;
    lastJob = ij;
  }
}

void freeIndexJob( struct indexjob_t *ij )
{
  zStringCollectionFree( &ij->objects );
  zParamCollectionFree( &ij->ctypes );
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
  zParamCollectionFree( &ij->charsets );
  zParamCollectionFree( &ij->accessCharsets );
  _ZFREE( ij->context, ij->accessFile);
#endif
  zReplaceFree( &ij->replaces );
  zStringCollectionFree( &ij->ignoreFileNames );
  zStringCollectionFree( &ij->ignoreDirNames );
  zStringCollectionFree( &ij->ignoreFullPathFiles );
  zStringCollectionFree( &ij->ignoreDirWithFile );
  zStringCollectionFree( &ij->ignoreFileURLs );
  zFree( ij->context, ij);
}

void freeIndexJobs( void );
void freeIndexJobs()
{
  if( indexJobs != NULL && indexJobs->sb != NULL )
    zStringBufferFree( indexJobs->sb );

  while( indexJobs != NULL )
  {
    struct indexjob_t *tmp = indexJobs;
    indexJobs = indexJobs->next;
    freeIndexJob( tmp );
  }
}

/***************************************************************************/
/*                                                                         */
/*  Default config                                                         */
/*                                                                         */
/***************************************************************************/

static const char * const defaultHtmlContent[] =
{
  "*.htm", "*.html", "*.shtml",
#if !defined( HAVE_CASEINDEP_FILE_NAMES )
  "*.HTM", "*.HTML", "*.SHTML",
#endif
  NULL
};

static const char * const defaultTextContent[] =
{
  "*.txt",
#if !defined( HAVE_CASEINDEP_FILE_NAMES )
  "*.TXT",
#endif
  NULL
};

static const char * const defaultWithoutContent[] =
{
  "*.jpg", "*.jpeg", "*.gif",
#if !defined( HAVE_CASEINDEP_FILE_NAMES )
  "*.JPG", "*.JPEG", "*.GIF",
#endif
  NULL
};


#define CTYPE_PLAIN_HTML      "text/html"
#define CTYPE_PLAIN_HTML1     "plain/html"
#define CTYPE_PLAIN_TEXT      "text/plain"
#define CTYPE_PLAIN_TEXT1     "plain/text"
#define CTYPE_OTHER           "*" "/" "*"

int getContentType( const char *name )
{
  if( strcasecmp( name, CTYPE_PLAIN_HTML) == 0 ||
      strcasecmp( name, CTYPE_PLAIN_HTML1) == 0 ) return CTYPE_HTML;
  if( strcasecmp( name, CTYPE_PLAIN_TEXT) == 0 ||
      strcasecmp( name, CTYPE_PLAIN_TEXT1) == 0 ) return CTYPE_TEXT;
  return CTYPE_UNKNOWN;
}

Boolean createDefaultIndexJob( struct zcontext_t *cnt, struct zstrbuf_t *sb)
{
  if( indexJobs == NULL )
  {
    struct indexjob_t *ij = ZNEW( cnt, struct indexjob_t);
    const char * const *p;
    char ctype;

    initIndexJob( cnt, ij, sb);

    for( ctype = CTYPE_HTML, p = defaultHtmlContent; *p != NULL; p++)
      if( !zParamCollectionAdd( &ij->ctypes, *p, &ctype, 1) ) return False;
    for( ctype = CTYPE_TEXT, p = defaultTextContent; *p != NULL; p++)
      if( !zParamCollectionAdd( &ij->ctypes, *p, &ctype, 1) ) return False;
    for( ctype = CTYPE_NONTEXT, p = defaultWithoutContent; *p != NULL; p++)
      if( !zParamCollectionAdd( &ij->ctypes, *p, &ctype, 1) ) return False;

    addIndexJob( ij );
  }

  return True;
}

/***************************************************************************/
/*                                                                         */
/*  Config                                                                 */
/*                                                                         */
/***************************************************************************/

static struct indexjob_t *currentJob = NULL;

#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
static struct zstatus_t cmethodTable[] =
{
  { "ByCharsetDirective", CMETHOD_BY_DIRECTIVE   },
  { "ByAccessFile",       CMETHOD_BY_ACCESS_FILE },
  { "ByMetaTag",          CMETHOD_BY_META_TAG    },
  { NULL }
};
#endif

Boolean initDefaultIndexJobConfig( struct zcontext_t *cnt, struct zconfsec_t *cur,
    unsigned int flags, const char *fileName, void *data)
{
  if( indexJobs != NULL )
  {
    /* XXX: warning: duplicate [defaults]... */
    currentJob = indexJobs;
    return True;
  }

  currentJob = ZNEW( cnt, struct indexjob_t);
  initIndexJob( cnt, currentJob, (struct zstrbuf_t *) data);
  return zInitDefaultConfig( cnt, cur, flags, fileName, data);
}

Boolean initIndexJobConfig( struct zcontext_t *cnt, struct zconfsec_t *cur,
    unsigned int flags, const char *fileName, void *data)
{
  if( indexJobs == NULL )
  {
    struct indexjob_t *ij = ZNEW( cnt, struct indexjob_t);
    initIndexJob( cnt, ij, (struct zstrbuf_t *) data);
    addIndexJob( ij );
  }

  currentJob = ZNEW( cnt, struct indexjob_t);
  initIndexJob( cnt, currentJob, (struct zstrbuf_t *) data);
  return zInitDefaultConfig( cnt, cur, flags, fileName, data);
}

void finishIndexJobConfig( struct zcontext_t *cnt, Boolean success,
    Boolean hasValue, void *data)
{
  if( success )
  {
    if( currentJob->defaults == NULL )
    {
      if( indexJobs == NULL )
        addIndexJob( currentJob );
      else if( indexJobs != currentJob )
        freeIndexJob( currentJob );
    }
    else if( zStringCollectionCount( &currentJob->objects ) == 0 )
    {
      /* XXX: warning: no index object in [job]... */
      freeIndexJob( currentJob );
    }
    else
      addIndexJob( currentJob );

    currentJob = NULL;
  }
  else
  {
    if( indexJobs != currentJob ) freeIndexJob( currentJob );
  }

  zFinishDefaultConfig( cnt, success, hasValue, data);
}

#ifdef __MSVC__
#pragma warning( disable: 4100)
#else
#pragma warn -par
#endif

int getIndexJobObjectValue( struct zcontext_t *cnt,
    struct zconfdef_t *cfg, char *value, void *data)
{
  if( currentJob->defaults == NULL )
  {
    /* XXX: warning: no index object in [defaults] is allowed... */
    return zpcOk;
  }

  zSetFlags( currentJob->flags, INDEXJOB_FLAG_HAVE_OBJECTS);
  return zConfigStringListValue( cnt, cfg, value, &currentJob->objects);
}

int getIndexJobIgnoreFileNamesValue( struct zcontext_t *cnt,
    struct zconfdef_t *cfg, char *value, void *data)
{
  zSetFlags( currentJob->flags, INDEXJOB_FLAG_HAVE_SELECTS);
  return zConfigStringListValue( cnt, cfg, value, &currentJob->ignoreFileNames);
}

int getIndexJobIgnoreDirNamesValue( struct zcontext_t *cnt,
    struct zconfdef_t *cfg, char *value, void *data)
{
  zSetFlags( currentJob->flags, INDEXJOB_FLAG_HAVE_SELECTS);
  return zConfigStringListValue( cnt, cfg, value, &currentJob->ignoreDirNames);
}

int getIndexJobIgnoreFullPathFilesValue( struct zcontext_t *cnt,
    struct zconfdef_t *cfg, char *value, void *data)
{
  zSetFlags( currentJob->flags, INDEXJOB_FLAG_HAVE_SELECTS);
  return zConfigStringListValue( cnt, cfg, value, &currentJob->ignoreFullPathFiles);
}

int getIndexJobIgnoreDirWithFileValue( struct zcontext_t *cnt,
    struct zconfdef_t *cfg, char *value, void *data)
{
  zSetFlags( currentJob->flags, INDEXJOB_FLAG_HAVE_SELECTS);
  return zConfigStringListValue( cnt, cfg, value, &currentJob->ignoreDirWithFile);
}

int getIndexJobIgnoreFileURLsValue( struct zcontext_t *cnt,
    struct zconfdef_t *cfg, char *value, void *data)
{
  zSetFlags( currentJob->flags, INDEXJOB_FLAG_HAVE_SELECTS);
  return zConfigStringListValue( cnt, cfg, value, &currentJob->ignoreFileURLs);
}

int getIndexJobReplaceValue( struct zcontext_t *cnt,
    struct zconfdef_t *cfg, char *value, void *data)
{
  char *ptr;
  Boolean wasError;
  char rule;

  zSetFlags( currentJob->flags, INDEXJOB_FLAG_HAVE_REPLACES);

  if( (value = zNextWord( value, &wasError, &cnt->nextItem)) == NULL )
    return wasError ? zpcNotClosed : zpcOk;
  (void) zUnescapeString( value, True);
  if( (rule = zReplaceRuleType( value )) == ZREPLACE_RULE_UNKNOWN )
    return zpcUnknown;

  if( (value = zNextWord( NULL, &wasError, &cnt->nextItem)) == NULL )
    return zpcUnknown;
  (void) zUnescapeString( value, True);
  if( (ptr = zNextWord( NULL, &wasError, &cnt->nextItem)) == NULL && wasError )
    return zpcUnknown;
  if( ptr != NULL ) (void) zUnescapeString( ptr, True);

  if( !zReplaceRuleAdd( &currentJob->replaces, rule, value, ptr,
#ifdef HAVE_CASEINDEP_FILE_NAMES
	 ZREPLACE_FLAG_CASEINDEP) )
#else
	 0) )
#endif
    return zpcNoMemory;

  return zpcOk;
}

int getIndexJobCtypesValue( struct zcontext_t *cnt,
    struct zconfdef_t *cfg, char *value, void *data)
{
  Boolean wasError;
  char ctype;

  zSetFlags( currentJob->flags, INDEXJOB_FLAG_HAVE_CTYPES);

  if( (value = zNextWord( value, &wasError, &cnt->nextItem)) == NULL )
    return wasError ? zpcNotClosed : zpcOk;
  (void) zUnescapeString( value, True);
  if( (ctype = getContentType( value )) == CTYPE_UNKNOWN ) ctype = CTYPE_NONTEXT;

  while( (value = zNextWord( NULL, &wasError, &cnt->nextItem)) != NULL )
    if( strcasecmp( value, FLUIDS_DEFAULT_STRING) == 0 )
    {
      const char * const *p;
      unsigned int flag;

      switch( ctype )
      {
        case CTYPE_HTML:
          p = defaultHtmlContent;
          flag = INDEXJOB_FLAG_HAVE_DEFAULT_HTML_CONTENT;
          break;
        case CTYPE_TEXT:
          p = defaultTextContent;
	  flag = INDEXJOB_FLAG_HAVE_DEFAULT_TEXT_CONTENT;
          break;
        default:
          p = defaultWithoutContent;
          flag = INDEXJOB_FLAG_HAVE_DEFAULT_WOUT_CONTENT;
      }

      if( !zCheckFlags( currentJob->flags, flag) )
        for( ; *p != NULL; p++)
          if( !zParamCollectionAdd( &currentJob->ctypes, *p, &ctype, 1) )
            return zpcNoMemory;

      zSetFlags( currentJob->flags, flag);
    }
    else
      if( !zParamCollectionAdd( &currentJob->ctypes, value, &ctype, 1) )
	return zpcNoMemory;

  return wasError ? zpcNotClosed : zpcOk;
}

#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
int getIndexJobCharsetsValue( struct zcontext_t *cnt,
    struct zconfdef_t *cfg, char *value, void *data)
{
  Boolean wasError;
  char charset;

  zSetFlags( currentJob->flags, INDEXJOB_FLAG_HAVE_CHARSETS);

  if( (value = zNextWord( value, &wasError, &cnt->nextItem)) == NULL )
    return wasError ? zpcNotClosed : zpcOk;
  (void) zUnescapeString( value, True);
  if( (charset = zCharsetType( value )) == ZCHARSET_UNKNOWN )
    return zpcUnknown;

  while( (value = zNextWord( NULL, &wasError, &cnt->nextItem)) != NULL )
    if( !zParamCollectionAdd( &currentJob->charsets, value, &charset, 1) )
      return zpcNoMemory;

  return wasError ? zpcNotClosed : zpcOk;
}

int getIndexJobCharsetMethodsValue( struct zcontext_t *cnt,
    struct zconfdef_t *cfg, char *value, void *data)
{
  Boolean wasError;
  int cmethod, count, i;

  count = 0;
  zSetFlags( currentJob->flags, INDEXJOB_FLAG_HAVE_CHARSET_METHODS);

  while( (value = zNextWord( value, &wasError, &cnt->nextItem)) != NULL )
  {
    (void) zUnescapeString( value, True);
    if( (cmethod = (int) zGetStatusByName( cmethodTable, value)) == 0 )
      return (cnt->errorStrParam = cfg->name, zpcInvalidValue);
    for( i = 0; i < CMETHOD_LAST; i++) if( cmethod == currentJob->charsetMethods[i] ) break;
    if( i >= CMETHOD_LAST ) currentJob->charsetMethods[count++] = cmethod;
    value = NULL;
  }

  return wasError ? zpcNotClosed : zpcOk;
}

int getIndexAccessFileValue( struct zcontext_t *cnt,
    struct zconfdef_t *cfg, char *value, void *data)
{
  zSetFlags( currentJob->flags, INDEXJOB_FLAG_HAVE_ACCESS_FILE);
  return zConfigTokenValue( cnt, cfg, value, (void *) &currentJob->accessFile);
}

int getIndexJobConvertToRussianValue( struct zcontext_t *cnt,
    struct zconfdef_t *cfg, char *value, void *data)
{
  Boolean wasError;

  zSetFlags( currentJob->flags, INDEXJOB_FLAG_HAVE_CONVERT_TO_RUSSIAN);

  while( (value = zNextWord( value, &wasError, &cnt->nextItem)) != NULL )
  {
    (void) zUnescapeString( value, True);

    /*
    if( value[0] == '\0' || value[1] == '\0' || value[2] != '\0' ||
        !isAlpha( value[0] ) || !isRussian( value[1] ) )
      return (cnt->errorStrParam = cfg->name, zpcInvalidValue);
    */

    if( value[0] == '\0' || value[1] == '\0' || value[2] != '\0' )
      return (cnt->errorStrParam = cfg->name, zpcInvalidValue);
    if( !isAlpha( value[0] ) )
      return (cnt->errorStrParam = cfg->name, zpcInvalidValue);
    if( !isRussian( value[1] ) )
      return (cnt->errorStrParam = cfg->name, zpcInvalidValue);

    if( currentJob->torTable == NULL ) currentJob->torTable = currentJob->_torTable;
    currentJob->_torTable[ value[0] & 0xff ] = value[1];

    value = NULL;
  }

  return wasError ? zpcNotClosed : zpcOk;
}

#endif

void freeIndexJobConfig( struct zcontext_t *cnt, void *data)
{
  freeIndexJobs();
}

#ifdef __MSVC__
#pragma warning( default: 4100)
#else
#pragma warn .par
#endif

Boolean defaultIndexJobConfig( struct zcontext_t *cnt, struct zstrbuf_t *sb)
{
  char ctype;
  const char * const *p;

  if( indexJobs == NULL )
  {
    struct indexjob_t *ij = ZNEW( cnt, struct indexjob_t);
    initIndexJob( cnt, ij, sb);
    addIndexJob( ij );
  }
  else
    return True;

  zSetFlags( indexJobs->flags, INDEXJOB_FLAG_HAVE_CTYPES);

  for( ctype = CTYPE_HTML, p = defaultHtmlContent; *p != NULL; p++)
    if( !zParamCollectionAdd( &indexJobs->ctypes, *p, &ctype, 1) )
      return False;

  for( ctype = CTYPE_TEXT, p = defaultTextContent; *p != NULL; p++)
    if( !zParamCollectionAdd( &indexJobs->ctypes, *p, &ctype, 1) )
      return False;

  for( ctype = CTYPE_NONTEXT, p = defaultWithoutContent; *p != NULL; p++)
    if( !zParamCollectionAdd( &indexJobs->ctypes, *p, &ctype, 1) )
      return False;

  return True;
}

/***************************************************************************/
/*                                                                         */
/*  Printing                                                               */
/*                                                                         */
/***************************************************************************/

static void printSelectCollection( struct zcontext_t *cnt,
    const char *header, struct zstrcoll_t *sc)
{
  unsigned int i;
  char lastParam = (char) 0xff;

  for( i = 0; i < sc->count; i++)
  {
    const char *string = sc->list[i];

    if( string[0] != lastParam )
      zprintf( cnt, "%s%s%s = \"%s\"",
        (i == 0) ? "" : "\n",
        (string[0] == '\0') ? "Accept" : "Ignore",
        header,
        &string[1]);
    else
      zprintf( cnt, " \"%s\"", &string[1]);

    lastParam = string[0];
  }

  zprintf( cnt, "\n");
}

static void printIndexJob( struct zcontext_t *cnt, struct indexjob_t *ij)
{
  struct zreplacerule_t *rr;
  Boolean first;
  const char *s;
  char last;
  int i;

/*  ࠧ */
  zprintf( cnt, ij->defaults == NULL ? "[defaults]\n" : "[job]\n");

/* ꥪ 樨 */
  if( ij->defaults != NULL )
    PRINT_COLLECTION_A( cnt, "IndexObject =", &ij->objects, 0);

/*  䠩 */
  if( zCheckFlags( ij->flags, INDEXJOB_FLAG_HAVE_CTYPES) )
  {
    if( zStringCollectionEmpty( &ij->ctypes ) )
    {
      if( ij->defaults != NULL ) zprintf( cnt, "ctype =\n");
    }
    else for( i = 0, last = CTYPE_UNKNOWN, first = True; i < ij->ctypes.count; i++)
    {
      if( last != ij->ctypes.list[i][0] )
      {
        if( !first ) zprintf( cnt, "\n");
        zprintf( cnt, "ctype = ");
        switch( last = ij->ctypes.list[i][0] )
        {
          case CTYPE_HTML:    s = CTYPE_PLAIN_HTML; break;
          case CTYPE_TEXT:    s = CTYPE_PLAIN_TEXT; break;
          default:
          case CTYPE_NONTEXT: s = CTYPE_OTHER; break;
        }
        zprintf( cnt, "%~s", s);
        first = False;
      }
      zprintf( cnt, " \"%s\"", &ij->ctypes.list[i][1]);
    }
    zprintf( cnt, "\n");
  }

#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
/* ⮤ । ஢ */
  if( zCheckFlags( ij->flags, INDEXJOB_FLAG_HAVE_CHARSET_METHODS) )
    if( ij->charsetMethods[0] == CMETHOD_NONE )
    {
      if( ij->defaults != NULL ) zprintf( cnt, "CharsetMethods =\n");
    }
    else
    {
      zprintf( cnt, "CharsetMethods =");
      for( i = 0; i < CMETHOD_LAST && ij->charsetMethods[i] != CMETHOD_NONE; i++)
         zprintf( cnt, " %~s", zGetStatusByValue( cmethodTable, ij->charsetMethods[i]));
      zprintf( cnt, "\n");
    }

/* ஢ */
  if( zCheckFlags( ij->flags, INDEXJOB_FLAG_HAVE_CHARSETS) )
    if( zStringCollectionEmpty( &ij->charsets ) )
    {
      if( ij->defaults != NULL ) zprintf( cnt, "charset =\n");
    }
    else
    {
      for( i = 0, last = ZCHARSET_UNKNOWN, first = True; i < ij->charsets.count; i++)
      {
        if( last != ij->charsets.list[i][0] )
        {
          if( !first ) zprintf( cnt, "\n");
          zprintf( cnt, "charset = %s", zCharsetName( ij->charsets.list[i][0], ZCHARSET_NAME_SHORT));
          first = False;
        }
        zprintf( cnt, " \"%s\"", &ij->charsets.list[i][1]);
      }
      zprintf( cnt, "\n");
    }

/*  㯠 */
  if( zCheckFlags( ij->flags, INDEXJOB_FLAG_HAVE_ACCESS_FILE) )
    if( ij->accessFile == NULL )
    {
      if( ij->defaults != NULL ) zprintf( cnt, "AccessFile =\n");
    }
    else
      zprintf( cnt, "AccessFile = \"%a\"\n", ij->accessFile);

/* ࠭  ᪨ 㪢 */
  if( zCheckFlags( ij->flags, INDEXJOB_FLAG_HAVE_CONVERT_TO_RUSSIAN) &&
      ij->torTable != NULL )
  {
    int i;
    zprintf( cnt, "ConvertToRussian =");
    for( i = 0; i < 128; i++)
      if( ij->torTable[i] != '\0' )
        zprintf( cnt, " \"%c%c\"", i, ij->torTable[i] & 0xff);
    zprintf( cnt, "\n");
  }
#endif

/* ८ࠧ  */
  if( zCheckFlags( ij->flags, INDEXJOB_FLAG_HAVE_REPLACES) )
    if( zReplaceEmpty( &ij->replaces ) )
    {
      if( ij->defaults != NULL ) zprintf( cnt, "replace =\n");
    }
    else for( rr = ij->replaces.rules; rr != NULL; rr = rr->next)
    {
      zprintf( cnt, "replace = %s \"%.*A\"", zReplaceRuleName( rr->type ), rr->length1, rr->string1);
      if( rr->type != ZREPLACE_RULE_PREPEND && rr->type != ZREPLACE_RULE_APPEND )
        zprintf( cnt, " \"%.*A\"", rr->length2, rr->string2);
      zprintf( cnt, "\n");
    }

/* ࠢ ⡮ */
  if( zCheckFlags( ij->flags, INDEXJOB_FLAG_HAVE_SELECTS) )
    if( zStringCollectionEmpty( &ij->ignoreFileNames ) &&
        zStringCollectionEmpty( &ij->ignoreDirNames ) &&
        zStringCollectionEmpty( &ij->ignoreFullPathFiles ) &&
        zStringCollectionEmpty( &ij->ignoreDirWithFile ) &&
        zStringCollectionEmpty( &ij->ignoreFileURLs ) )
    {
      if( ij->defaults != NULL ) zprintf( cnt, "IgnoreFileName =\n");
    }
    else
    {
      if( !zStringCollectionEmpty( &ij->ignoreFileNames ) )
	printSelectCollection( cnt, "FileName", &ij->ignoreFileNames);
      if( !zStringCollectionEmpty( &ij->ignoreDirNames ) )
	printSelectCollection( cnt, "DirName", &ij->ignoreFileNames);
      if( !zStringCollectionEmpty( &ij->ignoreFullPathFiles ) )
	printSelectCollection( cnt, "FullPathFile", &ij->ignoreFileNames);
      if( !zStringCollectionEmpty( &ij->ignoreDirWithFile ) )
	printSelectCollection( cnt, "DirWithFile", &ij->ignoreFileNames);
      if( !zStringCollectionEmpty( &ij->ignoreFileURLs ) )
        printSelectCollection( cnt, "FileURL", &ij->ignoreFileURLs);
    }

/* ⥫쭠  ப */
  zprintf( cnt, "\n");
}

void printIndexJobs( struct zcontext_t *cnt )
{
  struct indexjob_t *ij;

  for( ij = indexJobs; ij != NULL; ij = ij->next) printIndexJob( cnt, ij);
}
