/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */

#include "cfg.h"
#include "defs.h"
#include "zcontext.h"
#include "zcoll.h"
#include "zconfig.h"
#include "zerror.h"
#include "zstring.h"
#include "zstdio.h"

#include "fludata.h"
#include "values.h"

#include "indexjob.h"
#include "configur.h"

#define areaIndexing                   0x0001u

static const char * const defaultStopWords[] =
{
  "a", "all", "an", "and", "are", "can", "could", "did", "do", "for", "from",
  "had", "have", "it", "may", "not", "of", "that", "the", "this", "was",
  "will", "with", "you", "your",
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
  "", "", "", "", "", "",
#endif
  NULL
};

static const char * const defaultValidWords[] =
{
  "95", "98", "ar", "bg", "c", "cc", "cd", "cp", "dd", "ed",
  "h1", "h2", "h3", "h4", "h5", "h6", "fg", "id", "im", "ip",
  "ln", "ls", "mc", "mv", "nc", "ng", "nt", "od", "os", "pr",
  "re", "rm", "sh", "tr", "vi", "vm", "wc",
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
  "", "", "", "", "", "", "", "", "ndex",
#endif
  NULL
};

/***************************************************************************/
/*                                                                         */
/*  Config handlers                                                        */
/*                                                                         */
/***************************************************************************/

#ifdef __MSVC__
#pragma warning( disable: 4100)
#else
#pragma warn -par
#endif

static int getConfigVerboseValue( struct zcontext_t *cnt,
    struct zconfdef_t *cfg, char *value, void *data)
{
  setVerboseValue( cnt, value);
  return zpcOk;
}

static int getConfigWordsValue( struct zcontext_t *cnt,
    struct zconfdef_t *cfg, char *value, void *data)
{
  Boolean wasError;
  struct zstrcoll_t *sc = (struct zstrcoll_t *) cfg->value;

  while( (value = zNextWord( value, &wasError, &cnt->nextItem)) != NULL )
  {
    zUnescapeString( value, True);

    if( strcasecmp( value, FLUIDS_DEFAULT_STRING) == 0 )
    {
      if( !zCheckFlags( cfg->flags, ZCONFIG_FLAG_PROCESSED) )
      {
        const char * const *p;

        if( zCheckFlags( cfg->flags, ZCONFIG_FLAG_SPECIAL) )
	  for( p = defaultStopWords; *p != NULL; p++)
            zStringCollectionAdd( sc, *p, 0);
        else
          for( p = defaultValidWords; *p != NULL; p++)
            zStringCollectionAdd( sc, *p, 0);

        zSetFlags( cfg->flags, ZCONFIG_FLAG_PROCESSED);
      }
    }
    else if( strlen( value ) <= MAX_WORD_LENGTH )
    {
      zStringLower( value );
      zStringCollectionAdd( sc, value, 0);
    }

    value = NULL;
  }

  return wasError ? zpcNotClosed : zpcOk;
}

#ifdef __MSVC__
#pragma warning( default: 4100)
#else
#pragma warn .par
#endif

/***************************************************************************/
/*                                                                         */
/*  Main conf table                                                        */
/*                                                                         */
/***************************************************************************/

zint_t E_MaxMemoryVolum = MAX_MEMORY_VOLUM;
char *E_TempDir = NULL;
int E_MaxSwapFiles = MAX_SWAP_FILES;
Boolean E_FollowLinks = False;
int E_MaxContentLength = MAX_CONTENT_LENGTH;

char *E_IndexFile = NULL;
char *E_IndexName = NULL;
char *E_IndexDescription = NULL;
char *E_IndexPointer = NULL;
char *E_IndexAdmin = NULL;

static struct zconfdef_t mainConfTable[] =
{
  {
    "TempDir",
    areaIndexing,
    zConfigTokenValue,
    &E_TempDir,
    NULL,
    -1,
    0
  },
  {
    "MaxMemoryVolum",
    areaIndexing,
    zConfigIntegerValue,
    (char **) &E_MaxMemoryVolum,
    NULL,
    -1,
    ZCONFIG_FLAG_POSITIVE | ZCONFIG_FLAG_ZVALUE
  },
  {
    "MaxSwapFiles",
    areaIndexing,
    zConfigIntegerValue,
    (char **) &E_MaxSwapFiles,
    NULL,
    -1,
    ZCONFIG_FLAG_POSITIVE
  },
  {
    "IndexFile",
    areaIndexing,
    zConfigTokenValue,
    &E_IndexFile,
    NULL,
    -1,
    0
  },
  {
    "Verbose",
    areaIndexing,
    getConfigVerboseValue,
    NULL,
    NULL,
    -1,
    0
  },
  {
    "FollowLinks",
    areaIndexing,
    zConfigBooleanValue,
    (char **) &E_FollowLinks,
    NULL,
    -1,
    0
  },
  {
    "IndexName",
    areaIndexing,
    zConfigStringValue,
    &E_IndexName,
    NULL,
    -1,
    ZCONFIG_FLAG_CHECK_SPACES | ZCONFIG_FLAG_COMPRESS
  },
  {
    "IndexDescription",
    areaIndexing,
    zConfigStringValue,
    &E_IndexDescription,
    NULL,
    -1,
    ZCONFIG_FLAG_CHECK_SPACES | ZCONFIG_FLAG_COMPRESS
  },
  {
    "IndexPointer",
    areaIndexing,
    zConfigStringValue,
    &E_IndexPointer,
    NULL,
    -1,
    ZCONFIG_FLAG_CHECK_SPACES | ZCONFIG_FLAG_COMPRESS
  },
  {
    "IndexAdmin",
    areaIndexing,
    zConfigStringValue,
    &E_IndexAdmin,
    NULL,
    -1,
    ZCONFIG_FLAG_CHECK_SPACES | ZCONFIG_FLAG_COMPRESS
  },
  {
    "MaxContentLength",
    areaIndexing,
    zConfigIntegerValue,
    (char **) &E_MaxContentLength,
    NULL,
    -1,
    ZCONFIG_FLAG_POSITIVE
  },
  {
    NULL
  }
};

#ifdef __MSVC__
#pragma warning( disable: 4100)
#else
#pragma warn -par
#endif

static void freeMainConfig( struct zcontext_t *cnt, void *data)
{
  ZFREE( cnt, E_TempDir);
  ZFREE( cnt, E_IndexFile);
  ZFREE( cnt, E_IndexName);
  ZFREE( cnt, E_IndexDescription);
  ZFREE( cnt, E_IndexPointer);
  ZFREE( cnt, E_IndexAdmin);
}

#ifdef __MSVC__
#pragma warning( default: 4100)
#else
#pragma warn .par
#endif

/***************************************************************************/
/*                                                                         */
/*  Job conf table                                                         */
/*                                                                         */
/***************************************************************************/

static struct zconfdef_t jobConfTable[] =
{
  {
    "IndexObject",
    areaIndexing,
    getIndexJobObjectValue,
    NULL,
    NULL,
    0,
    0
  },
  {
    "Replace",
    areaIndexing,
    getIndexJobReplaceValue,
    NULL,
    NULL,
    0,
    0
  },
  {
    "Ctype",
    areaIndexing,
    getIndexJobCtypesValue,
    NULL,
    NULL,
    0,
    0
  },
  {
    "Charset",
    areaIndexing,
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
    getIndexJobCharsetsValue,
#else
    NULL,
#endif
    NULL,
    NULL,
    0,
    0
  },
  {
    "CharsetMethods",
    areaIndexing,
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
    getIndexJobCharsetMethodsValue,
#else
    NULL,
#endif
    NULL,
    NULL,
    0,
    0
  },
  {
    "IgnoreFileName",
    areaIndexing,
    getIndexJobIgnoreFileNamesValue,
    NULL,
    NULL,
    0,
    ZCONFIG_FLAG_LEAVE_ESCAPED | ZCONFIG_FLAG_PARAM_YES
  },
  {
    "AcceptFileName",
    areaIndexing,
    getIndexJobIgnoreFileNamesValue,
    NULL,
    NULL,
    0,
    ZCONFIG_FLAG_LEAVE_ESCAPED | ZCONFIG_FLAG_PARAM_NO
  },
  {
    "IgnoreDirName",
    areaIndexing,
    getIndexJobIgnoreDirNamesValue,
    NULL,
    NULL,
    0,
    ZCONFIG_FLAG_LEAVE_ESCAPED | ZCONFIG_FLAG_PARAM_YES
  },
  {
    "AcceptDirName",
    areaIndexing,
    getIndexJobIgnoreDirNamesValue,
    NULL,
    NULL,
    0,
    ZCONFIG_FLAG_LEAVE_ESCAPED | ZCONFIG_FLAG_PARAM_NO
  },
  {
    "IgnoreFullPathFile",
    areaIndexing,
    getIndexJobIgnoreFullPathFilesValue,
    NULL,
    NULL,
    0,
    ZCONFIG_FLAG_LEAVE_ESCAPED | ZCONFIG_FLAG_PARAM_YES
  },
  {
    "AcceptFullPathFile",
    areaIndexing,
    getIndexJobIgnoreFullPathFilesValue,
    NULL,
    NULL,
    0,
    ZCONFIG_FLAG_LEAVE_ESCAPED | ZCONFIG_FLAG_PARAM_NO
  },
  {
    "IgnoreDirWithFile",
    areaIndexing,
    getIndexJobIgnoreDirWithFileValue,
    NULL,
    NULL,
    0,
    ZCONFIG_FLAG_LEAVE_ESCAPED | ZCONFIG_FLAG_PARAM_YES
  },
  {
    "AcceptDirWithFile",
    areaIndexing,
    getIndexJobIgnoreDirWithFileValue,
    NULL,
    NULL,
    0,
    ZCONFIG_FLAG_LEAVE_ESCAPED | ZCONFIG_FLAG_PARAM_NO
  },
  {
    "IgnoreFileURL",
    areaIndexing,
    getIndexJobIgnoreFileURLsValue,
    NULL,
    NULL,
    0,
    ZCONFIG_FLAG_LEAVE_ESCAPED | ZCONFIG_FLAG_PARAM_YES
  },
  {
    "AcceptFileURL",
    areaIndexing,
    getIndexJobIgnoreFileURLsValue,
    NULL,
    NULL,
    0,
    ZCONFIG_FLAG_LEAVE_ESCAPED | ZCONFIG_FLAG_PARAM_NO
  },
  {
    "AccessFile",
    areaIndexing,
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
    getIndexAccessFileValue,
#else
    NULL,
#endif
    NULL,
    NULL,
    0,
    0
  },
  {
    "ConvertToRussian",
    areaIndexing,
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
    getIndexJobConvertToRussianValue,
#else
    NULL,
#endif
    NULL,
    NULL,
    0,
    0
  },
  {
    NULL
  }
};

/***************************************************************************/
/*                                                                         */
/*  Word conf table                                                        */
/*                                                                         */
/***************************************************************************/

struct zstrcoll_t E_ValidWords;
struct zstrcoll_t E_StopWords;
struct zstrcoll_t E_BadWords;
struct zstrcoll_t E_RejectWords;
int E_IgnoreFilesLimit = IGNORE_FILES_LIMIT;
int E_IgnorePercentLimit = IGNORE_PERCENT_LIMIT;
int E_MinWordLength = MIN_WORD_LENGTH;
int E_MaxWordLength = MAX_WORD_LENGTH;

static struct zconfdef_t wordConfTable[] =
{
  {
    "CommonWords",
    areaIndexing,
    getConfigWordsValue,
    (char **) &E_StopWords,
    NULL,
    -1,
    ZCONFIG_FLAG_SPECIAL
  },
  {
    "ValidWords",
    areaIndexing,
    getConfigWordsValue,
    (char **) &E_ValidWords,
    NULL,
    -1,
    0
  },
  {
    "BadWords",
    areaIndexing,
    zConfigStringListValue,
    (char **) &E_BadWords,
    NULL,
    -1,
    ZCONFIG_FLAG_LOWERCASE
  },
  {
    "RejectWords",
    areaIndexing,
    zConfigStringListValue,
    (char **) &E_RejectWords,
    NULL,
    -1,
    ZCONFIG_FLAG_LOWERCASE | ZCONFIG_FLAG_LEAVE_ESCAPED
  },
  {
    "IgnoreFilesLimit",
    areaIndexing,
    zConfigIntegerValue,
    (char **) &E_IgnoreFilesLimit,
    NULL,
    -1,
    ZCONFIG_FLAG_POSITIVE
  },
  {
    "IgnorePercentLimit",
    areaIndexing,
    zConfigIntegerValue,
    (char **) &E_IgnorePercentLimit,
    NULL,
    -1,
    ZCONFIG_FLAG_POSITIVE
  },
  {
    "MaxWordLength",
    areaIndexing,
    zConfigIntegerValue,
    (char **) &E_MaxWordLength,
    NULL,
    -1,
    ZCONFIG_FLAG_POSITIVE
  },
  {
    "MinWordLength",
    areaIndexing,
    zConfigIntegerValue,
    (char **) &E_MinWordLength,
    NULL,
    -1,
    ZCONFIG_FLAG_POSITIVE
  },
  {
    NULL
  }
};

#ifdef __MSVC__
#pragma warning( disable: 4100)
#else
#pragma warn -par
#endif

static void freeWordConfig( struct zcontext_t *cnt, void *data)
{
  zStringCollectionFree( &E_ValidWords );
  zStringCollectionFree( &E_StopWords );
  zStringCollectionFree( &E_BadWords );
  zStringCollectionFree( &E_RejectWords );
}

#ifdef __MSVC__
#pragma warning( default: 4100)
#else
#pragma warn .par
#endif

/***************************************************************************/
/*                                                                         */
/*  Config Table                                                           */
/*                                                                         */
/***************************************************************************/

static struct zconfsec_t confTable[] =
{
  {
    "main",
    areaIndexing,
    zInitDefaultConfig,
    zProcessDefaultConfig,
    NULL,
    freeMainConfig,
    mainConfTable
  },
  {
    "defaults",
    areaIndexing,
    initDefaultIndexJobConfig,
    zProcessDefaultConfig,
    finishIndexJobConfig,
    NULL,
    jobConfTable
  },
  {
    "job",
    areaIndexing,
    initIndexJobConfig,
    zProcessDefaultConfig,
    finishIndexJobConfig,
    freeIndexJobConfig,
    jobConfTable
  },
  {
    "words",
    areaIndexing,
    zInitDefaultConfig,
    zProcessDefaultConfig,
    NULL,
    freeWordConfig,
    wordConfTable
  },
  {
    NULL
  }
};

static Boolean defaultConfigure( struct zcontext_t *cnt, struct zstrbuf_t *sb)
{
  const char * const *p;

  if( !defaultIndexJobConfig( cnt, sb) )
  {
    cnt->printError( cnt, zerNoMemory, NULL);
    return False;
  }

  /* XXX: no memory ? */
  for( p = defaultStopWords; *p != NULL; p++)
    zStringCollectionAdd( &E_StopWords, *p, 0);
  for( p = defaultValidWords; *p != NULL; p++)
    zStringCollectionAdd( &E_ValidWords, *p, 0);

  return True;
}

Boolean configure( struct zcontext_t *cnt, const char *fileName, struct zstrbuf_t *sb)
{
  Boolean success;

  (void) zStringCollectionInit( cnt, &E_ValidWords, 0, 5, zcfSorted | zcfCheckDuplicate);
  (void) zStringCollectionInit( cnt, &E_StopWords, 0, 10, zcfSorted | zcfCheckDuplicate);
  (void) zStringCollectionInit( cnt, &E_BadWords, 0, 5, zcfSorted | zcfCheckDuplicate);
  (void) zStringCollectionInit( cnt, &E_RejectWords, 0, 5, zcfCheckDuplicate);

  if( fileName == NULL )
    success = defaultConfigure( cnt, sb);
  else
    success = zReadConfFile( cnt, fileName, NULL, confTable, areaIndexing, sb) ?
      cnt->confSuccess : False;

/* 쪨 fixup */
  if( E_MinWordLength <= 0 ) E_MinWordLength = 1;
  if( E_MaxWordLength > MAX_WORD_SIZE ) E_MaxWordLength = MAX_WORD_SIZE;
  if( E_MaxContentLength > MAX_CONTENT_LENGTH ) E_MaxContentLength = MAX_CONTENT_LENGTH;

  return success;
}

void freeConfMemory( struct zcontext_t *cnt )
{
  zFreeConfBlocks( cnt, confTable, NULL);
}

/***************************************************************************/
/*                                                                         */
/*  Printing                                                               */
/*                                                                         */
/***************************************************************************/

static void printMainConfigure( struct zcontext_t *cnt )
{
/*  ࠧ */
  zprintf( cnt, "[main]\n");

/* TempDir */
  if( E_TempDir != NULL )
    zprintf( cnt, "TempDir = \"%a\"\n", E_TempDir);

/* MaxMemoryVolum */
  zprintf( cnt, "MaxMemoryVolum = %" _ZINT_FORMAT "d\n", E_MaxMemoryVolum);

/* MaxSwapFiles */
  zprintf( cnt, "MaxSwapFiles = %d\n", E_MaxSwapFiles);

/* IndexFile */
  if( E_IndexFile != NULL )
    zprintf( cnt, "IndexFile = \"%a\"\n", E_IndexFile);

/* Verbose */
  zprintf( cnt, "Verbose = %d\n", cnt->verboseLevel);

/* FollowLinks */
  zprintf( cnt, "FollowLinks = %~s\n", E_FollowLinks ? "True" : "False");

/* IndexName */
  if( E_IndexName != NULL )
    zprintf( cnt, "IndexName = \"%a\"\n", E_IndexName);

/* IndexDescription */
  if( E_IndexDescription != NULL )
    zprintf( cnt, "IndexDescription = \"%a\"\n", E_IndexDescription);

/* IndexPointer */
  if( E_IndexPointer != NULL )
    zprintf( cnt, "IndexPointer = \"%a\"\n", E_IndexPointer);

/* IndexAdmin */
  if( E_IndexAdmin != NULL )
    zprintf( cnt, "IndexAdmin = \"%a\"\n", E_IndexAdmin);

/* ⥫쭠  ப */
  zprintf( cnt, "\n");
}

static void printWordConfigure( struct zcontext_t *cnt )
{
  int i;

/*  ࠧ */
  zprintf( cnt, "[words]\n" );

/* CommonWords */
  if( !zStringCollectionEmpty( &E_StopWords ) )
    PRINT_COLLECTION_A( cnt, "CommonWords =", &E_StopWords, 0);

/* ValidWords */
  if( !zStringCollectionEmpty( &E_ValidWords ) )
    PRINT_COLLECTION_A( cnt, "ValidWords =", &E_ValidWords, 0);

/* BadWords */
  if( !zStringCollectionEmpty( &E_BadWords ) )
    PRINT_COLLECTION_A( cnt, "BadWords =", &E_BadWords, 0);

/* RejectWords */
  if( !zStringCollectionEmpty( &E_RejectWords ) )
    PRINT_COLLECTION_S( cnt, "RejectWords =", &E_RejectWords, 0);

/* IgnoreFilesLimit */
  zprintf( cnt, "IgnoreFilesLimit = %d\n", E_IgnoreFilesLimit);

/* IgnorePercentLimit */
  zprintf( cnt, "IgnorePercentLimit = %d\n", E_IgnorePercentLimit);

/* MaxWordLength */
  zprintf( cnt, "MaxWordLength = %d\n", E_MaxWordLength);

/* MinWordLength */
  zprintf( cnt, "MinWordLength = %d\n", E_MinWordLength);

/* ⥫쭠  ப */
  zprintf( cnt, "\n");
}

void printConfigure( struct zcontext_t *cnt )
{
  printMainConfigure( cnt );
  printIndexJobs( cnt );
  printWordConfigure( cnt );
}
