/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstdio.h" /* <stdio.h> */
#include <stdlib.h>

#include "zcontext.h"
#include "zchars.h"
#include "zcharset.h"
#include "zcoll.h"
#include "zerror.h"
#include "zstdio.h"

#include "cfg.h"
#include "defs.h"
#include "error.h"
#include "fludata.h"
#include "values.h"

#include "dump.h"

#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) &&\
    defined( UKRAINIAN_SUPPORT ) && defined( UKRAINIAN_INTERFACE )
#define EFORMATS zErrorUkrainianFormats
#elif defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) &&\
    defined( RUSSIAN_INTERFACE )
#define EFORMATS zErrorRussianFormats
#else
#define EFORMATS NULL
#endif

static void printUsage( struct zcontext_t *cnt );
static void exitProgram( struct zcontext_t *cnt, int exitCode);
static void printError( struct zcontext_t *cnt, unsigned zint_t errorCode, const char *name);
static void memoryFail( struct zcontext_t *cnt, const char *prog);

static struct zstrcoll_t wordList;

void main( int argc, char **argv)
{
  struct zcontext_t dumpContext;
  struct fludata_t fluData;
  const char *fileName = NULL, *docName = NULL;
  Boolean doUsage = False, success = True;
  unsigned int options = 0;
  Boolean changed = False;
  int argx;
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
  char *ptr;

  zInitRussians();
#endif

/* 樠㥬 ࠧ த ⥪ */
  zContextInit( &dumpContext, printError, EFORMATS, memoryFail, 0);
  initFludata( &dumpContext, &fluData);
  zSetFlags( dumpContext.ioFlags, ZCONTEXT_IOFLAG_AUTOFLUSH);
  zStringCollectionInit( &dumpContext, &wordList, 0, 10, zcfDontFreeContent);
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) && defined( REMOTE_CHARSET )
  zSetRemoteCharset( &dumpContext, ARGS_CHARSET, REMOTE_CHARSET);
#endif

/* ࠥ 㬥 */
  for( argx = 1; argx < argc; argx++)
  {
    if( argv[argx][0] == '-' )
      switch( argv[argx][1] )
      {
        case 'c':
          zSetFlags( options, optFileList);
          changed = True;
          break;

#if defined( FLUIDS43 )
        case 'd':
          zSetFlags( options, optDirIndex);
          changed = True;
          break;
#endif

        case 'f':
          zSetFlags( options, optFileIndex);
          changed = True;
          break;

        case 'h':
          zSetFlags( options, optSuppressHeader);
          changed = True;
          break;

        case 'i':
          zSetFlags( options, optWordIndex);
          changed = True;
          break;

        case 'l':
          zSetFlags( options, optWordList);
          changed = True;
          break;

        case 'n':
          GET_ARGUMENT( docName );
          changed = True;
          break;

        case 'p':
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
	  GET_ARGUMENT( ptr );
          if( !doUsage )
          {
            int charset = zCharsetType( ptr );
            zSetRemoteCharset( &dumpContext, ARGS_CHARSET, charset);
          }
#endif
          changed = True;
          break;

        case 't':
          zSetFlags( options, optWordTrack);
          changed = True;
          break;

        case 'w':
          FILLIN_LIST( wordList );
          changed = True;
          break;

        default:
	case '?':
          doUsage = True;
          break;
      }
    else if( fileName == NULL )
      fileName = argv[argx];
    else
      doUsage = True;
  }

  if( fileName == NULL )
    if( changed )
      fileName = DEFAULT_INDEX_FILE_NAME;
    else
      doUsage = True;

  if( doUsage ) printUsage( &dumpContext );

  success = dump( &dumpContext, fileName, options, &wordList, docName);

  exitProgram( &dumpContext, success ? 0 : 1);
}

static void printUsage( struct zcontext_t *cnt )
{
  zprintf( cnt, "usage: fldump [-c] "
#if defined( FLUIDS43 )
                "[-d] "
#endif
                "[-f] [-h] [-i] [-l] [-n document-name] "
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
                "[-p charset] "
#endif
                "[-t] [-w word1 [-w word2 ...]] index-file\n\n"
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
                "options: defaults are in brackets\n"
#else
                "options:\n"
#endif
                "       -c : dump the compact file list of the index file\n"
#if defined( FLUIDS43 )
                "       -d : dump the directory index of the index file\n"
#endif
                "       -f : dump the file index of the index file\n"
                "       -h : suppress the header\n"
                "       -i : dump the word index of the index file\n"
                "       -l : dump the word list of the index file\n"
                "       -n : dump the word list for the desired document\n"
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
                "       -p : print out all messages in the desired charset [%s]\n"
#endif
                "       -t : dump the word track of the index file\n"
                "       -w : dump the index info for the selected word\n\n"
                "version: " VERSION "\n"
                "   docs: http://www.sbnet.ru/soft/fluids/\n"
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
#ifdef REMOTE_CHARSET
    , zCharsetName( REMOTE_CHARSET, ZCHARSET_NAME_SHORT)
#else
    , zCharsetName( LOCAL_CHARSET, ZCHARSET_NAME_SHORT)
#endif
#endif
    );

  exitProgram( cnt, 1);
}

static void exitProgram( struct zcontext_t *cnt, int exitCode)
{
  zStringCollectionFree( &wordList );
  zContextFree( cnt );
  exit( exitCode );
}

static void printError( struct zcontext_t *cnt, unsigned zint_t errorCode, const char *name)
{
  char line[1024+1024+100];
  if( _zErrorCode( errorCode ) == zecNone ) return;
  (void) fluGetErrorString( cnt, line, sizeof( line ), errorCode, name);
  zprintf( cnt, "err: %s\n", line);
}

static void memoryFail( struct zcontext_t *cnt, const char *prog)
{
  cnt->printError( cnt, zerNoMemory, prog);
  exitProgram( cnt, -1);
}
