/*
    FLUIdS - local search system
    Copyright (C) 1998, 2000  VVK (valera@sbnet.ru), CNII Center, Moscow

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "zdefs.h"
#include "_pstring.h" /* <string.h> */

#include "zalloc.h"
#include "zchars.h"
#include "zcharset.h"
#include "zstdio.h"
#include "ztime.h"

#include "cfg.h"
#include "defs.h"
#include "indxfile.h"
#include "structur.h"

#include "dump.h"
#include "fileinfo.h"

#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) &&\
    defined( UKRAINIAN_SUPPORT ) && defined( UKRAINIAN_INTERFACE )
#define TIME_STRINGS zTimeUkrainianStrings
#elif defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE ) &&\
    defined( RUSSIAN_INTERFACE )
#define TIME_STRINGS zTimeRussianStrings
#else
#define TIME_STRINGS NULL
#endif

/***************************************************************************/
/*                                                                         */
/*  Directory index                                                        */
/*                                                                         */
/***************************************************************************/

#if defined( FLUIDS43 )
static void printDir( struct zcontext_t *cnt, const char *name,
    Boolean last, unsigned int level, _fn_t startNum, _fn_t count,
    const char *header, const char **pnewHeader)
{
  if( header != NULL ) zprintf( cnt, "%s", header);
  if( level > 0 )
  {
    zprintf( cnt, "%s", last ? "*-- " : ">-- ");
    *pnewHeader = zaprintf( cnt, "%+s%s", header, last ? "    " : "|   ");
  }
  zprintf( cnt, "%s (" PRINTFORMAT_FILENUM, name, startNum);
  if( count > 1 ) zprintf( cnt, "-" PRINTFORMAT_FILENUM, startNum+count-1);
  zprintf( cnt, ")\n");
}

static Boolean dumpDirDownIndex( struct indexfile_t *pif,
    struct idir_ritem_t *up, unsigned int level, const char *header)
{
  struct idir_ritem_t *id, *head;
  Boolean success = True;

  if( !idirReadListDownRead( pif, up, &head, (up == NULL) ? 0 : up->offset) )
    return False;

  for( id = head; success && id != NULL; id = id->next)
  {
    const char *newHeader = NULL;
    char *dirName = zStrndup( pif->context, id->name, id->nameLength);
#if defined( RUSSIAN_SUPPORT ) && defined( RUSSIAN_RELEASE )
    if( pif->recodeTable != NULL ) zRecode8( dirName, pif->recodeTable);
#endif

    printDir( pif->context, dirName, (Boolean) (id->next == NULL), level, 
      id->startNum, id->count, header, &newHeader);
    zFree( pif->context, dirName);

    if( id->offset != 0 )
    {
      success = dumpDirDownIndex( pif, id, level+1, newHeader);
      idirReadListChainFree( &pif->dirList, id->down);
      id->down = NULL;
    }

    ZFREE( pif->context, newHeader);
  }

  if( up == NULL ) idirReadListChainFree( &pif->dirList, head);

  return success;
}

Boolean dumpDirIndex( struct indexfile_t *pif )
{
  return dumpDirDownIndex( pif, NULL, 0, NULL);
}
#endif

/***************************************************************************/
/*                                                                         */
/*  File index                                                             */
/*                                                                         */
/***************************************************************************/

static void printText( struct zcontext_t *cnt, const char *text, Boolean all)
{
  int i, size = 0;
  char line[80], *ptr;
  Boolean empty;

  while( isSpace( *text ) ) text++;
  if( *text == '\0' )
  {
    zprintf( cnt, "\n");
    return;
  }

#define SHIFT          (2 + sizeof( SHIFT_STRING ))
  for( i = 0; i < SHIFT-1; i++) line[i] = ' ';

  for( ;; )
  {
    empty = True;
    ptr = &line[SHIFT-1];
    size = sizeof( line ) - SHIFT;

    for( ;; )
    {
      for( i = 0; text[i] != '\0' && !isSpace( text[i] ); i++) continue;
      if( i > size )
      {
        if( empty )
        {
          strncmp( ptr, text, size);
          text += size;
        }
        break;
      }

      strncpy( ptr, text, i);
      ptr += i;
      text += i;
      size -= i;
      empty = False;

      while( isSpace( *text ) ) text++;
      if( *text != '\0' && size > 1 )
      {
	*ptr ++ = ' ';
        size--;
      }

      if( *text == '\0' || size == 0 ) break;
    }

    if( *(ptr - 1) == ' ' ) ptr--;
    *ptr = '\0';
    zprintf( cnt, "%s", line);

    if( *text != '\0' )
      zprintf( cnt, "\n");
    else
      break;
  }

  if( !all )
  {
    if( size >= 4 )
      zprintf( cnt, " ...");
    else
    {
      line[SHIFT-1] = line[SHIFT] = line[SHIFT+1] = '.';
      line[SHIFT+2] = '\0';
      zprintf( cnt, line);
    }
  }
  zprintf( cnt, "\n\n");
}

static Boolean printFileInfo( struct indexfile_t *pif, _fn_t filenum, Boolean all)
{
  struct zcontext_t *cnt = pif->context;
  struct flu_docentry_t docInfo;
  char st[MAX_STRUCTURE_STRING_SIZE], date[40];

  if( all ) zprintf( cnt, "%d:\n", filenum);

  if( !ifReadFileInfo( pif, filenum, &docInfo, True) )
  {
    zprintf( cnt, "\n");
    return False;
  }

  if( all )
  {
    struct tm tm;
    if( docInfo.lastModified != 0 )
      zTimeString( date, sizeof( date ), NULL, zMakeTime( docInfo.lastModified, &tm, 0), TIME_STRINGS);
    zprintf( cnt, "%-10s %s\n%-10s %s\n%-10s %" _ZOFF_FORMAT "d\n%-10s %s\n%-10s %s\n",
      "URL:", docInfo.url, "Title:", docInfo.title, "Size:", docInfo.size, "Date:",
      (docInfo.lastModified == 0) ? "???" : date,
      "Structure:", getStructureString( st, ifGetFileStructure( pif, filenum)));
    printText( cnt, docInfo.content, docInfo.contentAll);
  }
  else
    zprintf( cnt, "%s\n", docInfo.url);

  return True;
}

Boolean dumpFileIndex( struct indexfile_t *pif, Boolean all)
{
  _fn_t i;

  for( i = 1; i <= (_fn_t) pif->header.fileCount; i++)
    if( !printFileInfo( pif, i, all) ) return False;

  return True;
}

/***************************************************************************/
/*                                                                         */
/*  Find document                                                          */
/*                                                                         */
/***************************************************************************/

_fn_t findDocument( struct indexfile_t *pif, const char *docName)
{
  struct flu_docentry_t docInfo;
  _fn_t i;

  for( i = 1; i <= (_fn_t) pif->header.fileCount; i++)
  {
    if( !ifReadFileInfo( pif, i, &docInfo, False) ) return 0;
    if( strcmp( docName, docInfo.url) == 0 ) return i;
  }

  return 0;
}
