/*  options.c  */
/*  part of the fitsblink program  */
/*  routines for input of astrometry options */
/*  Jure Skvarc,  May 1998                   */
#include <stdio.h> 
#include <string.h>
#include <stdlib.h>
#include <forms.h>

#include "formblink.h"
#include "functs.h"

extern STATE state;
extern BLINK_FRAME *frame;


void 
logfileC(FL_OBJECT *obj, long val) 

{
  char *format = "Can't open file '%s!";
  char *mes;

  if (state.logfile && strlen(state.logfile) < strlen(fl_get_input(obj))) {
    free(state.logfile);
  }
  state.logfile = strdup((char *) fl_get_input(obj)); 
  trim_spaces(state.logfile);
  if (state.logfp != stdout) {
    fclose(state.logfp);
  }
  state.logfp = fopen(state.logfile, "a");
  if (state.logfp == NULL) {
    mes = (char *) myalloc(sizeof(char) * 
			   (strlen(state.logfile) + strlen(format) + 1),
			   "logfileC", "mes");
    sprintf(mes, format, state.logfile);
    fl_show_message(mes, "Output is redirected to stdout again.", "");
    fl_set_input(obj, "");
    free((void *) mes);
    state.logfp = stdout;
  }
}


/*  Write the minimum value into input field  */
void
set_minconst(float min)
{
  char temp[20];
  sprintf(temp, "%.1f", min);
  fl_set_input(state.options->minconstW, temp);
}


/*  Get a minimum value for a constellation to be accepted  */
void minconstC(FL_OBJECT *obj, long val) 

{
  float max;
  char t[80];
  int n = state.control[state.control_num];

  max = 0.5 * (frame[n].match.nstar + 1) * frame[n].match.nstar;
  frame[n].match.min = strtod(fl_get_input(obj), NULL);
  if (frame[n].match.min < 5.0) {
    sprintf(t, "I suggest that you enter a number between %.2f and %.2f", 
	    CONSTMIN, 0.5 * max);
    fl_show_message("Possible wrong input", 
		  "You can enter this number if you really want to\nbut the results will be rather senseless.", t);
  }
  if (frame[n].match.min >= max) {
    sprintf(t, "I suggest that you enter a number between %.2f and %.2f", 
	    CONSTMIN, 0.5 * max);
    fl_show_message("You will never get any match with this value",
		  "because it is too big.", t);
  }
  set_minconst(frame[n].match.min);
}


/*  Write the number of stars used for initial matching  into input field  */
/*-------------------------------------------------------------------------*/
void
set_initconst(int init)
{
  char temp[20];
  sprintf(temp, "%d", init);
  fl_set_input(state.options->initconstW, temp);
}

void initconstC(FL_OBJECT *obj, long val) 

{
  int n = state.control[state.control_num];

  frame[n].match.firstbright = strtol(fl_get_input(obj), NULL, 10);
  if (frame[n].match.firstbright < 10 || frame[n].match.firstbright > 100) {
    fl_show_message("You can enter this number if you really want to",
		  "but the results will be rather senseless.",
		  "Please use numbers between 10 and 100");
  }
  set_initconst(frame[n].match.firstbright);
}

/*  Write a number of stars in a constellation into input field  */
/*---------------------------------------------------------------*/
void
set_starconst(int nstar)
{
  char temp[20];
  sprintf(temp, "%d", nstar);
  fl_set_input(state.options->starconstW, temp);
}


void starconstC(FL_OBJECT *obj, long val)

{
  int n = state.control[state.control_num];

  frame[n].match.nstar = strtol(fl_get_input(obj), NULL, 10);
  if (frame[n].match.nstar < 6 || frame[n].match.nstar > 15) {
    fl_show_message("You can enter this number if you really want to",
		  "but the results will be rather senseless.",
		  "Please use numbers between 6 and 15");
  }
  set_starconst(frame[n].match.nstar);
}

/*  Write the maximal allowed positional error into input field */
/*--------------------------------------------------------------*/
void
set_starerror(float poserrmax)
{
  char temp[20];
  sprintf(temp, "%.1f", poserrmax);
  fl_set_input(state.options->starerrorW, temp);
}


void starerrorC(FL_OBJECT *obj, long val)

{
  int n = state.control[state.control_num];

  frame[n].match.poserrmax = strtod(fl_get_input(obj), NULL);
  if (frame[n].match.poserrmax < 1 || frame[n].match.poserrmax > 10) {
    fl_show_message("You can enter this number if you really want to",
		  "but the results will be rather senseless.",
		  "Please use numbers between 1 and 10");
  }
  set_starerror(frame[n].match.poserrmax);
}


/*  Write the maximal allowed positional error into input field */
/*--------------------------------------------------------------*/
void
set_maxres(float res)
{
  char temp[20];
  sprintf(temp, "%.1f", res);
  fl_set_input(state.options->maxresW, temp);
}


void maxresC(FL_OBJECT *obj, long val)

{
  int n = state.control[state.control_num];

  frame[n].match.maxres = strtod(fl_get_input(obj), NULL);
  if (frame[n].match.maxres < 0.5 || frame[n].match.maxres > 3) {
    fl_show_message("You can enter this number if you really want to",
		  "but the results will be rather senseless.",
		  "Please use numbers between 0.5 and 3");
  }
  set_maxres(frame[n].match.maxres);
}


/*  Write the minimum magnitude  */
/*-------------------------------*/
void
set_starminmag(float minmag)
{
  char temp[20];
  sprintf(temp, "%.1f", minmag);
  fl_set_input(state.options->starminmagW, temp);
}

void starminmagC(FL_OBJECT *obj, long val)
{
  int n = state.control[state.control_num];

  frame[n].map.minmag = strtod(fl_get_input(obj), NULL);
  if (frame[n].map.minmag >= frame[n].map.maxmag) {
    fl_show_message("The lowest magnitude should have lower",
		  "value than the highest magnitude.",
		  "Correct this, please!");
  }
  set_starminmag(frame[n].map.minmag);
}


/*  Write the maximum magnitude  */
/*-------------------------------*/
void
set_starmaxmag(float maxmag)
{
  char temp[20];
  sprintf(temp, "%.1f", maxmag);
  fl_set_input(state.options->starmaxmagW, temp);
}

void starmaxmagC(FL_OBJECT *obj, long val)

{
  int n = state.control[state.control_num];

  frame[n].map.maxmag = strtod(fl_get_input(obj), NULL);
  if (frame[n].map.minmag >= frame[n].map.maxmag) {
    fl_show_message("The lowest magnitude should have lower",
		  "value than the highest magnitude.",
		  "Correct this, please!");
  }
  set_starmaxmag(frame[n].map.maxmag);
}



/*  Write the sensitivity threshold  */
void
set_countsigma(float min)
{
  char temp[20];
  sprintf(temp, "%.1f", min);
  fl_set_input(state.options->countsigmaW, temp);
}


void
countsigmaC(FL_OBJECT *obj, long val)

{
  int n = state.control[state.control_num];

  frame[n].fits.sg_num = strtod(fl_get_input(obj), NULL);
  if (frame[n].fits.sg_num <= 0) {
    fl_show_message("Please type in a positive number!",
		  "Typical good values are between 2 and 5.",
		  "");
    frame[n].fits.sg_num = 0;
  }
  set_countsigma(frame[n].fits.sg_num);
}


/*  Write the minimal accepted object intensity  */
/*===============================================*/
void
set_countminval(int min)
{
  char temp[20];
  sprintf(temp, "%d", min);
  fl_set_input(state.options->countminvalW, temp);
}

/*  Input the minimal accepted intensity  */
/*========================================*/
void countminvalC(FL_OBJECT *obj, long val)

{
  int n = state.control[state.control_num];

  frame[n].fits.minbright = strtol(fl_get_input(obj), NULL, 10);
  if (frame[n].fits.minbright > 200) {
    fl_show_message("Just to let you know that using high",
		  "values of this parameter you may loose",
		  "a lot of good stars.");
  }
  if (frame[n].fits.minbright < 50 ) {
    fl_show_message("You may get a lot of false detections",
		  "if you use this value.",
		  "");
  }
  set_countminval(frame[n].fits.minbright);
}


/*  Write the minimal accepted star size  */
/*========================================*/
void
set_starminsize(int min)
{
  char temp[20];
  sprintf(temp, "%d", min);
  fl_set_input(state.options->starminsizeW, temp);
}


/*  Input the minimal accepted star size  */
/*========================================*/
void starminsizelC(FL_OBJECT *obj, long val)

{
  int n = state.control[state.control_num];

  frame[n].fits.minstar = strtol(fl_get_input(obj), NULL, 10);
  if (frame[n].fits.minstar > 2 || frame[n].fits.minstar < 1) {
    fl_show_message("You can experiment if you really want to,",
		  "but the suggested value is 1, or, ",
		  "if you want, 2.");
  }
  set_starminsize(frame[n].fits.minstar);
}

/*  Write the grid size  */
/*========================================*/
void
set_gridsize(int grid)
{
  char temp[20];
  sprintf(temp, "%d", grid);
  fl_set_input(state.options->stargridW, temp);
}

/*  Input the grid  size  */
/*========================================*/
void stargridC(FL_OBJECT *obj, long val)

{
  int n = state.control[state.control_num];

  frame[n].fits.maxsize = strtol(fl_get_input(obj), NULL, 10);
  if (frame[n].fits.maxsize > 100 || frame[n].fits.maxsize < 10) {
    fl_show_message("Please set this value somewhere,",
		  "between 10 and 100.",
		  "");
  }
  set_gridsize(frame[n].fits.maxsize);
}

/*  Write the insensitive edge size  */
/*========================================*/
void
set_edgesize(int edge)
{
  char temp[20];
  sprintf(temp, "%d", edge);
  fl_set_input(state.options->staredgeW, temp);
}


/*  Input the edge size  */
/*========================================*/
void staredgeC(FL_OBJECT *obj, long val)

{
  int n = state.control[state.control_num];

  frame[n].fits.border = strtol(fl_get_input(obj), NULL, 10);
  if (frame[n].fits.border > frame[n].fits.width / 2 || 
      frame[n].fits.border > frame[n].fits.height / 2) {
    fl_show_message("With this value you will get",
		  "0 detections.",
		  "");
  }
  set_edgesize(frame[n].fits.border);
}



/*  Set default values for matching stars  */
/*-----------------------------------------*/
void 
set_match_defaults(BLINK_FRAME *frame, int n)

{
  frame[n].match.min = CONSTMIN; 
  frame[n].match.nstar = CONSTSTAR; 
  frame[n].match.firstbright = FIRSTBRIGHT; 
  frame[n].match.poserrmax = POSERRMAX; 
  frame[n].match.magerrmax = MAGERRMAX; 
  frame[n].match.maxres = MAXRES; 
  frame[n].map.minmag = DEFAULT_MINMAG; 
  frame[n].map.maxmag = DEFAULT_MAXMAG;
  frame[n].fits.sg_num = DEFAULT_SGNUM;
  frame[n].fits.minbright = DEFAULT_MINBRIGHT;
  frame[n].fits.minstar = DEFAULT_MINSTAR;
  frame[n].fits.maxsize = DEFAULT_MAXSIZE;
  frame[n].fits.border = DEFAULT_BORDER;
}



void defaultoptionsC(FL_OBJECT *obj, long val)

{
  int n = state.control[state.control_num];
  set_match_defaults(frame, n);
  set_minconst(frame[n].match.min);
  set_initconst(frame[n].match.firstbright);
  set_starconst(frame[n].match.nstar);
  set_starerror(frame[n].match.poserrmax);
  set_maxres(frame[n].match.maxres);
  set_starminmag(frame[n].map.minmag);
  set_starmaxmag(frame[n].map.maxmag);
  set_starminsize(frame[n].fits.minstar);
  set_countsigma(frame[n].fits.sg_num);
  set_countminval(frame[n].fits.minbright);
  set_gridsize(frame[n].fits.maxsize);
  set_edgesize(frame[n].fits.border);
}


void
edit_options(FL_OBJECT *obj, int m)

{
  int n = state.control[state.control_num];
  /*  The window in not created yet, do it now  */
  if (state.options == NULL) {
    state.options = create_form_Options();
    fl_set_form_minsize(state.options->Options, 
		  state.options->Options->w, state.options->Options->h);
    set_match_defaults(frame, n);
  }
  /*  Deactivate OPTIONS item in the menu */
  fl_set_menu_item_mode(obj, m, FL_PUP_GREY);
  fl_show_form(state.options->Options,  FL_FIX_SIZE, FL_FULLBORDER | FL_PLACE_FREE_CENTER, "Image options");
  set_minconst(frame[n].match.min);
  set_initconst(frame[n].match.firstbright);
  set_starconst(frame[n].match.nstar);
  set_starerror(frame[n].match.poserrmax);
  set_maxres(frame[n].match.maxres);
  set_starminmag(frame[n].map.minmag);
  set_starmaxmag(frame[n].map.maxmag);
  fl_set_input(state.options->logfileW, state.logfile);
  set_starminsize(frame[n].fits.minstar);
  set_gridsize(frame[n].fits.maxsize); 
  set_edgesize(frame[n].fits.border); 
  set_countsigma(frame[n].fits.sg_num);
  set_countminval(frame[n].fits.minbright);
}

