/***********************************************************************
 * rexxcli.c - REXX/SQL for X/Open CLI and ODBC
 ***********************************************************************
 *
 * REXX/SQL. A REXX interface to SQL databases.
 * Copyright Mark Hessling, 1996-1997.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to 
 * address below.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                    Email:       M.Hessling@qut.edu.au
 * PO Box 203                       Phone:              +617 3802 0800
 * Bellara                          http://www.lightlink.com/hessling/
 * QLD 4507                         **** Author of THE & Rexx/SQL ****
 * Australia                        ****** Maintainer PDCurses *******
 *
 */

#include "dbdefine.h"
#if defined(SOLID_SERVER) && !defined(__OS2__)
# include "dbheader.h"
#endif
#if defined(UNIX)
# include "dbheader.h"
#endif
#include "rexxsql.h"
#include "dbheader.h"
#include "dbtypes.h"

/*
 * Definition of field descriptions
 */
typedef struct 
{
 UCHAR  cbuf[MAX_NAMELEN+1];             /* Buffer to hold column name */
 SWORD  cbufl;                                /* Length of column name */
 SWORD  dbtype;                                  /* Datatype of column */
 UDWORD prec;                                   /* Precision of column */
 SWORD  scale;                             /* Scale of number datatype */
 SWORD  nullok;                  /* Are nulls permitted? 0=no, <>0=yes */
 UCHAR  *rbuf;       /* Pointer to buffer to hold fetched column value */
 SDWORD rbufl;          /* Size of buffer to hold fetched column value */
 SDWORD retl;                                /* Return length of field */
 int    rcode;          /* Column return code - valid only after fetch */
 int    indp;                /* Indicator for field - after each fetch */
                   /* <0 - NULL; 0 - OK; >0 - length BEFORE truncation */
} FLDDSCR;

/*
 * Definition of parameter markers
 */
typedef struct 
{
 UCHAR  *buf;   /* Pointer to buffer containing parameter marker value */
 SDWORD buflen;                                /* Actual length of buf */
 SDWORD retlen;                         /* Return length of bind value */
} PMDSCR;

/*
 * Definition of SQL Work Area (SQLWA).
 */

typedef struct 
{
 char    *sql_stmt;             /* ptr to buffer to hold SQL statement */
 int     select;            /* 1 if statement is a SELECT, 0 otherwise */
 int     sql_stmt_sz;            /* size of allocated statement buffer */
 int     bind_cnt;                      /* number of bind values bound */
 SWORD   expr_cnt;             /* number of expressions in select list */
 FLDDSCR *fa[MAX_COLS];          /* array of pointers to column dscr's */
 PMDSCR  *pm[MAX_BINDVARS];      /* array of pointers to param markers */
} SQLWA;


/* 
 * Connection structure. Each connection requires one of these! 
 * NB! The field; name, MUST be the first entry in the structure.
 */
typedef struct 
{
 char       name[MAX_IDENTIFIER+1];                 /* Connection name */
 RXSQL_HDBC hdbc;                                 /* connection handle */
 void       *dflt_stmt;               /* Pointer to implicit statement */
 int        num_statements;           /* Number of explicit statements */
 SQLVAL    *db_opt;                           /* SqlGetInfo hash table */
} DBCON;

/* 
 * Statement structure. Each statement requires one of these!
 * NB! The field; name, MUST be the first entry in the structure.
 */
typedef struct 
{
 char        name[MAX_DB_CURSOR_NAME+1];             /* Statement name */
 RXSQL_HSTMT hstmt;                                /* statement handle */
 DBCON       *db;                   /* Pointer to connection structure */
 SQLWA       sqlwa;                                   /* SQL Work Area */
 ULONG       Argc;
 RXSTRING    *Argv;
} STMT;


/* Basic REXXSQL environment structure - allocated when 1st connection made! */
typedef struct 
{
 int     num_connections;              /* Number of active connections */
 int     max_connections;     /* Maximum number of connections allowed */
 DBCON   *current_connection;         /* Pointer to current connection */
 BUCKET  *db_tbl[TBL_CONNECTIONS];            /* Connection hash table */
 BUCKET  *stmt_tbl[TBL_STATEMENTS];            /* Statement hash table */
} DBENV;


char *column_attribute[] =
{   
 "NAME", "TYPE", "SIZE", "PRECISION", "SCALE", "NULLABLE", (char *)NULL
};

/*
 * ReadOnly variables
 *
 * ---------------------------------------------------------------------
 * Current version/platform
 */
static RXSTRING RexxSqlVersion;
static char RexxSqlVersionString[100];

/* ---------------------------------------------------------------------
 * Error value returned for errors returned by database as opposed to
 * those generated by the interface.
 */
#define DB_ERROR	(-1L)

/* Debugging flags */
int run_flags = 0;          /* can also be set via SQLVariable() */

/* Global Function Name - set on entry to every REXX/SQL function */
char FName[40];

extern ULONG RowLimit;
extern ULONG LongLimit;
extern ULONG SaveSQL;
extern ULONG AutoCommit;
extern ULONG IgnoreTruncate;
extern ULONG StandardPlacemarkers;
extern ULONG SupportsTransactions;
extern ULONG SupportsPlacemarkers;
extern ULONG SupportsDMLRowcount;
extern ULONG SupportsSqlGetData;
extern RXSTRING RXSNullStringOut;
extern char NullStringOut[MAX_IDENTIFIER+1];
extern RXSTRING RXSNullStringIn;
extern char NullStringIn[MAX_IDENTIFIER+1];

RXSTRING RXSDBMSName;
char DBMSName[512+1];

RXSTRING RXSDBMSVer;
char DBMSVer[50+1];

SQLVAL *opt_tbl = NULL;

#ifdef HAVE_PROTO
static void *SetDescribeColumns(DBCON *);
static void *SetDatatypes(DBCON *);
static void *SetVersion(void *);
static void *SetNullStringIn(void *);
static void *SetNullStringOut(void *);
static void *SetDBMSName(DBCON *);
static void *SetDBMSVer(DBCON *);
static void *SetSupportsSqlGetData(DBCON *);
static void *SetSupportsTransactions(DBCON *);
static int ValidateDataType(char *, int, DBCON *);
#else
static void *SetDescribeColumns();
static void *SetDatatypes();
static void *SetVersion();
static void *SetNullStringIn();
static void *SetNullStringOut();
static void *SetDBMSName();
static void *SetDBMSVer();
static void *SetSupportsSqlGetData();
static void *SetSupportsTransactions();
static int ValidateDataType();
#endif

/* ---------------------------------------------------------------------
 * The following table defines those values that can be used with the
 * SQLGetInfo() and SQLSetInfo() functions.
 */
/* name            settable  type         dyn/conn reqd                     maxlen  value func */
SQLVAL sqlvariable_vars[] =
{
 {NULL,NULL,"DEBUG",                1, TYPE_INT,     OPT_STATIC ,0, (void*)&run_flags,0},
 {NULL,NULL,"ROWLIMIT",             1, TYPE_INT,     OPT_STATIC ,0, (void*)&RowLimit,0},
 {NULL,NULL,"LONGLIMIT",            1, TYPE_INT,     OPT_STATIC ,0, (void*)&LongLimit,0},
 {NULL,NULL,"SAVESQL",              1, TYPE_BOOL,    OPT_STATIC ,0, (void*)&SaveSQL,0},
 {NULL,NULL,"AUTOCOMMIT",           1, TYPE_BOOL,    OPT_STATIC ,0, (void*)&AutoCommit,0},
 {NULL,NULL,"IGNORETRUNCATE",       1, TYPE_BOOL,    OPT_STATIC ,0, (void*)&IgnoreTruncate,0},
 {NULL,NULL,"NULLSTRINGIN",         1, TYPE_STRING,  OPT_STATIC ,0, NULL,   SQLVAL_NULLSTRINGIN},
 {NULL,NULL,"NULLSTRINGOUT",        1, TYPE_STRING,  OPT_STATIC ,0, NULL,   SQLVAL_NULLSTRINGOUT},
 {NULL,NULL,"STANDARDPLACEMARKERS", 1, TYPE_BOOL,    OPT_STATIC ,0, (void*)&StandardPlacemarkers,0},
 {NULL,NULL,"SUPPORTSDMLROWCOUNT",  0, TYPE_BOOL,    OPT_STATIC ,0, (void*)&SupportsDMLRowcount,0},
 {NULL,NULL,"SUPPORTSPLACEMARKERS", 0, TYPE_BOOL,    OPT_STATIC ,0, (void*)&SupportsPlacemarkers,0},
 {NULL,NULL,"VERSION",              0, TYPE_STRING,  OPT_STATIC ,0, NULL,   SQLVAL_VERSION},
 {NULL,NULL,"",                     0, 0,            0          ,0, NULL,0},
};

SQLVAL sqlgetinfo_vars[] =
{
 {NULL,NULL,"SUPPORTSTRANSACTIONS", 0, TYPE_BOOL,    OPT_STATIC ,0, NULL,   SQLVAL_SUPPORTSTRANSACTIONS},
 {NULL,NULL,"SUPPORTSSQLGETDATA",   0, TYPE_BOOL,    OPT_STATIC ,0, NULL,   SQLVAL_SUPPORTSSQLGETDATA},
 {NULL,NULL,"DESCRIBECOLUMNS",      0, TYPE_ARRAY,   OPT_DYNAMIC,0, NULL,   SQLVAL_DESCRIBECOLUMNS},
 {NULL,NULL,"DATATYPES",            0, TYPE_DATATYPE,OPT_DYNAMIC,0, NULL,   SQLVAL_DATATYPES},
 {NULL,NULL,"DBMSNAME",             0, TYPE_STRING,  OPT_STATIC ,0, NULL,   SQLVAL_DBMSNAME},
 {NULL,NULL,"DBMSVERSION",          0, TYPE_STRING,  OPT_STATIC ,0, NULL,   SQLVAL_DBMSVERSION},
 {NULL,NULL,"",                     0, 0,            0          ,0, NULL,   0},
};
               
/* ---------------------------------------------------------------------
 * Pointer to REXXSQL environment - initially unallocated!
 * Only one object of this type is required and this object is created
 * the first time a connection is made. This object is released whenever
 * the last connection is released or upon exit.
 */
static DBENV *DbEnv = (DBENV*)NULL;
               
/* ---------------------------------------------------------------------
 * These global variables hold the result of the last function call.
 */
RXSQL_SQLCODE_TYPE  SQLCA_SqlCode  = -1L;	/* Force a clear on startup */
RXSQL_ROWCOUNT_TYPE SQLCA_RowCount = -1L;	/* Force a clear on startup */
long SQLCA_IntCode                 = -1L;	/* Force a clear on startup */
RXSQL_SQLSTATE_TYPE SQLCA_SqlState[SQL_SQLSTATE_SIZE+1]; /* contains last SQLSTATE */

/* ---------------------------------------------------------------------
 * Handle for an SQL environment. There must be only one of these.
 */
RXSQL_HENV henv = 0L;


/*-----------------------------------------------------------------------------
 * Fetch the DB error and put into REXX "SQLCA." compound variable.
 *
 * Really should try to set multiple errors.
 *----------------------------------------------------------------------------*/
static void SetDBError

#ifdef HAVE_PROTO
    (RXSQL_HENV henvir, RXSQL_HDBC hdbc, RXSQL_HSTMT hstmt, SQLWA *swa)
#else
    (henvir, hdbc, hstmt, swa)
    RXSQL_HENV henvir;
    RXSQL_HDBC hdbc;
    RXSQL_HSTMT hstmt;
    SQLWA *swa;
#endif

{
 SDWORD    sqlcode=0L;
 char      *txt=NULL;
 UCHAR     msg[SQL_MAX_MESSAGE_LENGTH+1];
 SWORD     length=0;

 InternalFunctionPrologue("SetDBError");

 /* 
  * Get the Database error code and message 
  */
 SQLError(henvir, hdbc, hstmt, SQLCA_SqlState, &sqlcode, msg, (SWORD)sizeof(msg), &length);

 /* 
  * Get the statement text 
  */
 txt = (swa == (SQLWA*)NULL) ? "" : (swa->sql_stmt) ? swa->sql_stmt : "";

 /* 
  * Set SQLCA. variable 
  */
 SetSQLCA(sqlcode, SQLCA_SqlState, msg, txt);

 return;
}

/*-----------------------------------------------------------------------------
 * Allocate a DBENV object. Only one database environment is allocated.
 * This structure is allocated when the first connection is made and deallocated
 * when the last connection is released.
 * Note that statement names are global and hence are defined in this structure
 * and not in the connection structure as would be the case if they were unique
 * to a connection!
 *----------------------------------------------------------------------------*/
static DBENV *AllocDbEnvironment

#ifdef HAVE_PROTO
    (void)
#else
    ()
#endif

{
 DBENV  *dbenv=NULL;
 int    i=0;

 InternalFunctionPrologue("AllocDbEnvironment");

 if ((dbenv = (DBENV*)malloc(sizeof(DBENV))) == (DBENV*)NULL)
   {
    SetIntError(10,"out of memory", __FILE__,__LINE__);
    return (DBENV*)NULL;
   }
 dbenv->num_connections = 0;
 dbenv->max_connections = 0;
 dbenv->current_connection = (DBCON*)NULL;

 if (SQLAllocEnv(&henv) == SQL_ERROR)
   {
    if (henv == SQL_NULL_HENV)
       SetIntError(8,"unable to allocate environment", __FILE__,__LINE__);
    else
      {
       SetDBError(henv,SQL_NULL_HDBC,SQL_NULL_HSTMT,NULL);
       SQLFreeEnv(henv);
      }
    free(dbenv);
    return(DBENV*)NULL;
   }
 for (i = 0; i < TBL_CONNECTIONS; i++)
     dbenv->db_tbl[i] = (BUCKET*)NULL;
 for (i = 0; i < TBL_STATEMENTS; i++)
     dbenv->stmt_tbl[i] = (BUCKET*)NULL;
 return dbenv;
}

/*-----------------------------------------------------------------------------
 * Allocate a DBCON object. One of these is required for each database
 * connection. This structure is allocated when the connection is made and
 * deallocated when the connection is released.
 * Check to make sure we are not attempting to allocate more
 * connections than we are allowed.
 *----------------------------------------------------------------------------*/
static DBCON *AllocConnection

#ifdef HAVE_PROTO
    (char *name)
#else
    (name)
    char *name;
#endif

{
 DBCON       *db=NULL;
 char        tmp[61];
 SWORD cons=0,conslen=0;
 int    i=0;

 /*
  * If the number of connections is unlimited or this is the
  * first connection (dbenv->max_connections == 0) for both
  * situations, then continue...
  * ...else check that this connection does not exceed the
  * maximum number of connections.
  */

 InternalFunctionPrologue("AllocConnection");

 if (DbEnv->max_connections != 0
 &&  DbEnv->num_connections == DbEnv->max_connections)
   {
    sprintf(tmp,"reached maximum number of connections: %d",
                DbEnv->max_connections);
    SetIntError(27,tmp, __FILE__,__LINE__);
    return (DBCON*)NULL;
   }
 if ((db = (DBCON*)NewObject(sizeof(DBCON))) == (DBCON*)NULL)
   {
    SetIntError(10,"out of memory", __FILE__,__LINE__);
    return (DBCON*)NULL;
   }
 (void)strncpy(db->name, name, MAX_IDENTIFIER);
 db->dflt_stmt = (void*)NULL;
 db->db_opt = (SQLVAL*)NULL;
 db->num_statements = 0;
 if (SQLAllocConnect(henv,&(db->hdbc)) != SQL_SUCCESS)
   {
    SetDBError(henv,SQL_NULL_HDBC,SQL_NULL_HSTMT,NULL);
    free(db);
    return(DBCON*)NULL;
   }
 /*
  * Now we have a valid connection, see how many we are allowed
  */
#if 0
 SQLGetInfo(db->hdbc,SQL_ACTIVE_CONNECTIONS,&cons,sizeof(cons),&conslen);
 DbEnv->max_connections = cons;
#else
 DbEnv->max_connections = 0;
#endif
 return db;
}

/*-----------------------------------------------------------------------------
 * Open a database connection. This requires allocating a connection object
 * and making the connection to Database.
 * Check to make sure we are allowed more than 1 connection.
 *----------------------------------------------------------------------------*/
static int OpenConnection

#ifdef HAVE_PROTO
    (PSZ name, UCHAR *uid, SWORD uidlen, UCHAR *pwd, SWORD pwdlen, UCHAR *srv, SWORD srvlen, UCHAR *conn, SWORD connlen, DBCON **new_db)
#else
    (name, uid, uidlen, pwd, pwdlen, srv, srvlen, conn, connlen, new_db)
    PSZ name;
    UCHAR *uid;
    SWORD uidlen;
    UCHAR *pwd;
    SWORD pwdlen;
    UCHAR *srv;
    SWORD srvlen;
    UCHAR *conn;
    SWORD connlen;
    DBCON **new_db;
#endif

{
#define RET_BUFFER_LEN 1024
 DBCON   *db=NULL;
 UCHAR   buffer[RET_BUFFER_LEN+1];/* buffer for complete connect string */
 SWORD   buflen=0;
 UWORD   funcsupport=FALSE;
 int     rc=0,i=0;
 ULONG   warning=0;
 SQLVAL  *curr=NULL;
 void    *result=NULL;

 InternalFunctionPrologue("OpenConnection");

 if ((db = AllocConnection(name)) == (DBCON*)NULL)
    return (SQLCA_IntCode);

 /*
  * If we have an optional connect string call SQLDriverConnect()
  * rather than SQLConnect() This is not available under OS/2, Unix
  * with Solid Server.
  */
 if (connlen)
   {
#if defined(REXXSQL_SS_NO_SQLDRIVERCONNECT)
    FreeObject(db);
    return DB_ERROR;
#else
# if defined(UDBC_ODBC)
    rc = SQLDriverConnect(db->hdbc,(HWND)NULL,conn,connlen,buffer,RET_BUFFER_LEN,&buflen,SQL_DRIVER_COMPLETE);
# else
    rc = SQLDriverConnect(db->hdbc,(HWND)NULL,conn,connlen,buffer,RET_BUFFER_LEN,&buflen,SQL_DRIVER_NOPROMPT);
# endif
#endif
   }
 else
    rc = SQLConnect(db->hdbc,srv,srvlen,uid,uidlen,pwd,pwdlen);
 if (rc != SQL_SUCCESS
 &&  rc != SQL_SUCCESS_WITH_INFO)
   {
    SetDBError(SQL_NULL_HENV, db->hdbc, SQL_NULL_HSTMT,NULL);
    FreeObject(db);
    return DB_ERROR;
   }
 /*
  * Check if SQLConnectOption() function is valid...
  * Not available for Solid Server..grrr!
  */
#if !defined(REXXSQL_SS_NO_SQLGETFUNCTIONS)
 if (SQLGetFunctions(db->hdbc, SQL_API_SQLSETCONNECTOPTION, &funcsupport) != SQL_SUCCESS)
   {
    SetDBError(SQL_NULL_HENV, db->hdbc, SQL_NULL_HSTMT,NULL);
    FreeObject(db);
    return DB_ERROR;
   }
#else
 funcsupport = 1;  /* force this to TRUE for Solid Server */
#endif

 /*
  * Force the status of AUTOCOMMIT to OFF
  * We will do our own (more flexible) autocommit
  */
 if (funcsupport)
   {
    rc = SQLSetConnectOption(db->hdbc, SQL_AUTOCOMMIT, SQL_AUTOCOMMIT_OFF);
    if (rc != SQL_SUCCESS)
      {
       SetDBError(SQL_NULL_HENV, db->hdbc, SQL_NULL_HSTMT,NULL);
       if (strcmp(SQLCA_SqlState,"S1C00") == 0)
         {
          warning = DB_WARNING;
         }
       else
         {
          FreeObject(db);
          return DB_ERROR;
         }
      }
   }

 /*
  * Setup any connection-specific variable information
  */
 for (i=0;strlen(sqlgetinfo_vars[i].name)!=0;i++)
   {
    if (sqlgetinfo_vars[i].value)
       result = sqlgetinfo_vars[i].value;
    else
       result = GetSqlVariableValue(sqlgetinfo_vars[i].valtype,(void*)db);
    if ((curr = InstallSQLVariable(
                &rc,
                sqlgetinfo_vars[i].name,
                db->db_opt,
                sqlgetinfo_vars[i].user_update, 
                sqlgetinfo_vars[i].dtype, 
                0, 
                sqlgetinfo_vars[i].option,
                result)) == NULL)
      {
       return(rc);
      }
    if (db->db_opt == NULL)
       db->db_opt = curr;
   }
 *new_db = db;
 return warning;
}

/*-----------------------------------------------------------------------------
 * Allocate a STMT object. One of these is required for each statement
 * including the default statement for a connection. An instance of this object
 * is allocated when (i) a statement is prepared & (ii) the first time
 * SQLCOMMAND() is called for a connection (ie. the default statement is used).
 * Set the cursor name equal to the statment name so that "UPDATE WHERE
 * CURRENT OF cursor" commands can refer to the cursor.
 *----------------------------------------------------------------------------*/
static STMT *AllocStatement

#ifdef HAVE_PROTO
    (char *name, DBCON *db)
#else
    (name, db)
    char  *name;
    DBCON *db;
#endif

{
 STMT    *stmt=NULL;
 SQLWA   *swa=NULL;
 int     i=0;
 InternalFunctionPrologue("AllocStatement");


 if ((stmt = (STMT*)NewObject(sizeof(STMT))) == (STMT*)NULL)
    return (STMT*)NULL;
 stmt->db = db;
 (void)strncpy(stmt->name, name, MAX_DB_CURSOR_NAME);

 /* 
  * Initialise SQL Work Area 
  */
 swa = SWA(stmt);
 swa->bind_cnt = 0;
 swa->expr_cnt = 0;
 swa->sql_stmt = (char*)NULL;
 swa->sql_stmt_sz = 0;

 /* 
  * Set pointer for each column descriptor to NULL (unused) 
  */
 for (i = 0; i < MAX_COLS; i++)
    swa->fa[i] = (FLDDSCR*)NULL;

 /* 
  * Create bind value arrays (empty)
  */
 for (i = 0; i < MAX_BINDVARS; i++)
    swa->pm[i] = (PMDSCR*)NULL;

 /* 
  * Allocate a statement handle 
  */
 if (SQLAllocStmt(db->hdbc, &stmt->hstmt) != SQL_SUCCESS)
   {
    SetDBError(SQL_NULL_HENV, db->hdbc, SQL_NULL_HSTMT,NULL);
    FreeObject(stmt);
    stmt = (STMT*)NULL;
   }

 /* 
  * Set the cursor name for the statement 
  */
 if (SQLSetCursorName(stmt->hstmt,(UCHAR*)name,SQL_NTS) != SQL_SUCCESS)
   {
    SetDBError(SQL_NULL_HENV, SQL_NULL_HDBC, stmt->hstmt,NULL);
    FreeObject(stmt);
    stmt = (STMT*)NULL;
   }
 return stmt;
}

/*-----------------------------------------------------------------------------
 * Open a statement. This allocates a statement.
 *----------------------------------------------------------------------------*/
static int OpenStatement

#ifdef HAVE_PROTO
    (char *name, DBCON *db, STMT **new_stmt)
#else
    (name, db, new_stmt)
    char  *name;
    DBCON *db;
    STMT  **new_stmt;
#endif

{
 int     rc=0;
 STMT    *stmt=NULL;

 InternalFunctionPrologue("OpenStatement");
    
 if ((stmt = AllocStatement(name, db)) == (STMT*)NULL)
    return(SetIntError(10, "out of memory", __FILE__,__LINE__)); /* ??? */

 *new_stmt = stmt;
 return (rc==0) ? 0 : DB_ERROR;
}

/*-----------------------------------------------------------------------------
 * Disposes a SQL statement. This frees the statement.
 *----------------------------------------------------------------------------*/
static int ReleaseStatement

#ifdef HAVE_PROTO
    (STMT *stmt)
#else
    (stmt)
    STMT *stmt;
#endif

{
 SQLWA    *swa = SWA(stmt);
 FLDDSCR  *fd=NULL;
 PMDSCR   *pmd=NULL;
 int      rc=0, i=0;

 InternalFunctionPrologue("ReleaseStatement");

 stmt->Argv = FreeArgv(&stmt->Argc, stmt->Argv);

 /* 
  * Close any open cursor...
  */
 if (SQLFreeStmt(stmt->hstmt,SQL_CLOSE) != SQL_SUCCESS)
   {
    SetDBError(SQL_NULL_HENV,stmt->db->hdbc,SQL_NULL_HSTMT,swa);
    rc = DB_ERROR;
   }
 /* 
  * Release the statement 
  */
 if (SQLFreeStmt(stmt->hstmt,SQL_DROP) != SQL_SUCCESS)
   {
    SetDBError(SQL_NULL_HENV,stmt->db->hdbc,SQL_NULL_HSTMT,swa);
    rc = DB_ERROR;
   }

 /* 
  * Free up select column descriptors 
  */
 for (i = 0; i < MAX_COLS; i++) 
   {
    fd = swa->fa[i];
    if (fd == NULL)
       break;
    if (fd->rbuf)
       free(fd->rbuf);
    free(fd);
   }

 /* 
  * Free up parameter marker descriptors 
  */
 for (i = 0; i < MAX_BINDVARS; i++)
   {
    pmd = swa->pm[i];
    if (pmd == NULL)
       break;
    if (pmd->buf)
       free(pmd->buf);
    free(pmd);
   }
    
 /* 
  * Free sql statement buffer (if any) 
  */
 if (swa->sql_stmt)
    free(swa->sql_stmt);

 FreeObject(stmt);

 return (rc);
}

/*-----------------------------------------------------------------------------
 * Release a connection. This closes any open statements associated
 * with the connection, closes the connection and frees all resources
 * associated with it.
 * Any open transaction is comitted.
 *----------------------------------------------------------------------------*/
static int ReleaseConnection

#ifdef HAVE_PROTO
    (DBCON *db)
#else
    (db)
    DBCON *db;
#endif

{
 int    i=0, rc=0, last_error=0;
 STMT   *stmt=NULL, *t=NULL;

 InternalFunctionPrologue("ReleaseConnection");

 /* 
  * Remove the Connection structure from the hash structures etc. 
  */
 (void)RemoveObject(db);

 /* 
  * Decrement the count of active connections. 
  */
 DbEnv->num_connections--;

 /* 
  * Dispose all active statements for this connection. 
  */
 for (i = 0; db->num_statements && i < TBL_STATEMENTS; i++) 
   {
    stmt = (STMT*)FirstObject(i, DbEnv->stmt_tbl);
    while (stmt && db->num_statements) 
      {
       t = (STMT*)NextObject(stmt);
       if (stmt->db == db) 
         {
          RemoveObject(stmt);
          last_error = (rc = ReleaseStatement(stmt)) ? rc : last_error;
          db->num_statements--;
         }
       stmt = t;
      }
   }

 /* 
  * Dispose all informational data for this connection.
  */
 ll_free(db->db_opt);

 /* 
  * Dispose the default statement (if any). 
  */
 if (db->dflt_stmt)
    last_error = (rc = ReleaseStatement((STMT*)(db->dflt_stmt))) ? rc : last_error;

#if defined(HAVE_SQLENDTRAN)
 /* 
  * End all transactions
  */
 if (SQLEndTran(SQL_HANDLE_DBC,db->hdbc,SQL_COMMIT) == SQL_ERROR)
   {
    if (!last_error)
      {
       SetDBError(SQL_NULL_HENV, db->hdbc,SQL_NULL_HSTMT,NULL);
       last_error = SQLCA_SqlCode;
      }
   }
#else
 /* 
  * Commit the current transaction (if there is one)
  */
 if (SQLTransact(henv, db->hdbc,SQL_COMMIT) == SQL_ERROR)
   {
    if (!last_error)
      {
       SetDBError(SQL_NULL_HENV, db->hdbc, SQL_NULL_HSTMT, NULL);
       last_error = SQLCA_SqlCode;
      }
   }
#endif

 /* 
  * Disconnect from DB 
  */
 if (SQLDisconnect(db->hdbc) == SQL_ERROR)
   {
    if (!last_error)
      {
       SetDBError(SQL_NULL_HENV, db->hdbc,SQL_NULL_HSTMT,NULL);
       last_error = SQLCA_SqlCode;
      }
   }

 /* 
  * Release the connection 
  */
 if (SQLFreeConnect(db->hdbc) == SQL_ERROR)
   {
    if (!last_error)
      {
       SetDBError(SQL_NULL_HENV, db->hdbc,SQL_NULL_HSTMT,NULL);
       last_error = SQLCA_SqlCode;
      }
   }

 /* 
  * Free the connection structure 
  */
 FreeObject(db);

 return (last_error);
}

/*-----------------------------------------------------------------------------
 * Release the database environment. This releases all active connections
 * (if any).
 *----------------------------------------------------------------------------*/
int ReleaseDbEnvironment

#ifdef HAVE_PROTO
    (void)
#else
    ()
#endif

{
 int    i=0, rc=0, last_error=0;
 DBCON  *db=NULL, *t=NULL;

 InternalFunctionPrologue("ReleaseDbEnvironment");

 /* 
  * Ensure there is an environment to release! 
  */
 if (DbEnv == (DBENV*)NULL)
    return 0;

 /* 
  * Release all connections. 
  */
 for (i = 0; DbEnv->num_connections && i < TBL_CONNECTIONS; i++) 
   {
    db = (DBCON*)FirstObject(i, DbEnv->db_tbl);
    while (db && DbEnv->num_connections) 
      {
       t = (DBCON*)NextObject(db);
       last_error = (rc = ReleaseConnection(db)) ? rc : last_error;
       db = t;
      }
   }

 /* 
  * Release the environment 
  */
 if (SQLFreeEnv(henv) == SQL_ERROR)
   {
    SetDBError(henv, SQL_NULL_HDBC,SQL_NULL_HSTMT,NULL);
    last_error = SQLCA_SqlCode;
   }

 /* 
  * Free the DB environment 
  */
 free(DbEnv);
 DbEnv = (DBENV*)NULL;

 return (last_error);
}

/*-----------------------------------------------------------------------------
 * SYNOPSIS:  SQLCONNECT( [connection-name, ] [username], [password], <database source>, [connect string])
 *
 * ARGUMENTS: 
 *            0 - connection_name (optional)
 *            1 - username (optional)
 *            2 - password (optional)
 *            3 - database source (mandatory) - database name in DB2/2
 *                                            - data source name in ODBC
 *            4 - ODBC connect string (optional)
 *                if arg 4 supplied all other arguments MUST be empty
 *                (except connection_name)
 *
 * RETURNS :  0-success, <0-error.
 *----------------------------------------------------------------------------*/
ULONG RXSQL_APIENTRY SQLCONNECT

#ifdef HAVE_PROTO
    (RXSQL_PUCHAR		name,ULONG	argc,RXSTRING	argv[],RXSQL_PCSZ		stck,RXSTRING	*retstr)
#else
    (name, argc, argv, stck, retstr)
    RXSQL_PUCHAR		name;
    ULONG	argc;
    RXSTRING	argv[];
    RXSQL_PCSZ		stck;
    RXSTRING	*retstr;
#endif
{
 UCHAR uid[MAX_IDENTIFIER];
 SWORD uidlen=0;
 UCHAR pwd[MAX_IDENTIFIER];
 SWORD pwdlen=0;
 UCHAR srv_dsn[MAX_IDENTIFIER+1];
 SWORD srvlen=0;
 UCHAR *conn=NULL;
 SWORD connlen=0;
 char  dbname[MAX_IDENTIFIER+1];
 int   i=0, rc=0;
 char  tmp[128];
 DBCON *db=NULL;
 char  dsn_supplied=0;

 FunctionPrologue(name, argc, argv);

 if (SQLCA_SqlCode) ClearDBError();
 if (SQLCA_IntCode) ClearIntError();

 if (argc > 5) return 1;
 if (argc < 4) return 1;

 /*
  * If there is an optional connect string, ensure that none of
  * the other data arguments are supplied.
  */
 if (argc == 5
 &&  RXSTRLEN(argv[4]))
   {
    dsn_supplied = 1;
    if (RXSTRLEN(argv[1])
    ||  RXSTRLEN(argv[2])
    ||  RXSTRLEN(argv[3]))
      return ReturnError(retstr, 76, "<connect string> must be only argument");
   }

 if (RXSTRLEN(argv[1]))                            /* a userid passed */
    RXSTRCPY(uid,uidlen,RXSTRPTR(argv[1]),(SWORD)RXSTRLEN(argv[1]));
 else
    RXSTRCPY(uid,uidlen,"",0);

 if (RXSTRLEN(argv[2]))                          /* a password passed */
    RXSTRCPY(pwd,pwdlen,RXSTRPTR(argv[2]),(SWORD)RXSTRLEN(argv[2]));
 else
    RXSTRCPY(pwd,pwdlen,"",0);

 if (RXSTRLEN(argv[3]))                /* a DSN or server name passed */
    RXSTRCPY(srv_dsn,srvlen,RXSTRPTR(argv[3]),(SWORD)RXSTRLEN(argv[3]));
 else
#if defined(SOLID_SERVER)
    /*
     * Solid Server allows an empty DSN. This uses the default DSN
     */
    RXSTRCPY(srv_dsn,srvlen,"",0);
#else
    if (!dsn_supplied)
       return ReturnError(retstr, 75, "no database name supplied");
#endif

 /*
  * Get the name of the connection (default if not specified).
  */
 if (RXSTRLEN(argv[0])) 
   {
    if ((rc = MkIdentifier(argv[0], dbname, sizeof(dbname))))
       return ReturnInt(retstr, rc);
   }
 else 
    (void)strcpy(dbname, DEFAULT_CONNECTION);

 /* 
  * Allocate a DB environment if none exists! 
  */
 if (DbEnv == (DBENV*)NULL) 
   {
    if ((DbEnv = AllocDbEnvironment()) == (DBENV*)NULL)
       return ReturnInt(retstr,SQLCA_IntCode);
   }

 /* 
  * Make sure there is no existing connection with the same name 
  */
 if (FindObject(dbname, DbEnv->db_tbl, TBL_CONNECTIONS)) 
   {
    (void)sprintf(tmp, "connection already open with name \"%s\".", dbname);
    return ReturnError(retstr, 20, tmp);
   }

 /*
  * If there is an optional connect string, make a copy of it
  */
 if (dsn_supplied)
   {
    if ((conn = (UCHAR *)malloc((RXSTRLEN(argv[4])+1)*sizeof(UCHAR))) == NULL)
       return(SetIntError(10, "out of memory", __FILE__,__LINE__));
    connlen = (SWORD)RXSTRLEN(argv[4]);
    RXSTRCPY(conn,connlen,RXSTRPTR(argv[4]),(SWORD)RXSTRLEN(argv[4]));
   }

 /* 
  * Open a new connection for the given connect string. 
  */
 if ((rc = OpenConnection(dbname, uid, uidlen, pwd, pwdlen, 
           srv_dsn, srvlen, conn, connlen, &db)) == DB_ERROR)
     return ReturnInt(retstr, (long)rc);

 if (conn)
    free(conn);
 DbEnv->num_connections++;
 DbEnv->current_connection = db;
 (void)InsertObject(db, DbEnv->db_tbl, TBL_CONNECTIONS);

 return ReturnInt(retstr, rc);
}

/*-----------------------------------------------------------------------------
 * SYNOPSIS:  SQLDISCONNECT( [connection-name ] )
 *
 * RETURNS :  0-success, <0-error.
 *----------------------------------------------------------------------------*/
ULONG RXSQL_APIENTRY SQLDISCONNECT

#ifdef HAVE_PROTO
    (RXSQL_PUCHAR		name,ULONG	argc,RXSTRING	argv[],RXSQL_PCSZ		stck,RXSTRING	*retstr)
#else
    (name, argc, argv, stck, retstr)
    RXSQL_PUCHAR		name;
    ULONG	argc;
    RXSTRING	argv[];
    RXSQL_PCSZ		stck;
    RXSTRING	*retstr;
#endif
{
 int         rc = 0;
 char        dbname[MAX_IDENTIFIER+1];
 DBCON       *db=NULL;
 char        tmp[128];


 FunctionPrologue(name, argc, argv);

 if (SQLCA_SqlCode) ClearDBError();
 if (SQLCA_IntCode) ClearIntError();

 if (argc > 1) return 1;

 if (DbEnv == (DBENV*)NULL)
    return ReturnInt(retstr, 0L);

 if (argc && RXSTRLEN(argv[0])) 
   {
    /* 
     * A connection has been named 
     */
    if ((rc = MkIdentifier(argv[0], dbname, sizeof(dbname))))
       return ReturnInt(retstr, rc);

    /* 
     * Make sure the named connection exists. 
     */
    if ((db = FindObject(dbname, DbEnv->db_tbl, TBL_CONNECTIONS)) == (DBCON*)NULL) 
      {
       (void)sprintf(tmp,"connection \"%s\" is not open.", dbname);
       return ReturnError(retstr, 21, tmp);
      }
   }
 else 
   {
    if (DbEnv->current_connection)
       db = DbEnv->current_connection;
    else
       return ReturnError(retstr, 25, "no connection is current");
   }

 /*
  * If terminating the current connection then make it so there is
  * no current connection!
  */
 if (db == DbEnv->current_connection)
    DbEnv->current_connection = (DBCON*)NULL;

 /* 
  * Do the disconnection 
  */
 rc = ReleaseConnection(db);

 /* 
  * Free the environment if zero connections remaining 
  */
 if (DbEnv->num_connections == 0) 
   {
    free(DbEnv);
    DbEnv = (DBENV*)NULL;
   }
 return ReturnInt(retstr, (long)rc);
}

/*-----------------------------------------------------------------------------
 * SYNOPSIS:  SQLDEFAULT( [connection-name ] )
 *
 * RETURNS :  When called with 0 args : 0-success, <0-error.
 *         :  When called with 1 arg  : Name of current connection else "".
 *----------------------------------------------------------------------------*/
ULONG RXSQL_APIENTRY SQLDEFAULT

#ifdef HAVE_PROTO
    (RXSQL_PUCHAR		name,ULONG	argc,RXSTRING	argv[],RXSQL_PCSZ		stck,RXSTRING	*retstr)
#else
    (name, argc, argv, stck, retstr)
    RXSQL_PUCHAR		name;
    ULONG	argc;
    RXSTRING	argv[];
    RXSQL_PCSZ		stck;
    RXSTRING	*retstr;
#endif
{
 char        dbname[MAX_IDENTIFIER+1];
 DBCON       *db=NULL;
 char        tmp[128];
 int rc=0;

 FunctionPrologue(name, argc, argv);

 if (SQLCA_SqlCode) ClearDBError();
 if (SQLCA_IntCode) ClearIntError();

 if (argc > 1) return 1;

 if (argc && RXSTRLEN(argv[0])) 
   {
    /* 
     * Get the normalised name of the connection. 
     */
    if ((rc = MkIdentifier(argv[0], dbname, sizeof(dbname))))
       return ReturnInt(retstr, rc);

    /* 
     * Make sure we have an environment! 
     */
    if (DbEnv == (DBENV*)NULL)
       return ReturnError(retstr, 22, "no connections open.");

    /* 
     * Make sure the Connection Name is a valid one! 
     */
    if ((db = FindObject(dbname, DbEnv->db_tbl, TBL_CONNECTIONS)) == (DBCON*)NULL) 
      {
       (void)sprintf(tmp,"connection \"%s\" is not open.", dbname);
       return ReturnError(retstr, 21, tmp);
      }

    /* 
     * Make connection the default one! 
     */
    DbEnv->current_connection = db;

    return ReturnInt(retstr, 0L);
   }
 else 
   {
    if (DbEnv && DbEnv->current_connection) 
      {
       return ReturnString(retstr,DbEnv->current_connection->name,
                         strlen(DbEnv->current_connection->name));
      }
    else
       return ReturnString(retstr, "", 0);
   }
}

/*-----------------------------------------------------------------------------
 * SYNOPSIS:  Called by SQLCOMMIT( ) &  SQLROLLBACK( )
 *
 * RETURNS :  0-success, <0-error.
 *----------------------------------------------------------------------------*/
ULONG RXSQL_APIENTRY SQLTRANSACT

#ifdef HAVE_PROTO
    (RXSQL_PUCHAR		name,ULONG	argc,RXSTRING	argv[],RXSQL_PCSZ		stck,RXSTRING	*retstr,SWORD commit)
#else
    (name, argc, argv, stck, retstr, commit)
    RXSQL_PUCHAR		name;
    ULONG	argc;
    RXSTRING	argv[];
    RXSQL_PCSZ		stck;
    RXSTRING	*retstr;
    SWORD commit;
#endif
{ 
 int         rc = 0;
 DBCON       *db=NULL;


 FunctionPrologue(name, argc, argv);

 if (SQLCA_SqlCode) ClearDBError();
 if (SQLCA_IntCode) ClearIntError();

 if (argc > 1) return 1;

 if (DbEnv == (DBENV*)NULL)
     return ReturnError(retstr, 22, "no connections open.");

 if (DbEnv->current_connection)
    db = DbEnv->current_connection;
 else
    return ReturnError(retstr, 25, "no connection is current");

 if (SQLTransact(henv, db->hdbc,commit) == SQL_ERROR)
   {
    SetDBError(SQL_NULL_HENV, db->hdbc, SQL_NULL_HSTMT, NULL);
    rc = SQLCA_SqlCode;
   }

 return ReturnInt(retstr, (rc==0)?0:DB_ERROR);
}


/*-----------------------------------------------------------------------------
 * This function determines the number of decimal places (if any) in
 * the supplied string.
 *----------------------------------------------------------------------------*/
static int GetDecimals

#ifdef HAVE_PROTO
    (char *val,int vallen)
#else
    (val,vallen)
    char *val;
    int vallen;
#endif
{
 int i=0,numdec=0;

 InternalFunctionPrologue("GetDecimals");

 for (i=vallen;i>(-1);i--)
   {
    if (*(val+i) == '.')
      {
       numdec = vallen - i;
       break;
      }
   }
 return (numdec);
}

/*-----------------------------------------------------------------------------
 * This function validates the supplied datatype and attempts to set
 * the length and scale values correctly.
 *----------------------------------------------------------------------------*/
static int GetValidDatatype

#ifdef HAVE_PROTO
    (DBCON *db,char *str,int strlen,char *val,int vallen,SWORD *datatype, SDWORD *length, SWORD *scale)
#else
    (db,str,strlen,val,vallen,datatype,length,scale)
    DBCON *db;
    char *str;
    int strlen;
    char *val;
    int vallen;
    SWORD *datatype;
    SDWORD *length;
    SWORD *scale;
#endif
{
 int   i=0;
 int   found=0;
 int   bind_type=0;

 InternalFunctionPrologue("GetValidDatatype");

 /*
  * Check that the supplied string is valid.
  */

 bind_type = ValidateDataType(str,strlen,db);
 if (bind_type == (RXSQL_INVALID_DATATYPE))
    return(1);
 *datatype = (SWORD)bind_type;

 switch(*datatype)
   {
    case SQL_DECIMAL:
    case SQL_NUMERIC:
    case SQL_TIMESTAMP:
         *length = vallen+2;     /* extra 2 for sign and decimal point */
         *scale = GetDecimals(val,vallen); /* digits to right of decimal point */
         break;
    default:
         *length = vallen;
         *scale = 0;
         break;
   }
 return (0);
}

/*-----------------------------------------------------------------------------
 * Binds all arguments for standard placemarkers (by number)
 *----------------------------------------------------------------------------*/
static int BindByStandard

#ifdef HAVE_PROTO
    (STMT *stmt,ULONG argc,RXSTRING argv[])
#else
    (stmt, argc, argv)
    STMT          *stmt;
    ULONG         argc;
    RXSTRING      argv[];
#endif
{
 int      colno=0,i=0,file_bind=0,bind_type=0;
 int      rc=0;
 long     bindlen=0L;
 char     tmp[128];
 SQLWA    *swa=SWA(stmt);
 SWORD    pm=0;
 SDWORD   length=0;
 SWORD    scale=0;
 SWORD    datatype=0;
 PMDSCR   *pmd=(PMDSCR*)NULL;
 RXSTRING *Varg=NULL, *Darg=NULL;
 FILE     *fp=NULL;
 struct stat stat_buf;


 InternalFunctionPrologue("BindByStandard");

 /*
  * Check that the number of parameter markers in the statement (* 2)
  * matches the number of arguments passed.
  */
 if (argc % 2)
    return(SetIntError(62, "bind values must be paired", __FILE__,__LINE__));

 if (SQLNumParams(stmt->hstmt, &pm) != SQL_SUCCESS)
   {
    SetDBError(SQL_NULL_HENV, SQL_NULL_HDBC, stmt->hstmt,swa);
    return DB_ERROR;
   }
 if (pm * 2 != (SWORD)argc)
   {
    (void)sprintf(tmp, "%d bind values passed. %d expected",argc/2, pm);
    return(SetIntError(61, tmp, __FILE__,__LINE__));
   }

 /*
  * Bind all variables (if any) to placeholders in SQL statement. It is
  * assumed that the variables are static for the life of the SQL statement
  * i.e. their addresses remain fixed.
  */
 for (i=0,colno=0;i<(int)argc && i<MAX_BINDVARS;colno++)
   {
    Darg = &argv[i++];
    Varg = &argv[i++];
    /* 
     * Get the parameter marker descriptor
     */
    if (swa->pm[colno] == (PMDSCR*)NULL)
      {
       if ((pmd = (swa->pm[colno] = (PMDSCR*)malloc(sizeof(PMDSCR)))) == (PMDSCR*)NULL)
          return(SetIntError(10, "out of memory", __FILE__,__LINE__));
       pmd->buf = (char*)NULL;
       pmd->buflen = 0L;
       pmd->retlen = 0L;
      }
    else
       pmd = swa->pm[colno];
    /*
     * Check bind value type. If "FILE:", then get bind value from
     * the filename.
     */
    file_bind = 0;
    if (memcmp(RXSTRPTR(*Darg),"FILE:",5) == 0)
      {
       char *ptr = (char *)RXSTRPTR(*Darg);
       if (GetValidDatatype(stmt->db,ptr+5,RXSTRLEN(*Darg)-5,
                         RXSTRPTR(*Varg),RXSTRLEN(*Varg),
                         &datatype,&length,&scale))
         {
          (void)sprintf(tmp, "invalid datatype %s specified",ptr+5);
          return(SetIntError(64, tmp, __FILE__,__LINE__));
         }
       file_bind = 1;
      }
    else
      {
       if (GetValidDatatype(stmt->db,RXSTRPTR(*Darg),RXSTRLEN(*Darg),
                         RXSTRPTR(*Varg),RXSTRLEN(*Varg),
                         &datatype,&length,&scale))
         {
          (void)sprintf(tmp, "invalid datatype %s specified",RXSTRPTR(*Darg));
          return(SetIntError(64, tmp, __FILE__,__LINE__));
         }
      }
    if (file_bind)
      {
       if ((fp = fopen(RXSTRPTR(*Varg),"rb")) == NULL)
         {
          sprintf(tmp,"<open> on file <%s> failed: %s.",RXSTRPTR(*Varg),strerror(errno));
          return(SetIntError(86, tmp, __FILE__,__LINE__));
         }
       stat(RXSTRPTR(*Varg),&stat_buf);
       bindlen = (int)stat_buf.st_size;
       /*
        * In an attempt to save malloc() & free() calls we keep the old
        * buffer providing it is not grossly oversized. 
        */
       if ((size_t)pmd->buflen < (size_t)bindlen
       || (size_t)pmd->buflen > (size_t)(bindlen + 256))
         {
          /* Free the old buffer (if any) */
          if (pmd->buf)
            {
             free(pmd->buf);
             pmd->buflen = 0;
            }
          /* Allocate a new buffer */
          if ((pmd->buf = (UCHAR*)malloc(bindlen+1)) == (UCHAR*)NULL)
             return(SetIntError(10, "out of memory", __FILE__,__LINE__));
          pmd->buflen = bindlen;
         }
       if ((LONG)fread(pmd->buf,sizeof(char),bindlen,fp) != bindlen)
         {
          sprintf(tmp,"<read> on file <%s> failed: %s.",RXSTRPTR(*Varg),strerror(errno));
          return(SetIntError(86, tmp, __FILE__,__LINE__));
         }
       if (fclose(fp))
         {
          sprintf(tmp,"<close> on file <%s> failed: %s.",RXSTRPTR(*Varg),strerror(errno));
          return(SetIntError(86, tmp, __FILE__,__LINE__));
         }
       pmd->buf[bindlen] = '\0';
       if ((memcmp(pmd->buf,NullStringIn,bindlen)) == 0)
          /* indicate a NULL value */
          pmd->retlen = SQL_NULL_DATA;
       else
          pmd->retlen = pmd->buflen;
      }
    else
      {
       /*
        * In an attempt to save malloc() & free() calls we keep the old
        * buffer providing it is not grossly oversized. 
        */
       if ((size_t)pmd->buflen < RXSTRLEN(*Varg)
       || (size_t)pmd->buflen > RXSTRLEN(*Varg) + 256)
         {
          /* Free the old buffer (if any) */
          if (pmd->buf)
            {
             free(pmd->buf);
             pmd->buflen = 0;
            }
          /* Allocate a new buffer */
          if ((pmd->buf = (UCHAR*)malloc(RXSTRLEN(*Varg)+1)) == (UCHAR*)NULL)
             return(SetIntError(10, "out of memory", __FILE__,__LINE__));
          pmd->buflen = RXSTRLEN(*Varg);
         }
       memcpy(pmd->buf,RXSTRPTR(*Varg),RXSTRLEN(*Varg));
       pmd->buf[RXSTRLEN(*Varg)] = '\0';

       if (memcmp(RXSTRPTR(*Varg),NullStringIn,RXSTRLEN(*Varg)) == 0)
          /* indicate a NULL value */
          pmd->retlen = SQL_NULL_DATA;
       else 
          pmd->retlen = pmd->buflen;
      }
#if defined(UDBC_ODBC) && defined(OS2)
    if (rc = SQLSetParam(stmt->hstmt,
                         colno+1,
                         SQL_C_CHAR,
                         datatype,
                         length,
                         scale,
                         pmd->buf,
                         &pmd->retlen))
      {
       SetDBError(SQL_NULL_HENV, SQL_NULL_HDBC, stmt->hstmt,swa);
       return DB_ERROR;
      }
#else
    if (rc = SQLBindParameter(stmt->hstmt,
                              colno+1,
                              SQL_PARAM_INPUT,
                              SQL_C_CHAR,
                              datatype,
                              length,
                              scale,
                              pmd->buf,
                              pmd->buflen,
                              &pmd->retlen))
      {
       SetDBError(SQL_NULL_HENV, SQL_NULL_HDBC, stmt->hstmt,swa);
       return DB_ERROR;
      }
#endif
    }

 /*
  * If previous bind count is non-zero (i.e. a previous bind for 
  * this statement then check all values are re-bound. 
  */
 if (swa->bind_cnt && colno < swa->bind_cnt)
   {
    (void)sprintf(tmp, "%d bind values passed. %d expected", colno, swa->bind_cnt);
    return(SetIntError(61, tmp, __FILE__,__LINE__));
   }

 /* 
  * save the new bind count 
  */
 swa->bind_cnt = pm;

 return (0);
}

/*-----------------------------------------------------------------------------
 * This function binds all values (supplied as a list of arguments) to
 * the specified cursor.
 * Each variable to be bound, counsists of a pair of parameters:       
 * The first is the datatype of the parameter marker, which can be one of:
 *      CHAR
 *      INTEGER
 *      ....
 * The second is the actual value of the variable.
 *----------------------------------------------------------------------------*/
static int BindValues

#ifdef HAVE_PROTO
    (STMT *stmt,ULONG argc,RXSTRING argv[])
#else
    (stmt, argc, argv)
    STMT     *stmt;
    ULONG    argc;
    RXSTRING argv[];
#endif
{
 int      bind_by_array=0;
 int      arglen1=0,arglen2=0;
 int      rc=0;

 InternalFunctionPrologue("BindValues");

 if ((stmt->Argv = (RXSTRING*)malloc(2*MAX_BINDVARS*sizeof(RXSTRING))) == NULL)
    return(SetIntError(10, "out of memory", __FILE__,__LINE__));
 memset(stmt->Argv,0,2*MAX_BINDVARS*sizeof(RXSTRING));
 /*
  * Determine if the bind variables are being passed as arrays
  */
 if (argc == 2)
   {
    arglen1 = RXSTRLEN(argv[0]);
    arglen2 = RXSTRLEN(argv[1]);
    if (arglen1 && *(RXSTRPTR(argv[0])+arglen1-1) == '.'
    &&  arglen2 && *(RXSTRPTR(argv[1])+arglen2-1) == '.')
      {
       bind_by_array = 1;
       if ((rc = ArrayToArgv(&(argv[0]),&(argv[1]),&stmt->Argc,stmt->Argv)) != 0)
         {
          stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
          return(rc);
         }
      }
   }
 if (!bind_by_array)
   {
    if (ArgvToArgv(argc,argv,&stmt->Argc,stmt->Argv))
      {
       stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
       return(SetIntError(10, "out of memory", __FILE__,__LINE__));
      }
   }
 /* 
  * Bind by standard placemarkers
  */
 rc = BindByStandard(stmt,stmt->Argc,stmt->Argv);
 return (rc);
}

/*-----------------------------------------------------------------------------
 * Get a column expression descriptor.  Assumes the index is valid!
 *----------------------------------------------------------------------------*/
static FLDDSCR *GetFldDscr

#ifdef HAVE_PROTO
    (SQLWA *swa,int i)
#else
    (swa, i)
    SQLWA   *swa;
    int	    i;
#endif
{
 FLDDSCR *fd=NULL;

 InternalFunctionPrologue("GetFldDscr");

 fd = swa->fa[i];
 if (fd == (FLDDSCR*)NULL)
   {
    if ((fd = (swa->fa[i] = (FLDDSCR*)malloc(sizeof(FLDDSCR))))
                    == (FLDDSCR*)NULL) 
      {
       return (FLDDSCR*)NULL;
      }
    fd->rbuf = (char*)NULL;
    fd->rbufl = 0;
    fd->retl = 0;
    fd->rcode = 0;
   }
 return fd;
}

/*-----------------------------------------------------------------------------
 * Describes columns (expressions) and defines the output buffers for each
 * column for the select statement in nominated cursor.
 * Also binds the columns in the result set to the output buffers for the
 * subsequent fetch.
 * Returns the number of expressions in the select statement.
 *----------------------------------------------------------------------------*/
static int DefineExpressions

#ifdef HAVE_PROTO
    (STMT *stmt)
#else
    (stmt)
    STMT     *stmt;
#endif
{
 SQLWA       *swa=SWA(stmt);
 FLDDSCR     *fd=NULL;
 SWORD real_column_length=0;
 int         i=0;
 SDWORD  varlen=0;

 InternalFunctionPrologue("DefineExpressions");

 /* 
  * determine the number of columns in the result set 
  */
 if (SQLNumResultCols(stmt->hstmt,&(swa->expr_cnt)) == SQL_ERROR)
   {
    SetDBError(SQL_NULL_HENV, SQL_NULL_HDBC, stmt->hstmt,swa);
    return (DB_ERROR);
   }

 /* 
  * Describe & define buffer for each expression in the SELECT statement 
  */
 for (i = 0; i < swa->expr_cnt ; i++)
   {
    /* Get a new field descriptor */
    if ((fd = GetFldDscr(swa, i)) == (FLDDSCR*)NULL)
       return(SetIntError(10, "out of memory", __FILE__,__LINE__));

    fd->cbufl = sizeof(fd->cbuf) - 1;

    /* 
     * get general info about a column 
     */
    if (SQLDescribeCol(stmt->hstmt,(UWORD)i+1,(UCHAR*)fd->cbuf,(SWORD)fd->cbufl,(SWORD*)&real_column_length,
                       (SWORD*)&(fd->dbtype),(UDWORD*)&(fd->prec),(SWORD*)&(fd->scale),(SWORD*)&(fd->nullok)) == SQL_ERROR)
      {
       SetDBError(SQL_NULL_HENV, SQL_NULL_HDBC, stmt->hstmt,swa);
       return (DB_ERROR);
      }
    /* 
     * get the display size of the column 
     */
    if (SQLColAttributes(stmt->hstmt,(UWORD)i+1,(UWORD)SQL_COLUMN_DISPLAY_SIZE, NULL,(SWORD)0,NULL, (SDWORD*)&varlen) == SQL_ERROR)
      {
       SetDBError(SQL_NULL_HENV, SQL_NULL_HDBC, stmt->hstmt,swa);
       return (DB_ERROR);
      }
    /* 
     * If display size > DEFAULT_LONG_LIMIT, then restrict the size of the data,
     */
    if (varlen > DEFAULT_LONG_LIMIT)
       varlen = DEFAULT_LONG_LIMIT - 1;
    /* 
     * increment for terminating NUL character 
     */
    varlen++;
    /* 
     * Terminate the Column Name 
     */
    fd->cbufl = strlen(fd->cbuf);

    /* 
     * Set the column name to upper case
     */
    (void)make_upper(fd->cbuf);

    /*
     * In an attempt to save malloc() & free() calls we keep the old
     * buffer providing it is not grossly oversized. 
     */
    if ((size_t)fd->rbufl < (size_t)varlen || (size_t)fd->rbufl > (size_t)(varlen + 256))
      {
       /* 
        * Free the old buffer (if any) 
        */
       if (fd->rbufl) 
         {
          free(fd->rbuf);
          fd->rbufl = 0;
         }
       /* 
        * Allocate a new buffer 
        */
       if ((fd->rbuf = (UCHAR *)malloc(varlen)) == (UCHAR *)NULL)
          return(SetIntError(10, "out of memory", __FILE__,__LINE__));
      }
#if 1
    fd->rbufl = varlen-1;
#else
    fd->rbufl = varlen;
#endif


#if !defined(USE_GETDATA)
    /* 
     * bind the columns to the storage area used to save the data 
     */
#if 1
    if (SQLBindCol(stmt->hstmt,(UWORD)i+1,(SWORD)SQL_C_CHAR,(PTR)fd->rbuf,(SDWORD)varlen,(SDWORD*)&(fd->retl)) == SQL_ERROR)
#else
    if (SQLBindCol(stmt->hstmt,i+1,SQL_C_CHAR,fd->rbuf,fd->rbufl,&(fd->retl)) == SQL_ERROR)
#endif
      {
       SetDBError(SQL_NULL_HENV, SQL_NULL_HDBC, stmt->hstmt,swa);
       return (DB_ERROR);
      }
#endif

#if DEBUG
    printf("Name: %s DISPLAY_SIZE: %d REAL_COLUMN_LENGTH: %d RETL: %d PREC: %d\n",
           fd->cbuf, varlen, real_column_length, fd->retl, fd->prec);
#endif
   }

 return (swa->expr_cnt);
}

/*-----------------------------------------------------------------------------
 * Fetches the next row from the nominated cursor and returns the values
 * of the expressions for the fetched row into the compound variable with
 * name constructed as follows:
 *
 * For single fetches. eg. SQLFETCH('s1')
 *                     or  SQLFETCH('s1','') :
 *  <statement-name>.<column-name>
 *
 * For bulk fetches. eg. SQLCOMMAND(stmt1)
 *                   or  SQLFETCH('s1',0)
 *                   or  SQLFETCH('s1',1)
 *                   or  SQLFETCH('s1',20) :
 *  <statement-name>.<column-name>.<row-number>
 *
 * Note that the row-number always starts at 1!
 *
 * Returns:
 *	success:  0
 *	failure: database return code
 *----------------------------------------------------------------------------*/
static int FetchRow

#ifdef HAVE_PROTO
    (STMT *stmt,char *stem,ULONG rowcount)
#else
    (stmt, stem, rowcount)
    STMT    *stmt;
    char    *stem;
    ULONG   rowcount;
#endif
{
 FLDDSCR *fd=NULL;
 SQLWA   *swa=NULL;
 int	    rc=0, i=0;
 size_t  varlen=0;
 char    varname[MAX_IDENTIFIER+1+10];
 char    tmp[128];
 int     rlen=0;

 InternalFunctionPrologue("FetchRow");

 swa = SWA(stmt);

 rc = SQLFetch(stmt->hstmt);
 if (rc == SQL_NO_DATA_FOUND)
    return (rc);
 if (rc != SQL_SUCCESS)
   {
    SetDBError(SQL_NULL_HENV, SQL_NULL_HDBC, stmt->hstmt,swa);
    return DB_ERROR;
   }

 /* 
  * Get each expr value in turn 
  */
 for (i = 0; rc == 0 && i < swa->expr_cnt; i++)
   {
    fd = swa->fa[i];
        
#ifdef USE_GETDATA
    rc = SQLGetData(stmt->hstmt,(UWORD)i+1,(SWORD)SQL_C_CHAR, (PTR)fd->rbuf, (SDWORD)((fd->rbufl)+1), (SDWORD*)&(fd->retl));
    if (rc != SQL_SUCCESS)
      {
       SetDBError(SQL_NULL_HENV, SQL_NULL_HDBC, stmt->hstmt,swa);
       return (DB_ERROR);
      }
#endif
    /* 
     * Add each column value to the stem's values 
     */
    (void)sprintf(varname, rowcount ? "%s.%s.%lu" : "%s.%s",
                      stem, fd->cbuf, rowcount);
    varlen = strlen(varname);

    if (fd->retl == SQL_NULL_DATA)      /* returned value is NULL */
      {
       rc = SetRexxVariable(varname, varlen, RXSNullStringOut.strptr, RXSNullStringOut.strlength);
      }
    else 
      {
       if (fd->retl > fd->rbufl)  /* truncation or conversion occured */
         {
          if (IgnoreTruncate)
            rc = SetRexxVariable(varname, varlen, fd->rbuf, (size_t)fd->rbufl);
          else
            {
             sprintf(tmp,"Conversion/truncation occurred on column %s: Expecting %d, got %d",
                     fd->cbuf,fd->rbufl,fd->retl);
             return(SetIntError(15, tmp, __FILE__,__LINE__));
            }
         }
       else 
          rc = SetRexxVariable(varname, varlen, fd->rbuf, (size_t)fd->retl);
      }
   }
 if (rc)
    return(SetIntError(16, "unable to set REXX variable", __FILE__,__LINE__));
 return rc;
}

static int SetRowCountVar

#ifdef HAVE_PROTO
    (SQLWA *swa,char *stem_name,long rowcount)
#else
    (swa, stem_name, rowcount)
    SQLWA *swa;
    char *stem_name;
    long rowcount;
#endif
{
 int     i=0,rc=0;
 char    varname[MAX_IDENTIFIER*2+4], buf[11];

 InternalFunctionPrologue("SetRowCountVar");

 for (i = 0; i < swa->expr_cnt; i++) 
   {
    sprintf(varname, "%s.%s.0", stem_name,swa->fa[i]->cbuf);
    sprintf(buf, "%lu", rowcount);
    if (rc = SetRexxVariable(varname,strlen(varname),buf,strlen(buf)))
       return(rc);
   }
 return 0;
}

/*-----------------------------------------------------------------------------
 * SYNOPSIS: SQLCOMMAND(stem-name, sql-statement-text [, <bind-args>])
 *
 * RETURNS :  0-success < 0-error.
 *----------------------------------------------------------------------------*/
ULONG RXSQL_APIENTRY SQLCOMMAND

#ifdef HAVE_PROTO
    (RXSQL_PUCHAR		name,ULONG	argc,RXSTRING	argv[],RXSQL_PCSZ		stck,RXSTRING	*retstr)
#else
    (name, argc, argv, stck, retstr)
    RXSQL_PUCHAR		name;
    ULONG	argc;
    RXSTRING	argv[];
    RXSQL_PCSZ		stck;
    RXSTRING	*retstr;
#endif
{
 int         rc = 0, i=0;
 SDWORD      rowcount=0L;
 int         expr_cnt=0;
 DBCON       *db=NULL;
 STMT        *stmt=NULL;
 SQLWA       *swa=NULL;
 char        stem_name[MAX_IDENTIFIER+1];
 ULONG       warning=0;

 FunctionPrologue(name, argc, argv);

 if (SQLCA_SqlCode) ClearDBError();
 if (SQLCA_IntCode) ClearIntError();

 if (argc == 0) return 1;

 /* 
  * Get pointer to current connection 
  */
 if (DbEnv && DbEnv->current_connection)
    db = DbEnv->current_connection;
 else
    return ReturnError(retstr, 25, "no connection is current");

 if (argc == 1 || RXSTRLEN(argv[0]) == 0) /* No stem name specified! */
    (void)strcpy(stem_name, DEFAULT_STEM);
 else 
   {
    if ((rc = MkIdentifier(argv[0], stem_name, sizeof(stem_name))))
       return ReturnInt(retstr, rc);
   }

 /* 
  * If no default statement then create it! 
  */
 if ((stmt = (STMT*)(db->dflt_stmt)) == (STMT*)NULL) 
   {
    /* 
     * Open a statement for the default statement. 
     */
    if ((rc = OpenStatement(DEFAULT_STATEMENT, db, &stmt)))
       return ReturnInt(retstr, (long)rc);

    db->dflt_stmt = (void*)stmt;
   }

 swa = SWA(stmt);

 /*
  * If only 1 arg then it is the SQL-statement-text. If more than 1 args
  * then arg#1 is stem-name, arg#2 is text and rest are bind values.
  * 'i' is index (base zero) to sql-statement-text arg.
  */
 i = (argc == 1) ? 0 : 1;

 /* 
  * set flag indicating if SELECT statement supplied
  */
 swa->select = SelectStatement(RXSTRPTR(argv[i]),RXSTRLEN(argv[i]));

 /* 
  * Save the SQL statement if required 
  */
 if (SaveSQL && SaveSqlStatement(&(swa->sql_stmt), &(swa->sql_stmt_sz),
                                 RXSTRPTR(argv[i]),(int)RXSTRLEN(argv[i])))
    return ReturnError(retstr, 10, "out of memory");

 /* 
  * Prepare the statement 
  */
#if !defined(USE_EXECDIRECT)
 if (SQLPrepare(stmt->hstmt,(UCHAR*)RXSTRPTR(argv[i]), (SDWORD)RXSTRLEN(argv[i])) != SQL_SUCCESS)
   {
    SetDBError(SQL_NULL_HENV, SQL_NULL_HDBC, stmt->hstmt, swa);
    return ReturnInt(retstr, DB_ERROR);
   }
#endif

 /* 
  * Describe the expression list & define output buffers (if any). 
  */
 if ((expr_cnt = DefineExpressions(stmt)) < 0)
    return ReturnInt(retstr, (long)expr_cnt);

 /* 
  * Bind variables in WHERE/INSERT/UPDATE clause (if any). 
  */
 if (argc > 2) 
   {
    if ((rc = BindValues(stmt, argc-2, &argv[2])))
      {
       stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
       return ReturnInt(retstr, (long)rc);
      }
   }

 /* 
  * Execute the statement 
  * Check for any truncation errors
  */
 rc = SQLExecute(stmt->hstmt);
 if (rc == SQL_ERROR)
   {
    SetDBError(SQL_NULL_HENV, SQL_NULL_HDBC, stmt->hstmt, swa);
#if 0
    if (sqlcode == RXSQL_TRUNCATION_SQLCODE
    &&  !IgnoreTruncate)
#else
    if (IgnoreTruncate
    &&  strcmp(SQLCA_SqlState,"22001") == 0)
       warning = DB_WARNING;
    else
#endif
      {
       stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
       return ReturnInt(retstr, DB_ERROR);
      }
   }
 if (rc == SQL_SUCCESS_WITH_INFO)
   {
    SetDBError(SQL_NULL_HENV, SQL_NULL_HDBC, stmt->hstmt, swa);
    warning = DB_WARNING;
   }

 /* 
  * Test if select statement. If not then done else fetch rows. 
  */
 if (expr_cnt == 0) 
   { 
    if (SQLRowCount(stmt->hstmt,&rowcount) != SQL_SUCCESS)
      {
       stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
       SetDBError(SQL_NULL_HENV, SQL_NULL_HDBC, stmt->hstmt, swa);
       return ReturnInt(retstr, DB_ERROR);
      }
    if (AutoCommit) /* commit the statement */
      {
       if (SQLTransact(henv, db->hdbc,SQL_COMMIT) == SQL_ERROR)
         {
          stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
          SetDBError(SQL_NULL_HENV, SQL_NULL_HDBC, stmt->hstmt, swa);
          return ReturnInt(retstr, DB_ERROR);
         }
      }
   }
 else /* ...then must be a SELECT statement */
   {
    /* Fetch each row in turn */
    for (rowcount = 1; RowLimit == 0 || rowcount <= (SDWORD)RowLimit; rowcount++)
      {
       if ((rc = FetchRow(stmt, stem_name, rowcount)))
          break;
      }
    rowcount--;
    if (rc && rc != SQL_NO_DATA_FOUND)
      {
       stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
       return ReturnInt(retstr, (long)rc);
      }

    if ((rc = SetRowCountVar(swa, stem_name, rowcount)))
      {
       stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
       return ReturnInt(retstr, (long)rc);
      }

   }

 stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);

 SetRowCount(rowcount);

 /* 
  * Release the statement and associated storage. 
  */
 if ((rc = ReleaseStatement(stmt)))
    return ReturnInt(retstr, (long)rc);
 db->dflt_stmt = (STMT*)NULL;

 return ReturnInt(retstr, warning);
}

/*-----------------------------------------------------------------------------
 * SYNOPSIS:  SQLPREPARE(statement-name, sql-statement-text)
 *
 * RETURNS :  0-success, <0-error.
 *----------------------------------------------------------------------------*/
ULONG RXSQL_APIENTRY SQLPREPARE

#ifdef HAVE_PROTO
    (RXSQL_PUCHAR		name,ULONG	argc,RXSTRING	argv[],RXSQL_PCSZ		stck,RXSTRING	*retstr)
#else
    (name, argc, argv, stck, retstr)
    RXSQL_PUCHAR		name;
    ULONG	argc;
    RXSTRING	argv[];
    RXSQL_PCSZ		stck;
    RXSTRING	*retstr;
#endif
{
 int         rc=0;
 DBCON       *db=NULL;
 STMT        *stmt=NULL;
 SQLWA       *swa=NULL;
 char        stmt_name[MAX_DB_CURSOR_NAME+1];

 FunctionPrologue(name, argc, argv);

 if (SQLCA_SqlCode) ClearDBError();
 if (SQLCA_IntCode) ClearIntError();

 if (argc != 2) return 1;

 /* 
  * Get pointer to current connection 
  */
 if (DbEnv && DbEnv->current_connection)
    db = DbEnv->current_connection;
 else
    return ReturnError(retstr, 25, "no connection is current");

 if (RXSTRLEN(argv[0]) == 0)          /* No statement name specified! */
    return ReturnError(retstr, 23, "statement name omitted or null");
 else 
    if ((rc = MkIdentifier(argv[0], stmt_name, sizeof(stmt_name))))
       return ReturnInt(retstr, rc);

 /*
  * Find the named statement or create if necessary. We have to be a
  * bit careful here because the statement may exist but point to a
  * different database connection!
  */
 stmt = FindObject(stmt_name, DbEnv->stmt_tbl, TBL_STATEMENTS);

 if (stmt == (STMT*)NULL || stmt->db != db) 
   {
    if (stmt) 
      {
       /*
        * Statement is not for the same db, therefore we must dispose
        * & re-alloc it!
        */
       RemoveObject(stmt);
       ((DBCON*)stmt->db)->num_statements--;
       if ((rc = ReleaseStatement(stmt)))
          return ReturnInt(retstr, (long)rc);
      }

    /* 
     * Open a statement for this statement. 
     */
    if ((rc = OpenStatement(stmt_name, db, &stmt)))
       return ReturnInt(retstr, (long)rc);

    /* 
     * Insert this statement into the connection hash table. 
     */
    (void)InsertObject(stmt, DbEnv->stmt_tbl, TBL_STATEMENTS);

    db->num_statements++;
   }

 swa = SWA(stmt);
 /* 
  * set flag indicating if SELECT statement supplied 
  */
 swa->select = SelectStatement(RXSTRPTR(argv[1]),RXSTRLEN(argv[1]));

 /* 
  * Save the SQL statement if required
  */
 if (SaveSQL && SaveSqlStatement(&(swa->sql_stmt), &(swa->sql_stmt_sz),
                                 RXSTRPTR(argv[1]),(int)RXSTRLEN(argv[1])))
     return ReturnError(retstr, 10, "out of memory");

 swa->bind_cnt = 0;
 swa->expr_cnt = 0;

 /* 
  * Prepare the statement 
  */
 if (SQLPrepare(stmt->hstmt,(UCHAR*)RXSTRPTR(argv[1]), (SDWORD)RXSTRLEN(argv[1])) != SQL_SUCCESS)
   {
    SetDBError(SQL_NULL_HENV, SQL_NULL_HDBC, stmt->hstmt, swa);
    rc = DB_ERROR;
   }

 return ReturnInt(retstr, rc ? DB_ERROR : 0);
}

/*-----------------------------------------------------------------------------
 * Get a pointer to the nominated statement. Returns NULL on error.
 *----------------------------------------------------------------------------*/
static STMT *GetStatement

#ifdef HAVE_PROTO
    (RXSTRING	var,PSZ		buf)
#else
    (var, buf)
    RXSTRING	var; 
    PSZ		buf;
#endif
{
 STMT        *stmt=NULL;
 char        tmp[128];

 InternalFunctionPrologue("GetStatement");

 if (DbEnv == (DBENV*)NULL) 
   {
    SetIntError(22, "no connections open.",__FILE__,__LINE__);
    return (STMT*)NULL;
   }

 if (RXSTRLEN(var) == 0)
   {
    /* 
     * No statement name specified! 
     */
    SetIntError(23, "statement name omitted or null", __FILE__,__LINE__);
    return (STMT*)NULL;
   }

 /* 
  * Get the normalised form of the name 
  */
 if (MkIdentifier(var, buf, MAX_DB_CURSOR_NAME+1))
    return (STMT*)NULL;

 /* 
  * Find the named statement or create if necessary 
  */
 if ((stmt = FindObject(buf, DbEnv->stmt_tbl, TBL_STATEMENTS)) == (STMT*)NULL)
   {
    /*
     * Statement not an existing one!
     */
    (void)sprintf(tmp,"statement \"%s\" does not exist", buf);
    SetIntError(24, tmp,__FILE__,__LINE__);
    return (STMT*)NULL;
   }
 return stmt;
}

/*-----------------------------------------------------------------------------
 * SYNOPSIS:  SQLDISPOSE(statement-name)
 *
 * RETURNS :  0-success, <0-error.
 *----------------------------------------------------------------------------*/
ULONG RXSQL_APIENTRY SQLDISPOSE

#ifdef HAVE_PROTO
    (RXSQL_PUCHAR		name,ULONG	argc,RXSTRING	argv[],RXSQL_PCSZ		stck,RXSTRING	*retstr)
#else
    (name, argc, argv, stck, retstr)
    RXSQL_PUCHAR		name;
    ULONG	argc;
    RXSTRING	argv[];
    RXSQL_PCSZ		stck;
    RXSTRING	*retstr;
#endif
{
 int         rc=0;
 STMT        *stmt=NULL;
 DBCON       *db=NULL;
 char        stmt_name[MAX_DB_CURSOR_NAME+1];

 FunctionPrologue(name, argc, argv);

 if (SQLCA_SqlCode) ClearDBError();
 if (SQLCA_IntCode) ClearIntError();

 if (argc != 1) return 1;

 if ((stmt = GetStatement(argv[0], stmt_name)) == (STMT*)NULL)
     return ReturnInt(retstr, SQLCA_IntCode);
 
 /* 
  * Get pointer to statement's connection structure 
  */
 db = stmt->db;

 /* 
  * Dispose the statement 
  */
 RemoveObject(stmt);
 rc = ReleaseStatement(stmt);

 db->num_statements--;

 return ReturnInt(retstr, (long)rc);
}

/*-----------------------------------------------------------------------------
 * SYNOPSIS:  SQLEXEC (called by SQLEXECUTE() and SQLOPEN()
 *
 * RETURNS :  0-success,
 *           >0-number of rows affected for SQLEXECUTE(),
 *           <0-error.
 *----------------------------------------------------------------------------*/
ULONG RXSQL_APIENTRY SQLEXEC

#ifdef HAVE_PROTO
    (RXSQL_PUCHAR		name,ULONG	argc,RXSTRING	argv[],RXSQL_PCSZ		stck,RXSTRING	*retstr,int open)
#else
    (name, argc, argv, stck, retstr, open)
    RXSQL_PUCHAR		name;
    ULONG	argc;
    RXSTRING	argv[];
    RXSQL_PCSZ		stck;
    RXSTRING	*retstr;
    int open;
#endif
{
 int         rc=0,warning=0;
 int         expr_cnt=0;
 SDWORD  rowcount=0L;
 STMT        *stmt=NULL;
 SQLWA       *swa=NULL;
 char        stmt_name[MAX_DB_CURSOR_NAME+1];
 char        tmp[128];
 DBCON       *db=NULL;

 FunctionPrologue(name, argc, argv);

 if (SQLCA_SqlCode) ClearDBError();
 if (SQLCA_IntCode) ClearIntError();

 if (argc == 0) return 1;

 if ((stmt = GetStatement(argv[0], stmt_name)) == (STMT*)NULL)
    return ReturnInt(retstr, SQLCA_IntCode);
    
 swa = SWA(stmt);

 /*
  * Make sure we have a current connection
  */
 if (DbEnv == (DBENV*)NULL)
     return ReturnError(retstr, 22, "no connections open.");

 if (DbEnv->current_connection)
    db = DbEnv->current_connection;
 else
    return ReturnError(retstr, 25, "no connection is current");

 /*
  * Rest of args (if any) are bind values. Therefore bind the substitution
  * variables in WHERE/INSERT/UPDATE clause (if any).
  */
 argc--;

 if (argc) 
   {
    if ((rc = BindValues(stmt, argc, &argv[1])))
      {
       stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
       return ReturnInt(retstr, (long)rc);
      }
   }

 /*
  * This function can be called as SQLOPEN() or SQLEXECUTE(). These
  * operations are similar except for the describe. Use the function
  * name to determine what operation we are performing.
  */
 if (open) 
   {
    /* 
     * Describe the expression list & define output buffers (if any).
     */
    if ((expr_cnt = DefineExpressions(stmt)) < 0) 
      {
       stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
       return ReturnInt(retstr, (long)expr_cnt);
      }

    /* 
     * Since this is a call to SQLOPEN() the statement must be a query!
     */
    if (expr_cnt == 0) 
      { 
       if (!SEL(swa))
         {
          stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
          (void)sprintf(tmp,"statement \"%s\" is not a query.", stmt_name);
          return ReturnError(retstr, 13, tmp);
         }
       else
         {
          stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
          SetDBError(SQL_NULL_HENV, SQL_NULL_HDBC, stmt->hstmt, swa);
          return ReturnInt(retstr, DB_ERROR);
         }
      }
   }

 /* 
  * Execute the SQL statement. This is required for Queries, DDL & DML!
  */
 rc = SQLExecute(stmt->hstmt);
 if (rc == SQL_ERROR)
   {
    stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
    SetDBError(SQL_NULL_HENV, SQL_NULL_HDBC, stmt->hstmt, swa);
    return ReturnInt(retstr, DB_ERROR);
   }
 if (rc == SQL_SUCCESS_WITH_INFO)
   {
    SetDBError(SQL_NULL_HENV, SQL_NULL_HDBC, stmt->hstmt, swa);
    warning = DB_WARNING;
   }

 stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);

 /*
  * Return the ROWCOUNT.  For a query it will be zero at this stage.  For
  * a DML statement it will be the number of rows affected by the INSERT/
  * UPDATE/DELETE statement.
  */
 if (open)
    SetRowCount(rowcount);
 else
   {
    if (SQLRowCount(stmt->hstmt,&rowcount) != SQL_SUCCESS)
      {
       SetDBError(SQL_NULL_HENV, SQL_NULL_HDBC, stmt->hstmt, swa);
       return ReturnInt(retstr, DB_ERROR);
      }
    SetRowCount(rowcount);
    if (AutoCommit) /* commit the statement */
      {
       if (SQLTransact(henv, db->hdbc,SQL_COMMIT) == SQL_ERROR)
         {
          SetDBError(SQL_NULL_HENV, SQL_NULL_HDBC, stmt->hstmt, swa);
          return ReturnInt(retstr, DB_ERROR);
         }
      }
   }
 return ReturnInt(retstr, (long)warning);
}

/*-----------------------------------------------------------------------------
 * SYNOPSIS: SQLCLOSE(statement-name)
 *
 * RETURNS :  0-success, <0-error.
 *----------------------------------------------------------------------------*/
ULONG RXSQL_APIENTRY SQLCLOSE

#ifdef HAVE_PROTO
    (RXSQL_PUCHAR		name,ULONG	argc,RXSTRING	argv[],RXSQL_PCSZ		stck,RXSTRING	*retstr)
#else
    (name, argc, argv, stck, retstr)
    RXSQL_PUCHAR		name;
    ULONG	argc;
    RXSTRING	argv[];
    RXSQL_PCSZ		stck;
    RXSTRING	*retstr;
#endif
{
 int         rc=0;
 STMT        *stmt=NULL;
 SQLWA       *swa=NULL;
 char        stmt_name[MAX_DB_CURSOR_NAME+1];

 FunctionPrologue(name, argc, argv);

 if (SQLCA_SqlCode) ClearDBError();
 if (SQLCA_IntCode) ClearIntError();

 if (argc != 1) return 1;

 if ((stmt = GetStatement(argv[0], stmt_name)) == (STMT*)NULL)
     return ReturnInt(retstr, SQLCA_IntCode);
 
 swa = SWA(stmt);

 /* 
  * Inform database that operation is complete. This should never fail!
  */
 if (SQLFreeStmt(stmt->hstmt, SQL_CLOSE) != SQL_SUCCESS)
   {
    SetDBError(SQL_NULL_HENV, SQL_NULL_HDBC, stmt->hstmt, swa);
    rc = DB_ERROR;
   }

 return ReturnInt(retstr, rc ? DB_ERROR : 0);
}

/*-----------------------------------------------------------------------------
 * SYNOPSIS:  SQLFETCH(statement-name [, number-of-rows])
 *
 * RETURNS :  0-end-of-data,
 *           >0- single fetch: row number of last row fetched
 *           >0- group fetch : number of rows fetched if < number-of-rows then
 *                             end--of-data is indicated.
 *           <0-error.
 *----------------------------------------------------------------------------*/
ULONG RXSQL_APIENTRY SQLFETCH

#ifdef HAVE_PROTO
    (RXSQL_PUCHAR		name,ULONG	argc,RXSTRING	argv[],RXSQL_PCSZ		stck,RXSTRING	*retstr)
#else
    (name, argc, argv, stck, retstr)
    RXSQL_PUCHAR		name;
    ULONG	argc;
    RXSTRING	argv[];
    RXSQL_PCSZ		stck;
    RXSTRING	*retstr;
#endif
{
 long        rc=0;
 int         single_fetch=0;
 ULONG       num_rows=0L;
 ULONG       rowcount=0L;
 STMT        *stmt=NULL;
 SQLWA       *swa=NULL;
 char        stmt_name[MAX_DB_CURSOR_NAME+1];

 FunctionPrologue(name, argc, argv);

 if (SQLCA_SqlCode) ClearDBError();
 if (SQLCA_IntCode) ClearIntError();

 if (argc == 0 || argc > 2) return 1;

 if ((stmt = GetStatement(argv[0], stmt_name)) == (STMT*)NULL)
     return ReturnInt(retstr, SQLCA_IntCode);
 
 swa = SWA(stmt);

 /* 
  * Determine # of rows to fetch 
  */
 if (argc > 1 && RXSTRLEN(argv[1])) 
   {
    if (StrToInt(&argv[1], &num_rows)) 
       return ReturnError(retstr, 14, "<num-rows> is not a valid integer.");
    single_fetch = FALSE;
   }
 else
    single_fetch = TRUE;

 if (single_fetch) 
   {
    /* Fetch a single row */
    if ((rc = FetchRow(stmt, stmt_name, 0L)))
       rc = (rc == SQL_NO_DATA_FOUND) ? 0 : rc;
    else
       rc = 1; /* single row fetch, must return 1 */
   }
 else 
   {
   /* Fetch each row in turn */
    for (rowcount = 1; num_rows == 0 || rowcount <= num_rows; rowcount++)
      {
       if ((rc = FetchRow(stmt, stmt_name, rowcount)))
          break;
      }
    rowcount--;

    if (rc 
    &&  rc != SQL_NO_DATA_FOUND)
       return ReturnInt(retstr, (long)rc);

    if ((rc = SetRowCountVar(swa, stmt_name, rowcount)))
       return ReturnInt(retstr, (long)rc);

    rc = rowcount;
   }

 SetRowCount(rowcount);
 return ReturnInt(retstr, rc);
}

/*-----------------------------------------------------------------------------
 * SYNOPSIS:  SQLGETDATA(statement-name, column-name, start-byte, length [, file-name])
 *
 * RETURNS :  0-end-of-data,
 *           >0- bytes obtained - <= length
 *           <0-error.
 *----------------------------------------------------------------------------*/
ULONG RXSQL_APIENTRY SQLGETDATA

#ifdef HAVE_PROTO
    (RXSQL_PUCHAR		name,ULONG	argc,RXSTRING	argv[],RXSQL_PCSZ		stck,RXSTRING	*retstr)
#else
    (name, argc, argv, stck, retstr)
    RXSQL_PUCHAR		name;
    ULONG	argc;
    RXSTRING	argv[];
    RXSQL_PCSZ		stck;
    RXSTRING	*retstr;
#endif
{
#define LONG_BLOCK_SIZE 0x10000
 FLDDSCR *fd=NULL;
 long    rc=0,ret_len=0;
 ULONG   num_bytes=0L;
 ULONG   start_byte=0L;
 STMT    *stmt=NULL;
 SQLWA   *swa=NULL;
 char    stmt_name[MAX_IDENTIFIER+1];
 int     long_col=0,ext_type=0,i=0,off=0;
 size_t  varlen=0;
 FILE    *fp=NULL;

 FunctionPrologue(name, argc, argv);

 if (SQLCA_SqlCode) ClearDBError();
 if (SQLCA_IntCode) ClearIntError();

 if (argc < 4 || argc > 5) return 1;

 if ((stmt = GetStatement(argv[0], stmt_name)) == (STMT*)NULL)
     return ReturnInt(retstr, SQLCA_IntCode);
 
 swa = SWA(stmt);


 return ReturnInt(retstr, ret_len);
}

/*-----------------------------------------------------------------------------
 * Fetch the description for the column expression. Used by SQLDESCRIBE().
 *----------------------------------------------------------------------------*/
static int GetColumn

#ifdef HAVE_PROTO
    (STMT *stmt,int i,char *stem_name)
#else
    (stmt, i, stem_name)
    STMT   *stmt;
    int     i;
    char    *stem_name;
#endif
{
 FLDDSCR *fd=NULL;
 SQLWA	*swa=NULL;
 int     idx=0, rc=0;
 char    column_size[15], column_prec[15], column_scale[15], column_nullok[15];
#if OLD
 char    *column_type=NULL;
#else
 UCHAR   column_type[MAX_IDENTIFIER+1];
 SWORD   collen=0;
#endif
 char    name[MAX_IDENTIFIER+32];
 char    *value[NUM_DESCRIBE_COLUMNS];
 int     value_len[NUM_DESCRIBE_COLUMNS];

 InternalFunctionPrologue("GetColumn");

 swa = SWA(stmt);

 if (i >= swa->expr_cnt)
     return 1;

 fd = swa->fa[i];

 /* get the column type of the column */
 if (SQLColAttributes(stmt->hstmt,i+1,SQL_COLUMN_TYPE_NAME,column_type,sizeof(column_type),&collen,NULL) != SQL_SUCCESS)
    strcpy(column_type,"UNKNOWN");

 /* Set up the array */
 value[0] = fd->cbuf;
 value_len[0] = fd->cbufl;
 value[1] = column_type;
 value_len[1] = strlen(column_type);
 value[2] = column_size;
 value_len[2] = sprintf(column_size, "%ld", fd->rbufl);
 value[3] = column_prec;
 value_len[3] = sprintf(column_prec, "%d", fd->prec);
 value[4] = column_scale;
 value_len[4] = sprintf(column_scale, "%d", fd->scale);
 value[5] = column_nullok;
 value_len[5] = sprintf(column_nullok, "%d", fd->nullok);

 /* Output into Rexx variable */
 i++;
 for (idx = 0; idx < NUM_DESCRIBE_COLUMNS; idx++) 
   {
    sprintf(name, "%s.COLUMN.%s.%d", stem_name,column_attribute[idx], i);
    if (rc = SetRexxVariable(name,strlen(name),value[idx],value_len[idx]))
       break;
   }

 return rc;
}

/*-----------------------------------------------------------------------------
 * SYNOPSIS:  SQLDESCRIBE(statement-name [, stem-name])
 *
 * RETURNS : >0-number of columns in the prepared SELECT statement.
 *            0-prepared statement is not a SELECT statement.
 *           <0-error.
 *----------------------------------------------------------------------------*/
ULONG RXSQL_APIENTRY SQLDESCRIBE

#ifdef HAVE_PROTO
    (RXSQL_PUCHAR		name,ULONG	argc,RXSTRING	argv[],RXSQL_PCSZ		stck,RXSTRING	*retstr)
#else
    (name, argc, argv, stck, retstr)
    RXSQL_PUCHAR		name;
    ULONG	argc;
    RXSTRING	argv[];
    RXSQL_PCSZ		stck;
    RXSTRING	*retstr;
#endif
{
 int         i=0, len1=0, len2=0, rc=0;
 STMT        *stmt=NULL;
 SQLWA	*swa=NULL;
 char        **p=NULL, buf1[MAX_IDENTIFIER+32], buf2[16];
 char        stmt_name[MAX_DB_CURSOR_NAME+1];
 char        stem_name[MAX_IDENTIFIER+1];


 FunctionPrologue(name, argc, argv);

 if (SQLCA_SqlCode) ClearDBError();
 if (SQLCA_IntCode) ClearIntError();

 if (argc < 1 || argc > 2) return 1;

 if ((stmt = GetStatement(argv[0], stmt_name)) == (STMT*)NULL)
     return ReturnInt(retstr, SQLCA_IntCode);
 
 swa = SWA(stmt);

 /* Get the name of the stem into which to put output */
 if (argc < 2 || RXSTRLEN(argv[1]) == 0) /* No stem name specified! */
    (void)strcpy(stem_name, stmt_name);
 else if ((rc = MkIdentifier(argv[1], stem_name, sizeof(stem_name))))
    return ReturnInt(retstr, (long)rc);

 /* Describe the expression list & define output buffers (if any). */
 if ((rc = DefineExpressions(stmt)) < 0)
    return ReturnInt(retstr, (long)rc);

 /* Describe the parsed statement into REXX variables */
 for (rc = 0, i = 0; rc == 0; i++) 
   {
    rc = GetColumn(stmt, i, stem_name);
   }

 if (rc >= 0) 
   {
    len2 = sprintf(buf2,"%d", --i);
    for (p = column_attribute; *p && rc >= 0; p++) 
      {
       len1 = sprintf(buf1, "%s.COLUMN.%s.0", stem_name, *p);
       rc = SetRexxVariable(buf1, len1, buf2, len2);
      }
    rc = rc < 0 ? rc : i;
   }

 return ReturnInt(retstr, (long)rc);
}

int DBInitialise
#ifdef HAVE_PROTO
    (void)
#else
    ()
#endif
{
 return(0);
}

/*-----------------------------------------------------------------------------
 * Set up the Datatypes supported
 *----------------------------------------------------------------------------*/
static void *SetDatatypes

#ifdef HAVE_PROTO
    (DBCON *db)
#else
    (db)
    DBCON *db;
#endif

{
 RXSTRING_DT *dt=NULL;
 int          rc = 0, i=0;
 int          expr_cnt=0;
 RXSQL_HSTMT  hstmt;
 char         datatype[MAX_IDENTIFIER+1];
 SDWORD       rlen_dt=0;
 SDWORD       rlen_int=0;
 SWORD        internal=0;

 InternalFunctionPrologue("SetDatatypes");

 dt = (RXSTRING_DT *)malloc(sizeof(RXSTRING_DT)*(MAX_DATATYPES+1));

 if ((rc = SQLAllocStmt(db->hdbc, &hstmt)) != SQL_SUCCESS)
   {
    SetDBError(SQL_NULL_HENV,db->hdbc,SQL_NULL_HSTMT,NULL);
    return (NULL); /* ?? */
   }

 if ((rc = SQLGetTypeInfo(hstmt, SQL_ALL_TYPES)) != SQL_SUCCESS)
   {
    SetDBError(SQL_NULL_HENV,db->hdbc,hstmt,NULL);
    return (NULL);
   }

 if ((rc = SQLBindCol(hstmt, 1, SQL_C_CHAR, (RXSQL_SQLPOINTER)datatype, MAX_IDENTIFIER+1, &rlen_dt)) != SQL_SUCCESS)
   {
    SetDBError(SQL_NULL_HENV,db->hdbc,hstmt,NULL);
    return(NULL);
   }

 if ((rc = SQLBindCol(hstmt, 2, SQL_C_DEFAULT, (RXSQL_SQLPOINTER)&internal, sizeof(internal), &rlen_int)) != SQL_SUCCESS)
   {
    SetDBError(SQL_NULL_HENV,db->hdbc,hstmt,NULL);
    return(NULL);
   }

 while ((rc = SQLFetch(hstmt)) == SQL_SUCCESS)
   {
    dt[i].strptr = (char*)malloc(rlen_dt+1);
    memcpy(dt[i].strptr,datatype,rlen_dt);
    dt[i].strptr[strlen(datatype)] = '\0';
    dt[i].strlength = strlen(datatype);
    dt[i].internal = internal;
    i++;
   }
 dt[i].strptr = NULL;
 dt[i].strlength = 0;
 dt[i].internal = 0;
 if (SQLFreeStmt(hstmt,SQL_DROP) != SQL_SUCCESS)
   {
    SetDBError(SQL_NULL_HENV,db->hdbc,hstmt,NULL);
    return(NULL);
   }
 return((void*)dt);
}

/*-----------------------------------------------------------------------------
 * Set up the Describe Columns supported
 *----------------------------------------------------------------------------*/
static void *SetDescribeColumns

#ifdef HAVE_PROTO
    (DBCON *db)
#else
    (db)
    DBCON *db;
#endif

{
 int i=0;
 RXSTRING *dc=NULL;

 InternalFunctionPrologue("SetDescribeColumns");

 dc = (RXSTRING *)malloc(sizeof(RXSTRING)*(NUM_DESCRIBE_COLUMNS+1));
 for (i=0;i<NUM_DESCRIBE_COLUMNS;i++)
   {
    dc[i].strptr = (char*)malloc(strlen(column_attribute[i])+1);
    strcpy(dc[i].strptr,column_attribute[i]);
    dc[i].strlength = strlen(column_attribute[i]);
   }
 dc[i].strptr = NULL;
 dc[i].strlength = 0;
 return((void*)dc);
}

/*-----------------------------------------------------------------------------
 * Set up the DBMSName variable
 *----------------------------------------------------------------------------*/
static void *SetDBMSName

#ifdef HAVE_PROTO
    (DBCON *db)
#else
    (db)
    DBCON *db;
#endif

{
 SWORD    outlen=0;
 int      i=0;

 InternalFunctionPrologue("SetDBMSName");

 (void)SQLGetInfo(db->hdbc, SQL_DBMS_NAME, DBMSName, sizeof(DBMSName), &outlen);
 for (i=0;i<(int)strlen(DBMSName);i++)
   {
    if (DBMSName[i] == ' ')
       DBMSName[i] = '_';
   }
 MAKERXSTRING(RXSDBMSName,DBMSName,(ULONG)strlen(DBMSName));
 return((void*)&RXSDBMSName);
}

/*-----------------------------------------------------------------------------
 * Set up the DBMSVer variable
 *----------------------------------------------------------------------------*/
static void *SetDBMSVer

#ifdef HAVE_PROTO
    (DBCON *db)
#else
    (db)
    DBCON *db;
#endif

{
 SWORD    outlen=0;
 int      rc=0;

 InternalFunctionPrologue("SetDBMSVer");
 rc = SQLGetInfo(db->hdbc, SQL_DBMS_VER, DBMSVer, sizeof(DBMSVer), &outlen);
 MAKERXSTRING(RXSDBMSVer,DBMSVer,(ULONG)strlen(DBMSVer));
 return((void*)&RXSDBMSVer);
}

/*-----------------------------------------------------------------------------
 * Set up the SupportsTransactions variable
 *----------------------------------------------------------------------------*/
static void *SetSupportsTransactions

#ifdef HAVE_PROTO
    (DBCON *db)
#else
    (db)
    DBCON *db;
#endif

{
 SWORD    capable=0,outlen=0;
 int      rc=0;

 InternalFunctionPrologue("SetSupportsTransactions");
 rc = SQLGetInfo(db->hdbc, SQL_TXN_CAPABLE, &capable, sizeof(capable), &outlen);
 SupportsTransactions = (capable)?1L:0L;
 return((void*)&SupportsTransactions);
}

/*-----------------------------------------------------------------------------
 * Set up the SupportsSqlGetData variable
 *----------------------------------------------------------------------------*/
static void *SetSupportsSqlGetData

#ifdef HAVE_PROTO
    (DBCON *db)
#else
    (db)
    DBCON *db;
#endif

{
 InternalFunctionPrologue("SetSupportsSQLGetData");
 SupportsSqlGetData = SUPPORTSSQLGETDATA;
 return((void*)&SupportsSqlGetData);
}

#include "common.h"
