/* +------------------------------------------------------------------------+
   |  BASE64 encode/decode routines.     1996-09       Dave Lewis           |
   +------------------------------------------------------------------------+ */

#pragma strings(readonly)

#include <stdlib.h>

/*-----------------------------------------------------------------------------


   See RFC 1521 Section "5.2.  Base64 Content-Transfer-Encoding"

                            Table 1: The Base64 Alphabet

      Value Encoding  Value Encoding  Value Encoding  Value Encoding
           0 A            17 R            34 i            51 z
           1 B            18 S            35 j            52 0
           2 C            19 T            36 k            53 1
           3 D            20 U            37 l            54 2
           4 E            21 V            38 m            55 3
           5 F            22 W            39 n            56 4
           6 G            23 X            40 o            57 5
           7 H            24 Y            41 p            58 6
           8 I            25 Z            42 q            59 7
           9 J            26 a            43 r            60 8
          10 K            27 b            44 s            61 9
          11 L            28 c            45 t            62 +
          12 M            29 d            46 u            63 /
          13 N            30 e            47 v
          14 O            31 f            48 w         (pad) =
          15 P            32 g            49 x
          16 Q            33 h            50 y

-----------------------------------------------------------------------------*/

static unsigned char Base64CharSet[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
                                       "abcdefghijklmnopqrstuvwxyz"
                                       "0123456789+/";

#define BASE64PAD '='


static unsigned char Base64Bits[256] = /* Base64 characters 6 bit value */
{
// 0   1   2   3   4   5   6   7   8   9   A   B   C   D   E   F
  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,  // 0
  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,  // 1  A
  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 62, 64, 64, 64, 63,  // 2
  52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 64, 64, 64, 64, 64, 64,  // 3  S
  64,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14,  // 4
  15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 64, 64, 64, 64, 64,  // 5  C
  64, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,  // 6
  41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 64, 64, 64, 64, 64,  // 7  I
  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,  // 8
  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,  // 9  I
  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,  // A
  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,  // B
  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,  // C
  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,  // D
  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,  // E
  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64   // F

// 0   1   2   3   4   5   6   7   8   9   A   B   C   D   E   F
//64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,  // 0
//64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,  // 1  E
//64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,  // 2
//64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,  // 3  B
//64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 63, 64, 64, 64, 64,  // 4
//64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,  // 5  C
//62, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,  // 6
//64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,  // 7  D
//64, 26, 27, 28, 29, 30, 31, 32, 33, 34, 64, 64, 64, 64, 64, 64,  // 8
//64, 35, 36, 37, 38, 39, 40, 41, 42, 43, 64, 64, 64, 64, 64, 64,  // 9  I
//64, 64, 44, 45, 46, 47, 48, 49, 50, 51, 64, 64, 64, 64, 64, 64,  // A
//64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,  // B  C
//64,  0,  1,  2,  3,  4,  5,  6,  7,  8, 64, 64, 64, 64, 64, 64,  // C
//64,  9, 10, 11, 12, 13, 14, 15, 16, 17, 64, 64, 64, 64, 64, 64,  // D
//64, 64, 18, 19, 20, 21, 22, 23, 24, 25, 64, 64, 64, 64, 64, 64,  // E
//52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 64, 64, 64, 64, 64, 64   // F
};

/*-----------------------------------------------------------------------------

  Convert binary character array to Base64 character set

  length = size of the source buffer;

  target buffer must be of size ( ( length + 2 ) / 3 * 4 )

  returns length of target buffer

  --------------------------------------------------------------------------

       bits in work    chars in work     required shift
      --------------  ---------------   ----------------
             8               1                  2
            10               1                  4
            12               2                 6,0

  --------------------------------------------------------------------------

example encoding binariy structure:

unsigned char workarea[ ( sizeof( structure ) + 2 ) / 3 * 4 + 1 ];

workarea[ utBinToBase64( workarea, (unsigned char *)&structure, sizeof( structure ) ] = 0;

example decoding binariy structure:

utBase64ToBin( (unsigned char *)&structure, kvdata, strlen( kvdata ) );

-----------------------------------------------------------------------------*/

int utBinToBase64 ( unsigned char *target, unsigned char *source, int length )
{
  int work = 0;
  int mask = 0;
  int retlen = ( length + 2 ) / 3 * 4;

  while ( length-- )
  {
    work <<= 8;
    work  |= *source++;

    mask += 2;

    *target++ = Base64CharSet[ ( work >> mask ) & 0x3F ];

    if ( mask == 6 )
    {
      *target++ = Base64CharSet[ work & 0x3F ];
      mask = 0;
    }
  }

  if ( mask )
  {
    if ( mask == 2 )
    {
      *target++ = Base64CharSet[ ( work << 4 ) & 0x3F ];
      *target++ = BASE64PAD;
    }
    else
      *target++ = Base64CharSet[ ( work << 2 ) & 0x3F ];

    *target++ = BASE64PAD;
  }


  return ( retlen );
}


/*-----------------------------------------------------------------------------

  Convert Base64 character array to binary

  length = size of the source buffer;

  target buffer must be at least of size ( length / 4 * 3 )

  returns length of target buffer

  --------------------------------------------------------------------------

       bits in work    chars in work     required shift    :    mask
      --------------  ---------------   ----------------     ===========
             6               0                 n/a         :     0
            12               1                  4          :     6
            10               1                  2          :     4
             8               1                  0          :     2

-----------------------------------------------------------------------------*/

int utBase64ToBin ( unsigned char *target, unsigned char *source, int length )
{
  int work = 0;
  int mask = 0;
  int retlen = 0;

  while ( length-- )
  {
    if ( *source == BASE64PAD ) break;

    if ( Base64Bits[ *source ] < sizeof( Base64CharSet ) )
    {
      work <<= 6;
      work  |= Base64Bits[ *source ];

      if ( mask )
      {
        mask -= 2;

        *target++ = ( work >> mask ) & 0xFF;

        retlen++;
      }
      else mask = 6;
    }

    source++;
  }

  return ( retlen );
}

