/*****************************************************************************/
/*    TestCP.CMD A REXX Program to test all of the RxCP290 Functions         */
/*****************************************************************************/
/* This program demonstrates the proper use of all of the RxCP290 functions. */
/* Each Function has it's input and output parameters described here, along  */
/* with usage guidelines. Output from each function is printed to the screen */
/*****************************************************************************/

/*****************************************************************************/
/* Return Codes:                                                             */
/*                                                                           */
/* All RxCP290 functions return a value indicating success or failure. This  */
/* return code is returned via the normal REXX Function return procedure     */
/* (ie - rc = RxFunc() ).  The return code from an RxCP290 function is also  */
/* placed in the special variable CP290rc.  In all cases the returned value  */
/* will be the same in these two locations.                                  */
/*****************************************************************************/

/*****************************************************************************/
/*          RxLoadFuncsCP290 - Load the RxCP290 Functions into REXX          */
/*****************************************************************************/
/* Description:                                                              */
/*                                                                           */
/* This function loads the RxCP290 X-10 communications functions, so that    */
/* they are available to REXX programs.  Execution of this command makes     */
/* these functions available to all REXX programs in alll sessions under     */
/* OS/2.  These functions will remain available until explicitly dropped by  */
/* The RxDropFuncsCP290 function.                                            */
/*****************************************************************************/
/* Parameters:                                                               */
/*                                                                           */
/* None                                                                      */
/*****************************************************************************/
/* Returns:                                                                  */
/*                                                                           */
/* 0 if successful                                                           */
/* Non- zero returns indicate that none of the RxCP290 functions have been   */
/* loaded.                                                                   */
/*****************************************************************************/
/* Example:                                                                  */

                                               /* Load RxLoadFuncsCP290      */
call RxFuncAdd 'RxLoadFuncsCP290', 'RxCP290', 'RxLoadFuncsCP290'
call RxLoadFuncsCP290                          /* Call the RxCP290 function  */
say 'RxLoadFuncsCP290, CP290rc:   'CP290rc     /* Show results               */

/*****************************************************************************/
/*                RxInitCP290 - Initialize Communication Port                */
/*****************************************************************************/
/* Description:                                                              */
/*                                                                           */
/* RxInitCP290 initializes the serial port for communications with the CP290.*/
/* It initializes a Com Port Handle, which is used by all of the RxCP290     */
/* functions to determine which serial port to communicate with.  This Comm  */
/* Handle variable should be initialized in the REXX environment before      */
/* using this function.                                                      */
/*****************************************************************************/
/* Parameters:                                                               */
/*                                                                           */
/* This function takes two parameters. The first parameter is a string with  */
/* the name of the serial port the CP290 is connected to.  For this example, */
/* the CP290 is connected to COM2, so the parameter is 'COM2'. Note that     */
/* This parameter must be in all caps.                                       */
/*                                                                           */
/* The second parameter is the name of a REXX variable to store the Com Port */
/* Handle in.  This variable should have been initialized prior to calling   */
/* function.                                                                 */
/*****************************************************************************/
/* Returns:                                                                  */
/*                                                                           */
/* This function returns 0 when successful. Non-zero results indicate the    */
/* was not opened.                                                           */
/*****************************************************************************/
/* Example:                                                                  */

CP290 = ''                                    /* Init Com Port Handle var    */
rc = RxInitCP290('COM2', 'CP290')             /* Call the function           */
say 'RxInitCP290:   'rc', CP290rc:   'CP290rc /* print the results           */

/*****************************************************************************/
/*                   RxDiagCP290 - Perform CP290 Self-Test                   */
/*****************************************************************************/
/* Description:                                                              */
/*                                                                           */
/* This function commands the CP290 to perform it's internal diagnostic      */
/* routines.  Execution of these routines will take about 10 seconds.   All  */
/* information in the CP290 will be erased as a result of the execution of   */
/* this function.                                                            */
/*****************************************************************************/
/* Parameters:                                                               */
/*                                                                           */
/* This function takes a single parameter, the Com Port Handle produced by   */
/* the RxInitCP290 function.                                                 */
/*****************************************************************************/
/* Returns:                                                                  */
/*                                                                           */
/* This function returns zero if successful.  A return code of 1 indicates   */
/* a CP290 diagnostics failure not related to the serial communications.  A  */
/* return code of less than zero indicates a communications failure, which   */
/* may or may not be due to CP290 failure.                                   */
/*****************************************************************************/
/* Example:                                                                  */
 
rc = RxDiagCP290(CP290)                        /* Call Function              */
say 'RxDiagCP290:   'rc', CP290rc:   'CP290rc  /* Print return code          */

/*****************************************************************************/
/*                RxSetHCodeCP290 - Set CP290 Base House Code                */
/*****************************************************************************/
/* Description:                                                              */
/*                                                                           */
/* RxSetHcodeCP290 sets the CP290's Base Housecode.  All information in the  */
/* CP290 will be erased as a result of this function's execution.            */
/*****************************************************************************/
/* Parameters:                                                               */
/*                                                                           */
/* This function takes two parameters. The first parameter is the Comm Port  */
/* handle.                                                                   */
/*                                                                           */
/* The second parameter is a REXX string containing the new Housecode letter.*/
/* Valid housecodes are in the range "A" to "P".                             */
/*****************************************************************************/
/* Returns:                                                                  */
/*                                                                           */
/* RxSetHcode returns the CP290 ack status after execution. This value will  */
/* be zero if no information has been downloaded to the CP290. It will be a  */
/* 1 after download of any command.  This can be used to determine if the    */
/* CP290 has been powered off. A return code of -1 indicates a communication */
/* error with the CP290.                                                     */
/*****************************************************************************/
/* Example:                                                                  */
 
rc = RxSetHCodeCP290(CP290, "D")               /* Set Housecode "D"          */
say 'RxSetHCodeCP290: 'rc                      /* Print the results          */

/*****************************************************************************/
/*                 RxSetTimeCP290 - Set CP290 Base House Code                */
/*****************************************************************************/
/* Description:                                                              */
/*                                                                           */
/* RxSetTimeCP290 sets the CP290's internal clock from the host computer's   */
/* internal clock.                                                           */
/*****************************************************************************/
/* Parameters:                                                               */
/*                                                                           */
/* This function takes one parameter, the Comm Port handle                   */
/*****************************************************************************/
/* Returns:                                                                  */
/*                                                                           */
/* RxSetTime returns the CP290 ack status after execution. This value will   */
/* be zero if no information has been downloaded to the CP290. It will be a  */
/* 1 after download of any command.  This can be used to determine if the    */
/* CP290 has been powered off. A return code of -1 indicates a communication */
/* error with the CP290.                                                     */
/*****************************************************************************/
/* Example:                                                                  */

rc = RxSetTimeCP290(CP290)                     /* Set current Computer Time  */
say 'RxSetTimeCP290: ' rc                      /* Print the results          */

/*****************************************************************************/
/*                RxGetTimeCP290 - Get CP290 Time and House Code             */
/*****************************************************************************/
/* Description:                                                              */
/*                                                                           */
/* RxGetTimeCP290 returns the CP290's internal clock time and House Code.    */
/*****************************************************************************/
/* Parameters:                                                               */
/*                                                                           */
/* This function takes two parameters. The first parameter is the Comm Port  */
/* handle.                                                                   */
/*                                                                           */
/* The second parameter is the stem name of a REXX compound variable to      */
/* receive the results. The format of this stem variable is shown below, in  */
/* section that prints the results.                                          */
/*****************************************************************************/
/* Returns:                                                                  */
/*                                                                           */
/* RxGetTimeCP290 returns the number of characters received.  This should be */
/* equal to 12 for a normal call.                                            */
/*****************************************************************************/
/* Example:                                                                  */

rc = RxGetTimeCP290(CP290, 'result')    /* Get CP290 time                    */
say 'RxGetTimeCP290: ' rc               /* Print return code                 */
 
say 'Status   - Result.0: 'result.0     /* CP290 Ack Status                  */
say 'Minutes  - Result.1: 'result.1     /* Minutes:  valid 00 - 59           */
say 'Hours    - Result.2: 'result.2     /* Hours:    valid 00 - 23           */
say 'Day Week - Result.3: 'result.3     /* Day of Week: in "MON","TUE","WED" */
                                        /* "THU","FRI","SAT","SUN"           */
say 'Base     - Result.4: 'result.4     /* Base Housecode: valid "A" - "P"   */

/*****************************************************************************/
/*                 RxDirCmdCP290 - Send CP290 direct command                 */
/*****************************************************************************/
/* Description:                                                              */
/*                                                                           */
/* RxDirCmdCP290 sends a direct command to a single X-10 module              */
/*****************************************************************************/
/* Parameters:                                                               */
/*                                                                           */
/* This function takes five parameters. The first parameter is the Comm Port */
/* handle.                                                                   */
/*                                                                           */
/* The second parameter is the dim level, for the DIM function. Valid Values */
/* range from 00(dimmest) to 15(brightest).                                  */
/*                                                                           */
/* The third parameter is the function code. This can be one of "ON", "OFF", */
/* or "DIM".                                                                 */
/*                                                                           */
/* The fourth parameter is a REXX string containing the new Housecode letter.*/
/* Valid housecodes are in the range "A" to "P".                             */
/*                                                                           */
/* Finally, the last parameter is the module number to affect. Valid values  */
/* are in the range 1 through 16.                                            */
/*****************************************************************************/
/* Returns:                                                                  */
/*                                                                           */
/* RxDirCmd returns the CP290 ack status after execution. This value will    */
/* be zero if no information has been downloaded to the CP290. It will be a  */
/* 1 after download of any command.  This can be used to determine if the    */
/* CP290 has been powered off. A return code of -1 indicates a communication */
/* error with the CP290.                                                     */
/*****************************************************************************/
/* Example:                                                                  */

rc = RxDirCmdCP290(CP290, 8, "ON" ,"D", 4)     /* Turn on Module D4          */
say 'RxDirCmdCP290: 'rc', CP290rc:   'CP290rc

/*****************************************************************************/
/*                RxCmdUploadCP290 - Get CP290 execution report              */
/*****************************************************************************/
/* Description:                                                              */
/*                                                                           */
/* Every time the CP290 executes a command to an X-10 module, it sends a     */
/* Command Upload to the host computer. This is true whether the action was  */
/* caused by a RxDirCmdCP290 REXX Function, an event execution in the CP290, */
/* or manual control from the CP290 keypad.  This function receives this     */
/* report, and formats the uploaded information in a REXX compound variable. */
/*****************************************************************************/
/* Parameters:                                                               */
/*                                                                           */
/* This function takes two parameters. The first parameter is the Comm Port  */
/* handle.                                                                   */
/*                                                                           */
/* The second parameter is the stem name of a REXX compound variable to      */
/* receive the results. The format of this stem variable is shown below, in  */
/* section that prints the results.                                          */
/*****************************************************************************/
/* Usage Notes:                                                              */
/*                                                                           */
/* In this example, the call to RxCmdUploadCP290 is contained within a do    */
/* while loop. This waits until the command is received by the REXX program  */
/* before continuing.  If the com  port is left open for any period of time, */
/* this function should be called periodically to check to see if any events */
/* that would cause a report (such as manual key presses, or timer events).  */
/* This will keep the serial input and output buffers synchronized.          */
/*****************************************************************************/
/* Returns:                                                                  */
/*                                                                           */
/* RxCmd UploadCP290 returns the number of characters received.  This should */
/* be equal to 12 for a normal call. If no command upload has been received, */
/* this function will return 0.                                              */
/*****************************************************************************/
/* Example:                                                                  */

do while rc < 5
   rc = RxCmdUploadCP290(CP290, 'result')
end
 
say 'Status       - Result.0:   'result.0     /* CP290 ack status           */
say 'Hcode        - Result.1:   'result.1     /* House code: "A"-"P"        */
say 'Function     - Result.2:   'result.2     /* Function: "ON","OFF","DIM" */
say '# of Modules - Result.3.0: 'result.3.0   /* Number of modules affected */
do i = 1 to result.3.0
   say 'Module 'i'     - Result.3.'i': 'result.3.i /* Module number         */
end	
say 'Base          - Result.4:   'result.4     /* CP290 Base Housecode       */

/*****************************************************************************/
/*       RxDirCmdManyCP290 - Send Direct Command to Many X-10 Modules        */
/*****************************************************************************/
/* Description:                                                              */
/*                                                                           */
/* RxDirCmdManyCP290 sends a direct command to a series of X-10 modules      */
/*****************************************************************************/
/* Parameters:                                                               */
/*                                                                           */
/* This function takes five parameters. The first parameter is the Comm Port */
/* handle.                                                                   */
/*                                                                           */
/* The second parameter is the dim level, for the DIM function. Valid Values */
/* range from 00(dimmest) to 15(brightest).                                  */
/*                                                                           */
/* The third parameter is the function code. This can be one of "ON", "OFF", */
/* or "DIM".                                                                 */
/*                                                                           */
/* The fourth parameter is a REXX string containing the new Housecode letter.*/
/* Valid housecodes are in the range "A" to "P".                             */
/*                                                                           */
/* Finally, the last parameter is the stem name of a REXX compound variable  */
/* that contains a list of the modules to command. The format of this        */
/* variable is demonstrated in the example below.                            */
/*****************************************************************************/
/* Returns:                                                                  */
/*                                                                           */
/* RxDirCmdMany returns the CP290 ack status after execution. This value will*/
/* be zero if no information has been downloaded to the CP290. It will be a  */
/* 1 after download of any command.  This can be used to determine if the    */
/* CP290 has been powered off. A return code of -1 indicates a communication */
/* error with the CP290.                                                     */
/*****************************************************************************/
/* Example:                                                                  */

dcom.0 = 5                             /* 5 modules to command               */
dcom.1 = 4                             /* 1st module - Module  4             */
dcom.2 = 5                             /* 2nd module - Module  5             */
dcom.3 = 16                            /* 3rd module - Module 16             */
dcom.4 = 10                            /* 4th module - Module 10             */
dcom.5 = 7                             /* 5th module - Module  7             */

rc = RxDirCmdManyCP290(CP290, 8, "OFF" ,"D", "dcom")  /* call the function   */
say 'RxDirCmdManyCP290: 'rc', CP290rc:   'CP290rc

do while rc < 5
   rc = RxCmdUploadCP290(CP290, 'result')
end

say 'Status        - Result.0:   'result.0
say 'Hcode         - Result.1:   'result.1
say 'Function      - Result.2:   'result.2
say '# of Modules  - Result.3.0: 'result.3.0
do i = 1 to result.3.0
   say 'Module 'i'      - Result.3.'i': 'result.3.i
end	
say 'Base          - Result.4:   'result.4

/*****************************************************************************/
/*            RxDwnLdEvtCP290 - Download a Timer Event to the CP290          */
/*****************************************************************************/
/* Description:                                                              */
/*                                                                           */
/* RxDwnLdEvtCP290 downloads a Timer event to the CP290 for later execution. */
/*****************************************************************************/
/* Parameters:                                                               */
/*                                                                           */
/* This function takes three parameters.  The first parameter is the Comm    */
/* Port handle.                                                              */
/*                                                                           */
/* The second parameter is the event number.  Valid Values range from 0 to   */
/* 127.                                                                      */
/*                                                                           */
/* Finally, the last parameter is the stem name of a REXX compound variable  */
/* that contains the timer event information. The format of this variable is */
/* demonstrated in the example below. Note the similarity of this variable   */
/* format to the one used by RxGetEventsCP290.                               */
/*****************************************************************************/
/* Returns:                                                                  */
/*                                                                           */
/* RxDwnLdEvt returns the CP290 ack status after execution. This value will  */
/* be zero if no information has been downloaded to the CP290. It will be a  */
/* 1 after download of any command.  This can be used to determine if the    */
/* CP290 has been powered off. A return code of -1 indicates a communication */
/* error with the CP290.                                                     */
/*****************************************************************************/
/* Example:                                                                  */

event.0   = 0                /* Not used - here for compatibility with       */
                             /* RxGetEventsCP290 Compound Variable.          */

event.1   = "NORMAL"         /* Timer Mode - valid entries are "NORMAL",     */
                             /* "SECURITY","TODAY","TOMORROW", and "CLEAR"   */

event.2.0 = 4                /* Number of Days for Event: range 1-7          */

event.2.1 = "WED"            /* First Day - one of "MON","TUE","WED","THU",  */
event.2.2 = "THU"            /* "SAT", or "SUN".                             */
event.2.3 = "FRI"
event.2.4 = "MON"

event.3   = 15               /* Event Time - Hours:   Range  0-23            */

event.4   = 25               /* Event Time - Minutes: Range  0-59            */

event.5.0 = 4                /* Number of Modules for Event: Range 1-16      */
event.5.1 = 4                /* First Module: Range 1-16                     */
event.5.2 = 3                /* 2nd Module                                   */
event.5.3 = 16               /* 3rd Module                                   */
event.5.4 = 10               /* 4th Module                                   */

event.6   = "D"              /* Housecode: Range "A"-"P"                     */

event.7   = 7                /* Dim Level: Range 0(dimmest)-15(brightest)    */

event.8   = "DIM"            /* Function - One of "ON","OFF","DIM            */

rc = RxDwnldEvtCP290(CP290, 1,event)  /* call the function                   */
say 'RxDwnldEvtCP290:  'rc

/*****************************************************************************/
/*                RxGetEventsCP290 - Get CP290 Timer Events                  */
/*****************************************************************************/
/* Description:                                                              */
/*                                                                           */
/* This function gets all the Timer Event information from the CP290, and    */
/* makes it available to the calling REXX program in the form of a large     */
/* compound variable.                                                        */
/*****************************************************************************/
/* Parameters:                                                               */
/*                                                                           */
/* This function takes two parameters. The first parameter is the Comm Port  */
/* handle.                                                                   */
/*                                                                           */
/* The second parameter is the stem name of a REXX compound variable to      */
/* receive the results. The format of this compound variable is shown below, */
/* in the section that prints the results. This compound variable consists   */
/* of several structures in the same format as the compound variable used by */
/* RxDwnLdEvtCP290, for convenience.                                         */
/*****************************************************************************/
/* Returns:                                                                  */
/*                                                                           */
/* RxGetEventsCP290 returns the number of characters received.               */
/*****************************************************************************/
/* Example:                                                                  */

rc = RxGetEventsCP290(CP290, 'bigevt')    /* Call the function               */
say 'RxGetEventsCP290:  'rc

do i=0 to 127
 if BigEvt.i.0 = 1 then Do                /* If this value = 1, there is a   */
                                          /* corresponding event number.     */
                                          /* Otherwise, value is 0           */ 
  say 'BigEvt.'i'.0  :  'BigEvt.i.0       /* 1 = event, 0 = no event         */
  say 'BigEvt.'i'.1  :  'BigEvt.i.1       /* Timer Mode Sel - as RxDwnLdEvt  */
  say 'BigEvt.'i'.2.0:  'BigEvt.i.2.0     /* Number of Days - as RxDwnLdEvt  */
  Do j = 1 to BigEvt.i.2.0
   say 'BigEvt.'i'.2.'j':  'BigEvt.i.2.j  /* Day info - as RxDwnLdEvt        */
  end
  say 'BigEvt.'i'.3  :  'BigEvt.i.3       /* Hours:                          */
  say 'BigEvt.'i'.4  :  'BigEvt.i.4       /* Minutes:                        */
  say 'BigEvt.'i'.5.0:  'BigEvt.i.5.0     /* Number of Modules               */
  Do j = 1 to BigEvt.i.5.0
   say 'BigEvt.'i'.5.'j':  'BigEvt.i.5.j  /* Module number                   */
  end
  say 'BigEvt.'i'.6  :  'BigEvt.i.6       /* Housecode                       */
  say 'BigEvt.'i'.7  :  'BigEvt.i.7       /* Dim Level                       */
  say 'BigEvt.'i'.8  :  'Bigevt.i.8       /* Function                        */
  end
end

/*****************************************************************************/
/*                   RxCloseCP290 - Close CP290 Serial Port                  */
/*****************************************************************************/
/* Description:                                                              */
/*                                                                           */
/* RxCloseCP290 closes the serial port specified by the Com Port Handle,     */
/* freeing the port for use by other OS/2 applications.                      */
/*****************************************************************************/
/* Parameters:                                                               */
/*                                                                           */
/* This function takes one parameter, the Comm Port handle                   */
/*****************************************************************************/
/* Returns:                                                                  */
/*                                                                           */
/* Returns 0 if successful                                                   */
/*****************************************************************************/
/* Example:                                                                  */

rc = RxCloseCP290(CP290)
say 'RxCloseCP290:  'rc', CP290rc:   'CP290rc

/*****************************************************************************/
/*          RxDropFuncsCP290 - Drop the RxCP290 Functions from REXX          */
/*****************************************************************************/
/* Description:                                                              */
/*                                                                           */
/* This function drops the RxCP290 X-10 communications functions, so that    */
/* they are no longer available to REXX programs.  Execution of this command */
/* makes these functions unavailable to all REXX programs in all sessions    */
/* in OS/2.  These functions will remain unavailable until explicitly loaded */
/* by the RxLoadFuncsCP290 function.                                         */
/*****************************************************************************/
/* Parameters:                                                               */
/*                                                                           */
/* None                                                                      */
/*****************************************************************************/
/* Returns:                                                                  */
/*                                                                           */
/* 0 if successful                                                           */
/*****************************************************************************/
/* Example:                                                                  */

call RxDropFuncsCP290
say 'RxDropFuncsCP290, CP290rc:  'CP290rc

