/*
 * subcom.c -- Defines a simple REXX subcommand handler which prints its
 *             input string and converts it to uppercase.  In real life
 *             you'd actually want to parse the input string and do
 *             something useful based on it....
 */

#define INCL_REXXSAA

#include <os2.h>
#include <rexxsaa.h>

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "subcom.h"

/*
 * TEST -- A REXX subcommand handler.  It prints the command line to
 *         standard out and then returns the uppercase version of it.
 *
 *
 *  Return values:  If you return 0 from this function, the string
 *                  you assign to result will be assigned to the
 *                  RC variable in REXX.  If you return non-zero,
 *                  the return value will be converted to a signed
 *                  integer and then to a string and assigned to RC.
 */

RexxSubcomHandler Test;

ULONG Test( PRXSTRING command, PUSHORT flags, PRXSTRING result )
  {
    PVOID mem = NULL;
    ULONG i;

    /* Check for 'error' and 'failure', in which case we want
       to simulate what happens when an error/failure occurs */

    if( command->strlength > 0 ){
        if( stricmp( command->strptr, "error" ) == 0 ){
            *flags = RXSUBCOM_ERROR;
            return( -1 );
        } else if( stricmp( command->strptr, "failure" ) == 0 ){
            *flags = RXSUBCOM_FAILURE;
            return( -1 );
        }
    }

    /* Print out the command string.... it's OK to use printf
       because REXX guarantees that the string ends with a nullchar,
       although if you're expecting embedded nulls you should go
       by the string length */

    printf( "Handler 'Test' received command: '" );

    if( command->strptr != NULL && command->strlength > 0 ){
        printf( "%s", command->strptr );
    }

    printf( "'\n" );

    /* Allocate new buffer if necessary... don't use malloc or any
       other library routine -- use DosAllocMem only */

    if( command->strlength > result->strlength ){
        if( DosAllocMem( &mem, command->strlength,
                         PAG_COMMIT | PAG_READ | PAG_WRITE ) != 0 ){

            *flags = RXSUBCOM_ERROR;
            return( -1 );
        }

        result->strptr = mem;
    }

    /* Copy and convert... */

    for( i = 0; i < command->strlength; ++i ){
        result->strptr[i] = toupper( command->strptr[i] );
    }

    result->strlength = command->strlength;

    /* Set OK flag and return 0, meaning that the value we just
       copied into result will be assigned to RC */

    *flags = RXSUBCOM_OK;

    return( 0 );
  }

/*
 * Handler registration... before starting any REXX macros, we have
 * to register our subcommand handlers with the interpreter.  We just
 * keep a table of functions and register each one in sequence.
 */

typedef struct {
    PSZ name;
    PFN function;
} rxhandler_entry, *rxhandler_entry_p;

static rxhandler_entry REXXHandlers[] =
  {
    { "TEST",  (PFN) Test  },
    { NULL,    NULL  }
  };

/*
 * RegisterREXXHandlers -- Register our REXX command handler(s).  We
 *                         use RexxRegisterSubcomExe to do the
 *                         registration.
 *
 *  NOTE: RexxRegisterSubcomExe takes three parms, one is the name of
 *        the handler to register (the name should be all uppercase),
 *        the second is the address of the function to call, and the
 *        third is the address of an 8-byte block of user-defined
 *        data to store with the handler.  The function does not need
 *        to be exported.  After registration the function is only
 *        available to the process that registered it.
 *
 *        In contrast, RexxRegisterSubcomDll requires that the function
 *        be exported from a DLL and is then made available to all REXX
 *        programs in any process.
 *
 */

BOOL RegisterREXXHandlers( void )
  {
    rxhandler_entry_p ptr;

    for( ptr = REXXHandlers; ptr->name != NULL; ++ptr ){
        RexxRegisterSubcomExe( ptr->name, ptr->function, NULL );
    }

    return( TRUE );
  }

/*
 * DeregisterREXXHandlers -- Deregister each subcommand handler.  Strictly
 *                           speaking, this is not necessary since the
 *                           registrations are freed once the process exits,
 *                           but it's always good practice to clean up nicely.
 */

void DeregisterREXXHandlers( void )
  {
    rxhandler_entry_p ptr;

    for( ptr = REXXHandlers; ptr->name != NULL; ++ptr ){
        RexxDeregisterSubcom( ptr->name, NULL );
    }
  }
