#include <stdio.h>
/*
 *  Copyright 1995 Microsoft Corporation. All rights reserved.
 *  Developed by Ataman Software, Inc., ftp://rmii.com/pub2/ataman,
 *       info@ataman.com
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Modified by Patrick McPhee to be a loadable library, 1997/08/07
 * Changes copyright 1997, Patrick McPhee
 * $Header: G:/ptjm/rexx/w32funcs/RCS/w32funcs.c 1.3 1998/06/26 20:40:59 pmcphee Exp $
 */

#define STRICT
#include <windows.h>
#include <time.h>
#include <malloc.h>

/* use the interface header, rexxsaa.h, instead of the internal header,
 * rexx.h */
#define INCL_RXSHV
#define INCL_RXFUNC
#include "rexxsaa.h"

#include "w32funcs.h"

static HKEY GetHKEY(PRXSTRING str);
static unsigned char *eLog;

/* ataman code original raised a syntax error whenever anything went wrong,
 * for instance, if you tried to query a registry value which wasn't set.
 * I think you ought to be able to pass an invalid argument without it being
 * treated as a syntax error.  It does raise the question `how do we tell if
 * there's no value for a registry key?'  I return an empty string from from
 * the functions that return values, if there is no key matching the value.
 * If that's not good enough for you, I also set rc = 0 if the key existed,
 * and 1 if it didn't.  I'm not sure how usual it is to set rc from a
 * function, as opposed to an external command, but the ataman functions
 * don't really give a good way to figure it out.
 */

/* in the following lists of functions, I use the following conventions:
 *  key refers to a handle returned by one of the open functions, or
 *   to a key with a standard name, HKEY_LOCAL_MACHINE, HKEY_CLASSES_ROOT,
 *   HKEY_CURRENT_USER, or HKEY_USERS. These names can be in any case, and
 *   the `HKEY_' part is optional, since it's redundant.
 *  parameters in brackets are optional. parameters in braces are strings
 *   containing the name of a variable which will be used to return a value
 *  functions which return a key will return 0 to indicate failure.
 *   functions which return an arbitrary value will set rc if there's a
 *   problem of some sort
 *  functions which don't return any useful values return 0 for success and 1
 *   for failure
 *  these calls are meant to be upwardly compatible from the ataman calls
 *  but there is no warranty
 */

/* functions defined in this file:
 *  w32regopenkey(key, [subkeyname]) -> key
 *  w32regcreatekey(key, subkeyname) -> key
 *  w32regconnectregistry(hostname,key) -> key
 *  w32regclosekey(key) -> 0 (success) or 1 (failure)
 *  w32regqueryvalue(key, entry, [{type}]) -> the value (sets rc to 1 if no such entry)
 *  w32regqueryvaluetype(key, entry) -> type (use optional arg to w32regqueryvalue instead)
 *  w32regsetvalue(hkey, entry, type, data) -> 0 or 1
 *  w32regdeletekey(hkey, subkeyname) -> 0 or 1
 *  w32regenumkey(hkey, [index], [{stem}]) -> keyname, 0 or 1 (sets rc)
 *  w32regenumvalue(hkey, [index], [{stem}]) -> value, 0 or 1 (sets rc)
 *  w32regflushkey(hkey) -> 0 or 1
 *  w32reggetkeysecdesc(hkey) -> secdesc (sets rc)
 *  w32regsetkeysecdesc(hkey, secdesc) -> 0 or 1
 *  w32regqueryinfokey(hkey, infokey, [{stem}]) -> desired info, 0 or 1 (sets rc)
 *  w32regsavekey(hkey, filename) -> 0 or 1
 *  w32regrestorekey(hkey, filename) -> 0 or 1
 *  w32regloadkey(hkey, subkey, filename) -> 0 or 1
 *  w32regunloadkey(hkey, subkey) -> 0 or 1
 *
 *  w32openeventlog(host, source) -> handle, or 0
 *  w32closeeventlog(handle) -> 0 or 1
 *  w32getnumberofeventlogrecords(handle) -> the number, or -1
 *  w32openbackupeventlog([hostname], filename) -> handle, or 0
 *  w32backupeventlog(handle, filename) -> 0 or 1
 *  w32backupeventlog(handle, [filename]) -> 0 or 1
 *  w32findeventlogentry(handle, recnum, [{stem}]) -> 0 or 1
 *  w32geteventid() -> id (sets rc)
 *  w32geteventtype() -> type (sets rc)
 *  w32geteventcategory() -> category (sets rc)
 *  w32geteventnumstrings() -> num strings (sets rc)
 *  w32geteventtimewritten() -> time written (sets rc)
 *  w32geteventtimegenerated() -> time generated (sets rc)
 *  w32geteventstring(index) -> string (sets rc)
 *  w32geteventdata() -> data (sets rc)
 *  w32writeeventlog([hostname], source, [eventtype], [category], [eventid], [data], [string1, ...]) -> 0 or 1
 *  w32writeevent({stem}) -> 0 or 1
 */

static char * type_name(DWORD dwType);
static DWORD type_type(const char * const theType);


/* open a registry key.
 *  arg1: the parent key (an hkey or one of the standard names)
 *  arg2: the key to open
 */
rxfunc(w32regopenkey)
{
	char *subkeyname = NULL;
	HKEY hkey, hkeyRet;
	DWORD err;
	int rc = 0;

	checkparam(1, 2) ;

	hkey = GetHKEY(argv);

	if (argc > 1) {
		rxstrdup(subkeyname, argv[1]);
	}

	if ((err=RegOpenKeyEx(hkey, subkeyname, 0, MAXIMUM_ALLOWED, &hkeyRet)) != ERROR_SUCCESS) {
		result_zero();
	}
	else {
	    result->strlength = sprintf(result->strptr, "%lx", (unsigned long)hkeyRet);
	}
	 
	return rc;
}


/* hkey = w32_regcreatekey(hkey, subkeyname) */
rxfunc(w32regcreatekey)
{
	char *subkeyname = NULL;
	HKEY hkey, hkeyRet;
	DWORD err, disp;

	checkparam(2, 2) ;

	hkey = GetHKEY(argv);

	rxstrdup(subkeyname, argv[1]);

	if ((err=RegCreateKeyEx(hkey, subkeyname, 0, "", REG_OPTION_NON_VOLATILE,
		MAXIMUM_ALLOWED, NULL, &hkeyRet, &disp)) != ERROR_SUCCESS) {
		result_zero();
	}
	else {
	    result->strlength = sprintf(result->strptr, "%lx", (unsigned long)hkeyRet);
	}
	 
	return 0;
}


rxfunc(w32regconnectregistry)
{
	/* hkey = w32_regconnectregistry(hostname, hkey) */
	char *hostname = NULL;
	HKEY hkey, hkeyRet;
	DWORD err;

	checkparam(2, 2) ;

	rxstrdup(hostname, argv[0]);

	hkey = GetHKEY(argv+1);

	if ((err=RegConnectRegistry(hostname, hkey, &hkeyRet)) != ERROR_SUCCESS) {
		result_zero();
	}
	 
	result->strlength = sprintf(result->strptr, "%lx", (unsigned long)hkeyRet);
	return 0;
}


rxfunc(w32regclosekey)
{
	/* call w32RegCloseKey hkey */
	HKEY hkey;
	char * key;

	checkparam(1, 1) ;

	rxstrdup(key, argv[0]);

	sscanf(key, "%lx", (unsigned long *)&hkey);

	if (RegCloseKey(hkey) != ERROR_SUCCESS) {
	    result_one();
	}
	else {
	    result_zero();
	}

	return 0;
}



rxfunc(w32regqueryvalue)
{
	char *valuename = NULL;
	HKEY hkey;
	DWORD cbData, dwType;
	unsigned char *byData = NULL;
	register int i;

	checkparam(2, 3) ;

	hkey = GetHKEY(argv);

	rxstrdup(valuename, argv[1]);

	result->strlength = 0;
	if (RegQueryValueEx(hkey, valuename, NULL, &dwType, NULL, &cbData) != ERROR_SUCCESS) {
		rc_one();
	}
	else {
	    byData = alloca(cbData);

	    if (!byData) {
		    rc_one();
		    return 0;
	    }

	    if (RegQueryValueEx(hkey, valuename, NULL, &dwType, byData, &cbData) != ERROR_SUCCESS) {
		    rc_one();
		    return 0;
	    }

	    if (argc > 2)
		setavar(argv+2, type_name(dwType));

	    switch(dwType) {
	    case REG_DWORD:
		    {
		    result->strlength = sprintf(result->strptr, "%lu", *((unsigned long *)byData));
		    break;
		    }
	    case REG_EXPAND_SZ:
	    case REG_SZ:	
		    {
		    /* make sure there's enough space. the string in byData
		     * includes a terminating null which we don't want */
                    if (cbData) {
                       rxresize(result, cbData - 1);
                       memcpy(result->strptr, byData, result->strlength);
                    }
                    else
                       result->strlength = 0;
		    break;
		    }
	    default:	/* some kind of binary data */
		    {
		    rxresize(result, cbData*2+1);

		    result->strlength = 0;
		    for (i = 0; i < cbData; i++) {
			    result->strlength += sprintf(result->strptr+result->strlength, "%02x", (unsigned) byData[i]);
		    }
		    break;
		    }
	    }
	    rc_zero();
	}

	return 0;
}


/* return just the type of the registry entry. Since you almost never
 * want to do this without also querying the value of the entry, I added
 * an extra (optional) argument argument to w32regqueryvalue, which
 * holds the type. I leave this function for compatibility */
rxfunc(w32regqueryvaluetype)
{
	char *valuename = NULL;
	HKEY hkey;
	DWORD dwType;

	checkparam(2, 2) ;

	hkey = GetHKEY(argv);

	rxstrdup(valuename, argv[1]);

	if (RegQueryValueEx(hkey, valuename, NULL, &dwType, NULL, NULL) != ERROR_SUCCESS) {
		rc_one();
	}
	else {
	    rc_zero();
	    strcpy(result->strptr, type_name(dwType));
	    result->strlength = strlen(result->strptr);
	}

	return 0;
}



rxfunc(w32regsetvalue)
{
	char *valuename = NULL;
	char *type = NULL;
	char *data = NULL;
	HKEY hkey;
	DWORD dwType, cbData;
	unsigned char *byData = NULL;

	checkparam(4, 4) ;

	hkey = GetHKEY(argv);

	rxstrdup(valuename, argv[1]);
	rxstrdup(type, argv[2]);
	rxstrdup(data, argv[3]);

	dwType = type_type(type);

	/* assume success */
	result_zero();

	switch(dwType) {
	case REG_DWORD:
		{
		byData = alloca(sizeof (DWORD));
		if (!byData) {
			result_one();
		}
		else {
		    sscanf(data, "%lu", (unsigned long *)byData);
		    cbData = sizeof (DWORD);
		}
		break;
		}
	case REG_EXPAND_SZ:
	case REG_SZ:	
		{
		cbData = argv[3].strlength+1;
		byData = data;
		break;
		}
	case 0: result_one(); break;
	default:	/* some kind of binary data */
		{
		DWORD dw;
		char *p;

		cbData = argv[3].strlength;
		if ((cbData%2) != 0) {
			result_one();
		}
		else {
		    cbData /= 2;

		    byData = alloca(cbData);
		    if (!byData) {
			result_one();
		    }
		    else
		    for (p=data, dw=0; dw<cbData; dw++, p+=2) {
			char buf[3];
			unsigned int x;
			buf[0] = *p;
			buf[1] = *(p+1);
			buf[2] = '\0';
			if (sscanf(buf, "%02x", &x) != 1) {
				result_one();
			}
			byData[dw] = (unsigned char)x;
		    }
		}
		break;
		}
	}

	if (result->strptr[0] == '0')
	    if (RegSetValueEx(hkey, valuename, 0, dwType, byData, cbData) != ERROR_SUCCESS) {
		result_one();
	    }

	return 0;
}


/* w32expandenvironmentstrings is dropped because the functionality is
 * provided by, eg, value('PATH',,'ENVIRONMENT');
 */


rxfunc(w32regdeletekey)
{
	char *subkeyname = NULL;
	HKEY hkey;
	DWORD err;

	checkparam(2, 2) ;

	hkey = GetHKEY(argv);

	rxstrdup(subkeyname, argv[1]);

	if ((err=RegDeleteKey(hkey, subkeyname)) != ERROR_SUCCESS) {
		result_one();
	}
	else {
		result_zero();
	}
	 
	return 0;
}


rxfunc(w32regdeletevalue)
{
	char *valuename = NULL;
	HKEY hkey;

	checkparam(2, 2) ;

	hkey = GetHKEY(argv);

	rxstrdup(valuename, argv[1]);

	if (RegDeleteValue(hkey, valuename) != ERROR_SUCCESS) {
		result_one();
	}
	else {
		result_zero();
	}
	 
	return 0;
}


/* regenumkey either returns the name of the ith key under key,
 * (original syntax), or it gets all the key names and sticks them in
 * a stem variable (my syntax). If you specify the index argument,
 * it returns an empty string if there is no such key. If you don't
 * it returns 0 if it could get all the keys, or 1 otherwise.
 * It's an error to specify neither the index nor the stem name */
rxfunc(w32regenumkey)
{
	HKEY hkey;
	DWORD index;
	char keybuf[MAX_PATH+1];
	char * indptr;

	/* you must specify at least one of index or stem */
	if (argc == 1 || argc > 3 || (argc == 2 && argv[1].strptr == NULL) ||
	    (argc == 3 && argv[2].strptr == NULL && argv[1].strptr == NULL))
	    return BADARGS;

	hkey = GetHKEY(argv);

	if (!hkey) {
	    if (!argv[1].strptr)
		result_one();
	    rc_one();
	    return 0;
	}

	if (argv[1].strptr) {
	    rxstrdup(indptr, argv[1]);
	    if (sscanf(indptr, "%lu", &index) != 1) {
		rc_one();
	    }

	    if (RegEnumKey(hkey, index, keybuf, sizeof keybuf) != ERROR_SUCCESS) {
		rc_one();
	    }
	    else {
		rc_zero();
		rxresize(result, strlen(keybuf));
		memcpy(result->strptr, keybuf, result->strlength);
	    }
	}

	/* if we specified a stem variable, we need to loop through the
	 * keys and set up a stem variable. The variable name is assumed
	 * to end in a `.' */
	if (argc == 3) {
	    int keycount = 0, keysalloced = 100;
	    char ** keys = malloc(keysalloced * sizeof(char *));
	    register int i;
	    int rc = ERROR_SUCCESS;

	    rc_zero();

	    /* get all the keys */
	    for (i = 0, rc = RegEnumKey(hkey, i, keybuf, sizeof(keybuf));
		 rc == ERROR_SUCCESS;
		 i++, rc = RegEnumKey(hkey, i, keybuf, sizeof(keybuf))) {
		if (keysalloced <= keycount) {
		    keysalloced += 100;
		    keys = realloc(keys, keysalloced * sizeof(char *));
		}
		keys[keycount] = strdup(keybuf);
		keycount++;
	    }

	    /* now set the stem */
	    setastem(argv+2, keycount, keys);

	    /* clean up */
	    for (i = 0; i < keycount; i++)
		free(keys[i]);
	    free(keys);

	    /* if we weren't returning a value, set result to 0 */
	    if (!argv[1].strptr) {
		result_zero();
	    }
	}

	return 0;
}


/* this behaves similarly to w32regenumkey, but for entry names. */
rxfunc(w32regenumvalue)
{
	HKEY hkey;
	DWORD index;
	char byNameBuf[MAX_PATH+1];
	DWORD cbNameBuf;
	char * indptr;

	/* you must specify at least one of index or stem */
	if (argc == 1 || argc > 3 || (argc == 2 && argv[1].strptr == NULL) ||
	    (argc == 3 && argv[2].strptr == NULL && argv[1].strptr == NULL))
	    return BADARGS;

	hkey = GetHKEY(argv);

	if (!hkey) {
	    if (!argv[1].strptr)
		result_one();
	    rc_one();
	    return 0;
	}

	if (argv[1].strptr) {
	    rxstrdup(indptr, argv[1]);
	    if (sscanf(indptr, "%lu", &index) != 1) {
		rc_one();
	    }

	    cbNameBuf = sizeof(byNameBuf);
	    /* there's probably all kinds of useful information being thrown
	     * away here */
	    if (RegEnumValue(hkey, index, byNameBuf, &cbNameBuf, NULL, NULL, NULL, NULL) != ERROR_SUCCESS) {
		rc_one();
	    }
	    else {
		rc_zero();
		rxresize(result, strlen(byNameBuf));
		memcpy(result->strptr, byNameBuf, result->strlength);
	    }
	}

	/* if we specified a stem variable, we need to loop through the
	 * values and set up a stem variable. The variable name is assumed
	 * to end in a `.' */
	if (argc == 3) {
	    int valcount = 0, valsalloced = 100;
	    char ** vals = malloc(valsalloced * sizeof(char *));
	    register int i;
	    int rc = ERROR_SUCCESS;

	    rc_zero();

	    /* get all the values */
	    for (i = 0, cbNameBuf = sizeof(byNameBuf), rc = RegEnumValue(hkey, i, byNameBuf, &cbNameBuf, NULL, NULL, NULL, NULL);
		 rc == ERROR_SUCCESS;
		 i++, cbNameBuf = sizeof(byNameBuf), rc = RegEnumValue(hkey, i, byNameBuf, &cbNameBuf, NULL, NULL, NULL, NULL)) {

		if (valsalloced <= valcount) {
		    valsalloced += 100;
		    vals = realloc(vals, valsalloced * sizeof(char *));
		}
		vals[valcount] = strdup(byNameBuf);
		valcount++;
	    }

	    /* now set the stem */
	    setastem(argv+2, valcount, vals);

	    /* clean up */
	    for (i = 0; i < valcount; i++)
		free(vals[i]);
	    free(vals);
	    /* if we weren't returning a value, set result to 0 */
	    if (!argv[1].strptr) {
		result_zero();
	    }
	}

	return 0;
}


rxfunc(w32regflushkey)
{
	HKEY hkey;

	checkparam(1, 1) ;

	hkey = GetHKEY(argv);

	if (RegFlushKey(hkey) != ERROR_SUCCESS) {
	    result_one();
	}
	else {
	    result_zero();
	}

	return 0;
}


rxfunc(w32reggetkeysecdesc)
{
	HKEY hkey;
	PSECURITY_DESCRIPTOR psd = NULL;
	char *szbuf;
	DWORD cbsd, dw, err;
	SECURITY_INFORMATION si = OWNER_SECURITY_INFORMATION |
		GROUP_SECURITY_INFORMATION | DACL_SECURITY_INFORMATION;

	checkparam(1, 1) ;

	hkey = GetHKEY(argv);

	cbsd = 0;
	result->strlength = 0;
	if ((err=RegGetKeySecurity(hkey, si, NULL, &cbsd)) != ERROR_INSUFFICIENT_BUFFER) {
		rc_one();
		return 0;
	}

	psd = malloc(cbsd);
	if (!psd) {
		goto skipout;
	}

	szbuf = malloc((cbsd*2)+1);
	if (!szbuf) {
		goto skipout;
	}

	if (RegGetKeySecurity(hkey, si, psd, &cbsd) != ERROR_SUCCESS) {
		rc_one();
		goto skipout;
	}

	for (dw = 0; dw < cbsd; dw++) {
		sprintf(&szbuf[dw*2], "%02x", (unsigned) ((unsigned char *)psd)[dw]);
	}

	rxresize(result, cbsd*2);
	memcpy(result->strptr, szbuf, result->strlength);

	rc_zero();
skipout:
	if (psd) {
		free(psd);
	}

	if (szbuf) {
		free(szbuf);
	}

	return 0;
}



rxfunc(w32regsetkeysecdesc)
{
	char *secdesc = NULL;
	HKEY hkey;
	PSECURITY_DESCRIPTOR psd;
	DWORD cbpsd;
	DWORD dw;
	char *p;
	SECURITY_INFORMATION si = OWNER_SECURITY_INFORMATION |
		GROUP_SECURITY_INFORMATION | DACL_SECURITY_INFORMATION;

	checkparam(2, 2) ;

	hkey = GetHKEY(argv);

	rxstrdup(secdesc, argv[1]);

	cbpsd = strlen(secdesc);
	if ((cbpsd%2) != 0) {
		result_one();
		goto skipout;
	}
	cbpsd /= 2;

	psd = malloc(cbpsd);
	if (!psd) {
		result_one();
		goto skipout;
	}
	for (p=secdesc, dw=0; dw<cbpsd; dw++, p+=2) {
		char buf[3];
		unsigned int x;
		buf[0] = *p;
		buf[1] = *(p+1);
		buf[2] = '\0';
		if (sscanf(buf, "%02x", (unsigned) &x) != 1) {
			result_one();
			goto skipout;
		}
		((unsigned char *)psd)[dw] = (unsigned char)x;
	}

	if (RegSetKeySecurity(hkey, si, psd) == ERROR_SUCCESS) {
		result_zero();
	}

skipout:
	if (psd) {
		free(psd);
	}

	return 0;
}


rxfunc(w32regqueryinfokey)
{
	HKEY hkey;
	char *infoname;
	char numbuf[20];
	DWORD dwRet, dwDummy, err;
	FILETIME ftDummy;
	char chDummy[MAX_PATH+1];
	static const char * infothing[] = {
	    "NumSubKeys","MaxSubKeyName","NumValues","MaxValueName",
	    "MaxValueData"
	};
	DWORD infovalue[DIM(infothing)];
	register int i;

	checkparam(2, 3) ;

	hkey = GetHKEY(argv);

	rxstrdup(infoname, argv[1]);


	dwDummy = sizeof chDummy;
	if ((err=RegQueryInfoKey(hkey, chDummy, &dwDummy,
		NULL, infovalue, infovalue+1, &dwDummy,
		infovalue+2, infovalue+3, infovalue+4, &dwDummy, &ftDummy)) != ERROR_SUCCESS) {
		rc_one();
	}
	else {
		rc_zero();
	}

	/* default rc is -1 */
	dwRet = -1;
	for (i = 0; i < DIM(infothing); i++)
	    if (stricmp(infoname, infothing[i]) == 0) {
		dwRet = infovalue[i];
	    }

	result->strlength = sprintf(result->strptr, "%lu", dwRet);

	/* if we were passed a stem name, set a bunch of stem variables */
	if (argc == 3) {
	    SHVBLOCK svh[DIM(infothing)];
	    register int i;
	    char * name;

	    rxstrdup(name, argv[3]);

	    for (i = 0;  i < DIM(infothing); i++) {
		svh[i].shvnext = svh+i+1;
		svh[i].shvname.strptr = alloca(argv[2].strlength+15);
		svh[i].shvname.strlength = sprintf(svh[i].shvname.strptr, "%s%d", name, infothing[i]);
		svh[i].shvvalue.strptr = alloca(15);
		svh[i].shvvalue.strlength = sprintf(svh[i].shvvalue.strptr, "%d", infovalue[i]);
		svh[i].shvcode = RXSHV_SET;
	    }
	    svh[DIM(infothing)-1].shvnext = NULL;
	    RexxVariablePool(svh);
	}

	return 0;
}

const char se_backup_name[] = SE_BACKUP_NAME;
const char se_restore_name[] = SE_RESTORE_NAME;

static APIRET w32regsaverestorekey(ULONG argc, PRXSTRING argv, const char * saveparm, PRXSTRING result)
{


    char *filename;
    HKEY hkey;
    DWORD err;
    HANDLE ht;
    LUID luidValue;
    TOKEN_PRIVILEGES tkp;

    checkparam(2, 2) ;

    hkey = GetHKEY(argv);

    rxstrdup(filename, argv[1]);

    if (!OpenProcessToken(GetCurrentProcess(),
			  TOKEN_ADJUST_PRIVILEGES|TOKEN_QUERY, &ht) ||
	!LookupPrivilegeValue(NULL, saveparm, &luidValue)) {
	result_one();
    }

    else {
	tkp.PrivilegeCount = 1;
	tkp.Privileges[0].Luid = luidValue;
	tkp.Privileges[0].Attributes = SE_PRIVILEGE_ENABLED;

	if (!AdjustTokenPrivileges(ht, FALSE, &tkp, sizeof tkp, NULL, NULL) ||
	   (saveparm != se_backup_name && saveparm != se_restore_name) ||
	   (saveparm == se_backup_name && ((err=RegSaveKey(hkey, filename, NULL)) != ERROR_SUCCESS)) ||
	   (saveparm == se_restore_name && ((err=RegRestoreKey(hkey, filename, 0)) != ERROR_SUCCESS))) {
	    result_one();
	}
	else {
	    result_zero();
	}

	tkp.Privileges[0].Attributes = 0;

	if(!AdjustTokenPrivileges(ht, FALSE, &tkp, sizeof tkp, NULL, NULL)) {
	    result_one();
	}
    }

    return 0;
}


rxfunc(w32regsavekey)
{
    return w32regsaverestorekey(argc, argv, se_backup_name, result);
}

rxfunc(w32regrestorekey)
{
    return w32regsaverestorekey(argc, argv, se_restore_name, result);
}

static APIRET w32regloadunload(int nparms, ULONG argc, PRXSTRING argv, const char * saveparm, PRXSTRING result)
{
    char *filename;
    char *subkeyname;
    HKEY hkey;
    DWORD err;
    HANDLE ht;
    LUID luidValue;
    TOKEN_PRIVILEGES tkp;

    checkparam(nparms, nparms) ;

    hkey = GetHKEY(argv);

    rxstrdup(subkeyname, argv[1]);
    if (argc == 3)
	rxstrdup(filename, argv[2]);

    if (!OpenProcessToken(GetCurrentProcess(),
			  TOKEN_ADJUST_PRIVILEGES|TOKEN_QUERY, &ht) ||
	!LookupPrivilegeValue(NULL, saveparm, &luidValue)) {
	result_one();
    }

    else {
	tkp.PrivilegeCount = 1;
	tkp.Privileges[0].Luid = luidValue;
	tkp.Privileges[0].Attributes = SE_PRIVILEGE_ENABLED;

	if (!AdjustTokenPrivileges(ht, FALSE, &tkp, sizeof tkp, NULL, NULL) ||
	   (saveparm != se_backup_name && saveparm != se_restore_name) ||
	   (saveparm == se_backup_name && ((err=RegLoadKey(hkey, subkeyname, filename)) != ERROR_SUCCESS)) ||
	   (saveparm == se_restore_name && ((err=RegUnLoadKey(hkey, subkeyname)) != ERROR_SUCCESS))) {
	    result_one();
	}
	else {
	    result_zero();
	}

	tkp.Privileges[0].Attributes = 0;

	if(!AdjustTokenPrivileges(ht, FALSE, &tkp, sizeof tkp, NULL, NULL)) {
	    result_one();
	}
    }

    return 0;
}

rxfunc(w32regloadkey)
{
    return w32regloadunload(3, argc, argv, se_backup_name, result);
}

rxfunc(w32regunloadkey)
{
    return w32regloadunload(2, argc, argv, se_restore_name, result);
}


static struct hkey_map_s {
    char * name;
    HKEY hkey;
}  stdkeynames[] = {
    "CLASSES_ROOT", HKEY_CLASSES_ROOT,
    "CURRENT_CONFIG", HKEY_CURRENT_CONFIG,
    "CURRENT_USER", HKEY_CURRENT_USER,
    "DYN_DATA", HKEY_DYN_DATA,
    "HKEY_CLASSES_ROOT", HKEY_CLASSES_ROOT,
    "HKEY_CURRENT_CONFIG", HKEY_CURRENT_CONFIG,
    "HKEY_CURRENT_USER", HKEY_CURRENT_USER,
    "HKEY_DYN_DATA", HKEY_DYN_DATA,
    "HKEY_LOCAL_MACHINE", HKEY_LOCAL_MACHINE,
    "HKEY_PERFORMANCE_DATA", HKEY_PERFORMANCE_DATA,
    "HKEY_USERS", HKEY_USERS,
    "LOCAL_MACHINE", HKEY_LOCAL_MACHINE,
    "PERFORMANCE_DATA", HKEY_PERFORMANCE_DATA,
    "USERS", HKEY_USERS,
};

/* key must be one of the bits of text listed here, or some hex value */
static HKEY GetHKEY(PRXSTRING str) {
	HKEY hkey = 0;
	register int i;

	for (i = 0; i < DIM(stdkeynames); i++)
	    if (!strnicmp(str->strptr, stdkeynames[i].name, str->strlength)) {
		hkey = stdkeynames[i].hkey;
		break;
	    }

	if (!hkey) {
	    if (sscanf(str->strptr, "%lx", &hkey) != 1) {
		hkey = 0;
	    }
	}

	return hkey;
}


rxfunc(w32openeventlog)
{
    char *hostname = NULL;
    char *sourcename;
    HANDLE hRet;

    /* must have two arguments, although 1 is optional */
    if (argc != 2 || !argv[1].strptr) {
	return BADARGS;
    }

    if (argv[0].strptr) {
	rxstrdup(hostname, argv[0]);
    }

    rxstrdup(sourcename, argv[1]);

    if ((hRet=OpenEventLog(hostname, sourcename)) == NULL) {
	result_zero();
    }
    else {
	result->strlength = sprintf(result->strptr, "%lx", (unsigned long)hRet);
    }

    return 0;
}


rxfunc(w32closeeventlog)
{
    HANDLE h;
    char * hptr;

    checkparam(1, 1) ;

    rxstrdup(hptr, argv[0]);

    sscanf(hptr, "%lx", (unsigned long *)&h);

    if (!CloseEventLog(h)) {
	result_one();
    }
    else {
	result_zero();
    }

    if (eLog) {
	free(eLog);
	eLog = NULL;
    }

    return 0;
}



rxfunc(w32getnumberofeventlogrecords)
{
    HANDLE h;
    DWORD num;
    char * hptr;

    checkparam(1, 1) ;

    rxstrdup(hptr, argv[0]);

    sscanf(hptr, "%lx", (unsigned long *)&h);

    if (!GetNumberOfEventLogRecords(h, &num)) {
	num = -1;
    }

    result->strlength = sprintf(result->strptr, "%d", num);
    return 0;
}

rxfunc(w32openbackupeventlog)
{
    char *hostname = NULL;
    char *filename;
    HANDLE hRet;

    if (argc != 2 || !argv[1].strptr) {
	return BADARGS;
    }

    if (argv[0].strptr) {
	rxstrdup(hostname, argv[0]);
    }

    rxstrdup(filename, argv[1]);

    if ((hRet=OpenBackupEventLog(hostname, filename)) == NULL) {
	result_zero();
    }
	 
    else {
	result->strlength = sprintf(result->strptr, "%lx", (unsigned long)hRet);
    }

    return 0;
}


rxfunc(w32backupeventlog)
{
    char *filename;
    HANDLE h;
    char * hptr;

    checkparam(2, 2) ;

    rxstrdup(hptr, argv[0]);
    sscanf(hptr, "%lx", (unsigned long *)&h);

    rxstrdup(filename, argv[1]);

    if (!BackupEventLog(h, filename)) {
    	result_one();
    }
    else {
	result_zero();
    }

    return 0;
}


rxfunc(w32cleareventlog)
{
    char *filename = NULL;
    HANDLE h;
    char * hptr;

    checkparam(1, 2) ;

    rxstrdup(hptr, argv[0]);
    sscanf(hptr, "%lx", (unsigned long *)&h);

    if (argc == 2 && argv[1].strptr) {
	rxstrdup(filename, argv[1]);
    }

    if (!ClearEventLog(h, filename)) {
	result_one();
    }
    else {
	result_zero();
    }

    return 0;
}

static void seteventvar(PRXSTRING stemname, EVENTLOGRECORD * elr);

rxfunc(w32findeventlogentry)
{
    HANDLE h;
    DWORD recnum;
    DWORD cbRead, needed;
    char chDummy[1];
    char * hptr;

    checkparam(2, 3) ;

    rxstrdup(hptr, argv[0]);
    sscanf(hptr, "%lx", (unsigned long *)&h);

    rxstrdup(hptr, argv[1]);
    if (sscanf(hptr, "%lu", &recnum) != 1 || recnum == 0) {
	return BADARGS;
    }

    ReadEventLog(h, EVENTLOG_SEEK_READ|EVENTLOG_FORWARDS_READ, recnum,
		 &chDummy, 0, &cbRead, &needed);
    if (GetLastError() != ERROR_INSUFFICIENT_BUFFER) {
	result_one();
    }
    else {
	/* either eLog is NULL, so realloc will act like malloc, or it isn't,
	 * so realloc will possibly do nothing */
	eLog = realloc(eLog, needed);

	if (!eLog) {
	    result_one();
	}

	else if (!ReadEventLog(h, EVENTLOG_SEEK_READ|EVENTLOG_FORWARDS_READ, recnum,
			       eLog, needed, &cbRead, &needed)) {
	    result_one();
	}
	else {
	    result_zero();
	}

	if (argc == 3)
	    seteventvar(argv+2, (EVENTLOGRECORD *)eLog);
    }

    return 0;
}

rxfunc(w32geteventid)
{
    EVENTLOGRECORD *elr = (EVENTLOGRECORD *)eLog;

    checkparam(0, 0) ;

    if (!elr) {
	rc_one();
	result->strlength = sprintf(result->strptr, "-1");
    }
    else {
	rc_zero();
	result->strlength = sprintf(result->strptr, "%lu", elr->EventID);
    }

    return 0;
}



rxfunc(w32geteventtype)
{
    EVENTLOGRECORD *elr = (EVENTLOGRECORD *)eLog;

    checkparam(0, 0) ;

    if (!elr) {
	rc_one();
	result->strlength = sprintf(result->strptr, "-1");
    }
    else {
	rc_zero();
	result->strlength = sprintf(result->strptr, "%lu", elr->EventType);
    }

    return 0;
}



rxfunc(w32geteventcategory)
{
    EVENTLOGRECORD *elr = (EVENTLOGRECORD *)eLog;

    checkparam(0, 0) ;

    if (!elr) {
	rc_one();
	result->strlength = sprintf(result->strptr, "-1");
    }
    else {
	rc_zero();
	result->strlength = sprintf(result->strptr, "%lu", elr->EventCategory);
    }

    return 0;
}



rxfunc(w32geteventnumstrings)
{
    EVENTLOGRECORD *elr = (EVENTLOGRECORD *)eLog;

    checkparam(0, 0) ;

    if (!elr) {
	rc_one();
	result->strlength = sprintf(result->strptr, "-1");
    }
    else {
	rc_zero();
	result->strlength = sprintf(result->strptr, "%lu", elr->NumStrings);
    }

    return 0;
}


/* change from ataman: ataman returned date in asctime() format, which 
 * is Ddd Mmm d hh:mm:ss yyyy (ie, useless). I return time in
 * yyyy-mm-dd hh.mm.ss, which is one of the ISO formats.
 * The (enhanced) Regina  date function can be used to convert this
 * to some other format. (Actually I thought better of it, and only
 * return the ISO format in w32findeventlogentry) */
rxfunc(w32geteventtimewritten)
{
    EVENTLOGRECORD *elr = (EVENTLOGRECORD *)eLog;
    struct tm* tm;
    char * p;

    checkparam(0, 0) ;

    if (!elr) {
	rc_one();
	result->strlength = sprintf(result->strptr, "-1");
    }
    else {
	rc_zero();
	tm = localtime(&elr->TimeWritten);

#ifdef USE_ISO_DATES
	result->strlength = sprintf(result->strptr, "%04d-%02d-%02d %02d.%02d.%02d",
		tm->tm_year, tm->tm_mon+1, tm->tm_mday, tm->tm_hour,
		tm->tm_min, tm->tm_sec);
#else
	p = asctime(tm);
	result->strlength = strchr(p, '\n') - p;
	memcpy(result->strptr, p, result->strlength);
#endif
	}

    return 0;
}


/* see the comments to w32geteventtimewritten for the output format */
rxfunc(w32geteventtimegenerated)
{
    EVENTLOGRECORD *elr = (EVENTLOGRECORD *)eLog;
    struct tm* tm;
    char * p;

    checkparam(0, 0) ;

    if (!elr) {
	rc_one();
	result->strlength = sprintf(result->strptr, "-1");
    }
    else {
	rc_zero();
	tm = localtime(&elr->TimeGenerated);
#ifdef USE_ISO_DATES
	result->strlength = sprintf(result->strptr, "%04d-%02d-%02d %02d.%02d.%02d",
		tm->tm_year, tm->tm_mon+1, tm->tm_mday, tm->tm_hour,
		tm->tm_min, tm->tm_sec);
#else
	p = asctime(tm);
	result->strlength = strchr(p, '\n') - p;
	memcpy(result->strptr, p, result->strlength);
#endif
	}

    return 0;
}



rxfunc(w32geteventstring)
{
    EVENTLOGRECORD *elr = (EVENTLOGRECORD *)eLog;
    char *szbuf = NULL;
    DWORD dw;
    DWORD szwanted;
    char *p;
    char * hptr;

    checkparam(1, 1) ;

    if (!elr) {
	rc_one();
	result->strlength = 0;
    }
    else {
	rc_zero();

	rxstrdup(hptr, argv[0]);
	if (sscanf(hptr, "%lu", &szwanted) != 1) {
	    return BADARGS;
	}

	if (szwanted == 0 || szwanted > elr->NumStrings) {
	    return BADARGS;
	}

	for (p=&eLog[elr->StringOffset], dw=1; dw < szwanted; dw++) {
	    /* replaced while(*p) p++; on the theory strchr must be as fast,
	     * but it's probably faster, unless the compiler people blew it */
	    p = strchr(p, '\0');
	    if (p)
		p++;
	}

	if (p) {
	    rxresize(result, strlen(p));
	    memcpy(result->strptr, p, result->strlength);
	}
	else {
	    rc_one();
	    result->strlength = 0;
	}
    }

    return 0;
}



rxfunc(w32geteventdata)
{
    EVENTLOGRECORD *elr = (EVENTLOGRECORD *)eLog;
    DWORD dw;
    char *szbuf;
    unsigned char *p = (unsigned char *)&eLog[elr->DataOffset];

    checkparam(0, 0) ;

    szbuf = malloc((elr->DataLength*2)+1);

    if (!szbuf) {
	rc_one();
    }
    else {
	rc_zero();

	for (dw = 0; dw < elr->DataLength; dw++) {
	    sprintf(&szbuf[dw*2], "%02x", (unsigned)p[dw]);
	}

	rxresize(result, elr->DataLength * 2);
	memcpy(result->strptr, szbuf, result->strlength);

	free(szbuf);
    }

    return 0;
}
 

rxfunc(w32writeeventlog)
{
    WORD type;
    HANDLE h = NULL;
    char *hostname = NULL;
    char *source;
    DWORD datalen = 0;
    WORD category;
    DWORD eventid;
    WORD nStrings = 0;
    char **ppsz;
    unsigned char *pData = NULL;
    char * strptr;
    register int i;

    /* be default, assume there'll be an error */
    result_one();

    if (argc < 6 || !argv[1].strptr) {
	return BADARGS;
    }

    nStrings = (unsigned short)(argc-6);
    ppsz = alloca(nStrings * sizeof(*ppsz));

    /* set some strings */
    if (argv[0].strptr) {
	rxstrdup(hostname, argv[0]);
    }

    rxstrdup(source, argv[1]);

    if (argv[2].strptr) {
	rxstrdup(strptr, argv[2]);
	if (sscanf(strptr, "%hu", (unsigned short *)&type) != 1) {
	    goto blowup;
	}
    } else {
	type = EVENTLOG_ERROR_TYPE;
    }

    if (argv[3].strptr) {
	rxstrdup(strptr,argv[3]);
	if (sscanf(strptr, "%hu", (unsigned short *)&category) != 1) {
	    goto blowup;
	}
    } else {
	category = 0;
    }

    if (argv[4].strptr) {
	rxstrdup(strptr, argv[4]);
	if (sscanf(strptr, "%lu", (unsigned long *)&eventid) != 1) {
	    goto blowup;
	}
    } else {
	eventid = 0;
    }

    if (argv[5].strptr) {
	pData = RXSTRPTR(argv[5]);
	datalen = RXSTRLEN(argv[5]);
    }

    for (i = 6; i < argc; i++) {
	if (argv[i].strptr) {
	    rxstrdup(ppsz[i-6], argv[i]);
	} else {
	    ppsz[i-6] = "";
	}
    }

    if ((h=RegisterEventSource(hostname, source)) == NULL) {
	goto blowup;
    }

    if (!ReportEvent(h, type, category, eventid, NULL, nStrings, datalen, ppsz, pData)) {
	goto blowup;
    }

    if (!CloseEventLog(h)) {
	goto blowup;
    }

    result_zero();
    h = NULL;

blowup:
    if (h) {
	(void)CloseEventLog(h);
    }

    return 0;
}


static struct tm * correcttime(time_t thetime)
{
    struct tm * ptm;

    ptm = localtime(&thetime);

    /* if it's daylight savings time, we want to subtract an hour, and all
     * that that implies */
    if (ptm->tm_isdst) {
	thetime -= 3600;
	ptm = localtime(&thetime);
    }

    /* make it a 4 digit year */
    ptm->tm_year += 1900;

    return ptm;
}


/* set up a stem variable particular to reading an event log entry.
 * This is in lieu of calling all those getthis, getthat functions */
static void seteventvar(PRXSTRING stemname, EVENTLOGRECORD * elr)
{
    SHVBLOCK *evstem, * pv;
    char intdata[7][20], * stem;
    register int i = 0;
    register char * cp;
    struct tm * ptm;

    rxstrdup(stem, *stemname);

    evstem = alloca((9+elr->NumStrings)*sizeof(SHVBLOCK));
    memset(evstem, 0, sizeof(evstem));

    /* first drop it */
    pv = evstem;
    pv->shvnext = pv + 1;
    pv->shvname = *stemname;
    pv->shvcode = RXSHV_DROPV;

    /* now do each of the integer members */
    pv++;
    pv->shvnext = pv + 1;
    pv->shvname.strptr = alloca(stemname->strlength+3);
    pv->shvname.strlength = sprintf(pv->shvname.strptr, "%s%s", stem, "id");
    pv->shvvalue.strptr = intdata[i++];
    pv->shvvalue.strlength = sprintf(pv->shvvalue.strptr, "%lu", elr->EventID);
    pv->shvcode = RXSHV_SET;
    
    pv++;
    pv->shvnext = pv + 1;
    pv->shvname.strptr = alloca(stemname->strlength+5);
    pv->shvname.strlength = sprintf(pv->shvname.strptr, "%s%s", stem, "type");
    pv->shvvalue.strptr = intdata[i++];
    pv->shvvalue.strlength = sprintf(pv->shvvalue.strptr, "%d", elr->EventType);
    pv->shvcode = RXSHV_SET;
    
    pv++;
    pv->shvnext = pv + 1;
    pv->shvname.strptr = alloca(stemname->strlength+9);
    pv->shvname.strlength = sprintf(pv->shvname.strptr, "%s%s", stem, "category");
    pv->shvvalue.strptr = intdata[i++];
    pv->shvvalue.strlength = sprintf(pv->shvvalue.strptr, "%d", elr->EventCategory);
    pv->shvcode = RXSHV_SET;
    
    pv++;
    pv->shvnext = pv + 1;
    pv->shvname.strptr = alloca(stemname->strlength+11);
    pv->shvname.strlength = sprintf(pv->shvname.strptr, "%s%s", stem, "numstrings");
    pv->shvvalue.strptr = intdata[i++];
    pv->shvvalue.strlength = sprintf(pv->shvvalue.strptr, "%d", elr->NumStrings);
    pv->shvcode = RXSHV_SET;
    
    /* this is pointless, but pseudo-conventional */
    pv++;
    pv->shvnext = pv + 1;
    pv->shvname.strptr = alloca(stemname->strlength+9);
    pv->shvname.strlength = sprintf(pv->shvname.strptr, "%s%s", stem, "string.0");
    pv->shvvalue.strptr = intdata[i++];
    pv->shvvalue.strlength = sprintf(pv->shvvalue.strptr, "%d", elr->NumStrings);
    pv->shvcode = RXSHV_SET;
    
    pv++;
    pv->shvnext = pv + 1;
    pv->shvname.strptr = alloca(stemname->strlength+12);
    pv->shvname.strlength = sprintf(pv->shvname.strptr, "%s%s", stem, "timewritten");
    pv->shvvalue.strptr = intdata[i++];
    ptm = correcttime(elr->TimeWritten);

    pv->shvvalue.strlength = sprintf(pv->shvvalue.strptr, "%04d-%02d-%02d %02d.%02d.%02d",
    				     ptm->tm_year, ptm->tm_mon+1, ptm->tm_mday, ptm->tm_hour,
				     ptm->tm_min, ptm->tm_sec);
    pv->shvcode = RXSHV_SET;
    
    pv++;
    pv->shvnext = pv + 1;
    pv->shvname.strptr = alloca(stemname->strlength+14);
    pv->shvname.strlength = sprintf(pv->shvname.strptr, "%s%s", stem, "timegenerated");
    pv->shvvalue.strptr = intdata[i++];

    ptm = correcttime(elr->TimeGenerated);

    pv->shvvalue.strlength = sprintf(pv->shvvalue.strptr, "%04d-%02d-%02d %02d.%02d.%02d",
    				     ptm->tm_year, ptm->tm_mon+1, ptm->tm_mday, ptm->tm_hour,
				     ptm->tm_min, ptm->tm_sec);
    pv->shvcode = RXSHV_SET;
    
    /* now the data buffer */
    pv++;
    pv->shvnext = pv + 1;
    pv->shvname.strptr = alloca(stemname->strlength+5);
    pv->shvname.strlength = sprintf(pv->shvname.strptr, "%s%s", stem, "data");
    /* note -- we don't put this in hex. */
    pv->shvvalue.strptr = ((char *)elr) + elr->DataOffset;
    pv->shvvalue.strlength = elr->DataLength;
    pv->shvcode = RXSHV_SET;
    
    /* now do the strings */
    cp = ((char *)elr)+elr->StringOffset;
    for (i = 1; i <= elr->NumStrings; i++) {
	pv++;
	pv->shvnext = pv + 1;
	pv->shvname.strptr = alloca(stemname->strlength+15);
	pv->shvname.strlength = sprintf(pv->shvname.strptr, "%sstring.%d", stem, i);
	pv->shvvalue.strptr = cp;
	pv->shvvalue.strlength = strlen(cp);
	pv->shvcode = RXSHV_SET;
	cp = strchr(cp, '\0') + 1;
    }

    pv->shvnext = NULL;

    RexxVariablePool(evstem);
}


/* mapping between rexx strings and C symbolic names for registry data types
 * given that there are only 9 types, I'm just using a straight scan
 * of this array, although I sorted the entries, so you could use bsearch
 * if you wanted to.
 */
static struct type_map_s {
    char * name;
    DWORD type;
} theTypeMap[] = {
    "REG_BINARY", REG_BINARY,
    "REG_DWORD", REG_DWORD,
    "REG_DWORD_BIG_ENDIAN", REG_DWORD_BIG_ENDIAN,
    "REG_EXPAND_SZ", REG_EXPAND_SZ,
    "REG_LINK", REG_LINK,
    "REG_MULTI_SZ", REG_MULTI_SZ,
    "REG_NONE", REG_NONE,
    "REG_RESOURCE_LIST", REG_RESOURCE_LIST,
    "REG_SZ", REG_SZ
};

static char * type_name(DWORD dwType)
{
    register int i;

    for (i = 0; i < DIM(theTypeMap); i++) {
	if (dwType == theTypeMap[i].type)
	    return theTypeMap[i].name;
    }

    return "UNKNOWN";
 }

static DWORD type_type(const char * const theType)
{
    register int i;

    for (i = 0; i < DIM(theTypeMap); i++) {
	if (!stricmp(theType, theTypeMap[i].name))
	    return theTypeMap[i].type;
    }

    return 0;
}


/* load and unload all the functions */

static const struct {
   char * name;
   APIRET (APIENTRY*rxFcn)(PUCHAR fname, ULONG argc, PRXSTRING argv, PSZ pSomething, PRXSTRING result);
} funcTable[] = {
   "w32backupeventlog", w32backupeventlog,
   "w32callfunc", w32callfunc,
   "w32callproc", w32callproc,
   "w32cleareventlog", w32cleareventlog,
   "w32closeeventlog", w32closeeventlog,
   "w32createobject", w32createobject,
   "w32findeventlogentry", w32findeventlogentry,
   "w32geteventcategory", w32geteventcategory,
   "w32geteventdata", w32geteventdata,
   "w32geteventid", w32geteventid,
   "w32geteventnumstrings", w32geteventnumstrings,
   "w32geteventstring", w32geteventstring,
   "w32geteventtimegenerated", w32geteventtimegenerated,
   "w32geteventtimewritten", w32geteventtimewritten,
   "w32geteventtype", w32geteventtype,
   "w32getnumberofeventlogrecords", w32getnumberofeventlogrecords,
   "w32getobject", w32getobject,
   "w32getproperty", w32getproperty,
   "w32getsubobj", w32getsubobj,
   "w32openbackupeventlog", w32openbackupeventlog,
   "w32openeventlog", w32openeventlog,
   "w32putproperty", w32putproperty,
   "w32regclosekey", w32regclosekey,
   "w32regconnectregistry", w32regconnectregistry,
   "w32regcreatekey", w32regcreatekey,
   "w32regdeletekey", w32regdeletekey,
   "w32regdeletevalue", w32regdeletevalue,
   "w32regenumkey", w32regenumkey,
   "w32regenumvalue", w32regenumvalue,
   "w32regflushkey", w32regflushkey,
   "w32reggetkeysecdesc", w32reggetkeysecdesc,
   "w32regloadkey", w32regloadkey,
   "w32regopenkey", w32regopenkey,
   "w32regqueryinfokey", w32regqueryinfokey,
   "w32regqueryvalue", w32regqueryvalue,
   "w32regqueryvaluetype", w32regqueryvaluetype,
   "w32regrestorekey", w32regrestorekey,
   "w32regsavekey", w32regsavekey,
   "w32regsetkeysecdesc", w32regsetkeysecdesc,
   "w32regsetvalue", w32regsetvalue,
   "w32regunloadkey", w32regunloadkey,
   "w32releaseobject", w32releaseobject,
   "w32writeeventlog", w32writeeventlog,
   "w32dropfuncs", w32dropfuncs,
   "w32loadfuncs", w32loadfuncs        /* must be last, so we don't reload it */
};

/* load every function in that table. */
rxfunc(w32loadfuncs)
{
   register int i;

   /* can use RegisterFunctionExe since the addresses are already resolved in
    * the array above */
   for (i = 0; i < DIM(funcTable)-1; i++) {
      RexxRegisterFunctionExe(funcTable[i].name, funcTable[i].rxFcn);
      /* ought to test for an error */
   }

   result_zero();

   return 0;
}

/* unload every function in that table */
rxfunc(w32dropfuncs)
{
   register int i;

   for (i = 0; i < DIM(funcTable); i++) {
      RexxDeregisterFunction(funcTable[i].name);
      /* ought to test for an error */
   }

   result_zero();

   return 0;
}
