/******************************************************************
 * FILE.CMD
 *
 * This program calls the various REXX external functions provided in the FILEREXX DLL.
 *
 * Each function is called once to illustrate how it is used.
 *******************************************************************/


/* The FileLoadFuncs loads all the rest of the REXX functions in the FILEREXX DLL. */
/* So, we don't have to make calls to RxFuncAdd to add each one of those functions. Of */
/* course, in order to call FileLoadFuncs, we have to add that one. */
CALL RxFuncAdd 'FileLoadFuncs', 'FILEREXX', 'FileLoadFuncs'
CALL FileLoadFuncs



/* ================================ FileWrite ================================= */
/*** Demonstrate FileOpen by opening 'c:\blort' for writing. This should return non-0. Note the
     flags is 'on' to overwrite any existing file with the same name, or create a new file if it doesn't exist.
     The mode is 'rws' for Read/Write with Sharing. Note that we could do only Write with Sharing (ie, 'ws')
      since we aren't going to simultaneously read/write using this handle. That would be the wiser thing
      in case someone else has prevented read sharing. But, this is for demonstration. The attributes is
      0 for normal file, although we don't need to supply it as that's the default.
***/
handle = FileOpen('c:\blort', 'rws', 'on', 0)
SAY  handle"=FileOpen('c:\blort', 'rws', 'on', 0)"

/*** Demonstrate FileWrite by writing 10 characters to 'c:\blort'. This should return 10 in this case ***/
err = FileWrite(handle, '0123456789', 10)
SAY  err"=FileWrite(handle, '0123456789', 10)"

/* Incidentally, the count for FileWrite defaults to a 1. So, if you're writing 1 character, you can omit the count arg. */
err = FileWrite(handle, ".")
SAY  err"=FileWrite(handle, '.')"

/*** Demonstrate FileClose. This should return 0. ***/
err = FileClose(handle)
SAY err"=FileClose(handle)"




/* ================================ FileRead ================================= */
/*** Demonstrate FileOpen by opening 'c:\blort' for reading. This should return non-0, Note omit last arg ***/
handle = FileOpen('c:\blort', 'r', 'e')
SAY  handle"=FileOpen('c:\blort', 'r', 'e')"

/*** Demonstrate FileRead by reading 10 characters from 'c:\blort'. This should return '0123456789' ***/
contents = FileRead(handle, 10)
SAY  contents"=FileRead(handle, 10)"

/* Incidentally, the count for FileRead defaults to a 1. So, if you're reading 1 character, you can omit the count arg. */
contents = FileRead(handle)
SAY "one more character read ="contents

/*** Demonstrate FileClose. This should return 0. ***/
err = FileClose(handle)
SAY err"=FileClose(handle)"




/* ============================= FileOpen sharing ============================== */
/** OK. Let's open the same file twice to demonstrate that FileOpen shares access if we allow it. For
      Sharing, we're going to add the + or - signs after the 's' in order to specifically ask for READ sharing
      but no WRITE sharing (ie, 's+-').  If we don't specify these extra chars, it defaults to 's++' **/
handle = FileOpen('c:\blort', 'rs+-', 'e')
SAY  handle"=FileOpen('c:\blort', 'rs+-', 'e')  <- first handle"
handle2 = FileOpen('c:\blort', 'rs+-', 'e')
SAY  handle2"=FileOpen('c:\blort', 'rs+-', 'e')  <- second handle"

/** OK. Let's alternately read 1 byte from the file using both handles. You'll note that both
      handles have their own "current" position **/
DO i=1 TO 10
  SAY FileRead(handle)"=read from first handle"
  SAY FileRead(handle2)"=read from second handle"
END

/** Close the 2 handles */
err = FileClose(handle)
SAY err"=FileClose(handle)"
err = FileClose(handle2)
SAY err"=FileClose(handle2)"




/* ================================ FilePuts ================================= */
/** Use FilePuts to write 3 lines of text. To this example, we'll add error checking of return values.
       Note that we overwrite any existing file, or allow creation of the file if it doesn't exist. Also note
	that we DON'T allow write sharing on this file while its open. If someone else has already done
       likewise to us, then this open will fail.
 **/
handle = FileOpen('c:\blort', 'w', 'on')
IF handle = 0 THEN SAY "Error opening file"

line.1 = "This is the first line"
line.2 = "This is the second line"
line.3 = "This is the third     and last    line"

DO i=1 TO 3
    err = FilePuts(handle, line.i)
    IF err = 0 THEN SAY "Error writing to file"
END

/* Incidentally, if you supply a null line for FilePuts, it defaults to simply closing the current line. So, if you're using
    FileWrite() to write the characters of a line separately, as you might with CHAROUT, and you wish to close the
    line, simply do: */
err = FilePuts(handle);
IF err = 0 THEN SAY "Error upon writing the end of the line"

err = FileClose(handle)
IF err <> 0 THEN SAY "Error closing file"




/* ================================ FileGets ================================= */
/** Use FileGets to read those lines of text we just wrote above. Note: the default for FileOpen is
      READ with READ sharing (no WRITE share), 'e' flag, normal file. That's what I want here, so the
      FileOpen is really easy
**/
handle = FileOpen('c:\blort')
IF handle = 0 THEN SAY "Error opening file"

i=1
DO FOREVER
    myline = FileGets(handle)
    IF FILEERR = 0 THEN LEAVE
    IF FILEERR <> 0 THEN SAY "line "i"="myline
    i=i+1
END

/* Now before we close the file, seek to 5 bytes after beginning and read the first line again. We should
    skip the "This ". Note that we could omit the 3rd arg if it was a 1 (ie, seek from current position) */
err = FileSeek(handle, 5, 0)
IF err = "" THEN SAY "Error seeking within file"
IF err <> "" THEN SAY "New position within file ="err
myline = FileGets(handle)
IF FILEERR <> 0 THEN SAY "seek'ed line="myline

err = FileClose(handle)
IF err <> 0 THEN SAY "Error closing file"




/* ============================= FileGetInfo ================================ */
/** Use FileGetInfo to retrieve info about this REXX script (ie, assumed to be in current dir).
**/
handle = FileOpen('file.cmd')
IF handle \= 0 THEN DO
    err = FileGetInfo(handle, 'Info', 'SIZE|DATE|EASIZE')
    IF err = 0 THEN DO
	SAY "Size of file is" Info.0 "bytes"
	SAY "Size of file's EAs are" Info.3 "bytes"
	SAY "Last Write Date of file is" Info.1 "(month/day/year hour/minute/sec)"
    END
    err = FileClose(handle)
END




/* ============================= FileGets with parsing ============================== */
/** Use FileGets to read those same lines of text, but parse each line into separate args using a space as
      the separator character. So, for the first call to FileGets, it should return 5 args, "This", "is", "the",
      "first", and "line", stored in the variables FileArg.1, FileArg.2, FileArg.3, FileArg.4, and FileArg.5. FileArg.0
      should be set to 5. For the next call to FileGets, it should return 5 args also, "This", "is", "the",
      "second", and "line", stored in the variables FileArg.1, FileArg.2, FileArg.3, FileArg.4, and FileArg.5. FileArg.0
      should be set to 5. For the next call to FileGets, it should return 7 args, "This", "is", "the",
      "third", "and", "last, and "line", stored in the variables FileArg.1, FileArg.2, FileArg.3, FileArg.4, FileArg.5,
      FileArg.6, and FileArg.7. FileArg.0 should be set to 7. For the next call to FileGets, FileArg.0 should be
      set to 1, and FileArg.1 is a null string, since we wrote out an empty line in the above FilePuts example.
      For the next call to FileGets, FileArg.0 should be set to 0 since there aren't any more lines in the file.
	   Note that the separator arg can be many characters, for example '=,;' would separate args at any of the
      characters '=', ',', or ';'. If you want to include a ' or " in the separator string, enclose the ' within "" and
      the " within ''. For example, here's how to specify both quote characters are separators:  '"'"'". Yeah,
      it's ugly, but that's REXX.
	     Also note that for flags, we ask to trim leading and trailing spaces off of each arg. If we didn't
      want any flags enabled, we'd specify a null string (ie, "" or '').
**/
handle = FileOpen('c:\blort')

linenum=1
DO FOREVER
    FileGets(handle, 't', ' ')
    IF FILEARG.0 = 0 THEN LEAVE
    DO i = 1 to FileArg.0
	SAY "line #"linenum": Arg #"i"="FileArg.i
    END
    linenum = linenum+1
END

err = FileClose(handle)




/* ============================== FileWriteValue ============================ */
handle = FileOpen('c:\blort', 'w', 'on')

/*** Demonstrate FileWriteValue by writing 400000 LONG, 20000 SHORT, 6 CHAR, and -369000 LONG ***/
err = FileWriteValue(handle, 400000, 4)
SAY  "Writing 400000 ULONG"
err = FileWriteValue(handle, 20000, 2)
SAY  "Writing 20000 USHORT"
err = FileWriteValue(handle, 6, 1)
SAY  "Writing 6 UCHAR"
err = FileWriteValue(handle, -369000, 4)
SAY  "Writing -369000 LONG"

err = FileClose(handle)



/* ============================== FileReadValue ============================ */
handle = FileOpen('c:\blort', 'r', 'e')

/*** Demonstrate FileReadValue by reading 400000 LONG, 20000 SHORT, 6 CHAR, and -369000 LONG ***/
contents = FileReadValue(handle, 4)
SAY  "Reading "contents" ULONG"
contents = FileReadValue(handle, 2)
SAY  "Reading "contents" USHORT"
contents = FileReadValue(handle, 1)
SAY  "Reading "contents" UCHAR"
contents = FileReadValue(handle, 4, '-')
SAY  "Reading "contents" LONG"

err = FileClose(handle)



/* ============================ FileDeleteFile ============================ */
/* Delete that file that we created here */
err = FileDeleteFile('c:\blort')
IF err = 0 THEN SAY 'Temp file is deleted.'



/* =============================================================================== */
/* FileDropFuncs: This unloads all of the functions in the FILEREXX DLL. This is not necessary, and
    we could otherwise leave it open for some other REXX script */
CALL FileDropFuncs
