# wepstops.awk
#
# Copyright (c) 1995 by Magus. See the file WEPStoPS.TXT
# for information concerning distribution.
#
# Kevin Thompson     12 July 1995     Version 1.0
#
# This awk script reads in an EPS file produced by the 
# Microsoft Windows PostScript driver and converts it to a
# standard PostScript file, by inserting the page-delimiter 
# comment 
#
# %%Page: n n
#
# in the appropriate places (just before the SS procedure). 
# The script ignores genuine embedded EPS files, as denoted by
# surrounding %%BeginDocument:/%%EndDocument comments.
# If the file contains multiple concatenated files,
#
# Warning: If the file already has %%Page: comments in the
# proper places, this script may put in extra comments,
# resulting in empty pages on viewing.
#
# Usage:  awk -f wepstops.awk outfile=outputfilename < inputfilename
#
# where inputfilename and outputfilename are the names of the 
# original (EPS) file and modified (PS) files.  If the input
# file contains only one PostScript job, the output goes to
# outputfilename.  Otherwise, a file containing n PostScript jobs
# is split into one file per job, with file names 
# <stem>.1 through <stem>.n, where <stem> is the sequence of
# characters up to and including the last period in outputfilename.
#
# History:
# 20 March 1995  (Kevin Thompson): First beta release.
#
# 27 March 1995  (Kevin Thompson): Second beta.  
#    Fixed invalidaccess error in some files by disabling 
#    unnecessary save/restore (set fPP switch to false).
#
# 27 March 1995  (Kevin Thompson): Third beta.
#    Added code to split a file of concatenated PostScript
#    files into multiple files.  First file goes into
#    standard output.  The others go to wepstops.n, where
#    n is an integer which begins with 1.
#
# 29 March 1995 (Kevin Thompson): Fourth beta.
#    Added file closing code to fix open files bug.  
#    Improved naming convention for output files.
#
#  7 April 1995 (Kevin Thompson): Fifth beta.
#    Added test for the presence of page markers,
#    so the program refrains from inserting them
#    when they are already present.  Also improved test
#    for concatenated files by checking for %%EOF,
#    and fixed page numbering for multiple files.
#
# 19 April 1995 (Kevin Thompson): Sixth beta.
#    Improved test for concatenated files by
#    relaxing BOF pattern to match
#    PS-Adobe followed by a space, hyphen, or EOL.
#
# 12 July 1995 (Kevin Thompson): Release 1.0

# Initialize all variables.
BEGIN{
  depth = 0
  filecount = 0
  haspages = 0
  lastline = "%%EOF"
  pagecount = 0
}

# Increment or decrement the embedding depth, as appropriate.
$0 ~ /^%%BeginDocument:/ {depth = depth + 1}
$0 ~ /^%%EndDocument/ {depth = depth - 1}

# Test for a concatenated PostScript file.  If found, increment the
# destination counter and define a new destination file name.
# Skip the test for the firs line of the input file, since
# we know it begins a file and we want it written to standard output.

$0 ~ /%!PS-Adobe(-| |$)/ && depth == 0 && lastline ~ /^%%EOF/ {

# Increment the file counter and reset the page variables.
# Then save the original output file name,  or close the current 
# output file, as appropriate. 
  filecount = filecount + 1
  haspages = 0
  pagecount = 0
  if (filecount == 1)
    {firstout = outfile}
  else
    {close(outfile)}

# Determine the output file stem.  Append a period if none are present.
  numparts = split(firstout,fields,".")
  if (numparts == 1) { filestem = firstout "."}
  else
  {
    stemsize = length(firstout) - length(fields[numparts])
    filestem = substr(firstout,1,stemsize)
  }

# Name the next output file.
  outfile = filestem filecount
}

# Check for the presence of page markers.
$0 ~ /^%%Page:/ && depth == 0 {haspages = 1}

# Insert absent %%Page: n n comment for non-embedded boundaries, 
# writing to the appropriate destination.
$0 ~ /^SS$/ && depth == 0 && haspages == 0 {
  pagecount = pagecount + 1
  print "%%Page:", pagecount, pagecount > outfile
  print "/fPP false def  %  WEPStoPS: Disable troublesome save/restore" > outfile
}

# Write the current line to the appropriate destination, and
# save it to lastline if it is not empty.
{print > outfile}
$0 !~ /^[ ]*$/ {lastline = $0}

END {
# Close the current output file, then delete and rename files, if necessary.
# Display statistics regarding file creation.
  close(outfile)
  if (filecount == 1) {
    system("IF EXIST " firstout " DEL " firstout)
    system("RENAME " filestem "1 " firstout)
    print "Created one file: " firstout
  }
  else {
    print "Created " filecount " files: " filestem "1 through " filestem filecount
  }
}
