/* showdcp.cmd -- keyboard.dcp file explorer             950222 */
signal on halt

say 'Operating System/2  Keyboard.dcp file viewer'
say 'Version 1.07.000 Feb 22 1995'
say '(C) Copyright Martin Lafaix 1994, 1995'
say 'All rights reserved.'
say

/* keyboard layout flags names (order is significant!)          */
flg = 'ShiftAlt AltGrL AltGrR ShiftLock DefaultTable ShiftToggle AccentPass CapsShift MachDep RTL LangSel DefaultLayout'

/* extracting command line arguments                            */
arg infile x

if infile = '' | infile = '-h' | infile = '-H' then
   do
      say 'Usage:  showdcp [param] file [country [subcountry [cp [type]]]] [file2]'
      say
      say '        -h         - Access Help;'
      say '        -v[n]      - View matching layouts.  n is the detail level;'
      say '        -x         - Extract matching layouts;'
      say '        -a         - Add layout to file;'
      say '        -ds,t,c    - Define a key;'
      say '        -sk1,k2    - Swap key k1 and key k2;'
      say '        -fflag     - Set layout flag flag;'
      say '        -cflag     - Clear layout flag flag;'
      say
      say '        file       = keyboard.dcp file to use'
      say '        country    = country code (US, FR, ...) or *'
      say '        subcountry = subcountry code (189, 120, ...) or *'
      say '        cp         = code page (437, 850, ...) or *'
      say '        type       = keyboard type (0 = 89 keys, 1 = 101/102 keys) or *'
      say '        file2      = output or source file name (with -x or -a)'
      say
      say 'Possible flags (with -f and -c):'
      say
      say '   ShiftAlt    AltGrL     AltGrR   ShiftLock  DefaultTable  ShiftToggle'
      say '   AccentPass  CapsShift  MachDep  RTL        LangSel       DefaultLayout'
      exit
   end /* do */

opt = translate(left(infile,2))
select
   when opt = '-V' then do
      cmd = 'view'
      /* Find out the detail level  --  1 by default. */
      if datatype(substr(infile,3),'N') then
         lvl = substr(infile,3)
      else
         lvl = 1
      end
   when opt = '-X' then do
      cmd = 'extract'
      /* Find out the output file name (last command-line arg). */
      if words(x) < 2 then do
         say 'No output file name!'
         exit 1
         end
      outfile = word(x,words(x))
      if verify(outfile,'*?<>:|"','M') > 0 then do
         say 'Invalid output file name! ("'outfile'")'
         exit 1
         end
      x = subword(x,1,words(x)-1)
      end
   when opt = '-A' then do
      cmd = 'add'
      /* Find out the source file name (last command-line arg). */
      if words(x) < 2 then do
         say 'No output file name!'
         exit 1
         end
      outfile = word(x,words(x))
      if verify(outfile,'*?<>:|"','M') > 0 | stream(outfile,'c','query exists') = '' then do
         say 'Invalid or nonexistent output file name! ("'outfile'")'
         exit 1
         end
      x = subword(x,1,words(x)-1)
      end
   when opt = '-D' then do
      cmd = 'define'
      /* Find out the key scan code, type and definition. */
      parse value substr(infile,3) with kscan ',' ktype ',' kdef
      if \ datatype(kscan,'N') then do
         say 'Invalid scan code! (Not a number: "'kscan'")'
         exit 1
         end
      end
   when opt = '-S' then do
      cmd = 'swap'
      /* Find out what to swap */
      parse value substr(infile,3) with key1 ',' key2
      if datatype(key1,'N') = 0 then do
         say 'Invalid scan code! (Not a number: "'key1'")'
         exit 1
         end
      if datatype(key2,'N') = 0 then do
         say 'Invalid scan code! (Not a number: "'key2'")'
         exit 1
         end
      if key1 = key2 then do
         say 'Can''t swap a key with itself!'
         exit 2
         end
      if key2 < key1 then do
         key3 = key2
         key2 = key1
         key1 = key3
         end
      end
   when opt = '-F' then do
      cmd = 'set'
      /* Find out which flag to set*/
      flag = wordpos(translate(substr(infile,3)),translate(flg))
      if flag = 0 then do
         say 'Invalid flag! ("'substr(infile,3)'")'
         exit 1
         end
      end
   when opt = '-C' then do
      cmd = 'clear'
      /* Find out which flag to clear */
      flag = wordpos(translate(substr(infile,3)),translate(flg))
      if flag = 0 then do
         say 'Invalid flag! ("'substr(infile,3)'")'
         exit 1
         end
      end
otherwise
   cmd = ''
end  /* select */
 
if cmd \= '' then
   parse value x with infile x

if infile = '' then do
   say 'No KEYBOARD.DCP file specified!'
   exit 1
   end
if stream(infile,'c','query exists') = '' then do
   say 'Nonexistent KEYBOARD.DCP file! ("'infile'")'
   exit 1
end

parse value x '* * * *' with rcn rss rcp rty _

/* The following variables are now initialized:
**
** infile        - the keyboard.dcp file;
** cmd           - the requested command ('view', 'extract', ... or '');
** rcn           - the requested country abbrev (or '*');
** rss           - the requested subcountry ID (or '*');
** rcp           - the requested codepage (or '*');
** rty           - the requested keyboard type (or '*').
**
** These variables are initialized, if applicable:
**
** lvl           - the display level (0..4, 1 per default);
** kscan         - the key scan code (a decimal value);
** ktype         - the key type (a 16bits hexadecimal value);
** kdef          - the key definition (a hexadecimal string);
** key1          - the first key scan code (a decimal value);
** key2          - the second key scan code (another decimal value);
** outfile       - the output file name;
** flag          - the layout flag (a decimal one-based value).
*/

/* finding and reading the Index Table                          */
KLE.0 = 0
ito = readl()
call charin infile,ito
iec = readw()
do iec
   call getindex
   if (rcn = '*' | country = rcn) & (rss = '*' | rss = subcntr) & (rcp = '*' | rcp = cp) & (rty = '*' | rty = type) then
      call saveentry
end /* do */

call initialize

/* handling the found entries                                   */
do i = 1 to KLE.0
   call doentry i
end /* do */

call terminate

exit

initialize: /* do cmd-relative initializations */
   select
      when cmd = 'view' & KLE.0 > 0 & lvl >= 1 then do
         say 'Index Country SubCountry CodePage Offset Type ...'
         say '--------------------------------------------------------'
         end
   otherwise
   end  /* select */
   return
 
terminate: /* do cmd-relative terminations */
   if KLE.0 > 1 then do
      say
      say KLE.0 'entries found.'
      end
   else
   if KLE.0 = 1 then
      say 'One entry found.'
   else
      say 'No entry found!'
   return
 
doentry: /* handle entry #arg(1) */
   index = arg(1)
   offset = word(KLE.index,4)
   select
      when cmd = 'view' then do
         /*
         ** We here display the requested information.
         */
         if lvl >= 1 then say format(index,5) KLE.index
         if lvl >= 2 then do
            call getentry offset
            say
            say '      Table Header Information'
            say '      ------------------------'
            say '      'country||subcountry', Codepage: 'cp', Keyboard type: 'kbdtype', Keyboard subtype: 'kbdsubtype
            say '      'flagsdef1()
            say '      'flagsdef2()
            say '      Entry count: 'entrycount', Entry width: 'entrywidth', Table length: 'tablelen', TableTypeID: 'tabletypeid
            say
            end
         if lvl >= 3 then do
            say '      Scan Type char1 char2 char3 char4 char5 ...'
            say '      -------------------------------------------'
            do j = 1 to entrycount
               call getkeydef
               if keytype = 0 then iterate
               say '      'format(j,4) d2x(keytype,4) keydef()
            end /* do */
            say
            end
         if lvl >= 4 then do
            say '      Accent Table Entry'
            say '      ------------------'
            free = tablelen - 40 - entrycount * entrywidth
            j = 1
            empty = 1
            do while free > 0
               call getaccententry j
               if glyph \= 0 then do
                  say '      'format(j,2) ':' d2c(glyph) d2x(b1,2) d2x(b2,2) d2x(b3,2) d2x(b4,2) d2x(b5,2) strip(accententry,,'0'x)
                  empty = 0
                  end
               j = j + 1
               free = free - len
            end /* do */
            if empty then say '      <empty>'
            say
            end
         end
      when cmd = 'extract' then do
         /*
         ** We here extract the matching entry(ies) in outfile.
         */
         call getentry offset
         call emitentry offset
         end
      when cmd = 'define' then do
         /*
         ** We here modify the given key.
         */
         call getentry offset
         if kscan > entrycount then 
            say 'No such scancode!'
         else
         if length(x2c(kdef)) > entrywidth - 2 then
            say 'Definition too long!'
         else do
            call charin infile,,(kscan-1)*entrywidth
            call getkeydef
            say 'Replacing' format(kscan,4) d2x(keytype,4) keydef()
            say '     With' format(kscan,4) ktype kdef
            call charout infile,d2w(x2d(ktype))x2c(kdef),offset + 1 + 40 + (kscan-1)*entrywidth
            end
         end
      when cmd = 'swap' then do
         /*
         ** We here swap KeyDefs.  (key1 < key2)
         */
         call getentry offset
         call charin infile ,,(key1-1)*entrywidth
         key1def = charin(infile,,entrywidth)
         call charin infile ,,(key2-key1-1)*entrywidth
         key2def = charin(infile,,entrywidth)
         say 'Swapping' format(key1,4) c2x(key1def)
         say '    With' format(key2,4) c2x(key2def)
         call charout infile,key2def,offset+1+40+(key1-1)*entrywidth
         call charout infile,key1def,offset+1+40+(key2-1)*entrywidth
         end
      when cmd = 'set' then do
         /*
         ** We here set the specified flag.
         */
         call getentry offset
         say 'Replacing' d2x(flags,8) x2b(d2x(flags,8))
         flags = x2d(b2x(reverse(overlay('1',reverse(x2b(d2x(flags,8))),flag))))
         say '     With' d2x(flags,8) x2b(d2x(flags,8))
         call charin infile,offset+3,0
         call charout infile,d2l(flags)
         end
      when cmd = 'clear' then do
         /*
         ** We here clear the specified flag.
         */
         call getentry offset
         say 'Replacing' d2x(flags,8) x2b(d2x(flags,8))
         flags = x2d(b2x(reverse(overlay('0',reverse(x2b(d2x(flags,8))),flag))))
         say '     With' d2x(flags,8) x2b(d2x(flags,8))
         call charin infile,offset+3,0
         call charout infile,d2l(flags)
         end
   otherwise
   end  /* select */
   return

getaccententry: /* read a AccentTable entry, starting at current offset */
   /*
   ** The first six entries of the AccentTable have a fixed size (46
   ** bytes), but the width of the seventh entry may vary:
   **
   ** It's 46 bytes if the entry is empty.  Otherwise, it contains
   ** multiple subentries, and their size is stored in their first
   ** byte.
   */
   if arg(1) < 7 then
      len = 47
   else
      len = readb()
   if len = 0 then len = 47
 
   glyph = readb()
   b1 = readb()
   b2 = readb()
   b3 = readb()
   b4 = readb()
   b5 = readb()
   accententry = charin(infile,,len - 7)
   return
 
getkeydef: /* read a KeyDef entry, starting at current offset */
   /*
   ** We read the KeyDef chars in char1, char2, etc.  We have to
   ** read entrycount - sizeof(keytype) -- that is, entrycount - 2 --
   ** such chars.
   */
   keytype = readw()
   do char = 1 to entrywidth - 2
      call value 'char'char, readb()
   end /* do */
   return
 
getentry: /* read a KLE entry, starting at offset arg(1) */
   /*
   ** It just read the entry header.  It does not read keydefs and
   ** accent table entries.
   */
   call charin infile,arg(1)+1,0
   cp = readw()
   flags = readl()
   kbdtype = readw()
   kbdsubtype = readw()
   tablelen = readw()
   entrycount = readw()
   entrywidth = readw()
   country = reverse(charin(infile,,2))
   tabletypeid = readw()
   subcountry = charin(infile,,4)
   reserved = charin(infile,,16)
   return
 
emitentry: /* write current entry to output file */
   /*
   ** arg(1) is table offset;
   ** tablelen is the table length.
   */
   say 'Extracting table' country subcountry cp '('tablelen' bytes)'
   buff = charin(infile,arg(1)+1,tablelen)
   call charout outfile, buff
   return
 
saveentry: /* record a matching keyboard layout entry (KLE) */
   /*
   ** country is the KLE country code;
   ** subcntr is the KLE subcountry code;
   ** cp is the KLE codepage;
   ** offset is the KLE offset from the beginning of infile;
   ** type is the KLE keyboard type;
   ** w1, w2 and w3 are three fields whose meaning is currently
   ** unknown.
   */
   count = KLE.0 + 1
   KLE.count = left(country,7) left(subcntr,10) format(cp,8) format(offset,6) format(type,4) w1 w2
   KLE.0 = count
   return

getindex: /* read an index entry */
   w1 = readw()
   country = reverse(charin(infile,,2))
   subcntr = charin(infile,,4)
   w2 = readw()
   cp = readw()
   type = readw()
   offset = readl()
   return

flagsdef1: /* convert the KLE's flag to an human-readable representation */
   return d2x(flags,8) x2b(d2x(flags,8))
 
flagsdef2: /* convert the KLE's flag to a(nother) verbose representation */
   procedure expose flags flg
   bs = reverse(x2b(d2x(flags)))
   r = ''
   do i = 1 to length(bs)
      if substr(bs,i,1) then
         r = r word(flg,i)
   end /* do */
   return strip(r)
 
keydef: /* convert a KeyDef to an human-readable representation */
   result = ''
   do kd = 1 to entrywidth - 2
      result = result||v(value('char'kd))
   end /* do */
   return result
 
v: /* convert a decimal value to an human-readable representation */
   if arg(1) < 32 then
      return ' 0x'd2x(arg(1),2)' '
   else
      return '  "'d2c(arg(1))'" '
 
readb:   /* read one byte from infile */
   return x2d(c2x(charin(infile,,1)))
 
readw:   /* read one word from infile */
   return w2d(charin(infile,,2))

readl:   /* read one long from infile */
   return l2d(charin(infile,,4))

skip:    /* skip arg(1) chars */
   return charin(infile,,arg(1))

bit:     /* return bit arg(2) of arg(1) */
   return substr(x2b(d2x(arg(1),4)), arg(2),1)

w2d:     /* littleendian word to decimal */
   w = c2x(arg(1))
   return x2d(substr(w,3,2)substr(w,1,2))

d2w:     /* decimal to littleendian word */
   w = d2x(arg(1),4)
   return x2c(substr(w,3,2)substr(w,1,2))

l2d:     /* littleendian long to decimal */
   l = c2x(arg(1))
   return x2d(substr(l,7,2)substr(l,5,2)substr(l,3,2)substr(l,1,2))

d2l:     /* decimal to littleindian long */
   l = d2x(arg(1),8)
   return x2c(substr(l,7,2)substr(l,5,2)substr(l,3,2)substr(l,1,2))
 
halt:    /* abort operation */
   say '^C'
   exit
