//-------------------------------------------------------------------------
// The following is part of the VIOWIN user-interface library source code.
// The code is copyright (c) 1994-1995 by Larry Salomon, Jr. and may not
// be used in a for-profit application without the expressed, written
// consent of Larry Salomon, Jr.  All rights reserved.
//-------------------------------------------------------------------------
#define INCL_VIO
#include "vwdefs.h"
#include <string.h>
#include <ctype.h>

SHORT EXPENTRY vwDrawText(HVWWND hwndWnd,
                          SHORT sLenText,
                          PCHAR pchText,
                          PRECTL prclRect,
                          LONG lForeClr,
                          LONG lBackClr,
                          USHORT usFlags)
//-------------------------------------------------------------------------
// This function draws the specified text in the specified window.  It
// accepts the following flags:  DT_LEFT, DT_CENTER, DT_RIGHT, DT_TOP,
// DT_VCENTER, DT_BOTTOM, DT_WORDBREAK, and DT_ERASERECT.
//
// Input:  hwndWnd - handle to the window to draw within
//         sLenText - length of the text to draw.  -1 means the string
//                    is null-terminated and the entire string should be
//                    drawn
//         pchText - points to the text to draw
//         prclRect - specifies a rectangle to clip the text within.
//                    This is AND'd with the bounding rectangle of the
//                    window to determine the actual clipping rectangle of
//                    the text.
//         lForeClr, lBackClr - specify the foreground and background
//                              color of the text
//         usFlags - one or more DT_* flags
// Returns:  the number of characters drawn if successful, 0 otherwise
//-------------------------------------------------------------------------
{
   RECTL rclWnd;
   RECTL rclCopy;
   RECTL rclClip;
   SHORT sPos;
   USHORT usX;
   USHORT usY;
   SHORT sSearch;
   BYTE bAttr;

   if (!vwIsWindow(hwndWnd)) {
      return 0;
   } /* endif */

   //----------------------------------------------------------------------
   // If DT_WORDBREAK is specified, only DT_LEFT and DT_TOP may also be
   // specified.
   //----------------------------------------------------------------------
   if ((usFlags & DT_WORDBREAK)!=0) {
      if (usFlags & (DT_LEFT | DT_CENTER | DT_RIGHT)!=DT_LEFT) {
         return 0;
      } /* endif */

      if (usFlags & (DT_TOP | DT_VCENTER | DT_BOTTOM)!=DT_TOP) {
         return 0;
      } /* endif */
   } /* endif */

   //----------------------------------------------------------------------
   // Calculate the true clipping rectangle
   //----------------------------------------------------------------------
   rclWnd.xLeft=hwndWnd->swpSwp.lX;
   rclWnd.xRight=rclWnd.xLeft+hwndWnd->swpSwp.ulCx-1;
   rclWnd.yBottom=hwndWnd->swpSwp.lY;
   rclWnd.yTop=rclWnd.yBottom+hwndWnd->swpSwp.ulCy-1;

   if (prclRect!=NULL) {
      rclCopy=*prclRect;

      rclCopy.xLeft+=hwndWnd->swpSwp.lX;
      rclCopy.xRight+=hwndWnd->swpSwp.lX;
      rclCopy.yBottom+=hwndWnd->swpSwp.lY;
      rclCopy.yTop+=hwndWnd->swpSwp.lY;

      vwIntersectRect(&rclClip,&rclCopy,&rclWnd);
   } else {
      rclClip=rclWnd;
   } /* endif */

   //----------------------------------------------------------------------
   // If the clipping rectangle is empty, return.
   //----------------------------------------------------------------------
   if (vwIsRectEmpty(&rclClip)) {
      return 0;
   } /* endif */

   if (sLenText==-1) {
      sLenText=strlen(pchText);
   } /* endif */

   //----------------------------------------------------------------------
   // If the text exceeds the clipping rectangle, determine the place
   // to stop drawing.  This only gets complicated if DT_WORDBREAK was
   // specified.
   //----------------------------------------------------------------------
   if (sLenText>(rclClip.xRight-rclClip.xLeft+1)) {
      sLenText=rclClip.xRight-rclClip.xLeft+1;

      if ((usFlags & DT_WORDBREAK)!=0) {
         sSearch=sLenText;

         while ((sSearch>0) && (!isspace(pchText[sSearch]))) {
            sSearch--;
         } /* endwhile */

         if (sSearch!=0) {
            sLenText=sSearch;
         } /* endif */

         while ((sSearch>0) && (isspace(pchText[sSearch]))) {
            sSearch--;
         } /* endwhile */

         if (sSearch!=0) {
            sLenText=sSearch+1;
         } /* endif */
      } /* endif */
   } /* endif */

   //----------------------------------------------------------------------
   // Determine the positioning of the text within the clipping rectangle
   //----------------------------------------------------------------------
   sPos=0;

   if ((usFlags & DT_CENTER)!=0) {
      usX=rclClip.xLeft+(rclClip.xRight-rclClip.xLeft+1)/2-sLenText/2;

      if (usX<rclClip.xLeft) {
         sPos=rclClip.xLeft-usX;
         usX=rclClip.xLeft;
      } /* endif */
   } else
   if ((usFlags & DT_RIGHT)!=0) {
      usX=rclClip.xRight-sLenText;
   } else {
      usX=rclClip.xLeft;
   } /* endif */

   if ((usFlags & DT_VCENTER)!=0) {
      usY=rclClip.yBottom+(rclClip.yTop-rclClip.yBottom+1)/2;
   } else
   if ((usFlags & DT_BOTTOM)!=0) {
      usY=rclClip.yBottom;
   } else {
      usY=rclClip.yTop;
   } /* endif */

   //----------------------------------------------------------------------
   // Erase if needed
   //----------------------------------------------------------------------
   if ((usFlags & DT_ERASERECT)!=0) {
      vwFillRect(hwndWnd,prclRect,lBackClr);
   } /* endif */

   //----------------------------------------------------------------------
   // Draw the text.
   //----------------------------------------------------------------------
   bAttr=((((BYTE)lBackClr) & 0x07)<<4)|(((BYTE)lForeClr) & 0x07);

   VioWrtCharStrAtt(&pchText[sPos],
                    sLenText,
                    vwQuerySysValue(VWSV_CYSCREEN)-1-usY,
                    usX,
                    &bAttr,
                    NULLHANDLE);

   return sLenText;
}

BOOL EXPENTRY vwFillRect(HVWWND hwndWnd,PRECTL prclRect,LONG lColor)
//-------------------------------------------------------------------------
// This function fills a rectangle with the specified color.
//
// Input:  hwndWnd - handle to the window to draw within
//         prclRect - points to the RECTL structure specifying the
//                    rectangle to fill.  This is required.
//         lColor - the color to use
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------
{
   RECTL rclWnd;
   RECTL rclCopy;
   RECTL rclClip;
   BYTE abCell[2];

   if (!vwIsWindow(hwndWnd)) {
      return FALSE;
   } /* endif */

   if (prclRect==NULL) {
      return FALSE;
   } /* endif */

   rclWnd.xLeft=hwndWnd->swpSwp.lX;
   rclWnd.xRight=rclWnd.xLeft+hwndWnd->swpSwp.ulCx-1;
   rclWnd.yBottom=hwndWnd->swpSwp.lY;
   rclWnd.yTop=rclWnd.yBottom+hwndWnd->swpSwp.ulCy-1;

   rclCopy=*prclRect;

   rclCopy.xLeft+=hwndWnd->swpSwp.lX;
   rclCopy.xRight+=hwndWnd->swpSwp.lX;
   rclCopy.yBottom+=hwndWnd->swpSwp.lY;
   rclCopy.yTop+=hwndWnd->swpSwp.lY;

   vwIntersectRect(&rclClip,&rclCopy,&rclWnd);

   abCell[0]=' ';
   abCell[1]=((((BYTE)lColor) & 0x07)<<4);

   VioScrollDn(vwQuerySysValue(VWSV_CYSCREEN)-1-rclClip.yTop,
               rclClip.xLeft,
               vwQuerySysValue(VWSV_CYSCREEN)-1-rclClip.yBottom,
               rclClip.xRight,
               999,
               abCell,
               NULLHANDLE);
   return TRUE;
}
