/*
$Revision:   1.28  $
$Date:   Jan 26 1999 17:58:00  $

*/

/*
 * Copyright 1993-1996
 * Quatech Inc.
 * Akron, Ohio, U.S.A.
 * All Rights Reserved
 *
 *
 * Title:    SENDBI.C
 *
 * Function: Bit synchronous transmit program
 *
 * Notes:    This program transmits on a bit synchronous link between two
 *           MPA-series boards connected by a cable.  The receiving board
 *           will typically run the complimenting RECBI program.
 *
 *           DMA is turned on by default.  Use the /t0 option on the
 *           command line to not use DMA.
 *
 *           A straight-through cable can be used if one board is DTE and
 *           the other is DCE.  The MPA-100 and MPA-102 can be configured for
 *           DTE or DCE with jumper selections.  The MPA-200/300 must be
 *           purchased as either DTE or DCE and cannot be user-configured.
 *           The MPAP-100 is only available for DTE.
 *
 *           If this program is run using two DTEs or two DCEs, a null-modem
 *           cable must be used.  At a minimum, the Tx and Rx data signals
 *           must be cross-connected.
 *
 *           By default, this program transmits its Tx clock for the receiver
 *           to use as a receive clock.  If this desired, the clock signals
 *           should also be cross-connected.  Otherwise, the output Tx clock
 *           can be ignored, or turned off by changing pMPA_cfg->clock_source
 *           to 0x02.
 *
 *
 *
 * DISCLAIMER --- This program is for demonstration purposes only.
 *
 *                Quatech makes no claims about the suitability of this
 *                example program for any specific application.
 *
 */


// Ensure that board type macro has been set.
#if   defined MPA200
#elif defined MPA102
#else
   #error You must define a board type macro for the compiler!
   #error Use /DMPA102 for the MPA-102.
   #error Use /DMPA200 for the MPA-100/200/300 (or MPAP-x00)
#endif


// Syncdrive include files used

#ifdef __OS2__
#include <os2.h>
#include "syncos2.h"
#endif

#ifdef _WINDOWS
#include <windows.h>
   #ifndef __WIN95__
        #include "snc31dll.h"
   #else
        #include "snc95dll.h"
   #endif
#endif

#include "syncdriv.h"
#include "mpa-x00.h"

// Standard C include files used

#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <signal.h>


// Global variables

// Counter elements to count frames transmitted.
long    counter;
char digits[15];


// This is the string we will transmit.
char *pMsgString = "Jim's Chicago Style Pizza and German Beer Stand ";


// configuration array pointer.
struct channel_cfg _far *pMPA_cfg;


// Transmit buffer pointer.
union com_block _far *pTxBuffer;


// Buffer length, including space for overhead and the counter bytes.
short BufferLen;


#ifdef _WINDOWS
// Windows memory handle for config array.
HGLOBAL  ConfigHandle;
#endif


// Other miscellaneous variables.
short       rc;


void init_config_array(void)
{
   // Set up all initial values in the configuration array.

   pMPA_cfg->signature           = 0xcc;

#ifdef MPA102
   pMPA_cfg->structure_type      = 0x93;
#endif
#ifdef MPA200
   pMPA_cfg->structure_type      = 0x90;
#endif

   pMPA_cfg->clock_rate          = 9830400L;
   pMPA_cfg->board_number        = 0;        // arbitrarily chosen
   pMPA_cfg->channel             = 0x0;
   pMPA_cfg->operating_mode      = 0x08;
   pMPA_cfg->line_control        = 0xF0;
   pMPA_cfg->options             = 0x0;
   pMPA_cfg->base_address        = 0x300;
   pMPA_cfg->tx_dma_channel      = 3;
   pMPA_cfg->rx_dma_channel      = 0;
   pMPA_cfg->tx_interrupt        = 5;
   pMPA_cfg->rx_interrupt        = 5;
   pMPA_cfg->crc_mode            = 0x82;
   pMPA_cfg->clock_source        = 0x06;
   pMPA_cfg->rx_baud_rate        = 9600;
   pMPA_cfg->tx_baud_rate        = 9600;
   pMPA_cfg->read_timeout        = 0;
   pMPA_cfg->write_timeout       = 0;
   pMPA_cfg->sync_char_1         = 0x0;
   pMPA_cfg->sync_char_2         = 0x7e;
   pMPA_cfg->protocol_dependent0 = 0;
   pMPA_cfg->protocol_dependent1 = 0;
   pMPA_cfg->protocol_dependent2 = 0;
   pMPA_cfg->protocol_dependent3 = 0;        // 0x01 = FIFO mode
   pMPA_cfg->protocol_dependent4 = 0;
   pMPA_cfg->protocol_dependent5 = 0;
   pMPA_cfg->protocol_dependent6 = 0;
   pMPA_cfg->protocol_dependent7 = 0;

}


// Convert ascii hex to unsigned int.
USHORT hextoui(char *numbertext)
{
    return((USHORT)strtol(numbertext, (char **)NULL, 16));
}

// Convert ascii decimal to unsigned int.
USHORT atoui(char *numbertext)
{
    return((USHORT)strtol(numbertext, (char **)NULL, 10));
}

// Convert ascii decimal to unsigned char.
UCHAR atouc(char *numbertext)
{
    return((UCHAR)strtol(numbertext, (char **)NULL, 10));
}

// Convert ascii decimal to unsigned long.
ULONG atoul(char *numbertext)
{
    return((ULONG)strtol(numbertext, (char **)NULL, 10));
}


void process_options(int argc, char *argv[])
{
   int     index;
   char    *charptr;

   // Loop through all command line parameters, skipping the program name.
   for (index = 1; index <= (argc-1); index++)
      {
      charptr = argv[index];

      // First character of parameter must be a slash.
      if (*charptr == '/')
         {
         charptr++;

         // The switch type follows the slash.  Decode the type and
         //   pass a pointer to the variable data to the conversion
         //   routine.

         switch (*charptr)
            {
            // Change base address.
            case 'a':
            case 'A':
               pMPA_cfg->base_address = hextoui(++charptr);
               break;

            // Change IRQ.
            case 'i':
            case 'I':
               pMPA_cfg->tx_interrupt = atouc(++charptr);
               pMPA_cfg->rx_interrupt = pMPA_cfg->tx_interrupt;
               break;

            // Change baud rate.
            case 'b':
            case 'B':
               pMPA_cfg->rx_baud_rate = atoul(++charptr);
               pMPA_cfg->tx_baud_rate = pMPA_cfg->rx_baud_rate;
               break;

            // Change Tx DMA channel.
            case 't':
            case 'T':
               pMPA_cfg->tx_dma_channel = atouc(++charptr);
               break;

            
            default:
               printf("\nSENDBI (options)\n");
               printf("\nOptions:\n");
               printf("/A<base address in hex>\n");
               printf("/I<irq>\n");
               printf("/B<baud rate>\n");
               printf("/T<transmit dma channel>\n");
               exit(-1);
               break;
            }
         }
      }

}

short transmit(void)
{
   counter++;
        ltoa(counter, digits, 10);

   strcpy ((char *)pTxBuffer->BitT.comm_buffer,pMsgString);

        strcat(pTxBuffer->BitT.comm_buffer, digits);

   // Initialize the transmit buffer comm_block

   pTxBuffer->BitT.buffer_length = strlen((char*)pTxBuffer->BitT.comm_buffer);
   pTxBuffer->BitT.buffer_status = 0;
   pTxBuffer->BitT.idle_flag =  0;

#ifdef __OS2__
   rc = sync_transmit (pMPA_cfg, pTxBuffer, 10000);
#elif defined _WINDOWS
   rc = sync_transmit (pMPA_cfg, pTxBuffer, 0);
#else  // DOS
   rc = sync_transmit (pMPA_cfg, pTxBuffer);
#endif

   if (rc)
      {
      printf("Error starting frame transmission %04x", rc);
      return(-1);
      }

   while (!(pTxBuffer->BitT.buffer_status & (TxDone | TxAbort | TxError)));

   if (pTxBuffer->BitT.buffer_status & TxError)
      printf("Transmitter Error\n");
   if (pTxBuffer->BitT.buffer_status & TxAbort)
      printf("Transmitter underrun\n");

   return(pTxBuffer->BitT.buffer_status);

}

void SyncDriveExit(void)
{
   // This exit routine takes care of required cleanup.
   // It's important not to exit with any locked memory lying around!

   // Print current buffer status.
   if (pTxBuffer)
      printf("Tx buffer status = %02x \n",pTxBuffer->BitT.buffer_status);

   // Stop the transmitter.
   rc = sync_abort_message(pMPA_cfg);

   // Release the channel.
   rc = sync_release(pMPA_cfg);

   // Free up the Tx buffer.
   if (pMPA_cfg->tx_dma_channel != 0)
      rc = sync_free_dma_buffers();
   else
      if (pTxBuffer)
         free(pTxBuffer);

   // Release the channel config array memory.
   free(pMPA_cfg);

}

void AbnormalExit(int signal)
{
   // This function ensures that the exit handler gets
   // called if Ctrl-C, Ctrl-Break, etc. occurs.
   exit(0);
}





void main(int argc, char *argv[])
{
   time_t t0, t1;
   USHORT  DriverRelease;
   USHORT  DLLRelease;

   
// Setup handlers for Ctrl-C, Ctrl-Break, etc.
   signal(SIGINT, AbnormalExit);
   signal(SIGTERM, AbnormalExit);

#if defined(__OS2__) || defined(__WIN95__)
   signal(SIGBREAK, AbnormalExit);
#endif

   atexit(SyncDriveExit);

   // Allocate memory for the channel configuration array.
   if ((pMPA_cfg = malloc(sizeof(struct channel_cfg))) == NULL)
      {
      printf("Cannot allocate config array!\n");
      exit(-1);
      }
   
   printf("\n*** Quatech MPA-series adapter transmit example program ***\n\n");
   printf("Bit synchronous link between two boards, connected by a cable.\n");
   printf("See the source code for cabling info.\n\n");
   printf("Press Ctrl-Break to stop the program.\n\n");

   // Initialize the channel configuration array.

   init_config_array();

   // Check the SyncDrive version.
   rc = sync_version(pMPA_cfg, &DriverRelease, &DLLRelease);
   if (rc != SYNC_SUCCESS)
   {
      printf("Syncdrive release level error!\n\n");
      printf("Driver release level is %04x, DLL release level is %04x\n",
             DriverRelease, DLLRelease);
      exit(-1);
   }
   printf("Syncdrive release level is %04x\n\n", DriverRelease); 

   
   // Now process any options that override the defaults.

   if (argc > 1)
       process_options(argc, argv);


    // Buffer length is the string, the com_block structure overhead, and
    //  some extra space for the counter bytes.

    BufferLen = strlen(pMsgString) + COMBLOCK_OVERHEAD + 0x10;


    // Allocate the Tx buffer.  Use special allocate routine if using DMA.

    if (pMPA_cfg->tx_dma_channel != 0)
       {
       if (sync_alloc_dma_buffer(pMPA_cfg, BufferLen, &pTxBuffer) != 0)
          {
          printf("DMA Buffer Allocation Failed\n");
          exit(-1);
          }
       }
    else
       {
       if ((pTxBuffer = malloc(BufferLen)) == NULL)
          {
          printf("Buffer Allocation Failed\n");
          exit(-1);
          }
       }



   // Configure and open the channel.

#ifdef MPA102
   printf("Channel configuration status = %04x \n",config_MPA102(pMPA_cfg));
#endif
#ifdef MPA200
   printf("Channel configuration status = %04x \n",config_MPA200(pMPA_cfg));
#endif


   // Print configuration details to the screen.

   printf("\nBase address = %03x\n", pMPA_cfg->base_address);
   printf("Transmit interrupt = %i\n", pMPA_cfg->tx_interrupt);
   if (pMPA_cfg->tx_dma_channel != 0)
      printf("Transmit DMA channel = %i\n", pMPA_cfg->tx_dma_channel);
   else
      printf("DMA not in use.\n");
   printf("Clockrate = %.li\n", pMPA_cfg->clock_rate);
   printf("Baudrate = %.li\n\n", pMPA_cfg->tx_baud_rate);

#ifdef __WIN95__
   printf("\nFor best performance, run this application from a full screen Command Prompt.\n\n");
#endif
   printf("Press any key to start.\n");
   while(!kbhit());
   getch();


   while(1)
      {
      // initialize and transmit buffer.

      transmit();


      // put terminator in buffer to accomadate the printf statement.

      pTxBuffer->BitT.comm_buffer[pTxBuffer->BitT.buffer_length+10] = 0;


      // and print transmitted buffer

      printf("%s \n", pTxBuffer->BitT.comm_buffer);

      // Wait for a second before sending the next frame.
      t0 = time( &t0 );
      while( (time( &t1 ) - t0) < 1 );

#ifdef _WINDOWS
      // Windows "QuickWin" app won't respond to Ctrl-Break unless some
      //  console I/O is done.  This can impact performance under DOS,
      //  so we'll make it conditional to Windows.
      if (kbhit())
         getch();
#endif

      }

}
