/*
$Revision:   1.26  $
$Date:   Jan 26 1999 17:57:58  $

*/

/*
 * Copyright 1993-1996
 * Quatech Inc.
 * Akron, Ohio, U.S.A.
 * All Rights Reserved
 *
 *
 * Title:    RECBI.C
 *
 * Function: Bit synchronous receive program
 *
 * Notes:    This program receives on a bit synchronous link between two
 *           MPA-series boards connected by a cable.  The transmitting board
 *           will typically run the complimenting SENDBI program.
 *
 *           DMA is turned on by default.  Use the /r0 option on the
 *           command line to not use DMA.
 *
 *           A straight-through cable can be used if one board is DTE and
 *           the other is DCE.  The MPA-100 and MPA-102 can be configured for
 *           DTE or DCE with jumper selections.  The MPA-200/300 must be
 *           purchased as either DTE or DCE and cannot be user-configured.
 *           The MPAP-100 is only available for DTE.
 *
 *           If this program is run using two DTEs or two DCEs, a null-modem
 *           cable must be used.  At a minimum, the Tx and Rx data signals
 *           must be cross-connected.  By default, this program receives its
 *           Rx clock, so the clock signals should also be cross-connected.
 *
 *           Alternatively, the pMPA_cfg->clock_source could be set to 0x16,
 *           which would cause the Rx clock to be locally sourced from the
 *           baud rate generator.  This would avoid the need for cross-
 *           connected clock signals.  If this is done, the user must specify
 *           the proper baud rate (/b option) on the command line when the
 *           program is started.
 *
 *
 * DISCLAIMER --- This program is for demonstration purposes only.
 *
 *                Quatech makes no claims about the suitability of this
 *                example program for any specific application.
 *
 */


// Ensure that board type macro has been set.
#if   defined MPA200
#elif defined MPA102
#else
   #error You must define a board type macro for the compiler!
   #error Use /DMPA102 for the MPA-102.
   #error Use /DMPA200 for the MPA-100/200/300 (or MPAP-x00)
#endif


// Syncdrive include files used

#ifdef __OS2__
#include <os2.h>
#include "syncos2.h"
#endif

#ifdef _WINDOWS
#include <windows.h>
   #ifndef __WIN95__
        #include "snc31dll.h"
   #else
        #include "snc95dll.h"
   #endif
#endif

#include "syncdriv.h"
#include "mpa-x00.h"


// Standard C include files used

#include <conio.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>



// Global variables

// configuration array pointer.
struct channel_cfg _far *pMPA_cfg;


// Receive buffer pointer.
union com_block _far *pRxBuffer;


#ifdef _WINDOWS
// Windows memory handle for config array.
HGLOBAL  ConfigHandle;
#endif


// Other miscellaneous variables.
int         ErrorCount = 0;
short       rc;



void init_config_array(void)
{
   // Set up all initial values in the configuration array.

   pMPA_cfg->signature           = 0xcc;

#ifdef MPA102
   pMPA_cfg->structure_type      = 0x93;
#endif
#ifdef MPA200
   pMPA_cfg->structure_type      = 0x90;
#endif

   pMPA_cfg->clock_rate          = 9830400L;
   pMPA_cfg->board_number        = 0;        // arbitrarily chosen
   pMPA_cfg->channel             = 0;
   pMPA_cfg->operating_mode      = 0x08;     // HDLC, NRZ (No PLL)
   pMPA_cfg->line_control        = 0xF0;
   pMPA_cfg->options             = 0x0;
   pMPA_cfg->base_address        = 0x300;
   pMPA_cfg->tx_dma_channel      = 0;
   pMPA_cfg->rx_dma_channel      = 1;
   pMPA_cfg->tx_interrupt        = 5;
   pMPA_cfg->rx_interrupt        = 5;
   pMPA_cfg->crc_mode            = 0x82;
   pMPA_cfg->clock_source        = 0x06;     // X1 mode, Rx Src = RTXC  (NRZ, No PLL)
   pMPA_cfg->rx_baud_rate        = 9600;
   pMPA_cfg->tx_baud_rate        = 9600;
   pMPA_cfg->read_timeout        = 0;
   pMPA_cfg->write_timeout       = 0;
   pMPA_cfg->sync_char_1         = 0x0;
   pMPA_cfg->sync_char_2         = 0x7e;
   pMPA_cfg->protocol_dependent0 = 0;
   pMPA_cfg->protocol_dependent1 = 0;
   pMPA_cfg->protocol_dependent2 = 0;
   pMPA_cfg->protocol_dependent3 = 0;        // 0x01 = FIFO mode
   pMPA_cfg->protocol_dependent4 = 0;
   pMPA_cfg->protocol_dependent5 = 0;
   pMPA_cfg->protocol_dependent6 = 0;
   pMPA_cfg->protocol_dependent7 = 0;

}

// Convert ascii hex to unsigned int.
USHORT hextoui(char *numbertext)
{
    return((USHORT)strtol(numbertext, (char **)NULL, 16));
}

// Convert ascii decimal to unsigned int.
USHORT atoui(char *numbertext)
{
    return((USHORT)strtol(numbertext, (char **)NULL, 10));
}

// Convert ascii decimal to unsigned char.
UCHAR atouc(char *numbertext)
{
    return((UCHAR)strtol(numbertext, (char **)NULL, 10));
}

// Convert ascii decimal to unsigned long.
ULONG atoul(char *numbertext)
{
    return((ULONG)strtol(numbertext, (char **)NULL, 10));
}


void process_options(int argc, char *argv[])
{
   int     index;
   char    *charptr;

   // Loop through all command line parameters, skipping the program name.
   for (index = 1; index <= (argc-1); index++)
      {
      charptr = argv[index];

      // First character of parameter must be a slash.
      if (*charptr == '/')
         {
         charptr++;

         // The switch type follows the slash.  Decode the type and
         //   pass a pointer to the variable data to the conversion
         //   routine.

         switch (*charptr)
            {
            // Change base address.
            case 'a':
            case 'A':
                pMPA_cfg->base_address = hextoui(++charptr);
                break;

            // Change IRQ.
            case 'i':
            case 'I':
                pMPA_cfg->tx_interrupt = atouc(++charptr);
                pMPA_cfg->rx_interrupt = pMPA_cfg->tx_interrupt;
                break;

            // Change baud rate (ignored if receiving the Rx clock).
            case 'b':
            case 'B':
                pMPA_cfg->rx_baud_rate = atoul(++charptr);
                pMPA_cfg->tx_baud_rate = pMPA_cfg->rx_baud_rate;
                break;

            // Change Rx DMA channel.
            case 'r':
            case 'R':
                pMPA_cfg->rx_dma_channel = atouc(++charptr);
                break;

            
            default:
                printf("\nRECBI (options)\n");
                printf("\nOptions:\n");
                printf("/A<base address in hex>\n");
                printf("/I<irq>\n");
                printf("/B<baud rate>\n");
                printf("/R<receive dma channel>\n\n");
                exit(-1);
                break;
            }
         }
      }

}


// Routine sets up parameters in the receive buffer and
// initializes the receive hardware.

short receive(void)
{
   // Initialize the transmit buffer comm_block.

   pRxBuffer->BitR.buffer_length = buffer_size;
   pRxBuffer->BitR.buffer_status = 0;       // initialize buffer status to 0

#ifdef __OS2__
   rc = sync_receive(pMPA_cfg,pRxBuffer,10000);
#elif defined _WINDOWS
   rc = sync_receive(pMPA_cfg,pRxBuffer,0);
#else  // DOS
   rc = sync_receive(pMPA_cfg,pRxBuffer);
#endif

   if (rc)
      {
      printf("Error starting frame reception %04x", rc);
      return(-1);
      }


   // Wait for frame to complete.
   while (!(pRxBuffer->BitR.buffer_status & (RxDone | RxFull | RxAbort | RxError)))
      {
#ifdef _WINDOWS
      // Windows "QuickWin" app won't respond to Ctrl-Break unless some
      //  console I/O is done.  This can impact performance under DOS,
      //  so we'll make it conditional to Windows.
      if (kbhit())
         getch();
#endif
      }


   if (pRxBuffer->BitR.buffer_status & RxError)
      printf("Receiver Error\n");
   if (pRxBuffer->BitR.buffer_status & RxAbort)
      printf("Received Abort\n");
   if (pRxBuffer->BitR.buffer_status & RxFull)
       printf("Received Overrun\n");


   return(pRxBuffer->BitR.buffer_status);

}

void SyncDriveExit(void)
{
   // This exit routine takes care of required cleanup.
   // It's important not to exit with any locked memory lying around!

   // Print current buffer status.
   if (pRxBuffer)
      printf("Rx buffer status = %02x \n",pRxBuffer->BitR.buffer_status);

   // Stop the receiver.
   rc = sync_command(pMPA_cfg, 4);

   // Release the channel.
   rc = sync_release(pMPA_cfg);

   // Free up the Rx buffer.
   if (pMPA_cfg->rx_dma_channel != 0)
      sync_free_dma_buffers();
   else
      {
      if (pRxBuffer)
         free(pRxBuffer);
      }

   // Release the channel config array memory.
   free(pMPA_cfg);

}



void AbnormalExit(int signal)
{
   // This function ensures that the exit handler gets
   // called if Ctrl-C, Ctrl-Break, etc. occurs.
   exit(0);
}






void main(int argc, char *argv[])
{
   USHORT  DriverRelease;
   USHORT  DLLRelease;

   
   // Setup handlers for Ctrl-C, Ctrl-Break, etc.
   signal(SIGINT, AbnormalExit);
   signal(SIGTERM, AbnormalExit);

#if defined(__OS2__) || defined(__WIN95__)
   signal(SIGBREAK, AbnormalExit);
#endif

   atexit(SyncDriveExit);

   // Allocate memory for the channel configuration array.
   if ((pMPA_cfg = malloc(sizeof(struct channel_cfg))) == NULL)
      {
      printf("Cannot allocate config array!\n");
      exit(-1);
      }
   
   printf("\n*** Quatech MPA-series adapter receive example program ***\n\n");
   printf("Bit synchronous link between two boards, connected by a cable.\n");
   printf("See the source code for cabling info.\n\n");
   printf("Press Ctrl-Break to stop the program.\n\n");


   // Initialize the channel configuration array.

   init_config_array();

   // Check the SyncDrive version.
   rc = sync_version(pMPA_cfg, &DriverRelease, &DLLRelease);
   if (rc != SYNC_SUCCESS)
   {
      printf("Syncdrive release level error!\n\n");
      printf("Driver release level is %04x, DLL release level is %04x\n",
             DriverRelease, DLLRelease);
      exit(-1);
   }
   printf("Syncdrive release level is %04x\n\n", DriverRelease); 

   
   // Now process any options that override the defaults.

   if (argc > 1)
       process_options(argc, argv);


   // Allocate the Rx buffer.  Use special allocate routine if using DMA.

   if (pMPA_cfg->rx_dma_channel != 0)
      {
      if (sync_alloc_dma_buffer(pMPA_cfg,buffer_size + COMBLOCK_OVERHEAD,&pRxBuffer) != 0)
         {
         printf("DMA Buffer Allocation Failed\n");
         exit(-1);
         }
      }
   else
      {
      if ((pRxBuffer = malloc(buffer_size + COMBLOCK_OVERHEAD)) == NULL)
         {
         printf("Buffer Allocation Failed\n");
         exit(-1);
         }
      }

   // Configure and open the channel.

#ifdef MPA102
   printf("Channel configuration status = %04x \n", config_MPA102(pMPA_cfg));
#endif
#ifdef MPA200
   printf("Channel configuration status = %04x \n", config_MPA200(pMPA_cfg));
#endif


   // Print configuration details to the screen.

   printf("\nBase address = %03x\n", pMPA_cfg->base_address);
   printf("Receive interrupt = %i\n", pMPA_cfg->rx_interrupt);
   if (pMPA_cfg->rx_dma_channel != 0)
      printf("Receive DMA channel = %i\n", pMPA_cfg->rx_dma_channel);
   else
      printf("DMA not in use.\n");
   printf("Clockrate = %.li\n", pMPA_cfg->clock_rate);
   printf("Baudrate = %.li\n\n", pMPA_cfg->rx_baud_rate);

#ifdef __WIN95__
   printf("\nFor best performance, run this application from a full screen Command Prompt.\n\n");
#endif

   while(1)
      {
      // initialize receiver, and receive frame

      receive();


      // put terminator in buffer to accommodate the printf statement, and compare.

      pRxBuffer->BitR.comm_buffer[pRxBuffer->BitR.buffer_pointer - 2] = 0;


      // print received buffer.

      printf("%s \n",pRxBuffer->BitR.comm_buffer);


      }
}
