/*
$Revision:   1.14  $
$Date:   Jan 26 1999 17:57:56  $

*/

/*
 * Copyright 1996
 * Quatech Inc.
 * Akron, Ohio, U.S.A.
 * All Rights Reserved
 *
 *
 * Title:    QRECBI.C
 *
 * Function: Bit synchronous receive program using a receive queue
 *
 * Notes:    This program receives on a bit synchronous link between two
 *           MPA-series boards connected by a cable.  The transmitting board
 *           will typically run the complimenting QSENDBI program.
 *
 *           DMA is turned ob by default.  Use the /r0 option on the
 *           command line to turn off DMA.
 *
 *           A straight-through cable can be used if one board is DTE and
 *           the other is DCE.  The MPA-100 and MPA-102 can be configured for
 *           DTE or DCE with jumper selections.  The MPA-200/300 must be
 *           purchased as either DTE or DCE and cannot be user-configured.
 *           The MPAP-100 is only available for DTE.
 *
 *           If this program is run using two DTEs or two DCEs, a null-modem
 *           cable must be used.  At a minimum, the Tx and Rx data signals
 *           must be cross-connected.  By default, this program receives its
 *           Rx clock, so the clock signals should also be cross-connected.
 *
 *           Alternatively, the pMPA_cfg->clock_source could be set to 0x16,
 *           which would cause the Rx clock to be locally sourced from the
 *           baud rate generator.  This would avoid the need for cross-
 *           connected clock signals.  If this is done, the user must specify
 *           the proper baud rate (/b option) on the command line when the
 *           program is started.
 *
 *
 * DISCLAIMER --- This program is for demonstration purposes only.
 *
 *                Quatech makes no claims about the suitability of this
 *                example program for any specific application.
 *
 */


// Ensure that board type macro has been set.
#if   defined MPA200
#elif defined MPA102
#else
   #error You must define a board type macro for the compiler!
   #error Use /DMPA102 for the MPA-102.
   #error Use /DMPA200 for the MPA-100/200/300 (or MPAP-x00)
#endif


// Syncdrive include files used

#ifdef __OS2__
#include <os2.h>
#include "syncos2.h"
#endif

#ifdef _WINDOWS
#include <windows.h>
   #ifndef __WIN95__
        #include "snc31dll.h"
   #else
        #include "snc95dll.h"
   #endif
#endif

#include "syncdriv.h"
#include "mpa-x00.h"


// Standard C include files used

#include <conio.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>

#define MAX_BUFFERS 1000


// Global variables

// Number of buffers in the receive queue.  Set default here.
USHORT num_buffers = 100;


// This is the string we expect to receive.
char *pMsgString = "Jim's Chicago Style Pizza and German Beer Stand";


// configuration array pointer.
struct channel_cfg _far *pMPA_cfg;


// Buffer length, default set here.
USHORT BufferLen = 1024;


// Receive buffer pointers.
union com_block _far *pRxBuffer[MAX_BUFFERS];


// Receive queue pointer.
struct BufferQueue _far *pRxQueue;


// Queue mode.
ULONG  QueueMode = QUEUE_RING;


#ifdef _WINDOWS
// Windows memory handle for config array.
HGLOBAL  ConfigHandle;
#endif


// Other miscellaneous variables.
USHORT       index;
USHORT       done;
USHORT       this_rx;
USHORT       rc;



void init_config_array(void)
{
   // Set up all initial values in the configuration array.

   pMPA_cfg->signature           = 0xcc;

#ifdef MPA102
   pMPA_cfg->structure_type      = 0x93;
#endif
#ifdef MPA200
   pMPA_cfg->structure_type      = 0x90;
#endif

   pMPA_cfg->clock_rate          = 9830400L;
   pMPA_cfg->board_number        = 0;        // arbitrarily chosen
   pMPA_cfg->channel             = 0x0;
   pMPA_cfg->operating_mode      = 0x08;
   pMPA_cfg->line_control        = 0xf0;
   pMPA_cfg->options             = 0x0;
   pMPA_cfg->base_address        = 0x300;
   pMPA_cfg->tx_dma_channel      = 0;
   pMPA_cfg->rx_dma_channel      = 1;
   pMPA_cfg->tx_interrupt        = 5;
   pMPA_cfg->rx_interrupt        = 5;
   pMPA_cfg->crc_mode            = 0x82;
   pMPA_cfg->clock_source        = 0x06;     // RxCLK src=RTxC (x1,NRZ,No PLL)
   pMPA_cfg->rx_baud_rate        = 9600;
   pMPA_cfg->tx_baud_rate        = 9600;
   pMPA_cfg->read_timeout        = 0;
   pMPA_cfg->write_timeout       = 0;
   pMPA_cfg->sync_char_1         = 0;
   pMPA_cfg->sync_char_2         = 0x7e;
   pMPA_cfg->protocol_dependent0 = 0;
   pMPA_cfg->protocol_dependent1 = 0;
   pMPA_cfg->protocol_dependent2 = 0;
   pMPA_cfg->protocol_dependent3 = 0;        // 0x01 = FIFO mode (85230 only)
   pMPA_cfg->protocol_dependent4 = 0;
   pMPA_cfg->protocol_dependent5 = 0;
   pMPA_cfg->protocol_dependent6 = 0;
   pMPA_cfg->protocol_dependent7 = 0;

}

// Convert ascii hex to unsigned int.
USHORT hextoui(char *numbertext)
{
    return((USHORT)strtol(numbertext, (char **)NULL, 16));
}

// Convert ascii decimal to unsigned int.
USHORT atoui(char *numbertext)
{
    return((USHORT)strtol(numbertext, (char **)NULL, 10));
}

// Convert ascii decimal to unsigned char.
UCHAR atouc(char *numbertext)
{
    return((UCHAR)strtol(numbertext, (char **)NULL, 10));
}

// Convert ascii decimal to unsigned long.
ULONG atoul(char *numbertext)
{
    return((ULONG)strtol(numbertext, (char **)NULL, 10));
}


void process_options(int argc, char *argv[])
{
   int     index;
   char    *charptr;

   // Loop through all command line parameters, skipping the program name.
   for (index = 1; index <= (argc-1); index++)
      {
      charptr = argv[index];

      // First character of parameter must be a slash.
      if (*charptr == '/')
         {
         charptr++;

         // The switch type follows the slash.  Decode the type and
         //   pass a pointer to the variable data to the conversion
         //   routine.

         switch (*charptr)
            {
            // Change base address.
            case 'a':
            case 'A':
               pMPA_cfg->base_address = hextoui(++charptr);
               break;

            // Change IRQ.
            case 'i':
            case 'I':
               pMPA_cfg->tx_interrupt = atouc(++charptr);
               pMPA_cfg->rx_interrupt = pMPA_cfg->tx_interrupt;
               break;

            // Change baud rate (ignored if receiving the Rx clock).
            case 'b':
            case 'B':
               pMPA_cfg->rx_baud_rate = atoul(++charptr);
               pMPA_cfg->tx_baud_rate = pMPA_cfg->rx_baud_rate;
               break;

            // Change Rx DMA channel.
            case 'r':
            case 'R':
               pMPA_cfg->rx_dma_channel = atouc(++charptr);
               break;

            // Run the queue in oneshot mode.
            case 'o':
            case 'O':
               QueueMode = QUEUE_ONESHOT;
               break;

            // Let queue ring mode continue running despite overruns.
            case 'w':
            case 'W':
               QueueMode |= QUEUE_ALLOW_OVERRUN;
               break;

            // Change number of buffers to use.
            case 'n':
            case 'N':
               num_buffers = atoui(++charptr);
               if(num_buffers > MAX_BUFFERS)
               {
                  printf("ERROR--The maximum number of buffers is %d.\n", MAX_BUFFERS);
                  exit(-1);
               }           
               break;

            // Change size of buffers to use.
            case 's':
            case 'S':
               BufferLen = atoui(++charptr);
               break;


            default:
               printf("\nQRECBI (options)\n");
               printf("\nOptions:\n");
               printf("/A<base address in hex>\n");
               printf("/I<irq>\n");
               printf("/B<baud rate>\n");
               printf("/R<receive dma channel>\n\n");
               printf("/N<number of buffers>\n");
               printf("/S<size of buffers in bytes>\n");
               printf("/O for oneshot mode\n");
               printf("/W for ignore queue overrun\n");
               exit(-1);
               break;
            }
         }
      }

}


void validate_rx(union com_block _far *pRxBuffer, short index)
{
   // put terminator in buffer to accommodate the printf statement and print it.
   pRxBuffer->BitR.comm_buffer[pRxBuffer->BitR.buffer_pointer - 2] = 0;
   printf("%s \n",pRxBuffer->BitR.comm_buffer);

   // Report if the buffer wasn't what we expected.
   if (strncmp((char *)pRxBuffer->BitR.comm_buffer,
               (char *)pMsgString,
               strlen((char *)pMsgString)) != 0)
      {
      printf("Strings not the same.  Buffer %i.  Rx Length = %i.\n",
              index, strlen((char *)pRxBuffer->BitR.comm_buffer));
      }

   // Clear the status so the buffer can be reused.
   pRxBuffer->BitR.buffer_status  = 0;
}



void SyncDriveExit(void)
{
   // This exit routine takes care of required cleanup.
   // It's important not to exit with any locked memory lying around!

   // Print current buffer queue status.
   if (pRxQueue)
      printf("Rx queue status %8lx \n",pRxQueue->QueueStatus);

   // Stop the receiver.  This will stop the Rx queue as well.
   rc = sync_command(pMPA_cfg, 4);

   // Free the Rx queue.
   rc = sync_free_queue(pMPA_cfg, pRxQueue);

   // Release the channel.
   rc = sync_release(pMPA_cfg);

   // Free up the Rx buffers.
   if (pMPA_cfg->rx_dma_channel != 0)
      rc = sync_free_dma_buffers();
   else
      {
      for (index = 0; index < num_buffers; index++)
         {
         if (pRxBuffer[index])
            free(pRxBuffer[index]);
         }
      }

   // Release the channel config array memory.
   free(pMPA_cfg);

}



void AbnormalExit(int signal)
{
   // This function ensures that the exit handler gets
   // called if Ctrl-C, Ctrl-Break, etc. occurs.
   exit(0);
}






void main(int argc, char *argv[])
{
   USHORT  DriverRelease;
   USHORT  DLLRelease;

   
   // Setup handlers for Ctrl-C, Ctrl-Break, etc.
   signal(SIGINT, AbnormalExit);
   signal(SIGTERM, AbnormalExit);

#if defined(__OS2__) || defined(__WIN95__)
   signal(SIGBREAK, AbnormalExit);
#endif
   atexit(SyncDriveExit);


   // Allocate memory for the channel configuration array.
   if ((pMPA_cfg = malloc(sizeof(struct channel_cfg))) == NULL)
      {
      printf("Cannot allocate config array!\n");
      exit(-1);
      }


   printf("\n*** Quatech MPA-series adapter receive queue example program ***\n\n");
   printf("Bit synchronous link between two boards, connected by a cable.\n");
   printf("See the source code for cabling info.\n\n");
   printf("Press Ctrl-Break to stop the program.\n\n");


   // Initialize the channel configuration array.

   init_config_array();

   // Check the SyncDrive version.
   rc = sync_version(pMPA_cfg, &DriverRelease, &DLLRelease);
   if (rc != SYNC_SUCCESS)
   {
      printf("Syncdrive release level error!\n\n");
      printf("Driver release level is %04x, DLL release level is %04x\n",
             DriverRelease, DLLRelease);
      exit(-1);
   }
   printf("Syncdrive release level is %04x\n\n", DriverRelease); 

   
   // Now process any options that override the defaults.

   if (argc > 1)
       process_options(argc, argv);


   // Allocate the Rx buffers.  Use special allocate routine if using DMA.
   for (index = 0; index < num_buffers; index++)
      {
      if (pMPA_cfg->rx_dma_channel != 0)
         {
         if (sync_alloc_dma_buffer(pMPA_cfg, BufferLen, &pRxBuffer[index]) != 0)
            {
            printf("DMA Buffer %i Allocation Failed\n", index);
            sync_free_dma_buffers();
            exit(-1);
            }
         }
      else
         {
         if ((pRxBuffer[index] = malloc(BufferLen)) == NULL)
            {
            printf("Buffer %i Allocation Failed\n", index);
            exit(-1);
            }
         }
      }

   // Display DMA buffer allocation if DMA buffers have been allocated.
   if (pMPA_cfg->rx_dma_channel != 0)
   {
      printf("Allocated %i DMA buffers of %i bytes (%li bytes total)\n",
          num_buffers, BufferLen, num_buffers * BufferLen);
   } 


   // Configure and open the channel.

#ifdef MPA102
   printf("Channel configuration status = %04x \n", config_MPA102(pMPA_cfg));
#endif
#ifdef MPA200
   printf("Channel configuration status = %04x \n", config_MPA200(pMPA_cfg));
#endif


   // Print configuration details to the screen.

   printf("\nBase address = %03x\n", pMPA_cfg->base_address);
   if (pMPA_cfg->rx_dma_channel != 0)
      printf("Receive DMA channel = %i\n", pMPA_cfg->rx_dma_channel);
   else
      printf("DMA not in use.\n");
   printf("Receive interrupt = %i\n", pMPA_cfg->rx_interrupt);
   printf("Clockrate = %.li\n", pMPA_cfg->clock_rate);
   printf("Baudrate = %.li\n\n", pMPA_cfg->tx_baud_rate);


   // Initialize the receive buffer com_blocks.  Set buffer_length to the
   // maximum buffer size, allowing space for Syncdrive's overhead (see the
   // manual's description of com_block structures).  Clear the buffer_status
   // on all buffers.

   for (index = 0; index < num_buffers; index++)
      {
      pRxBuffer[index]->BitR.buffer_length  = BufferLen - COMBLOCK_OVERHEAD;
      pRxBuffer[index]->BitR.buffer_status  = 0;
      }


   // Create a receive queue from the com_block array.
   // Run in continuous ring mode.

   rc = sync_register_queue(pMPA_cfg, num_buffers, &pRxBuffer[0], 0,
                            QueueMode | QUEUE_RECEIVE, &pRxQueue);
   if (rc)
      {
      printf("register queue failed, status = %04x\n", rc);
      exit(-1);
      }

   printf("Rx queue registered, %i buffers\n", num_buffers);


   // Initialize loop control.

   done = FALSE;
   this_rx = pRxQueue->CurrentBuffer;


   // Start the receive queue.

#ifdef __OS2__
   rc = sync_receive_queue(pMPA_cfg, pRxQueue, 0);
#elif defined _WINDOWS
   rc = sync_receive_queue(pMPA_cfg, pRxQueue, 0);
#else
   rc = sync_receive_queue(pMPA_cfg, pRxQueue);
#endif

   if (rc)
      {
      printf("sync_receive_queue failed, status = %04x\n", rc);
      exit(-1);
      }

#ifdef __WIN95__
   printf("\nFor best performance, run this application from a full screen Command Prompt.\n\n");
#endif
   printf("Rx queue now running...\n");



   // Loop until user aborts or a queue error (queue overrun) occurs.
   // Whenever a frame is received, check it, print it, and recycle the buffer.

   while ((!done) && (pRxQueue->QueueStatus & QUEUE_RUNNING))
      {
      while (pRxQueue->CurrentBuffer != this_rx)
         {
         validate_rx(pRxBuffer[this_rx], this_rx);
         this_rx = ++this_rx % num_buffers;
         }

#ifdef _WINDOWS
         // Windows "QuickWin" app won't respond to Ctrl-Break unless some
         //  console I/O is done.  This can impact performance under DOS,
         //  so we'll make it conditional to Windows.
         if (kbhit())
            getch();
#endif

      }

}

