/*--------------------------------------------------------------------------*/
/* RSPGEN.CMD                                              25th July 1994   */
/*                                            Last update: 26th August 1994 */
/*                                                                          */
/* Winterthur Insurance, Switzerland, R. Schnyder                           */
/*--------------------------------------------------------------------------*/
/* This procedure creates a new response file using an already existing     */
/* one.                                                                     */
/* The entries in the source file which should be changed have to be marked */
/* with keywords. A keyword can be any name with a leading and trailing %.  */
/* If a keyword has been found in the source file it will be replaced with  */
/* specific value from the ini file.                                        */
/*                                                                          */
/* The ini file contains variables and response file directories.           */
/* The variables are placed in the VARS section and the response file       */
/* to generate in the RSPFILES section.                                     */
/*                                                                          */
/* The target response files will be determined using the keyword TARGETRSP */
/* from the ini file.                                                       */
/*                                                                          */
/* If the ini includes lines which begins with an * this lines will be      */
/* ignored (Remarks!)                                                       */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* RSPGEN Syntax: RSPGEN inifile                                            */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* Inifile example:                                                         */
/*                                                                          */
/* [VARS]                                                                   */
/* * This is a silly remark                                                 */
/* TargetRSP = XYZ                                                          */
/* TRAddress = 400000000000                                                 */
/* [RSPFILES]                                                               */
/* G:\CID\IMG\LAPS\RSPGEN.RSP                                               */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* Generation example:                                                      */
/* RSPGEN.RSP                Generation        XYZ.RSP                      */
/*                                                                          */
/* UPDATEMODE = 1           -> RSPGEN ->       UPDATEMODE = 1               */
/* ADDRESS = %TRAddress%                       ADDRESS = 400000000000       */
/*                                                                          */
/*--------------------------------------------------------------------------*/

PARSE ARG IniFile .

/*--------------------------------------------------------------------------*/
/* Variables                                                                */
/*--------------------------------------------------------------------------*/

Version   = '1.2a'
KeyWords  = 0                    /* Counter for keywords       */
RSPFiles  = 0                    /* Counter for response files */
Section   = ''                   /* Current section            */
TargetRSP = ''                   /* Target response file name  */

/*--------------------------------------------------------------------------*/
/* Copyright                                                                */
/*--------------------------------------------------------------------------*/

say 'Response File Generator, Version' Version
say '(C) 1994 Winterthur Insurance, R. Schnyder'
say ''

/*--------------------------------------------------------------------------*/
/* Check parameters                                                         */
/*--------------------------------------------------------------------------*/
if IniFile = '' then do
  say 'Syntax: RSPGEN IniFile'
  exit 1
end

/*--------------------------------------------------------------------------*/
/* Does the file exists?                                                    */
/*--------------------------------------------------------------------------*/
if stream(IniFile,'C','QUERY EXISTS') == '' then do
  say 'Error: File' IniFile 'is missing!'
  exit 1
end

/*--------------------------------------------------------------------------*/
/* Read the inifile                                                         */
/*--------------------------------------------------------------------------*/
say 'Reading' IniFile '...'

do while lines(IniFile)
  IniLine = linein(IniFile)

  /**********************/
  /* Determine section  */
  /**********************/
  select 
    /* Remarks or empty line */
    when left(IniLine,1) == '*' | strip(IniLine,'Trailing',' ') == '' then
      nop
    /* VARS */
    when left(translate(IniLine),6) = '[VARS]' then 
      Section = 'V'
    /* RSPFILES */
    when left(translate(IniLine),10) = '[RSPFILES]' then
      Section = 'R'
    otherwise
      select
        /****************************/
        /* In section of variables? */
        /****************************/
        when Section == 'V' then do
          /* Add the keyword to the array */
          /* Element 0 = name             */
          /* Element 1 = value            */
          KeyWords = KeyWords +1
          DelimPos = pos('=',IniLine)
          if DelimPos == 0 then do
            say 'Error: Unrecognized line "'IniLine'"'
            exit 1
          end
          ArrKeyWord.KeyWords.0 = strip(substr(translate(IniLine),1,DelimPos -1),'Both',' ')
          ArrKeyWord.KeyWords.1 = strip(substr(IniLine,DelimPos +1),'Both',' ')
        end
        /****************************/
        /* In section of rsp files? */
        /****************************/
        when Section == 'R' then do
          /* Add the response file to the array */
          RSPFiles = RSPFiles +1
          ArrRSPFile.RSPFiles = translate(strip(IniLine,'t',' '))
          /* Append .RSP if necessary */
          if pos('.',right(ArrRSPFile.RSPFiles,4)) == 0 then
            ArrRSPFile.RSPFiles = ArrRSPFile.RSPFiles || '.RSP'
        end
        otherwise
      end
  end
end
/* Close ini file */
call stream IniFile,'C','CLOSE'

/*--------------------------------------------------------------------------*/
/* Check the ini entries                                                    */
/*--------------------------------------------------------------------------*/
if KeyWords == 0 then do
  say 'Error: Missing key words!'
  exit 1
end
if RSPFiles == 0 then do
  say 'Error: Missing response file entries!'
  exit 1
end

/*--------------------------------------------------------------------------*/
/* Determine target response file name                                      */
/*--------------------------------------------------------------------------*/
do i = 1 to KeyWords
  /* Target response file? */
  if ArrKeyWord.i.0 == 'TARGETRSP' then do
    TargetRSP = ArrKeyWord.i.1
    if right(TargetRSP,4) <> '.RSP' then
      TargetRSP = TargetRSP || '.RSP'
  end
end

/*--------------------------------------------------------------------------*/
/* Does the target response file name is supported?                         */
/*--------------------------------------------------------------------------*/
if TargetRSP == '' then do
  say 'Error: Entry "TargetRSP" is missing!'
  exit 1
end

/*--------------------------------------------------------------------------*/
/* Create response files                                                    */
/*--------------------------------------------------------------------------*/

do i = 1 to RSPFiles
  CurrSource = ArrRSPFile.i
  CurrTarget = substr(ArrRSPFile.i,1,lastpos('\',ArrRSPFile.i)) || TargetRSP
  '@if exist' CurrTarget 'del' CurrTarget

  say 'Reading response file' CurrSource'...'
  if stream(CurrSource,'C','QUERY EXISTS') == '' then do
    say 'Error: File' CurrSource 'is missing!'
    exit 1
  end

  say 'Generating response file' CurrTarget'...'
  /********************************************/
  /* Parse all the source response file lines */
  /********************************************/
  do while lines(CurrSource)
    RSPLine    = linein(CurrSource)	/* Input           */
    RSPUpLine  = translate(RSPLine)	/* Input uppercase */
    NewRSPLine = RSPLine		/* Output          */

    /***********************************/
    /* Check all the keywords per Line */
    /* -> %KEYWORD% in response file   */
    /***********************************/
    do KeySearch = 1 to KeyWords
      KeyPos = pos('%' || ArrKeyWord.KeySearch.0 || '%',RSPUpLine)
      if KeyPos > 0 then do
        NewRSPLine = substr(RSPLine,1,KeyPos -1)
        NewRSPLine = NewRSPLine || ArrKeyWord.KeySearch.1
        NewRSPLine = NewRSPLine || substr(RSPLine,KeyPos + length(ArrKeyWord.KeySearch.0) +2)
        /* Only one keyword per line! */
        leave
      end
    end
    /***********************/
    /* Write to the output */
    /***********************/
    if lineout(CurrTarget,NewRSPLine) <> 0 then do
      say 'Error: Could not successfully write to' TargetRSP'!'
      exit 1
    end
  end
  /* Close files */
  call stream CurrTarget,'C','CLOSE'
  call stream CurrSource,'C','CLOSE'
end

say ''
say 'All response files have been created.'

exit 0
