/*
** THIS SOFTWARE IS SUBJECT TO COPYRIGHT PROTECTION AND IS OFFERED ONLY
** PURSUANT TO THE 3DFX GLIDE GENERAL PUBLIC LICENSE. THERE IS NO RIGHT
** TO USE THE GLIDE TRADEMARK WITHOUT PRIOR WRITTEN PERMISSION OF 3DFX
** INTERACTIVE, INC. A COPY OF THIS LICENSE MAY BE OBTAINED FROM THE 
** DISTRIBUTOR OR BY CONTACTING 3DFX INTERACTIVE INC(info@3dfx.com). 
** THIS PROGRAM IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER 
** EXPRESSED OR IMPLIED. SEE THE 3DFX GLIDE GENERAL PUBLIC LICENSE FOR A
** FULL TEXT OF THE NON-WARRANTY PROVISIONS.  
** 
** USE, DUPLICATION OR DISCLOSURE BY THE GOVERNMENT IS SUBJECT TO
** RESTRICTIONS AS SET FORTH IN SUBDIVISION (C)(1)(II) OF THE RIGHTS IN
** TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 252.227-7013,
** AND/OR IN SIMILAR OR SUCCESSOR CLAUSES IN THE FAR, DOD OR NASA FAR
** SUPPLEMENT. UNPUBLISHED RIGHTS RESERVED UNDER THE COPYRIGHT LAWS OF
** THE UNITED STATES.  
** 
** COPYRIGHT 3DFX INTERACTIVE, INC. 1999, ALL RIGHTS RESERVED
**
** $Revision: 1.1 $
** $Date: 2000/02/25 00:38:01 $
*/

#ifndef __TEXUS_H__
#define __TEXUS_H__

#ifdef __cplusplus 
extern "C" { 
#endif

#include <stdlib.h>
#include <stdio.h>

#ifndef __3DFX_H__
/*
** basic data types
*/
typedef unsigned char   FxU8;
typedef unsigned short  FxU16;
typedef short           FxI16;
typedef unsigned long   FxU32;
typedef long            FxI32;
typedef int             FxBool;

/*
** fundamental types
*/

#define FXTRUE    1
#define FXFALSE   0
#endif  /* __3DFX_H__ */


#ifndef __GLIDE_H__
/* These are defined so we can use them without glide */
typedef FxU32 GrTextureFormat_t;
#define GR_TEXFMT_8BIT                  0x0
#define GR_TEXFMT_RGB_332                               GR_TEXFMT_8BIT
#define GR_TEXFMT_YIQ_422               0x1
#define GR_TEXFMT_ALPHA_8               0x2 /* (0..0xFF) alpha     */
#define GR_TEXFMT_INTENSITY_8           0x3 /* (0..0xFF) intensity */
#define GR_TEXFMT_ALPHA_INTENSITY_44    0x4
#define GR_TEXFMT_P_8                   0x5
#define GR_TEXFMT_RSVD1                 0x6
#define GR_TEXFMT_RSVD2                 0x7
#define GR_TEXFMT_16BIT                 0x8
#define GR_TEXFMT_ARGB_8332                     GR_TEXFMT_16BIT
#define GR_TEXFMT_AYIQ_8422             0x9
#define GR_TEXFMT_RGB_565               0xa
#define GR_TEXFMT_ARGB_1555             0xb
#define GR_TEXFMT_ARGB_4444             0xc
#define GR_TEXFMT_ALPHA_INTENSITY_88    0xd
#define GR_TEXFMT_AP_88                 0xe
#define GR_TEXFMT_RSVD4                 0xf

typedef FxI32 GrLOD_t;
#ifdef GLIDE3
#define GR_LOD_256              0x8
#define GR_LOD_128              0x7
#define GR_LOD_64               0x6
#define GR_LOD_32               0x5
#define GR_LOD_16               0x4
#define GR_LOD_8                0x3
#define GR_LOD_4                0x2
#define GR_LOD_2                0x1
#define GR_LOD_1                0x0
#else
#define GR_LOD_256              0x0
#define GR_LOD_128              0x1
#define GR_LOD_64               0x2
#define GR_LOD_32               0x3
#define GR_LOD_16               0x4
#define GR_LOD_8                0x5
#define GR_LOD_4                0x6
#define GR_LOD_2                0x7
#define GR_LOD_1                0x8
#endif 

typedef FxI32 GrAspectRatio_t;
#ifdef GLIDE3
#define GR_ASPECT_8x1   3       /* 8W x 1H */
#define GR_ASPECT_4x1   2       /* 4W x 1H */
#define GR_ASPECT_2x1   1       /* 2W x 1H */
#define GR_ASPECT_1x1   0       /* 1W x 1H */
#define GR_ASPECT_1x2  -1       /* 1W x 2H */
#define GR_ASPECT_1x4  -2       /* 1W x 4H */
#define GR_ASPECT_1x8  -3       /* 1W x 8H */
#else
#define GR_ASPECT_8x1 0x0       /* 8W x 1H */
#define GR_ASPECT_4x1 0x1       /* 4W x 1H */
#define GR_ASPECT_2x1 0x2       /* 2W x 1H */
#define GR_ASPECT_1x1 0x3       /* 1W x 1H */
#define GR_ASPECT_1x2 0x4       /* 1W x 2H */
#define GR_ASPECT_1x4 0x5       /* 1W x 4H */
#define GR_ASPECT_1x8 0x6       /* 1W x 8H */
#endif /* GLIDE3 */

/*
** 3DF texture file structs
*/
typedef struct
{
  FxU32               width, height;
  int                 small_lod, large_lod;
  GrAspectRatio_t     aspect_ratio;
  GrTextureFormat_t   format;
} Gu3dfHeader;

typedef struct
{
  FxU8  yRGB[16];
  FxI16 iRGB[4][3];
  FxI16 qRGB[4][3];
  FxU32 packed_data[12];
} GuNccTable;

typedef struct {
    FxU32 data[256];
} GuTexPalette;

typedef union {
    GuNccTable   nccTable;
    GuTexPalette palette;
} GuTexTable;

typedef struct
{
  Gu3dfHeader  header;
  GuTexTable   table;
  void        *data;
  FxU32        mem_required;    /* memory required for mip map in bytes. */
} Gu3dfInfo;

#endif          // __GLIDE_H__

#define GR_TEXFMT_32BIT                                 0x10
#define GR_TEXFMT_ARGB_8888                             GR_TEXFMT_32BIT
#define GR_TEXFMT_RGB_888                               0x11
#define GR_TEXFMT_ANY                                   0x7fffffff

// Save typing fingers
#define GR_TEXFMT_A_8                                   GR_TEXFMT_ALPHA_8
#define GR_TEXFMT_I_8                                   GR_TEXFMT_INTENSITY_8
#define GR_TEXFMT_AI_44                                 GR_TEXFMT_ALPHA_INTENSITY_44
#define GR_TEXFMT_AI_88                                 GR_TEXFMT_ALPHA_INTENSITY_88

#define TX_DITHER_NONE                                  0x00000000
#define TX_DITHER_4x4                                   0x00000001
#define TX_DITHER_ERR                                   0x00000002
#define TX_DITHER_MASK                                  0x0000000f

#define TX_COMPRESSION_STATISTICAL                      0x00000000
#define TX_COMPRESSION_HEURISTIC                        0x00000010
#define TX_COMPRESSION_MASK                             0x000000f0

#define TX_CLAMP_DISABLE                                0x00000000
#define TX_CLAMP_ENABLE                                 0x00000100
#define TX_CLAMP_MASK                                   0x00000f00

#define TX_AUTORESIZE_DISABLE                           0x00000000
#define TX_AUTORESIZE_GROW                              0x00001000
#define TX_AUTORESIZE_SHRINK                            0x00002000
#define TX_AUTORESIZE_MASK                              0x0000f000

#define TX_TARGET_PALNCC_BESTFIT                        0x00000000
#define TX_TARGET_PALNCC_SOURCE                         0x00010000
#define TX_TARGET_PALNCC_MASK                           0x000f0000

/*
 * (GaryMcT)
 * TX_FIXED_PAL_QUANT_DIST -    use color distance to map a true color
 *                              image back into a user specified  palette.
 *                              Use this when you have a lot of different
 *                              palettes.       
 * TX_FIXED_PAL_QUANT_TABLE-    use an inverse palette to map colors 
 *                              into a user specfied palette.
 *                              Use this method when you have a few
 *                              palettes.  Will run faster if you
 *                              process textures with the same 
 *                              palette sequentially since the palette
 *                              is compiled internally to a lookup table.
 */
#define TX_FIXED_PAL_QUANT_DIST                         0x00000000
#define TX_FIXED_PAL_QUANT_TABLE                        0x00100000
#define TX_FIXED_PAL_QUANT_MASK                         0x00f00000

#define TX_WRITE_3DF                                    0x00000000
#define TX_WRITE_TGA                                    0x00000001
#define TX_WRITE_MASK                                   0x0000000f

/*
 * Publicly accessible functions.
 */

/*
 * txInit3dfInfo:
 * Set up the target parameters for a texture conversion.
 *      info                    - Glide texture structure to set up.
 *      destFormat              - format that we will be converting to.
 *      destWidth, destHeight   - geometry of the target texture image.
 *                                Are modified if TX_AUTORESIZE is enabled.
 *      mipLevels               - number of mipmap levels in the target texture.
 *                                Either specifies the number of mipmap levels
 *                                to create, or -1 for all possible mipmap levels.
 *      flags -
 *         auto-resample mode   - TX_AUTORESIZE_*
 */
size_t txInit3dfInfo( Gu3dfInfo *info, GrTextureFormat_t destFormat,
                      int *destWidth, int *destHeight,
                      int mipLevels, FxU32 flags );

size_t txInit3dfInfoFromFile( FILE *file, 
                              Gu3dfInfo *info, GrTextureFormat_t destFormat,
                              int *destWidth, int *destHeight,
                              int mipLevels, FxU32 flags );

/*
 * txConvert: convert from an in memory texture to a Glide Gu3dfInfo.
 *      info - target texture info. . is set up with txInit3dfInfo.
 *      srcFormat - format of in-memory source texture.
 *      srcWidth, srcHeight - geometry of in memory source texture.
 *                            This is not limited in size or 
 *                            aspect ratio.
 *      flags - 
 *         dither mode                  - TX_DITHER_*
 *         compression                  - TX_COMPRESS_*
 *         clamp mode                   - TX_CLAMP_*
 *         palette conversion mode      - TX_FIXED_PAL_QUANT_*
 */
FxBool txConvert( Gu3dfInfo *info, GrTextureFormat_t srcFormat,
                  int srcWidth, int srcHeight,
                  const void *srcImage, FxU32 flags,
                  const void *palNcc );

FxBool txConvertFromFile( FILE *file, Gu3dfInfo *info, 
                          FxU32 flags, const void *palNcc );

FxBool txWrite( Gu3dfInfo *info, FILE *fp, FxU32 flags );

/*
 * Error handling.
 */
typedef void (*TxErrorCallbackFnc_t)( const char *string, FxBool fatal );

void txErrorSetCallback( TxErrorCallbackFnc_t fnc );

/*
 * TEXUS MEMORY ALLOCATION ROUTINES.
 * These are not to be called directly. . .they are only here
 * so that you can redefine them to your own memory management
 * routines.
 */
void    *txMalloc( size_t size );
void    txFree( void *ptr );
void    *txRealloc( void *ptr, size_t size );

#ifdef __cplusplus 
}
#endif

#endif  /* __TEXUS_H__ */
