/***************************************************************************/
/* NETDIST Installation Command File                                         */
/***************************************************************************/
/* This command file accepts a set of input parameters that are fairly     */
/* common among CID-enabled install programs.  All of the IBM OS/2 cid-    */
/* enabled programs use them, for example.                                 */
/*                                                                         */
/*    /R:<path>respfile    gives the name of the response file.  This is a */
/*                         required parameter.                             */
/*                                                                         */
/*    /S:<path>            names the directory where the source files (the */
/*                         files from which the install is done) reside.   */
/*                         This is typically a network drive.  If not      */
/*                         specified, the path defaults to the directory   */
/*                         from which this program was loaded.             */
/*                                                                         */
/*    /G:<path>            names another directory where other needed      */
/*                         might be located (imbedded response files,      */
/*                         for example).                                   */
/*                                                                         */
/*    /T:<drive>           names the target directory where the files are  */
/*                         to be loaded. If CS2 is installed, the default  */
/*                         will be the same drive, if not, then boot drive.*/
/*                         NETDIST will be appended to default drive to for*/
/*                         the default target path.                        */
/*                                                                         */
/*    /L1:<path>file       names a file where history and log information  */
/*                         will be placed. Default is no logging           */
/***************************************************************************/

arg inparms
                                     /* If NVDM/2 invoked us, and this is not */
                                     /* the first invocation, we must be done */

                /******************************************/
                /* The environment variable               */
                /*    REMOTE_INSTALL_NEXT_STATE           */
                /* is specified by LAN CID Utility.  See  */
                /* the LCU guide for information on this  */
                /* variable.                              */
                /******************************************/
state = VALUE('REMOTE_INSTALL_NEXT_STATE',,'OS2ENVIRONMENT')
IF state <> "" THEN
  IF state <> 0 THEN
    DO
      CALL ENDLOCAL
      EXIT 0
    END
/****************************  Start  *************************************/
/* This call sets: CS2Drive TCPDrive CMDrive ComDrive CFGDrive            */
Call FindDriveData
if inparms <> ""
THEN DO
   WkDir = CFGDrive"\OS2\INSTALL\ESAWKDIR"   /* Create a working directory.     */
   ADDRESS CMD "@MkDir" WkDir ">NUL 2>&1"
                                         /* create log and history file names */
   hist = WkDir"\cifinst.hst"
   ADDRESS CMD "@ERASE" hist ">NUL 2>&1"
   histout = ">>"hist" 2>>&1"
   /* Sets variables logfile srcpath respfile TargetDrive */
   IF \ParseParms(inparms)
   THEN DO
     /* Unable to parse line parms completely */
     irc = C2D('1604'x, 2);         /* Set 'error' return code */
   ELSE DO
      /* Sets variables NodeName , MasterName , Protocols , UpdateConfig */
      IF \ParseResponseFile(respfile)
      THEN DO
         /* Unable to parse response file input */
         irc = C2D('1604'x, 2);
      END /* if */
      ELSE DO
         Path = TargetDrive"\NETDIST"
         if  \VerifySpace( TargetDrive , "4000000" )
         THEN DO
            ADDRESS CMD "@ECHO" "Not adequate space on target drive "TargetDrive  histout
            irc = C2D('1208'x, 2)
         END
         ELSE DO
            "PKUNZIP2 -o -x "srcpath"\client.zip" TargetPath Histout
            ADDRESS CMD "@COPY "Respfile TargetPath"\CIFDM.PRF"
            /* Now append Histout to logfile and complete installation */
            irc = CompleteInstall( hist )
         END /* end of else adequate space available */
      END /* end of else */
   END /* else */
   IF irc = 0 THEN
      ADDRESS CMD "@ECHO NETDIST CID Install was successful." histout
   IF logfile <> "" THEN
   DO
      ADDRESS CMD "@type" hist ">"logfile
   END
   CALL ENDLOCAL
   EXIT irc
END /* end of if parsing parms */
ELSE DO
   Diskette = GetLoadDir()
   NodeName     = ""
   CIFDM = VALUE('CIFDM',,'OS2ENVIRONMENT')
   IF CIFDM <> ""
   THEN DO
      IF Exists( CIFDM"\cifdm.prf" )
      THEN DO
         Migrate = "YES"
      END
      ELSE Migrate = "NO"
   END /* IF */
   ELSE Migrate = "NO"
   /******************************************************************/
   /* Migration will keep the old profile and other values, so we run*/
   /* the next code only for a new installation.                     */
   /******************************************************************/
   IF Migrate == "NO"
   THEN DO
      FileName = Diskette"\master.dat"
      IF Exists( FileName )
      THEN DO
         MasterName = STRIP( linein( FileName ) )
         Result = stream( FileName , "c" , "Close" )
      END
      ELSE MasterName   = ""
      OrigMasterName = MasterName /* store for comparison later */
      AdjacentName = ""
      UpdateConfig = "Y"
      "echo off"
      IF ( CS2Drive \= "" ) THEN Path = CS2Drive":\NETDIST"
                            ELSE Path = "C:\NETDIST"
      Path = TRANSLATE(Path)
      Init = "NO"
      Done = "NO"
      "cls"
      SAY "                         Network Distributor"
      SAY "                        Installation Program"
      SAY ""
      SAY "This program asks questions about your configuration before installing"
      SAY "Network Distributor.  After each question, this program presents a"
      SAY "default response, enclosed in parentheses."
      SAY ""
      SAY "In response to each question, do one of the following:"
      SAY ""
      SAY "    .   If you want to accept the default response, press Enter."
      SAY ""
      SAY "    .   If you want to enter your own response, type your response"
      SAY "        and press Enter."
      SAY ""
      SAY "    .   If you do not understand the question and want to access"
      SAY "        online help for the question, type question mark (?)"
      SAY "        and press Enter."
      SAY ""
      SAY "The Communications System Installation guides describe how to setup,"
      SAY "install, and use Network Distributor.  It is recommended that you"
      SAY "read this document before using this program."
      SAY "Hit <enter> to continue"
      pull ans
      DO while ( Init == "NO" )
         Gotpath = "NO"
         DO While ( GotPath == "NO" )
            SAY
            SAY "Enter the drive and path where you want to install Network"
            SAY "Distributor. The following size space is required:"
            SAY "5 Mb on a Master Node"
            SAY "4 Mb on a Client Node"
            SAY
            SAY "Enter the drive and path where you want to install Network"
            SAY "Distributor ("Path"):"
            pull NewPath .
            GotPath = "YES"
            IF ( NewPath == "?" )
            THEN DO
               CLS
               SAY "HELP:  Enter the fully qualified drive and path name of the directory"
               SAY "where you want to install the Network Distributor files.  If the"
               SAY "directory that you specify does not exist, this program creates the"
               SAY "directory."
               NewPath = ""
               GotPath = "NO"
            END
            IF ( NewPath \= "" )
            THEN DO
               Path = NewPath;
            END
         END /* end of do while not entered path */
         "CLS"
         GotPath = "NO"
         DO while ( GotPath == "NO" )
            SAY
            SAY "Installing NETDIST into: "Path
            SAY
            SAY "Is this correct (Y) ?"
            SAY
            parse Pull PathGood .
            GotPath = "YES"
            IF ( PathGood == "?" )
            THEN DO
               CLS
               SAY
               SAY "Verify Installation Path"
               SAY "------------------------"
               SAY "Enter Y or y to indicate the correct path, "
               SAY "or any other value to indicate the path is incorrect."
               SAY
               GotPath = "NO"
               PathGood = ""
            END
         END /* end while did not get path ok yet */
         IF ( PathGood == "Y" ) | ( PathGood == "y" ) | ( PathGood = "" )
         THEN DO
            OldDir = DIRECTORY()
            NewDir = DIRECTORY( Path )
            IF ( NewDir \= Path )
            THEN DO
               IF SUBSTR(Path, LENGTH(Path) , 1 ) \= '\' THEN Path = Path'\'
               PathDrive = FILESPEC( "drive" , Path )
               PathDir   = FILESPEC( "path" , Path )
               PathDrive
               PathDir = SUBSTR( PathDir , 1 , LENGTH( PathDir ) -1 )
               'mkdir 'PathDir
               Path = SUBSTR( Path , 1 , LENGTH( Path ) -1 )
               NewDir = DIRECTORY( Path )
               IF ( NewDir \= Path )
               THEN DO
                  say "ERROR - Unable to change to directory " Path
                  Result = DIRECTORY( OldDir )
                  exit 999
               END /* end of if unable to create it either */
            END /* end of if invalid path */
            SAY "Please wait .... Network Distributor is verifying"
            SAY "disk space and installing initial files."
            PathDrive = FILESPEC( "drive" , Path )
            if ( VerifySpace( PathDrive , "4000000" ) == 0 )
            THEN DO
               SAY "Not enough space on "PathDrive "installation halted."
               EXIT 99
            END
            SAY "Now loading starter files..."
            "copy "Diskette"PKUNZIP2.EXE >NUL 2>&1"
            "PKUNZIP2 -o -x "Diskette"\client.zip >NUL 2>&1"
            Call FindDAEData
            Init = "YES"
         END /* end of if path verified */
      END /* end of while not init yet */
      Protocols = ""
      Call FindCommData
      IF ( CMDrive \= "" )
      THEN DO
         IF ( PriAdapterType \= "00" ) THEN Protocols = Protocols"N0"
         IF ( SecAdapterType \= "00" ) THEN Protocols = Protocols"N1"
      END
      IF ( TCPDrive \= "" ) THEN Protocols = Protocols"T0"

      DO while ( Done == "NO" )
         "cls"
         GotNodeOk = "NO"
         DO while ( GotNodeOk == "NO" )
            SAY "Enter the name of the local node ("NodeName") :"
            parse pull NewNodeName .
            GotNodeOk = "YES"
            IF ( NewNodeName == "?" )
            THEN DO
               CLS
               SAY
               SAY "Local Node Name"
               SAY "---------------"
               SAY "The Local Node Name is used to establish the "
               SAY "Distributed Application Environment node name, as "
               SAY "well as the node name for NETDIST. The node name is "
               SAY "case sensitive. "
               SAY
               SAY "The default value for this is obtained from the "
               SAY "DAE control blocks if present on the system. "
               SAY
               GotNodeOk = "NO"
               NewNodeName == ""
            END
            IF ( NewNodeName \= "" )
            THEN DO
               NodeName = NewNodeName
            END
         END /* end of while */

         IF ( MasterName == "" ) THEN MasterName = NodeName
         GotNodeOk = "NO"
         DO while ( GotNodeOk == "NO" )
            SAY "Enter the name of the master node ("MasterName") :"
            parse pull NewMasterName .
            GotNodeOk = "YES"
            IF NewMasterName == "?"
            THEN DO
               CLS
               SAY "HELP:  Enter the name of the node you assigned to be the master node."
               SAY "You must assign one Network Distributor node to be the master node."
               SAY "The master node controls all the Network Distributor nodes on the"
               SAY "network."
               SAY
               GotNodeOk = "NO"
               NewMasterName = ""
            END /* end of if help */
            if NewMasterName \= ""
            THEN DO
               MasterName = NewMasterName
               IF ( AdjacentName == "" ) THEN AdjacentName = MasterName
            END
         END /* while not GotNodeOk */
         IF ( AdjacentName == "" ) THEN AdjacentName = MasterName

         GotNodeOk = "NO"
         DO while ( GotNodeOk == "NO" )
            SAY "Enter the name of the adjacent node, if required ("AdjacentName") :"
            parse pull NewAdjacentName .
            GotNodeOk = "YES"
            if ( NewAdjacentName == "?" )
            THEN DO
               CLS
               SAY "HELP:  If your configuration requires an adjacent node, enter the name"
               SAY "of the node you assigned to be the adjacent node."
               SAY ""
               SAY "An adjacent node is required if:"
               SAY ""
               SAY "   .  The master node contains support for only one protocol (TCP/IP"
               SAY "      or NETBIOS)."
               SAY ""
               SAY "      and"
               SAY ""
               SAY "   .  A Network Distributor node does not contain support for the"
               SAY "      protocol used on the master node."
               SAY ""
               SAY "The adjacent node can be any Network Distributor node that"
               SAY "contains support for both the TCP/IP and NETBIOS protocols."
               NewAdjacentName = ""
               GotNodeOk = "NO"
            END /* end of if they want help */
            IF NewAdjacentName = "" then AdjacentName = MasterName
                                    else AdjacentName = NewAdjacentName
         END /* end of while not ok on data */
         ProtRight = "NO"
         CLS
         DO while ProtRight == "NO"
            SAY "The next question asks you to enter the protocols supported on"
            SAY "the local node.  To specify a single protocol, enter one of the"
            SAY "following the following abbreviations (for example, N0):"
            SAY "     N0     NETBIOS on adapter 0"
            SAY "     N1     NETBIOS on adapter 1"
            SAY "     T0     TCP/IP"
            SAY "     I0     No protocol (stand-alone node)"
            SAY "To specify multiple protocols, enter two or more abbreviations, with"
            SAY "no delimiters (for example, N0N1)."
            SAY ""
            SAY "Enter the protocols supported on the local node ("Protocols"):"
            pull NewProtocols .
            if ( NewProtocols == "?" )
            THEN DO
               CLS
               SAY "HELP:  Enter the protocols supported on the node on which you are"
               SAY "installing Network Distributor.  Each Network Distributor node must"
               SAY "contain support for either the NETBIOS or TCP/IP protocol.  If your"
               SAY "configuration requires an adjacent node, the adjacent node must"
               SAY "contain support for both the NETBIOS and TCP/IP protocol."
               SAY ""
               SAY "To specify a single protocol, enter one of the following the following"
               SAY "abbreviations (for example, N0):"
               SAY "     N0     NETBIOS on adapter 0"
               SAY "     N1     NETBIOS on adapter 1"
               SAY "     T0     TCP/IP"
               SAY "     I0     No protocol (stand-alone node)"
               SAY "To specify multiple protocols, enter two or more abbreviations, with"
               SAY "no delimiters (for example, N0N1)."
            END /* end of if help */
            ELSE DO
               IF ( NewProtocols \= "" ) THEN Protocols = NewProtocols
               CheckProt = Protocols
               IF ( CheckProt \= "" ) THEN ProtRight = "YES"
               DO while CheckProt \= ""
                  TestProt = SUBSTR( CheckProt , 1 , 2 )
                  CheckProt = SUBSTR( CheckProt , 3 )
                  if (TestProt \= "N0" )&(TestProt \= "N1" )&( TestProt \= "T0" )&(TestProt \= "I0")
                  THEN DO
                     ProtRight = "NO"
                     SAY "Protocol "TestProt "not valid, try again..."
                  END
               END /* end of do while */
            END /* end of else do */
         END /* end of do while */

         "cls"
         OkAnswer = "NO"
         DO While ( OkAnswer == "NO" )
            SAY "The next question asks if you want this program to make the required"
            SAY "updates to your system files.  If you answer N (no), you must do the"
            SAY "following tasks after completing the installation:"
            SAY ""
            SAY "     .   Add the line CALL STARTCIF to the STARTUP.CMD file."
            SAY "     .   Add "Path" to the PATH statement in the CONFIG.SYS"
            SAY "         file."
            SAY "     .   Add "Path" to the DPATH statement in the CONFIG.SYS"
            SAY "         file."
            SAY "     .   Add the line SET CIFDM="Path" to the CONFIG.SYS file."
            SAY ""
            SAY "Do you want this program to make the required updates your system"
            SAY "files ("UpdateConfig"):"
            pull NewUpdateConfig .
            OkAnswer = "YES"
            IF ( NewUpdateConfig == "?" )
            THEN DO
               SAY "HELP:  Type Y (yes) and press Enter if you want this program to"
               SAY "automatically make the updates.  Type N (no) and press Enter if you"
               SAY "want to update the files manually after completing installation. All"
               SAY "files modified by this operation are backed up in the original directory"
               SAY "with a numeric extension."
               OkAnswer = "NO"
            END
            IF (( NewUpdateConfig == "Y" ) | ( NewUpdateConfig == "N" )) THEN UpdateConfig = NewUpdateConfig
         END /* end of while */
         "cls"
         SAY "About to configure Network Distributor using the "
         SAY "following information:"
         SAY
         SAY "Master Node:"MasterName
         SAY "Adjacent Node:"AdjacentName
         SAY "Local Node:"NodeName
         SAY "Protocols on the Local Node:"Protocols
         SAY "Update system files:"UpdateConfig
         SAY
         IF ( NodeName == MasterName )
         THEN DO
            SAY "This will be the Master Node..."
            SAY
         END /* end of then do */
         SAY "Is this information correct (Y)"
         pull ans .
         if ( ans == "Y" ) | ( ans == "" ) then Done = "YES"
      END /* end of do while not done */
      SAY "Now creating the required subdirectories..."
      "mkdir AIX >NUL 2>&1"
      "mkdir OS232 >NUL 2>&1"
      IF ( OrigMasterName \= MasterName )
      THEN DO
         say "Storing Master Name on diskette..."
         if Exists( FileName )
         THEN DO
            "del "FileName
         END
         Result = lineout( FileName , MasterName )
         Result = stream( FileName , "C" , "Close" )
      END /* IF */
   END /* end of if NOT migrating node */
   ELSE DO
      "echo off"
      "CLS"
      SAY "                         Network Distributor"
      SAY "                        Installation Program"
      SAY ""
      SAY "                       ************************"
      SAY "           *********** Previous Version Located **************"
      SAY "                       ************************"
      SAY ""
      SAY "Performing Network Distributor Updates...."
      SAY ""
      SAY "Hit <enter> to continue"
      pull ans
      /* This is a migration, we must terminate CIF and unzip the client */
      /* files first. */
      /* We change current directory to the CIF directory now */
      Path = CIFDM /* derived from environment variable earlier */
      OldDir = DIRECTORY()
      NewDir = DIRECTORY( Path )
      IF ( NewDir \= Path )
      THEN DO
         IF SUBSTR(Path, LENGTH(Path) , 1 ) \= '\' THEN Path = Path'\'
         PathDrive = FILESPEC( "drive" , Path )
         PathDir   = FILESPEC( "path" , Path )
         PathDrive
         NewDir = DIRECTORY( Path )
         IF ( NewDir \= Path )
         THEN DO
            say "ERROR - Unable to change to directory " Path
            Result = DIRECTORY( OldDir )
            exit 999
         END /* end of if unable to create it either */
      END /* end of if invalid path */
      'CIFDMTRM'
      "PKUNZIP2 -o -x "Diskette"\client.zip >NUL 2>&1"
      ProfileFile = "cifdm.prf"
      TopologyFile = "" /* always use the local one */
      Result = STREAM( ProfileFile ,"c" , "open" )
      DO UNTIL \LINES( ProfileFile )
         Newline=linein(ProfileFile)
         PARSE VAR NewLine Key "=" Value
         IF Key = "NODENAME"
         THEN DO
            NodeName = Value
         END /* if */
         IF Key = "MASTNAME"
         THEN DO
            MasterName = Value
         END /* if */
      END /* do until */
      Result = STREAM( ProfileFile , "c" , "close" )
      IF ( NodeName <> MasterName )
      THEN DO
         SAY "It is not valid to reinstall Network Distributor on any node except the master"
         SAY "Please remove Network Distributor from this system if you wish to override, or"
         SAY "go to the master node " MasterName " to perform this install"
         EXIT 99
      END /* IF */
   END /* ELSE migration node */
END /* end of else doing normal installation */

IF ( NodeName == MasterName )
THEN DO
   /* This is the master Node */
   PathDrive = FILESPEC( "drive" , Path )
   if ( Migrate == "NO" )
   THEN DO
      if ( VerifySpace( PathDrive , "5000000" ) == 0 )
      THEN DO
         SAY "Not enough space on "PathDrive "installation halted."
         EXIT 99
      END
   END /* IF not migration */
   say "Loading master files now...."
   "Copy "Diskette"\netdist.doc 1>NUL 2>&1"
   "Copy "Diskette"\migrate.doc 1>NUL 2>&1"
   "PKUNZIP2 -o -x "Diskette"\base.zip 1>NUL 2>&1"
   CorrectDisk = "NO"
   DO WHILE ( CorrectDisk == "NO" )
      "cls"
      SAY
      say "Insert Network Distributor Installation diskette number 2"
      say "Press enter key when the diskette is inserted"
      pull resp
      Call VerifyDiskette Diskette "CIF02"
   END
   SAY
   SAY "Diskette 2 verified, loading files...."
   SAY
   "PKUNZIP2 -o -x "Diskette"\base2.zip 1>NUL 2>&1"
   IF TopologyFile \= ""
   THEN DO
      SAY
      SAY "Using Topology data found in "TopologyFile
      "copy "TopologyFile "topology.dat"
   END
end  /* Do */
IF ( Migrate == "NO" )
THEN DO
   /* Now we create the user profile for this node */
   ProfileFile = "cifdm.prf"
   IF Exists( ProfileFile ) THEN "DEL "ProfileFile
   Buffer = "NODENAME="NodeName
   retcode = LINEOUT( ProfileFile , Buffer );
   Buffer = "MASTNAME="MasterName
   retcode = LINEOUT( ProfileFile , Buffer );
   Buffer = "PROFNAME="MasterName
   retcode = LINEOUT( ProfileFile , Buffer );
   Buffer = "USEREDIT="CFGDrive":\OS2\E.EXE"
   retcode = LINEOUT( ProfileFile , Buffer );
   Buffer = "PROTOCOL="Protocols
   retcode = LINEOUT( ProfileFile , Buffer );
   Buffer = "ADJCNAME="AdjacentName
   retcode = LINEOUT( ProfileFile , Buffer );
   IF ( NodeName == MasterName )
   THEN DO
      Buffer = "SESSIONS=64"
   END
   ELSE DO
      Buffer = "SESSIONS=4"
   END
   retcode = LINEOUT( ProfileFile , Buffer );
   Buffer = "SESSCLNT=2"
   retcode = LINEOUT( ProfileFile , Buffer );
   Buffer = "GETFREMT=Y"
   retcode = LINEOUT( ProfileFile , Buffer );
   Buffer = "PUTFREMT=Y"
   retcode = LINEOUT( ProfileFile , Buffer );
   Buffer = "COMMREMT=Y"
   retcode = LINEOUT( ProfileFile , Buffer );
   Buffer = "BOOTCOMM=Y"
   retcode = LINEOUT( ProfileFile , Buffer );
   Buffer = "CRONREMT=Y"
   retcode = LINEOUT( ProfileFile , Buffer );
   Buffer = "PROFREMT=Y"
   retcode = LINEOUT( ProfileFile , Buffer );
   Buffer = "USNETCFG=Y"
   retcode = LINEOUT( ProfileFile , Buffer );
   Buffer = "GETFLOCL=Y"
   retcode = LINEOUT( ProfileFile , Buffer );
   Buffer = "PUTFLOCL=Y"
   retcode = LINEOUT( ProfileFile , Buffer );
   Buffer = "COMMLOCL=Y"
   retcode = LINEOUT( ProfileFile , Buffer );
   retcode = stream( ProfileFile , "c" , "close" )
   IF ( NodeName == MasterName )
   THEN DO
      "copy "ProfileFile "cifmast.prf >NUL 2>&1"
      CronFile = "cifdmcrn.dat"
      retcode = stream( CronFile , "c" , "query exists" )
      if ( retcode == "" )
      THEN DO
         /* On the master, need a cron to run the pruner process */
         Buffer = "0 13 * * * cifdmprn"  /* every night at 11 pm */
         retcode = LINEOUT( CronFile , Buffer );
         retcode = stream( CronFile , "c" , "close" )
      END /* if */
   END /* if */
END /* if new installation */
ELSE DO
   /* Now it is time to do the update and define object calls to put */
   /* the new files into the system.   */
   /* This release adds some new files to system */
   "CIFDMAPI APPEND  BASE CIFDMNEW.BAS APPEND INSTALL CIFDMNEW.INS"
   "CIFDMAPI REPLACE BASE CIFDMINI.BAS REPLACE INSTALL CIFDMINI.INS"
END /* else migration node */


/* Certain files are loaded by a different name to avoid stepping on */
/* the real file during a migration install. If the real copy does   */
/* not exist at this point, we must copy the shipped one over it.    */
if \Exists( "Topology.dat" ) THEN "copy topology.ins topology.dat"
if \Exists( "mnemonic.dat" ) THEN "copy mnemonic.ins mnemonic.dat"
if \Exists( "service.cif"  ) THEN "copy service.ins service.cif"
IF ( Migrate == "NO" ) THEN Result = CompleteInstall( "-" )
ELSE DO
   Call StartCIF
   "CLS"
   SAY "Update Successful..."
   SAY ""
   SAY "Updates have been made to Network Distributor code on the master "
   SAY "node only. See MIGRATE.DOC for details on completing this set of"
   SAY "updates. Network Distributor functionality from the Master node "
   SAY "only will be limited until the migration is completed."
   SAY ""
END /* ELSE */
temp = Directory( OldDir )
exit Result
/***************************************************************/
CompleteInstall : procedure expose Path TCPDrive UpdateConfig Protocols
/* Now lets update config.sys for CIF variable setting */
arg Logfile
Temp = Value( 'CIFDM' , Path , 'OS2ENVIRONMENT' )
IF TCPDrive == "" THEN TCPServ = "-"
                  ELSE TCPServ = TCPDrive":\\TCPIP\\ETC\\SERVICES"
IF ( POS( "T0", Protocols ) == 0 ) THEN TCPServ = "-"

IF UpdateConfig == "Y" THEN "CIFDMINT install.scr " LogFile TCPServ
SAY
SAY "Continuing with installation...standby"
IF LogFile <> "-"
THEN Path'\cifdmini >>'LogFile "2>&1"
ELSE Path'\cifdmini'
IniRc = rc
if \Exists( "cifdmnb.exe" )  THEN "copy cifdmnb.new  cifdmnb.exe"
if \Exists( "cifdmcrn.exe" ) THEN "copy cifdmcrn.new cifdmcrn.exe"
if \Exists( "cifdmtc.exe" )  THEN "copy cifdmtc.new  cifdmtc.exe"
IF IniRc \= 0
THEN DO
   CLS
   SAY "ERROR"
   SAY
   SAY "NETDIST Initialization failed with return code "IniRc
   SAY "See documentation for information on return codes."
   return IniRc
END
ELSE DO
   CLS
   SAY
   SAY "NETDIST Installation Successful..."
   SAY
   IF UpdateConfig == "N"
   THEN DO
      SAY
      SAY "IMPORTANT:Do not forget to make the required system file updates"
      SAY "see documentation for details."
   END /* end of if they did not have us update config */
   SAY "There will be one or more processes left running to allow unattended "
   SAY "and timed execution of commands. See documentation for more information."
END /* end of else it worked */
Call StartCIF
return 1
/****************************************************/
FindDriveData : Procedure expose CS2Drive TCPDrive CMDrive ComDrive CFGDrive

CS2Drive = ""
TCPDrive = ""
ComDrive = ""
CMDrive = ""
OS2CMD        = VALUE( COMSPEC , , OS2ENVIRONMENT )
PARSE VAR OS2CMD ConfigDrive ':' .
DriveList=ConfigDrive"CDEFGHIJKLMNOPQRSTUVW"
DO I = 1 to LENGTH( DriveList )
   CFGDrive = SUBSTR( DriveList , I , 1 )
   IF Exists( CFGDrive":\CONFIG.SYS" ) then
   DO
      InputFile = CFGDrive":\CONFIG.SYS"
      I = 9999
   END /* end of if found config.sys */
END /* end of if do through drives */
Result = STREAM( InputFile , "c" , "open" )
DO UNTIL \lines(InputFile)
   Newline=linein(InputFile)
   Newline=TRANSLATE( NewLine ) /* always uppercase it first */
   /**************************************************/
   /* We also want to look for cmlib on path lists   */
   /**************************************************/
   IF POS( "CMLIB" , Newline ) > 0  then
   DO
      /* Lets get the drive letter for now */
      x = POS( "CMLIB" , Newline )
      DO while (( x > 1 ) & ( SUBSTR( NewLine , x , 1 ) \= ":" ))
         x = x -1
      end
      x = x - 1 /* backup one more to find drive letter */
      IF x > 0 then
      DO
         NewCMDrive = SUBSTR( NewLine , x , 1 )
         IF CMDrive == "" then
         DO
            CMDrive = NewCMDrive
            NewCMDrive = ""
         end /* end of if it was not set */
      end /* end of if x valid */
   end /* end of if we found CMLIB on a line */
   IF POS( "TCPIP" , Newline ) > 0  then
   DO
      /* Lets get the drive letter for now */
      x = POS( "TCPIP" , Newline )
      DO while (( x > 1 ) & ( SUBSTR( NewLine , x , 1 ) \= ":" ))
         x = x -1
      end
      x = x - 1 /* backup one more to find drive letter */
      IF x > 0  then
      DO
         NewTCPDrive = SUBSTR( NewLine , x , 1 )
         IF TCPDrive == "" then
         DO
            TCPDrive = NewTCPDrive
            NewTCPDrive = ""
         end /* end of if it was not set */
      end /* end of if x valid */
   end /* end of if we found TCPIP on a line */
   /**************************************************/
   /* We also want to look for ibmcom on path lists  */
   /**************************************************/
   IF POS( "IBMCOM" , Newline ) > 0  then
   DO
      /* Lets get the drive letter for now */
      x = POS( "IBMCOM" , Newline )
      DO while (( x > 1 ) & ( SUBSTR( NewLine , x , 1 ) \= ":" ))
         x = x -1
      end
      x = x - 1 /* backup one more to find drive letter */
      IF x > 0 then
      DO
         NewComDrive = SUBSTR( NewLine , x , 1 )
         IF ComDrive == "" then
         DO
            ComDrive = NewComdrive
            NewComDrive = ""
         end /* end of if it was not set */
      end /* end of if x valid */
   end /* end of if we found IBMCOM on a line */
   /**************************************************/
   /* Now find the CS2 Install path                  */
   /**************************************************/
   IF POS( "CS2\DLL" , Newline ) > 0  then
   DO
      /* Lets get the drive letter for now */
      x = POS( "CS2\DLL" , Newline )
      DO while (( x > 1 ) & ( SUBSTR( NewLine , x , 1 ) \= ":" ))
         x = x -1
      end
      x = x - 1 /* backup one more to find drive letter */
      IF x > 0 then
      DO
         NewCS2Drive = SUBSTR( NewLine , x , 1 )
         IF CS2Drive == "" then
         DO
            CS2Drive = NewCS2Drive
            NewCS2Drive = ""
         end /* end of if it was not set */
         else
         DO
            IF CS2Drive \= NewCS2Drive then
            DO
               SAY "More than one CS2 drive found in Config.sys, unable"
               SAY "to determine Node name conclusively"
            end /* end of if they were not equal */
         end /* end of if x is valid */
      end /* end of else CS2drive not already assigned */
   end /* end of if we found CS2\DLL on a line */
end /* end of do until */
Result = STREAM( InputFile , "c" , "close" )
return;

/*******************************************************/
/* Routine to test for file existance                  */
/*******************************************************/
EXISTS :
   ARG Fn
   Result = STREAM( Fn , "C" , "Query Exists" )
   IF TRANSLATE( Result ) == "" then Return 0
   ELSE Return 1
end /* end of Exists function */
return 0

FindDaeData : Procedure expose CS2Drive NodeName TopologyFile
/*******************************************************/
/* Now use the DAE drive if found to locate CS2 Values */
/*******************************************************/
TopologyFile = ""   /* will set if found */
CALL FindDriveData  /* Locate CS2 drive */
IF CS2Drive \= "" then
DO
   OutputFile = "gidgos2.tmp"
   Datafile   = "gidgos2.cfg"
   OutBuff = "Output( "DataFile" )"
   x = Lineout( OutputFile , OutBuff )
   OutBuff = "Browse("
   x = Lineout( OutputFile , OutBuff )
   OutBuff = "NIB_Segments_Disk ("
   x = Lineout( OutputFile , OutBuff )
   OutBuff = "    File_name( "CS2Drive":\cs2\sys\cs2cbf00.cfs )"
   x = Lineout( OutputFile , OutBuff )
   OutBuff = "    Configuration() ) )"
   x = Lineout( OutputFile , OutBuff )
   if Exists( CS2Drive":\cs2\toolkit\cs2cuoff.exe" )
   THEN DO
      CS2Drive":\cs2\toolkit\cs2cuoff -i" Outputfile " >NUL 2>&1"
   END
   ELSE DO
      "cs2cuoff -i" Outputfile " >NUL 2>&1"
   END /* None found */
   InputFile = DataFile
   Result = STREAM( InputFile , "c" , "open" )
   DO UNTIL \Lines( Inputfile )
      NewLine = Linein( InputFile )
      PARSE VAR Newline Type "(" Value ")"
      Type = STRIP( Type )
      SELECT
         when Type == "Current_node_name" then
         DO
            NodeName = STRIP( Value )
         end /* end of when current node name */
      otherwise
      end  /* select */
   end /* do */
   Result = STREAM( OutputFile ,"c", "close" )
   "erase "OutputFile  " >NUL 2>&1"
   Result = STREAM( DataFile ,"c", "close" )
   "erase "DataFile " >NUL 2>&1"
   InputFile = CS2Drive":\cs2\netcfg\topology.dat"
   if Exists( InputFile )
   THEN DO
      TopologyFile = InputFile
   END
end /* end of if cs2drive exists */
return

FindCommData : procedure expose CMDrive ComDrive PriAdapterType SecAdapterType
ConfigFile = ""
OS2CMD        = VALUE( COMSPEC , , OS2ENVIRONMENT )
PARSE VAR OS2CMD ConfigDrive ':' .
DriveList=ConfigDrive"CDEFGHIJKLMNOPQRSTUVW"
PriAdapterType = "00"
SecAdapterType = "00"
/*******************************************************/
/* As a last ditch, lets try to find the Comm Mgr Disk */
/*******************************************************/
Level = ""
IF ConfigFile == "" then
DO
   /* We did not find one referenced in config.sys */
   IF CMDrive == "" then
   DO
      DO I = 1 to LENGTH( DriveList )
         NewCMDrive = SUBSTR( DriveList , I , 1 )
         IF Exists( NewCMDrive":\CMLIB\ACS.PRO" ) then
         DO
            CMDrive = NewCMDrive
         end /* end of we found the file to read */
      end /* do */
   end /* end of if no drive set either */
   IF CMDrive \= "" then
   DO
      InputFile=CMDrive":\CMLIB\SYSLEVEL.ACS"
      Level="EE"
      DO UNTIL \lines(InputFile)
         Newline=linein(InputFile)
         IF POS( "Extended" , Newline ) > 0 THEN Level="ES"
         IF POS( "anager/2" , Newline ) > 0 THEN Level="CM"
      END /* end of do until eof */
      Result = STREAM( InputFile ,"c", "close" )
      IF ( ComDrive == "" ) & ( Level \= "EE" ) then
      DO
         DO I = 1 to LENGTH( DriveList )
            NewComDrive = SUBSTR( DriveList , I , 1 )
            IF Exists( NewComDrive":\IBMCOM\PROTOCOL.INI" ) then
            DO
               ComDrive = NewComDrive
            end /* end of we found the file to read */
         end /* do */
      end /* end of if no drive set either */
   end /* end of if CMDrive found */
   /*******************************************************/
   /* Now we can attempt to find the default Comm. Manager*/
   /* filename from the ACS.PRO file                      */
   /*******************************************************/
   IF CMDrive \= "" then
   DO
      InputFile = CMDrive":\CMLIB\ACS.PRO"
      DO UNTIL \lines( Inputfile )
         Newline = Linein( InputFile )
         IF POS( "ACSCFG" , NewLine ) > 0  then
         DO
            x = POS( "ACSCFG" , NewLine ) + 6
            DO while ( \( DATATYPE( SUBSTR( NewLine , x , 1 ) , "A" ) ) )
               x = x + 1
            end /* end of do while not at end of string */
            ConfigFN=""
            /* Now extract the file name from the binary data */
            DO while ( DATATYPE( SUBSTR( NewLine , x , 1 ) , "A" ) )
               c = SUBSTR( NewLine , x , 1 )
               x = x + 1
               ConfigFN=ConfigFN""c
            end /* end of do while not at end of string */
            ConfigFile = CMDrive":\CMLIB\"ConfigFN".CFG"
            /* Check to make sure we found the right one */
            IF \Exists( ConfigFile )  then ConfigFile = ""
         end /* end of if config found in profile */
      end /* end of do while file left */
      Result = STREAM( InputFile ,"c", "close" )
   end /* end of still no cmdrive available */
end /* end of if no config file located yet */
IF ConfigFile == "" then
DO
   InputFile = "C:\IBMLVL.INI" /* On 2.0 nodes, contains default Comm Mgr file */
   ConfigFN = ""
   DO UNTIL \Lines( InputFile )
      NewLine = Linein( InputFile )
      P = Pos( "CONFIG_FILE_DEF" , NewLine )
      IF P > 0  then
      DO
         P = P + LENGTH( "CONFIG_FILE_DEF" ) + 1
         DO WHILE DATATYPE( SUBSTR( NewLine , P , 1 ) , "A" )
            c = SUBSTR( NewLine , P , 1 )
            ConfigFN = ConfigFN""c
            P = P + 1
         end /* do */
      end /* end of if found */
   end /* end of do until */
   Result = STREAM( InputFile ,"c", "close" )
   ConfigFile = CMDrive":\CMLIB\"ConfigFN".CFG"
   /* Check to make sure we found the right one */
   IF \Exists( ConfigFile )  then ConfigFile = ""
end /* end of if still no luck on config file */
/*******************************************************/
/* Now we can look to see which Comm Manager this is   */
/* and get the required values.                        */
/*******************************************************/
IF CMDrive \= "" then
DO
   ConfigFile=TRANSLATE( ConfigFile ) /* make it uppercase */
   PARSE VAR ConfigFile . ':' . 'CMLIB\' ConfigFN '.CFG'
   InputFile=CMDrive":\CMLIB\SYSLEVEL.ACS"
   IF Level = "" then
   DO
      Level="EE"
      DO UNTIL \lines(InputFile)
         Newline=linein(InputFile)
         IF POS( "Extended" , Newline ) > 0 THEN Level="ES"
         IF POS( "anager/2" , Newline ) > 0 THEN Level="CM"
      END /* end of do until eof */
      Result = STREAM( InputFile ,"c", "close" )
      IF ( ComDrive == "" ) & ( Level == "ES" ) then
      DO
         DO I = 1 to LENGTH( DriveList )
            NewComDrive = SUBSTR( DriveList , I , 1 )
            IF Exists( NewComDrive":\IBMCOM\PROTOCOL.INI" ) then
            DO
               ComDrive = NewComDrive
            end /* end of we found the file to read */
         end /* do */
         IF ComDrive == "" then
         DO
            /* default to same as comm mgr */
            ComDrive = CMDrive
         end /* end of else tell them we cannot find it */
      end /* end of if no drive set either */
   end /* end of if still not found level yet */
   /**************************************************************/
   /* If this is EE, then we need to check adapter types from fil*/
   /**************************************************************/
   IF level = "EE" then
   DO
      InputFile = ConfigFile
      /* This is an EE node, we can get information from the file about */
      /* adapters on AT's etc. */
      Result=STREAM( Inputfile , "C" , "OPEN READ" )

      Result=STREAM( InputFile , "C" , "SEEK 16989" ) /* Position of 1st adapt */
      IF Result == "16989" then
      DO
         CPName = charin( InputFile ,, 8 )
         NetName = charin( InputFile ,, 8 )
         XID = charin( InputFile ,, 5 )
      END /* end of if able to seek to spot */
      Result=STREAM( InputFile , "C" , "SEEK 54919" ) /* Position of 1st adapt */
      IF Result == "54919" then
      DO
         HexType = CharIn( InputFile )
         Type = RIGHT( C2D( HexType ), 2 , "0" )
         IF Type \= "00" then
         DO
            PriAdapterType = Type
            MsgNum = 12 + Type
            PriAdapterText = Messages.MsgNum
            PriAdapterAddr = Messages.12
            Result=STREAM( InputFile , "C" , "SEEK 54921" ) /* Position of 1st addr */
            IF Result == "54921" then
            DO
               HexLcl = CharIn( InputFile )
               IF HexLcl = '02'X then
               DO
                  HexVal = Charin( Inputfile ) /* throw one away */
                  /* they are not using universal Address */
                  PriAdapterAddr = ""
                  DO i = 1 to 12
                     HexVal = Charin( InputFile )
                     PriAdapterAddr = PriAdapterAddr""HexVal
                  END
               end /* end of if using local address */
            end /* end of if able to seek */
         end /* end of if able to seek to the right spot */
      end /* end of if an adapter found configured */

      Result = STREAM( InputFile , "C" , "SEEK 55151" ) /* Position of 2st adapt */
      IF Result == "55151" then
      DO
         HexType = CharIn( InputFile )
         Type = RIGHT( C2D( HexType ), 2 , "0" )
         IF Type \= "00" then
         DO
            SecAdapterType = Type
            MsgNum = 12 + Type
            SecAdapterText = Messages.MsgNum
            SecAdapterAddr = Messages.12
            Result = STREAM( InputFile , "C" , "SEEK 55153" ) /* Position of 1st addr */
            IF Result == "55153" then
            DO
               HexLcl = CharIn( InputFile )
               IF HexLcl = '02'X then
               DO
                  HexVal = Charin( Inputfile ) /* throw one away */
                  /* they are not using universal Address */
                  SecAdapterAddr = ""
                  DO i = 1 to 12
                     HexVal = Charin( InputFile )
                     SecAdapterAddr = SecAdapterAddr""HexVal
                  END
               end /* end of if using local address */
            end /* end of if able to seek */
         end /* end of if able to seek to right spot */
      end /* end of if the second adapter is configured in file */
      Result = STREAM( Inputfile , "C" , "close" )
   end /* end of else this is EE , so go look for adapters */
   /**************************************************************/
   /* Now look to see if both adapter addresses have been found  */
   /* If not then look for acslan.log to see if they are there   */
   /**************************************************************/
   IF Level == "EE" then
   DO
      IF ( PriAdapterAddr == Messages.12 ) | ( SecAdapterAddr == Messages.12 ) then
      DO
         InputFile = CMDrive":\CMLIB\ACSLAN.LOG"
         DO UNTIL \lines( InputFile )
            NewLine = Linein( InputFile )
            PARSE VAR NewLine . Num . . . Check Address
            IF Num == "0" then
            DO
               IF Check == "address" then
               DO
                  IF PriAdapterAddr == Messages.12 then
                     PriAdapterAddr = SUBSTR( Address , 1 , 12 )
               end /* end of if address card */
            end /* end of if correct card */
            IF Num == "1" then
            DO
               IF Check == "address" then
               DO
                  IF SecAdapterAddr == Messages.12 then
                     SecAdapterAddr = SUBSTR( Address , 1 , 12 )
               end /* end of if address card */
            end /* end of if correct card */
         end /* end of while not at end of file */
         Result = STREAM( InputFile ,"c", "close" )
      end /* end of if adapter addresses unknown */
   end /* end of if EE */
   /*******************************************************/
   /* Now we need to look in protocol.ini for ES values   */
   /*******************************************************/
   IF (Level == "ES")|(Level == "CM") then
   DO
      NifType="NONE"
      InputFile=ComDrive":\IBMCOM\protocol.ini"
      DO UNTIL \lines(InputFile)
         Newline=linein(InputFile)
         IF SUBSTR( NewLine , 1 , 1 ) = "[" then
         DO
            /* We are now entering a new section */
            NifType = STRIP( SUBSTR( NewLine , 2 ) )
            NifType = SUBSTR( NifType , 1 , LENGTH( NifType ) - 1 )
            NifType = TRANSLATE( NifType )
            SELECT
               when NifType == "LANDD_NIF" then
               DO
                  /* Lan Nif contains many values we need 802.3 DLC */
               end /* end of Landd nif */
               when Niftype = "IBMNETA_NIF" then
               DO
                  /* the NETA is loaded when putting DLC onto PCNet Card */
               end /* end of ibmneta nif */
               when NifType == "IBMTOK_NIF" then
               DO
                  /* The IBMTOK loads DLC onto token Ring Card */
               end /* end of when IBMTOK nif */
               when NifType == "NETBEUI_NIF" then
               DO
                  /* the NETBEUI_nif contains the Netbios DLC data */
               end /* end of when netbios dlc nif */
               when NifType == "ELNKMC_NIF" then
               DO
                  /* Ethernet dlc Nif */
               end /* end of ethernet dlc */
               otherwise
               DO
                  /* skip the other sections for now */
                  Niftype = ""
               end /* end of otherwise do */
            end  /* select */
         end /* then do */
         else
         DO
            /* Now processing a statement from the NIF sections */
            NewLine = TRANSLATE( NewLine )
            PARSE VAR NewLine ValType '=' PriVal ',' SecVal
            ValType = STRIP( ValType)
            SELECT
               when NifType == "LANDD_NIF" then
               DO
                  SELECT
                     when ValType == "BINDINGS" then
                     DO
                        /* Bindings match CARD type to Prim, Secondary */
                        SELECT
                           when PriVal == "IBMTOK_NIF" then
                           DO
                              /* Primary is token ring */
                              PriAdapterType = "TR"
                           end /* end of primary TR */
                           when PriVal == "IBMNETA_NIF" then
                           DO
                              /* Primary is PC Net */
                              PriAdapterType = "PC"
                           end /* end of primary  */
                           when PriVal == "ELNKMC_NIF" then
                           DO
                              /* Primary is Ethernet */
                              PriAdapterType = "ET"
                           end /* end of primary ethernet */
                           otherwise
                        end  /* select */
                        SELECT
                           when SecVal == "IBMTOK_NIF" then
                           DO
                              /* Secondary is token ring */
                              SecAdapterType = "TR"
                           end /* end of secondary TR */
                           when SecVal == "IBMNETA_NIF" then
                           DO
                              /* Secondary is PC Net */
                              SecAdapterType = "PC"
                           end /* end of secondary  */
                           when SecVal == "ELNKMC_NIF" then
                           DO
                              /* Secondary is Ethernet */
                              SecAdapterType = "ET"
                           end /* end of secondary ethernet */
                           otherwise
                        end  /* select */
                     end /* end of bindings */
                     when ValType == "NETADDRESS" then
                     DO
                        /* Locally administered Addresses go here */
                        IF STRIP( PriVal ) \= "" then
                        DO
                           PriAdapterAddr = STRIP( TRANSLATE( PriVal , "  " , '"T' ) )
                        end /* end of if primary specified */
                        IF STRIP( SecVal ) \= "" then
                        DO
                           SecAdapterAddr = STRIP( TRANSLATE( SecVal , "  " , '"T' ) )
                        end /* end of if secondary specified */
                     end /* end of NETADDRESS */
                     otherwise
                  end  /* select */
               end /* end of when landd nif */
               when NifType == "IBMNETA_NIF" then
               DO
                  SELECT
                     when ValType == "ADAPTER" then
                     DO
                        /* Primary or secondary coded in Prival as 0 or 1 */
                        IF PriVal == 0  then
                        DO
                           PriAdapterType = "PC"
                        end /* end of when 0 */
                        else
                        DO
                           SecAdapterType = "PC"
                        end /* end of when not 0 */
                     end /* end of when adapter */
                  otherwise
                  end  /* select */
               end /* end of when PC NET card */
               when NifType == "IBMTOK_NIF" then
               DO
                  SELECT
                     when ValType == "PRIMARY" then
                     DO
                        /* this is primary adapter */
                        PriAdapterType = "TR"
                     end /* end of primary */
                     when ValType == "ALTERNATE" then
                     DO
                        /* This is secondary adapter */
                        SecAdapterType = "TR"
                     end /* end of alternate */
                  otherwise
                  end  /* select */
               end /* end of token ring nif */
            otherwise
            end  /* select */
         end /* end of else not first card in section */
      end /* end of do until out of file */
      Result = STREAM( InputFile ,"c", "close" )
   end /* end of if Extended Services */
end /* end of if comm mgr installed, found drive */
return


VerifyDiskette : procedure expose CorrectDisk
ARG Drive Label
CorrectDisk = "NO"
Tempfile = Temp.out
"VOL "Drive "> "TempFile "2>&1"
DO until \Lines( TempFile )
   Buffer = linein( TempFile )
   IF ( POS( Label , Buffer ) > 0 ) THEN CorrectDisk = "YES"
END
Result = stream( TempFile , "C" , "Close" )
"del "TempFile
return

VerifySpace :
ARG Drive, Space

/* command to locate free space */
return 0
/******************************************************************************/
/*  GetLoadDir()                                                             |*/
/*                                                                            */
/*  Extract the name of the directory from which this module was              */
/*  loaded.                                                                   */
/******************************************************************************/
GetLoadDir: PROCEDURE

PARSE SOURCE "OS/2" "COMMAND" progname
loaddir = FILESPEC('drive', progname) || FILESPEC('path', progname)
loaddir = strip(loaddir)
loaddir = STRIP(loaddir, 'trailing', '\')

return loaddir

/****************************************************************/
/* ParseParms() - read the command line and set things up       */
/****************************************************************/
ParseParms: PROCEDURE EXPOSE TargetDrive srcpath histout respfile,
                             logfile genpath loadpath CS2Drive CFGDrive
PARSE ARG inparms

genpath = ""
respfile = ""
logfile = ""
IF CS2Drive <> "" THEN TargetDrive = CS2Drive
ELSE TargetDrive = CFGDrive

DO i = 1 to WORDS(inparms)
  parm = WORD(inparms, i)
  IF (POS('=', parm) > 0) THEN
    PARSE VAR parm clflag '=' clval;
  ELSE
    PARSE VAR parm clflag ':' clval;
  clval = STRIP(clval, 'both');
  clflag = TRANSLATE(clflag);     /* force flag to upper case            */
  SELECT
     WHEN (clflag = "/S") THEN    srcpath = clval;
     WHEN (clflag = "/R") THEN    respfile  = clval;
     WHEN (clflag = "/G") THEN    genpath   = clval;
     WHEN (clflag = "/L1") THEN   logfile = clval;
     WHEN (clflag = "/T")  THEN   TargetDrive = clval;
     OTHERWISE  DO
                    ADDRESS CMD "@ECHO" "Invalid command line parameter specified "parm histout
                    return 0
                END
  END
END
IF (respfile = "")
THEN DO
    ADDRESS CMD "@ECHO" "Response File required parameter not specified " histout
    RETURN 0
END
IF STREAM(respfile, 'c', 'query exists') = ""
THEN DO
    ADDRESS CMD "@ECHO" "Unable to read Response File "respfile histout
    RETURN 0
END
loadpath = GetLoadDir()
IF srcpath = "" THEN
  srcpath = loadpath
IF TargetDrive <> "" THEN
  TargetDrive = LEFT(TargetDrive, 1) || ":"

path = VALUE('PATH',,'OS2ENVIRONMENT')
dpath = VALUE('DPATH',,'OS2ENVIRONMENT')
IF genpath <> "" THEN
  DO
    p = genpath
    IF (LENGTH(p) = 2) & (SUBSTR(p, 2, 1) = ":") THEN
      p = p"\"
    path = p || ";" || path
    dpath = p || ";" || dpath
  END
IF srcpath <> "" THEN
  DO
    p = srcpath
    IF (LENGTH(p) = 2) & (SUBSTR(p, 2, 1) = ":") THEN
      p = p"\"
    path = p || ";" || path
    dpath = p || ";" || dpath
  END
IF loadpath <> srcpath THEN
  DO
    p = loadpath
    IF (LENGTH(p) = 2) & (SUBSTR(p, 2, 1) = ":") THEN
      p = p"\"
    path = p || ";" || path
    dpath = p || ";" || dpath
  END
call VALUE 'PATH',path,'OS2ENVIRONMENT'
call VALUE 'DPATH',dpath,'OS2ENVIRONMENT'
ADDRESS CMD "@ECHO" inparms logfile
RETURN 1
/***************************************************************/
/* ParseResponseFile(file) - configures variables from response*/
/***************************************************************/
ParseResponseFile : procedure expose NodeName MasterName Protocols UpdateConfig,
                    histout
arg respf
InputFile = respf
NodeName = ""
MasterName = ""
Protocols  = ""
UpdateConfig = "Y" /* always on CID */
Result = STREAM( InputFile ,"c" , "open" )
DO UNTIL \LINES( InputFile )
   Newline=linein(InputFile)
   PARSE VAR NewLine Key "=" Value
   SELECT
     WHEN (Key == "NODENAME") THEN NodeName = Value;
     WHEN (Key == "MASTNAME") THEN MasterName = Value;
     WHEN (Key == "PROTOCOL") THEN Protocols = Value;
     OTHERWISE;
   END
END /* do until */
Result = STREAM( Inputfile , "c" , "close" )
IF (NodeName == "")|(MasterName=="")|(Protocols=="")
THEN DO
   ADDRESS CMD "@ECHO" "Response File is missing values" histout
   return 0
END
return 1
