/*
 *  linux/fs/hpfs/alloc.c
 *
 *  Mikulas Patocka (mikulas@artax.karlin.mff.cuni.cz), 1998
 *
 *  adding & removing files & directories
 */

#include "hpfs_fn.h"

int hpfs_mkdir(struct inode *dir, const char *name, int len, int mode)
{
	struct quad_buffer_head qbh0;
	struct buffer_head *bh;
	struct hpfs_dirent *de;
	struct fnode *fnode;
	struct dnode *dnode;
	struct inode *result;
	fnode_secno fno;
	dnode_secno dno;
	int r;
	struct hpfs_dirent dee;
	if (!dir) return -ENOENT;
	if (!S_ISDIR(dir->i_mode)) {
		iput(dir);
		return -ENOTDIR;
	}
	adjust_length((char *)name, &len);
	if (chk_hpfs_name((char *)name, len)) {
		iput(dir);
		return -ENAMETOOLONG;
	}	
	if (!(fnode = alloc_fnode(dir->i_sb, dir->i_hpfs_dno, &fno, &bh))) goto bail;
	if (!(dnode = alloc_dnode(dir->i_sb, fno, &dno, &qbh0, 1))) goto bail1;
	memset(&dee, 0, sizeof dee);
	dee.directory = 1;
	if (!(mode & 0222)) dee.read_only = 1;
	dee.archive = 1;
	dee.fnode = fno;
	dee.creation_date = dee.write_date = dee.read_date = gmt_to_local(CURRENT_TIME);
	hpfs_lock_inode(dir);
	r = add_dirent(dir, (char *)name, len, &dee, 0);
	if (r == 1) goto bail2;
	if (r == -1) {
		brelse(bh);
		brelse4(&qbh0);
		free_sectors(dir->i_sb, fno, 1);
		free_dnode(dir->i_sb, dno);
		hpfs_unlock_inode(dir);
		iput(dir);
		return -EEXIST;
	}
	fnode->len = len;
	memcpy(fnode->name, name, len > 15 ? 15 : len);
	fnode->up = dir->i_ino;
	fnode->dirflag = 1;
	fnode->btree.n_free_nodes = 7;
	fnode->btree.n_used_nodes = 1;
	fnode->btree.first_free = 0x14;
	fnode->u.external[0].disk_secno = dno;
	fnode->u.external[0].file_secno = -1;
	dnode->root_dnode = 1;
	dnode->up = fno;
	de = add_de(dir->i_sb, dnode, "\001\001", 2, 0);
	de->creation_date = de->write_date = de->read_date = gmt_to_local(CURRENT_TIME);
	if (!(mode & 0222)) de->read_only = 1;
	de->first = de->hidden = de->system = de->directory = 1;
	de->fnode = fno;
	/*de = add_de(dir->i_sb, dnode, "\377", 1, 0);
	de->last = 1;*/
	mark_buffer_dirty(bh, 1);
	mark_4buffers_dirty(&qbh0);
	brelse(bh);
	brelse4(&qbh0);
	dir->i_nlink++;
	hpfs_unlock_inode(dir);
	lock_iget(dir->i_sb, 1);
	if ((result = iget(dir->i_sb, fno))) {
		result->i_ctime = result->i_mtime = result->i_atime = local_to_gmt(dee.creation_date);
		result->i_hpfs_ea_size = 0;
		if (dee.read_only) result->i_mode &= ~0222;
		/*printk("%08x\n", result->i_nlink);*/
		if (result->i_uid != current->fsuid ||
		    result->i_gid != current->fsgid ||
		    result->i_mode != (mode | S_IFDIR)) {
		    	result->i_dirt = 1;
			result->i_uid = current->fsuid;
			result->i_gid = current->fsgid;
			result->i_mode = mode | S_IFDIR;
		}
		unlock_iget(dir->i_sb);
		iput(result);
	} else unlock_iget(dir->i_sb);
	iput(dir);
	return 0;
	bail2:
	brelse4(&qbh0);
	free_dnode(dir->i_sb, dno);
	hpfs_unlock_inode(dir);
	bail1:
	brelse(bh);
	free_sectors(dir->i_sb, fno, 1);
	bail:
	iput(dir);
	return -ENOSPC;
}

int hpfs_create(struct inode *dir, const char *name, int len, int mode,
		       struct inode **result)
{
	struct buffer_head *bh;
	struct fnode *fnode;
	fnode_secno fno;
	int r;
	struct hpfs_dirent dee;
	*result = NULL;
	if (!dir) return -ENOENT;
	if (!S_ISDIR(dir->i_mode)) {
		iput(dir);
		return -ENOTDIR;
	}
	if (!S_ISREG(mode)) {
		iput(dir);
		return -EINVAL;
	}
	adjust_length((char *)name, &len);
	if (chk_hpfs_name((char *)name, len)) {
		iput(dir);
		return -ENAMETOOLONG;
	}	
	if (!(fnode = alloc_fnode(dir->i_sb, dir->i_hpfs_dno, &fno, &bh))) goto bail;
	memset(&dee, 0, sizeof dee);
	if (!(mode & 0222)) dee.read_only = 1;
	dee.archive = 1;
	dee.fnode = fno;
	dee.creation_date = dee.write_date = dee.read_date = gmt_to_local(CURRENT_TIME);
	hpfs_lock_inode(dir);
	r = add_dirent(dir, (char *)name, len, &dee, 0);
	if (r == 1) goto bail1;
	if (r == -1) {
		brelse(bh);
		free_sectors(dir->i_sb, fno, 1);
		hpfs_unlock_inode(dir);
		iput(dir);
		return -EEXIST;
	}
	fnode->len = len;
	memcpy(fnode->name, name, len > 15 ? 15 : len);
	fnode->up = dir->i_ino;
	mark_buffer_dirty(bh, 1);
	brelse(bh);
	hpfs_unlock_inode(dir);
	lock_iget(dir->i_sb, 2);
	if ((*result = iget(dir->i_sb, fno))) {
		decide_conv(*result, (char *)name, len);
		(*result)->i_ctime = (*result)->i_mtime = (*result)->i_atime = local_to_gmt(dee.creation_date);
		(*result)->i_hpfs_ea_size = 0;
		if (dee.read_only) (*result)->i_mode &= ~0222;
		if ((*result)->i_blocks == -1) (*result)->i_blocks = 512;
		if ((*result)->i_size == -1) (*result)->i_size = 0;
		if ((*result)->i_uid != current->fsuid ||
		    (*result)->i_gid != current->fsgid ||
		    (*result)->i_mode != (mode | S_IFREG)) {
		    	(*result)->i_dirt = 1;
			(*result)->i_uid = current->fsuid;
			(*result)->i_gid = current->fsgid;
			(*result)->i_mode = mode | S_IFREG;
		}
	}
	unlock_iget(dir->i_sb);
	iput(dir);
	return 0;
	bail1:
	brelse(bh);
	free_sectors(dir->i_sb, fno, 1);
	hpfs_unlock_inode(dir);
	bail:
	iput(dir);
	return -ENOSPC;
}

int hpfs_mknod(struct inode *dir, const char *name, int len, int mode,
		      int rdev)
{
	struct buffer_head *bh;
	struct fnode *fnode;
	fnode_secno fno;
	int r;
	struct hpfs_dirent dee;
	struct inode *result = NULL;
	if (!dir) return -ENOENT;
	if (dir->i_sb->s_hpfs_eas < 2) {
		iput(dir);
		return -EPERM;
	}
	if (!S_ISDIR(dir->i_mode)) {
		iput(dir);
		return -ENOTDIR;
	}
	if (!S_ISBLK(mode) && !S_ISCHR(mode) && !S_ISFIFO(mode) && !S_ISSOCK(mode)) {
		iput(dir);
		return -EINVAL;
	}
	adjust_length((char *)name, &len);
	if (chk_hpfs_name((char *)name, len)) {
		iput(dir);
		return -ENAMETOOLONG;
	}	
	if (!(fnode = alloc_fnode(dir->i_sb, dir->i_hpfs_dno, &fno, &bh))) goto bail;
	memset(&dee, 0, sizeof dee);
	if (!(mode & 0222)) dee.read_only = 1;
	dee.archive = 1;
	dee.fnode = fno;
	dee.creation_date = dee.write_date = dee.read_date = gmt_to_local(CURRENT_TIME);
	hpfs_lock_inode(dir);
	r = add_dirent(dir, (char *)name, len, &dee, 0);
	if (r == 1) goto bail1;
	if (r == -1) {
		brelse(bh);
		free_sectors(dir->i_sb, fno, 1);
		hpfs_unlock_inode(dir);
		iput(dir);
		return -EEXIST;
	}
	fnode->len = len;
	memcpy(fnode->name, name, len > 15 ? 15 : len);
	fnode->up = dir->i_ino;
	mark_buffer_dirty(bh, 1);
	brelse(bh);
	hpfs_unlock_inode(dir);
	lock_iget(dir->i_sb, 2);
	if ((result = iget(dir->i_sb, fno))) {
		result->i_ctime = result->i_mtime = result->i_atime = local_to_gmt(dee.creation_date);
		result->i_hpfs_ea_size = 0;
		if (result->i_blocks == -1) result->i_blocks = 512;
		if (result->i_size == -1) result->i_size = 0;
		result->i_mode = mode;
		result->i_uid = current->fsuid;
		result->i_gid = current->fsgid;
		if (!S_ISFIFO(mode)) result->i_rdev = to_kdev_t(rdev);
		result->i_nlink = 1;
		result->i_size = 0;
		result->i_blocks = 1;
		result->i_op = NULL;
		if (S_ISBLK(result->i_mode)) result->i_op = (struct inode_operations *) &blkdev_inode_operations;
		if (S_ISCHR(result->i_mode)) result->i_op = (struct inode_operations *) &chrdev_inode_operations;
		if (S_ISFIFO(result->i_mode)) init_fifo(result);
		result->i_dirt = 1;
		unlock_iget(dir->i_sb);
		iput(result);
	} else unlock_iget(dir->i_sb);
	iput(dir);
	return 0;
	bail1:
	brelse(bh);
	free_sectors(dir->i_sb, fno, 1);
	hpfs_unlock_inode(dir);
	bail:
	iput(dir);
	return -ENOSPC;
}

extern const struct inode_operations hpfs_symlink_iops;

int hpfs_symlink(struct inode *dir, const char *name, int len,
			const char *symlink)
{
	struct buffer_head *bh;
	struct fnode *fnode;
	fnode_secno fno;
	int r;
	struct hpfs_dirent dee;
	struct inode *result;
	if (!dir) return -ENOENT;
	if (dir->i_sb->s_hpfs_eas < 2) {
		iput(dir);
		return -EPERM;
	}
	if (!S_ISDIR(dir->i_mode)) {
		iput(dir);
		return -ENOTDIR;
	}
	adjust_length((char *)name, &len);
	if (chk_hpfs_name((char *)name, len)) {
		iput(dir);
		return -ENAMETOOLONG;
	}
	/*if (chk_hpfs_name((char *)symlink, strlen(symlink))) {
		iput(dir);
		return -ENAMETOOLONG;
	}*/
	if (!(fnode = alloc_fnode(dir->i_sb, dir->i_hpfs_dno, &fno, &bh))) goto bail;
	memset(&dee, 0, sizeof dee);
	dee.archive = 1;
	dee.fnode = fno;
	dee.creation_date = dee.write_date = dee.read_date = gmt_to_local(CURRENT_TIME);
	hpfs_lock_inode(dir);
	r = add_dirent(dir, (char *)name, len, &dee, 0);
	if (r == 1) goto bail1;
	if (r == -1) {
		brelse(bh);
		free_sectors(dir->i_sb, fno, 1);
		hpfs_unlock_inode(dir);
		iput(dir);
		return -EEXIST;
	}
	fnode->len = len;
	memcpy(fnode->name, name, len > 15 ? 15 : len);
	fnode->up = dir->i_ino;
	mark_buffer_dirty(bh, 1);
	brelse(bh);
	hpfs_unlock_inode(dir);
	lock_iget(dir->i_sb, 2);
	if ((result = iget(dir->i_sb, fno))) {
		result->i_ctime = result->i_mtime = result->i_atime = local_to_gmt(dee.creation_date);
		result->i_hpfs_ea_size = 0;
		if (result->i_blocks == -1) result->i_blocks = 512;
		if (result->i_size == -1) result->i_size = 0;
		result->i_mode = S_IFLNK | 0777;
		result->i_uid = current->fsuid;
		result->i_gid = current->fsgid;
		result->i_blocks = 1;
		result->i_size = strlen(symlink);
		result->i_op = (struct inode_operations *) &hpfs_symlink_iops;
		if ((fnode = map_fnode(dir->i_sb, fno, &bh))) {
			set_ea(result, fnode, "SYMLINK", (char *)symlink, strlen(symlink));
			mark_buffer_dirty(bh, 1);
			brelse(bh);
		}
		result->i_dirt = 1;
		unlock_iget(dir->i_sb);
		iput(result);
	} else unlock_iget(dir->i_sb);
	iput(dir);
	return 0;
	bail1:
	brelse(bh);
	free_sectors(dir->i_sb, fno, 1);
	hpfs_unlock_inode(dir);
	bail:
	iput(dir);
	return -ENOSPC;
}

int hpfs_unlink(struct inode *dir, const char *name, int len)
{
	struct quad_buffer_head qbh;
	struct hpfs_dirent *de;
	struct inode *inode;
	dnode_secno dno;
	fnode_secno fno;
	int depth, r;
	adjust_length((char *)name, &len);
	hpfs_lock_inode(dir);
	if (!(de = map_dirent(dir, dir->i_hpfs_dno, (char *)name, len, &dno, &qbh, &depth))) {
		hpfs_unlock_inode(dir);
		iput(dir);
		return -ENOENT;
	}	
	if (de->first) {
		brelse4(&qbh);
		hpfs_unlock_inode(dir);
		iput(dir);
		return -EPERM;
	}
	if (de->directory) {
		brelse4(&qbh);
		hpfs_unlock_inode(dir);
		iput(dir);
		return -EISDIR;
	}
	lock_iget(dir->i_sb, 2);
	if ((inode = iget(dir->i_sb, de->fnode))) {
		if (inode->i_count > 1) {
			brelse4(&qbh);
			unlock_iget(dir->i_sb);
			hpfs_unlock_inode(dir);
			iput(inode);
			iput(dir);
			return -EBUSY;
		}
	} else printk("HPFS: could not get inode %08x ... trying to continue\n", de->fnode);
	unlock_iget(dir->i_sb);
	fno = de->fnode;
	if ((r = remove_dirent(dir, dno, de, &qbh, depth)) == 1) hpfs_error(dir->i_sb, "there was error when removing dirent");
	hpfs_unlock_inode(dir);
	if (r != 2) {
		if (inode) {
			inode->i_nlink = 0;
			inode->i_dirt = 0;
			iput(inode);
			if (inode->i_count) printk("HPFS: Linus, why do you allow such crazy multitasking in VFS? What can I do now?\n");
			else clear_inode(inode);
		}
		remove_fnode(dir->i_sb, fno);
	} 
	if (r==2 && inode) iput(inode);
	iput(dir);
	return r == 2 ? -ENOSPC : 0;
}

int hpfs_rmdir(struct inode *dir, const char *name, int len)
{
	struct quad_buffer_head qbh;
	struct hpfs_dirent *de;
	struct inode *inode;
	dnode_secno dno;
	fnode_secno fno;
	int n_items = 0;
	int depth, r;
	adjust_length((char *)name, &len);
	hpfs_lock_inode(dir);
	if (!(de = map_dirent(dir, dir->i_hpfs_dno, (char *)name, len, &dno, &qbh, &depth))) {
		hpfs_unlock_inode(dir);
		iput(dir);
		return -ENOENT;
	}	
	if (de->first) {
		brelse4(&qbh);
		hpfs_unlock_inode(dir);
		iput(dir);
		return -EPERM;
	}
	if (!de->directory) {
		brelse4(&qbh);
		hpfs_unlock_inode(dir);
		iput(dir);
		return -ENOTDIR;
	}
	count_dnodes(dir->i_sb, fnode_dno(dir->i_sb, de->fnode), NULL, NULL, &n_items);
	if (n_items) {
		brelse4(&qbh);
		hpfs_unlock_inode(dir);
		iput(dir);
		return -ENOTEMPTY;
	}
	lock_iget(dir->i_sb, 1);
	if ((inode = iget(dir->i_sb, de->fnode))) {
		if (inode->i_count > 1) {
			brelse4(&qbh);
			unlock_iget(dir->i_sb);
			hpfs_unlock_inode(dir);
			iput(inode);
			iput(dir);
			return -EBUSY;
		}
	} else printk("HPFS: could not get inode %08x ... trying to continue\n", de->fnode);
	unlock_iget(dir->i_sb);
	fno = de->fnode;
	if ((r = remove_dirent(dir, dno, de, &qbh, depth)) == 1) hpfs_error(dir->i_sb, "there was error when removing dirent");
	hpfs_unlock_inode(dir);
	if (r != 2) {
		if (inode) {
			inode->i_nlink = 0;
			inode->i_dirt = 0;
			iput(inode);
			if (inode->i_count) printk("HPFS: Linus, why do you allow such crazy multitasking in VFS? What can I do now?\n");
			else clear_inode(inode);
		}
		dir->i_nlink--;
		remove_fnode(dir->i_sb, fno);
	}
	if (r == 2 && inode) iput(inode);
	iput(dir);
	return r == 2 ? -ENOSPC : 0;
}

int hpfs_readlink(struct inode *i, char *buf, int len)
{
	struct fnode *fnode;
	struct buffer_head *bh;
	char *symlink;
	int slen;
	if (!S_ISLNK(i->i_mode)) {
		iput(i);
		return -EINVAL;
	}
	if (!(fnode = map_fnode(i->i_sb, i->i_ino, &bh))) {
		iput(i);
		return -EIO;
	}
	if (!(symlink = get_ea(i->i_sb, fnode, "SYMLINK", &slen))) {
		iput(i);
		return -EFSERROR;
	}
	brelse(bh);
	if (slen > len) slen = len;
	memcpy_tofs(buf, symlink, slen);
	kfree(symlink);
	iput(i);
	return slen;
}

int hpfs_follow_link(struct inode *dir, struct inode *inode, int flag,
			    int mode, struct inode **res_inode)
{
	char *link;
	int len;
	int error;
	struct buffer_head *bh;
	struct fnode *fnode;
	if (!dir) {
		dir = current->fs->root;
		dir->i_count++;
	}
	if (!inode) {
		iput(dir);
		return -ENOENT;
	}
	if (!S_ISLNK(inode->i_mode)) {
		iput(dir);
		*res_inode = inode;
		return 0;
	}
	if (current->link_count > 5) {
		iput(dir);
		iput(inode);
		return -ELOOP;
	}
	if (!(fnode = map_fnode(inode->i_sb, inode->i_ino, &bh))) {
		iput(dir);
		iput(inode);
		return -EIO;
	}
	if (!(link = get_ea(inode->i_sb, fnode, "SYMLINK", &len))) {
		brelse(bh);
		iput(dir);
		iput(inode);
		return -EFSERROR;
	}
	brelse(bh);
	UPDATE_ATIME(inode);
	current->link_count++;
	error = open_namei(link, flag, mode, res_inode, dir);
	current->link_count--;
	kfree(link);
	iput (inode);
	return error;
}

int hpfs_rename(struct inode *old_dir, const char *old_name, int old_len,
		struct inode *new_dir, const char *new_name, int new_len,
		int must_be_dir)
{
	struct inode *i;
	struct quad_buffer_head qbh;
	struct hpfs_dirent *dep;
	struct hpfs_dirent de;
	dnode_secno dno;
	struct fnode *fnode;
	struct buffer_head *bh;
	int depth;
	int c;
	int err = 0;
	if (!old_dir || !new_dir) {
		/*if (old_dir) iput(old_dir);*/
		if (new_dir) iput(new_dir);
		return -ENOENT;
	}
	if (!S_ISDIR(old_dir->i_mode) || !S_ISDIR(new_dir->i_mode)) {
		err = -ENOTDIR;
		goto end2;
	}
	adjust_length((char *)old_name, &old_len);
	adjust_length((char *)new_name, &new_len);
	if (chk_hpfs_name((char *)new_name, new_len) || chk_hpfs_name((char *)old_name, old_len)) {
		err = -ENAMETOOLONG;
		goto end2;
	}
	if ((c = hpfs_lookup(old_dir, old_name, old_len, &i))) {
		err = c;
		goto end2;
	}	
	if (i->i_sb != old_dir->i_sb || i->i_sb != new_dir->i_sb) {
		err = -EINVAL; /* Do not allow to move mount points */
		goto end1_;
	}
	if (new_dir != old_dir) {
		int c1, c2 = 0;
		fnode_secno fno = new_dir->i_ino;
		if (old_dir->i_sb->s_hpfs_root == i->i_ino) {
			err = -EINVAL;
			goto end1_;
		}
		while (fno != old_dir->i_sb->s_hpfs_root) {
			if (old_dir->i_sb->s_hpfs_chk)
				if (stop_cycles(old_dir->i_sb, fno, &c1, &c2, "hpfs_rename")) {
					err = -EFSERROR;
					goto end1_;
				}
			if (fno == i->i_ino) {
				err = -EINVAL;
				goto end1_;
			}
			if (!(fnode = map_fnode(new_dir->i_sb, fno, &bh))) {
				err = -EFSERROR;
				goto end1_;
			}
			fno = fnode->up;
			brelse(bh);
		}
	}

#define DEBUG_CL /*printk("lock %d, queue %p\n", old_dir->i_sb->s_hpfs_creation_de_lock, old_dir->i_sb->s_hpfs_creation_de);*/

	if (new_dir != old_dir || compare_names(old_dir->i_sb, (char *)old_name, old_len, (char *)new_name, new_len, 0)) {
		new_dir->i_count++;
		err = hpfs_unlink(new_dir, new_name, new_len);
		if (err == -ENOENT) err = 0;
		if (err) goto end1_;
	}
	
	DEBUG_CL;
	hpfs_lock_3inodes(old_dir, new_dir, i);
	if (!(dep = map_dirent(old_dir, old_dir->i_hpfs_dno, (char *)old_name, old_len, &dno, &qbh, &depth))) {
		hpfs_error(i->i_sb, "lookup succeeded but map dirent failed");
		err = -ENOENT;
		goto end1;
	}
	copy_de(&de, dep);
	DEBUG_CL;
	lock_creation(old_dir->i_sb);
	c = add_dirent(new_dir, (char *)new_name, new_len, &de, depth + 2);
	DEBUG_CL;
	if (c == -1) {
		dnode_secno down = 0;
		if (new_dir != old_dir) {
			brelse4(&qbh);
			err = -EEXIST;
			goto end;
		}
		if (check_free_dnodes(old_dir->i_sb, depth + 2)) {
			brelse4(&qbh);
			err = -ENOSPC;
			goto end;
		}
		if (de.down) down = de_down_pointer(&de);
		delete_de(old_dir->i_sb, qbh.data, dep);
		mark_4buffers_dirty(&qbh);
		brelse4(&qbh);
		if (add_to_dnode(old_dir, dno, (char *)new_name, new_len, &de, down)) err = -EFSERROR;
		goto end;
	}
	if (c == 1) {
		err = -EFSERROR;
		goto end;
	}
	brelse4(&qbh);
	DEBUG_CL;
	if (!(dep = map_dirent(old_dir, old_dir->i_hpfs_dno, (char *)old_name, old_len, &dno, &qbh, &depth))) {
		hpfs_error(i->i_sb, "lookup succeeded but map dirent failed #2");
		err = -EFSERROR;
	} else if (remove_dirent(old_dir, dno, dep, &qbh, 0)) err = -EFSERROR;
	DEBUG_CL;
	i->i_hpfs_parent_dir = new_dir->i_ino;
	if ((fnode = map_fnode(i->i_sb, i->i_ino, &bh))) {
		if (fnode->dirflag) {
			old_dir->i_nlink--;
			new_dir->i_nlink++;
		}
		if (fnode->up != new_dir->i_ino || fnode->len != new_len ||
		    memcmp(fnode->name, new_name, new_len>15?15:new_len)) {
			fnode->up = new_dir->i_ino;
			fnode->len = new_len;
			memcpy(fnode->name, new_name, new_len>15?15:new_len);
			if (new_len < 15)
				memset(&fnode->name[new_len], 0, 15 - new_len);
			mark_buffer_dirty(bh, 1);
		}
		brelse(bh);
	}
	end:
	DEBUG_CL;
	unlock_creation(old_dir->i_sb);
	end1:
	hpfs_unlock_3inodes(old_dir, new_dir, i);
	end1_:
	iput(i);
	end2:
	iput(new_dir);
	/*iput(old_dir);*/
	return err;
}

