/*
 *  linux/fs/hpfs/alloc.c
 *
 *  Mikulas Patocka (mikulas@artax.karlin.mff.cuni.cz), 1998
 *
 *  file VFS functions
 */

#include "hpfs_fn.h"

/*
 * generic_file_read often calls bmap with non-existing sector,
 * so we must ignore such errors.
 */

secno hpfs_bmap(struct inode *inode, unsigned file_secno)
{
	unsigned n, disk_secno;
	struct fnode *fnode;
	struct buffer_head *bh;
	if (((inode->i_size + 511) >> 9) <= file_secno) return 0;
	n = file_secno - inode->i_hpfs_file_sec;
	if (n < inode->i_hpfs_n_secs) return inode->i_hpfs_disk_sec + n;
	if (!(fnode = map_fnode(inode->i_sb, inode->i_ino, &bh))) return 0;
	disk_secno = bplus_lookup(inode->i_sb, inode, &fnode->btree, file_secno, bh);
	if (disk_secno == -1) return 0;
	return disk_secno;
}

void hpfs_truncate(struct inode *i)
{
	if (IS_IMMUTABLE(i)) return /*-EPERM*/;
	i->i_hpfs_n_secs = 0;
	truncate_btree(i->i_sb, i->i_ino, 1, ((i->i_size + 511) >> 9));
	i->i_blocks = 1 + ((i->i_size + 511) >> 9);
	i->i_dirt = 1;
}

/*
static void __wait_on_inode(struct inode * inode)
{
        struct wait_queue wait = { current, NULL };

        add_wait_queue(&inode->i_wait, &wait);
	repeat:
        current->state = TASK_UNINTERRUPTIBLE;
        if (inode->i_lock) {
		schedule();
		goto repeat;
	}
	remove_wait_queue(&inode->i_wait, &wait);
	current->state = TASK_RUNNING;
}

static inline void wait_on_inode(struct inode * inode)
{
        if (inode->i_lock) __wait_on_inode(inode);
}



static inline void lock_inode(struct inode * inode)
{
        wait_on_inode(inode);
	inode->i_lock = 1;
}
*/


int hpfs_file_read(struct inode *inode, struct file *filp, char *buf, int count)
{
	int i,j;
	int a = generic_file_read(inode, filp, buf, count);
	/*lock_inode(inode);*/
	/*down(&inode->i_sem);*/
	if (inode->i_hpfs_conv != CONV_TEXT || a < 0) return a;
	for (i = 0, j = 0; i < a; i++) {
		char c = get_user(buf + i);
		if (c != '\r') {
			if (i != j) put_user(c, buf + j);
			j++;
		}
	}
	return j;
}

int hpfs_file_write(struct inode *i, struct file *filp, const char *buf,
		    int count)
{
	int carry, error = 0;
	const char *start = buf;
	if (!i) return -EINVAL;
	if (!S_ISREG(i->i_mode)) return -EINVAL;
	if (IS_IMMUTABLE(i)) return -EPERM;
	if (filp->f_flags & O_APPEND) filp->f_pos = i->i_size;
	if (count <= 0) return 0;
	if ((unsigned)(filp->f_pos+count) >= 0x8000000U || (unsigned)count >= 0x80000000U) return -EFBIG;
	carry = 0;
	while (count || carry) {
		int ii, add = 0;
		secno sec = 0; /* Go away, uninitialized variable warning */
		int offset, size, written;
		char ch;
		struct buffer_head *bh;
		char *data;
		/* To be fixed: writing far behind file end !!!*/
		offset = filp->f_pos & 0x1ff;
		size = count > 0x200 - offset ? 0x200 - offset : count;
		if ((filp->f_pos >> 9) < ((i->i_size + 0x1ff) >> 9)) {
			i->i_hpfs_n_secs = 0;
			if (!(sec = hpfs_bmap(i, filp->f_pos >> 9))) {
				hpfs_error(i->i_sb, "bmap failed, file %08x, fsec %08x", i->i_ino, filp->f_pos >> 9);
				error =- EFSERROR;
				break;
			}
		} else for (ii = (i->i_size + 0x1ff) >> 9, add = 1; ii <= filp->f_pos >> 9; ii++) {
			if ((sec = add_sector_to_btree(i->i_sb, i->i_ino, 1, ii)) == -1) {
				hpfs_truncate(i);
				return -ENOSPC;
			}
			if (filp->f_pos != i->i_size)
				if ((data = get_sector(i->i_sb, sec, &bh))) {
					memset(data, 0, 512);
					mark_buffer_dirty(bh, 0);
					brelse(bh);
				}
			i->i_size = 0x200 * ii + 1;
			i->i_blocks++;
			i->i_dirt = 1;
			if (i->i_sb->s_hpfs_chk >= 2) {
				secno bsec;
				bsec = hpfs_bmap(i, ii);
				if (sec != bsec) {
					hpfs_error(i->i_sb, "sec == %08x, bmap returns %08x", sec, bsec);
					error = -EFSERROR;
					break;
				}
			}	
			PRINTK(("file_write: added %08x\n", sec));
		}
		if (!sec || sec == 15) {
			hpfs_error(i->i_sb, "bmap returned empty sector");
			error = -EFSERROR;
			break;
		}
		if (i->i_sb->s_hpfs_chk)
			if (chk_sectors(i->i_sb, sec, 1, "data")) {
				error = -EFSERROR;
				break;
			}
		/*if ((!offset && size == 0x200) || add) printk("g %03x %03x %08x\n",size,offset,sec);
		else printk("m %03x %03x %08x\n",size,offset,sec);*/
		if ((!offset && size == 0x200) || add)
			data = get_sector(i->i_sb, sec, &bh);
		else data = map_sector(i->i_sb, sec, &bh, 0);
		if (!data) {
			error = -EIO;
			break;
		}
		if (i->i_hpfs_conv != CONV_TEXT) {
			memcpy_fromfs(data + offset, buf, written = size);
			buf += size;
		} else {
			int left;
			char *to;
			/* LF->CR/LF conversion, stolen from fat fs */
			written = left = 0x200 - offset;
			to = (char *) bh->b_data + (filp->f_pos & 0x1ff);
			if (carry) {
				*to++ = '\n';
				left--;
				carry = 0;
			}
			for (size = 0; size < count && left; size++) {
				if ((ch = get_user(buf++)) == '\n') {
					*to++ = '\r';
					left--;
				}
				if (!left) carry = 1;
				else {
					*to++ = ch;
					left--;
				}
			}
			written -= left;
		}
		update_vm_cache(i, filp->f_pos, bh->b_data + (filp->f_pos & 0x1ff), written);
		filp->f_pos += written;
		if (filp->f_pos > i->i_size) {
			i->i_size = filp->f_pos;
			i->i_dirt = 1;
		}
		mark_buffer_dirty(bh, 0);
		brelse(bh);
		count -= size;
	}
	if (start == buf) return error;
	i->i_mtime = CURRENT_TIME;
	i->i_dirt = 1;
	return buf - start;
}

