/*
 *  linux/fs/hpfs/alloc.c
 *
 *  Mikulas Patocka (mikulas@artax.karlin.mff.cuni.cz), 1998
 *
 *  directory VFS functions
 */

#include "hpfs_fn.h"

int hpfs_dir_read(struct inode *inode, struct file *filp, char *name, int len)
{
	return -EISDIR;
}

void hpfs_dir_release(struct inode *inode, struct file *filp)
{
	del_pos(inode, &filp->f_pos);
}

int hpfs_readdir(struct inode *inode, struct file *filp, void * dirent,
		 filldir_t filldir)
{
	struct quad_buffer_head qbh;
	struct hpfs_dirent *de;
	int lc;
	long old_pos;
	char *tempname;
	int c1, c2 = 0;

	/*PRINTK(("hpfs_readdir(%08x,%08x)\n",inode->i_ino,filp->f_pos));*/

	if (!inode) return -EBADF;
	if (!S_ISDIR(inode->i_mode)) return -EBADF;
	if (inode->i_sb->s_hpfs_chk) {
		if (chk_sectors(inode->i_sb, inode->i_ino, 1, "dir fnode"))
			return -EFSERROR;
		if (chk_sectors(inode->i_sb, inode->i_hpfs_dno, 4, "dir dnode"))
			return -EFSERROR;
	}
	if (inode->i_sb->s_hpfs_chk >= 2) {
		struct buffer_head *bh;
		struct fnode *fno;
		int e = 0;
		if (!(fno = map_fnode(inode->i_sb, inode->i_ino, &bh)))
			return -EIOERROR;
		if (!fno->dirflag) {
			e = 1;
			hpfs_error(inode->i_sb, "not a directory, fnode %08x",inode->i_ino);
		}
		if (inode->i_hpfs_dno != fno->u.external[0].disk_secno) {
			e = 1;
			hpfs_error(inode->i_sb, "corrupted inode: i_hpfs_dno == %08x, fnode -> dnode == %08x", inode->i_hpfs_dno, fno->u.external[0].disk_secno);
		}
		brelse(bh);
		if (e) return -EFSERROR;
	}
	lc = inode->i_sb->s_hpfs_lowercase;
	if (filp->f_pos == -2) { /* diff -r requires this (note, that diff -r */
		filp->f_pos = -3; /* also fails on msdos filesystem) */
		return 0;
	}
	if (filp->f_pos == -3) return -ENOENT;
	
	hpfs_lock_inode(inode);
	
	while (1) {
		again:
		/* This won't work when cycle is longer than number of dirents
		   accepted by filldir, but what can I do?
		   maybe killall -9 ls helps */
		if (inode->i_sb->s_hpfs_chk)
			if (stop_cycles(inode->i_sb, filp->f_pos, &c1, &c2, "hpfs_readdir")) {
				hpfs_unlock_inode(inode);
				return -EFSERROR;
			}
		if (filp->f_pos == -2) {
			/*PRINTK(("hpfs_readdir: exiting...\n"));*/
			hpfs_unlock_inode(inode);
			return 0;
		}
		if (filp->f_pos == 3 || filp->f_pos == 4 || filp->f_pos == 5) {
			printk("HPFS: warning: pos==%d\n",(int)filp->f_pos);
			hpfs_unlock_inode(inode);
			return 0;
		}
		if (filp->f_pos == 0) {
			if (filldir(dirent, ".", 1, filp->f_pos, inode->i_ino) < 0) return 0;
			/*PRINTK(("hpfs_readdir: '.' placed\n"));*/
			filp->f_pos = -1;
		}
		if (filp->f_pos == -1) {
			if (filldir(dirent, "..", 2, filp->f_pos, inode->i_hpfs_parent_dir) < 0) return 0;
			/*PRINTK(("hpfs_readdir: '..' placed\n"));*/
			filp->f_pos = 1;
		}
		if (filp->f_pos == 1) {
			filp->f_pos = ((loff_t) de_as_down_as_possible(inode->i_sb, inode->i_hpfs_dno) << 4) + 1;
			add_pos(inode, &filp->f_pos);
			filp->f_version = inode->i_version;
		}
			/*if (filp->f_version != inode->i_version) {
				hpfs_unlock_inode(inode);
				return -ENOENT;
			}*/	
			old_pos = filp->f_pos;
			if (!(de = map_pos_dirent(inode, &filp->f_pos, &qbh))) {
				hpfs_unlock_inode(inode);
				return -EIOERROR;
			}
			/*PRINTK(("hpfs_readdir: map_pos_dirent: %08x -> %08x\n",old_pos,filp->f_pos));*/
			if (de->first || de->last) {
				if (inode->i_sb->s_hpfs_chk) {
					if (de->first && !de->last && (de->namelen != 2 || de ->name[0] != 1 || de->name[1] != 1 || de->last)) hpfs_error(inode->i_sb, "hpfs_readdir: bad ^A^A entry; pos = %08x", old_pos);
					if (de->last && (de->namelen != 1 || de ->name[0] != 255)) hpfs_error(inode->i_sb, "hpfs_readdir: bad \\377 entry; pos = %08x", old_pos);
				}
				brelse4(&qbh);
				goto again;
			}
			tempname = translate_hpfs_name(inode->i_sb, de->name, de->namelen, lc, de->not_8x3);
			if (filldir(dirent, tempname, de->namelen, old_pos, de->fnode) < 0) {
				/*PRINTK(("hpfs_readdir: '%c',%d not placed\n",tempname[0],de->namelen));*/
				filp->f_pos = old_pos;
				if (tempname != (char *)de->name) kfree(tempname);
				brelse4(&qbh);
				hpfs_unlock_inode(inode);
				return 0;
			}
			/*PRINTK(("hpfs_readdir: '%c',%d placed\n",tempname[0],de->namelen));*/
			if (tempname != (char *)de->name) kfree(tempname);
			brelse4(&qbh);
	}
}

/*
 * lookup.  Search the specified directory for the specified name, set
 * *result to the corresponding inode.
 *
 * lookup uses the inode number to tell read_inode whether it is reading
 * the inode of a directory or a file -- file ino's are odd, directory
 * ino's are even.  read_inode avoids i/o for file inodes; everything
 * needed is up here in the directory.  (And file fnodes are out in
 * the boondocks.)
 *
 *    - M.P.: this is over, sometimes we've got to read file's fnode for eas
 *	      inode numbers are just fnode sector numbers
 */

int hpfs_lookup(struct inode *dir, const char *name, int len,
		struct inode **result)
{
	struct quad_buffer_head qbh;
	struct hpfs_dirent *de;
	struct inode *inode;
	ino_t ino;

	*result = 0;
	if (dir == 0) return -ENOENT;
	adjust_length((char *)name, &len);
	hpfs_lock_inode(dir);
	if (!S_ISDIR(dir->i_mode)) goto bail;
	/*
	 * Read in the directory entry. "." is there under the name ^A^A .
	 * Always read the dir even for . and .. in case we need the dates.
	 *
	 * M.P.: No - we can't read '^A^A' for current directory. In some cases
	 * the information under '^A^A' is incomplete (missing ea_size, bad
	 * fnode number) - chkdsk ignores such errors and it doesn't seem to
	 * matter under OS/2.
	 */
	if (name[0] == '.' && len == 1) {
		*result = dir;
		dir->i_count++;
		goto end;
	} else if (name[0] == '.' && name[1] == '.' && len == 2) {
		struct buffer_head *bh;
		struct fnode *fnode;
		if (dir->i_ino == dir->i_sb->s_hpfs_root) {
			*result = dir;
			dir->i_count++;
			goto end;
		}
		if (dir->i_hpfs_parent_dir == dir->i_sb->s_hpfs_root) {
			*result = dir->i_sb->s_mounted;
			dir->i_sb->s_mounted->i_count++;
			goto end;
		}
		if (!(fnode = map_fnode(dir->i_sb, dir->i_hpfs_parent_dir, &bh))) goto bail;
		de = map_fnode_dirent(dir->i_sb, dir->i_hpfs_parent_dir, fnode, &qbh);
		brelse(bh);
	} else
		de = map_dirent(dir, dir->i_hpfs_dno, (char *) name, len, NULL, &qbh, NULL);

	/*
	 * This is not really a bailout, just means file not found.
	 */

	if (!de)
		goto bail;

	/*
	 * Get inode number, what we're after.
	 */

	ino = de->fnode;

	/*
	 * Go find or make an inode.
	 */

	lock_iget(dir->i_sb, de->directory || (de->ea_size && dir->i_sb->s_hpfs_eas) ? 1 : 2);
	if (!(inode = iget(dir->i_sb, ino))) {
		unlock_iget(dir->i_sb);
		hpfs_error(inode->i_sb, "hpfs_lookup: can't get inode");
		goto bail1;
	}
	decide_conv(inode, (char *)name, len);
	unlock_iget(dir->i_sb);

	if ((de->has_acl || de->has_xtd_perm) && !(dir->i_sb->s_flags & MS_RDONLY)) {
		hpfs_error(inode->i_sb, "ACLs or XPERM found. This is probably HPFS386. This driver doesn't support it now. Send me some info on these structures");
		goto bail1;
	}

	/*
	 * Fill in the info from the directory if this is a newly created
	 * inode.
	 */

	if (!inode->i_ctime) {
		if (!(inode->i_ctime = local_to_gmt(de->creation_date)))
			inode->i_ctime = 1;
		inode->i_mtime = local_to_gmt(de->write_date);
		inode->i_atime = local_to_gmt(de->read_date);
		inode->i_hpfs_ea_size = de->ea_size;
		if (!inode->i_hpfs_ea_mode && de->read_only)
			inode->i_mode &= ~0222;
		if (!de->directory) {
			if (inode->i_size == -1) {
				inode->i_size = de->file_size;
			/*
			 * i_blocks should count the fnode and any anodes.
			 * We count 1 for the fnode and don't bother about
			 * anodes -- the disk heads are on the directory band
			 * and we want them to stay there.
			 */
				inode->i_blocks = 1 + ((inode->i_size + 511) >> 9);
			}
		}
	}

	brelse4(&qbh);

	/*
	 * Made it.
	 */

	*result = inode;
	end:
	hpfs_unlock_inode(dir);
	iput(dir);
	return 0;

	/*
	 * Didn't.
	 */
	bail1:
	
	brelse4(&qbh);
	
	bail:

	hpfs_unlock_inode(dir);
	iput(dir);
	return -ENOENT;
}

