(*******************************************************************
 *
 *  TTRaster.Pas                                              v 1.2
 *
 *  The FreeType glyph rasterizer.
 *
 *  Copyright 1996 David Turner, Robert Wilhelm and Werner Lemberg
 *
 *  This file is part of the FreeType project, and may only be used
 *  modified and distributed under the terms of the FreeType project
 *  license, LICENSE.TXT. By continuing to use, modify or distribute
 *  this file you indicate that you have read the license and
 *  understand and accept it fully.
 *
 *  NOTES : This version supports the following :
 *
 *    - direct grayscaling
 *    - sub-banding
 *    - drop-out modes 4 and 5
 *    - second pass for complete drop-out control ( bitmap only )
 *    - variable precision
 *
 *   Re-entrancy is _not_ planned.
 *
 *   Changes between 1.1 and 1.2 :
 *
 *     - no more trace tables, now uses linked list to sort
 *       coordinates.
 *
 *     - reduced code size using function dispatch within a generic
 *       draw_sweep function.
 *
 *     - added variable precision for finer rendering at small ppems
 *
 *
 *   Note that its interface may change in the future.
 *
 ******************************************************************)

Unit TTRASTER;

interface

uses
{$IFDEF OS2}
     Use32,
{$ENDIF}
     FreeType,
     TTTypes;

{ $DEFINE ASSERT}   (* Check assertions during debugging *)

{ $DEFINE DEBUG}   (* Displays fatal errors with Writeln *)

{ $DEFINE DEBUG3}  (* Pset pixels *during* arc computations, you must *)
                   (* activate DEBUG as well for this one to compile  *)

{$DEFINE REVERSE} (* Allows the rendering of 'illegal' glyphs like  *)
                  (* those found in "CocaCola.ttf" ...              *)

{ $DEFINE CONST_PREC} (* Uses a fixed 6 bits precision for all coordinates *)

{ $DEFINE SMOOTH}  (* To use "smooth-levels" rendering (16 levels) *)

const

  (* Flow constants :                                           *)
  (*                                                            *)
  (* The flow of a bitmap refers to the way lines are oriented  *)
  (* within the bitmap data, i.e. the orientation of the Y      *)
  (* coordinate axis.                                           *)
  (*                                                            *)

  TTFlowDown  = -1; (* Bitmap is oriented from top to bottom  *)
  TTFlowUp    =  1; (* Bitmap is oriented from bottom to top  *)
  TTFlowError =  0; (* Indicates that an error occured during *)
                    (* computation                            *)

  Err_Ras_None       =  0;
  Err_Ras_NotIni     = -2;  (* Rasterizer not Initialized    *)
  Err_Ras_Overflow   = -3;  (* Profile Table Overflow        *)
  Err_Ras_Neg_H      = -4;  (* Negative Height encountered ! *)
  Err_Ras_Invalid    = -5;  (* Invalid value encountered !   *)

type

  (* This structure is used to describe a BitMap or PixMap to the   *)
  (* rasterizer                                                     *)

  PRasterBlock = ^TRasterBlock;
  TRasterBlock = record
                   Rows   : longint;      (* rows number of the bitmap    *)
                   Cols   : longint;      (* columns (bytes) per row      *)
                   Width  : longint;      (* pixels per row               *)
                   Flow   : longint;      (* bit/pixmap's flow            *)
                   Buffer : Pointer;      (* bit/pixmap data              *)
                   Size   : longint;      (* bit/pixmap data size (bytes) *)
                 end;

  (* This structure is a parameter block for each Render_ call *)

  PGlyphRecord = ^TGlyphRecord;
  TGlyphRecord = record
                   numConts  : longint;         (* Contours nymber in glyph *)
                   endPoints : PUShort;         (* Contours end points      *)
                   Points    : longint;         (* points number            *)
                   XCoord    : TT_PCoordinates; (* x coordinates table      *)
                   YCoord    : TT_PCoordinates; (* y coordinates table      *)
                   Flag      : TT_PTouchTable;  (* flags table              *)
                  end;

var
  Raster_Error : Integer;
  (* Global raster error variable *)

function Init_Rasterizer( profBuffer   : Pointer;
                          profSize     : longint;
                          grayBuffer   : Pointer;
                          grayLength   : integer  ) : longint;

  (* Initializes the rasterizer, specifying the render pool as   *)
  (* well as the gray lines buffer ( put NULL if gray-levels     *)
  (* rasterization is not needed by your application )           *)

function Render_Glyph( var glyph  : TGlyphRecord;
                           target : PRasterBlock;
                           scan   : byte ) : boolean;

  (* Render one glyph in the target bitmap, using drop-out control *)
  (* mode 'scan'                                                   *)

function Render_Gray_Glyph( var glyph   : TGlyphRecord;
                                target  : PRasterBlock;
                                scan    : byte;
                                palette : pointer      ) : boolean;

  (* Render one gray-level glyph in the target pixmap              *)
  (* palette points to an array of 5 colors used for the rendering *)
  (* use nil to reuse the last palette. Default is VGA graylevels  *)

{$IFDEF SMOOTH}
function Render_Smooth_Glyph( var glyph   : TGlyphRecord;
                                  target  : PRasterBlock;
                                  scan    : Byte;
                                  palette : pointer      ) : boolean;
{$ENDIF}

procedure Set_High_Precision( High : boolean );

  (* Set rendering precision. Should be set to TRUE for small sizes only *)
  (* ( typically < 20 ppem )                                             *)

implementation

uses
     TTCalc        { used for MulDiv }
{$IFDEF DEBUG}
     ,FullScr      { Used to access VRAM pointer VIO during DEBUG }
{$ENDIF}
     ;


const
  MaxBezier  = 32;       (* Maximum number of stacked Bziers.    *)
                         (* Setting this constant to more than 32 *)
                         (* is a pure waste of space              *)

  Pixel_Bits = 6;        (* fractional bits of input coordinates  *)

type

  TEtats  = ( Indetermine, Ascendant, Descendant, Plat );

  PProfile = ^TProfile;
  TProfile = record
               Flow    : Int;       (* ascending or descending Profile *)
               Height  : Int;       (* Profile's height in scanlines   *)
               Start   : Int;       (* Profile's starting scanline     *)
               Offset  : ULong;     (* offset of first coordinate in   *)
                                    (* render pool                     *)

               Link    : PProfile;  (* link used in several cases      *)

               X       : Longint;   (* current coordinate during sweep *)
               CountL  : Int;       (* number of lines to step before  *)
                                    (* this Profile becomes drawable   *)

               next    : PProfile; (* next Profile of the same contour *)
             end;

  TProfileList = record            (* simple record to manage linked lists *)
                   Head : TProfile;
                   Tail : TProfile;
                 end;
  PProfileList = ^TProfileList;

  (* I use the classic trick of two dummy records for the head and tail  *)
  (* of a linked list, this reduces tests in insertion/deletion/sorting  *)
  (* NOTE :  used during sweeps only                                     *)

  TBand = record
            Y_Min : Int;
            Y_Max : Int;
          end;

  (* Simple record used to implement a stack of bands, required *)
  (* by the sub-banding mechanism                               *)

const
  AlignProfileSize = ( sizeOf(TProfile) + 3 ) div 4;
  (* You may need to compute this according to your prefered alignement *)

  LMask : array[0..7] of Byte
        = ($FF,$7F,$3F,$1F,$0F,$07,$03,$01);

  RMask : array[0..7] of Byte
        = ($80,$C0,$E0,$F0,$F8,$FC,$FE,$FF);

  (* left and right fill bitmasks *)

type
  Function_Sweep_Init = procedure( var min, max : Int );

  Function_Sweep_Span = procedure( y     : Int;
                                   x1    : TT_F26dot6;
                                   x2    : TT_F26dot6;
                                   Left  : PProfile;
                                   Right : PProfile );

  Function_Sweep_Step = procedure;

  (* prototypes used for sweep function dispatch *)

{$IFNDEF CONST_PREC}

var
  Precision_Bits  : Int;       (* Fractional bits of Raster coordinates *)
  Precision       : Int;
  Precision_Half  : Int;
  Precision_Step  : Int;       (* Bezier subdivision minimal step       *)
  Precision_Shift : Int;       (* Shift used to convert coordinates     *)
  Precision_Mask  : Longint;   (* integer truncatoin mask               *)

{$ELSE}

const
  Precision_Bits  = 6;
  Precision       = 1 shl Precision_Bits;
  Precision_Half  = Precision div 2;
  Precision_Step  = Precision_Half;
  Precision_Shift = 0;
  Precision_Mask  = -Precision;

{$ENDIF}

var
  cProfile  : PProfile;  (* current Profile                *)
  fProfile  : PProfile;  (* head of Profiles linked list   *)
  oProfile  : PProfile;  (* old Profile                    *)
  gProfile  : PProfile;  (* last Profile in case of impact *)

  nProfs   : Int;        (* current number of Profiles *)

  Etat     : TEtats;     (* State of current trace *)

  Fresh    : Boolean;    (* Indicates a new Profile which 'Start' field *)
                         (* must be set                                 *)

  Joint    : Boolean;    (* Indicates that the last arc stopped sharp *)
                         (* on a scan-line. Important to get rid of   *)
                         (* doublets                                  *)

  Buff     : PStorage;   (* Profiles buffer a.k.a. Render Pool *)
  MaxBuff  : ULong;      (* current render pool's size         *)
  profCur  : ULong;      (* current render pool cursor         *)

  Cible      : TRasterBlock; (* Description of target map *)

  BWidth     : integer;
  BCible     : PByte;   (* target bitmap buffer *)
  GCible     : PByte;   (* target pixmap buffer *)

  Band_Stack : array[1..16] of TBand;
  Band_Top   : Int;

  TraceOfs : Int;      (* current offset in target bitmap  *)
  TraceG   : Int;      (* current offset in targer pixmap  *)

  gray_min_x : Int;    (* current min x during gray rendering *)
  gray_max_x : Int;    (* current max x during gray rendering *)

  (* Dispatch variables : *)

  Proc_Sweep_Init : Function_Sweep_Init;  (* Sweep initialisation *)
  Proc_Sweep_Span : Function_Sweep_Span;  (* Span drawing         *)
  Proc_Sweep_Drop : Function_Sweep_Span;  (* Drop out control     *)
  Proc_Sweep_Step : Function_Sweep_Step;  (* Sweep line step      *)

  Arcs     : Array[0..2*MaxBezier] of
              record                   (* A stack of points used to     *)
               X, Y : LongInt          (* store and recursively split   *)
              end;                     (* our Bzier arcs.              *)

  CurArc   : Int;                      (* stack's top                   *)

  XCoord,                     (* current x coordinates array *)
  YCoord   : TT_PCoordinates; (* current y coordinates array *)

  Flags    : PByte;   (* current flags array     *)
  Outs     : PUShort; (* current endpoints array *)

  nPoints,            (* current number of points   *)
  nContours : Int;    (* current number of contours *)

  LastX,              (* Last and extrema coordinates during *)
  LastY,              (* rendering                           *)
  MinY,
  MaxY     : LongInt;

  DropOutControl : Byte;  (* current drop-out control mode *)

  Count_Table : array[0..255] of Word;
  (* Look-up table used to quickly count set bits in a gray 2x2 cell *)

  Count_Table2 : array[0..255] of Word;
  (* Look-up table used to quickly count set bits in a gray 2x2 cell *)

  Grays : array[0..4] of Byte;
  (* gray palette used during gray-levels rendering *)
  (* 0 : background .. 4 : foreground               *)

  Gray_Lines  : PByte;   { 2 intermediate bitmap lines         }
  Gray_Width  : integer; { width of the 'gray' lines in pixels }

{$IFDEF SMOOTH}
  Smooth_Cols : integer;
  Smooths : array[0..16] of Byte;
  (* smooth palette used during smooth-levels rendering *)
  (* 0 : background...16 : foreground                   *)

  smooth_pass : integer;
{$ENDIF}

  Second_Pass : boolean;
  (* indicates wether an horizontal pass should be performed  *)
  (* to control drop-out accurately when calling Render_Glyph *)
  (* Note that there is no horizontal pass during gray render *)

  (* better set it off at ppem >= 18                          *)


{$IFDEF DEBUG3}
(****************************************************************************)
(*                                                                          *)
(* Function:     Pset                                                       *)
(*                                                                          *)
(* Description:  Used only in the "DEBUG3" state.                           *)
(*                                                                          *)
(*               This procedure simply plots a point on the video screen    *)
(*               Note that it relies on the value of cProfile->start,       *)
(*               which may sometimes not be set yet when Pset is called.    *)
(*               This will usually result in a dot plotted on the first     *)
(*               screen scanline ( far away its original position ).        *)
(*                                                                          *)
(*               This "bug" means not that the current implementation is    *)
(*               buggy, as the bitmap will be rendered correctly, so don't  *)
(*               panic if you see 'flying' dots in debugging mode           *)
(*                                                                          *)
(*                                                                          *)
(* Input:        None                                                       *)
(*                                                                          *)
(* Returns:      Nada                                                       *)
(*                                                                          *)
(****************************************************************************)

procedure PSet;
var c  : byte;
    o  : Int;
    xz : LongInt;
begin
  xz := Buff^[profCur] div Precision;

  with cProfile^ do
   begin

    case Flow of
      TTFlowUp   : o := 80 * (profCur-Offset+Start) + xz div 8;
      TTFlowDown : o := 80 * (Start-profCur+offset) + xz div 8;
     end;

    if o > 0 then
     begin
      c := Vio^[o] or ( $80 shr ( xz and 7 ));
      Vio^[o] := c;
     end
   end;
end;

(****************************************************************************)
(*                                                                          *)
(* Function:    Clear_Band                                                  *)
(*                                                                          *)
(* Description: Clears a Band on screen during DEBUG3 rendering             *)
(*                                                                          *)
(* Input:       y1, y2   top and bottom of screen-wide band                 *)
(*                                                                          *)
(* Returns:     Nada.                                                       *)
(*                                                                          *)
(****************************************************************************)

procedure ClearBand( y1, y2 : Int );
var
  Y : Int;
  K : Word;
begin
  K := y1*80;
  FillChar( Vio^[k], (y2-y1+1)*80, 0 );
end;
{$ENDIF}

{$IFNDEF CONST_PREC}

(****************************************************************************)
(*                                                                          *)
(* Function:    Set_High_Precision                                          *)
(*                                                                          *)
(* Description: Sets precision variables according to param flag            *)
(*                                                                          *)
(* Input:       High     set to True for high precision ( typically for     *)
(*                       ppem < 18 ), false otherwise.                      *)
(*                                                                          *)
(****************************************************************************)

procedure Set_High_Precision( High : boolean );
begin
  second_pass := High;

  if High then
    begin
      Precision_Bits := 10;
      Precision_Step := 128;
    end
  else
    begin
      Precision_Bits := 6;
      Precision_Step := 32;
    end;

  Precision       := 1 shl Precision_Bits;
  Precision_Half  := Precision shr 1;
  Precision_Shift := Precision_Bits - Pixel_Bits;
  Precision_Mask  := -Precision;
end;

{$ENDIF}

(****************************************************************************)
(*                                                                          *)
(* Function:    New_Profile                                                 *)
(*                                                                          *)
(* Description: Creates a new Profile in the render pool                    *)
(*                                                                          *)
(* Input:       AEtat    state/orientation of the new Profile               *)
(*                                                                          *)
(* Returns:     True on sucess                                              *)
(*              False in case of overflow or of incoherent Profile          *)
(*                                                                          *)
(****************************************************************************)

function New_Profile( AEtat : TEtats ) : boolean;
begin

  if fProfile = NIL then
    begin
      cProfile := PProfile( @Buff^[profCur] );
      fProfile := cProfile;
      inc( profCur, AlignProfileSize );
    end;

  if profCur >= MaxBuff then
    begin
      Raster_Error := Err_Ras_Overflow;
      New_Profile  := False;
      exit;
    end;

  with cProfile^ do
    begin

      Case AEtat of

        Ascendant  : Flow := TTFlowUp;
        Descendant : Flow := TTFlowDown;
      else
{$IFDEF DEBUG}
        Writeln('ERROR : Incoherent Profile' );
        Halt(30);
{$ELSE}
        New_Profile  := False;
        Raster_Error := Err_Ras_Invalid;
        exit;
{$ENDIF}
      end;

      Start   := 0;
      Height  := 0;
      Offset  := profCur;
      Link    := nil;
      next   := nil;
    end;

  if gProfile = nil then gProfile := cProfile;

  Etat  := AEtat;
  Fresh := True;
  Joint := False;

  New_Profile := True;
end;

(****************************************************************************)
(*                                                                          *)
(* Function:    End_Profile                                                 *)
(*                                                                          *)
(* Description: Finalizes the current Profile.                              *)
(*                                                                          *)
(* Input:       None                                                        *)
(*                                                                          *)
(* Returns:     True on success                                             *)
(*              False on overflow or incoherency.                           *)
(*                                                                          *)
(****************************************************************************)

function End_Profile : boolean;
var
  H          : Int;
  oldProfile : PProfile;
begin
  H := profCur - cProfile^.Offset;

  if H < 0 then
    begin
      End_Profile  := False;
      Raster_Error := Err_Ras_Neg_H;
      exit;
    end;

  if H > 0 then
    begin
      oldProfile       := cProfile;
      cProfile^.Height := H;
      cProfile         := PProfile( @Buff^[profCur] );

      inc( profCur, AlignProfileSize );

      cProfile^.Height  := 0;
      cProfile^.Offset  := profCur;
      oldProfile^.next := cProfile;
      inc( nProfs );
    end;

  if profCur >= MaxBuff then
    begin
      End_Profile  := False;
      Raster_Error := Err_Ras_Overflow;
      exit;
    end;

  Joint := False;

  End_Profile := True;
end;

(****************************************************************************)
(*                                                                          *)
(* Function:    Finalize_Profile_Table                                      *)
(*                                                                          *)
(* Description: Adjusts all links in the Profiles list                      *)
(*                                                                          *)
(* Input:       None                                                        *)
(*                                                                          *)
(* Returns:     Nada                                                        *)
(*                                                                          *)
(****************************************************************************)

procedure Finalize_Profile_Table;
var
  n : int;
  p : PProfile;
begin

  n := nProfs;

  if n > 1 then
    begin

      P := fProfile;

      while n > 1 do with P^ do
        begin
          Link := PProfile( @Buff^[ Offset + Height ] );
          P    := Link;

          dec( n );
        end;

      P^.Link := nil;

    end
  else
    fProfile := nil;

end;

(****************************************************************************)
(*                                                                          *)
(* Function:    Split_Bezier                                                *)
(*                                                                          *)
(* Description: Subdivises one Bezier arc into two joint                    *)
(*              sub-arcs in the Bezier stack.                               *)
(*                                                                          *)
(* Input:       None ( subdivised bezier is taken from the top of the       *)
(*              stack )                                                     *)
(*                                                                          *)
(* Returns:     Nada                                                        *)
(*                                                                          *)
(****************************************************************************)

procedure Split_Bezier;
var
  X1, Y1, X2, Y2 : LongInt;
begin
  with Arcs[CurArc+2] do begin x1 := x; y1 := y; end;
  with Arcs[CurArc]   do begin x2 := x; y2 := y; end;

  with Arcs[CurArc+4] do begin x := x1; y := y1; end;
  with Arcs[CurArc+1] do
   begin
    inc(x1,x); inc(y1,y);
    inc(x2,x); inc(y2,y);
   end;

  with Arcs[CurArc+3] do begin x := (x1+1) div 2; y := (y1+1) div 2; end;
  with Arcs[CurArc+1] do begin x := (x2+1) div 2; y := (y2+1) div 2; end;
  with Arcs[CurArc+2] do
   begin
    x:=( x1+x2+2 ) div 4;
    y:=( y1+y2+2 ) div 4;
   end;

  Inc( CurArc,2);
end;

(****************************************************************************)
(*                                                                          *)
(* Function:    Push_Bezier                                                 *)
(*                                                                          *)
(* Description: Clears the Bezier stack and pushes a new Arc on top of it.  *)
(*                                                                          *)
(* Input:       x1,y1 x2,y2 x3,y3  new Bezier arc                           *)
(*                                                                          *)
(* Returns:     nada                                                        *)
(*                                                                          *)
(****************************************************************************)

procedure PushBezier( x1, y1, x2, y2, x3, y3 : LongInt );
begin
  curArc:=0;

  with Arcs[CurArc+2] do begin x:=x1; y:=y1; end;
  with Arcs[CurArc+1] do begin x:=x2; y:=y2; end;
  with Arcs[ CurArc ] do begin x:=x3; y:=y3; end;
end;

(****************************************************************************)
(*                                                                          *)
(* Function:    Line_Up                                                     *)
(*                                                                          *)
(* Description: Compute the x-coordinates of an ascending line segment      *)
(*              and stores them in the render pool.                         *)
(*                                                                          *)
(* Input:       x1,y1 x2,y2  Segment start (x1,y1) and end (x2,y2) points   *)
(*                                                                          *)
(* Returns:     True on success                                             *)
(*              False if Render Pool overflow.                              *)
(*                                                                          *)
(****************************************************************************)

function Line_Up( x1, y1, x2, y2 : LongInt ) : boolean;
var
  Dx, Dy               : LongInt;
  e1, e2, f1, f2, size : Int;
  Ix, Rx, Ax           : LongInt;
begin
  Line_Up := True;

  Dx := x2-x1; Dy := y2-y1;

  if (Dy <= 0) or (y2 < MinY) or (y1 > MaxY) then exit;

  if y1 < MinY then
   begin
    x1 := x1 + MulDiv( Dx, MinY-y1, Dy );
    e1 := MinY div Precision;
    f1 := 0;
   end
  else
   begin
    e1 := y1 div Precision;
    f1 := y1 and (Precision-1);
   end;

  if y2 > MaxY then
   begin
    (* x2 := x2 + MulDiv( Dx, MaxY-y2, Dy ); *)
    e2 := MaxY div Precision;
    f2 := 0;
   end
  else
   begin
    e2 := y2 div Precision;
    f2 := y2 and (Precision-1);
   end;

  if f1 > 0 then
    if e1 <> e2 then inc( e1 )
                else exit
  else
    if Joint then dec( profCur );

  Joint := (f2 = 0);

  (* Indicates that the segment stopped sharp on a ScanLine *)

  if Fresh then
   begin
    cProfile^.Start := e1;
    Fresh           := False;
   end;

  size := ( e2-e1 )+1;
  if ( profCur + size >= MaxBuff ) then
   begin
     Line_Up      := False;
     Raster_Error := Err_Ras_Overflow;
     exit;
   end;

  if Dx = 0 then
    begin

      while size > 0 do
      begin
       Buff^[profCur] := x1; {$IFDEF DEBUG3} Pset; {$ENDIF}
       inc( profCur );
       dec( size );
      end;

    end
  else

    if Dx > 0 then
      begin

       Ix := (Precision*Dx) div Dy;
       Rx := (Precision*Dx) mod Dy;
       Ax := 0;

       if f1 > 0 then
       begin
         inc( x1, ((Precision-f1)*Dx) div Dy );
         dec( Ax, ((Precision-f1)*Dx) mod Dy );
         if Ax < 0 then
           begin
             inc( x1 );
             inc( Ax, Dy );
           end;
       end;

       while size > 0 do
       begin
         Buff^[profCur] := x1; {$IFDEF DEBUG3} Pset; {$ENDIF}
         inc( profCur );

         inc( x1, Ix );
         dec( Ax, Rx );
         if Ax < 0 then
           begin
             inc( x1 );
             inc( Ax, Dy );
           end;
         dec( size );
       end;

      end
    else
      begin

        Dx := -Dx;
        Ix := (Precision*Dx) div Dy;
        Rx := (Precision*Dx) mod Dy;
        Ax := Dy-1;

        if f1 > 0 then
        begin
          dec( x1, ((Precision-f1)*Dx) div Dy );
          dec( Ax, ((Precision-f1)*Dx) div Dy );
          if Ax < 0 then
            begin
              dec( x1 );
              inc( Ax, Dy );
            end;
        end;

        while size > 0 do
        begin
          Buff^[profCur] := x1; {$IFDEF DEBUG3} Pset; {$ENDIF}
          inc( profCur );

          dec( x1, Ix );
          dec( Ax, Rx );
          if Ax < 0 then
            begin
              dec( x1 );
              inc( Ax, Dy );
            end;
          dec( size );
        end;
    end;

end;

(****************************************************************************)
(*                                                                          *)
(* Function:    Line_Down                                                   *)
(*                                                                          *)
(* Description: Compute the x-coordinates of a descending line segment      *)
(*              and stores them in the render pool.                         *)
(*                                                                          *)
(* Input:       x1,y1 x2,y2  Segment start (x1,y1) and end (x2,y2) points   *)
(*                                                                          *)
(* Returns:     True on success                                             *)
(*              False if Render Pool overflow.                              *)
(*                                                                          *)
(****************************************************************************)

function Line_Down( x1, y1, x2, y2 : LongInt ): boolean;
var
  Dx, Dy                : LongInt;
  e1, e2, f1, f2, size  : Int;

  Ix, Rx, Ax            : LongInt;
begin
  Line_Down := True;

  Dx := x2-x1; Dy := y2-y1;

  if (Dy >= 0) or (y1 < MinY) or (y2 > MaxY) then exit;

  if y1 > MaxY then
   begin
    x1 := x1 + MulDiv( Dx, MaxY-y1, Dy );
    e1 := MaxY div Precision;
    f1 := 0;
   end
  else
   begin
    e1:= y1 div Precision;
    f1:= y1 and (Precision-1);
   end;

  if y2 < MinY then
   begin
    (* x2 := x2 + MulDiv( Dx, MinY-y2, Dy ); *)
    e2 := MinY div Precision;
    f2 := 0;
   end
  else
   begin
    e2 := y2 div Precision;
    f2 := y2 and (Precision-1);
   end;

  if (f1 = 0) and Joint then
  begin
    dec( profCur );
    Joint := false;
  end;

  if f2>0 then
    if e2 <> e1 then inc( e2 )
                else exit
  else
    Joint:=True;

  (* Indicates that the segment stopped sharp on a scanline. *)

  If Fresh then
   begin
    cProfile^.Start := e1;
    Fresh           := False;
   end;

  size := ( e1-e2 )+1;
  if ( profCur + size >= MaxBuff ) then
   begin
     Line_Down := False;
     Raster_Error  := Err_Ras_Overflow;
     exit;
   end;

  Dy := -Dy;

  if Dx = 0 then
    begin

      while size > 0 do
      begin
        Buff^[profCur] := x1; {$IFDEF DEBUG3} Pset; {$ENDIF}
        inc( profCur );
        dec( size );
      end;

    end
  else
    if Dx > 0 then
    begin

      Ix := (Dx*Precision) div Dy;
      Rx := (Dx*Precision) mod Dy;
      Ax := 0;

      if f1 > 0 then
      begin
        inc( x1, (f1*Dx) div Dy );
        dec( Ax, (f1*Dx) mod Dy );
        if Ax < 0 then
          begin
            inc( x1 );
            inc( Ax, Dy );
          end;
      end;

      while size > 0 do
      begin
        Buff^[profCur] := x1; {$IFDEF DEBUG3} Pset; {$ENDIF}
        inc( profCur );
        inc( x1, Ix );
        dec( Ax, Rx );

        if Ax < 0 then
          begin
            inc( Ax, Dy );
            inc( x1 );
          end;

        dec( size );
      end;

    end
  else
    begin

      Dx := -Dx;

      Ix := (Dx*Precision) div Dy;
      Rx := (Dx*Precision) mod Dy;
      Ax := Dy-1;

      if f1 > 0 then
      begin
        dec( x1, (f1*Dx) div Dy );
        dec( Ax, (f1*Dx) mod Dy );
        if Ax < 0 then
          begin
            dec( x1 );
            inc( Ax, Dy );
          end;
      end;

      while size > 0 do
      begin
        Buff^[profCur] := x1; {$IFDEF DEBUG3} Pset; {$ENDIF}
        inc( profCur );
        dec( x1, Ix );
        dec( Ax, Rx );

        if Ax < 0 then
          begin
            inc( Ax, Dy );
            dec( x1 );
          end;

        dec( size );
      end;

    end;

end;


(****************************************************************************)
(*                                                                          *)
(* Function:    Bezier_Up                                                   *)
(*                                                                          *)
(* Description: Compute the x-coordinates of an ascending bezier arc        *)
(*              and stores them in the render pool.                         *)
(*                                                                          *)
(* Input:       None.The arc is taken from the top of the Bezier stack.     *)
(*                                                                          *)
(* Returns:     True on success                                             *)
(*              False if Render Pool overflow.                              *)
(*                                                                          *)
(****************************************************************************)

function Bezier_Up : boolean;
var
  x1, y1, x2, y2, e, e2, e0 : LongInt;
  debArc, f1                : Int;

begin
  Bezier_Up := True;

  y1 := Arcs[curArc+2].y;
  y2 := Arcs[ curArc ].y;

  if ( y2 < MinY ) or ( y1 > MaxY ) then
   begin
    dec( curArc,2 );
    exit;
   end;

  e2 := y2 and Precision_Mask;

  if e2 > MaxY then e2 := MaxY;

  e0 := MinY;

  if y1 < MinY then e := MinY
  else
   begin
    e  := (y1+Precision-1) and Precision_Mask;
    f1 := y1 and (Precision-1);
    e0 := e;

    if f1 = 0 then
     begin

      if Joint then begin dec(profCur); Joint:=False; end;
      (* ^ Ce test permet d'viter les doublons *)

      Buff^[profCur] := Arcs[curArc+2].x;
      {$IFDEF DEBUG3} Pset; {$ENDIF}
      inc( profCur );

      (* Remarque au sujet du dbordement de table :     *)
      (*                                                 *)
      (*  Nous savons dj que profCur < MaxBuff, il     *)
      (*  y a donc la place pour au moins 1 ordonne     *)
      (*  et nous n'avons pas besoin de faire le test    *)
      (*  ici !                                          *)
      (*                                                 *)

      (* Note on table overflow :                        *)
      (*                                                 *)
      (* We already know that profCur < MaxBuff here,    *)
      (* so there is room for at least 1 coordinate      *)
      (* and we don't need to test overflow there !      *)
      (*                                                 *)

      inc( e, Precision );

     end
   end;

  if Fresh then
   begin
    cProfile^.Start := e0 div Precision;
    Fresh := False;
   end;

  (* overflow ? *)
  if ( profCur + (e2-e) div Precision + 1 >= MaxBuff ) then
    begin
      Bezier_Up := False;
      Raster_Error := Err_Ras_Overflow;
      exit;
    end;

  debArc := curArc;

  while ( curArc >= debArc ) and ( e <= e2 ) do
   begin
    Joint := False;

    y2 := Arcs[CurArc].y;

    if y2 = e then

     begin

      Joint := True;

      Buff^[profCur] := Arcs[curArc].x;
      {$IFDEF DEBUG3} Pset; {$ENDIF}
      inc( profCur );

      inc( e, Precision );
      dec( curArc, 2 );
     end

    else
     if y2 < e then dec( curArc, 2 )

    else
     begin

      y1 := Arcs[curArc+2].y;

      if y2-y1 < Precision_Step then

       begin

        x1 := Arcs[curArc+2].x;
        x2 := Arcs[ curArc ].x;

        dec( y2, y1 );

        Buff^[profCur] := x1 + ( (x2-x1)*(e-y1) + y2 div 2) div y2;
        {$IFDEF DEBUG3} Pset; {$ENDIF}
        inc( profCur );

        dec( curArc, 2 );
        inc( e, Precision );
       end

      else
        Split_Bezier;

        (* Note : Inlining the split_bezier function seems to  *)
        (*        increase speed by such a little amount that  *)
        (*        I kept the procedure call there              *)

     end;
   end;

  curArc := debArc-2;

end;


(****************************************************************************)
(*                                                                          *)
(* Function:    Bezier_Down                                                 *)
(*                                                                          *)
(* Description: Compute the x-coordinates of a descending bezier arc        *)
(*              and stores them in the render pool.                         *)
(*                                                                          *)
(* Input:       None. Arc is taken from the top of the Bezier stack.        *)
(*                                                                          *)
(* Returns:     True on success                                             *)
(*              False if Render Pool overflow.                              *)
(*                                                                          *)
(****************************************************************************)

function Bezier_Down : boolean;
var
  x1, y1, x2, y2, e, e0, e2 : LongInt;
  f1, debArc                : Int;

begin
  Bezier_Down := True;

  y1 := Arcs[curArc+2].y;
  y2 := Arcs[ curArc ].y;

  if ( y1 < MinY ) or ( y2 > MaxY ) then
   begin
    dec( curArc,2 );
    exit;
   end;

  e2 := (y2+Precision-1) and Precision_Mask;

  if e2 < MinY then e2 := MinY;

  e0 := MaxY;

  if y1 > MaxY then e := MaxY
  else
   begin
    e  := y1 and Precision_Mask;
    f1 := y1 and (Precision-1);
    e0 := e;

    if f1 = 0 then

     begin

      if Joint then begin dec( profCur ); Joint:=False; end;
      (* ^ Ce test permet d'viter les doublons *)

      Buff^[profCur] := Arcs[curArc+2].x;
      {$IFDEF DEBUG3} Pset; {$ENDIF}
      inc( profCur );

      (* Remarque au sujet du dbordement de table :     *)
      (*                                                 *)
      (*  Nous savons dj que profCur < MaxBuff, il     *)
      (*  y a donc la place pour au moins 1 ordonne     *)
      (*  et nous n'avons pas besoin de faire le test    *)
      (*  ici !                                          *)
      (*                                                 *)

      (* Note on table overflow :                        *)
      (*                                                 *)
      (* We already know that profCur < MaxBuff here,    *)
      (* so there is room for at least 1 coordinate      *)
      (* and we don't need to test overflow there !      *)
      (*                                                 *)

      dec( e, Precision );
     end
   end;

  if Fresh then
   begin
    cProfile^.Start := e0 div Precision;
    Fresh := False;
   end;

  if ( profCur + (e - e2) div Precision + 1 >= MaxBuff ) then
    begin
      Raster_Error   := Err_Ras_Overflow;
      Bezier_Down := False;
      exit;
    end;

  debArc := curArc;

  while ( curArc >= debArc ) and ( e >= e2 ) do
   begin
    Joint:=False;

    y2 := Arcs[CurArc].y;

    if y2 = e then

     begin
      Joint:=True;

      Buff^[profCur] := Arcs[curArc].x;
      {$IFDEF DEBUG3} Pset; {$ENDIF}
      inc( profCur );

      dec( e, Precision );
      dec( curArc,2 );
     end

    else
     if y2 > e then dec( curArc,2 )

    else
     begin
      y1 := Arcs[curArc+2].y;

      if (y1-y2)< Precision_Step then
       begin
        x1 := Arcs[curArc+2].x;
        x2 := Arcs[ curArc ].x;

        Buff^[profCur] := x1 + ( (x2-x1)*(e-y1) div (y2-y1) );
        {$IFDEF DEBUG3} Pset; {$ENDIF}
        inc( profCur );

        dec( curArc,2 );
        dec( e, Precision );
       end

      else
        Split_Bezier;

        (* Note : Inlining the split_bezier function seems to  *)
        (*        increase speed by such a little amount that  *)
        (*        I kept the procedure call there              *)

     end;
   end;

 curArc := debArc-2;

end;


(****************************************************************************)
(*                                                                          *)
(* Function:    Line_To                                                     *)
(*                                                                          *)
(* Description: Injects a new line segment and adjust Profiles list.        *)
(*                                                                          *)
(* Input:       x, y : segment endpoint ( start point in LastX,LastY )      *)
(*                                                                          *)
(* Returns:     True on success                                             *)
(*              False if Render Pool overflow or Incorrect Profile          *)
(*                                                                          *)
(****************************************************************************)

function Line_To( x, y : LongInt ) : boolean;
begin
  Line_To := False;

  case Etat of

    Indetermine : if y > lastY then
                    if not New_Profile( Ascendant ) then exit else
                  else
                   if y < lastY then
                    if not New_Profile( Descendant ) then exit;

    Ascendant   : if y < lastY then
                   begin
                    if not End_Profile or
                       not New_Profile( Descendant ) then exit;
                   end;

    Descendant  : if y > LastY then
                   begin
                    if not End_Profile or
                       not New_Profile( Ascendant ) then exit;
                   end;
   end;

  Case Etat of
    Ascendant  : if not Line_Up  ( LastX, LastY, X, Y ) then exit;
    Descendant : if not Line_Down( LastX, LastY, X, Y ) then exit;
   end;

  LastX := x;
  LastY := y;

  Line_To := True;
end;

(****************************************************************************)
(*                                                                          *)
(* Function:    Bezier_To                                                   *)
(*                                                                          *)
(* Description: Injects a new bezier arc and adjust Profiles list.          *)
(*                                                                          *)
(* Input:       x,   y : arc endpoint ( start point in LastX, LastY )       *)
(*              Cx, Cy : control point                                      *)
(*                                                                          *)
(* Returns:     True on success                                             *)
(*              False if Render Pool overflow or Incorrect Profile          *)
(*                                                                          *)
(****************************************************************************)

function Bezier_To( x, y, Cx, Cy : LongInt ) : boolean;
var
  y1, y2, y3, x3 : LongInt;
  Etat_Bez       : TEtats;
begin
  Bezier_To := False;

  PushBezier( LastX, LastY, Cx, Cy, X, Y );

  while ( curArc>=0 ) do
   begin
    y1 := Arcs[curArc+2].y;
    y2 := Arcs[curArc+1].y;
    y3 := Arcs[curArc].y;
    x3 := Arcs[curArc].x;

    (* On dtermine l'tat du bzier courant *)

    if y1 = y2 then
     begin

      if y2 = y3 then Etat_Bez := Plat
      else
      if y2 > y3 then Etat_Bez := Descendant
      else
                      Etat_Bez := Ascendant;
     end

    else
    if y1 > y2 then
     begin

      if y2 >= y3 then Etat_Bez := Descendant
      else
                       Etat_Bez := Indetermine;
     end

    else
     begin

      if y2 <= y3 then Etat_Bez := Ascendant
      else
                       Etat_Bez := Indetermine;
     end;


    (* On agit en consquence *)

    case Etat_Bez of

      Plat        : dec( curArc, 2 );

      Indetermine : Split_Bezier;

    else

      if Etat <> Etat_Bez then
        begin

          if Etat <> Indetermine then
            if not End_Profile then exit;

          if not New_Profile( Etat_Bez ) then exit;

        end;

      case Etat of

        Ascendant  : if not Bezier_Up then exit;
        Descendant : if not Bezier_Down then exit;

      end;

    end;
   end;

  LastX := x3;
  LastY := y3;

  Bezier_To := True;
end;

(****************************************************************************)
(*                                                                          *)
(* Function:    Curve_To                                                    *)
(*                                                                          *)
(* Description: Injects several following Bezier arcs.                      *)
(*                                                                          *)
(* Input:       x,   y : arc endpoint ( start point in LastX, LastY )       *)
(*                                                                          *)
(*              firstCtrl, lastCtrlint : first and last control point       *)
(*                                       index.                             *)
(* Returns:     True on success                                             *)
(*              False if Render Pool overflow or Incorrect Profile          *)
(*                                                                          *)
(****************************************************************************)

function CurveTo( x, y : LongInt; FirstCtrl, LastCtrl : Int ) : boolean;
var
  xz, yz, cx, cy : LongInt;
begin

  CurveTo := False;

  xz := XCoord^[FirstCtrl] shl Precision_Shift;
  yz := YCoord^[FirstCtrl] shl Precision_Shift;

  inc( FirstCtrl );

  while FirstCtrl <= LastCtrl do
   begin

    cx := ( xz + XCoord^[FirstCtrl] shl Precision_Shift ) div 2;
    cy := ( yz + YCoord^[FirstCtrl] shl Precision_Shift ) div 2;

    if not Bezier_To( cx, cy, xz, yz ) then exit;

    xz := XCoord^[FirstCtrl] shl Precision_Shift;
    yz := YCoord^[FirstCtrl] shl Precision_Shift;

    inc( FirstCtrl );

   end;

  if not Bezier_To( x, y, xz, yz ) then exit;

  CurveTo := True;

end;

(****************************************************************************)
(*                                                                          *)
(* Function:    Convert_Glyph                                               *)
(*                                                                          *)
(* Description: Converts a glyph into a series of segments and arcs         *)
(*              and make a Profiles list with them.                         *)
(*                                                                          *)
(* Input:       _xCoord, _yCoord : coordinates tables.                      *)
(*                                                                          *)
(*              Uses the 'Flag' table too.                                  *)
(*                                                                          *)
(* Returns:     True on success                                             *)
(*              False if any error was encountered during render.           *)
(*                                                                          *)
(****************************************************************************)

Function Convert_Glyph( _xCoord, _yCoord : TT_PCoordinates ) : boolean;
var
  i, j, First, Last, Start : Int;

  y1, y2, y3 : LongInt;

  lastProfile : PProfile;

begin
  Convert_Glyph := False;

  j        := 0;
  fProfile := NIL;
  Joint    := False;
  Fresh    := False;

  XCoord := _XCoord;
  YCoord := _YCoord;

  cProfile         := PProfile( @Buff^[profCur] );
  cProfile^.Offset := profCur;
  nProfs           := 0;

  for i := 0 to nContours-1 do
   begin

    Etat     := Indetermine;
    First    := j;
    LastX    := xCoord^[j] shl Precision_Shift;
    LastY    := yCoord^[j] shl Precision_Shift;
    Start    := 0;
    gProfile := nil;

    inc(j);

    while j <= Outs^[i] do
     begin

      if Flags^[j] and 1 = 0 then  (* OFF Curve *)

        if Start = 0 then
          begin
           Start := j;
           Last  := j;
          end
         else
          inc( Last )

      else                     (* ON Curve *)
       if Start <> 0 then
        begin
         if not CurveTo( XCoord^[j] shl Precision_Shift,
                         YCoord^[j] shl Precision_Shift,
                         Start,
                         Last ) then exit;
         Start := 0;
        end
       else
         if not Line_To( XCoord^[j] shl Precision_Shift,
                         YCoord^[j] shl Precision_Shift ) then exit;

      inc( j );
     end;

    if Start <> 0 then
      if not CurveTo( XCoord^[First] shl Precision_Shift,
                      YCoord^[First] shl Precision_Shift,
                      Start,
                      Last )
          then exit else
     else
      if not Line_To( XCoord^[First] shl Precision_Shift,
                      YCoord^[First] shl Precision_Shift ) then exit;

    (* We _must_ take care of the case when the first and last arcs join  *)
    (* while having the same orientation                                  *)

    if ( lastY and (Precision-1) = 0 ) and
       ( lastY >= MinY ) and
       ( lastY <= MaxY ) then

      if ( gProfile <> nil ) and                  (* gProfile can be nil    *)
         ( gProfile^.Flow = cProfile^.Flow ) then  (* if the contour was    *)
                                                 (* too small to be drawn *)
           dec( profCur );

    lastProfile := cProfile;

    if not End_Profile then exit;

    if gProfile <> nil then lastProfile^.next := gProfile;

   end;

  Finalize_Profile_Table;

  Convert_Glyph := True;
end;


  (************************************************)
  (*                                              *)
  (*  Init_Linked                                 *)
  (*                                              *)
  (*    Init an empty linked list.                *)
  (*                                              *)
  (************************************************)

  procedure Init_Linked( var L : TProfileList );
  begin
    L.Head.Link := @L.Tail;
    L.Tail.Link := @L.Tail;

    L.Tail.X := $7FFFFFFF;
  end;

  (************************************************)
  (*                                              *)
  (*  InsNew :                                    *)
  (*                                              *)
  (*    Inserts a new Profile in a linked list.   *)
  (*                                              *)
  (************************************************)

  procedure InsNew( List    : PProfileList;
                    Profile : PProfile
                  );
  var
    Old, Current : PProfile;
  begin

    Old     := @List   ^.Head;
    Current :=  Old    ^.Link;

    repeat

      if Profile^.X < Current^.X then
      begin
        Profile^.Link := Current;
        old    ^.Link := Profile;
        exit;
      end;

      Old     := Current;
      Current := Current^.Link;

    until false;

  end;

  (************************************************)
  (*                                              *)
  (*  DelOld :                                    *)
  (*                                              *)
  (*    Removes an old Profile from a linked list *)
  (*                                              *)
  (************************************************)


  procedure DelOld( List    : PProfileList;
                    Profile : PProfile );
  var
    Old, Current, Next : PProfile;
  begin

    Old     := @List   ^.Head;
    Current :=  Old    ^.Link;
    Next    :=  Current^.Link;

    while Next <> Current do
    begin

      if current = Profile then
      begin
        old^.Link := Next;
        exit;
      end;

      Old     := Current;
      Current := Next;
      Next    := Current^.Link;
    end;

    (* we should never get there, unless the Profile was not part of *)
    (* the list                                                      *)

    {$IFDEF ASSERT}
    Writeln('(Raster:DelOld) Incoherent deletion');
    halt(9);
    {$ENDIF}
  end;

  (************************************************)
  (*                                              *)
  (*  Sort :                                      *)
  (*                                              *)
  (*    Sorts 'quickly' (??) a trace list.        *)
  (*                                              *)
  (************************************************)

  procedure Sort( List : PProfileList );
  var
    Old, Current, Nextp : PProfile;
  begin

    (* First, recompute coordinates *)

    Current := List^.Head.Link;
    Nextp   := Current^.Link;

    while current <> nextp do with current^ do
    begin
      X := Buff^[offset];
      inc( offset, flow );
      dec( height );

      current := nextp;
      nextp   := current^.Link;
    end;

    (* Then, do the sort *)

    Old     := @List   ^.Head;
    Current :=  Old    ^.Link;
    Nextp   :=  Current^.Link;

    while current <> nextp do
    begin

      if current^.x > nextp^.x then
        begin
          old    ^.link := nextp;
          current^.link := nextp^.link;
          nextp  ^.link := current;

          old     := @List ^.Head;
          current :=  old  ^.Link;
        end
      else
        begin
          old     := current;
          current := Nextp;
        end;

      nextp := current^.Link;
    end;

  end;



{$F+ Far calls are necessary for function pointers under BP7}
{    This flag is currently ignored by the Virtual Compiler }

(***********************************************************************)
(*                                                                     *)
(*  Vertical Sweep Procedure Set :                                     *)
(*                                                                     *)
(*  These three routines are used during the vertical black/white      *)
(*  sweep phase by the generic Draw_Sweep function.                    *)
(*                                                                     *)
(***********************************************************************)

procedure Vertical_Sweep_Init( var min, max : Int );
begin
  TraceOfs   := Cible.Cols * min;
  gray_min_x := 0;
  gray_max_x := 0;
end;



procedure Vertical_Sweep_Span( y     : Int;
                               x1,
                               x2    : TT_F26dot6;
                               Left,
                               Right : PProfile );
var
  e1, e2  : Longint;
  c1, c2  : Int;
  f1, f2  : Int;

  j : Int;
begin

  e1 := (( x1+Precision-1 ) and Precision_Mask) div Precision;
  e2 := ( x2 and Precision_Mask ) div Precision;

  if (e2 >= 0) and (e1 < BWidth) then

    begin

      if e1 < 0       then e1 := 0;
      if e2 >= BWidth then e2 := BWidth-1;

      c1 := e1 shr 3;
      c2 := e2 shr 3;

      f1 := e1 and 7;
      f2 := e2 and 7;

      if gray_min_X > c1 then gray_min_X := c1;
      if gray_max_X < c2 then gray_max_X := c2;

      j := TraceOfs + c1;

      if c1 = c2 then
        BCible^[j] := BCible^[j] or ( LMask[f1] and Rmask[f2] )

      else
       begin
         BCible^[j] := BCible^[j] or LMask[f1];

         if c2>c1+1 then
           FillChar( BCible^[j+1], c2-c1-1, $FF );

         inc( j, c2-c1 );

         BCible^[j] := BCible^[j] or RMask[f2];
       end
    end;

end;

procedure Vertical_Sweep_Drop( y     : Int;
                               x1,
                               x2    : TT_F26dot6;
                               Left,
                               Right : PProfile );
var
  e1, e2  : Longint;
  c1, c2  : Int;
  f1, f2  : Int;

  j : Int;
begin

  (* Drop-out control *)

  e1 := ( x1+Precision-1 ) and Precision_Mask;
  e2 := x2 and Precision_Mask;

  (* We are guaranteed that x2-x1 <= Precision here *)

  if e1 > e2 then
   if e1 = e2 + Precision then

    case DropOutControl of

      (* Drop-out Control Rule #3 *)
      1 : e1 := e2;

      4 : begin
            e1 := ((x1+x2+1) div 2 + Precision-1) and Precision_Mask;
            e2 := e1;
          end;

      (* Drop-out Control Rule #4 *)

      (* The spec is not very clear regarding rule #4. It       *)
      (* presents a method that is way too costly to implement  *)
      (* while the general idea seems to get rid of 'stubs'.    *)
      (*                                                        *)
      (* Here, we only get rid of stubs recognized when :       *)
      (*                                                        *)
      (*  upper stub :                                          *)
      (*                                                        *)
      (*   - P_Left and P_Right are in the same contour         *)
      (*   - P_Right is the successor of P_Left in that contour *)
      (*   - y is the top of P_Left and P_Right                 *)
      (*                                                        *)
      (*  lower stub :                                          *)
      (*                                                        *)
      (*   - P_Left and P_Right are in the same contour         *)
      (*   - P_Left is the successor of P_Right in that contour *)
      (*   - y is the bottom of P_Left                          *)
      (*                                                        *)

      2,5 : begin

            (* upper stub test *)

            if ( Left^.next = Right ) and
               ( Left^.Height <= 0 )  then exit;

            (* lower stub test *)

            if ( Right^.next = Left ) and
               ( Left^.Start = y   ) then exit;

            (* Check that the rightmost pixel is not already set *)

            e1 := e1 div Precision;

            c1 := e1 shr 3;
            f1 := e1 and 7;

            if ( e1 >= 0 ) and ( e1 < BWidth ) and
               ( BCible^[TraceOfs+c1] and ($80 shr f1) <> 0 ) then
              exit;

            case DropOutControl of
              2 : e1 := e2;
              5 : e1 := ((x1+x2+1) div 2 + Precision-1) and Precision_Mask;
            end;

            e2 := e1;

          end;
    else
      exit;  (* unsupported mode *)
    end

   else
  else
    e2 := e1;   (* when x1 = e1, x2 = e2, e2 = e1 + 64 *)

  e1 := e1 div Precision;

  if (e1 >= 0) and (e1 < BWidth) then
    begin

      c1 := e1 shr 3;
      f1 := e1 and 7;

      if gray_min_X > c1 then gray_min_X := c1;
      if gray_max_X < c1 then gray_max_X := c1;

      j := TraceOfs + c1;

      BCible^[j] := BCible^[j] or ($80 shr f1);
    end;
end;



procedure Vertical_Sweep_Step;
begin
  inc( TraceOfs, Cible.Cols );
end;


(***********************************************************************)
(*                                                                     *)
(*  Horizontal Sweep Procedure Set :                                   *)
(*                                                                     *)
(*  These three routines are used during the horizontal black/white    *)
(*  sweep phase by the generic Draw_Sweep function.                    *)
(*                                                                     *)
(***********************************************************************)

procedure Horizontal_Sweep_Init( var min, max : Int );
begin
  (* Nothing, really *)
end;


procedure Horizontal_Sweep_Span( y     : Int;
                                 x1,
                                 x2    : TT_F26dot6;
                                 Left,
                                 Right : PProfile );
var
  e1, e2  : Longint;
  c1, c2  : Int;
  f1, f2  : Int;

  j : Int;
begin

  e1 := ( x1+(Precision-1) ) and Precision_Mask;
  e2 := x2 and Precision_Mask;

  (* We are here guaranteed that x2-x1 > Precision *)

   c1 := y shr 3;
   f1 := y and 7;

   if (e1 >= 0) then
   begin
     e1 := e1 shr Precision_Bits;
     j  := c1 + e1*Cible.cols;
     if e1 < Cible.Rows then
       BCible^[j] := BCible^[j] or ($80 shr f1);
   end;

   if (e2 >= 0) then
   begin
     e2 := e2 shr Precision_Bits;
     j  := c1 + e2*Cible.Cols;
     if (e2 <> e1) and (e2 < Cible.Rows) then
       BCible^[j] := BCible^[j] or ($80 shr f1);
   end;

end;

procedure Horizontal_Sweep_Drop( y     : Int;
                                 x1,
                                 x2    : TT_F26dot6;
                                 Left,
                                 Right : PProfile );
var
  e1, e2  : Longint;
  c1, c2  : Int;
  f1, f2  : Int;

  j : Int;
begin

  e1 := ( x1+(Precision-1) ) and Precision_Mask;
  e2 := x2 and Precision_Mask;

  (* During the horizontal sweep, we only take care of drop-outs *)

  if e1 > e2 then
   if e1 = e2 + Precision then

    case DropOutControl of

      0 : exit;

      (* Drop-out Control Rule #3 *)
      1 : e1 := e2;

      4 : begin
            e1 := ( (x1+x2) div 2 +Precision div 2 ) and Precision_Mask;
            e2 := e1;
          end;

      (* Drop-out Control Rule #4 *)

      (* The spec is not very clear regarding rule #4. It       *)
      (* presents a method that is way too costly to implement  *)
      (* while the general idea seems to get rid of 'stubs'.    *)
      (*                                                        *)

      2,5 : begin

              (* rightmost stub test *)

              if ( Left^.next = Right ) and
                 ( Left^.Height <= 0 )  then exit;

              (* leftmost stub test *)

              if ( Right^.next = Left ) and
                 ( Left^.Start = y   ) then exit;

              (* Check that the upmost pixel is not already set *)

              e1 := e1 div Precision;

              c1 := y shr 3;
              f1 := y and 7;

              j := c1 + e1*Cible.Cols;

              if ( e1 >= 0 ) and ( e1 < Cible.Rows ) and
                 ( BCible^[j] and ($80 shr f1) <> 0 ) then exit;

              case DropOutControl of
                2 : e1 := e2;
                5 : e1 := ((x1+x2) div 2 + Precision_Half) and Precision_Mask;
              end;

              e2 := e1;
            end;
    else
      exit;  (* Unsupported mode *)
    end;

   c1 := y shr 3;
   f1 := y and 7;

   if (e1 >= 0) then
   begin
     e1 := e1 shr Precision_Bits;
     j  := c1 + e1*Cible.cols;
     if e1 < Cible.Rows then BCible^[j] := BCible^[j] or ($80 shr f1);
   end;

end;



procedure Horizontal_Sweep_Step;
begin
  (* Nothing, really *)
end;

(***********************************************************************)
(*                                                                     *)
(*  Vertical Gray Sweep Procedure Set :                                *)
(*                                                                     *)
(*  These two   routines are used during the vertical gray-levels      *)
(*  sweep phase by the generic Draw_Sweep function.                    *)
(*                                                                     *)
(*                                                                     *)
(*  NOTES :                                                            *)
(*                                                                     *)
(*  - The target pixmap's width *must* be a multiple of 4              *)
(*                                                                     *)
(*  - you have to use the function Vertical_Sweep_Span for             *)
(*    the gray span call.                                              *)
(*                                                                     *)
(***********************************************************************)

procedure Gray_Sweep_Init( var min, max : Int );
begin
  min        :=  min and -2;
  max        :=  (max + 3) and -2;
  TraceOfs   :=  0;
  TraceG     :=  Cible.Cols * ( min div 2 );
  gray_min_x :=  Cible.Cols;
  gray_max_x := -Cible.Cols;
end;



procedure Gray_Sweep_Step;
var
  j, c1, c2 : Int;
begin
  inc( TraceOfs, Gray_Width );
  if TraceOfs > Gray_Width then
  begin

    if gray_max_X >= 0 then
    begin

      if gray_max_x > cible.cols-1 then gray_max_x := cible.cols-1;

      if gray_min_x < 0 then gray_min_x := 0;

      j := TraceG + gray_min_x*4;

      for c1 := gray_min_x to gray_max_x do
      begin

        c2 := Count_Table[ BCible^[c1           ] ] +
              Count_Table[ BCible^[c1+Gray_Width] ];

        if c2 <> 0 then
        begin

          BCible^[c1           ] := 0;
          BCible^[c1+Gray_Width] := 0;

          GCible^[j] := GCible^[j] or Grays[ (c2 and $F000) shr 12 ]; inc(j);
          GCible^[j] := GCible^[j] or Grays[ (c2 and $0F00) shr  8 ]; inc(j);
          GCible^[j] := GCible^[j] or Grays[ (c2 and $00F0) shr  4 ]; inc(j);
          GCible^[j] := GCible^[j] or Grays[ (c2 and $000F)        ]; inc(j);

        end
        else
          inc( j, 4 );

      end;
    end;

    TraceOfs := 0;
    inc( TraceG, Cible.Cols );

    gray_min_x :=  Cible.Cols;
    gray_max_x := -Cible.Cols;

  end;

end;


{$IFDEF SMOOTH}

(***********************************************************************)
(*                                                                     *)
(*  Vertical Smooth Sweep Procedure Set :                              *)
(*                                                                     *)
(*  These two   routines are used during the vertical smooth-levels    *)
(*  sweep phase by the generic Draw_Sweep function.                    *)
(*                                                                     *)
(*                                                                     *)
(*  NOTES :                                                            *)
(*                                                                     *)
(*  - The target pixmap's width *must* be a multiple of 2              *)
(*                                                                     *)
(*  - you have to use the function Vertical_Sweep_Span for             *)
(*    the smooth span call.                                            *)
(*                                                                     *)
(***********************************************************************)

procedure Smooth_Sweep_Init( var min, max : Int );
var
  i : integer;
begin
  min        :=  min and -4;
  max        :=  (max + 7) and -4;
  TraceOfs   :=  0;
  TraceG     :=  Cible.Cols * ( min div 4 );
  gray_min_x :=  Cible.Cols;
  gray_max_x := -Cible.Cols;

  smooth_pass := 0;
(*
  for i := 0 to Smooth_Cols-1 do
    GCible^[i] := 0;
*)
end;



procedure Smooth_Sweep_Step;
var
  j, c1, c2 : Int;
begin

  if gray_max_X >= 0 then
  begin

    if gray_max_x > cible.cols-1 then gray_max_x := cible.cols-1;

    if gray_min_x < 0 then gray_min_x := 0;

    j := TraceG + gray_min_x*2;

    for c1 := gray_min_x to gray_max_x do
    begin

     c2 := Count_Table2[ BCible^[c1] ];

      if c2 <> 0 then
      begin
        inc( GCible^[j], c2 shr 4  ); inc(j);
        inc( GCible^[j], c2 and 15 ); inc(j);

        BCible^[c1] := 0;
      end
      else
        inc( j, 2 );
    end;

  end;

  traceOfs := 0;
  inc( smooth_pass );

  if smooth_pass >= 4 then
  begin

    j := TraceG + gray_min_x*2;

    for c1 := gray_min_x to gray_max_x do
    begin
      c2 := GCible^[j]; GCible^[j] := Smooths[c2]; inc(j);
      c2 := GCible^[j]; GCible^[j] := Smooths[c2]; inc(j);
    end;

    smooth_pass := 0;
    inc( TraceG, Cible.Cols );

    gray_min_x :=  Cible.Cols;
    gray_max_x := -Cible.Cols;
  end;

end;

{$ENDIF}

{$F-  End of dispatching functions definitions }


(********************************************************************)
(*                                                                  *)
(*  Generic Sweep Drawing routine                                   *)
(*                                                                  *)
(*                                                                  *)
(*                                                                  *)
(********************************************************************)

function Draw_Sweep : boolean;

label
  Skip_To_Next;

var
  y, k,
  I, J   : Int;
  P, Q   : PProfile;

  Top,
  Bottom,
  min_Y,
  max_Y  : Int;

  x1, x2, xs, e1, e2 : LongInt;

  Wait  : TProfileList;

  Draw_Left  : TProfileList;
  Draw_Right : TProfileList;

  Drop_Left  : TProfileList;
  Drop_Right : TProfileList;

  P_Left,  Q_Left  : PProfile;
  P_Right, Q_Right : PProfile;

  Phase     : Int;
  dropouts  : Int;

begin

  Draw_Sweep := False;

  (* Init the empty linked lists *)

  Init_Linked( Wait );

  Init_Linked( Draw_Left  );
  Init_Linked( Draw_Right );

  Init_Linked( Drop_Left  );
  Init_Linked( Drop_Right );

  (* First, compute min Y and max Y *)

  P     := fProfile;
  max_Y := MinY div Precision;
  min_Y := MaxY div Precision;

  while P <> nil do
   with P^ do
    begin
     Q := P^.Link;

     Case Flow of

       TTFlowUp : begin
                   Bottom := Start;
                   Top    := Start + Height - 1;
                  end;

       TTFlowDown : begin
                     Bottom := Start - Height + 1;
                     Top    := Start;

                     Start  := Bottom;
                     Offset := Offset+Height-1;
                    end;
      else
        (* Severe Error here !! *)
        Raster_Error := Err_Ras_Invalid;
        exit;
      end;

     if min_Y > Bottom then min_Y := Bottom;
     if max_Y < Top    then max_Y := Top;

     X := 0;
     InsNew( @Wait, P );

     P := Q;
   end;

  (* Now inits the sweeps *)

  Proc_Sweep_Init( min_Y, max_Y );

  (* Then compute the distance of each Profile to min Y *)

  P := Wait.Head.Link;
  Q := P^.Link;

  while P <> Q do
  begin
    with P^ do CountL := (Start-min_Y);
    P := Q;
    Q := P^.Link;
  end;

  (* Let's go *)

  for y := min_Y to max_Y do
   begin

    (* Look in the wait list for new activations *)

    P := Wait.Head.Link;
    Q := P^.Link;

    while P <> Q do with P^ do
    begin

      if CountL = 0 then
        begin
          DelOld( @Wait, P );
          case Flow of
            TTFlowUp   : InsNew( @Draw_Left, P );
            TTFlowDown : InsNew( @Draw_Right, P );
          end
        end

      else
        dec( CountL );

      P := Q;
      Q := P^.Link;
    end;

    (* Sort the drawing lists *)

    Sort( @Draw_Left );
    Sort( @Draw_Right );

    (* Let's trace *)

    dropouts  := 0;

    P_Left  := Draw_Left .Head.Link;
    P_Right := Draw_Right.Head.Link;

    Q_Left  := P_Left^ .Link;
    Q_Right := P_Right^.Link;

    while ( P_Left <> Q_Left ) do
    begin

      {$IFDEF ASSERT}
      if Q_Right = P_Right then
        Halt(11);
      {$ENDIF}

      x1 := P_Left^ .X;
      x2 := P_Right^.X;

{$IFDEF REVERSE}
      if x1 > x2 then
        begin
          xs := x1;
          x1 := x2;
          x2 := xs;
        end;
{$ENDIF}

      if ( x2-x1 <= Precision ) then
        begin
          e1 := ( x1+Precision-1 ) and Precision_Mask;
          e2 := x2 and Precision_Mask;

          if (e1 > e2) or (e2 = e1 + Precision) then
          begin
            P_Left^.x  := x1;
            P_Right^.x := x2;

            inc( dropouts );

            DelOld( @Draw_Left,  P_Left );
            DelOld( @Draw_Right, P_Right );

            InsNew( @Drop_Left,  P_Left );
            InsNew( @Drop_Right, P_Right );

            goto Skip_To_Next;
          end
        end;

      Proc_Sweep_Span( y, x1, x2, P_Left, P_Right );

      (* We finalize the Profile if needed *)

      if P_Left ^.height = 0 then
          DelOld( @Draw_Left,  P_Left  );

      if P_Right^.height = 0 then
          DelOld( @Draw_Right, P_Right );

  Skip_To_Next:

      P_Left  := Q_Left;
      P_Right := Q_Right;

      Q_Left  := P_Left ^.Link;
      Q_Right := P_Right^.Link;

    end;

    {$IFDEF ASSERT}
    if Q_Right <> P_Right then
      Halt(10);
    {$ENDIF}

    (* Now perform the dropouts only _after_ the span drawing *)

    P_Left  := Drop_Left. Head.Link;
    P_Right := Drop_Right.Head.Link;

    while ( dropouts > 0 ) do
    begin

      Q_Left  := P_Left^. Link;
      Q_Right := P_Right^.Link;

      DelOld( @Drop_Left, P_Left );
      DelOld( @Drop_Right, P_Right );

      Proc_Sweep_Drop( y, P_Left^.x, P_Right^.x, P_Left, P_Right );

      if P_Left^.height > 0 then
        InsNew( @Draw_Left, P_Left );

      if P_Right^.height > 0 then
        InsNew( @Draw_Right, P_Right );

      P_Left  := Q_Left;
      P_Right := Q_Right;

      dec( dropouts );
    end;

    (* Step to next line *)

    Proc_Sweep_Step;

  end;

  Draw_Sweep := True;

end;

(****************************************************************************)
(*                                                                          *)
(* Function:    Render_Single_Pass                                          *)
(*                                                                          *)
(* Description: Performs one sweep with sub-banding.                        *)
(*                                                                          *)
(* Input:       _XCoord, _YCoord : x and y coordinates arrays               *)
(*                                                                          *)
(* Returns:     True on success                                             *)
(*              False if any error was encountered during render.           *)
(*                                                                          *)
(****************************************************************************)

function Render_Single_Pass( _XCoord, _YCoord : TT_PCoordinates ) : boolean;
var
  i, j, k : Int;
begin
  Render_Single_Pass := False;

  while Band_Top > 0 do

    begin

      with Band_Stack[ Band_Top ] do
        begin
          MaxY   := longint(Y_Max) * Precision;
          MinY   := longint(Y_Min) * Precision;
        end;

      profCur  := 0;
      Raster_Error := Err_Ras_None;

      if not Convert_Glyph( _XCoord, _YCoord ) then
        begin

          if Raster_Error <> Err_Ras_Overflow then exit;
          Raster_Error := Err_Ras_None;

          (* sub-banding *)

          {$IFDEF DEBUG3}
          ClearBand( MinY shr Precision_Bits, MaxY shr Precision_Bits );
          {$ENDIF}

          with Band_Stack[Band_Top] do
            begin
              I := Y_Min;
              J := Y_Max;
            end;

          K := ( I + J ) div 2;

          if ( Band_Top >= 8 ) or ( K <= I ) then
            begin
              Band_Top := 0;
              Raster_Error := Err_Ras_Invalid;
              exit;
            end
          else
            begin

              with Band_Stack[Band_Top+1] do
                begin
                  Y_Min := K;
                  Y_Max := J;
                end;

              Band_Stack[Band_Top].Y_Max := K-1;

              inc( Band_Top );
            end
        end
      else
        begin
          if ( fProfile <> nil ) then
            if not Draw_Sweep then exit;
          dec( Band_Top );
        end;

    end;

  Render_Single_Pass := true;
end;

(****************************************************************************)
(*                                                                          *)
(* Function:    Render_Glyph                                                *)
(*                                                                          *)
(* Description: Renders a glyph in a bitmap.      Sub-banding if needed     *)
(*                                                                          *)
(* Input:       AGlyph   Glyph record                                       *)
(*                                                                          *)
(* Returns:     True on success                                             *)
(*              False if any error was encountered during render.           *)
(*                                                                          *)
(****************************************************************************)

function Render_Glyph( var glyph  : TGlyphRecord;
                           target : PRasterBlock;
                           scan   : Byte ) : boolean;

begin

 Render_Glyph := False;

 if Buff = nil then
   begin
     Raster_Error := Err_Ras_NotIni;
     exit;
   end;

 if target <> nil then
   cible := target^;

 Outs      := Glyph.endPoints;
 Flags     := PByte(glyph.Flag);
 nPoints   := Glyph.Points;
 nContours := Glyph.numConts;

 DropOutControl := scan;

 Raster_Error := Err_Ras_None;

 (* Vertical Sweep *)

 Proc_Sweep_Init := Vertical_Sweep_Init;
 Proc_Sweep_Span := Vertical_Sweep_Span;
 Proc_Sweep_Drop := Vertical_Sweep_Drop;
 Proc_Sweep_Step := Vertical_Sweep_Step;

 Band_Top            := 1;
 Band_Stack[1].Y_Min := 0;
 Band_Stack[1].Y_Max := Cible.Rows-1;

 BWidth := Cible.width;
 BCible := PByte( Cible.Buffer );

 if not Render_Single_Pass( Glyph.XCoord, Glyph.YCoord ) then exit;

 (* Horizontal Sweep *)

 if Second_Pass  then
 begin

   Proc_Sweep_Init := Horizontal_Sweep_Init;
   Proc_Sweep_Span := Horizontal_Sweep_Span;
   Proc_Sweep_Drop := Horizontal_Sweep_Drop;
   Proc_Sweep_Step := Horizontal_Sweep_Step;

   Band_Top            := 1;
   Band_Stack[1].Y_Min := 0;
   Band_Stack[1].Y_Max := Cible.Width-1;

   BWidth := Cible.rows;
   BCible := PByte( Cible.Buffer );

   if not Render_Single_Pass( Glyph.YCoord, Glyph.XCoord ) then exit;

 end;

 Render_Glyph := True;
end;

(****************************************************************************)
(*                                                                          *)
(* Function:    Render_Gray_Glyph                                           *)
(*                                                                          *)
(* Description: Renders a glyph with grayscaling. Sub-banding if needed     *)
(*                                                                          *)
(* Input:       AGlyph   Glyph record                                       *)
(*                                                                          *)
(* Returns:     True on success                                             *)
(*              False if any error was encountered during render.           *)
(*                                                                          *)
(****************************************************************************)

function Render_Gray_Glyph( var glyph   : TGlyphRecord;
                                target  : PRasterBlock;
                                scan    : Byte;
                                palette : pointer      ) : boolean;
begin

 Render_Gray_Glyph := False;

 if target <> nil then
   cible := target^;

 if palette <> nil then
   move( palette^, Grays, 5 );

 Outs      := Glyph.endPoints;
 Flags     := PByte(glyph.Flag);
 nPoints   := Glyph.Points;
 nContours := Glyph.numConts;

 DropOutControl := scan;

 Raster_Error := Err_Ras_None;

 Band_Top            := 1;
 Band_Stack[1].Y_Min := 0;
 Band_Stack[1].Y_Max := 2*Cible.Rows - 1;

 BWidth := Gray_Width;

 if BWidth > Cible.cols div 4 then BWidth := Cible.cols div 4;

 BWidth := BWidth*8;
 BCible := PByte( Gray_Lines   );
 GCible := PByte( Cible.Buffer );

 Proc_Sweep_Init := Gray_Sweep_Init;
 Proc_Sweep_Span := Vertical_Sweep_Span;
 Proc_Sweep_Drop := Vertical_Sweep_Drop;
 Proc_Sweep_Step := Gray_Sweep_Step;

 if not Render_Single_Pass( Glyph.XCoord, Glyph.YCoord ) then exit;

 Render_Gray_Glyph := True;
 exit;

end;

{$IFDEF SMOOTH}
(****************************************************************************)
(*                                                                          *)
(* Function:    Render_Smooth_Glyph                                         *)
(*                                                                          *)
(* Description: Renders a glyph with grayscaling. Sub-banding if needed     *)
(*                                                                          *)
(* Input:       AGlyph   Glyph record                                       *)
(*                                                                          *)
(* Returns:     True on success                                             *)
(*              False if any error was encountered during render.           *)
(*                                                                          *)
(****************************************************************************)

function Render_Smooth_Glyph( var glyph   : TGlyphRecord;
                                  target  : PRasterBlock;
                                  scan    : Byte;
                                  palette : pointer      ) : boolean;
begin

 Render_Smooth_Glyph := False;

 if target <> nil then
   cible := target^;
(*
 if palette <> nil then
   move( palette^, Grays, 5 );
*)
 Outs      := Glyph.endPoints;
 Flags     := PByte(glyph.Flag);
 nPoints   := Glyph.Points;
 nContours := Glyph.numConts;

 DropOutControl := scan;

 Raster_Error := Err_Ras_None;

 Band_Top            := 1;
 Band_Stack[1].Y_Min := 0;
 Band_Stack[1].Y_Max := 4*Cible.Rows - 1;

 BWidth := Smooth_Cols;

 if BWidth > Cible.cols then BWidth := Cible.cols;

 BWidth := BWidth*8;
 BCible := PByte( Gray_Lines   );
 GCible := PByte( Cible.Buffer );

 Proc_Sweep_Init := Smooth_Sweep_Init;
 Proc_Sweep_Span := Vertical_Sweep_Span;
 Proc_Sweep_Drop := Vertical_Sweep_Drop;
 Proc_Sweep_Step := Smooth_Sweep_Step;

 if not Render_Single_Pass( Glyph.XCoord, Glyph.YCoord ) then exit;

 Render_Smooth_Glyph := True;
 exit;

end;

{$ENDIF}

(****************************************************************************)
(*                                                                          *)
(* Function:    Init_Rasterizer                                             *)
(*                                                                          *)
(* Description: Initializes the rasterizer.                                 *)
(*                                                                          *)
(* Input:       rasterBlock   target bitmap/pixmap description              *)
(*              profBuffer    pointer to the render pool                    *)
(*              profSize      size in bytes of the render pool              *)
(*                                                                          *)
(* Returns:     1 ( always, but we should check parameters )                *)
(*                                                                          *)
(****************************************************************************)

function Init_Rasterizer( profBuffer   : Pointer;
                          profSize     : longint;
                          grayBuffer   : Pointer;
                          grayLength   : integer
                        )
                        : longint;
var
  i, j, c, l : integer;
const
  Default_Grays : array[0..4] of Byte
                = ( 0, 23, 27, 29, 31 );

  Default_Smooths : array[0..16] of Byte
                  = ( 0,  20, 20, 21, 22, 23, 24, 25,
                      26, 27, 28, 29, 30, 31, 31, 31, 31 );

begin
  Buff    := PStorage(profBuffer);
  MaxBuff := (profSize div 4) - AlignProfileSize;

  Gray_Lines  := grayBuffer;
  Gray_Width  := grayLength div 2;

{$IFDEF SMOOTH}
  Smooth_Cols := grayLength div 4;
{$ENDIF}

  { Initialisation of Count_Table }

  for i := 0 to 255 do
  begin

    l := 0;
    j := i;

    for c := 0 to 3 do
    begin

      l := l shl 4;

      if ( j and $80 <> 0 ) then inc(l);
      if ( j and $40 <> 0 ) then inc(l);

      j := (j shl 2) and $FF;

    end;

    Count_table[i] := l;

  end;

  (* default Grays takes the gray levels of the standard VGA *)
  (* 256 colors mode                                                *)

  Grays[0] := 0;
  Grays[1] := 23;
  Grays[2] := 27;
  Grays[3] := 29;
  Grays[4] := 31;


{$IFDEF SMOOTH}

  { Initialisation of Count_Table2 }

  for i := 0 to 255 do
  begin

    l := 0;
    j := i;

    for c := 0 to 1 do
    begin

      l := l shl 4;

      if ( j and $80 <> 0 ) then inc(l);
      if ( j and $40 <> 0 ) then inc(l);
      if ( j and $20 <> 0 ) then inc(l);
      if ( j and $10 <> 0 ) then inc(l);

      j := (j shl 4) and $FF;

    end;

    Count_table2[i] := l;

  end;

  move( Default_Smooths, Smooths, 17 );
{$ENDIF}

  DropOutControl := 2;
  Raster_Error       := Err_Ras_None;

  Init_Rasterizer := 1;
end;

procedure Cycle_DropOut;
begin
  case DropOutControl of

    0 : DropOutControl := 1;
    1 : DropOutControl := 2;
    2 : DropOutControl := 4;
    4 : DropOutControl := 5;
  else
    DropOutControl := 0;
  end;
end;


begin
  MaxBuff := 0;
  Buff    := nil;
  profCur := 0;

{$IFNDEF CONST_PREC}
  Set_High_Precision( False );
{$ENDIF}

end.
