/****************************************************************************/
/*                                                                          */
/*  The FreeType project - a free and portable quality TrueType renderer.   */
/*                                                                          */
/*  Copyright 1996, 1997 by                                                 */
/*  D. Turner, R.Wilhelm, and W. Lemberg                                    */
/*                                                                          */
/*  View : A simple font viewer.  Now supports hinting and grayscaling      */
/*         with the '-g' option.                                            */
/*                                                                          */
/*                                                                          */
/*  Keys :                                                                  */
/*                                                                          */
/*  x :   fine counter_clockwise rotation                                   */
/*  c :   fine clockwise rotation                                           */
/*                                                                          */
/*  v :   fast counter_clockwise rotation                                   */
/*  b :   fast clockwise rotation                                           */
/*                                                                          */
/*  + :   fast scale up                                                     */
/*  - :   fast scale down                                                   */
/*  u :   fine scale down                                                   */
/*  j :   fine scale up                                                     */
/*                                                                          */
/*  l :   go to next glyph                                                  */
/*  k :   go to previous glyph                                              */
/*                                                                          */
/*  o :   go to tenth next glyph                                            */
/*  i :   go to tenth previous glyph                                        */
/*                                                                          */
/*  0 :   go to hundredth next glyph                                        */
/*  9 :   go to hundredth previous glyph                                    */
/*                                                                          */
/*  ESC :   exit                                                            */
/*                                                                          */
/*                                                                          */
/*  NOTE : This is just a test program that is used to show off and         */
/*         debug the current engine; which is still in alpha. In no         */
/*         way does it shows the final high-level interface that            */
/*         client applications will use. Wait for at least a beta for       */
/*         this.                                                            */
/*                                                                          */
/****************************************************************************/

#ifdef ARM
#include "std.h"
#include "graflink.h"
#endif

#include "tttypes.h"
#include "tterror.h"
#include "ttcalc.h"
#include "tttables.h"
#include "ttmemory.h"
#include "ttraster.h"
#include "ttindex.h"
#include "ttins.h"
#include "ttfile.h"
#include "ttexec.h"
#include "tttables.h"

#include "stdio.h"
#include "stdlib.h"
#include "string.h"
#include "math.h"    /* libc ANSI */

#include "gmain.h"
#include "gevents.h"
#include "gdriver.h"


#ifdef ARM
#include "armsup.c" /* pull in our routines */
#endif

#define  Pi         3.1415926535

#define  MAXPTSIZE  500                /* dtp */
#define  Center_X   ( Bit.width / 2 )  /* dtp */
#define  Center_Y   ( Bit.rows  / 2 )  /* dtp */

#define  Profile_Buff_Size  64000      /* Size of the render pool   */
                                       /* Minimum is around 4 Kb    */
                                       /* experiment to see how its */
                                       /* size impacts on raster    */

                                       /* performance..             */
#define  Font_Buff_Size     512000     /* this buffer holds all     */
                                       /* font specific data.       */


  char                Header[128];

  TT_Stream           stream;
  PResident_Record    resident;
  PInstance_Record    instance;
  PExecution_Context  exec;

  PStorage            Font_Buffer;

  int  num_glyphs;

  Int  ptsize;

  TGlyphRecord   gRec;

  TRasterBlock   Bit;

  Int            Rotation;
  Int            Fail;
  Int            Num;
  unsigned char  autorun;

  char           GrayLines[1024];
  int            gray_render;

  TT_PCMapTable  OurcmapIndex;


  static void  ClearData();


  void  Set_Raster_Area()
  {
    Bit.rows  = vio_Height;   /* The whole window */
    Bit.width = vio_Width;

    if ( gray_render )
    {
      Bit.cols  = Bit.width;
      Bit.flow  = TT_Flow_Up;
      Bit.size  = Bit.rows * Bit.width;
    }
    else
    {
      Bit.cols  = ( Bit.width + 7 ) / 8;    /* convert to # of bytes */
      Bit.flow  = TT_Flow_Up;
      Bit.size  = Bit.rows * Bit.cols;      /* number of bytes in buffer */
    }
  }


  int  Init_Engine( PInstance_Record  instance,
                    Int               maxres ) 
  {
    PByte  p;
    Int    temp;


    Set_Raster_Area();   /* setup maximum raster size */
  
    Bit.bitmap = (void*)malloc( (int)Bit.size );
    if ( !Bit.bitmap ) 
    {
      printf( "ERROR: Not enough memory to allocate bitmap!\n" );
      exit( 1 );
    }
  
    p = (PByte)malloc( Profile_Buff_Size );
    if ( !p )
    {
      printf( "ERROR: Not enough memory to allocate render pool!\n" );
      exit( 1 );
    }

    if ( gray_render )
      InitRasterizer( (long*)p, Profile_Buff_Size, GrayLines, 1024 );
    else
      InitRasterizer( (long*)p, Profile_Buff_Size, NULL, 0 );

    ClearData();

    temp = MAX( resident->maxProfile.maxContours,
                resident->maxProfile.maxCompositeContours );
    gRec.outStarts = (PUShort) malloc( temp * sizeof( UShort ) );

    temp = MAX( resident->maxProfile.maxPoints,
                resident->maxProfile.maxCompositePoints );
    gRec.xCoord = (PStorage)malloc( temp * sizeof( TT_F26Dot6 ) );
    gRec.yCoord = (PStorage)malloc( temp * sizeof( TT_F26Dot6 ) );
    gRec.flag   = (PByte)   malloc( temp );

    if ( gRec.outStarts == NULL || gRec.xCoord == NULL ||
         gRec.yCoord == NULL || gRec.flag == NULL )
      return 1;

    return 0;
  }
  
  
  static void  ClearData()
  {
    if ( gray_render )
      memset( Bit.bitmap, gray_palette[0], Bit.size );
    else
      memset( Bit.bitmap, 0, Bit.size );
  }


  Bool  Reset_Scale( Int  pointSize )
  {
    if ( !Reset_Context( instance, pointSize, 96 ) )
    {
      printf( "error = %d\n", exec->error );
      Panic1( "could not reset execution context\n" );
    }

    return SUCCESS;
  }
  

  Bool  LoadTrueTypeChar( Int  idx )
  {
    PGlyph  pG;
    Short   numP, numC;
    Int     j;

    PExecution_Context  exec = instance->exec;


    /* Reset_Context( instance->exec );  */

    if ( idx < 0 || idx > num_glyphs )
      return FAILURE;
    
    if ( !Load_TrueType_Glyph( instance, idx, FALSE ) )
      return FAILURE;
  
    pG   = instance->glyph;
    numP = instance->pts.n;
    numC = instance->glyph->numberOfContours;
  
    if ( numP <= 0 || numC <= 0 )
      return FAILURE;
  
    gRec.outlines = numC;
    gRec.points   = numP;

    TT_Transform_Glyph( instance );     /* this call scales the glyph */

    if ( !Load_Glyph_Instructions( instance, 0 ) )
      Panic1( "could not load instructions\n" );

    if ( !Run_Context( instance ) )
      return FAILURE;

    numP -= 2;  /* remove phantom points */

    for ( j = 0; j < numP; j++ )
    {
      gRec.xCoord[j] = exec->pts.cur_x[j];
      gRec.yCoord[j] = exec->pts.cur_y[j];
      gRec.flag  [j] = exec->pts.touch[j] & TT_Flag_On_Curve;
    }

    for ( j = 0; j < numC; ++j )
      gRec.outStarts[j] = instance->glyph->contours[j].finish;
  
    return SUCCESS;
  }


  Bool  ConvertRaster()
  {
    if ( gray_render )
      return Render_Gray_Glyph( &gRec, &Bit, 2, gray_palette );
    else
      return Render_Glyph( &gRec, &Bit, 2 );
  }


  Bool  Render_All( Int  first_glyph )
  {
    TT_F26Dot6  start_x, start_y, step_x, step_y, x, y;
    Int         scale_shift, i, j;


    start_x = 4;
    start_y = vio_Height - 50;

    if ( gray_render )
      scale_shift = 1;
    else
      scale_shift = 0;

    exec = (PExecution_Context)instance->exec;

    step_x = exec->ppem + 4;
    step_y = exec->ppem + 10;

    x = start_x;
    y = start_y;

    i = first_glyph;

    while ( i < num_glyphs )
    {
      if ( LoadTrueTypeChar( i ) )
      {
        for ( j = 0; j < exec->pts.n - 2; j++ )
        {
          gRec.xCoord[j] = ( (gRec.xCoord[j] + (x*64)) << scale_shift ) - 32;
          gRec.yCoord[j] = ( (gRec.yCoord[j] + (y*64)) << scale_shift ) - 32;
        }

        ConvertRaster();

        x += ( exec->pts.cur_x[exec->pts.n - 1] / 64 ) + 1;

        if ( x + exec->ppem > vio_Width )
        {
          x  = start_x;
          y -= step_y;

          if ( y < 10 ) return SUCCESS;
        }
      }
      i++;
    }

    return SUCCESS;
  }


  int  Process_Event( TEvent*  event )
  {
    switch ( event->what )
    {
    case event_Quit:            /* ESC */
      return 0;
      
    case event_Rotate_Glyph:
      Rotation = ( Rotation + event->info ) & 1023;
      break;

    case event_Scale_Glyph:
      ptsize += event->info;
      if ( ptsize < 1 )         ptsize = 1;
      if ( ptsize > MAXPTSIZE ) ptsize = MAXPTSIZE;
      break;

    case event_Change_Glyph:
      Num += event->info;
      if (Num < 0)           Num = 0;
      if (Num >= num_glyphs) Num = num_glyphs - 1;
      break;
    }
    return 1;
  }


  int  main( int  argc, char**  argv ) 
  {
    int     i, old_ptsize;
    char    filename[128+4];
    char*   execname;
  
    TEvent  event;
 

    Font_Buffer = (PStorage)malloc( Font_Buff_Size );
    if ( !Font_Buffer )
    {
      printf( "Error: Could not even allocate font pool!\n" );
      exit( 1 );
    }

    Init_FontPool( Font_Buffer, Font_Buff_Size );

    execname = argv[0];

    if ( argc > 1 && !strcmp( argv[1], "-g" ) )
    {
      argc--;
      argv++;
      gray_render = 1;
    }

    if ( argc != 3 )
    {
      printf( "View: simple TrueType interpreter tester - part of the FreeType project\n" );
      printf( "-----------------------------------------------------------------------\n\n" );
      printf( "Usage: %s [-g] ppem fontname[.ttf]\n\n", execname );
      exit( 1 );
    }

    if ( sscanf( argv[1], "%d", &ptsize ) == EOF )
      ptsize = 64;

    i = strlen( argv[2] );
    while ( i > 0 && argv[2][i] != '\\' )
    {
      if ( argv[2][i] == '.' )
        i = 0;
      i--;
    }

    filename[128] = 0;

    strncpy( filename, argv[2], 128 );
    if ( i >= 0 )
      strncpy( filename + strlen(filename), ".ttf", 4 );

    if ( !TT_Open_File( filename , &stream ) )
    {
      printf( "Error, could not find/open %s\n", filename );
      exit( 1 );
    }

    if ( !TT_Load_Resident_Table( stream, &resident ) ) 
    {
      printf( "ERROR: Could not load data from %s\n", filename );
      exit( 1 );
    }

    num_glyphs = resident->numGlyphs;
        
    if ( !TT_Load_Instance_Data( resident, &instance ) ) 
    {
      printf( "ERROR: Could not open instance from %s\n", filename );
      exit( 1 );
    }

    if ( !Create_Context( instance ) )
      Panic1( "could not create execution context\n" );

    exec = (PExecution_Context)instance->exec;

    if ( gray_render ) 
    {
      if ( !SetGraphScreen( Graphics_Mode_Gray ) )
        Panic1( "could not set grayscale graphics mode\n" );
    }
    else
    {
      if ( !SetGraphScreen( Graphics_Mode_Mono ) )
        Panic1( "could not set mono graphics mode\n" );
    }

    Init_Engine( instance, ptsize );

    Set_High_Precision( TRUE );

    Reset_Scale( ptsize );

    old_ptsize = ptsize;

    Fail = 0;
    Num  = 0;

    for ( ;; )
    {
      Render_All( Num );

      sprintf( Header, "Glyph : %4d  ptsize : %4d", Num, ptsize );
#ifndef X11
#ifndef OS2
      Print_XY( 0, 0, Header );
#endif
#endif

      Display_Bitmap_On_Screen( Bit.bitmap, Bit.rows, Bit.cols );

      ClearData();

      Get_Event( &event );
      if ( !Process_Event( &event ) ) goto Fin;

      if ( ptsize != old_ptsize )
      {
        if ( !Reset_Scale( ptsize ) )
          Panic1( "Could not resize font\n" );
        old_ptsize = ptsize;
      }
    }

  Fin:
    RestoreScreen();
    TT_Close_File( stream );

    printf( "execution completed succesfully\n" );
    printf( "fails = %d\n", Fail );

    return 0;
}


/* End */
