/*******************************************************************
 *
 *  ttraster.h                                                 v 1.2
 *
 *  The FreeType glyph rasterizer.
 *
 *  Copyright 1996, 1997 by
 *  David Turner, Robert Wilhelm, and Werner Lemberg
 *
 *  This file is part of the FreeType project, and may only be used
 *  modified and distributed under the terms of the FreeType project
 *  license, LICENSE.TXT. By continuing to use, modify or distribute
 *  this file you indicate that you have read the license and
 *  understand and accept it fully.
 *
 *  NOTES:
 *
 *  This version supports the following:
 *
 *    - direct grayscaling
 *    - sub-banding
 *    - drop-out modes 4 and 5
 *    - second pass for complete drop-out control (bitmap only)
 *    - variable precision
 *
 *   Re-entrancy is _not_ planned.
 *
 *   Changes between 1.1 and 1.2:
 *
 *     - no more trace tables, now uses linked list to sort
 *       coordinates.
 *     - reduced code size using function dispatch within a generic
 *       draw_sweep function.
 *     - added variable precision for finer rendering at small ppems.
 *
 *
 *   Note that its interface may change in the future.
 *
 ******************************************************************/

#ifndef TTRASTER_H
#define TTRASTER_H

#ifdef __cplusplus
extern "C" {
#endif

  /* Flow constants :                                           */
  /*                                                            */
  /* The flow of a bitmap refers to the way lines are oriented  */
  /* within the bitmap data, i.e. the orientation of the Y      */
  /* coordinate axis.                                           */
  /*                                                            */

#define TT_Flow_Down  -1  /* Bitmap is oriented from top to bottom */
#define TT_Flow_Up     1  /* Bitmap is oriented from bottom to top */
#define TT_Flow_Error  0  /* an error occured during rendering     */



#define Raster_Err_None              0
#define Raster_Err_Not_Ini           1  /* Raster was not initialized      */
#define Raster_Err_Overflow          2  /* Profile table overflow          */
#define Raster_Err_Neg_Height        3  /* Negative height encountered!    */
#define Raster_Err_Invalid           4  /* Invalid value encountered       */
#define Raster_Err_Gray_Unsupported  5  /* Graylevel rendering unsupported */

  /* A structure used to describe the target bitmap or pixmap to the   */
  /* renderer. Note that there is nothing in this structure that gives */
  /* the nature of the buffer.                                         */
  typedef struct  _TRasterBlock
  {
    unsigned long  rows;    /* Number of rows                    */
    unsigned long  cols;    /* Number of columns (bytes) per row */
    unsigned long  width;   /* Number of pixels per line         */
    int            flow;    /* Bitmap orientation                */

    void*          bitmap;  /* bit/pixmap buffer                 */
    unsigned long  size;    /* bit/pixmap size in bytes          */
  } TRasterBlock;
  typedef TRasterBlock*  PRasterBlock;

  /* A structure used to describe the source glyph to the renderer. */
  typedef struct  _TGlyphRecord
  {
    unsigned long  outlines;    /* number of contours in glyph          */
    short*         outStarts;   /* points to an array of each contour's */
                                /* start point index                    */
    unsigned long  points;      /* number of points in the glyph        */
    long*          xCoord;      /* table of x coordinates               */
    long*          yCoord;      /* table of y coordinates               */
    char*          flag;        /* table of flags                       */
  } TGlyphRecord;
  typedef TGlyphRecord*  PGlyphRecord;

  extern int  Raster_Error;     /* Global Raster Error variable */

  /* Initializes the rasterizer, specifying the render pool as   */
  /* well as the gray lines buffer (put NULL if gray-levels      */
  /* rasterization is not needed by your application).           */
  int  InitRasterizer( void*  profBuffer,
                       long   profSize,
                       void*  grayBuffer,
                       int    grayLength );

  /* Render one glyph in the target bitmap, using drop-out control */
  /* mode 'scan'.                                                  */
  int  Render_Glyph( TGlyphRecord*  glyph,
                     TRasterBlock*  target,
                     char           scan );

  /* Render one gray-level glyph in the target pixmap.              */
  /* Palette points to an array of 5 colors used for the rendering. */
  /* Use NULL to reuse the last palette. Default is VGA graylevels. */
  int  Render_Gray_Glyph( TGlyphRecord*  glyph,
                          TRasterBlock*  target,
                          char           scan,
                          char*          palette );

  void  Set_High_Precision( int   High );
  /* Enables horizontal drop-out and high precision (slow) rendering */
  /* based on the 'High' boolean argument                            */

  /* NOTE : This interface is due to change in the future */

#ifdef __cplusplus
}
#endif

#endif /* RASTER_H */


/* End */
