/*******************************************************************
 *
 *  ttexec.c                                                     1.0
 *
 *    Execution context routines.
 *
 *  Copyright 1996, 1997 by
 *  David Turner, Robert Wilhelm, and Werner Lemberg.
 *
 *  This file is part of the FreeType project, and may only be used
 *  modified and distributed under the terms of the FreeType project
 *  license, LICENSE.TXT. By continuing to use, modify or distribute 
 *  this file you indicate that you have read the license and
 *  understand and accept it fully.
 *
 ******************************************************************/

#include "tttypes.h"
#include "tterror.h"
#include "tttables.h"
#include "ttexec.h"
#include "ttins.h"
#include "ttcalc.h"

  static const TGraphicsState  Default_GraphicsState =
  {
    TRUE, 68, 9, 3,
    { 0x4000, 0 },
    { 0x4000, 0 },
    { 0x4000, 0 },
    1, 1, 1, 0, 1, 64, 1, 0, 0, 0, TRUE, 0, 0, 0
  };

/*******************************************************************
 *
 *  Function    :  Goto_CodeRange
 *
 *  Description :  Switch to a new code range (updates Code and IP).
 *
 *  Input  :  exec    target execution context
 *            range   new execution code range       
 *            IP      new IP in new code range       
 *
 *  Output :  SUCCESS on success. FAILURE on error (no code range).
 *
 *****************************************************************/

  Bool  Goto_CodeRange( PExecution_Context exec, Int range, Int IP )
  {
    PCodeRange  cr;

    if ( range < 1 || range > 3 )
    {
      exec->error = TT_Err_Bad_Argument;
      return FAILURE;
    }

    cr = &exec->codeRangeTable[ range-1 ];

    if ( cr->Base == NULL )
    {                               /* invalid code range */
      exec->error = TT_Err_Invalid_Coderange;
      return FAILURE;
    }

    /* NOTE : Because the last instruction of a program may be a CALL */
    /*        which will return to the first byte *after* the code    */
    /*        range, we test for IP <= Size, instead of IP < Size.  */

    if ( IP > cr->Size )
    {
      exec->error = TT_Err_Code_Overflow;
      return FAILURE;
    }

    exec->code     = cr->Base;
    exec->codeSize = cr->Size;
    exec->IP       = IP;
    exec->curRange = range;

    return SUCCESS;
  }

/*******************************************************************
 *
 *  Function    :  Get_CodeRange
 *
 *  Description :  Returns a pointer to a given code range. Should
 *                 be used only by the debugger. Returns NULL if
 *                 'range' is out of current bounds.
 *
 *  Input  :  exec    target execution context
 *            range   new execution code range       
 *
 *  Output :  Pointer to the code range record. NULL on failure.
 *
 *****************************************************************/

  PCodeRange  Get_CodeRange( PExecution_Context exec, Int range )
  {
    if ( range < 1 || range > 3 )
      return NULL;
    else                /* arrays start with 1 in pas, and with 0 in c */
      return &exec->codeRangeTable[ range-1 ];
  }


/*******************************************************************
 *
 *  Function    :  Set_CodeRange
 *
 *  Description :  Sets a code range.
 *
 *  Input  :  exec    target execution context
 *            range   code range index
 *            base    new code base
 *            length  sange size in bytes
 *
 *  Output :  SUCCESS on success. FAILURE on error.
 *
 *****************************************************************/

  Bool  Set_CodeRange( PExecution_Context exec,
                       Int                range,
                       void*              base,
                       Int                length )
  {
    if ( range < 1 || range > 3 )
      return FAILURE;

    exec->codeRangeTable[ range-1 ].Base = (unsigned char*)base;
    exec->codeRangeTable[ range-1 ].Size = length;

    return SUCCESS;
  }

/*******************************************************************
 *
 *  Function    :  Clear_CodeRange
 *
 *  Description :  clears a code range.
 *
 *  Input  :  exec    target execution context
 *            range   code range index
 *
 *  Output :  SUCCESS on success. FAILURE on error.
 *
 *  Notes  : Does not set the Error variable.
 *
 *****************************************************************/

  Bool Clear_CodeRange( PExecution_Context exec, Int range )
  {
    if ( range < 1 || range > 3 )
      return FAILURE;

    exec->codeRangeTable[ range-1 ].Base = NULL;
    exec->codeRangeTable[ range-1 ].Size = 0;

    return SUCCESS;
  }

/*******************************************************************
 *
 *  Function    :  Create_Context
 *
 *  Description :  Creates a new execution context for a newly
 *                 created instance. Note that in our current
 *                 implementation, the context, as well as other
 *                 instance-specific tables like FDefs, IDefs,
 *                 execution stack, etc .. are created by the
 *                 instance creation function within tttables.c
 *
 *                 However, we may later change some of these
 *                 allocations to make a more flexible library
 *                 ( e.g. allocating an execution stack 'on-demand'
 *                   would be a good thing, memory-wise ).
 *
 *  Input  :  ins     fresh new instance
 *
 *  Output :  SUCCESS on success. FAILURE on error.
 *
 *  Notes  : Does not set the Error variable.
 *
 *****************************************************************/


 Bool Create_Context( PInstance_Record  ins )
 {
   PResident_Record   fontres = ins->fontRes;
   PExecution_Context exec;

   exec = (PExecution_Context)(ins->exec);
   /* currently, the context must be allocated by the instance  */
   /* creation function in tttables.c. We may replace this line */
   /* by a call to malloc later.                                */

   exec->owner = (void*)ins;

   exec->error = TT_Err_Ok;

   exec->numFDefs = ins->numFDefs;
   exec->FDefs    = ins->FDefs;
   /* This is a pointer to the table allocated in tttables.c */

   exec->numIDefs = ins->numIDefs;
   exec->IDefs    = ins->IDefs;
   /* This is a pointer to the table allocated in tttables.c */

   exec->callTop   = 0;
   exec->callSize  = ins->callSize;
   exec->callStack = ins->callStack;
   /* This is a pointer to the table allocated in tttables.c */

   exec->storeSize = ins->storeSize;
   exec->storage   = ins->storage;

   exec->stackSize = ins->stackSize;
   exec->stack     = ins->stack;
   exec->top       = 0;

   exec->period    = 64;
   exec->phase     = 0;
   exec->threshold = 0;

   exec->ppem      = 0;
   exec->pointSize = 0;
   exec->scale1    = 0;
   exec->scale2    = 1;

   exec->pts      = ins->pts;
   exec->twilight = ins->twilight;
   /* Note that the glyph loader may change the values of exec->pts    */
   /* and exec->twilight when loading and instructing composite glyphs */

   exec->zp0      = exec->twilight;
   exec->zp1      = exec->twilight;
   exec->zp2      = exec->twilight;
   /* we're going to run the font program */

   exec->instruction_trap = 0;

   exec->numContours = 0;
   exec->endContours = ins->endContours;

   /* Set default graphics state */

   exec->GS = exec->default_GS = Default_GraphicsState;

   exec->cvtSize = ins->cvtSize;
   exec->cvt     = ins->cvt;
   /* This is a pointer to the table allocated in tttables.c */

   exec->F_dot_P = 0x10000;

   exec->compensations[0] = 0;
   exec->compensations[1] = 0;
   exec->compensations[2] = 0;
   exec->compensations[3] = 0;

   Set_CodeRange( exec, 
                  TT_CodeRange_Font, 
                  fontres->fontProgram,
                  fontres->fontPgmSize );
   /* allows font program execution */

   Clear_CodeRange( exec, TT_CodeRange_Cvt   );
   Clear_CodeRange( exec, TT_CodeRange_Glyph );
   /* Disable CVT and glyph programs coderanges */

    if ( fontres->fontPgmSize > 0 )
    {
      if ( !Goto_CodeRange( exec, TT_CodeRange_Font, 0 ) )
        return FAILURE;

      if ( !RunIns( exec ) )
      {
        ins->error = exec->error;
        return FAILURE;
      }
    }

   return SUCCESS;
 }


 Bool Reset_Context( PInstance_Record  ins,
                     Int               pointSize,
                     Int               resolution  )
 {
   Int i;

   PExecution_Context exec;
   PResident_Record   fontres;

   exec    = (PExecution_Context)ins->exec;
   fontres = ins->fontRes;

   Set_CodeRange( exec,
                  TT_CodeRange_Cvt,
                  fontres->cvtProgram,
                  fontres->cvtPgmSize );

   Clear_CodeRange( exec, TT_CodeRange_Glyph );

   for ( i = 0; i < exec->storeSize; i++ ) exec->storage[i] = 0;

   exec->pointSize  = pointSize * 64;
   exec->ppem       = ( pointSize * resolution + 36 ) / 72;
   exec->resolution = resolution;
   exec->scale1     = exec->ppem * 64;
   exec->scale2     = fontres->fontHeader.Units_Per_EM;

   exec->instruction_trap = 0;

   exec->top     = 0;
   exec->callTop = 0;

   /* Scale the cvt values to the new ppem */

   for ( i = 0; i < exec->cvtSize; i++ )
     exec->cvt[i] = MulDiv_Round( fontres->cvt[i], 
                                  exec->scale1, 
                                  exec->scale2 );

   /* All twilight points are originally zero */

   for ( i = 0; i < exec->twilight.n; i++ )
   {
     exec->twilight.org_x[i] = 0;
     exec->twilight.org_y[i] = 0;
     exec->twilight.cur_x[i] = 0;
     exec->twilight.cur_y[i] = 0;
   }

   exec->GS = exec->default_GS = Default_GraphicsState;

   if ( fontres->cvtPgmSize > 0 )
   {
     if ( !Goto_CodeRange( exec, TT_CodeRange_Cvt, 0 ) )
       return FAILURE;

     if ( !RunIns( exec ) )
     {
       ins->error = exec->error;
       return FAILURE;
     }

     exec->default_GS = exec->GS;
   }

   return SUCCESS;
 }


 Bool  Run_Context( PInstance_Record ins )
 {
   PExecution_Context  exec;

   Bool result;

   exec = (PExecution_Context)ins->exec;

   if ( !Goto_CodeRange( exec, TT_CodeRange_Glyph, 0 ) )
     return FAILURE;

   exec->pts = ins->pts;
   exec->numContours = ins->numContours;

   exec->zp0 = exec->pts;
   exec->zp1 = exec->pts;
   exec->zp2 = exec->pts;

   exec->GS  = exec->default_GS;

   exec->GS.gep0 = 1;
   exec->GS.gep1 = 1;
   exec->GS.gep2 = 1;

   exec->GS.projVector.x = 0x4000;
   exec->GS.projVector.y = 0x0000;
   exec->GS.freeVector.x = 0x4000;
   exec->GS.freeVector.y = 0x0000;
   exec->GS.dualVector.x = 0x4000;
   exec->GS.dualVector.y = 0x0000;

   exec->GS.round_state = 1;

   exec->top = 0;
   /* some glyphs leave something on the stack !! */
   /* we must empty it                            */

   result = RunIns( exec );

   ins->error = exec->error;

   return result;
 }
