/*
 * Copyright (C) 2000-2001 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: utils.c,v 1.4 2001/11/30 00:53:51 f1rmb Exp $
 *
 */
#define _POSIX_PTHREAD_SEMANTICS 1      /* for 5-arg getpwuid_r on solaris */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <pwd.h>
#include <time.h>
#include <sys/types.h>
#include <pthread.h>

#include "compat.h"

/*
 *
 */
void *xine_xmalloc(size_t size) {
  void *ptr;

  if((ptr = malloc(size)) == NULL) {
    fprintf(stderr, "%s: malloc() failed: %s.\n",
            __XINE_FUNCTION__, strerror(errno));
    return NULL;
  }

  memset(ptr, 0, size);

  return ptr;
}

typedef struct mem_aligned_s {
  struct mem_aligned_s *next;
  void *ptr;
  void *aligned_ptr;
} mem_aligned_t;

static mem_aligned_t *mem_aligned;
static pthread_mutex_t mem_aligned_mutex;

/*
 *
 */
void *xine_xmalloc_aligned (size_t alignment, size_t size) {
  char *pMem, *ptr;
  mem_aligned_t *reg;

  ptr = pMem = xine_xmalloc (size+alignment);

  while ((int) pMem % alignment)
    pMem++;

  pthread_mutex_lock (&mem_aligned_mutex);
  reg = malloc( sizeof(mem_aligned_t) );
  reg->next = mem_aligned;
  reg->ptr = ptr;
  reg->aligned_ptr = pMem;
  mem_aligned = reg;
  pthread_mutex_unlock (&mem_aligned_mutex);

  return pMem;
}


void xine_free_aligned( void *p ) {
  mem_aligned_t *reg, *last;

  pthread_mutex_lock (&mem_aligned_mutex);

  last = NULL;
  reg = mem_aligned;
  while( reg != NULL ) {
    if ( reg->aligned_ptr == p ) {
      free( reg->ptr );
      if( last )
        last->next = reg->next;
      else
        mem_aligned = reg->next;
      free( reg );
      pthread_mutex_unlock (&mem_aligned_mutex);
      return;
    }
    last = reg;
    reg = reg->next;
  }
  pthread_mutex_unlock (&mem_aligned_mutex);
}


void xine_init_mem_aligned(void) {
  mem_aligned = NULL;
  pthread_mutex_init (&mem_aligned_mutex, NULL);
}

/*
 *
 */
const char *xine_get_homedir(void) {
  struct passwd *pw = NULL;
  char *homedir = NULL;
#ifdef HAVE_GETPWUID_R
  int ret;
  struct passwd pwd;
  char *buffer = NULL;
  int bufsize = 128;

  buffer = (char *) xine_xmalloc(bufsize);

  if((ret = getpwuid_r(getuid(), &pwd, buffer, bufsize, &pw)) < 0) {
#else
  if((pw = getpwuid(getuid())) == NULL) {
#endif
    if((homedir = getenv("HOME")) == NULL) {
      fprintf(stderr, "Unable to get home directory, set it to /tmp.\n");
      homedir = strdup("/tmp");
    }
  }
  else {
    if(pw)
      homedir = strdup(pw->pw_dir);
  }


#ifdef HAVE_GETPWUID_R
  if(buffer)
    free(buffer);
#endif

  return homedir;
}

/*
 *
 */
char *xine_chomp(char *str) {
  char *pbuf;

  pbuf = str;

  while(*pbuf != '\0') pbuf++;

  while(pbuf > str) {
    if(*pbuf == '\r' || *pbuf == '\n' || *pbuf == '"') pbuf = '\0';
    pbuf--;
  }

  while(*pbuf == '=' || *pbuf == '"') pbuf++;

  return pbuf;
}


/*
 * A thread-safe usecond sleep
 */
void xine_usec_sleep(unsigned usec) {
#if HAVE_NANOSLEEP
  /* nanosleep is prefered on solaris, because it's mt-safe */
  struct timespec ts;

  ts.tv_sec =   usec / 1000000;
  ts.tv_nsec = (usec % 1000000) * 1000;
  nansleep(&ts, NULL);
#else
#ifdef __EMX__
  _sleep2(usec);
#else
  usleep(usec);
#endif
#endif
}

