/* 

LBMix - A mixer for Crystal Semiconductor IOCTL90 by Lesha Bogdanow
Copyright (C) 1999-2000  Lesha Bogdanow

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */

#include "LBMix.h"
#include "ioctl90.h"
#include "mixerapi.h"
#include "PipeMix.h"
#include "call32.h"

#define PIPE_WAIT 200		// Wait for pipe (ms)
#define DEF_INFO_BUF_SIZE 1024	// Size for info buffer if driver does not return info

// Additional flags for mixerapi
#define FLAG_MUTE 1
#define FLAG_RELEASE 2

// User window messages parameters
#define USER_INIT 0
#define USER_QUERY 1
#define USER_CHILDFONTCHANGE 2
#define USER_CHILDBGNDCHANGE 3
#define USER_CHILDFGNDCHANGE 4
#define USER_RECREATE 5


HAB   hab;			    /* anchor block for the process */
HMQ   hmq;		    /* handle to the process' message queue */
HWND  hwndMain=0;			/* handle to the main client window */
HWND  hwndMainFrame=0;		/* handle to the main window frame */


// Definitions for Controls structure
#define FLAG_NONE	0	// Dummy
#define FLAG_ONLYLEFT	1	// Only left control is used
#define FLAG_STEREO	2	// Both is valid. 
#define FLAG_MUTABLE	4	// Mute is valid
#define FLAG_LOCKABLE	8	// Can be locked, lock is valid
#define FLAG_MULTIPLE	0x10	// This input selector supports multiple selections
#define FLAG_ONLYRIGHT	0x20	// Only right control is used

// Flags for .both
#define BOTH_BOTH	1	// Set by main window checkbox
#define BOTH_MONO	2	// Set from setup

#define NO_INPUT_SEL    0	/* Value to place into .sel field when control
				   does not appear in input selector */
enum CtlTypes{
   TYPE_DEFAULT,
   TYPE_SELECTOR,
   TYPE_3D,
   TYPE_TONE};

struct t_Controls{
   int setfunc;		// IOCTL set function no
   int qfunc;		// IOCTL query function no
   int flags;		// Flags, see above
   enum CtlTypes type;	// Control type
   int sel;		// Value representing the control in input selector or NO_INPUT_SEL
   HWND hwndGroup,hwndLeft,hwndRight,hwndMute,hwndBoth;	// Window handles of controls
   HWND hwndLock,hwndLdesc,hwndRdesc,hwndRect;		// ...continued
   int activeID,nameID,defnameID,monoID;		// Settings ctl IDs, default name string ID, mono ID
   LONG FgColor,BgColor,SlLColor,SlRColor;		// Colors. For listbox: SlLColor/SlRColor = bgnd/fgnd.
   int supported;					// TRUE if supported
   int active;						// TRUE if active
   unsigned char name[32];				// name
   int left,right,mute,both,lock;			// Current values
   } Controls[]={ 
{1,0x11,FLAG_STEREO,TYPE_DEFAULT,NO_INPUT_SEL,			// Master
0,0,0,0,0,0,0,0,0,
IDC_MASTACT,IDC_MASTNAME,IDS_MASTER,IDC_MASTMONO,
-1,-1,-1,-1,
1,0,"",100,100,0,0,0},					// Unactive on default
{0x4D,0x6D,FLAG_STEREO|FLAG_MUTABLE|FLAG_LOCKABLE,TYPE_DEFAULT,NO_INPUT_SEL,	// DAC
0,0,0,0,0,0,0,0,0,
IDC_DACACT,IDC_DACNAME,IDS_DAC,IDC_DACMONO,
-1,-1,-1,-1,
1,1,"",100,100,0,0,0},
{0x43,0x63,FLAG_STEREO|FLAG_MUTABLE,TYPE_DEFAULT,I90SRC_LINE,		// Line
0,0,0,0,0,0,0,0,0,
IDC_LINEACT,IDC_LINENAME,IDS_LINE,IDC_LINEMONO,
-1,-1,-1,-1,
1,1,"",100,100,0,0,0},
{0x44,0x64,FLAG_STEREO|FLAG_MUTABLE,TYPE_DEFAULT,I90SRC_CD,		// CD
0,0,0,0,0,0,0,0,0,
IDC_CDACT,IDC_CDNAME,IDS_CD,IDC_CDMONO,
-1,-1,-1,-1,
1,1,"",100,100,0,0,0},
{0x42,0x62,FLAG_STEREO|FLAG_MUTABLE,TYPE_DEFAULT,I90SRC_MIC,		// Mic
0,0,0,0,0,0,0,0,0,
IDC_MICACT,IDC_MICNAME,IDS_MIC,IDC_MICMONO,
-1,-1,-1,-1,
1,1,"",100,100,0,0,0},
{0x41,0x61,FLAG_STEREO|FLAG_MUTABLE,TYPE_DEFAULT,I90SRC_PHONE,		// Phone
0,0,0,0,0,0,0,0,0,
IDC_PHONEACT,IDC_PHONENAME,IDS_PHONE,IDC_PHONEMONO,
-1,-1,-1,-1,
1,1,"",100,100,0,0,0},
{0x45,0x65,FLAG_STEREO|FLAG_MUTABLE,TYPE_DEFAULT,I90SRC_VIDEO,		// Video
0,0,0,0,0,0,0,0,0,
IDC_VIDEOACT,IDC_VIDEONAME,IDS_VIDEO,IDC_VIDEOMONO,
-1,-1,-1,-1,
1,1,"",100,100,0,0,0},
{0x46,0x66,FLAG_STEREO|FLAG_MUTABLE,TYPE_DEFAULT,I90SRC_AUX,		// Aux
0,0,0,0,0,0,0,0,0,
IDC_AUXACT,IDC_AUXNAME,IDS_AUX,IDC_AUXMONO,
-1,-1,-1,-1,
1,1,"",100,100,0,0,0},
{0x40,0x60,FLAG_ONLYLEFT|FLAG_MUTABLE,TYPE_DEFAULT,NO_INPUT_SEL,	// MonoIn
0,0,0,0,0,0,0,0,0,
IDC_MONOACT,IDC_MONONAME,IDS_MONOIN,0,
-1,-1,-1,-1,
1,1,"",100,100,0,0,0},
{0x4C,0x6C,FLAG_MUTABLE,TYPE_3D,NO_INPUT_SEL,				// 3D
0,0,0,0,0,0,0,0,0,
IDC_3DACT,IDC_3DNAME,IDS_3D,0,
-1,-1,-1,-1,
1,1,"",100,100,1,0,0},
{0x4B,0x6B,FLAG_NONE,TYPE_TONE,NO_INPUT_SEL,				// Tone
0,0,0,0,0,0,0,0,0,
IDC_TONEACT,IDC_TONENAME,IDS_TONE,0,
-1,-1,-1,-1,
1,1,"",50,50,0,0,0},
{0x4F,0x6F,FLAG_ONLYLEFT|FLAG_MUTABLE|FLAG_LOCKABLE,TYPE_DEFAULT,NO_INPUT_SEL,	// ADC
0,0,0,0,0,0,0,0,0,
IDC_ADCACT,IDC_ADCNAME,IDS_ADC,0,
-1,-1,-1,-1,
1,0,"",100,100,0,0,0},		// ADC is marked inactive initially
{0x4E,0x6E,FLAG_LOCKABLE|FLAG_ONLYLEFT,TYPE_SELECTOR,NO_INPUT_SEL,	// Source (must be the last one)
0,0,0,0,0,0,0,0,0,
IDC_SELACT,IDC_SELNAME,IDS_INPUTSEL,0,
-1,-1,-1,-1,
1,1,"",100,0,0,0,0}
};
int NControls=13;		// Total number of controls

int	ApiLevel=0;			// ApiLevel
int	RestoreState=FALSE;		// Restore mixer settings on startup
int	AutoMini=FALSE;			// Minimize on losing focus
int	ChildDlg=FALSE;			// True while displaying a child window

char *InfoBuf=NULL;
size_t InfoBufSize=0;

#define MAX_APP_NAME_LEN 32
CHAR	szIniApp[MAX_APP_NAME_LEN]="LBMix";
int	UsingPipe=FALSE;
int	PipeSupport=FALSE;		// Provide pipe API
TID	PipeTID=0;			// TID for pipe server
HPIPE	hPipe;				// Pipe handle for pipe server
char	szPipeName[CCHMAXPATH]="\\PIPE\\MIXER";
int	Changing=FALSE;		// Set to TRUE while setting sliders
#define MAX_INST_NAME 64
char	szInstName[MAX_INST_NAME+1]="";		// Instance name
int	CellsInLine=DEF_CELLS_IN_LINE;
int	AutoExit=FALSE;
int	ShowMenu=TRUE;
int	ShowTitlebar=TRUE;
CHAR	szIconFile[CCHMAXPATH]="";
HPOINTER hIcon=0;
int	NarrowSliders=FALSE;
int	NSWidth;			// Narrow slider width
int	ShortSliders=FALSE;

char szLock[32];
char szMute[32];
char szOff[32];
char szBoth[32];
char szSp[32];
char szCt[32];
char szLF[32];
char szHF[32];

PFNWP	OldButtonWndProc,OldStaticWndProc,OldListboxWndProc,OldSliderWndProc;

struct t_Batch {
   struct t_Batch *next;
   char name[32];
   int leftop;			// -1 - decrease, 0 - set, 1 - increase
   int leftval;
   int rightop;
   int rightval;
   int mute;			// -1 - unmute, 0 - none, 1 - mute
   int lock;			// -1 - unlock, 0 - none, 1 - lock
   } *Batch=NULL;

const CHAR	szKeyWindowPos[]="WindowPos";
const CHAR	szKeyRestore[]="RestoreState";
const CHAR	szKeyAutoMini[]="AutoMinimize";
const CHAR      szLeftSuf[]="Left";
const CHAR      szRightSuf[]="Right";
const CHAR      szMuteSuf[]="Mute";
const CHAR      szBothSuf[]="Both";
const CHAR      szLockSuf[]="Lock";
const CHAR      szActiveSuf[]="Active";
const CHAR      szNameSuf[]="Name";
const CHAR      szForeGndSuf[]="FGndColor";
const CHAR      szBackGndSuf[]="BGndColor";
const CHAR      szSlLClrSuf[]="SlLColor";
const CHAR      szSlRClrSuf[]="SlRColor";
const CHAR      szCtlPref[]="Ctl";
const CHAR      szKeyInstName[]="InstanceName";
const CHAR      szKeyCellsInLine[]="CellsInLine";
const CHAR      szKeyShowMenu[]="ShowMenu";
const CHAR      szKeyShowTitlebar[]="ShowTitlebar";
const CHAR      szKeyIconFile[]="IconFile";
const CHAR      szKeyLockName[]="LockName";
const CHAR      szKeyMuteName[]="MuteName";
const CHAR      szKeyOffName[]="OffName";
const CHAR      szKeyBothName[]="BothName";
const CHAR      szKeyLFName[]="LFName";
const CHAR      szKeyHFName[]="HFName";
const CHAR      szKeySpName[]="SpName";
const CHAR      szKeyCtName[]="CtName";
const CHAR      szKeyNarrowSliders[]="NarrowSliders";
const CHAR      szKeyShortSliders[]="ShortSliders";

int PipeCommand(char *Buff,ULONG Size) {
   ULONG ulRC;
   HFILE hFile;
   ULONG A;
   ULONG rc;

   rc=DosWaitNPipe(szPipeName,PIPE_WAIT);
   if (rc) return rc;
   rc=DosOpen(szPipeName,&hFile,&A,0,0,OPEN_ACTION_OPEN_IF_EXISTS,
                OPEN_ACCESS_READWRITE|OPEN_SHARE_DENYNONE,NULL);
   if (rc) return rc;
   rc=DosWrite(hFile,Buff,strlen(Buff),&A);
   if (rc) return rc;
   rc=DosRead(hFile,Buff,Size,&A);		// This is WRONG
   if (rc) return rc;
   Buff[A]=0;
   return DosClose(hFile);
   }


// Set Master volume/balance (0-0x7FFF) by calling PDD IOCTL
static int MasterSetVB(unsigned short vol, unsigned short bal) {
   MCI_TRACK_INFO TrackInfo;
   MCI_AUDIO_CHANGE AudioChange;
   MCI_AUDIO_CONTROL AudioCtl;
   ULONG ulSize,ulRC;
   void * _Seg16 pul16;                   // defines a 16:16 pointer
   extern HFILE hDriver;

   TrackInfo.usMasterVolume=vol;
   TrackInfo.usDitherPct=0;		// ???
   TrackInfo.usMasterVolumeDelay=0;
   TrackInfo.usMasterBalance=bal;
   TrackInfo.usMasterBalanceDelay=0;
   pul16=(void * _Seg16)FlatToSel((unsigned long)(&TrackInfo));
   AudioChange.pvDevInfo=pul16;
   AudioChange.prMoreInputs=NULL;
   AudioChange.prMoreOutputs=NULL;
   AudioChange.pvModeInfo=NULL;
   AudioCtl.usIOCtlRequest=AUDIO_CHANGE;
   pul16=(void * _Seg16)FlatToSel((unsigned long)(&AudioChange));
   AudioCtl.pbRequestInfo=pul16;
   AudioCtl.ulPosition=0;

   ulSize=sizeof(AudioCtl);
   ulRC = DosDevIOCtl(hDriver,AUDIO_IOCTL_CAT,AUDIO_CONTROL,
                         NULL,0,NULL,&AudioCtl,ulSize,&ulSize);
   return ulRC;
   }

// Calculate value for master volume/balance
static unsigned short MasterVal(int v) {
   unsigned short rc;

   if (v<0) v=0;
   rc=(v<<15)/100;
   if (rc>0x7FFF) rc=0x7FFF;
   return rc;
   }

// Set Master volume by calling PDD IOCTL
static int MasterSet(MIXSTRUCT *MixStruct) {
   int v,b;

   if (MixStruct->VolumeL>MixStruct->VolumeR) v=MixStruct->VolumeL;
   else v=MixStruct->VolumeR;
   b=(100+MixStruct->VolumeL-MixStruct->VolumeR)/2;	// Balance
   return MasterSetVB(MasterVal(v),MasterVal(b));
   }

int CallSet(int Fn,MIXSTRUCT *MixStruct) {
   char Buff[32];
   int rc;

   if (!Fn) {
      mciSetSysValue(MSV_MASTERVOLUME,&(MixStruct->VolumeL));
      sprintf(Buff,"MASTERAUDIO VOLUME %d",MixStruct->VolumeL);
      return mciSendString(Buff,NULL,0,0,0);
      }
   if (!UsingPipe) {
      if (Fn<0x20) return MasterSet(MixStruct);
      else return mixerapiIOCTL90(Fn,MixStruct,sizeof(MIXSTRUCT));
      }
   sprintf(Buff,"%x %d %d %x",Fn,MixStruct->VolumeL,MixStruct->VolumeR,
           MixStruct->Mute);
   rc=PipeCommand(Buff,31);
   if (rc) return rc;
// Should check for REP_OK here. Though this return code is never checked
   return 0;   
   }

int CallQuery(int Fn,MIXSTRUCT *MixStruct) {
   char Buff[32];
   int rc;

   if (!Fn) {
      mciQuerySysValue(MSV_MASTERVOLUME,&(MixStruct->VolumeL));
      return 0;
      }
   if (!UsingPipe) {
      if (Fn<0x20) return 0;
      else return mixerapiIOCTL90(Fn,MixStruct,sizeof(MIXSTRUCT));
      }
   _itoa(Fn,Buff,16);
   rc=PipeCommand(Buff,31);
   if (rc) return rc;
   rc=sscanf(Buff,"%x %d %d %x",&Fn,&MixStruct->VolumeL,&MixStruct->VolumeR,
             &MixStruct->Mute);
   if (rc!=4) return 1;
   return 0;
   }

void SetControl(int no) {
   MIXSTRUCT MixStruct;

   if (Controls[no].supported&&Controls[no].active) {
      MixStruct.VolumeL=Controls[no].left;
      if (Controls[no].flags&FLAG_ONLYLEFT) MixStruct.VolumeR=0;
      else MixStruct.VolumeR=Controls[no].right;
      if (Controls[no].mute) MixStruct.Mute=FLAG_MUTE;
      else MixStruct.Mute=0;
      if (Controls[no].flags&FLAG_LOCKABLE) {
         if (ApiLevel>1) {
            if (!Controls[no].lock) MixStruct.Mute|=FLAG_RELEASE;
            }
         else Controls[no].lock=TRUE;
         }
      CallSet(Controls[no].setfunc,&MixStruct);
      }
   }
void DisplayControl(int no) {
   int EnabledL,i,j;
   HWND hwndList;

   if (!(Controls[no].supported&&Controls[no].active)) return;
   Changing=TRUE;
   if (Controls[no].flags&FLAG_LOCKABLE) EnabledL=Controls[no].lock;
   else EnabledL=TRUE;

   if (!(Controls[no].flags&FLAG_ONLYRIGHT)) {
      WinEnableWindow(Controls[no].hwndLeft,EnabledL);
      if (Controls[no].supported) {
         if (Controls[no].type==TYPE_SELECTOR) {
            hwndList=Controls[no].hwndLeft;
            for (i=0,j=0;i<NControls;i++) if ((Controls[i].sel!=NO_INPUT_SEL)&&(Controls[i].supported)) {
               if (Controls[i].sel&Controls[no].left)
                  WinSendMsg(hwndList,LM_SELECTITEM,(MPARAM)j,(MPARAM)TRUE);
               else
                  WinSendMsg(hwndList,LM_SELECTITEM,(MPARAM)j,(MPARAM)FALSE);
               j++;
               }
            }
         else {
            j=Controls[no].left;
            if (ShortSliders) j/=2;
            WinSendMsg(Controls[no].hwndLeft,
               SLM_SETSLIDERINFO,MPFROM2SHORT(SMA_SLIDERARMPOSITION,
               SMA_INCREMENTVALUE), (MPARAM)j);
            }
         }
      }
   if (!(Controls[no].flags&FLAG_ONLYLEFT)) {
      WinEnableWindow(Controls[no].hwndRight,EnabledL);
      if (Controls[no].supported) {
         j=Controls[no].right;
         if (ShortSliders) j/=2;
         WinSendMsg(Controls[no].hwndRight,
            SLM_SETSLIDERINFO,MPFROM2SHORT(SMA_SLIDERARMPOSITION,
            SMA_INCREMENTVALUE), (MPARAM)j);
         }
      }
   if (Controls[no].flags&FLAG_MUTABLE) {
      WinEnableWindow(Controls[no].hwndMute,EnabledL);
      if (Controls[no].supported) WinSendMsg(Controls[no].hwndMute,BM_SETCHECK,
            (MPARAM)Controls[no].mute,0);
      }
   if (Controls[no].flags&FLAG_STEREO) {
      if (Controls[no].supported) WinSendMsg(Controls[no].hwndBoth,BM_SETCHECK,
            (MPARAM)Controls[no].both,0);
      }
   if (Controls[no].flags&FLAG_LOCKABLE) {
      if ((ApiLevel<=1)&&(Controls[no].lock))
         WinEnableWindow(Controls[no].hwndLock,FALSE);
      else
         WinEnableWindow(Controls[no].hwndLock,TRUE);
      if (Controls[no].supported) WinSendMsg(Controls[no].hwndLock,BM_SETCHECK,
            (MPARAM)Controls[no].lock,0);
      }
   Changing=FALSE;
   return;
   }

void QueryControl(int no) {
   MIXSTRUCT MixStruct;

   if (Controls[no].supported) {
      MixStruct.VolumeL=Controls[no].left;
      MixStruct.VolumeR=Controls[no].right;
      if (!CallQuery(Controls[no].qfunc,&MixStruct)) {
         Controls[no].left=MixStruct.VolumeL;
         Controls[no].right=MixStruct.VolumeR;
         Controls[no].mute=(MixStruct.Mute&FLAG_MUTE);
         if (Controls[no].flags&FLAG_LOCKABLE)
            Controls[no].lock=!(MixStruct.Mute&FLAG_RELEASE);
         }
      }
   return;
   }

void SetControls() {
   int i;

   for (i=0;i<NControls;i++) SetControl(i);
   return;
   }
void DisplayControls() {
   int i;

   for (i=0;i<NControls;i++) DisplayControl(i);
   return;
   }

void QueryControls() {
   int i;

   for (i=0;i<NControls;i++) QueryControl(i);
   return;
   }

// Hide titlebar if it is disabled
void HideTitle() {
   SWP swp,swpFrame,swpMenu;
   HWND hwndMenu;
   int cy;

   if (ShowTitlebar) return;

   if (ShowMenu) {
      hwndMenu=WinWindowFromID(hwndMainFrame,FID_MENU);
      WinQueryWindowPos(hwndMenu,&swpMenu);
      }

   WinQueryWindowPos(hwndMain,&swp);
   WinQueryWindowPos(hwndMainFrame,&swpFrame);
   cy=swpFrame.cy-2*swp.y;
   if (ShowMenu) {
      WinSetWindowPos(hwndMenu,HWND_TOP,swpMenu.x,
                   swpFrame.cy-swp.y-swpMenu.cy,
                   swpMenu.cx, swpMenu.cy,SWP_SIZE|SWP_MOVE|SWP_ZORDER);
      cy-=swpMenu.cy;
      }
   WinSetWindowPos(hwndMain,HWND_TOP,swp.x,swp.y,swpFrame.cx-2*swp.x,
                   cy,SWP_SIZE|SWP_MOVE|SWP_ZORDER);
   }

void SetClipSiblings(HWND hwnd) {
   ULONG ul;

   ul=WinQueryWindowULong(hwnd,QWL_STYLE);
   ul|=WS_CLIPSIBLINGS;
   WinSetWindowULong(hwnd,QWL_STYLE,ul);
   return;
   }

// Query control colors and store them into Controls array
void QueryCtlColors() {
   int i;

   for (i=0;i<NControls;i++) if (Controls[i].hwndRect) {
      if (!WinQueryPresParam(Controls[i].hwndRect,PP_BACKGROUNDCOLOR,
                            PP_BACKGROUNDCOLORINDEX,NULL,sizeof(LONG),
                            &Controls[i].BgColor,QPF_ID2COLORINDEX))
         Controls[i].BgColor=-1;
      if (!WinQueryPresParam(Controls[i].hwndRect,PP_FOREGROUNDCOLOR,
                                 PP_FOREGROUNDCOLORINDEX,NULL,sizeof(LONG),
                                 &Controls[i].FgColor,QPF_ID2COLORINDEX))
         Controls[i].FgColor=-1;
      if (Controls[i].type==TYPE_SELECTOR) {
         if (Controls[i].hwndLeft) {
            if (!WinQueryPresParam(Controls[i].hwndLeft,PP_BACKGROUNDCOLOR,
                                 PP_BACKGROUNDCOLORINDEX,NULL,sizeof(LONG),
                                 &Controls[i].SlLColor,QPF_ID2COLORINDEX))
               Controls[i].SlLColor=-1;
            if (!WinQueryPresParam(Controls[i].hwndLeft,PP_FOREGROUNDCOLOR,
                                 PP_FOREGROUNDCOLORINDEX,NULL,sizeof(LONG),
                                 &Controls[i].SlRColor,QPF_ID2COLORINDEX))
               Controls[i].SlRColor=-1;
            }
         }
      else {
         if (Controls[i].hwndLeft) {
            if (!WinQueryPresParam(Controls[i].hwndLeft,PP_BUTTONBACKGROUNDCOLOR,
                                 0,NULL,sizeof(LONG),
                                 &Controls[i].SlLColor,0))
               Controls[i].SlLColor=-1;
            }
         if (Controls[i].hwndRight) {
            if (!WinQueryPresParam(Controls[i].hwndRight,PP_BUTTONBACKGROUNDCOLOR,
                                 0,NULL,sizeof(LONG),
                                 &Controls[i].SlRColor,0))
               Controls[i].SlRColor=-1;
            }
         }
      }
   }

// Create the main window
void MainCreate() {
   ULONG ctlData= FCF_TASKLIST | FCF_DLGBORDER | FCF_AUTOICON | FCF_ACCELTABLE
                  | FCF_ICON | FCF_TITLEBAR | FCF_SYSMENU;  
   char szName[MESSAGELEN];

   if (hwndMainFrame) {
      QueryCtlColors();
      WinStoreWindowPos((char *)szIniApp,(char *)szKeyWindowPos,hwndMainFrame);
      WinDestroyWindow(hwndMainFrame);
      hwndMainFrame=0;
      }
   if (hIcon) {
      WinFreeFileIcon(hIcon);
      hIcon=0;
      }

   WinLoadString(hab, 0, IDS_SWTITLE, MESSAGELEN, szName);
   if (*szInstName) {
      strncat(szName," - ",MESSAGELEN);
      strncat(szName,szInstName,MESSAGELEN);
      }
   if (ShowMenu) ctlData|=FCF_MENU;
   if (ShowTitlebar) ctlData|=FCF_MINBUTTON;
   hwndMainFrame = WinCreateStdWindow(HWND_DESKTOP,0,
			(PVOID)&ctlData,(PSZ)"MAIN",(PSZ)szName,0,
                        (HMODULE)NULL,ID_MAIN,(PHWND)&hwndMain);
   if (!ShowTitlebar) {
      SetClipSiblings(WinWindowFromID(hwndMainFrame,FID_TITLEBAR));
      SetClipSiblings(WinWindowFromID(hwndMainFrame,FID_SYSMENU));
      }
   if (*szIconFile) {
      hIcon=WinLoadFileIcon(szIconFile,FALSE);
      if (hIcon) WinSendMsg(hwndMainFrame,WM_SETICON,(MPARAM) hIcon,0);
      }
   return;
   }

void ResizeMain(int x,int y) {
   SWP swp;

   WinQueryWindowPos(hwndMainFrame,&swp);
   WinSetWindowPos(hwndMainFrame,0,swp.x,swp.y+swp.cy-y,
                   x,y,SWP_SIZE|SWP_MOVE);
   }

// Return string width in points
int StrWidth(HPS hps, char *str) {
   POINTL aptlPoints[TXTBOX_COUNT];
   int l,r;

   GpiQueryTextBox(hps,strlen(str),str,TXTBOX_COUNT,aptlPoints);
   if (aptlPoints[TXTBOX_TOPRIGHT].x>aptlPoints[TXTBOX_BOTTOMRIGHT].x)
      r=aptlPoints[TXTBOX_TOPRIGHT].x;
   else r=aptlPoints[TXTBOX_BOTTOMRIGHT].x;
   if (aptlPoints[TXTBOX_TOPLEFT].x<aptlPoints[TXTBOX_BOTTOMLEFT].x)
      l=aptlPoints[TXTBOX_TOPLEFT].x;
   else l=aptlPoints[TXTBOX_BOTTOMLEFT].x;
   return r-l;
   }

LONG RGBMix(LONG c1, LONG c2) {
   LONG rc;

   rc=((c1&0xFF)+(c2&0xFF))>>1;
   rc|=(((c1&0xFF00)+(c2&0xFF00))>>1)&0xFF00;
   rc|=(((c1&0xFF0000)+(c2&0xFF0000))>>1)&0xFF0000;
   return rc;
   }

// Create the main window controls
void MainWindow() {
   int i,no,xcells,ycells,x,y,t,actc;
   SLDCDATA sldcData;
   SWP swp;
   HPS	  hps;
   FONTMETRICS   fmMetrics;
   PPRESPARAMS ppres;
   LONG Dark,Light;

// Control layout flags
#define CLF_LOCK 1
#define CLF_MUTE 2
#define CLF_OFF  4
#define CLF_BOTH 8
#define CLF_LDLF 0x10
#define CLF_RDHF 0x20
#define CLF_LISTS 0x40			// List - single sel
#define CLF_LEFT 0x80
#define CLF_RIGHT 0x100
#define CLF_LDSP 0x200
#define CLF_RDCT 0x400
#define CLF_LISTM 0x800			// List - multi sel
   struct t_CtlLay{			// Control layout structure
      int no;				// Control no
      int celly;			// y in cells upper = 1, lower = 2
      int flags;
      int checkx,checkw;		// Checkbox x, max width
      int dscw;				// Description max width
      int ldscx;
      int rdscx;
      int leftx;			// Or listbox
      int rightx;
      int boxw;
      } *CtlLay;
   int sliderw, listw;			/* Slider, listbox widths */
   int sliderh,listh,checkh,boxh;	/* Cell slider, listbox, checkbox/
                                           description, groupbox heights */
   int locky,mutey,bothy,slidery;	/* Box, lock, mute/off/listbox,
                                           both/description, slider y */
   int spacex,spacey;			/* Inter-control spacing */
   int cellspx,cellspy;			/* Inter-cell spacing */
   int fieldx,fieldy;		/* Field left/right, top/bottom */
   int interx,interb,intert;	/* Interior field left/right, top, bottom */

   int borderw,borderh;		/* Window border width/height */


   borderw=WinQuerySysValue(HWND_DESKTOP,SV_CXDLGFRAME);
   borderh=WinQuerySysValue(HWND_DESKTOP,SV_CYDLGFRAME);

   HideTitle();

   actc=0;
   for (no=0;no<NControls;no++)	{			// Count active controls
      if (Controls[no].supported&&Controls[no].active) actc++;
      Controls[no].hwndGroup=0;
      Controls[no].hwndLeft=0;
      Controls[no].hwndRight=0;
      Controls[no].hwndMute=0;
      Controls[no].hwndBoth=0;
      Controls[no].hwndLock=0;
      Controls[no].hwndLdesc=0;
      Controls[no].hwndRdesc=0;
      Controls[no].hwndRect=0;
      }
   if (!actc) {
      ResizeMain(EMPTY_W,EMPTY_H);
      return;
      }
   CtlLay=malloc(actc*sizeof(struct t_CtlLay));
   xcells=actc;
   ycells=1;
   if (xcells>CellsInLine) {
      xcells=xcells/2+xcells%2;
      ycells++;
      }

   for (x=0,y=1,i=0,no=0;no<NControls;no++)		// Distribute controls
      if (Controls[no].supported&&Controls[no].active) {
         CtlLay[i].no=no;
         CtlLay[i].celly=y;
         i++;
         x++;
         if (x>=xcells) {
            x=0;
            y++;
            }
         }

/* Sizes common for all controls */
    hps = WinGetPS(hwndMain);
    GpiQueryFontMetrics (hps,sizeof (FONTMETRICS),&fmMetrics);
#define FONTH (fmMetrics.lMaxBaselineExt)
#define FONTW (fmMetrics.lAveCharWidth)

    if (NarrowSliders) sliderw=NSWidth;
    else sliderw=SLIDER_W_S;
    if (ShortSliders) sliderh=SLIDER_H_S;
    else sliderh=SLIDER_H_N;

    listw=LIST_W_O+LIST_W_M*FONTW;

    checkh=FONTH+FONTH/CHECK_H_D;
    if (checkh<CHECK_H_MIN) checkh=CHECK_H_MIN;

    spacex=SPACE_X_O+FONTW/SPACE_X_D;
    spacey=SPACE_Y_O+FONTH/SPACE_Y_D;    

    cellspx=CELLSP_X_O+FONTW*CELLSP_X_M/CELLSP_X_D;
    cellspy=CELLSP_Y_O+FONTH*CELLSP_Y_M/CELLSP_Y_D;

    fieldx=FIELD_X_O+FONTW*FIELD_X_M/FIELD_X_D;
    fieldy=FIELD_Y_O+FONTH*FIELD_Y_M/FIELD_Y_D;

    interx=INTER_X_O+FONTW*INTER_X_M/INTER_X_D;
    intert=INTER_T_O+FONTH*INTER_T_M/INTER_T_D;
    interb=INTER_B_O+FONTH*INTER_B_M/INTER_B_D;

    locky=interb;
    mutey=locky+spacey+checkh;
    bothy=mutey+spacey+checkh;
    slidery=bothy+spacey+checkh;
    listh=3*spacey+2*checkh+sliderh;
    boxh=slidery+sliderh+intert;

    for(i=0;i<actc;i++) {		// Calculate flags/control sizes

       t=0;				// Calculate flags
       no=CtlLay[i].no;
       if (Controls[no].flags&FLAG_LOCKABLE) t|=CLF_LOCK;
       if (Controls[no].type==TYPE_SELECTOR) {
          if (Controls[no].flags&FLAG_MULTIPLE) t|=CLF_LISTM;
          else t|=CLF_LISTS;
          }
       else if ((Controls[no].type==TYPE_3D)||(Controls[no].type==TYPE_TONE)){ 
          if (Controls[no].flags&FLAG_MUTABLE) t|=CLF_OFF;
          if (!(Controls[no].flags&FLAG_ONLYRIGHT)) {
             t|=CLF_LEFT;
             if (Controls[no].type==TYPE_3D) t|=CLF_LDSP;
             else t|=CLF_LDLF;
             }
          if (!(Controls[no].flags&FLAG_ONLYLEFT)) {
             t|=CLF_RIGHT;
             if (Controls[no].type==TYPE_3D) t|=CLF_RDCT;
             else t|=CLF_RDHF;
             }
          }
       else {			// default control type
          if (Controls[no].flags&FLAG_MUTABLE) t|=CLF_MUTE;
          if (!(Controls[no].both&BOTH_MONO)&&
             (Controls[no].flags&FLAG_STEREO)) t|=CLF_BOTH;
          if (!(Controls[no].flags&FLAG_ONLYRIGHT)) t|=CLF_LEFT;
          if (!(Controls[no].both&BOTH_MONO)&&
             !(Controls[no].flags&FLAG_ONLYLEFT))  t|=CLF_RIGHT;
          }
       CtlLay[i].flags=t;

// Calculate control sizes

      CtlLay[i].boxw=BOX_W_O+FONTW*BOX_W_M/BOX_W_D+StrWidth(hps,Controls[no].name);
      if (t&CLF_LOCK)
         CtlLay[i].checkw=StrWidth(hps,szLock);
      else CtlLay[i].checkw=0;
      if (t&CLF_OFF) {
         x=StrWidth(hps,szOff);
         if (x>CtlLay[i].checkw) CtlLay[i].checkw=x;
         }
      if (t&CLF_MUTE) {
         x=StrWidth(hps,szMute);
         if (x>CtlLay[i].checkw) CtlLay[i].checkw=x;
         }
      if (t&CLF_BOTH) {
         x=StrWidth(hps,szBoth);
         if (x>CtlLay[i].checkw) CtlLay[i].checkw=x;
         }
      CtlLay[i].checkw+=CHECK_W_O+FONTW*CHECK_W_M/CHECK_W_D;
      if (t&CLF_LDLF) 
         CtlLay[i].dscw=StrWidth(hps,szLF);
      else CtlLay[i].dscw=0;
      if (t&CLF_LDSP) {
         x=StrWidth(hps,szSp);
         if (x>CtlLay[i].dscw) CtlLay[i].dscw=x;
         }
      if (t&CLF_RDHF) {
         x=StrWidth(hps,szHF);
         if (x>CtlLay[i].dscw) CtlLay[i].dscw=x;
         }
      if (t&CLF_RDCT) {
         x=StrWidth(hps,szCt);
         if (x>CtlLay[i].dscw) CtlLay[i].dscw=x;
         }
      CtlLay[i].dscw+=DESC_W_O+FONTW*DESC_W_M/DESC_W_D;

// Calculate box size

      if ((t&(CLF_LOCK|CLF_MUTE|CLF_OFF|CLF_BOTH))
         &&(CtlLay[i].checkw>CtlLay[i].boxw)) CtlLay[i].boxw=CtlLay[i].checkw;
      if ((t&(CLF_LISTS|CLF_LISTM))&&(listw>CtlLay[i].boxw))
            CtlLay[i].boxw=listw;
      else {
         if (t&(CLF_LDLF|CLF_LDSP|CLF_RDHF|CLF_RDCT)) {
            if ((t&(CLF_LDLF|CLF_LDSP))&&(t&(CLF_RDHF|CLF_RDCT))) {
               if ((2*CtlLay[i].dscw+spacex)>CtlLay[i].boxw)
                  CtlLay[i].boxw=2*CtlLay[i].dscw+spacex;
               }
            else {
               if (CtlLay[i].dscw>CtlLay[i].boxw)
                  CtlLay[i].boxw=CtlLay[i].dscw;
               }
            }
         if (t&(CLF_LEFT|CLF_RIGHT)) {
            if ((t&CLF_LEFT)&&(t&CLF_RIGHT)) {
               if ((2*sliderw+spacex)>CtlLay[i].boxw)
                  CtlLay[i].boxw=2*sliderw+spacex;
               }
            else if (sliderw>CtlLay[i].boxw)
                  CtlLay[i].boxw=sliderw;
            }
         }
      CtlLay[i].boxw+=2*interx;

// Calculate controls x positions

      if (t&(CLF_LOCK|CLF_MUTE|CLF_OFF|CLF_BOTH)) {
         CtlLay[i].checkx=(CtlLay[i].boxw-CtlLay[i].checkw)/2;
         if (CtlLay[i].checkx<interx) CtlLay[i].checkx=interx;
         }
      if (t&(CLF_LISTS|CLF_LISTM)) {
         CtlLay[i].leftx=(CtlLay[i].boxw-listw)/2;
         if (CtlLay[i].leftx<interx) CtlLay[i].leftx=interx;
         }
      else {
         if (t&(CLF_LDLF|CLF_LDSP|CLF_RDHF|CLF_RDCT)) {
            if ((t&(CLF_LDLF|CLF_LDSP))&&(t&(CLF_RDHF|CLF_RDCT))) {
               x=CtlLay[i].boxw/3;
               CtlLay[i].ldscx=x-CtlLay[i].dscw/2;
               CtlLay[i].rdscx=2*x-CtlLay[i].dscw/2;
               if ((CtlLay[i].rdscx-CtlLay[i].ldscx-CtlLay[i].dscw)<spacex) {
                  CtlLay[i].ldscx=(CtlLay[i].boxw-spacex)/2-CtlLay[i].dscw;
                  CtlLay[i].rdscx=(CtlLay[i].boxw+spacex)/2;
                  }
               if (CtlLay[i].ldscx<interx) CtlLay[i].ldscx=interx;
               if (CtlLay[i].rdscx>(CtlLay[i].boxw-interx-CtlLay[i].dscw))
                   CtlLay[i].rdscx=CtlLay[i].boxw-interx-CtlLay[i].dscw;
               }
            else if (t&(CLF_LDLF|CLF_LDSP)) {
               CtlLay[i].ldscx=(CtlLay[i].boxw-CtlLay[i].dscw)/2;
               if (CtlLay[i].ldscx<interx) CtlLay[i].ldscx=interx;
               }
            else {
               CtlLay[i].rdscx=(CtlLay[i].boxw-CtlLay[i].dscw)/2;
               if (CtlLay[i].rdscx<interx) CtlLay[i].rdscx=interx;
               }
            }
         if (t&(CLF_LEFT|CLF_RIGHT)) {
            if ((t&CLF_LEFT)&&(t&CLF_RIGHT)) {
               x=CtlLay[i].boxw/3;
               CtlLay[i].leftx=x-sliderw/2;
               CtlLay[i].rightx=2*x-sliderw/2;
               if ((CtlLay[i].rightx-CtlLay[i].leftx-sliderw)<spacex) {
                  CtlLay[i].leftx=(CtlLay[i].boxw-spacex)/2-sliderw;
                  CtlLay[i].rightx=(CtlLay[i].boxw+spacex)/2;
                  }
               if (CtlLay[i].leftx<interx) CtlLay[i].leftx=interx;
               if (CtlLay[i].rightx>(CtlLay[i].boxw-interx-sliderw))
                   CtlLay[i].rightx=CtlLay[i].boxw-interx-sliderw;
               }
            else if (t&CLF_LEFT) {
               CtlLay[i].leftx=(CtlLay[i].boxw-sliderw)/2;
               if (CtlLay[i].leftx<interx) CtlLay[i].leftx=interx;
               }
            else {
               CtlLay[i].rightx=(CtlLay[i].boxw-sliderw)/2;
               if (CtlLay[i].rightx<interx) CtlLay[i].rightx=interx;
               }
            }
         }
      }       // big layout calculation loop

// Calculate/set window size
   y=boxh*ycells+cellspy*(ycells-1)+2*fieldy+2*borderh;
   if (ShowTitlebar) {
      WinQueryWindowPos(WinWindowFromID(hwndMainFrame,FID_TITLEBAR),&swp);
      y+=swp.cy;
      }
   if (ShowMenu) {
      WinQueryWindowPos(WinWindowFromID(hwndMainFrame,FID_MENU),&swp);
      y+=swp.cy;
      }
   x=0;
   for (i=0;(i<actc)&&(CtlLay[i].celly==1);i++) {
      if (x) x+=cellspx;
      x+=CtlLay[i].boxw;
      }
   if (ycells>1) {
      t=0;
      for (;i<actc;i++) {
         if (t) t+=cellspx;
         t+=CtlLay[i].boxw;
         }
      if (t>x) x=t;
      }
   x+=2*fieldx+2*borderw;
   ResizeMain(x,y);
   WinReleasePS(hps);

   Changing=TRUE;

   x=fieldx;
   y=(ycells-1)*(boxh+cellspy)+fieldy;
   for(i=0;i<actc;i++) {		// Display controls
      no=CtlLay[i].no;
      if ((Controls[no].FgColor!=-1)||(Controls[no].BgColor!=-1)) {
         if ((Controls[no].FgColor!=-1)&&(Controls[no].BgColor!=-1)) {
            ppres=malloc(sizeof(ULONG)*7);
            ppres->cb=sizeof(ULONG)*6;
            ppres->aparam[0].id=PP_BACKGROUNDCOLOR;
            ppres->aparam[0].cb=sizeof(LONG);
            memcpy(&ppres->aparam[0].ab,&Controls[no].BgColor,sizeof(LONG));
            ppres->aparam[1].id=PP_FOREGROUNDCOLOR;
            ppres->aparam[1].cb=sizeof(LONG);
            memcpy(&ppres->aparam[1].ab,&Controls[no].FgColor,sizeof(LONG));
            }
         else {
            ppres=malloc(sizeof(ULONG)*4);
            ppres->cb=sizeof(ULONG)*3;
            ppres->aparam[0].cb=sizeof(LONG);
            if (Controls[no].FgColor!=-1) {
               ppres->aparam[0].id=PP_FOREGROUNDCOLOR;
               memcpy(&ppres->aparam[0].ab,&Controls[no].FgColor,sizeof(LONG));
               }
            else {
               ppres->aparam[0].id=PP_BACKGROUNDCOLOR;
               memcpy(&ppres->aparam[0].ab,&Controls[no].BgColor,sizeof(LONG));
               }
            }
         }
      else ppres=NULL; 
      Controls[no].hwndRect=WinCreateWindow(hwndMain,"CONTROL",NULL,
                       WS_VISIBLE,x,y,CtlLay[i].boxw,boxh,
                       hwndMain,HWND_TOP,0,NULL,ppres);
      if (ppres) free(ppres);
      Controls[no].hwndGroup=WinCreateWindow(Controls[no].hwndRect,WC_STATIC,Controls[no].name,
                       WS_VISIBLE|SS_GROUPBOX,0,0,CtlLay[i].boxw,boxh,
                       Controls[no].hwndRect,HWND_TOP,0,NULL,NULL);
      OldStaticWndProc=WinSubclassWindow(Controls[no].hwndGroup,StaticWndProc);
      if (CtlLay[i].flags&CLF_LOCK) {
         Controls[no].hwndLock=WinCreateWindow(Controls[no].hwndRect,WC_BUTTON,szLock,
                       WS_VISIBLE|BS_AUTOCHECKBOX|WS_TABSTOP,
                       CtlLay[i].checkx,locky,CtlLay[i].checkw,checkh,
                       Controls[no].hwndRect,HWND_TOP,IDC_LOCK|no,NULL,NULL);
         OldButtonWndProc=WinSubclassWindow(Controls[no].hwndLock,ButtonWndProc);
         }
      if (CtlLay[i].flags&(CLF_LISTS|CLF_LISTM)) {
         t=WS_VISIBLE|WS_TABSTOP;
         if (CtlLay[i].flags&CLF_LISTM) t|=LS_MULTIPLESEL;
         if (Controls[no].SlLColor!=-1) {
            Light=RGBMix(Controls[no].SlLColor,RGB_WHITE);
            Dark=RGBMix(Controls[no].SlLColor,RGB_BLACK);
            if (Controls[no].SlRColor!=-1) {
               ppres=malloc(sizeof(ULONG)*13);
               ppres->cb=sizeof(ULONG)*12;
               ppres->aparam[3].id=PP_FOREGROUNDCOLOR;
               ppres->aparam[3].cb=sizeof(LONG);
               memcpy(&ppres->aparam[3].ab,&Controls[no].SlRColor,sizeof(LONG));
               }
            else {
               ppres=malloc(sizeof(ULONG)*10);
               ppres->cb=sizeof(ULONG)*9;
               }
            ppres->aparam[0].id=PP_BACKGROUNDCOLOR;
            ppres->aparam[0].cb=sizeof(LONG);
            memcpy(&ppres->aparam[0].ab,&Controls[no].SlLColor,sizeof(LONG));
            ppres->aparam[1].id=PP_BORDERLIGHTCOLOR;
            ppres->aparam[1].cb=sizeof(LONG);
            memcpy(&ppres->aparam[1].ab,&Light,sizeof(LONG));
            ppres->aparam[2].id=PP_BORDERDARKCOLOR;
            ppres->aparam[2].cb=sizeof(LONG);
            memcpy(&ppres->aparam[2].ab,&Dark,sizeof(LONG));
            }
         else if (Controls[no].SlRColor!=-1) {
               ppres=malloc(sizeof(ULONG)*4);
               ppres->cb=sizeof(ULONG)*3;
               ppres->aparam[0].id=PP_FOREGROUNDCOLOR;
               ppres->aparam[0].cb=sizeof(LONG);
               memcpy(&ppres->aparam[0].ab,&Controls[no].SlRColor,sizeof(LONG));
               }
         else ppres=NULL;
         Controls[no].hwndLeft=WinCreateWindow(Controls[no].hwndRect,WC_LISTBOX,NULL,
                       t,CtlLay[i].leftx,mutey,listw,listh,
                       Controls[no].hwndRect,HWND_TOP,IDC_LIST|no,NULL,ppres);
         OldListboxWndProc=WinSubclassWindow(Controls[no].hwndLeft,ListboxWndProc);
         if (ppres) free(ppres);
         for (i=0;i<NControls;i++)
            if ((Controls[i].sel!=NO_INPUT_SEL)&&(Controls[i].supported))
                WinSendMsg(Controls[no].hwndLeft,LM_INSERTITEM,MPFROMSHORT(LIT_END),
                           MPFROMP((PSZ)Controls[i].name));
         }
      else {
         if (CtlLay[i].flags&CLF_OFF) {
            Controls[no].hwndMute=WinCreateWindow(Controls[no].hwndRect,WC_BUTTON,szOff,
                       WS_VISIBLE|BS_AUTOCHECKBOX|WS_TABSTOP,
                       CtlLay[i].checkx,mutey,CtlLay[i].checkw,checkh,
                       Controls[no].hwndRect,HWND_TOP,IDC_MUTE|no,NULL,NULL);
            OldButtonWndProc=WinSubclassWindow(Controls[no].hwndMute,ButtonWndProc);
            }
         else if (CtlLay[i].flags&CLF_MUTE) {
            Controls[no].hwndMute=WinCreateWindow(Controls[no].hwndRect,WC_BUTTON,szMute,
                       WS_VISIBLE|BS_AUTOCHECKBOX|WS_TABSTOP,
                       CtlLay[i].checkx,mutey,CtlLay[i].checkw,checkh,
                       Controls[no].hwndRect,HWND_TOP,IDC_MUTE|no,NULL,NULL);
            OldButtonWndProc=WinSubclassWindow(Controls[no].hwndMute,ButtonWndProc);
            }
         if (CtlLay[i].flags&CLF_BOTH) {
            Controls[no].hwndBoth=WinCreateWindow(Controls[no].hwndRect,WC_BUTTON,szBoth,
                       WS_VISIBLE|BS_AUTOCHECKBOX|WS_TABSTOP,
                       CtlLay[i].checkx,bothy,CtlLay[i].checkw,checkh,
                       Controls[no].hwndRect,HWND_TOP,IDC_BOTH|no,NULL,NULL);
            OldButtonWndProc=WinSubclassWindow(Controls[no].hwndBoth,ButtonWndProc);
            }
         else {
            if (CtlLay[i].flags&CLF_LDSP) {
               Controls[no].hwndLdesc=WinCreateWindow(Controls[no].hwndRect,WC_STATIC,szSp,
                       WS_VISIBLE | SS_TEXT | DT_VCENTER | DT_CENTER,
                       CtlLay[i].ldscx,bothy,CtlLay[i].dscw,checkh,
                       Controls[no].hwndRect,HWND_TOP,0,&sldcData,NULL);
               OldStaticWndProc=WinSubclassWindow(Controls[no].hwndLdesc,StaticWndProc);
               }
            else if (CtlLay[i].flags&CLF_LDLF) {
               Controls[no].hwndLdesc=WinCreateWindow(Controls[no].hwndRect,WC_STATIC,szLF,
                       WS_VISIBLE | SS_TEXT | DT_VCENTER | DT_CENTER,
                       CtlLay[i].ldscx,bothy,CtlLay[i].dscw,checkh,
                       Controls[no].hwndRect,HWND_TOP,0,&sldcData,NULL);
               OldStaticWndProc=WinSubclassWindow(Controls[no].hwndLdesc,StaticWndProc);
               }
            if (CtlLay[i].flags&CLF_RDCT) {
               Controls[no].hwndRdesc=WinCreateWindow(Controls[no].hwndRect,WC_STATIC,szCt,
                       WS_VISIBLE | SS_TEXT | DT_VCENTER | DT_CENTER,
                       CtlLay[i].rdscx,bothy,CtlLay[i].dscw,checkh,
                       Controls[no].hwndRect,HWND_TOP,0,&sldcData,NULL);
               OldStaticWndProc=WinSubclassWindow(Controls[no].hwndRdesc,StaticWndProc);
               }
            else if (CtlLay[i].flags&CLF_RDHF) {
               Controls[no].hwndRdesc=WinCreateWindow(Controls[no].hwndRect,WC_STATIC,szHF,
                       WS_VISIBLE | SS_TEXT | DT_VCENTER | DT_CENTER,
                       CtlLay[i].rdscx,bothy,CtlLay[i].dscw,checkh,
                       Controls[no].hwndRect,HWND_TOP,0,&sldcData,NULL);
               OldStaticWndProc=WinSubclassWindow(Controls[no].hwndRdesc,StaticWndProc);
               }
            }


         sldcData.cbSize=sizeof(SLDCDATA);
         if (ShortSliders) sldcData.usScale1Increments=51;
         else sldcData.usScale1Increments=101;
         sldcData.usScale1Spacing=0;
         t=WS_VISIBLE | SLS_VERTICAL | SLS_SNAPTOINCREMENT |
                       SLS_PRIMARYSCALE1 | WS_TABSTOP;
         if (NarrowSliders) t|= SLS_OWNERDRAW;
         if (CtlLay[i].flags&CLF_LEFT) {
            if (Controls[no].SlLColor!=-1) {
               Light=RGBMix(Controls[no].SlLColor,RGB_WHITE);
               Dark=RGBMix(Controls[no].SlLColor,RGB_BLACK);
               ppres=malloc(sizeof(ULONG)*10);
               ppres->cb=sizeof(ULONG)*9;
               ppres->aparam[0].id=PP_BUTTONBACKGROUNDCOLOR;
               ppres->aparam[0].cb=sizeof(LONG);
               memcpy(&ppres->aparam[0].ab,&Controls[no].SlLColor,sizeof(LONG));
               ppres->aparam[1].id=PP_BORDERLIGHTCOLOR;
               ppres->aparam[1].cb=sizeof(LONG);
               memcpy(&ppres->aparam[1].ab,&Light,sizeof(LONG));
               ppres->aparam[2].id=PP_BORDERDARKCOLOR;
               ppres->aparam[2].cb=sizeof(LONG);
               memcpy(&ppres->aparam[2].ab,&Dark,sizeof(LONG));
               }
            else ppres=NULL;
            Controls[no].hwndLeft=WinCreateWindow(Controls[no].hwndRect,WC_SLIDER,NULL,t,
                       CtlLay[i].leftx,slidery,sliderw,sliderh,
                       Controls[no].hwndRect,HWND_TOP,IDC_LEFT|no,&sldcData,ppres);
            OldSliderWndProc=WinSubclassWindow(Controls[no].hwndLeft,SliderWndProc);
            if (ppres) free(ppres);
            }
         if (CtlLay[i].flags&CLF_RIGHT) {
            if (Controls[no].SlRColor!=-1) {
               Light=RGBMix(Controls[no].SlRColor,RGB_WHITE);
               Dark=RGBMix(Controls[no].SlRColor,RGB_BLACK);
               ppres=malloc(sizeof(ULONG)*10);
               ppres->cb=sizeof(ULONG)*9;
               ppres->aparam[0].id=PP_BUTTONBACKGROUNDCOLOR;
               ppres->aparam[0].cb=sizeof(LONG);
               memcpy(&ppres->aparam[0].ab,&Controls[no].SlRColor,sizeof(LONG));
               ppres->aparam[1].id=PP_BORDERLIGHTCOLOR;
               ppres->aparam[1].cb=sizeof(LONG);
               memcpy(&ppres->aparam[1].ab,&Light,sizeof(LONG));
               ppres->aparam[2].id=PP_BORDERDARKCOLOR;
               ppres->aparam[2].cb=sizeof(LONG);
               memcpy(&ppres->aparam[2].ab,&Dark,sizeof(LONG));
               }
            else ppres=NULL;
            Controls[no].hwndRight=WinCreateWindow(Controls[no].hwndRect,WC_SLIDER,NULL,t,
                       CtlLay[i].rightx,slidery,sliderw,sliderh,
                       Controls[no].hwndRect,HWND_TOP,IDC_RIGHT|no,&sldcData,ppres);
            OldSliderWndProc=WinSubclassWindow(Controls[no].hwndRight,SliderWndProc);
            if (ppres) free(ppres);
            }
         }
      x+=CtlLay[i].boxw+cellspx;
      if ((i+1)<actc) {
         if (CtlLay[i].celly!=CtlLay[i+1].celly) {
            x=fieldx;
            y-=boxh+cellspy;
            }
         }
      }
   Changing=FALSE;
   free(CtlLay);
   DisplayControls();
   return;
   }


// Open device, query available controls and set up Controls structure
// return TRUE if Ok.
int ApiInit() {
   unsigned char ApiMap[256];
   int i;
   MIXSTRUCT MixStruct;
   MIXMSGBUF MixMsgBuf;
   ULONG Level;
   char Buff[PIPE_BUF_MAX+1];
   char *tp;

   if (!UsingPipe) {
      if (mixerapiInit(MyCallBack)) UsingPipe=TRUE;	// Try pipe API
      else {
         if (!mixerapiIOCTL90(GETAPIMAP,ApiMap,256)) {
            for (i=0;i<NControls;i++)
               if ((Controls[i].setfunc>=0x20)&&(!(ApiMap[Controls[i].setfunc]&&ApiMap[Controls[i].qfunc]))) {
                  Controls[i].supported=0;
                  Controls[i].active=0;
                  }
            }
         if (!mixerapiIOCTL90(APILEVELQUERY,&Level,sizeof(ULONG))) ApiLevel=Level;
         }
      }

   if (UsingPipe) {
      strcpy(Buff,"80");		// Api Level
      if (PipeCommand(Buff,PIPE_BUF_MAX)) return 0;
      ApiLevel=atoi(Buff+3);
      }

// Check if query functions work
   for (i=0;i<NControls;i++) if (Controls[i].supported) {
       if (CallQuery(Controls[i].qfunc,&MixStruct)) {
            Controls[i].supported=0;
            Controls[i].active=0;
            }
       else {
            if (Controls[i].type==TYPE_SELECTOR) {
               if (MixStruct.Mute&4) Controls[i].flags|=FLAG_MULTIPLE;
               }
            else if (Controls[i].type==TYPE_TONE) {
               if (!(MixStruct.Mute&1)) Controls[i].flags|=FLAG_ONLYRIGHT;
               if (!(MixStruct.Mute&2)) Controls[i].flags|=FLAG_ONLYLEFT;
               }
            else if (Controls[i].type==TYPE_3D) {
               if (!(MixStruct.Mute&2)) Controls[i].flags|=FLAG_ONLYRIGHT;
               if (!(MixStruct.Mute&4)) Controls[i].flags|=FLAG_ONLYLEFT;
               }
            }
       }
   for (i=0;i<NControls;i++) if (Controls[i].supported) break; 
   if (i==NControls) return 0;	// No controls are available 
/* If API Level >1 then default for lockables is "active" */
   if (ApiLevel>1) for (i=0;i<<NControls;i++) {
      if ((Controls[i].supported)&&(Controls[i].flags&FLAG_LOCKABLE))
         Controls[i].active=TRUE;
      }

   if (!UsingPipe) {
      if (ApiMap[MSGBUF]) {
         MixMsgBuf.pBuffer=0;
         MixMsgBuf.ulSize=0;
         MixMsgBuf.fClear=FALSE;
         if (!mixerapiIOCTL90(MSGBUF,&MixMsgBuf,sizeof(MIXMSGBUF))) {
            InfoBufSize=MixMsgBuf.ulSize;
            InfoBuf=malloc(InfoBufSize);
            MixMsgBuf.pBuffer=(ULONG)InfoBuf;
            MixMsgBuf.ulSize=InfoBufSize;
            MixMsgBuf.fClear=FALSE;
            if (mixerapiIOCTL90(MSGBUF,&MixMsgBuf,sizeof(MIXMSGBUF))) {
               free(InfoBuf);
               InfoBuf=NULL;
               }
            else InfoBuf[MixMsgBuf.ulSize]=0;
            }
         }
      }
   else { 
      strcpy(Buff,"83");
      if (!PipeCommand(Buff,PIPE_BUF_MAX)) {
         tp=strchr(Buff,' ');
         if (tp) {
            if (*tp) tp++;
            if (*tp) {
               InfoBufSize=strlen(tp)+1;
               InfoBuf=malloc(InfoBufSize);
               strcpy(InfoBuf,tp);
               }
            }
         }
      }

   if (!InfoBuf) {
      InfoBufSize=DEF_INFO_BUF_SIZE;
      InfoBuf=malloc(InfoBufSize);
      InfoBuf[0]=0;
      }
   if (!InfoBuf[0]) {
      strcpy(InfoBuf,"Driver does not return info.");
      strcat(InfoBuf,"\nAPI Level: ");
      _itoa(ApiLevel,InfoBuf+strlen(InfoBuf),10);
      strcat(InfoBuf,"\nMixer state change callback is ");
      if (!ApiMap[CALLBACKREG]) strcat(InfoBuf,"not ");
      strcat(InfoBuf,"supported.");
      strcat(InfoBuf,"\nSupported controls:");
      for (i=0;i<NControls;i++) 
         if ((Controls[i].supported)&&(Controls[i].setfunc>=0x20)) {
            strcat(InfoBuf,"\n");
            strcat(InfoBuf,Controls[i].name);
            }
      strcat(InfoBuf,"\n");
      }

   return 1;
   }

void ApiClose() {
   if (!UsingPipe) mixerapiDeInit();
   }

// Set controls names to defaults
void NameInit() {
   int i;

   for (i=0;i<NControls;i++)
      WinLoadString(hab, 0, Controls[i].defnameID, 31, Controls[i].name);
   }

void LoadProfile() {
   unsigned char tname[32];
   char Buff[32];
   int i;
   unsigned char *SuffPos;

   if (PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,(char *)szKeyRestore,NULL,Buff,32))
      RestoreState=atoi(Buff);
   if (PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,(char *)szKeyAutoMini,NULL,Buff,32))
      AutoMini=atoi(Buff);
   PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,
              (char *)szKeyInstName,NULL,szInstName,MAX_INST_NAME);
   if (PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,(char *)szKeyCellsInLine,NULL,Buff,32))
      CellsInLine=atoi(Buff);
   if (PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,(char *)szKeyShowMenu,NULL,Buff,32))
      ShowMenu=atoi(Buff);
   if (PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,(char *)szKeyShowTitlebar,NULL,Buff,32))
      ShowTitlebar=atoi(Buff);
   PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,
              (char *)szKeyIconFile,NULL,szIconFile,CCHMAXPATH);
   PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,
              (char *)szKeyLockName,NULL,szLock,32);
   PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,
              (char *)szKeyOffName,NULL,szOff,32);
   PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,
              (char *)szKeyMuteName,NULL,szMute,32);
   PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,
              (char *)szKeyBothName,NULL,szBoth,32);
   PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,
              (char *)szKeyLFName,NULL,szLF,32);
   PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,
              (char *)szKeyHFName,NULL,szHF,32);
   PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,
              (char *)szKeySpName,NULL,szSp,32);
   PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,
              (char *)szKeyCtName,NULL,szCt,32);
   if (PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,
                             (char *)szKeyNarrowSliders,NULL,Buff,32))
      NarrowSliders=atoi(Buff);
   if (PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,
                             (char *)szKeyShortSliders,NULL,Buff,32))
      ShortSliders=atoi(Buff);

   for (i=0;i<NControls;i++) {
      strcpy(tname,szCtlPref);
      _itoa(Controls[i].setfunc,tname+strlen(tname),16);
      SuffPos=tname+strlen(tname);

      strcpy(SuffPos,szLeftSuf);
      if (PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,NULL,Buff,32))
         Controls[i].left=atoi(Buff);
      strcpy(SuffPos,szRightSuf);
      if (PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,NULL,Buff,32))
         Controls[i].right=atoi(Buff);
      strcpy(SuffPos,szMuteSuf);
      if (PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,NULL,Buff,32))
         Controls[i].mute=atoi(Buff);
      strcpy(SuffPos,szBothSuf);
      if (PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,NULL,Buff,32))
         Controls[i].both=atoi(Buff);
      if (ApiLevel>1) {
         strcpy(SuffPos,szLockSuf);
         if (PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,NULL,Buff,32))
            Controls[i].lock=atoi(Buff);
         }
      strcpy(SuffPos,szActiveSuf);
      if (PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,NULL,Buff,32))
         Controls[i].active=atoi(Buff);
      strcpy(SuffPos,szNameSuf);
      PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,NULL,Controls[i].name,32);
      strcpy(SuffPos,szBackGndSuf);
      if (PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,NULL,Buff,32))
         Controls[i].BgColor=atoi(Buff);
      strcpy(SuffPos,szForeGndSuf);
      if (PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,NULL,Buff,32))
         Controls[i].FgColor=atoi(Buff);
      strcpy(SuffPos,szSlLClrSuf);
      if (PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,NULL,Buff,32))
         Controls[i].SlLColor=atoi(Buff);
      strcpy(SuffPos,szSlRClrSuf);
      if (PrfQueryProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,NULL,Buff,32))
         Controls[i].SlRColor=atoi(Buff);
      }
   return;
   }

void SaveProfile() {
   unsigned char tname[32];
   char Buff[32];
   int i;
   unsigned char *SuffPos;

   _itoa(RestoreState,Buff,10);
   PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,(char *)szKeyRestore,Buff);
   _itoa(AutoMini,Buff,10);
   PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,(char *)szKeyAutoMini,Buff);
   PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,
                         (char *)szKeyInstName,szInstName);
   _itoa(CellsInLine,Buff,10);
   PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,(char *)szKeyCellsInLine,Buff);
   _itoa(ShowMenu,Buff,10);
   PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,(char *)szKeyShowMenu,Buff);
   _itoa(ShowTitlebar,Buff,10);
   PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,(char *)szKeyShowTitlebar,Buff);
   PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,(char *)szKeyIconFile,szIconFile);
   PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,(char *)szKeyLockName,szLock);
   PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,(char *)szKeyOffName,szOff);
   PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,(char *)szKeyMuteName,szMute);
   PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,(char *)szKeyBothName,szBoth);
   PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,(char *)szKeyHFName,szHF);
   PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,(char *)szKeyLFName,szLF);
   PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,(char *)szKeySpName,szSp);
   PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,(char *)szKeyCtName,szCt);
   _itoa(NarrowSliders,Buff,10);
   PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,(char *)szKeyNarrowSliders,Buff);
   _itoa(ShortSliders,Buff,10);
   PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,(char *)szKeyShortSliders,Buff);

   for (i=0;i<NControls;i++) if (Controls[i].supported) {
      strcpy(tname,szCtlPref);
      _itoa(Controls[i].setfunc,tname+strlen(tname),16);
      SuffPos=tname+strlen(tname);

      if (!(Controls[i].flags&FLAG_ONLYRIGHT)) {
         strcpy(SuffPos,szLeftSuf);
         _itoa(Controls[i].left,Buff,10);
         PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,Buff);
         }
      if (!(Controls[i].flags&FLAG_ONLYLEFT)) {
         strcpy(SuffPos,szRightSuf);
         _itoa(Controls[i].right,Buff,10);
         PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,Buff);
         }
      if (Controls[i].flags&FLAG_MUTABLE) {
         strcpy(SuffPos,szMuteSuf);
         _itoa(Controls[i].mute,Buff,10);
         PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,Buff);
         }
      if (Controls[i].flags&FLAG_STEREO) {
         strcpy(SuffPos,szBothSuf);
         _itoa(Controls[i].both,Buff,10);
         PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,Buff);
         }
      if (Controls[i].flags&FLAG_LOCKABLE) {
         strcpy(SuffPos,szLockSuf);
         _itoa(Controls[i].lock,Buff,10);
         PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,Buff);
         }
      strcpy(SuffPos,szActiveSuf);
      _itoa(Controls[i].active,Buff,10);
      PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,Buff);
      strcpy(SuffPos,szNameSuf);
      PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,Controls[i].name);
      strcpy(SuffPos,szBackGndSuf);
      if (Controls[i].BgColor!=-1) {
         _ltoa(Controls[i].BgColor,Buff,10);
         PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,Buff);
         }
      else 
         PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,NULL);
      strcpy(SuffPos,szForeGndSuf);
      if (Controls[i].FgColor!=-1) {
         _ltoa(Controls[i].FgColor,Buff,10);
         PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,Buff);
         }
      else 
         PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,NULL);
      strcpy(SuffPos,szSlLClrSuf);
      if (Controls[i].SlLColor!=-1) {
         _ltoa(Controls[i].SlLColor,Buff,10);
         PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,Buff);
         }
      else 
         PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,NULL);
      strcpy(SuffPos,szSlRClrSuf);
      if (Controls[i].SlRColor!=-1) {
         _ltoa(Controls[i].SlRColor,Buff,10);
         PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,Buff);
         }
      else 
         PrfWriteProfileString(HINI_USERPROFILE,(char *)szIniApp,tname,NULL);
      }
   return;
   }

int ParseParms(int argc, char *argv[]) {
   int i;
   int PDDMaster;
   char *tp;
   struct t_Batch *BLast=NULL;

   PDDMaster=FALSE;
   for (i=1;i<argc;i++) {
     if (!stricmp(argv[i],"-i"))  {
        i++;
        if (i==argc) return FALSE;		// Syntax error
        strncpy(szIniApp,argv[i],MAX_APP_NAME_LEN-1);
        }
     else if (!stricmp(argv[i],"-d"))  {
        i++;
        if (i==argc) return FALSE;		// Syntax error
        strcpy(szPddName,"\\DEV\\");
        strncpy(szPddName+5,argv[i],8);
        strupr(szPddName);
        }
     else if (!stricmp(argv[i],"-p"))
        UsingPipe=TRUE;
     else if (!stricmp(argv[i],"-pn"))  {
        i++;
        if (i==argc) return FALSE;		// Syntax error
        strncpy(szPipeName,argv[i],CCHMAXPATH-1);
        }
     else if (!stricmp(argv[i],"-mv")) PDDMaster=TRUE;
     else if (!stricmp(argv[i],"-x")) AutoExit=TRUE;
     else if (!stricmp(argv[i],"-ps")) PipeSupport=TRUE;
     else if (!stricmp(argv[i],"-nsw"))  {
        i++;
        if (i==argc) return FALSE;		// Syntax error
        NSWidth=atoi(argv[i]);
        }
     else {					// Batch command
        strupr(argv[i]);
        tp=strchr(argv[i],':');
        if (!tp) return FALSE;			// Syntax error
        if ((tp==argv[i])||(!tp[1])) return FALSE;	// Syntax error
        if (!BLast) {
           Batch=malloc(sizeof(struct t_Batch));
           BLast=Batch;
           }
        else {
           BLast->next=malloc(sizeof(struct t_Batch));
           BLast=BLast->next;
           }
        BLast->next=NULL;
        *tp=0;
        strncpy(BLast->name,argv[i],31);
        BLast->name[31]=0;
        tp++;
        BLast->leftop=1;
        BLast->leftval=0;
        BLast->rightop=1;
        BLast->rightval=0;
        BLast->mute=0;
        BLast->lock=0;
        if (*tp) {
           if (*tp!=':') {
              if (*tp=='+') {
                 BLast->leftop=1;
                 tp++;
                 }
              else if (*tp=='-') {
                 BLast->leftop=-1;
                 tp++;
                 }
              else BLast->leftop=0;
              BLast->leftval=atoi(tp);
              tp=strchr(tp,':');
              }
           if (tp) {
              tp++;
              if (*tp) {
                 if (*tp!=':') {
                    if (*tp=='+') {
                       BLast->rightop=1;
                       tp++;
                       }
                    else if (*tp=='-') {
                       BLast->rightop=-1;
                       tp++;
                       }
                    else BLast->rightop=0;
                    BLast->rightval=atoi(tp);
                    }
                 while (tp) {
                    tp=strchr(tp,':');
                    if (tp) {
                       tp++;
                       if (!memcmp(tp,"+M",2)) BLast->mute=1;
                       else if (!memcmp(tp,"-M",2)) BLast->mute=-1;
                       else if (!memcmp(tp,"+L",2)) BLast->lock=1;
                       else if (!memcmp(tp,"-L",2)) BLast->lock=-1;
                       else return FALSE; 		// Syntax error
                       }
                    }
                 }
              }
           }
        }
     }

   if (!PDDMaster) for (i=0;i<NControls;i++) if (Controls[i].qfunc<0x20) {
      Controls[i].qfunc=0;
      Controls[i].setfunc=0;
      Controls[i].flags&=~FLAG_STEREO;
      Controls[i].flags|=FLAG_ONLYLEFT;
      }
   return TRUE;
   }

// Process the batch commands
int DoBatch() {
   struct t_Batch *BLast,*BNext;
   int i;
   char tbuff[32];

   BLast=Batch;
   while (BLast) {
/*      fprintf(stderr,"Batch cmd: Name: %s, left: %d:%d, right: %d:%d, mute: %d, lock: %d\n",
              BLast->name,BLast->leftop,BLast->leftval,BLast->rightop,BLast->rightval,
              BLast->mute,BLast->lock); */
      for(i=0;i<NControls;i++) if (Controls[i].supported) {
         strcpy(tbuff,Controls[i].name);
         strupr(tbuff);
         if (!memcmp(BLast->name,tbuff,strlen(BLast->name))) break;
         }
      if (i==NControls) return FALSE;
      if (!(Controls[i].flags&FLAG_ONLYRIGHT)) {
         if (BLast->leftop==0) Controls[i].left=BLast->leftval;
         else if (BLast->leftop<0) Controls[i].left-=BLast->leftval;
         else Controls[i].left+=BLast->leftval;
         if (Controls[i].left<0) Controls[i].left=0;
         if (Controls[i].left>100) Controls[i].left=100;
         if (!(Controls[i].flags&FLAG_ONLYLEFT)&&(Controls[i].both))
            Controls[i].right=Controls[i].left;
         }
      if (!(Controls[i].flags&FLAG_ONLYLEFT)) {
         if (BLast->rightop==0) Controls[i].right=BLast->rightval;
         else if (BLast->rightop<0) Controls[i].right-=BLast->rightval;
         else Controls[i].right+=BLast->rightval;
         if (Controls[i].right<0) Controls[i].right=0;
         if (Controls[i].right>100) Controls[i].right=100;
         if (!(Controls[i].flags&FLAG_ONLYLEFT)&&(Controls[i].both))
            Controls[i].left=Controls[i].right;
         }
      if (Controls[i].flags&FLAG_MUTABLE) {
         if (BLast->mute<0) Controls[i].mute=FALSE;
         else if (BLast->mute>0) Controls[i].mute=TRUE;
         }
      if (Controls[i].flags&FLAG_LOCKABLE) {
         if (BLast->lock<0) Controls[i].lock=FALSE;
         else if (BLast->lock>0) Controls[i].lock=TRUE;
         }
      SetControl(i);
      BNext=BLast->next;
      free(BLast);
      BLast=BNext;
      }
   return TRUE;
   }


// Pipe server - process command
int ProcessCmd(char *Buff, int tokens, int cmd, int p1, int p2, int p3) {
   int i,right,mute;

   if ((cmd==0x01)||((cmd&0xF0)==0x40)) {		// Set command
      if (tokens<2) p1=100;	// VolumeL
      if (tokens<3) p2=p1;	// VolumeR
      if (tokens<4) p3=0;	// Mute
      for (i=0;i<NControls;i++) 
          if ((Controls[i].setfunc==cmd)&&(Controls[i].supported)) {
             Controls[i].left=p1;
             Controls[i].right=p2;
             Controls[i].mute=(p3&FLAG_MUTE);
             if (Controls[i].flags&FLAG_LOCKABLE)
                Controls[i].lock=!(p3&FLAG_RELEASE);
             SetControl(i);
             WinPostMsg( hwndMain, WM_USER, (MPARAM)USER_QUERY,(MPARAM)0 );
             strcat(Buff,REP_OK);
             return TRUE;
             }
      }
   else if ((cmd==0x11)||((cmd&0xF0)==0x60)) {		// Query command
      for (i=0;i<NControls;i++) 
          if ((Controls[i].qfunc==cmd)&&(Controls[i].supported)) {

             if (Controls[i].flags&FLAG_ONLYLEFT) right=0;
             else right=Controls[i].right;
             if ((Controls[i].mute)&&(Controls[i].flags&FLAG_MUTABLE))
                mute=FLAG_MUTE;
             else mute=0;
             if ((Controls[i].flags&FLAG_LOCKABLE)&&(!Controls[i].lock))
                mute|=FLAG_RELEASE;
             if (Controls[i].type==TYPE_3D) {
                if (!(Controls[i].flags&FLAG_ONLYRIGHT)) mute|=2;
                if (!(Controls[i].flags&FLAG_ONLYLEFT)) mute|=4;
                }
             else if (Controls[i].type==TYPE_TONE) {
                if (!(Controls[i].flags&FLAG_ONLYRIGHT)) mute|=1;
                if (!(Controls[i].flags&FLAG_ONLYLEFT)) mute|=2;
                }
             else if (Controls[i].type==TYPE_SELECTOR) {
                if (!(Controls[i].flags&FLAG_MULTIPLE)) mute|=4;
                }
             sprintf(Buff+strlen(Buff),"%d %d %x",
                     Controls[i].left,Controls[i].right,mute);
             return TRUE;
             }
      }
   strcat(Buff,REP_UNSUPPORTED);
   return TRUE;
   }

VOID APIENTRY PipeServer(ULONG Dummy) {
   char Buff[PIPE_BUF_MAX+1];
   ULONG cbActual;
   int cmd,p1,p2,p3,tokens;
   char *tp;


   while(1) {
      DosConnectNPipe(hPipe);
      DosRead(hPipe,Buff,PIPE_BUF_MAX,&cbActual);
      Buff[cbActual]=0;
      tp=Buff;
      while ((*tp>=0x20)||(*tp==9)) {
         if (*tp==9) *tp=0x20;
         tp++;
         }
      *tp=0;
      tokens=0;
      tp=Buff;
      while (*tp==0x20) tp++;
      if (*tp) {
         tokens++;
         cmd=strtol(tp,&tp,16);
         if (tp) while (*tp==0x20) tp++;
         if (!*tp) tp=NULL;
         if (tp) {
            tokens++;
            p1=strtol(tp,&tp,10);
            if (tp) while (*tp==0x20) tp++;
            if (!*tp) tp=NULL;
            if (tp) {
               tokens++;
               p2=strtol(tp,&tp,10);
               if (tp) while (*tp==0x20) tp++;
               if (!*tp) tp=NULL;
               if (tp) {
                  tokens++;
                  p3=strtol(tp,NULL,16);
                  }
               }
            }
         }
      if (tokens) {
         if (cmd>0xF) _itoa(cmd,Buff,16);
         else {
            Buff[0]='0';
            _itoa(cmd,Buff+1,16);
            }
         strcat(Buff," ");
         if (cmd==0x80) _itoa(ApiLevel,Buff+strlen(Buff),10);	// API level
         else if (cmd==0x83) {					// Message buffer
            if (InfoBuf) strncat(Buff,InfoBuf,PIPE_BUF_MAX);
            }
         else ProcessCmd(Buff,tokens,cmd,p1,p2,p3);
         DosWrite(hPipe,Buff,strlen(Buff),&cbActual);
         }
      DosResetBuffer(hPipe);
      DosDisConnectNPipe(hPipe);
      }   
   }

int PipeSupInit() {
   if ((UsingPipe)||(!PipeSupport)) return TRUE;
   if(DosCreateNPipe((char *)szPipeName,&hPipe,NP_ACCESS_DUPLEX,
                     NP_WAIT|NP_TYPE_BYTE|NP_READMODE_BYTE|1,PIPE_BUF_MAX,
                     PIPE_BUF_MAX,0)) return FALSE;
   DosCreateThread(&PipeTID,PipeServer,0,0,8192);   
   return TRUE;
   }


INT main(int argc, char *argv[])
{
   QMSG qmsg;					      /* message structure */

   hab = WinInitialize(0);
   if(!hab)  {
       DosBeep(BEEP_WARN_FREQ, BEEP_WARN_DUR);
       return(RETURN_ERROR);
   }
   hmq = WinCreateMsgQueue(hab, 0);
   if(!hmq)  {
       DosBeep(BEEP_WARN_FREQ, BEEP_WARN_DUR);
       WinTerminate(hab);
       return(RETURN_ERROR);
   }
   if(!Init()) {
       MessageBox(HWND_DESKTOP, IDMSG_INITFAILED, "Error !",
			   MB_OK | MB_ERROR | MB_MOVEABLE, TRUE);
       DosExit(EXIT_PROCESS, RETURN_ERROR);
       }
   if(!ParseParms(argc,argv)) {
       MessageBox(HWND_DESKTOP, IDMSG_SYNTAXERROR, "Error !",
			   MB_OK | MB_ERROR | MB_MOVEABLE, TRUE);
       DosExit(EXIT_PROCESS, RETURN_ERROR);
       }
   NameInit();
   if(!ApiInit()) {
       MessageBox(HWND_DESKTOP, IDMSG_APIINITFAILED, "Error !",
			   MB_OK | MB_ERROR | MB_MOVEABLE, TRUE);
       DosExit(EXIT_PROCESS, RETURN_ERROR);
       }
   LoadProfile();
   if (RestoreState) SetControls();
   else QueryControls();
   if (!DoBatch()) {
      MessageBox(HWND_DESKTOP, IDMSG_BADCTLNAME, "Error !",
			   MB_OK | MB_ERROR | MB_MOVEABLE, TRUE);
      DosExit(EXIT_PROCESS, RETURN_ERROR);
      }
  if (!PipeSupInit()) {
      MessageBox(HWND_DESKTOP, IDMSG_PIPECREAT, "Error !",
			   MB_OK | MB_ERROR | MB_MOVEABLE, TRUE);
      DosExit(EXIT_PROCESS, RETURN_ERROR);
      }
   MainCreate();
   if (!AutoExit) {
      while(WinGetMsg(hmq, (PQMSG)&qmsg, 0L, 0L, 0L))
   	      WinDispatchMsg(hmq, (PQMSG)&qmsg);
      SaveProfile();
      }
   ApiClose();
   if (InfoBuf) free(InfoBuf);
   DosExit(EXIT_PROCESS, RETURN_SUCCESS);
   return 0;
}							       /* main() */

BOOL Init(VOID) {
   int w;

   if(DosExitList(EXLST_ADD, ExitProc)) return FALSE;

   if(!WinRegisterClass(hab,(PSZ)"MAIN",(PFNWP)MainWndProc,CS_SIZEREDRAW,0))
      return FALSE;
   if(!WinRegisterClass(hab,(PSZ)"CONTROL",(PFNWP)ControlWndProc,0,0))
      return FALSE;

   WinLoadString(hab, 0, IDS_LOCK, 32, szLock);
   WinLoadString(hab, 0, IDS_OFF, 32, szOff);
   WinLoadString(hab, 0, IDS_MUTE, 32, szMute);
   WinLoadString(hab, 0, IDS_BOTH, 32, szBoth);
   WinLoadString(hab, 0, IDS_SP, 32, szSp);
   WinLoadString(hab, 0, IDS_LF, 32, szLF);
   WinLoadString(hab, 0, IDS_CT, 32, szCt);
   WinLoadString(hab, 0, IDS_HF, 32, szHF);

   w=WinQuerySysValue(HWND_DESKTOP,SV_CXSCREEN);
   if (w<=640) NSWidth=SLIDER_W_N640;
   else if (w<=800) NSWidth=SLIDER_W_N800;
   else if (w<=1024) NSWidth=SLIDER_W_N1024;
   else if (w<=1152) NSWidth=SLIDER_W_N1152;
   else if (w<=1280) NSWidth=SLIDER_W_N1280;
   else NSWidth=SLIDER_W_N1600;

   return TRUE;
}							  /* Init() */

VOID APIENTRY ExitProc(ULONG usTermCode) {
   if (PipeTID) {
      DosKillThread(PipeTID);
      DosClose(hPipe);
      }
   if (hwndMainFrame) WinDestroyWindow(hwndMainFrame);
   if (hIcon) WinFreeFileIcon(hIcon);
   WinDestroyMsgQueue(hmq);
   WinTerminate(hab);
   DosExitList(EXLST_EXIT, (PFNEXITLIST)0L);	/* termination complete */
   return;
   }
LONG MessageBox(HWND hwndOwner, LONG IdMsg, PSZ pszMsg, LONG fsStyle,
		     BOOL bBeep)
{
    CHAR szText[MESSAGELEN];
    LONG usRet;

    if(!WinLoadMessage(hab, (HMODULE)NULL, IdMsg, MESSAGELEN, (PSZ)szText)) {
	WinAlarm(HWND_DESKTOP, WA_ERROR);
	return RETURN_ERROR;
        }
    if(bBeep) WinAlarm(HWND_DESKTOP, WA_ERROR);

    usRet = WinMessageBox(HWND_DESKTOP,
			 hwndOwner,
			 szText,
			 (PSZ)pszMsg,
			 IDM_MSGBOX,
			 fsStyle);
    return usRet;
}						    /* MessageBox() */


void MainControl(HWND hwnd,MPARAM mp1) {
   int i,j,k,val;

   if ((!Changing)&&((SHORT2FROMMP(mp1)==BN_CLICKED)||
       (SHORT2FROMMP(mp1)==LN_SELECT)||(SHORT2FROMMP(mp1)==SLN_CHANGE)||
       (SHORT2FROMMP(mp1)==SLN_SLIDERTRACK))) {
      i=LOUSHORT(mp1)&0xFF;	// Control no
      switch (LOUSHORT(mp1)&0xFF00) {
         case IDC_LIST:
            Controls[i].left=0;
            val=LIT_FIRST;
            while(1) {
               val=(int)WinSendMsg(Controls[i].hwndLeft,
                        LM_QUERYSELECTION,(MPARAM)val,(MPARAM)NULL);
               if (val==LIT_NONE) break;
               for (k=0,j=0;k<NControls;k++) if ((Controls[k].sel!=NO_INPUT_SEL)&&(Controls[k].supported)) {
                  if (j==val) {
                     Controls[i].left|=Controls[k].sel;
                     break;
                     }
                  else j++;
                  }
               if (!(Controls[i].flags&FLAG_MULTIPLE)) break;
               };
            break;
         case IDC_LEFT: 
            Controls[i].left=(int)WinSendMsg(Controls[i].hwndLeft,
                 SLM_QUERYSLIDERINFO, MPFROM2SHORT(SMA_SLIDERARMPOSITION,
                 SMA_INCREMENTVALUE),(MPARAM)NULL);
            if (ShortSliders) Controls[i].left*=2;
            if ((Controls[i].flags&FLAG_STEREO)&&Controls[i].both)
                Controls[i].right=Controls[i].left;
            break;
         case IDC_RIGHT: 
            Controls[i].right=(int)WinSendMsg(Controls[i].hwndRight,
                    SLM_QUERYSLIDERINFO, MPFROM2SHORT(SMA_SLIDERARMPOSITION,
                    SMA_INCREMENTVALUE),(MPARAM)NULL);
            if (ShortSliders) Controls[i].right*=2;
            if ((Controls[i].flags&FLAG_STEREO)&&Controls[i].both)
                Controls[i].left=Controls[i].right;
            break;
         case IDC_MUTE:
            Controls[i].mute=(int)WinSendMsg(Controls[i].hwndMute,
                                                    BM_QUERYCHECK,0,0);
            break;
         case IDC_BOTH:
            Controls[i].both=(int)WinSendMsg(Controls[i].hwndBoth,
                                                    BM_QUERYCHECK,0,0);
            if ((Controls[i].both)&&(Controls[i].right!=Controls[i].left)) {
                Controls[i].left=(Controls[i].left+Controls[i].right)/2;
                Controls[i].right=Controls[i].left;
                }
         case IDC_LOCK:
            Controls[i].lock=(int)WinSendMsg(Controls[i].hwndLock,
                                                    BM_QUERYCHECK,0,0);
            break;
         default:
            return;
         }
      SetControl(i);
      QueryControl(i);
      DisplayControl(i);
      }
   return;
   }

void Minimize() {
   SWP swpCurrent;

   WinQueryWindowPos(hwndMainFrame,&swpCurrent);
   if (!(swpCurrent.fl&SWP_MINIMIZE))
      WinSetWindowPos(hwndMainFrame,0,swpCurrent.x,swpCurrent.y,
                   swpCurrent.cx,swpCurrent.cy,
                   SWP_MINIMIZE|SWP_SIZE|SWP_MOVE);
   return;
   }

void MainCommand(HWND hwnd,MPARAM mp1) {

   switch(LOUSHORT(mp1)) {
      case IDC_SETTINGS:
         ChildDlg=TRUE;
         WinDlgBox(HWND_DESKTOP, hwndMain, (PFNWP)SettingsDlgProc,
                   (HMODULE) NULL, IDD_SETTINGS, NULL);
         ChildDlg=FALSE;
         MainCreate();
         break;
      case IDC_ABOUT:
         ChildDlg=TRUE;
         WinDlgBox(HWND_DESKTOP, hwndMain, (PFNWP)AboutDlgProc,
			(HMODULE) NULL, IDD_ABOUT, NULL);
         ChildDlg=FALSE;
         break;
      case IDC_INFO:
         ChildDlg=TRUE;
         WinDlgBox(HWND_DESKTOP, hwndMain, (PFNWP)InfoDlgProc,
			(HMODULE) NULL, IDD_INFO, NULL);
         ChildDlg=FALSE;
         break;
      case IDC_EXIT:
         WinPostMsg( hwndMain, WM_CLOSE, (MPARAM)0,(MPARAM)0 );
         break;
      case IDC_MOVE:
         WinSendMsg(hwndMainFrame, WM_TRACKFRAME, MPFROMSHORT( TF_MOVE | TF_SETPOINTERPOS ),
                     0);
         break;
      case IDC_MINIMIZE:
         Minimize();
         break;
      }
   return;
   }

MRESULT EXPENTRY MainWndProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
   SWP swpCurrent;
   HPS	  hps;				/* Handle for painting		 */
   RECTL  rect;				/* Rectangle struct for painting */
   LONG lColor;
   MRESULT rc;
   static HWND hwndMenu;
   POINTL point;
   char *FName;

     switch (msg) {
       case WM_CREATE:
          WinPostMsg( hwnd, WM_USER, (MPARAM)USER_INIT,(MPARAM)0 );
 	  break;
       case WM_USER:
          switch ((int)mp1) {
             case USER_INIT:
                hwndMenu=WinLoadMenu(hwndMain,0,ID_POPUP);   
                lColor=WinQuerySysColor(HWND_DESKTOP,SYSCLR_DIALOGBACKGROUND,0);
                WinSetPresParam(hwndMain,PP_BACKGROUNDCOLOR,sizeof(lColor),&lColor);
                if (!WinRestoreWindowPos((char *)szIniApp,(char *)szKeyWindowPos,
                                          hwndMainFrame)) {
                  WinQueryTaskSizePos(hab,0,&swpCurrent);
                  WinSetWindowPos(hwndMainFrame,0,swpCurrent.x,swpCurrent.y,
                                  swpCurrent.cx,swpCurrent.cy,SWP_SIZE|SWP_MOVE);
                   }
                WinSetWindowPos(hwndMainFrame,0,0,0,0,0,SWP_SHOW | SWP_ACTIVATE);
                MainWindow();
                break;
             case USER_QUERY:
                QueryControls();
                DisplayControls();
                break;
             case USER_CHILDFONTCHANGE:
                FName=malloc(256);
                if (WinQueryPresParam((HWND)mp2,PP_FONTNAMESIZE,0,NULL,256,
                                   FName,0))
                   WinSetPresParam(hwnd,PP_FONTNAMESIZE,strlen(FName)+1,FName);
                else 
                   WinRemovePresParam(hwnd,PP_FONTNAMESIZE);
                free(FName);
                break;
             case USER_RECREATE:
                MainCreate();
                break;
             }
          break;
       case WM_CLOSE:
          QueryCtlColors();
          ChildDlg=TRUE;	// To prevent temporary minimization
          WinStoreWindowPos((char *)szIniApp,(char *)szKeyWindowPos,hwndMainFrame);
          WinPostMsg( hwndMain, WM_QUIT, (MPARAM)0,(MPARAM)0 );
          break;
       case WM_PRESPARAMCHANGED:
          if ((ULONG)mp1==PP_FONTNAMESIZE) MainCreate();
          else {
             WinQueryWindowRect(hwndMain,&rect);
             WinInvalidateRect(hwndMain,&rect,TRUE);
             }
          break;
       case WM_PAINT:
          hps = WinBeginPaint(hwnd, 0L,NULL);
          GpiCreateLogColorTable(hps,0,LCOLF_RGB,0,0,NULL);
          if (!WinQueryPresParam(hwndMain,PP_BACKGROUNDCOLOR,
                                 PP_BACKGROUNDCOLORINDEX,NULL,sizeof(lColor),
                                 &lColor,QPF_ID2COLORINDEX))
             lColor=WinQuerySysColor(HWND_DESKTOP,SYSCLR_WINDOW,0);
          WinQueryWindowRect(hwndMain,&rect);
          WinFillRect(hps, (PRECTL)&rect, lColor);
          WinEndPaint(hps);
          break;
       case WM_BUTTON2CLICK:
          WinQueryPointerPos(HWND_DESKTOP, &point);
          WinMapWindowPoints(HWND_DESKTOP,hwndMain,&point,1);
          WinPopupMenu(hwndMain,hwnd,hwndMenu,point.x, point.y,0,
                      PU_VCONSTRAIN | PU_HCONSTRAIN | PU_MOUSEBUTTON1);
          break;          
      case WM_BUTTON1MOTIONSTART:
      case WM_BUTTON2MOTIONSTART:
         WinSendMsg( hwndMainFrame, WM_TRACKFRAME, MPFROMSHORT( TF_MOVE | TF_SETPOINTERPOS ),
                     0);
         break;
      case WM_BUTTON1DBLCLK:
         Minimize();
         break;
       case WM_COMMAND:
       case WM_SYSCOMMAND:
          MainCommand(hwnd,mp1);
          break;
       case WM_CONTROL:
          MainControl(hwnd,mp1);
          break;
       case WM_SIZE:
	  rc=WinDefWindowProc(hwnd, msg, mp1, mp2);
          HideTitle();
          return rc;
       case WM_ACTIVATE:
          if ((!SHORT1FROMMP(mp1))&&(AutoMini)&&(!ChildDlg)) Minimize();
          if (SHORT1FROMMP(mp1))
             WinPostMsg( hwndMain, WM_USER, (MPARAM)USER_QUERY,(MPARAM)0 );
       default:	
	  return WinDefWindowProc(hwnd, msg, mp1, mp2);
       }
   return (MRESULT)0L;
}

int PPChange=FALSE;
MRESULT EXPENTRY ControlWndProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
   HPS	  hps;				/* Handle for painting		 */
   RECTL  rect;				/* Rectangle struct for painting */
   MRESULT rc;
   LONG lColor;

     switch (msg) {
       case WM_BUTTON2CLICK:
       case WM_BUTTON1MOTIONSTART:
       case WM_BUTTON2MOTIONSTART:
       case WM_CONTROL:
          return WinSendMsg(hwndMain,msg, mp1, mp2);
          break;

       case WM_USER:
          PPChange=TRUE;
          if ((int)mp1==USER_CHILDBGNDCHANGE) {
             if (WinQueryPresParam((HWND)mp2,PP_BACKGROUNDCOLOR,
                                 PP_BACKGROUNDCOLORINDEX,NULL,sizeof(lColor),
                                 &lColor,QPF_ID2COLORINDEX))
                WinSetPresParam(hwnd,PP_BACKGROUNDCOLOR,sizeof(lColor),&lColor);
             else {
                WinRemovePresParam(hwnd,PP_BACKGROUNDCOLOR);
                WinRemovePresParam(hwnd,PP_BACKGROUNDCOLORINDEX);
                }
             }
          else if ((int)mp1==USER_CHILDFGNDCHANGE) {
             if (WinQueryPresParam((HWND)mp2,PP_FOREGROUNDCOLOR,
                                 PP_FOREGROUNDCOLORINDEX,NULL,sizeof(lColor),
                                 &lColor,QPF_ID2COLORINDEX))
                WinSetPresParam(hwnd,PP_FOREGROUNDCOLOR,sizeof(lColor),&lColor);
             else {
                WinRemovePresParam(hwnd,PP_FOREGROUNDCOLOR);
                WinRemovePresParam(hwnd,PP_FOREGROUNDCOLORINDEX);
                }
             }
          PPChange=FALSE;
          break;
       case WM_PRESPARAMCHANGED:
          if ((ULONG)mp1==PP_FONTNAMESIZE)	// Must POST
             WinPostMsg(hwndMain,WM_USER,(MPARAM)USER_CHILDFONTCHANGE,(MPARAM)hwnd);
          else {
             WinQueryWindowRect(hwndMain,&rect);
             WinInvalidateRect(hwndMain,&rect,TRUE);
             }
          break;
       case WM_PAINT:
          hps = WinBeginPaint(hwnd, 0L,NULL);
          GpiCreateLogColorTable(hps,0,LCOLF_RGB,0,0,NULL);
          if (!WinQueryPresParam(hwnd,PP_BACKGROUNDCOLOR,
                                 PP_BACKGROUNDCOLORINDEX,NULL,sizeof(lColor),
                                 &lColor,QPF_ID2COLORINDEX))
             lColor=WinQuerySysColor(HWND_DESKTOP,SYSCLR_WINDOW,0);
          WinQueryWindowRect(hwnd,&rect);
          WinFillRect(hps, (PRECTL)&rect, lColor);
          WinEndPaint(hps);
          break;
      case WM_DRAWITEM:
         if (((LOUSHORT(mp1)&0xFF00)==IDC_LEFT)||
            ((LOUSHORT(mp1)&0xFF00)==IDC_RIGHT)) {
            if (((POWNERITEM)mp2)->idItem ==SDA_BACKGROUND)
               return (MRESULT)TRUE;
            if (((POWNERITEM)mp2)->idItem==SDA_SLIDERARM) {
               RECTL  rect;
               POINTL pnt;
               HPS hps;
               HWND hwndSl;

               hwndSl=WinWindowFromID(hwnd,LOUSHORT(mp1));
               rect=((POWNERITEM)mp2)->rclItem;
               hps=((POWNERITEM)mp2)->hps;

               GpiCreateLogColorTable(hps,0,LCOLF_RGB,0,0,NULL);
               if (!WinQueryPresParam(hwndSl,PP_BUTTONBACKGROUNDCOLOR,
                                 0,NULL,sizeof(lColor),
                                 &lColor,0))
                  lColor=WinQuerySysColor(HWND_DESKTOP,SYSCLR_BUTTONMIDDLE,0);
               GpiSetColor(hps,lColor);
               pnt.x=rect.xLeft+6;
               pnt.y=rect.yTop-2;
               GpiMove(hps,&pnt);
               pnt.x=rect.xRight-7;
               pnt.y=rect.yBottom+2;
               GpiBox(hps,DRO_FILL,&pnt,1,1);
               if (!WinQueryPresParam(hwndSl,PP_BORDERLIGHTCOLOR,
                                 0,NULL,sizeof(lColor),
                                 &lColor,0))
                  lColor=RGB_WHITE;
               GpiSetColor(hps,lColor);
               pnt.x=rect.xLeft+6;
               pnt.y=rect.yTop;
               GpiMove(hps,&pnt);
               pnt.x=rect.xLeft;
               pnt.y=rect.yBottom;
               GpiBox(hps,DRO_FILL,&pnt,1,1);
               pnt.x=rect.xRight-5;
               pnt.y=rect.yTop-2;
               GpiBox(hps,DRO_FILL,&pnt,1,1);
               if (!WinQueryPresParam(hwndSl,PP_BORDERDARKCOLOR,
                                 0,NULL,sizeof(lColor),
                                 &lColor,0))
                  lColor=0x808080;
               GpiSetColor(hps,lColor);
               pnt.x=rect.xRight;
               pnt.y=rect.yBottom;
               GpiMove(hps,&pnt);
               pnt.x=rect.xRight-6;
               pnt.y=rect.yTop;
               GpiBox(hps,DRO_FILL,&pnt,1,1);
               pnt.x=rect.xLeft+6;
               pnt.y=rect.yBottom+2;
               GpiBox(hps,DRO_FILL,&pnt,1,1);
               return (MRESULT)TRUE;
               }
            }
         break;
       default:	
	  return WinDefWindowProc(hwnd, msg, mp1, mp2);
       }
   return (MRESULT)0L;
}

void SubClassWndProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2) {

   if ((msg==WM_PRESPARAMCHANGED)&&(!PPChange)) {
      if ((ULONG)mp1==PP_FONTNAMESIZE)
             WinPostMsg(hwndMain,WM_USER,(MPARAM)USER_CHILDFONTCHANGE,(MPARAM)hwnd);
      else if ((ULONG)mp1==PP_BACKGROUNDCOLOR)
             WinPostMsg(WinQueryWindow(hwnd,QW_PARENT),WM_USER,
                        (MPARAM)USER_CHILDBGNDCHANGE,(MPARAM)hwnd);
      else if ((ULONG)mp1==PP_FOREGROUNDCOLOR)
             WinPostMsg(WinQueryWindow(hwnd,QW_PARENT),WM_USER,
                        (MPARAM)USER_CHILDFGNDCHANGE,(MPARAM)hwnd);
      }
   }
MRESULT EXPENTRY ButtonWndProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2) {
   SubClassWndProc(hwnd, msg, mp1, mp2);
   if ((msg==WM_BUTTON2CLICK)||(msg==WM_BUTTON2MOTIONSTART))
      return WinSendMsg(hwndMain,msg, mp1, mp2);
   return OldButtonWndProc(hwnd, msg, mp1, mp2);
   }
MRESULT EXPENTRY StaticWndProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2) {
   SubClassWndProc(hwnd, msg, mp1, mp2);
   return OldStaticWndProc(hwnd, msg, mp1, mp2);
   }
MRESULT EXPENTRY ListboxWndProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2) {
   LONG lColor,lColor1;

   if ((msg==WM_PRESPARAMCHANGED)&&((ULONG)mp1==PP_BACKGROUNDCOLOR)) {
      if (!WinQueryPresParam(hwnd,PP_BACKGROUNDCOLOR,
                                 PP_BACKGROUNDCOLORINDEX,NULL,sizeof(lColor),
                                 &lColor,QPF_ID2COLORINDEX)) lColor=-1;
      if (lColor==-1) {
         WinRemovePresParam(hwnd,PP_BORDERLIGHTCOLOR);
         WinRemovePresParam(hwnd,PP_BORDERDARKCOLOR);
         }
      else {
         lColor1=RGBMix(lColor,RGB_WHITE);
         WinSetPresParam(hwnd,PP_BORDERLIGHTCOLOR,sizeof(lColor1),&lColor1);
         lColor1=RGBMix(lColor,RGB_BLACK);
         WinSetPresParam(hwnd,PP_BORDERDARKCOLOR,sizeof(lColor1),&lColor1);
         }
      }
   else if ((msg!=WM_PRESPARAMCHANGED)||((ULONG)mp1!=PP_FOREGROUNDCOLOR))
      SubClassWndProc(hwnd, msg, mp1, mp2);
   if ((msg==WM_BUTTON2CLICK)||(msg==WM_BUTTON2MOTIONSTART))
      return WinSendMsg(hwndMain,msg, mp1, mp2);
   return OldListboxWndProc(hwnd, msg, mp1, mp2);
   }
MRESULT EXPENTRY SliderWndProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2) {
   LONG lColor,lColor1;
   RECTL  rect;				/* Rectangle struct for painting */
 
   if ((msg==WM_PRESPARAMCHANGED)&&((ULONG)mp1==PP_FOREGROUNDCOLOR)) {
      if (WinQueryPresParam(hwnd,PP_FOREGROUNDCOLOR,
                                 PP_FOREGROUNDCOLORINDEX,NULL,sizeof(lColor),
                                 &lColor,QPF_ID2COLORINDEX|QPF_NOINHERIT)) {
         WinSetPresParam(hwnd,PP_BUTTONBACKGROUNDCOLOR,sizeof(lColor),&lColor);
         lColor1=RGBMix(lColor,RGB_WHITE);
         WinSetPresParam(hwnd,PP_BORDERLIGHTCOLOR,sizeof(lColor1),&lColor1);
         lColor1=RGBMix(lColor,RGB_BLACK);
         WinSetPresParam(hwnd,PP_BORDERDARKCOLOR,sizeof(lColor1),&lColor1);
         WinPostMsg(hwndMain,WM_USER,(MPARAM)USER_RECREATE,(MPARAM)0);
         }
      }
   else SubClassWndProc(hwnd, msg, mp1, mp2);
   return OldSliderWndProc(hwnd, msg, mp1, mp2);
   }

int MyCallBack() {
   WinPostMsg(hwndMain, WM_USER, (MPARAM)USER_QUERY,(MPARAM)0 );
   return;
   }

/************************************************************************\
|            Small dialogs                                               |
\************************************************************************/

MRESULT EXPENTRY AboutDlgProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
   CHAR	szTitle[MESSAGELEN];
   MRESULT sRC;

     switch (msg) {
       case WM_INITDLG:
          WinLoadString(hab, 0, IDS_TITLE, MESSAGELEN, szTitle);
          WinSetWindowText(WinWindowFromID(hwnd,IDC_ABOUTAPP),szTitle);
 	  break;
       case WM_COMMAND:
          if (LOUSHORT(mp1)==IDC_OK) WinDismissDlg(hwnd, TRUE);
          break;
       default:	
	  sRC = WinDefDlgProc(hwnd, msg, mp1, mp2);
	  return sRC;
       }
   return (MRESULT)0L;
}

MRESULT EXPENTRY InfoDlgProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
   MRESULT sRC;
   IPT iptOffset;
   HWND hwndMLE;

     switch (msg) {
       case WM_INITDLG:
          {
          char tbuf[80];
          char tbuf1[10];

          hwndMLE=WinWindowFromID(hwnd,IDC_INFOMLE);
          WinSendMsg(hwndMLE, MLM_DISABLEREFRESH, NULL, NULL);
          WinSendMsg(hwndMLE, MLM_SETSEL, MPFROMSHORT(NULL),
                     (MPARAM) WinSendMsg(hwndMLE, MLM_QUERYTEXTLENGTH,NULL, NULL));
          WinSendMsg(hwndMLE, MLM_CLEAR, NULL, NULL);
          WinSendMsg(hwndMLE, MLM_SETIMPORTEXPORT,
                       MPFROMP((PBYTE)InfoBuf),(MPARAM)strlen(InfoBuf));
          iptOffset=0;
          WinSendMsg(hwndMLE, MLM_IMPORT, MPFROMP(&iptOffset),
                          (MPARAM)strlen(InfoBuf));
          WinSendMsg(hwndMLE, MLM_ENABLEREFRESH, NULL, NULL);
          if (UsingPipe) {
             WinLoadString(hab, 0, IDS_USINGPIPE, 79, tbuf);
             strncat(tbuf,szPipeName,79);
             }
          else {
             WinLoadString(hab, 0, IDS_USINGDRV, 79, tbuf);
             strncat(tbuf,szPddName+5,79);
             }
          if (strlen(tbuf)<50) {
             WinLoadString(hab, 0, IDS_APILEVEL, 79-strlen(tbuf), tbuf+strlen(tbuf));
             _itoa(ApiLevel,tbuf1,10);
             strncat(tbuf,tbuf1,79);
             }
          WinSetWindowText(WinWindowFromID(hwnd,IDC_INFODRV),tbuf);
 	  break;
          }
       case WM_COMMAND:
          if (LOUSHORT(mp1)==IDC_OK) WinDismissDlg(hwnd, TRUE);
          break;
       default:	
	  sRC = WinDefDlgProc(hwnd, msg, mp1, mp2);
	  return sRC;
       }
   return (MRESULT)0L;
}


/************************************************************************\
|                   Settings dialogs                                     |
\************************************************************************/

static HWND hwndControls1,hwndControls2,hwndOther,hwndLayout;

static	 CHAR	    Drive[]={0,0,0};
/*********************************************************************
 *
 *  Name:   OpenFileDialog
 *
 *  Purpose: open the standard file open dialog as file extention
 *	     and return the filename
 *
 *  Usage:   called when the user needs to supply a name for
 *	     the file to be opened
 *
 *  Method:  calls the standard file open dialog to get the
 *	     file name.
 *
 *  Parameters: HWD hwnd	 Handle of the owner window.
 *		PSZ szTitle	 Title of open dialog.
 *		PSZ pszFileExt	 File extention. (for example : *.txt)
 *		PSZ pszFullPath	 PSZ for returning the file name.
 *
 *  Returns: TRUE if successful in getting a file name, FALSE
 *		if not in pushing CANCEL
 *	     PSZ pszFullPath pointer to filename (full path)
 *
 **********************************************************************/
BOOL OpenFileDialog(HWND hwndOwner,
		    PSZ szTitle,
		    PSZ szFileExt,
		    PSZ szFullPath,PSZ szOkButton)
{
   static   PSZ	       ppszdefaultdrivelist[] = {NULL};

	    FILEDLG    fdg;		  /* file dialog structure	     */

   fdg.cbSize = sizeof(FILEDLG);	  /* Size of FILEDLG.		     */
   fdg.pszTitle = szTitle;		  /* String to display in title bar. */
   fdg.pszOKButton = szOkButton;	  /* String to display in OK button. */
   fdg.ulUser = 0L;			  /* User defined field.	     */
   fdg.fl = FDS_HELPBUTTON |		  /* FDS flags.			     */
	    FDS_CENTER |
	    FDS_OPEN_DIALOG;
   fdg.pfnDlgProc = (PFNWP)NULL;
   fdg.lReturn = 0L;			  /* Result code from dialog dismissal. */
   fdg.lSRC = 0L;			  /* System return code.	  */
   fdg.hMod = 0;			  /* Custom file dialog template. */
   fdg.usDlgId = 0;			  /* Custom file dialog ID.	  */
   fdg.x = 0;				  /* X coordinate of the dialog.  */
   fdg.y = 0;				  /* Y coordinate of the dialog.  */

   /* set selected fully qualified path */
   strcpy( fdg.szFullFile, szFileExt);

   fdg.pszIType = 0L;			  /* Pointer to string containing   */
					  /*   Initial EA type filter.	    */
   fdg.papszITypeList = 0L;		  /* Pointer to table of pointers   */
					  /*  that point to null terminated */
   if (!Drive[0])			  /*  Type strings.		    */
       fdg.pszIDrive = 0L;		  /* Pointer to string containing   */
   else					  /*  initial drive.		    */
       fdg.pszIDrive = Drive;

   fdg.papszIDriveList = (PAPSZ)ppszdefaultdrivelist;
					  /* Pointer to table of pointers   */
					  /*  that point to null terminated */
					  /*  Drive strings.		    */
   fdg.sEAType = 0;			  /* Selected file's EA Type.       */
   fdg.papszFQFilename = 0;		  /* Pointer to table of pointers   */
					  /*  point to null terminated	    */
					  /*  FQFname strings.		    */
   fdg.ulFQFCount = 0;			  /* Numbers of file selected.	    */

   /* get the file */

   if (!WinFileDlg(HWND_DESKTOP, hwndOwner, (PFILEDLG)&fdg))
      return FALSE;

   if (fdg.lReturn == DID_CANCEL)
      return FALSE;

   /* copy file name into file name buffer */

   strcpy(szFullPath, fdg.szFullFile);
/*   strcpy(szFileExt, fdg.szFullFile); */
   strncpy(Drive,fdg.szFullFile,2);	    /* Always Contains Drive letter */

   return TRUE;
} /* End of OpenFileDialog */

void SettingsInit(HWND hwnd) {
   int i;
   char Buff[32];
   HWND hwndBook;
   ULONG ulPageId;
   PAGEINFO pginfo;
   PDLGTEMPLATE pDlgt;   

   hwndBook=WinWindowFromID(hwnd,IDC_NOTEBOOK);

   ulPageId = (ULONG)WinSendMsg(hwndBook,BKM_INSERTPAGE, NULL,
        MPFROM2SHORT((BKA_AUTOPAGESIZE | BKA_MAJOR | BKA_STATUSTEXTON),
        BKA_LAST));
   WinLoadString(hab, 0, IDS_ACTIVECTLS, 31, Buff);
   WinSendMsg(hwndBook,BKM_SETTABTEXT, MPFROMLONG(ulPageId),MPFROMP(Buff));
   WinLoadString(hab, 0, IDS_PAGE12, 31, Buff);
   WinSendMsg(hwndBook,BKM_SETSTATUSLINETEXT, MPFROMLONG(ulPageId),MPFROMP(Buff));
   DosGetResource(0,RT_DIALOG,IDD_CONTROLS1,(PPVOID)&pDlgt);
   hwndControls1=WinCreateDlg(hwnd,hwndBook,ControlsDlgProc,pDlgt,NULL);
   DosFreeResource((PVOID)pDlgt);
   WinSendMsg(hwndBook,BKM_SETPAGEWINDOWHWND,MPFROMLONG(ulPageId),
              MPFROMHWND(hwndControls1));

   ulPageId = (ULONG)WinSendMsg(hwndBook,BKM_INSERTPAGE, NULL,
        MPFROM2SHORT((BKA_AUTOPAGESIZE | BKA_STATUSTEXTON),
        BKA_LAST));
   WinLoadString(hab, 0, IDS_PAGE22, 31, Buff);
   WinSendMsg(hwndBook,BKM_SETSTATUSLINETEXT, MPFROMLONG(ulPageId),MPFROMP(Buff));
   DosGetResource(0,RT_DIALOG,IDD_CONTROLS2,(PPVOID)&pDlgt);
   hwndControls2=WinCreateDlg(hwnd,hwndBook,ControlsDlgProc,pDlgt,NULL);
   DosFreeResource((PVOID)pDlgt);
   WinSendMsg(hwndBook,BKM_SETPAGEWINDOWHWND,MPFROMLONG(ulPageId),
              MPFROMHWND(hwndControls2));
  
   ulPageId = (ULONG)WinSendMsg(hwndBook,BKM_INSERTPAGE, NULL,
        MPFROM2SHORT((BKA_AUTOPAGESIZE | BKA_MAJOR),
        BKA_LAST));
   WinLoadString(hab, 0, IDS_LAYOUT, 31, Buff);
   WinSendMsg(hwndBook,BKM_SETTABTEXT, MPFROMLONG(ulPageId),MPFROMP(Buff));
   DosGetResource(0,RT_DIALOG,IDD_LAYOUT,(PPVOID)&pDlgt);
   hwndLayout=WinCreateDlg(hwnd,hwndBook,LayoutDlgProc,pDlgt,NULL);
   DosFreeResource((PVOID)&pDlgt);
   WinSendMsg(hwndBook,BKM_SETPAGEWINDOWHWND,MPFROMLONG(ulPageId),
              MPFROMHWND(hwndLayout));

   ulPageId = (ULONG)WinSendMsg(hwndBook,BKM_INSERTPAGE, NULL,
        MPFROM2SHORT((BKA_AUTOPAGESIZE | BKA_MAJOR),
        BKA_LAST));
   WinLoadString(hab, 0, IDS_OTHER, 31, Buff);
   WinSendMsg(hwndBook,BKM_SETTABTEXT, MPFROMLONG(ulPageId),MPFROMP(Buff));
   DosGetResource(0,RT_DIALOG,IDD_OTHER,(PPVOID)&pDlgt);
   hwndOther=WinCreateDlg(hwnd,hwndBook,OtherDlgProc,pDlgt,NULL);
   DosFreeResource((PVOID)&pDlgt);
   WinSendMsg(hwndBook,BKM_SETPAGEWINDOWHWND,MPFROMLONG(ulPageId),
              MPFROMHWND(hwndOther));

   return;
   }

void SettingsCommand(HWND hwnd,MPARAM mp1) {
   int i;
   char Buff[32];
   unsigned int OldBase;
   HWND hwndCtl;

   switch(LOUSHORT(mp1)) {
      case IDC_CANCEL:
         WinDismissDlg(hwnd, FALSE);
         break;
      case IDC_DEFAULTS:
         WinSendDlgItemMsg(hwndLayout, IDC_SHOWMENU,BM_SETCHECK,(MPARAM)TRUE,(MPARAM)0);
         WinSendDlgItemMsg(hwndLayout, IDC_SHOWTITL,BM_SETCHECK,(MPARAM)TRUE,(MPARAM)0);
         WinSendDlgItemMsg(hwndLayout, IDC_NARROWSL,BM_SETCHECK,(MPARAM)FALSE,(MPARAM)0);
         WinSendDlgItemMsg(hwndLayout, IDC_SHORTSL,BM_SETCHECK,(MPARAM)FALSE,(MPARAM)0);
         WinLoadString(hab, 0, IDS_LOCK, 32, Buff);
         WinSetWindowText(WinWindowFromID(hwndLayout,IDC_LOCKNAME),Buff);
         WinLoadString(hab, 0, IDS_OFF, 32, Buff);
         WinSetWindowText(WinWindowFromID(hwndLayout,IDC_OFFNAME),Buff);
         WinLoadString(hab, 0, IDS_MUTE, 32, Buff);
         WinSetWindowText(WinWindowFromID(hwndLayout,IDC_MUTENAME),Buff);
         WinLoadString(hab, 0, IDS_BOTH, 32, Buff);
         WinSetWindowText(WinWindowFromID(hwndLayout,IDC_BOTHNAME),Buff);
         WinLoadString(hab, 0, IDS_SP, 32, Buff);
         WinSetWindowText(WinWindowFromID(hwndLayout,IDC_SPNAME),Buff);
         WinLoadString(hab, 0, IDS_LF, 32, Buff);
         WinSetWindowText(WinWindowFromID(hwndLayout,IDC_LFNAME),Buff);
         WinLoadString(hab, 0, IDS_CT, 32, Buff);
         WinSetWindowText(WinWindowFromID(hwndLayout,IDC_CTNAME),Buff);
         WinLoadString(hab, 0, IDS_HF, 32, Buff);
         WinSetWindowText(WinWindowFromID(hwndLayout,IDC_HFNAME),Buff);
         _itoa(DEF_CELLS_IN_LINE,Buff,10);
         WinSetWindowText(WinWindowFromID(hwndLayout,IDC_MAXCTLS),Buff);
         WinSendDlgItemMsg(hwndOther, IDC_RESTORE,BM_SETCHECK,(MPARAM)FALSE,0);
         WinSendDlgItemMsg(hwndOther, IDC_AUTOMINI,BM_SETCHECK,(MPARAM)FALSE,0);
         WinSetWindowText(WinWindowFromID(hwndOther,IDC_INSTNAME),"");
         WinSetWindowText(WinWindowFromID(hwndOther,IDC_ICONFILE),"");
         for (i=0;i<NControls;i++) {
            if (WinWindowFromID(hwndControls1,Controls[i].nameID))
               hwndCtl=hwndControls1;
            else hwndCtl=hwndControls2;
            if ((ApiLevel>1)||(!(Controls[i].flags&FLAG_LOCKABLE)))
               WinSendDlgItemMsg(hwndCtl,Controls[i].activeID,BM_SETCHECK,
                                 (MPARAM)TRUE,0);
            else WinSendDlgItemMsg(hwndCtl,Controls[i].activeID,
                                   BM_SETCHECK,(MPARAM)TRUE,0);
            WinLoadString(hab, 0, Controls[i].defnameID, 31, Buff);
            WinSetWindowText(WinWindowFromID(hwndCtl,Controls[i].nameID),
                             Buff);
            if (Controls[i].flags&FLAG_STEREO)
               WinSendDlgItemMsg(hwndCtl,Controls[i].monoID,
                                   BM_SETCHECK,(MPARAM)FALSE,0);
            } 
         break;
      case IDC_OK:
         RestoreState=(int)WinSendDlgItemMsg(hwndOther,IDC_RESTORE,
                                             BM_QUERYCHECK,0,0);
         AutoMini=(int)WinSendDlgItemMsg(hwndOther,IDC_AUTOMINI,BM_QUERYCHECK,
                                         0,0);
         WinQueryWindowText(WinWindowFromID(hwndOther,IDC_INSTNAME),
                            MAX_INST_NAME,szInstName);
         WinQueryWindowText(WinWindowFromID(hwndLayout,IDC_MAXCTLS),32,Buff);
         i=atoi(Buff);
         WinQueryWindowText(WinWindowFromID(hwndLayout,IDC_LOCKNAME),32,szLock);
         WinQueryWindowText(WinWindowFromID(hwndLayout,IDC_MUTENAME),32,szMute);
         WinQueryWindowText(WinWindowFromID(hwndLayout,IDC_OFFNAME),32,szOff);
         WinQueryWindowText(WinWindowFromID(hwndLayout,IDC_BOTHNAME),32,szBoth);
         WinQueryWindowText(WinWindowFromID(hwndLayout,IDC_SPNAME),32,szSp);
         WinQueryWindowText(WinWindowFromID(hwndLayout,IDC_CTNAME),32,szCt);
         WinQueryWindowText(WinWindowFromID(hwndLayout,IDC_LFNAME),32,szLF);
         WinQueryWindowText(WinWindowFromID(hwndLayout,IDC_HFNAME),32,szHF);
         ShowMenu=(int)WinSendDlgItemMsg(hwndLayout,IDC_SHOWMENU,
                                             BM_QUERYCHECK,0,0);
         ShowTitlebar=(int)WinSendDlgItemMsg(hwndLayout,IDC_SHOWTITL,
                                             BM_QUERYCHECK,0,0);
         NarrowSliders=(int)WinSendDlgItemMsg(hwndLayout,IDC_NARROWSL,
                                             BM_QUERYCHECK,0,0);
         ShortSliders=(int)WinSendDlgItemMsg(hwndLayout,IDC_SHORTSL,
                                             BM_QUERYCHECK,0,0);
         WinQueryWindowText(WinWindowFromID(hwndOther,IDC_ICONFILE),CCHMAXPATH,szIconFile);
         if ((i>=MIN_CELLS_IN_LINE)&&(i<=MAX_CELLS_IN_LINE)) CellsInLine=i;
         for(i=0;i<NControls;i++) if (Controls[i].supported) {
            if (WinWindowFromID(hwndControls1,Controls[i].nameID))
               hwndCtl=hwndControls1;
            else hwndCtl=hwndControls2;
            Controls[i].active=(int)WinSendDlgItemMsg(hwndCtl,
                                    Controls[i].activeID,BM_QUERYCHECK,0,0);
            WinQueryWindowText(WinWindowFromID(hwndCtl,Controls[i].nameID),
                               32,Controls[i].name);
            if (Controls[i].flags&FLAG_STEREO) {
               if (WinSendDlgItemMsg(hwndCtl,
                                    Controls[i].monoID,BM_QUERYCHECK,0,0))
                  Controls[i].both|=BOTH_MONO;
               else Controls[i].both&=~BOTH_MONO;
               }

            }
         SaveProfile();
         WinDismissDlg(hwnd, FALSE);
         break;
      }
   return;
   }

MRESULT EXPENTRY SettingsDlgProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
   MRESULT sRC;

     switch (msg) {
       case WM_INITDLG:
          SettingsInit(hwnd);
 	  break;
       case WM_CLOSE:
          WinDismissDlg(hwnd, FALSE);
          break;
       case WM_COMMAND:
          SettingsCommand(hwnd,mp1);
          break;
       default:	
	  sRC = WinDefDlgProc(hwnd, msg, mp1, mp2);
	  return sRC;
       }
   return (MRESULT)0L;
}

MRESULT EXPENTRY ControlsDlgProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
   MRESULT sRC;
   int i;

     switch (msg) {
       case WM_INITDLG:
          for(i=0;i<NControls;i++) if (WinWindowFromID(hwnd,Controls[i].nameID)) {
             WinSendDlgItemMsg(hwnd,Controls[i].activeID,BM_SETCHECK,(MPARAM)Controls[i].active,0);
             WinSetWindowText(WinWindowFromID(hwnd,Controls[i].nameID),Controls[i].name);
             if (Controls[i].flags&FLAG_STEREO)
                WinSendDlgItemMsg(hwnd,Controls[i].monoID,BM_SETCHECK,(MPARAM)(Controls[i].both&BOTH_MONO),0);
             if (Controls[i].supported) {
                WinEnableWindow(WinWindowFromID(hwnd,Controls[i].activeID),TRUE);
                WinEnableWindow(WinWindowFromID(hwnd,Controls[i].nameID),TRUE);
                if (Controls[i].flags&FLAG_STEREO)
                   WinEnableWindow(WinWindowFromID(hwnd,Controls[i].monoID),TRUE);
                }
             }
 	  break;
       case WM_CLOSE:
          WinDismissDlg(hwnd, FALSE);
          break;
       default:	
	  sRC = WinDefDlgProc(hwnd, msg, mp1, mp2);
	  return sRC;
       }
   return (MRESULT)0L;
}

MRESULT EXPENTRY LayoutDlgProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
   MRESULT sRC;
   char Buff[16];

     switch (msg) {
       case WM_INITDLG:

          WinSetWindowText(WinWindowFromID(hwnd,IDC_LOCKNAME),szLock);
          WinSetWindowText(WinWindowFromID(hwnd,IDC_OFFNAME),szOff);
          WinSetWindowText(WinWindowFromID(hwnd,IDC_MUTENAME),szMute);
          WinSetWindowText(WinWindowFromID(hwnd,IDC_BOTHNAME),szBoth);
          WinSetWindowText(WinWindowFromID(hwnd,IDC_LFNAME),szLF);
          WinSetWindowText(WinWindowFromID(hwnd,IDC_HFNAME),szHF);
          WinSetWindowText(WinWindowFromID(hwnd,IDC_CTNAME),szCt);
          WinSetWindowText(WinWindowFromID(hwnd,IDC_SPNAME),szSp);
          _itoa(CellsInLine,Buff,10);
          WinSetWindowText(WinWindowFromID(hwnd,IDC_MAXCTLS),Buff);
          WinSendDlgItemMsg(hwnd, IDC_SHOWMENU,BM_SETCHECK,(MPARAM)ShowMenu,0);
          WinSendDlgItemMsg(hwnd, IDC_SHOWTITL,BM_SETCHECK,(MPARAM)ShowTitlebar,0);
          WinSendDlgItemMsg(hwnd, IDC_NARROWSL,BM_SETCHECK,(MPARAM)NarrowSliders,0);
          WinSendDlgItemMsg(hwnd, IDC_SHORTSL,BM_SETCHECK,(MPARAM)ShortSliders,0);
 	  break;
       case WM_CLOSE:
          WinDismissDlg(hwnd, FALSE);
          break;
       default:	
	  sRC = WinDefDlgProc(hwnd, msg, mp1, mp2);
	  return sRC;
       }
   return (MRESULT)0L;
}


const char szExt[]="*.ico;*.exe;*.dll";
MRESULT EXPENTRY OtherDlgProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
   MRESULT sRC;
   char Buff[16];

     switch (msg) {
       case WM_INITDLG:
          WinSendDlgItemMsg(hwnd, IDC_RESTORE,BM_SETCHECK,(MPARAM)RestoreState,0);
          WinSendDlgItemMsg(hwnd, IDC_AUTOMINI,BM_SETCHECK,(MPARAM)AutoMini,0);
          WinSendMsg(WinWindowFromID(hwnd,IDC_INSTNAME),EM_SETTEXTLIMIT,(MPARAM)MAX_INST_NAME,0);
          WinSetWindowText(WinWindowFromID(hwnd,IDC_INSTNAME),szInstName);
          WinSendMsg(WinWindowFromID(hwnd,IDC_ICONFILE),EM_SETTEXTLIMIT,(MPARAM)CCHMAXPATH,0);
          WinSetWindowText(WinWindowFromID(hwnd,IDC_ICONFILE),szIconFile);
 	  break;
       case WM_CLOSE:
          WinDismissDlg(hwnd, FALSE);
          break;
       case WM_COMMAND:
          if(LOUSHORT(mp1)==IDC_ICONLOC) {
             char szTit[MESSAGELEN];

             WinLoadString(hab, 0, IDS_SELECTICON, MESSAGELEN, szTit);
             WinLoadString(hab, 0, IDS_OK, 15, Buff);
             if (OpenFileDialog(WinQueryWindow(hwnd,QW_PARENT),szTit,
                                (char *)szExt,szIconFile,Buff)) {
                WinSetWindowText(WinWindowFromID(hwnd,IDC_ICONFILE),szIconFile);
                }
             }
          break;
       default:	
	  sRC = WinDefDlgProc(hwnd, msg, mp1, mp2);
	  return sRC;
       }
   return (MRESULT)0L;
}
