/* 

Pipe Mixer for Crystal Semi mode 2 chips by Lesha Bogdanow
Copyright (C) 1999-2000  Lesha Bogdanow

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */

#define USE_OS2_TOOLKIT_HEADERS
#define INCL_DOSPROCESS
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <os2.h>
#include <sys\hw.h>

#include "ioctl90.h"
#include "PipeMixPvt.h"

const char szLogo[]="PipeMixer demon for Crystal mode 2 chips. v.0.03";

int	BasePort=0;
int	LockRate=0;
int	AccessedPort=0;

MIXSTRUCT Mono={0,0,0};		
MIXSTRUCT AUX1={0,0,0};		// IOCTL90: Line 
MIXSTRUCT AUX2={0,0,0};		// IOCTL90: CD
MIXSTRUCT Line={0,0,0};		// IOCTL90: Aux
MIXSTRUCT DAC={2,0,0};		// Released initially
MIXSTRUCT ADC={2,0,0};		// Released initially
MIXSTRUCT RecMux={2,0,0};	// Released initially

int	LockPriority=PRTYC_TIMECRITICAL;
TID	LockTID=0;

static void SetRegister(unsigned char idx,unsigned char data) {
   if (BasePort) {
      if (BasePort!=AccessedPort) {
         _portaccess(BasePort,BasePort+1);
         AccessedPort=BasePort;
         }
      _outp8(BasePort,idx);
      _outp8(BasePort+1,data);
      }
   }

static unsigned char GetRegister(unsigned char idx) {
   if (BasePort) {
      if (BasePort!=AccessedPort) {
         _portaccess(BasePort,BasePort+1);
         AccessedPort=BasePort;
         }
      _outp8(BasePort,idx);
      return _inp8(BasePort+1);
      }
   return 0;
   }

static void Limit100(MIXSTRUCT *ms) {
   if (ms->VolumeL>100) ms->VolumeL=100;
   if (ms->VolumeR>100) ms->VolumeR=100;
   }

static void _GetInp(MIXSTRUCT *ms, int rl, int rr) {
   unsigned char t;

   t=GetRegister(rl);
   if (t&0x80) ms->Mute|=1;
   t&=0x1F;
   ms->VolumeL=(32-t)*100/32;
   t=GetRegister(rr)&0x1F;
   ms->VolumeR=(32-t)*100/32;
   Limit100(ms);
   }

// Get hardware state at startup
static void GetState() {
   unsigned char t;

   t=GetRegister(0);
   switch (t&0xC0) {
      case 0:
         RecMux.VolumeL=I90SRC_AUX;
         break;
      case 0x40:
         RecMux.VolumeL=I90SRC_LINE;
         break;
      case 0x80:
         RecMux.VolumeL=I90SRC_MIC;
         break;
      default:
         RecMux.VolumeL=I90SRC_CD;
      }
   t=(t&0xF);
   if (t==0xF) ADC.Mute|=1;
   ADC.VolumeL=(16-t)*100/16;
   Limit100(&ADC);

   _GetInp(&AUX1,2,3);
   _GetInp(&AUX2,4,5);

   t=GetRegister(6);
   if (t&0x80) DAC.Mute|=1;
   t&=0x3F;
   DAC.VolumeL=(64-t)*100/64;
   t=GetRegister(7)&0x3F;
   DAC.VolumeR=(64-t)*100/64;
   Limit100(&DAC);

   _GetInp(&Line,18,19);

   t=GetRegister(26);
   if (t&0x80) Mono.Mute|=1;
   t&=0xF;
   Mono.VolumeL=(16-t)*100/16;
   Limit100(&Mono);
   }

static void _SetADC(int n) {
   unsigned char t;

   if (ADC.Mute&1) t=15;
   else t=16-(ADC.VolumeL*16)/100;
   if (t>15) t=15;
   t=(GetRegister(n)&0xC0) | t;
   SetRegister(n,t);
   }
static void SetADC() {
   if (ADC.Mute&2) return;
   _SetADC(0);
   _SetADC(1);
   }
static void SetRecMux() {
   unsigned char t;

   if (RecMux.Mute&2) return;

   if (RecMux.VolumeL&I90SRC_LINE) t=0x40;
   else if (RecMux.VolumeL&I90SRC_AUX) t=0;
   else if (RecMux.VolumeL&I90SRC_MIC) t=0x80;
   else t=0xC0;
   SetRegister(0,(GetRegister(0)&0x3F)|t);
   SetRegister(1,(GetRegister(1)&0x3F)|t);
   }
static void _SetInp(MIXSTRUCT *ms, int rl, int rr) {
   unsigned char t;

   t=32-(ms->VolumeL*32)/100;
   if (t>31) t=31;
   if (ms->Mute&1) t|=0x80;
   SetRegister(rl,t);
   t=32-(ms->VolumeR*32)/100;
   if (t>31) t=31;
   if (ms->Mute&1) t|=0x80;
   SetRegister(rr,t);
   }
static void SetAUX1() {
   _SetInp(&AUX1,2,3);
   }
static void SetAUX2() {
   _SetInp(&AUX2,4,5);
   }
static void SetDAC() {
   unsigned char t;

   if (DAC.Mute&2) return;

   t=64-(DAC.VolumeL*64)/100;
   if (t>63) t=63;
   if (DAC.Mute&1) t|=0x80;
   SetRegister(6,t);
   t=64-(DAC.VolumeR*64)/100;
   if (t>63) t=63;
   if (DAC.Mute&1) t|=0x80;
   SetRegister(7,t);
   }
static void SetLine() {
   _SetInp(&Line,18,19);
   }
static void SetMono() {
   unsigned char t;

   t=16-(Mono.VolumeL*16)/100;
   if (t>15) t=15;
   if (Mono.Mute&1) t|=0x80;
   SetRegister(26,t);
   }

VOID APIENTRY LockThread(ULONG Dummy) {
   DosSetPriority(PRTYS_THREAD, LockPriority, 0, 0);

   while(1) {
      DosSleep(LockRate);
      SetDAC();
      SetADC();
      SetRecMux();
      };
   }

// Initialize mixer. Return TRUE if Ok
int MixerInit(int argc, char *argv[]) {
   int i;

   for (i=1;i<argc;i++) {
     if (!stricmp(argv[i],"-b"))  {
        i++;
        if (i==argc) return FALSE;		// Syntax error
        BasePort=strtoul(argv[i],NULL,16)&0xFFFF;
        }
     else if (!stricmp(argv[i],"-l"))  {
        i++;
        if (i==argc) return FALSE;		// Syntax error
        LockRate=atoi(argv[i]);
        }
     else return FALSE;				// Syntax error
     }
   if (!BasePort) return FALSE;
   GetState();
   if (LockRate) DosCreateThread(&LockTID,LockThread,0,0,8192);
   ApiLevel=2;
   if (Debug) printf("API Level: %d, Base port: %x, Lock rate: %d\n",ApiLevel,
                      BasePort,LockRate);
   return TRUE;
   }

// Deinitialize mixer
void MixerClose() {
   if (LockTID) DosKillThread(LockTID);
   }


/* Process a command. Buff is an output buffer, the function number and
   a trailing space are already there. tokens - number of command tokens
   cmd - command, p1, p2, p3 - parameters. If some of them are not specified
   then default values (p1=100, p2=p1, p3=0) are supplied
*/
int ProcessCmd(char *Buff, int tokens, int cmd, int p1, int p2, int p3) {
   MIXSTRUCT MixStruct;

   if ((cmd&0xF0)==0x40) {	// Set command
      MixStruct.VolumeL=p1;
      MixStruct.VolumeR=p2;
      MixStruct.Mute=p3;

      switch(cmd) {
         case 0x40:
            Mono=MixStruct;
            SetMono();
            break;
         case 0x42:		// Fake MIC
            break;
         case 0x43:		// Line
            AUX1=MixStruct;
            SetAUX1();
            break;
         case 0x44:		// CD
            AUX2=MixStruct;
            SetAUX2();
            break;
         case 0x46:		// Aux
            Line=MixStruct;
            SetLine();
            break;
         case 0x4D:		// DAC
            DAC=MixStruct;
            SetDAC();
            break;
         case 0x4E:		// RecMux
            RecMux=MixStruct;
            SetRecMux();
            break;
         case 0x4F:		// ADC
            ADC=MixStruct;
            SetADC();
            break;
         default:
            strcat(Buff,REP_UNSUPPORTED);
            return TRUE;
         }
      strcat(Buff,REP_OK);
      }
   else if ((cmd&0xF0)==0x60) {	// Query command
      switch(cmd) {
         case 0x60:
            MixStruct=Mono;
            break;
         case 0x62:		// Fake MIC
            MixStruct.VolumeL=0;
            MixStruct.VolumeR=0;
            MixStruct.Mute=1;
            break;
         case 0x63:		// Line
            MixStruct=AUX1;
            break;
         case 0x64:		// CD
            MixStruct=AUX2;
            break;
         case 0x66:		// Aux
            MixStruct=Line;
            break;
         case 0x6D:		// DAC
            MixStruct=DAC;
            break;
         case 0x6E:		// RecMux
            MixStruct=RecMux;
            break;
         case 0x6F:		// ADC
            MixStruct=ADC;
            break;
         default:
            strcat(Buff,REP_UNSUPPORTED);
            return TRUE;
         }
      sprintf(Buff+strlen(Buff),"%d %d %x",MixStruct.VolumeL,MixStruct.VolumeR,MixStruct.Mute);
      }
   else strcat(Buff,REP_UNSUPPORTED);
   return TRUE;
   }

