;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;
;;;                Centre for Speech Technology Research                  ;;
;;;                     University of Edinburgh, UK                       ;;
;;;                       Copyright (c) 1996,1997                         ;;
;;;                        All Rights Reserved.                           ;;
;;;                                                                       ;;
;;;  Permission to use, copy, modify,  and licence this software and its  ;;
;;;  documentation for any purpose, is hereby granted without fee,        ;;
;;;  subject to the following conditions:                                 ;;
;;;   1. The code must retain the above copyright notice, this list of    ;;
;;;      conditions and the following disclaimer.                         ;;
;;;   2. Any modifications must be clearly marked as such.                ;;
;;;   3. Original authors' names are not deleted.                         ;;
;;;                                                                       ;;
;;;  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        ;;
;;;  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      ;;
;;;  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   ;;
;;;  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     ;;
;;;  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    ;;
;;;  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   ;;
;;;  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          ;;
;;;  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       ;;
;;;  THIS SOFTWARE.                                                       ;;
;;;                                                                       ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Support for an SGML based mark-up language        
;;;  Based on
;;;  Sproat, R., Taylor, P, Tanenblatt, M. and Isard, A. "A Markup    
;;;  Language for Text-to-Speech Synthesis", Eurospeech97, Rhodes,    
;;;  Greece, 1997.                                                    
;;;
;;;  Note this doesn't follow that work completely there are some
;;;  additions and some things missing.  The support in Festival is
;;;  to make it easy to immplement such mark up languages.
;;;
;;;  This is *not* an official version of STML 
;;;

(set! stml_omitted_mode nil)
(set! stml_word_features_stack nil)

(define (stml_token_to_words utt token name)
  "(stml_token_to_words utt token name)
STML mode token specific analysis."
  (cond
   (stml_omitted_mode
    ;; Don't say anything during this time
    (format t "ommiting %s\n" name)
    nil)
   ((string-equal "1" (utt.streamitem.feat utt token "stml_phonemes"))
    ;; Each token is a phoneme here
    (streamitem.set_feat token "phonemes" 
			 (format nil "%l" (list name)))
    (list name))
   ((string-equal "1" (utt.streamitem.feat utt token "stml_literal"))
    ;; Only deal with spell here
    (let ((subwords) (subword))
      (streamitem.set_feat token "pos" token.letter_pos)
      (mapcar
       (lambda (letter)
	 ;; might be symbols or digits
	 (set! subword (stml_previous_token_to_words utt token letter))
	 (if subwords
	     (set! subwords (append subwords subword))
	     (set! subwords subword)))
       (symbolexplode name))
      subwords))
   (t  
    (stml_previous_token_to_words utt token name))))

(defvar stml_elements
'(
  ("(STML" (ATTLIST UTT)
    ;; required to identify type 
    (voice_rab_diphone)  ;; so we know what state we start in
    nil
  )
  (")STML" (ATTLIST UTT)
    ;; required to identify end token
    (xxml_synth UTT)  ;;  Synthesis the remaining tokens
    nil
  )
  ;; Utterance break elements
  ("(LANGUAGE" (ATTLIST UTT)
   (xxml_synth UTT)
   ;; Select a new language
   (select_language (car (xxml_attval "ID" ATTLIST)))
   nil)
  ("(GENRE" (ATTLIST UTT)
   (xxml_synth UTT)
   ;; Select a new language
   (format t "GENRE %s not supported: ignored\n" 
	   (car (xxml_attval "GENRE" ATTLIST)))
   nil)
  ("(SPEAKER" (ATTLIST UTT)
   (xxml_synth UTT)
   ;; Select a new voice
   (cond
    ((equal? (car (xxml_attval "NAME" ATTLIST)) 'male1)
     (voice_rab_diphone))
    ((equal? (car (xxml_attval "NAME" ATTLIST)) 'male2)
     (voice_don_diphone))
    ((equal? (car (xxml_attval "NAME" ATTLIST)) 'male3)
     (voice_kd_diphone))
    (t
     (print "STML: selecting unknown voice")
     (voice_rab_diphone)))
   nil)
  ("(BOUND" (ATTLIST UTT)
   (if (string-equal "4" (car (xxml_attval "STRENGTH" ATTLIST)))
       (begin
	 (xxml_synth UTT)
	 nil)
       (let ((last_token (car (last (utt.stream UTT 'Token)))))
	 (if last_token
	     (streamitem.set_feat last_token "pbreak" "B"))
	 UTT)))
  ("(DIV" (ATLIST UTT)
   (xxml_synth UTT)
   (format t "DIV: utterance break and type ignored\n")
   nil)
  ("(PHONETIC" (ATTLIST UTT)
   (stml_push_word_features)
   (if (string-equal "NATIVE" (car (xxml_attval "SCHEME" ATTLIST)))
       (set! xxml_word_features 
	     (cons (list "stml_phonemes" "1") xxml_word_features))
       (format t "LITERAL: ignoring non-spell mode\n"))
   UTT)
  (")PHONETIC" (ATTLIST UTT)
   (set! xxml_word_features (stml_pop_word_features))
   UTT)
  ("(LITERAL" (ATTLIST UTT)
   ;; A spell type mode
   (stml_push_word_features)
   (if (string-equal "SPELL" (car (xxml_attval "MODE" ATTLIST)))
       (set! xxml_word_features 
	     (cons (list "stml_literal" "1") xxml_word_features))
       (format t "LITERAL: ignoring non-spell mode\n"))
   UTT)
  (")LITERAL" (ATTLIST UTT)
   (set! xxml_word_features (stml_pop_word_features))
   UTT)
  ("(DEFINE" (ATTLIST UTT)
    (xxml_synth UTT)
    (if (not (string-equal "NATIVE" (car (xxml_attval "SCHEME" ATTLIST))))
	(format t "DEFINE: unsupport SCHEME %s, definition ignored\n"
		(car (xxml_attval "SCHEME" ATTLIST)))
	(lex.add.entry
	 (list
	  (car (xxml_attval "WORDS" ATTLIST))   ;; head form
	  nil          ;; pos
	  (lex.syllabify.phstress (xxml_attval "PRONS" ATTLIST)))))
    nil)
  ("(OMITTED" (ATTLIST UTT)
   (xxml_synth UTT)
   (set! stml_omitted_mode t)
   nil)
  (")OMITTED" (ATTLIST UTT)
   (set! stml_omitted_mode nil)
   nil)
  ("(CALL" (ATTLIST UTT)
   (xxml_synth UTT)
   (if (string-matches (car (xxml_attval "ENGID" ATTLIST)) "festival.*")
       (let ((comstr ""))
	 (mapcar
	  (lambda (c) (set! comstr (string-append comstr " " c)))
	  (xxml_attval "COMMAND" ATTLIST))
	 (eval (read-from-string comstr))))
   nil)
  ;;; This may cause an utterance boundary
  ("(AIMG" (ATTLIST UTT)
   (xxml_synth UTT)
   (if (not stml_omitted_mode)
       (apply_hooks tts_hooks
		    (eval (list 'Utterance 'Wave 
				(car (xxml_attval "SRC" ATTLIST))))))
   nil)
  ("(EMPH" (ATTLIST UTT)
   ;; Festival is particularly bad at adding specific emphasis
   ;; that's what happens when you use statistical methods that
   ;; don't include any notion of emphasis
   ;; This is *not* recursive
   (stml_push_word_features)
   (set! xxml_word_features 
	 (cons (list "EMPH" "1") xxml_word_features))
   UTT)
  (")EMPH" (ATTLIST UTT)
   (set! xxml_word_features (stml_pop_word_features))
   UTT)
  ("(RATE" (ATTLIST UTT)
   ;; This is *not* recursive
   (stml_push_word_features)
   (let ((rate  (xxml_attval "SPEED" ATTLIST)))
     (if rate
	 (set! xxml_word_features 
	       (cons (list "dur_stretch" (parse-number (car rate)))
		     xxml_word_features)))
     UTT))
  (")RATE" (ATTLIST UTT)
   (set! xxml_word_features (stml_pop_word_features))
   UTT)
  ("(INTONAT" (ATTLIST UTT)
   (format t "INTONAT: ignored\n")
   UTT)
  ("(WORD" (ATTLIST UTT)
   ;; a word in-line
   (let ((name   (xxml_attval "NAME" ATTLIST))
	 (pos    (xxml_attval "POS" ATTLIST))
	 (accent (xxml_attval "ACCENT" ATTLIST))
	 (tone   (xxml_attval "TONE" ATTLIST))
	 (phonemes (xxml_attval "PHONEMES" ATTLIST))
	 token)
     (utt.streamitem.insert UTT 'Token)  ;; add new Token
     (set! token (utt.stream.tail UTT 'Token))
     (streamitem.set_name token (car name))
     (if pos (streamitem.set_feat token "pos" (car pos)))
     (if accent (streamitem.set_feat token "accent" (car accent)))
     (if tone (streamitem.set_feat token "tone" (car tone)))
     (if phonemes (streamitem.set_feat token "phonemes" 
				       (format nil "%l" phonemes)))
     UTT))
))

(define (stml_init_func)
  "(stml_init_func)
Initialisation for STML mode"
  (voice_rab_diphone)
  (set! stml_previous_elements xxml_elements)
  (set! xxml_elements stml_elements)
  (set! stml_previous_token_to_words english_token_to_words)
  (set! english_token_to_words stml_token_to_words)
  (set! token_to_words stml_token_to_words))

(define (stml_exit_func)
  "(stml_exit_func)
Exit function for STML mode"
  (set! xxml_elements stml_previous_elements)
  (set! token_to_words stml_previous_token_to_words)
  (set! english_token_to_words stml_previous_token_to_words))

(define (stml_push_word_features)
"(stml_push_word_features)
Save current word features on stack."
  (set! stml_word_features_stack 
	(cons xxml_word_features stml_word_features_stack)))

(define (stml_pop_word_features)
"(stml_pop_word_features)
Pop word features from stack."
  (let ((r (car stml_word_features_stack)))
    (set! stml_word_features_stack (cdr stml_word_features_stack))
    r))

(set! tts_text_modes
   (cons
    (list
      'stml   ;; mode name
      (list         ;; email mode params
       (list 'init_func stml_init_func)
       (list 'exit_func stml_exit_func)
       '(analysis_type xxml)
       (list 'filter 
	     (format nil "%s -D %s " sgml_parse_progname libdir))))
    tts_text_modes))

(provide 'stml-mode)
