/*
    /dev/audio "emulation" program
    Copyright (C) 1998 by Andrew Zabolotny <bit@eltech.ru>
  
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.
  
    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.
  
    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <io.h>
#include <fcntl.h>
#include <gnu/getopt.h>
#define INCL_DOS
#include "libDART.h"

static char *programname;
static char *programversion = "0.1.0";

static struct option long_options[] =
{
  {"help", no_argument, 0, 'h'},
  {"version", no_argument, 0, 'v'},
  {"bytes", no_argument, 0, 'b'},
  {"words", no_argument, 0, 'w'},
  {"rate", required_argument, 0, 'r'},
  {"channels", required_argument, 0, 'c'},
  {"format", required_argument, 0, 'f'},
  {"device", required_argument, 0, 'd'},
  {0, no_argument, 0, 0}
};

static void display_help ()
{
  printf ("\n/dev/audio emulator version %s\n", programversion);
  printf ("Copyleft (L) 1998 FRIENDS software\n\n");
  printf ("Usage: devaudio [option/s] [filename/-]\n");
  printf ("By default devaudio will read raw data from stdin and dump it to the default\n");
  printf ("wave audio device. If you specify a filename, the data will be read from\n");
  printf ("the specified file. The data stream is interpreted depending on options:\n\n");
  printf ("  -d# --device=#    Choose audio device index (default = 0)\n");
  printf ("  -b  --bytes       One byte (8 bit) per sample\n");
  printf ("  -w  --words       One word (16 bit) per sample (default)\n");
  printf ("  -r# --rate=#      Define sampling rate (default = 44100Hz)\n");
  printf ("  -c# --channels=#  Define number of channels (default = 2 - stereo)\n");
  printf ("  -f# --format=#    Define sample format (default = pcm)\n");
  printf ("                    Supported formats: pcm, adpcm, ibm_cvsd,\n");
  printf ("                    alaw mulaw, oki_adpcm, dvi_adpcm, digistd,\n");
  printf ("                    digifix, avc_adpcm, ibm_adpcm, ibm_mulaw,\n");
  printf ("                    ibm_alaw, ct_adpcm, mpeg1\n");
  printf ("  -h  --help        Display usage help\n");
  printf ("  -v  --version     Show program version\n");
}

static void CheckError (DART &dart)
{
  if (dart.Error ())
  {
    printf ("%s: DART error: %s\n", programname, dart.GetErrorString ());
    exit (-1);
  }
}

static int input_handle = 0;

size_t dartCallback (void *Buffer, size_t BufferSize)
{
  size_t bytesread = 1, count = 0;
  while ((bytesread) && (count < BufferSize))
  {
    bytesread = read (input_handle, &((char *)Buffer) [count], BufferSize - count);
    count += bytesread;
  }
  return ((int)count == -1) ? 0 : count;
}

int main (int argc, char *argv[])
{
  int device_index = 0;
  int bits_per_sample = BPS_16;
  int sampling_rate = 44100;
  int sample_channels = 2;
  long sample_format = MCI_WAVE_FORMAT_PCM;
  char *input_file = NULL;

  programname = argv[0];

  int c;
  while ((c = getopt_long (argc, argv, "d:bwr:c:f:hv", long_options, (int *) 0)) != EOF)
    switch (c)
    {
      case '?':
        // unknown option
        return -1;
      case 'b':
        bits_per_sample = BPS_8;
        break;
      case 'w':
        bits_per_sample = BPS_16;
        break;
      case 'f':
        if (!stricmp (optarg, "pcm"))
          sample_format = MCI_WAVE_FORMAT_PCM;
        else if (!stricmp (optarg, "adpcm"))
          sample_format = MCI_WAVE_FORMAT_ADPCM;
        else if (!stricmp (optarg, "ibm_cvsd"))
          sample_format = MCI_WAVE_FORMAT_IBM_CVSD;
        else if (!stricmp (optarg, "alaw"))
          sample_format = MCI_WAVE_FORMAT_ALAW;
        else if (!stricmp (optarg, "mulaw"))
          sample_format = MCI_WAVE_FORMAT_MULAW;
        else if (!stricmp (optarg, "oki_adpcm"))
          sample_format = MCI_WAVE_FORMAT_OKI_ADPCM;
        else if (!stricmp (optarg, "dvi_adpcm"))
          sample_format = MCI_WAVE_FORMAT_DVI_ADPCM;
        else if (!stricmp (optarg, "digistd"))
          sample_format = MCI_WAVE_FORMAT_DIGISTD;
        else if (!stricmp (optarg, "digifix"))
          sample_format = MCI_WAVE_FORMAT_DIGIFIX;
        else if (!stricmp (optarg, "avc_adpcm"))
          sample_format = MCI_WAVE_FORMAT_AVC_ADPCM;
        else if (!stricmp (optarg, "ibm_adpcm"))
          sample_format = MCI_WAVE_FORMAT_IBM_ADPCM;
        else if (!stricmp (optarg, "ibm_mulaw"))
          sample_format = MCI_WAVE_FORMAT_IBM_MULAW;
        else if (!stricmp (optarg, "ibm_alaw"))
          sample_format = MCI_WAVE_FORMAT_IBM_ALAW;
        else if (!stricmp (optarg, "ct_adpcm"))
          sample_format = MCI_WAVE_FORMAT_CT_ADPCM;
        else if (!stricmp (optarg, "mpeg1"))
          sample_format = MCI_WAVE_FORMAT_MPEG1;
        else
        {
          printf ("%s: unknown sample format -- %s\n", programname, optarg);
          return -1;
        }
        break;
      case 'r':
        sampling_rate = atoi (optarg);
        if ((sampling_rate > 100000) || (sampling_rate < 1000))
        {
          printf ("%s: invalid sample rate -- %d\n", programname, sampling_rate);
          return -1;
        }
        break;
      case 'c':
        sample_channels = atoi (optarg);
        if ((sample_channels > 8) || (sample_channels < 1))
        {
          printf ("%s: invalid number of channels -- %d\n", programname, sample_channels);
          return -1;
        }
        break;
      case 'd':
        device_index = atoi (optarg);
        if ((device_index > 10) || (device_index < 0))
        {
          printf ("%s: invalid device index -- %d\n", programname, device_index);
          return -1;
        }
        break;
      case 'h':
        display_help ();
        return 0;
      case 'v':
        printf ("%s version %s\n", programname, programversion);
        return 0;
      default:
        printf ("c = %d  at  optind = %d\n", c, optind);
        break;
    }

  // Interpret the non-option arguments as file names
  for (; optind < argc; ++optind)
  {
    if (input_file == 0)
      input_file = argv [optind];
    else
      printf ("%s: excess command line argument -- %s\n", programname, argv [optind]);
  }

#if 0
  printf ("reading file [%s]\n", input_file ? input_file : "stdin");
  printf ("format: %ld %dbpp at %dHz / %d channels\n", sample_format,
    bits_per_sample, sampling_rate, sample_channels);
#endif

  // Open input file
  if ((input_file != NULL)
   && (strcmp (input_file, "-") != 0))
    input_handle = open (input_file, O_RDONLY | O_BINARY);
  if (input_handle < 0)
  {
    printf ("%s: cannot open file '%s'\n", programname, input_file);
    return -1;
  }
  setmode (input_handle, O_BINARY);	// Set handle to binary mode

  DART dart (device_index, bits_per_sample, sampling_rate, sample_format,
    sample_channels, 10);
  CheckError (dart);

  dart.SetInputCallback (dartCallback);
  dart.Play ();
  CheckError (dart);

  int retcode = 0;
  while (!dart.EndOfStream)
  {
    if (_read_kbd (0, 0, 0) >= 0)
    {
      retcode = 3;
      break;
    }
    DosSleep (100);
  }

  return retcode;
}
