/*
 *        Copyright (C) 1996  Active Software, Inc.
 *                  All rights reserved.
 *
 * @(#) InsetsConverter.java 1.10 - last change made 07/18/96
 */

package sunsoft.jws.visual.rt.type;

import java.awt.Insets;
import java.util.*;

/**
 * Converts Insets to strings and back again.
 * An example of the inset string format: "top=14;left=5;bottom=10;right=4".
 *
 * @see Insets
 * @version 	1.10, 07/18/96
 */
public class InsetsConverter extends Converter {

  /**
   * Converts an instance of Insets to its string representation.
   *
   * @param obj an instance of Insets
   */
  public String convertToString(Object obj) {
    if (obj == null)
      return "";

    String s = "";
    Insets insets = (Insets)obj;

    if (insets.top != 0)
      s = s + "top=" + insets.top + ";";
    if (insets.left != 0)
      s = s + "left=" + insets.left + ";";
    if (insets.bottom != 0)
      s = s + "bottom=" + insets.bottom + ";";
    if (insets.right != 0)
      s = s + "right=" + insets.right + ";";

    return s;
  }

  private int getIntegerFromTable(Hashtable table, String key) {
    String value = (String) table.get(key);
    if (value != null) {
      try {
	return Integer.valueOf(value).intValue();
      }
      catch (NumberFormatException ex) {
	throw new ParseException("Badly formatted insets value: " + value);
      }
    }
    else {
      return(0);
    }
  }

  /**
   * Returns a new instance of Insets according to the string representation
   * given.
   *
   * @exception ParseException when there is a format error in the string
   */
  public Object convertFromString(String s) {
    if (s == null || s.length() == 0)
      return null;

    SubFieldTokenizer sft = new SubFieldTokenizer(s);
    Hashtable table = sft.getHashtable();
    Insets insets = new Insets(0, 0, 0, 0);

    Enumeration e = table.keys();
    while (e.hasMoreElements()) {
      String key = (String)e.nextElement();
      if (!key.equals("top") && !key.equals("bottom") && 
	  !key.equals("left") && !key.equals("right")) {
	throw new ParseException("Illegal inset value: " + key);
      }
    }

    if (table.containsKey("top"))
      insets.top = getIntegerFromTable(table, "top");
    if (table.containsKey("left"))
      insets.left = getIntegerFromTable(table, "left");
    if (table.containsKey("bottom"))
      insets.bottom = getIntegerFromTable(table, "bottom");
    if (table.containsKey("right"))
      insets.right = getIntegerFromTable(table, "right");

    return insets;
  }

  /**
   * Returns code to create an instance of Insets like the one given.
   *
   * @param obj an instance of Insets
   */
  public String convertToCode(Object obj) {
    if (obj == null) {
      return "null";
    }
    else {
      Insets i = (Insets)obj;
      return("new java.awt.Insets(" +
	     i.top + ", " + i.left + ", " + i.bottom + ", " + i.right + ")");
	     
    }
  }
}
