/*
 *        Copyright (C) 1996  Active Software, Inc.
 *                  All rights reserved.
 *
 * @(#) StatusBar.java 1.9 - last change made 08/01/96
 */

package sunsoft.jws.visual.rt.awt;

import sunsoft.jws.visual.rt.base.Global;
import sunsoft.jws.visual.rt.base.Util;
import java.awt.*;

/**
 * A label that shows a single line of text for while.  After a set
 * time period the text is blanked.  It's useful for a status bar at
 * the bottom of a frame.
 *
 * @version 1.9, 08/01/96
 */
public class StatusBar extends Canvas implements Runnable {
  private String text;
  private long wakeupTime;
  private long timeout;
  
  public StatusBar(String text) {
    wakeupTime = 0;
    timeout = 7000;
    setFont(new Font("Helvetica", Font.BOLD, 14));

    Thread thread = new Thread(this);
    thread.setDaemon(true);
    thread.start();
  }
  
  public StatusBar() {
    this("");
  }
  
  public synchronized void setTimeout(long millis) {
    timeout = millis;
    resetTimer(true);
  }

  public void setText(String text) {
    setText(text, true);
  }

  public void setText(String text, boolean shouldTimeout) {
    if (text != this.text && (this.text == null || !this.text.equals(text))) {
      this.text = text;
      repaint();
    }

    resetTimer(shouldTimeout);
  }
  
  public String getText() {
    return text;
  }
  
  public void paint(Graphics g) {
    Dimension d = size();
    
    g.setColor(getBackground());
    Global.util.draw3DRect(g, 0, 0, d.width-1, d.height-1,
			   Util.WIN95_SUNKEN, 1);

    if (text != null) {
      g.setColor(getForeground());
      FontMetrics fm = g.getFontMetrics();
      g.drawString(text, 5, fm.getAscent() + 3);
    }
  }
  
  public Dimension minimumSize() {
    Graphics g = getGraphics();
    int w = 10;
    int h = 6;

    if (g != null) {
      FontMetrics fm = g.getFontMetrics();
      if (text != null)
	w += fm.stringWidth(text);
      h += fm.getHeight();
    }

    return new Dimension(w, h);
  }
  
  public Dimension preferredSize() {
    return minimumSize();
  }
  
  public synchronized void run() {
    long currentTime = System.currentTimeMillis();
    
    while (true) {
      try {
	if (wakeupTime == 0)
	  wait();
	else
	  wait(wakeupTime - currentTime);
      }
      catch (java.lang.InterruptedException ex) {
      }
      
      currentTime = System.currentTimeMillis();
      if (wakeupTime != 0 && wakeupTime < currentTime) {
	text = null;
	repaint();
	wakeupTime = 0;
      }
    }
  }
  
  private synchronized void resetTimer(boolean shouldTimeout) {
    if (timeout > 0 && shouldTimeout && text != null && !text.equals("")) {
      wakeupTime = System.currentTimeMillis() + timeout;
      notify();
    }
    else {
      wakeupTime = 0;
    }
  }
}
