/*
 *        Copyright (C) 1996  Active Software, Inc.
 *                  All rights reserved.
 *
 * @(#) RootDialog.java 1.33 - last change made 07/29/96
 */

package sunsoft.jws.visual.rt.awt;

import sunsoft.jws.visual.rt.base.DesignerAccess;
import sunsoft.jws.visual.rt.base.*;

import java.awt.*;
import java.awt.peer.ComponentPeer;

public class RootDialog extends Dialog implements RootWindow {
  private RWHelper helper;
  private Thread eventThread;
  private boolean skipValidate = false;

  private boolean isShowing = false;
  private boolean peerShowing = false;
  private boolean hasShown = false;

  public RootDialog(Frame parent, boolean modal) {
    super(parent, modal);
    initHelper();
  }

  public RootDialog(Frame parent, String title, boolean modal) {
    super(parent, title, modal);
    initHelper();
  }

  private void initHelper() {
    Class c = DesignerAccess.getRootWindowHelperClass();
    if (c != null) {
      try {
	helper = (RWHelper)c.newInstance();
	helper.setWindow(this);
      }
      catch (Exception ex) {
      }
    }
  }

  /**
   * Don't do a validate during a show.
   */
  public void validate() {
    if (!skipValidate){
      super.validate();
    }
    else {
      // Modal dialogs block in show, so the flag needs to be unset here
      skipValidate = false;
    }
  }

  /**
   * Make modal dialogs in JDK 1.0 behave the same as
   * modal dialogs in JDK 1.0.2.
   * <p>
   * Note: This workaround fails if show is invoked by the
   * AWT callback thread.
   */
  public void show() {
    //
    // Call show only if it is necessary.  When the peer's show method
    // is called, the peer will call validate back on the AWT component.
    //
    // This is also a workaround for a problem on Windows where calling
    // show on a modal dialog that is already showing will cause the
    // application to be locked up after the dialog is hidden.
    //
    if (isShowing()) {
      if (!Global.isWindows())
	toFront();
      return;
    }

    //
    // Workaround for the bug where the following combination of
    // show and hide actions on two modal dialogs will cause the
    // application to hang:
    //
    // Show dialog1, Show dialog2, Hide dialog2, Hide dialog1
    // Show dialog1, Show dialog2, Hide dialog2, Hide dialog1
    //
    if (Global.isWindows() && isModal() && hasShown) {
      Point p = location();
      Dimension d = size();

      removeNotify();
      addNotify();

      reshape(p.x, p.y, d.width, d.height);

      // Validate gets called by Windows during the call to show, so we
      // don't call validate here for performance reasons.  Also, sometimes
      // things shift slightly when validate is called the second time,
      // and this looks bad.
      //
      // validate();
    }

    isShowing = true;
    hasShown = true;
    peerShowing = true;

    skipValidate = true;
    super.show();
    skipValidate = false;

    if (isModal() && (Global.isWindows() || Global.javaVersion() < 1.02)) {
      String name = Thread.currentThread().getName();
      name = name.substring(0, 3);
      if (!name.equals("AWT"))
	blockShow();
    }
  }

  private synchronized void blockShow() {
    boolean interrupted = true;
    while (interrupted) {
      try {
	wait();
	interrupted = false;
      }
      catch(InterruptedException ex) {
	interrupted = true;
      }
    }
  }

  /**
   * Make dialogs on Windows behave the same as
   * dialogs on Motif.
   */
  public void pack() {
    Component parent = getParent();
    ComponentPeer peer = getPeer();

    if (parent != null && parent.getPeer() == null) {
      parent.addNotify();
    }
    if (peer == null) {
      addNotify();
    }

    Dimension d = preferredSize();
    Point p = getDialogLocation(d);

    // Adjust the location the first time the dialog is mapped after
    // the peer has been created.  There is an AWT bug that screws
    // up the location of the dialog, so we have to make this adjustment.
    if (Global.isMotif() && (peer == null)) {
      p.x -= d.width/2;
      p.y -= d.height/2;
    }

    reshape(p.x, p.y, d.width, d.height);

    validate();
  }

  private Point getDialogLocation(Dimension prefSize) {
    Frame frame = (Frame)getParent();
    Point p = frame.location();
    Dimension fsize = frame.size();
    Dimension dsize = prefSize;

    p.x += (fsize.width - dsize.width)/2;
    p.y += (fsize.height - dsize.height)/2;

    return p;
  }

  /**
   * Make modal dialogs in JDK 1.0 behave the same as
   * modal dialogs in JDK 1.0.2.
   */
  public synchronized void hide() {
    isShowing = false;
    peerShowing = false;

    super.hide();

    if (isModal() && (Global.isWindows() || Global.javaVersion() < 1.02)) {
      try {
	notify();
      }
      catch (IllegalMonitorStateException ex) {
      }
    }
  }

  /**
   * Workaround for bug where calling reshape on a dialog
   * that already has a peer will cause the dialog to be mapped.  We
   * definitely don't want the dialog to be mapped in this situation.
   */
  public void reshape(int x, int y, int w, int h) {
    super.reshape(x, y, w, h);
    if (Global.isMotif() && getPeer() != null)
      peerShowing = true;
  }

  public void toFront() {
    super.toFront();
    isShowing = true;
    peerShowing = true;
  }

  public boolean handleEvent(Event evt) {
    if (evt.id == Event.WINDOW_MOVED && peerShowing && !isShowing) {
      ((ComponentPeer)getPeer()).hide();
      peerShowing = false;
    }

    return super.handleEvent(evt);
  }

  /**
   * Make sure the insets aren't screwed up on Windows.  The dialog
   * will come up too small the first time it is shown if we don't
   * fix the insets here.
   */
  public Insets insets() {
    Insets insets = super.insets();

    if (Global.isWindows()) {
      if (insets.top < 10)
	insets.top = 25;
      if (insets.bottom < 5)
	insets.bottom = 5;
      if (insets.left < 5)
	insets.left = 5;
      if (insets.right < 5)
	insets.right = 5;
    }

    return insets;
  }

  /**
   * Event forwarding to groups.
   *
   * (see comment in GBPanel.java)
   */
  public boolean postEvent(Event evt) {
    // AWT Bug: JDK-1.0 AWT gives a WINDOW_ICONIFY event to a Dialog
    // when it is shown (causing us to obediently iconify the windows in the
    // dialog's group, including the dialog iteself.)  AWT then gives a
    // WINDOW_DEICONFY event to the same Dialog, resulting in a show and an
    // infinite loop of hiding and showing.  Dialogs can't be iconfied
    // anyway, so just what is this event doing here!!?
    //
    // Work-around: throw away these two spurious events on Dialogs before
    // they can muck up the works.
    //
    if (evt.id == Event.WINDOW_ICONIFY || evt.id == Event.WINDOW_DEICONIFY)
      return true;

    boolean marked = VJPanel.markEvent(evt, this);
    boolean handled = super.postEvent(evt);

    if (marked)
      VJPanel.forwardEvent(evt, this);
    return handled;
  }

  public void select() {
    if (helper != null)
      helper.select();
  }

  public void unselect() {
    if (helper != null)
      helper.unselect();
  }

  public void layoutMode() {
    if (helper != null)
      helper.layoutMode();
  }

  public void previewMode() {
    if (helper != null)
      helper.previewMode();
  }

  public Dimension previewSize() {
    if (helper != null)
      return helper.previewSize();
    else
      return null;
  }

  public void addNotify() {
    if (helper != null)
      helper.addNotify();
    super.addNotify();
  }

  public void removeNotify() {
    hasShown = false;
    if (Global.isWindows())
      reshapeZero(this);

    if (helper != null)
      helper.removeNotify();
    super.removeNotify();
  }

  /**
   * Recursively reshapes the component and all its children to zero.
   * You need to do this on Windows when removeNotify is called.
   * Otherwise when you call addNotify, the children are offset
   * about 25 extra pixels from the top on the window, and the bottom
   * 25 pixels will get clipped off.  I don't know why this happens,
   * but I do know that reshaping everything back to zero avoids
   * the problem.  I noticed the problem on WindowsNT, but not as
   * often on Windows95.
   */
  private void reshapeZero(Component comp) {
    comp.reshape(0, 0, 0, 0);

    if (comp instanceof Container) {
      Container cntr = (Container)comp;
      int count = cntr.countComponents();

      for (int i=0; i<count; i++)
	reshapeZero(cntr.getComponent(i));
    }
  }

  public void layout() {
    if (helper != null)
      helper.layout();
    else
      RootFrame.updateInsets(this);

    super.layout();
  }

  public void paint(Graphics g) {
    if (helper != null)
      helper.paint(g);
    super.paint(g);
  }

  public boolean mouseDown(Event evt, int x, int y) {
    if (helper != null)
      return helper.mouseDown(evt, x, y);
    return false;
  }
}
