/**
 * PanelFrame.java
 *
 * Copyright 1996-1996 Active Software Inc.
 *
 * @version @(#)PanelFrame.java 1.18 96/05/03
 * @author  Tilman Sporkert
 */

package sunsoft.jws.visual.rt.awt;

import sunsoft.jws.visual.rt.base.DesignerAccess;
import java.awt.*;

/**
 * A Java AWT Panel that implements some Tk Frame features, and Windows 95/NT
 * borders.
 *
 * A PanelFrame can be used just like a panel, but it lets you draw a frame
 * at the edge. It supports the relief and borderWidth features of a Tk
 * frame.
 * 
 * The default is RELIEF_FLAT with a borderwith of 1.
 *
 * PanelFrame also supports an inset so that the contents of it can be inset
 * properly from the borders.
 *
 * @author  Tilman Sporkert
 */
public class PanelFrame  extends java.awt.Panel {
  
public static final int RELIEF_FLAT   = 0;
public static final int RELIEF_SUNKEN = 1;
public static final int RELIEF_RAISED = 2;
public static final int RELIEF_GROOVE = 3;
public static final int RELIEF_RIDGE  = 4;
public static final int WIN95_FIELD_BORDER = 5;
    
    int borderWidth = 1;
    int relief = RELIEF_FLAT;
    int inset = 0;
private String title = null;
    public int titleHeight = 0;
    public int titleAscent = 0;
    public int titleRightEdge = 0;
    
public PanelFrame() {
}
    
public PanelFrame(int relief, int borderWidth) {
    this(null, relief, borderWidth, borderWidth);
}
    
public PanelFrame(int relief, int borderWidth, int inset) {
    this(null, relief, borderWidth, inset);
}
    
public PanelFrame(String title, int relief, int borderWidth, int inset) {
    setTitle(title);
    setBorderWidth(borderWidth);
    setRelief(relief);
    setInset(inset);
}
    
public void setTitle(String title) {
    this.title = title;
    }

public String getTitle() {
    return title;
    }

public void setBorderWidth(int borderWidth) {
    // is borderWidth valid?
    if (borderWidth < 0)
	return;
    
    this.borderWidth = borderWidth;
    if (borderWidth > inset)
	inset = borderWidth;
}
    
public int getBorderWidth() {
    return borderWidth;
}

public void setInset(int inset) {
    if (inset >= 0)
	this.inset = inset;
}


public int getInset() {
    return inset;
}

    
public void setRelief(int relief) {
    // is relief valid?
    if ((relief >= RELIEF_FLAT) && (relief <= WIN95_FIELD_BORDER))
	this.relief = relief;
}
    
public int getRelief() {
    return relief;
}

private boolean checkMetrics(Graphics g) {
    if ((title != null) && (titleRightEdge == 0)) {
	if (g == null)
	    g = getGraphics();
	if (g == null)
	    return false;
	Font font = getFont();
	if (font == null)
	    return false;
	FontMetrics fm = getFontMetrics(font);
	titleRightEdge = fm.stringWidth(title) + 8;
	titleHeight = fm.getAscent() + fm.getDescent();
	titleAscent = fm.getAscent();
    }
    return true;
}
    
public void paint(Graphics g) {
    if (relief == RELIEF_FLAT)
	return;
    
    synchronized (DesignerAccess.mutex) {
	
	Color orig_color = g.getColor();
	int stopx = 0;
	int startx = 0;
	
	if ((title != null) && checkMetrics(g)) 
	    g.drawString(title, 7, titleAscent);


	Insets i = super.insets();
	Dimension d = size();
	
	int b;
	
	// the colors to use
	Color bg = getParent().getBackground().brighter();
	Color fg = getParent().getBackground().darker();
	
	// for groove and and ridge: where to switch colors
	int middle = borderWidth / 2;
	
	for (b = 0; b < borderWidth; b++) {
	    
	    Color top_left_color = fg;
	    Color bottom_right_color = fg;
	    
	    switch (relief) {
	    case RELIEF_RAISED:
		top_left_color = bg;
		bottom_right_color = fg;
		break;
	    case RELIEF_SUNKEN:
		top_left_color = fg;
		bottom_right_color = bg;
		break;
	    case RELIEF_GROOVE:
		if (b < middle) {
		    top_left_color = fg;
		    bottom_right_color = bg;
		} else {
		    top_left_color = bg;
		    bottom_right_color = fg;
		}
		break;
	    case RELIEF_RIDGE:
		// settle on colors for groove and ridge
		if (b < middle) {
		    top_left_color = bg;
		    bottom_right_color = fg;
		} else {
		    top_left_color = fg;
		    bottom_right_color = bg;
		}
		break;
	    case WIN95_FIELD_BORDER:
		if (b < middle) {
		    top_left_color = fg;
		    bottom_right_color = Color.white;
		} else {
		    top_left_color = Color.black;
		    bottom_right_color = getParent().getBackground();
		}
		break;
	    }
	    
	    // this can screw up if the two borders don't fit...
	    int left = i.left + b;
	    int right = d.width - i.right - b - 1;
	    int top = i.top + b;
	    if (titleAscent > 0)
		top += titleAscent / 2;
	    int bottom = d.height - i.bottom - b - 1;
	    
	    // now draw the thing...
	    g.setColor(top_left_color);
	    if (titleRightEdge > 0) {
		g.drawLine(left, top, i.left + 5, top); // top
		g.drawLine(i.left + titleRightEdge, top, right, top); // top
	    } else
		g.drawLine(left, top, right, top); // top
	    g.drawLine(left, top, left, bottom); // left
	    
	    g.setColor(bottom_right_color);
	    g.drawLine(left, bottom, right, bottom); // bottom
	    g.drawLine(right, top, right, bottom); // right
	}
	g.setColor(orig_color);
    }
}
    
public Insets insets() {
    Insets i = super.insets();
    if (relief != RELIEF_FLAT) {
	i.left += inset;
	i.right += inset;
	i.bottom += inset;
	i.top += inset;
	if (checkMetrics(null) && (titleAscent > borderWidth))
	    i.top += titleAscent - borderWidth;
    }
    return i;
}
}
