import java.io.IOException;
import socket.TelnetIO;

/** Wrapper for a Java Telnet call. 
 * Supports remote commands.
 * Uses code from Matthias L. Jugel's and Marcus Meiner's freeware
 * <a href="http://www.first.gmd.de/persons/leo/java/Telnet">Telnet Applet</a>.
 * Note - their Applet is under GNU copyleft, so any commercial use of
 * this code is restricted. See their above site for details.
 * @author George Ruban 3/4/97
 * @see socket.TelnetIO
 */
public class TelnetWrapper
{
  /** The telnet connection. That which is wrapped. */
  private protected TelnetIO tio;
  /** Set to true for System.out.println debugging. */
  private final static boolean debug = false;
  /** The current prompt on the remote system. */
  private String prompt;

  /** The default prompt used by all TelnetWrappers unless specifically
   * overridden.
   * @see #setPrompt
   */
  private static String defaultPrompt = "$ ";

  /** The default login name used by TelnetWrappers.
   * If defaultLogin and defaultPassword are both non-null
   * when a TelnetWrapper is created, the TelnetWrapper will attempt
   * to login.
   */
  private static String defaultLogin = null;

  /** The default password used by TelnetWrappers.
   * If defaultLogin and defaultPassword are both non-null
   * when a TelnetWrapper is created, the TelnetWrapper will attempt
   * to login.
   */
  private static String defaultPassword = null;
  

  /** Skip any received data until the token appears. */
  public void wait(String token) throws IOException
  {
    if(debug) System.out.println("wait(" + token + ")...");
    String tmp;
    do {
      tmp = receive();
    } while(tmp.indexOf(token) == -1);
    if(debug) System.out.println("wait(" + token + ") successful.");
  }

  /** Returns bytes available to be read.  Since they haven't been
   * negotiated over, this could be misleading...
   * -- removed because available is not in TelnetIO
  public int available() throws IOException
  {
    return tio.available();
  } */
	
  /** Returns a String from the telnet connection. Blocks
   * until one is available. No guarantees that the string is in
   * any way complete.
   * NOTE: uses Java 1.0.2 style String-bytes conversion.*/
  public String receive() throws IOException
  {
    String s = new String(receiveBytes(), 0);
    if(debug) System.out.println(s);
    return s;
  }

  /** Returns a byte array. Blocks until data is available. */
  public byte[] receiveBytes() throws IOException
  {
    return tio.receive();
  }

  /** Returns all data received up until a certain token. */
  public String receiveUntil(String token) throws IOException
  {
    String s = new String();
    do
    {
      s += receive();
    } while(s.indexOf(token) == -1);
    return s;
  }
  
  /** Sends a String to the remote host.
   * NOTE: uses Java 1.0.2 style String-bytes conversion.
   */
  public void send(String s) throws IOException
  {
    if(debug) System.out.println(s);
    byte[] buf = new byte[s.length()];
    s.getBytes(0, s.length(), buf, 0);
    tio.send(buf);
  }

  /** Sends a line to the remote host, returns all data before the prompt.
   * Since telnet seems to rely on carriage returns ('\r'), 
   * one will be appended to the sent string, if necessary.
   * @param command command line to send
   * @return whatever data the command produced before the prompt.
   * @see #setPrompt
   */
  public String sendLine(String command) throws IOException
  {
    if(command.charAt(command.length() -1) != '\r') 
      command += "\r";
    send(command);
    String s = receiveUntil(prompt);

    // telnet typically echoes the command with a \r\n ending...
    return s.substring(command.length() + 1, s.indexOf(prompt));
  }
  
  /** Sends bytes over the telnet connection. */
  public void send(byte[] buf) throws IOException
  {
    tio.send(buf);
  }
  
  /** Logs in as a particular user and password. 
    * Returns after receiving prompt. */
  public void login(String loginName, String password) throws IOException
  {
    wait("login:");
    send(loginName + "\r");
    wait("Password:");
    sendLine(password + "\r");
  }
    
  /** Connects to the default telnet port on the given host. */
  public TelnetWrapper(String host) throws IOException
  {
    tio = new TelnetIO();
    setPrompt(defaultPrompt);
    tio.connect(host);
    if(defaultLogin != null && defaultPassword != null)
    {
      login(defaultLogin, defaultPassword);
    }
  }
  
  /** Sets the expected prompt. 
   * If this function is not explicitly called, the default prompt is used.
   * @see #setDefaultPrompt
   */
  public void setPrompt(String prompt)
  {
    if(prompt == null) throw new IllegalArgumentException("null prompt.");
    this.prompt = prompt;
  }

  /** Sets the default prompt used by all TelnetWrappers.
   * This can be specifically overridden for a specific instance.
   * The default prompt starts out as "$ " until this function is called.
   * @see #setPrompt
   */
  public static void setDefaultPrompt(String prompt)
  {
    if(prompt == null) throw new IllegalArgumentException("null prompt.");
    defaultPrompt = prompt;
  }

  /** Sets the default login used by TelnetWrappers.
   * If this method is called with non-null login and password,
   * all TelnetWrappers will attempt to login when first created.
   * @param login login name to use
   * @param password password to use
   * @see #login
   * @see #unsetLogin
   */
  public static void setLogin(String login, String password)
  {
    if(login == null || password == null)
      throw new IllegalArgumentException("null login or password.");
    defaultLogin = login;
    defaultPassword = password;
  }


  /** Turns off the default login of TelnetWrappers.
   * After this method is called, TelnetWrappers will not
   * login until that method is explicitly called.
   * @see #setLogin
   * @see #login
   */
  public static void unsetLogin()
  {
    defaultLogin = defaultPassword = null;
  }
  
  /** Ends the telnet connection. */
  public void disconnect() throws IOException
  {
    if(tio != null) tio.disconnect();
    tio = null;
  }
  
  /** Ends the telnet connection. */
  public void finalize()
  {
    try
    {
      disconnect();
    }
    catch(IOException e)
    {} // after all, what can be done at this point?
  }  

  /** Telnet test driver.
   * Modeled after the IOtest.java example in the Telnet Applet.
   * Logs in to stems, creates a timestamped file in /tmp, lists the
   * /tmp directory to System.out, disconnects.  Shows off several
   * TelnetWrapper methods.
   */
  public static void main(String args[]) throws IOException
  {
    java.util.Date now = new java.util.Date();
    String timestamp = now.getYear() + "-" +
		(now.getMonth()+1) + "-" + now.getDate() + "-" +
		  now.getHours() + ":" + now.getMinutes() + ":" +
		    now.getSeconds();
    TelnetWrapper telnet = new TelnetWrapper("someHost"); 
    
    // setting the correct prompt ahead of time is very important 
    // if you want to use login and sendLine
    telnet.setPrompt("$ ");
    telnet.login("myLoginName", "myPassword");

    // this is how you have to do it otherwise
    telnet.send("touch /tmp/TELNET_WRAPPER-" + timestamp + "\r");
    telnet.wait("$ ");

    // sendLine 1: adds the \r automatically, 2: waits for the prompt
    // before returning 3: returns what was printed from the command
    String ls = telnet.sendLine("ls /tmp");
    System.out.println(ls);

    // clean up
    telnet.disconnect();

  }
}
