/* This PlugIn will be responsible for handling user
 * defined triggers and gags, configurable 
 * $Id: Trigger.java 1.2 2000/07/24 14:01:27 jeffnik Exp jeffnik $
 */

/* The trigger plug-in and JamochaMUD are
 * Copyright (C) 1998-2000 Jeff Robinson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version. *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

package plugins;

// These import statements are specific to this plug-in
import MuckConn;
import MuckMain;
import java.awt.*;
import java.awt.event.*;

import java.io.File;
import java.io.RandomAccessFile;

import java.util.Vector;

/* This PlugIn will be responsible for handling user
 * defined triggers and gags, configurable 
 * @version $Id: Trigger.java 1.2 2000/07/24 14:01:27 jeffnik Exp jeffnik $
 * @author Jeff Robinson
 */

public class Trigger implements PlugInterface, ActionListener {

	// Variables
	Dialog ruleFrame;
	boolean inProcess = false;	// this stops multiple instances of the dialogue box
	Choice ruleName = new Choice();
	Vector mainRules;

	// File locations and paths
	String pathSeparator = new String(java.io.File.separator);
	String userDirectory = (String)MuckConn.jmVars.get("UserDirectory");
	String plugIns = new String(userDirectory + "plugins");
	String triggerDir = new String(plugIns + pathSeparator + "TriggerDir");
	File triggerFile = new File(triggerDir + pathSeparator + ".trigger.rc");

	private static final String RULES = "$RULES:";
	private static final String RULENAME = "$NAME:";
	private static final String COLOUR = "$COLOUR:";

	public void setLoader(Object o) {
	}

	public String PlugInName() {
		return "Gags & Triggers";
	}

	public String PlugInDescription() {
		return "A plug-in for user-defined gags (hiding unwanted output) and triggers (highlighting input in a number of ways, including sound or colours).";
	}

	public String PlugInType() {
		return "Output";
	}

	/** The main method of Trigger.  Here we are passed a string from 
	 * main program, will parse it for triggers, and then return whatever
	 * has been dictated the the plugins, or lack thereof
	 */
	public String PlugMain(String jamochaString) {
		String finalString = new String(ParseGags(jamochaString));
		return finalString;
	}

	public void PlugInProperties() {
		// We'll call the configuration dialogue for this plugin.
		// Since it is rather extensive, we've put it into its own
		// package.  (That way we can store the user configs there, too).
		if (!inProcess) {
			inProcess = true;
			SelectRule();
		}
	}

	public void InitialiseAtLoad(){
		// We'll call a class in the TriggerDir to scoop any existing .trigger.rc info
		ReadTriggers();
	}

	public void SetAtHalt(){
	}

	/** Do an "installation check", and read in triggers	
	 * if they already exist.  If not, create the proper space
	 */
	private void ReadTriggers() {
		File tD = new File(triggerDir);

		// First, check to see if the Trigger directory exists
		if (!tD.exists()) {
			// There is no TriggerDir, so we'll create one
			tD.mkdir();
			// Obviously there will be no .trigger.rc file, so we can stop here.
			return;
		}
		// Check to see if the .trigger.rc file exists
		if (!triggerFile.exists()) return;	// No trigger file, no need to continue

		mainRules = new Vector(0, 1);
		// We'll open the .trigger.rc file and read in the rulesets
		RandomAccessFile reader;
		boolean loop = true;

		try {
			reader = new RandomAccessFile(triggerFile.toString(), "r");

			String line;
			StringBuffer fullLine = new StringBuffer("");
			while (loop) {
				line = new String(reader.readLine());

				if (line == null || line.trim().equals("")) {
					loop = false;
					break;
				}

				line.trim();
				fullLine.append(line);

				if (line != null && line.indexOf("$DONE") > 0) {
					mainRules.addElement(fullLine.toString());
					fullLine = new StringBuffer("");
				}

			}

		} catch(Exception e) {
			System.out.println("Trigger plugin could not access " + triggerFile + ", exception " + e);
			return;
		}

		SetChoices();
	}

	/** This displays a list of available rules, allowing the
	 * user to add, edit, or delete rules. */
	private void SelectRule() {
		ruleFrame = new Dialog(MuckMain.textWindow, "Select a rule", false);
		GridBagLayout layout = new GridBagLayout();
		GridBagConstraints constraints = new GridBagConstraints();
		ruleFrame.setLayout(layout);

		Label tempLabel = new Label("Rule name:");
		constraints.gridwidth = 1;
		constraints.gridheight = 1;
		constraints.gridx = 0;
		constraints.gridy = 0;
		constraints.insets = new Insets(7, 7, 2, 2);
		constraints.fill = GridBagConstraints.BOTH;
		layout.setConstraints(tempLabel, constraints);
		ruleFrame.add(tempLabel);

		// ruleName = new Choice();
		constraints.gridwidth = 3;
		constraints.gridheight = 1;
		constraints.gridx = 1;
		constraints.gridy = 0;
		constraints.insets = new Insets(7, 2, 2, 7);
		constraints.fill = GridBagConstraints.BOTH;
		layout.setConstraints(ruleName, constraints);
		ruleFrame.add(ruleName);

		Button tempButton = new Button("Add a rule");
		constraints.gridwidth = 1;
		constraints.gridheight = 1;
		constraints.gridx = 1;
		constraints.gridy = 1;
		constraints.insets = new Insets(2, 2, 20, 2);
		constraints.fill = GridBagConstraints.BOTH;
		tempButton.setActionCommand("AddNewRule");
		tempButton.addActionListener(this);
		layout.setConstraints(tempButton, constraints);
		ruleFrame.add(tempButton);

		tempButton = new Button("Edit this rule");
		constraints.gridwidth = 1;
		constraints.gridheight = 1;
		constraints.gridx = 2;
		constraints.gridy = 1;
		constraints.insets = new Insets(2, 2, 20, 2);
		constraints.fill = GridBagConstraints.BOTH;
		tempButton.setActionCommand("EditThisRule");
		tempButton.addActionListener(this);
		layout.setConstraints(tempButton, constraints);
		ruleFrame.add(tempButton);

		tempButton = new Button("Remove this rule");
		constraints.gridwidth = 1;
		constraints.gridheight = 1;
		constraints.gridx = 3;
		constraints.gridy = 1;
		constraints.insets = new Insets(2, 2, 20, 7);
		constraints.fill = GridBagConstraints.BOTH;
		tempButton.setActionCommand("RemoveThisRule");
		tempButton.addActionListener(this);
		layout.setConstraints(tempButton, constraints);
		ruleFrame.add(tempButton);

		tempButton = new Button("Accept changes");
		constraints.gridwidth = 1;
		constraints.gridheight = 1;
		constraints.gridx = 0;
		constraints.gridy = 3;
		constraints.insets = new Insets(2, 7, 7, 2);
		constraints.fill = GridBagConstraints.BOTH;
		tempButton.setActionCommand("AcceptChanges");
		tempButton.addActionListener(this);
		layout.setConstraints(tempButton, constraints);
		ruleFrame.add(tempButton);

		tempButton = new Button("Cancel changes");
		constraints.gridwidth = 1;
		constraints.gridheight = 1;
		constraints.gridx = 3;
		constraints.gridy = 3;
		constraints.insets = new Insets(2, 2, 7, 7);
		constraints.fill = GridBagConstraints.BOTH;
		tempButton.setActionCommand("CancelChanges");
		tempButton.addActionListener(this);
		layout.setConstraints(tempButton, constraints);
		ruleFrame.add(tempButton);

		// Set the size of the dialogue
		ruleFrame.pack();
		ruleFrame.setVisible(true);

	}

	public void actionPerformed(ActionEvent e) {
		String cmd = new String(e.getActionCommand());
		System.out.println(cmd);

		if (cmd.equals("CancelChanges")) {
			// User has opted out of changes
			ruleFrame.setVisible(false);
			ruleFrame.dispose();
			inProcess = false;
		}

	}

	/** read the new vector into the choices... */
	private void SetChoices() {
		ruleName = new Choice();
		String tempName;
		for (int i = 0; i < mainRules.size(); i++) {
			// tempName = new String(RuleName((String)mainRules.elementAt(i)));
			tempName = new String(PullRule(mainRules.elementAt(i), RULENAME));

			if (tempName != null) {
				ruleName.add(tempName);
			}
		}
	}

	/** Go through the rules and see if have anything that matches */
	private String ParseGags(String input) {
		StringBuffer grinder = new StringBuffer(input);

		// Loop through all the rules...
		int numRules = 0;
		boolean gag = false;
		for (int set = 0; set < mainRules.size(); set++) {

			// How many rules are in this rule set?
			numRules = Integer.parseInt(PullRule(mainRules.elementAt(set), RULES));

			for (int j = 0; j < numRules; j++) {
				if (CheckRule(input, set, j)) {
					gag = true;
				} else {
					gag = false;
					break;
				}
			}

			// We'll see if this gag is appropriate...
			if (gag) {
				grinder = new StringBuffer(ApplyGag(grinder, set));
			}

		}

		return grinder.toString();
	}

	private boolean CheckRule(String input, int set, int rule) {
		String chkRule = new String(PullRule(mainRules.elementAt(set), "$" + rule + ":"));
		String match = (chkRule.substring(chkRule.indexOf(":") + 1)).toLowerCase();
		input = input.toLowerCase();
		// match.toLowerCase();
		boolean state = false;
		if (chkRule.startsWith("AND") && (input.indexOf(match) > 0 || input.startsWith(match))) state = true;
		if (chkRule.startsWith("OR") && (input.indexOf(match) > 0 || input.startsWith(match))) state = true;
		if (chkRule.startsWith("NOT") && input.indexOf(match) < 1) state = true;
	
		return state;
	}

	/** Apply the Gag to the String we were passed from the main program */
	private String ApplyGag(StringBuffer sb, int set) {
		String colour = PullRule(mainRules.elementAt(set), COLOUR);
		colour = new String(colour.toLowerCase());
		String colourCode = new String();
		if (colour.equals("gag")) return "";	// Not a colour, but a gag!
		if (colour.equals("black")) colourCode = '\u001b' + "[30m";
		if (colour.equals("red")) colourCode = '\u001b' + "[31m";
		if (colour.equals("green")) colourCode = '\u001b' + "[32m";
		if (colour.equals("yellow")) colourCode = '\u001b' + "[33m";
		if (colour.equals("blue")) colourCode = '\u001b' + "[34m";
		if (colour.equals("magenta")) colourCode = '\u001b' + "[35m";
		if (colour.equals("cyan")) colourCode = '\u001b' + "[36m";
		if (colour.equals("white")) colourCode = '\u001b' + "[37m";

		if (sb.charAt(0) != '\u001b') {
			sb.insert(0, colourCode);
			sb.append('\u001b' + "[0m");
		} 
		return sb.toString();
	}

	/** Search through the string for the rule name,
	 * and extract its information
	 */
	private String PullRule(Object ruleObj, String ruleName) {
		String rule = ruleObj.toString();
		int start, end;
		start = rule.indexOf(ruleName);
		start = start + ruleName.length();
		end = rule.indexOf("$", start + 1);

		return rule.substring(start, end);
	}
}
