/**
 * JMTFKeys emulates key controls of TinyFugue in JamochaMUD
 * $Id: JMTFKeys.java 1.3 2000/07/24 14:01:00 jeffnik Exp $
 */

/* JamochaMUD, a Muck/Mud client program
 * Copyright (C) 1998-2000 Jeff Robinson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version. *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

import java.awt.event.KeyEvent;

public class JMTFKeys {

	/** This method searches through the submitted string, checking for any
	 * TinyFugue keystrokes.  If one is found, we can return 'true', so that the
	 * rest of the event is not processed by DataIn.java
         * @version $Id: JMTFKeys.java 1.3 2000/07/24 14:01:00 jeffnik Exp $
         * @author Jeff Robinson
	 */
	public static boolean JMTFKeyStroke(int keyStroke) {
		boolean rC = true;

		switch (keyStroke) {
			case KeyEvent.VK_B:
				// Go left, to the beginning of the word
				FindWordStart();
				break;
			case KeyEvent.VK_F:
				// Go right, to end of the word
				FindWordEnd();
				break;
			case KeyEvent.VK_K:
				// Delete from cursor to end of line
				DEOL();
				break;
			case KeyEvent.VK_N:
				// Was 'Recall next input line', but now shows the pop-up
				DataIn.ShowPopup(null, -1, -1);
				break;
			case KeyEvent.VK_P:
				// Recall previous line
				PreviousLine();
				break;
			case KeyEvent.VK_Q:
				// resets the 'suspend' flag on the display (if true)
				DataIn.SpoolText();
				break;
			case KeyEvent.VK_S:
				// 'Suspends' the display.  (Halts scrollback)
				MuckMain.PauseText();
				break;
			case KeyEvent.VK_T:
				// Transposes characters
				TransposeChars();
				break;
			case KeyEvent.VK_U:
				// erase the entire line at the cursor's position
				EraseLine();
				break;
			case KeyEvent.VK_W:
				// erase the previous word
				ErasePreviousWord();
				break;
			default:
				rC = false;
		}
		return rC;
	}

	/** Delete from the cursor to the end of the line */
	private static void DEOL() {
		String wText = DataIn.dataText.getText();

		DataIn.dataText.setText(wText.substring(0, DataIn.dataText.getCaretPosition()));
	}

	/** Erase the entire line at the cursor's position */
	private static void EraseLine() {
		// First, we'll locate the position of the cursor, and the
		// beginning and end points of its line
		int cursor = DataIn.dataText.getCaretPosition();
		int row = rowNumber(cursor);
		int width = DataIn.dataText.getColumns();

		// Determine the row end, to see if it the whole width of the area or not
		int rowEnd = 0;
		if ((row * width) > cursor) {
			rowEnd = cursor;
		} else {
			rowEnd = row * width;
		}
	
		// Now, we can't really *erase* the line, so we just grab the test
		// on either side of the line and make a new String
		StringBuffer rString = new StringBuffer("");
		String workString = DataIn.dataText.getText();

		if ((row - 1) * width > 0) {
			// Append any text before the 'selection'
			rString.append(workString.substring(0, (row - 1) * width));
		}

		if (rowEnd < workString.length()) {
			// Append any text left after the 'selection'
			rString.append(workString.substring(rowEnd, workString.length()));
		}

		// Now we can set the new text in place of the old
		DataIn.dataText.setText((rString.toString()).trim());
		
	}

	/** Erase the word proceeding the cursor */
	private static void ErasePreviousWord() {
		// Erase the word preceeding the cursor
		String wText = new String(DataIn.dataText.getText());
		int cursor = DataIn.dataText.getCaretPosition();

		// Do a check to see if the cursor is sitting at the 'zero' position
		if (cursor < 1) return;

		// Now search backwards from the cursor's position till we find the
		// first space back, and delete the area in between
		int start = cursor - 1;

		while (start >= 0 && wText.charAt(start) != ' ') start--;

		// Now that we have the beginning and end, we'll assemble a new String
		StringBuffer nText = new StringBuffer("");
		
		if (start > 0) nText.append(wText.substring(0, start));

		if (cursor < wText.length()) nText.append(wText.substring(cursor, wText.length()));

		// Now set the text to this new String
		DataIn.dataText.setText(nText.toString().trim());

	}

	/** Find the beginning of the first word to the left of the cursor */
	private static void FindWordStart() {
		int cursor = DataIn.dataText.getCaretPosition() - 1;
		String wText = DataIn.dataText.getText();

		// Check to see if we're at the beginning of the line
		if (cursor < 1) return;
		
		while (cursor > 0 && wText.charAt(cursor) != ' ') cursor--;

		
		DataIn.dataText.setCaretPosition(cursor);

	}

	/** Find the end of the word to the right of the cursor */
	private static void FindWordEnd() {
		int cursor = DataIn.dataText.getCaretPosition();
		String wText = DataIn.dataText.getText();
		
		while (cursor < wText.length() && wText.charAt(cursor) != ' ') cursor++;

		if (cursor < wText.length()) cursor++;

		DataIn.dataText.setCaretPosition(cursor);
	}

	/** Recall the previously input line (if any)	*/
	private static void PreviousLine() {
		DataIn.JMSetFromScrollBack(-1);
	}

	/** Find the current row that the cursor is on */
	private static int rowNumber(int cursor) {
		// First, we'll locate the position of the cursor, and the
		// beginning and end points of its line
		int width = DataIn.dataText.getColumns();
		int row = 1;

		while ((width * row) < cursor) row++;

		return (row -1);
	}

	/** Transpose the character the caret is on with the one prior to it */
	private static void TransposeChars() {
		int cursor = DataIn.dataText.getCaretPosition();
		String wText = DataIn.dataText.getText();

		// Check to see if the cursor is in a 'legal' position
		if (cursor < 1 || cursor == wText.length()) return;

		StringBuffer fText = new StringBuffer("");
		fText.append(wText.substring(0, cursor - 1));
		fText.append(wText.charAt(cursor));
		fText.append(wText.charAt(cursor - 1));
		fText.append(wText.substring(cursor +1, wText.length()));
		
		DataIn.dataText.setText(fText.toString().trim());
		DataIn.dataText.setCaretPosition(cursor);
		
	}

	// This section is for TinyFugue emulated @/ commands
}
