
/**
 * This class manages events hookups between event source beans and 
 * target methods on target beans.
 * It does this for each hookup by generating a .java file for an adaptor
 * class and then compiling and loading the adaptor.
 */

package sun.beanbox;

import java.util.*;
import java.beans.*;
import java.lang.reflect.*;

public class HookupManager {

    static void hookup(EventSetDescriptor esd,
		Method listenerMethod,
		Wrapper sourceWrapper,
		Wrapper targetWrapper,
		Method targetMethod) {

	try {
	    // If we were smarter we might cache and reuse hookups.

	    Object source = sourceWrapper.getBean();
	    Object target = targetWrapper.getBean();
	    String hookupName = generateHookup(esd, listenerMethod, source, target, targetMethod);

	    if (hookupName == null) {
		// The compile failed.
	        System.err.println("Could not create event adaptor.");
		return;
	    }

	    SimpleClassLoader loader = SimpleClassLoader.ourLoader;

	    javaFiles.addElement(hookupName);
 	    Class hookupClass = loader.loadFromFile(hookupName);
	    Object hookup = hookupClass.newInstance();

	    Method setTargetMethod = findMethod(hookupClass, "setTarget");   
	    Object args[] = new Object[1];
	    args[0] = target;
	    setTargetMethod.invoke(hookup, args);

	    sourceWrapper.addEventTarget(esd.getName(), targetWrapper, hookup);

	} catch (Exception ex) {
	    System.err.println("Hookup caught " + ex);
	    ex.printStackTrace();
	}

    }

    static Vector getHookupFiles() {
	return javaFiles;
    }

    static String generateHookup(EventSetDescriptor esd, Method listenerMethod, Object source,
		Object target, Method targetMethod) {

	String id = getId();
	String className = "___Hookup_" + id;
	String fileName = tmpDir + java.io.File.separator + className + ".java";
	String targetType = target.getClass().getName();

	// Create an appropriate subdirectory.
	java.io.File tmp = new java.io.File(tmpDir);
	tmp.mkdirs();

	// Open the new java source file,
	java.io.PrintWriter out = null;
	try {
	    java.io.FileWriter fout = new java.io.FileWriter(fileName);
	    out = new java.io.PrintWriter(fout);
	} catch (Exception ex) {
	    System.err.println("Couldn't open hookup file " + fileName);
	    System.err.println("   " + ex);
	    return null;
	}

	out.println("// Automatically generated event hookup file.");
	out.println("");
	out.println("package " + packageName + ";");

	// These imports are in case the target or listener types are
	// in the default package.
	out.println("import " + targetType + ";");
	out.println("import " + esd.getListenerType().getName() + ";");

	out.println("");
	out.println("public class " + className + " implements " + 
			esd.getListenerType().getName() + ", java.io.Serializable {");
	out.println("");
	out.println("    public void setTarget(" + targetType + " t) {");
	out.println("        target = t;");
	out.println("    }");

	Method methods[] = esd.getListenerMethods();
	for (int k = 0; k < methods.length; k++) {
	    out.println("");
	    out.print("    public void " + methods[k].getName() + "(");
	    Class argTypes[] = methods[k].getParameterTypes();
	    for (int i = 0; i < argTypes.length; i++) {
	        if (i > 0) {
		    out.print(", ");
	        }
	        out.print("" + argTypes[i].getName() + " arg" + i);
	    }
	    out.print(")");
	    
	    /**
             * At the moment getExceptionTypes fails to return the
	     * PropertyVetoException.  I've hacked around the problem.
	     */
	    /*
	     * Class[] exceptionTypes = methods[k].getExceptionTypes();
	     * if (exceptionTypes.length > 0) {
	     * out.print("\n         throws");
	     * for (int i = 0; i < exceptionTypes.length; i++)
	     *     out.print(((i != 0) ? ", " : " ") + exceptionTypes[i].getName());
	     * out.print(" ");
	     * }
	    */
	    if ("vetoableChange".equals(methods[k].getName())) {
	        out.print("     throws java.beans.PropertyVetoException");
	    }

	    out.println(" {");
	    if (listenerMethod.getName() == methods[k].getName()) {
	        out.print("        target." + targetMethod.getName() + "(");
		// Either the targetMethod must take zero args, or the
		// same args as the listenerMethod.
		if (targetMethod.getParameterTypes().length != 0) {
	            for (int i = 0; i < argTypes.length; i++) {
	                if (i > 0) {
		            out.print(", ");
	                }
	                out.print("arg" + i);
		    }
	        }
	        out.println(");");
    	    }
	    out.println("    }");
	}
	out.println("");
	out.println("    private " + targetType + " target;");
	out.println("}");
	out.close();

	boolean ok = compile(fileName);	
	if (!ok) {
	    return null;
	}

	String fullClassName = packageName+"."+className;
	String pathToFile = tmpDir+java.io.File.separatorChar+className+".class";
	return pathToFile;
    }

    /**
     * A classpath corresponding to all the JAR files that have been
     * loaded so far.
     */

    static String loadedJarFiles() {
	String path = System.getProperty("java.class.path");
	String sep = System.getProperty("path.separator");
	Vector allJI = BeanBoxFrame.getToolBox().getLoadedJarInfo();

	StringBuffer allJars = new StringBuffer(path);
	Enumeration e = allJI.elements();
	JarInfo prev = null;	// REMIND!! clean this up
	while (e.hasMoreElements()) {
	    JarInfo ji = (JarInfo) e.nextElement();
	    if (ji == null) {
		// the built-in BeanBox container
		continue;
	    }
	    if (ji == prev) {
		// we already dealt with this one
		continue;
	    }
	    prev = ji;
	    allJars.append(sep);
	    allJars.append(ji.getJarName());
	}
	return allJars.toString();
    }


    static boolean compile(String fileName) {
	// Run the javac compiler inside the current address space.
	String args[] = new String[4];
	args[0] = "-classpath";
	args[1] = loadedJarFiles();
	args[2] = "-nowarn";
	args[3] = fileName;

	sun.tools.javac.Main compiler;
	
	try {
	    compiler = new sun.tools.javac.Main(System.err, "javac");
	} catch (Throwable th) {
	    System.err.println("");
	    System.err.println("Could not load JDK compiler class sun.tools.javac.Main :");
	    System.err.println("    " + th);
	    System.err.println("");
	    System.err.println("Check that the version of \"java\" that you are running");
	    System.err.println("is the one supplied with the JDK1.1 (which includes the");
	    System.err.println("compiler classes) and not some other version of \"java\"");
	    System.err.println("which has been shipped with some other product.");
	    System.err.println("");
	    return false;
	}

	return compiler.compile(args);
    }

    static String getId() {
	java.util.Date now = new java.util.Date();
	long id = now.getTime()/10;
	return Long.toHexString(id);
    }

    private static Method findMethod(Class cls, String methodName) {
	try {
            Method methods[] = cls.getMethods();
	    for (int i = 0; i < methods.length; i++) {
	        Method method = methods[i];
	        if (method.getName().equals(methodName)) {
		    return method;
		}
	    }
	    throw new Error("method " + methodName + " not found");
        } catch (Exception ex) {
	    throw new Error("findMethod caught : " + ex);
        }
    }

    static String shortPackageName = "sun.beanbox";
    static String shortTmpDir = BeanBoxFrame.getTmpDir();
    static String packageName;
    static String tmpDir;
    static Vector javaFiles = new Vector();

    static {
	packageName =
	    shortTmpDir.replace(java.io.File.separatorChar, '.')+"."+shortPackageName;
	tmpDir = packageName.replace('.', java.io.File.separatorChar);
    }
}

