/*  ----------------------------------------------------------------<Prolog>-
    Name:       xilrwp.c
    Title:      Xitami WSX agent for long-running web process (LRWP) protocol
    Package:    Xitami

    Written:    97/10/29    Robin P. Dunn  Total Control Software
    Revised:    98/01/07

    Copyright:  Copyright (c) 1998 Total Control Software
    License:    This is free software; you can redistribute it and/or modify
                it under the terms of the SMT License Agreement as provided
                in the file LICENSE.TXT.  This software is distributed in
                the hope that it will be useful, but without any warranty.
 ------------------------------------------------------------------</Prolog>-*/

#include "smtdefn.h"                    /*  SMT Agent Definitions            */
#include "smthttpl.h"                   /*  SMT HTTP library                 */

#include "xilrwp.h"                     /* LWRP definitions, etc.            */


/*- Definitions -------------------------------------------------------------*/

#define AGENT_NAME          LRWP_NAME

#define LRWP_REQUEST        "LRWP_REQUEST"
#define LRWP_PEERCLOSED     "LRWP_PEERCLOSED"
#define LRWP_JOBCOMP        "LRWP_JOBCOMP"

#define WSX_REQUEST         "WSX_REQUEST"


#define _MASTER             "_MASTER"
#define _PEER               "_PEER"
#define _ROUTER             "_ROUTER"



/*---------------------------------------------------------------------------*/

typedef struct_smt_wsx_request WSXREQ;

typedef struct {
    char*   name;
    char*   vhost;
    NODE    peerList;                   /* A list of PTCB's */
    NODE    waitingRequests;            /* A list of REQ's */
} RTRData;


typedef struct {                        /* Peer Thread control block         */
    void    *next, *prev;
    THREAD* thread;
    int     busy;
} PTCB;

typedef struct {                        /* Waiting requests                  */
    void    *next, *prev;
    WSXREQ* wsxreq;
    QID     replyQ;
} REQ;



typedef struct {                        /*  Thread context block:            */
    /* all types use this */
    int         thread_type;

    /* master and peers */
    int         handle;

    /* peers only */
    PTCB*       ptcb;
    RTRData*    rtr;
    QID         replyQ;
    char        sizebuff[10];
    size_t      numtoread;
    size_t      numread;
    char*       buffer;

    /* router only */
    Bool        termWhenEmpty;
} LRWP_TCB;


/*- Function prototypes -----------------------------------------------------*/

static char* makeTableKey(char* name, char* vhost);


/*- Global variables used in this source file only --------------------------*/

static LRWP_TCB  *tcb;                   /*  Address thread context block     */
#define TCB     LRWP_TCB

static QID  operq,                       /*  Operator console event queue     */
            /*logq, */                   /*  Logging agent event queue        */
            sockq,                       /*  Socket agent event queue         */
            tranq;                       /*  Transfer Agent                   */

static char scratch[LINE_MAX + 32];      /*  Scratch buffer                   */


static SYMTAB*  peerTable = NULL;
static THREAD*  rtrThread;
static THREAD*  http_thread;


#include "xilrwp.d"                     /*  Include dialog data              */

/********************   INITIALISE AGENT - ENTRY POINT   *********************/

int xilrwp_init (void)
{
    THREAD  *thread;
    AGENT   *agent;                     /*  Handle for our agent             */
#   include "xilrwp.i"                  /*  Include dialog interpreter       */

    /*  Change any of the agent properties that you need to                  */
    agent->router      = FALSE;        /*  FALSE = default                  */
    agent->max_threads = 0;            /*  0 = default                      */

    /*                      Method name     Event value      Priority        */
    /*  Shutdown event comes from Kernel                                     */
    method_declare(agent, "SHUTDOWN",      shutdown_event,  SMT_PRIORITY_MAX);

    /*  Reply events from socket agent                                       */
    method_declare(agent, "SOCK_INPUT_OK",  sock_input_ok_event,       0);
    method_declare(agent, "SOCK_CLOSED",    sock_closed_event,         0);
    method_declare(agent, "SOCK_ERROR",     sock_error_event,          0);
    method_declare(agent, "SOCK_TIMEOUT",   sock_error_event,          0);
    method_declare(agent, "SOCK_WRITE_OK",  sock_write_ok_event,       0);
    method_declare(agent, "TRAN_PUT_OK",    sock_write_ok_event,       0);
    method_declare(agent, "TRAN_CLOSED",    sock_closed_event,         0);
    method_declare(agent, "TRAN_ERROR",     sock_error_event,          0);

    /*  Private methods used to pump the initial thread events               */
    method_declare(agent, _MASTER,        master_event,   0);
    method_declare(agent, _PEER,          peer_event,     0);
    method_declare(agent, _ROUTER,        router_event,   0);

    /* other methods this agent supports       */
    method_declare(agent, WSX_REQUEST,      wsx_request_event,          0);
    method_declare(agent, LRWP_REQUEST,     lrwp_request_event,         0);
    method_declare(agent, LRWP_PEERCLOSED,  peer_closed_event,          0);
    method_declare(agent, LRWP_JOBCOMP,     peer_job_completed_event,   0);


    /*  Ensure that operator console is running, else start it up            */
    smtoper_init();
    if ((thread = thread_lookup(SMT_OPERATOR, "")) != NULL)
        operq = thread->queue->qid;
    else
        return(-1);

    /*  Ensure that socket agent is running, else start it up                */
    smtsock_init();
    if ((thread = thread_lookup(SMT_SOCKET, "")) != NULL)
        sockq = thread->queue->qid;
    else
        return(-1);

    /*  Ensure that transfer agent is running, else start it up              */
    smttran_init();
    if ((thread = thread_lookup(SMT_TRANSFER, "")) != NULL)
        tranq = thread->queue->qid;
    else
        return(-1);

    /*  Create initial thread to manage listening socket port              */
    if ((thread = thread_create(AGENT_NAME, "")) != NULL) {
        SEND(&thread->queue->qid, _MASTER, "");  /* pump the first event   */
        ((TCB *)thread->tcb)->thread_type = master_event;
        ((TCB *)thread->tcb)->handle      = 0;
      }
    else
        return (-1);

    /*  Create thread to act as the request router, and to respond to the WSX
        messages */
    if ((rtrThread = thread_create(AGENT_NAME, "main")) != NULL) {
        SEND(&rtrThread->queue->qid, _ROUTER, "");  /* pump the first event   */
        ((TCB *)rtrThread->tcb)->thread_type = router_event;
        ((TCB *)rtrThread->tcb)->handle      = 0;
      }
    else
        return (-1);

    /*  Signal to caller that we initialised okay */
    return(0);
}

/*************************   INITIALISE THE THREAD   *************************/

MODULE initialise_the_thread (THREAD *thread)
{
    /* Don't do anything...  The thread's first event is pumped in... */
}


/*************************   TERMINATE THE THREAD   **************************/

MODULE terminate_the_thread (THREAD *thread)
{
    tcb = thread->tcb;                 /*  Point to thread's context        */
    switch (tcb->thread_type) {
        case master_event:
            if (tcb->handle)
                close_socket(tcb->handle);
            break;

        case peer_event:
            if (tcb->handle)
                close_socket(tcb->handle);
            if (tcb->ptcb)
                node_destroy(tcb->ptcb);
            if (tcb->rtr)
                SEND(&rtrThread->queue->qid, LRWP_PEERCLOSED,
                        makeTableKey(tcb->rtr->name, tcb->rtr->vhost));
            break;

        case router_event:
            if (peerTable) {
                tcb->termWhenEmpty = TRUE;
                return;       /* bail out here so thread won't terminate yet */
            }
            break;
    }

    the_next_event = terminate_event;
}



/***************************   OPEN MASTER SOCKET   **************************/

MODULE open_master_socket (THREAD *thread)
{
    int     port;
    char*   cPort;

    tcb = thread->tcb;                 /*  Point to thread's context        */

    cPort = http_config(config, "lrwp:port");
    if (!cPort || !*cPort)
        cPort = LRWP_PORT;
    port = atoi(cPort) + ip_portbase;
    tcb->handle = passive_TCP(cPort, 5);
    if (tcb->handle == INVALID_SOCKET) {
        sendfmt(&operq, "ERROR",
                LRWP_NAME ": Could not open LRWP port %d", port);
        sendfmt(&operq, "ERROR", connect_errlist[connect_error()]);
        raise_exception(fatal_event);
      }
    else {
        sendfmt(&operq, "INFO",
                 LRWP_NAME ": ready for LRWP connections on port %d", port);
    }

    /* We'll need the main thread in HTTP agent later... */
    http_thread = thread_lookup (SMT_HTTP, "main");
}


/*************************   WAIT FOR SOCKET INPUT   *************************/

MODULE wait_for_socket_input (THREAD *thread)
{
    tcb = thread->tcb;                 /*  Point to thread's context        */

    send_input(&sockq, 0, tcb->handle, 0);
}

/******************************   SETUP ROUTER   *****************************/

MODULE setup_router (THREAD *thread)
{
    tcb = thread->tcb;                 /*  Point to thread's context        */
    tcb->termWhenEmpty = FALSE;
}



/************************   ACCEPT PEER CONNECTION   ***********************/

MODULE accept_peer_connection (THREAD *thread)
{
    sock_t  slave_socket;               /*  Connected socket                 */
    THREAD  *peer_thread;              /*  Handle to child threads          */

    tcb = thread->tcb;                 /*  Point to thread's context        */

    slave_socket = accept_socket(tcb->handle);
    if (slave_socket != INVALID_SOCKET) {
        peer_thread = thread_create(AGENT_NAME, "");
        if (peer_thread) {
            SEND(&peer_thread->queue->qid, _PEER, "");
            ((TCB *)peer_thread->tcb)->thread_type = peer_event;
            ((TCB *)peer_thread->tcb)->handle      = slave_socket;
            ((TCB *)peer_thread->tcb)->ptcb        = NULL;
            ((TCB *)peer_thread->tcb)->rtr         = NULL;
        }
    }
    else if (sockerrno != EAGAIN && sockerrno != EWOULDBLOCK) {
        sendfmt(&operq, "ERROR",
                 LRWP_NAME ": could not accept connection: %s", sockmsg());
        raise_exception(exception_event);
    }
}


/**************************   SIGNAL SOCKET ERROR   **************************/

MODULE signal_socket_error (THREAD *thread)
{
    tcb = thread->tcb;                 /*  Point to thread's context        */
    sendfmt(&operq, "ERROR",
            LRWP_NAME ": error on socket: %s", sockmsg());
}


/************************   SHUTDOWN THE APPLICATION   ***********************/

MODULE shutdown_the_application (THREAD *thread)
{
    tcb = thread->tcb;                 /*  Point to thread's context        */
    smt_shutdown();                    /*  Halt the application             */
}


/*********************   READ APP NAME AND START ROUTER   *********************/

static char* makeTableKey(char* name, char* vhost)
{
    sprintf(scratch, "%s:%s", (*vhost ? vhost : VHOST_ANY), name);
    return scratch;
}

MODULE read_app_name_and_start_router (THREAD *thread)
{
    int         num;
    char        buff[LRWP_MAXSTARTUP];
    char*       webmask;
    char*       appName;
    char*       vhost;
    char*       filterExt;
    char*       key;
    PTCB*       ptcb;
    RTRData*    rtr;
    SYMBOL*     rtrSym;
    char        delim = '\xFF';
    char*       errMsg1 = "ERROR: Malformed startup string";
    char*       errMsg2 = "ERROR: Unauthorized access";

    tcb = thread->tcb;                 /*  Point to thread's context        */


    /* get the appName, vhost, and filter extensions */
    num = read_TCP(tcb->handle, buff, LRWP_MAXSTARTUP);
    *(buff+num) = 0;               /* make sure it is NULL terminated */

    /* parse out the components of the startup string */
    appName = buff;
    if ((vhost = strchr(appName, delim)) == NULL ||
        (*(vhost++) = 0) ||        /* terminate the appName string */
        (filterExt = strchr(vhost, delim)) == NULL ||
        (*(filterExt++) = 0))      /* terminate the vhost string */ {

            /* If parsing failed, send back an error message */
            write_TCP(tcb->handle, errMsg1, strlen(errMsg1));
            strcpy(scratch, errMsg1);
            the_next_event = startup_error_event;
            return;
    }

    /* make sure connection is allowed */
    webmask = http_config(config, "lrwp:webmask");
    if (streq(webmask, "local"))
        webmask = socket_localaddr(tcb->handle);
    if (!socket_is_permitted(socket_peeraddr(tcb->handle), webmask)) {
        write_TCP(tcb->handle, errMsg2, strlen(errMsg2));
        strcpy(scratch, errMsg2);
        the_next_event = startup_error_event;
        return;
    }

    /* send OK acknowledgement */
    write_TCP(tcb->handle, "OK", 2);

    /* create the table if this is the first peer. */
    if (!peerTable)
        peerTable = sym_create_table();

    if (*appName == '/')
        appName++;

    /* find or create the router symbol in the peer symbol table*/
    key = makeTableKey(appName, vhost);
    rtrSym = sym_lookup_symbol(peerTable, key);
    if (! rtrSym) {
        rtrSym = sym_create_symbol(peerTable, key, NULL);
        rtrSym->data = mem_alloc(sizeof(RTRData));
        rtr = rtrSym->data;
        rtr->name  = mem_strdup(appName);
        rtr->vhost = mem_strdup(*vhost ? vhost : VHOST_ANY);
        node_reset(&rtr->peerList);
        node_reset(&rtr->waitingRequests);

        send_wsx_install(&http_thread->queue->qid,
                        (*vhost ? vhost : NULL),
                        appName,
                        LRWP_NAME);
    }
    rtr = rtrSym->data;

    /* tell the router about this thread */
    ptcb = node_create(&rtr->peerList, sizeof(PTCB));
    ptcb->thread = thread;
    ptcb->busy = 0;

    /* give this thread a pointer to the peer thread control block */
    tcb->ptcb = ptcb;
    tcb->rtr = rtr;

    sendfmt(&operq, "INFO",
            LRWP_NAME ": Peer %s connected for %s host",
                tcb->rtr->name, tcb->rtr->vhost);
}


/***********************   SEND REQUEST DATA TO PEER   ***********************/

MODULE send_request_data_to_peer (THREAD *thread)
{
    REQ*    req;
    char    sizebuf[10];
    long    size;

    tcb = thread->tcb;
    req = (REQ*)thread->event->body;
    tcb->replyQ = req->replyQ;


        /* **** Should set and handle timeout on these... */
        /* send the size as a zero-filled field, followed by the env data */
    sprintf(sizebuf, "%09d", req->wsxreq->symbols_size);
    send_write(&sockq, 0, tcb->handle, 9, (byte*)sizebuf, 0);
    send_write(&sockq, 0, tcb->handle, req->wsxreq->symbols_size,
                        req->wsxreq->symbols_data, 0);

        /* The post data goes the same way... */
    if (req->wsxreq->post_data[0] == '@') {
        /* Send the file... */
        size = get_file_size(req->wsxreq->post_data+1);
        sprintf(sizebuf, "%09ld", size);
        send_write(&sockq, 0, tcb->handle, 9, (byte*)sizebuf, 0);
        if (size)
            send_put_file(&tranq, tcb->handle, req->wsxreq->post_data+1);
    }
    else {
        /* Send the buffer */
        sprintf(sizebuf, "%09ld", size = strlen(req->wsxreq->post_data));
        send_write(&sockq, 0, tcb->handle, 9, (byte*)sizebuf, 0);
        if (size)
            send_write(&sockq, 0, tcb->handle, (word)size,
                       (byte*)req->wsxreq->post_data, 0);
    }

        /* free the request data */
    free_smt_wsx_request((void**)&req->wsxreq);

        /* set ourselves up for the return data */
    tcb->sizebuff[0] = 0;
}



/***************************   READ PEER DATA SIZE  **************************/

MODULE read_peer_data_size (THREAD *thread)
{
    int     len;

    tcb = thread->tcb;

    len = read_TCP(tcb->handle, tcb->sizebuff, 9);
    if (len != 9) {
        trace(LRWP_NAME ": read_peer_data **** did not read full size!");
        raise_exception(sock_error_event);
        return;
    }
    tcb->sizebuff[len] = 0;          /* NULL terminate it */
    tcb->numtoread = atoi(tcb->sizebuff);
    tcb->numread = 0;
    tcb->buffer = mem_alloc(tcb->numtoread+1);
}


/*****************************   READ PEER DATA   ****************************/

MODULE read_peer_data (THREAD *thread)
{
    int     len;
    tcb = thread->tcb;

    len = read_TCP(tcb->handle, tcb->buffer+tcb->numread,
                                tcb->numtoread-tcb->numread);
    if (len == SOCKET_ERROR) {
        if (sockerrno == EAGAIN || sockerrno == EWOULDBLOCK)
                return;
        else
        if (sockerrno == EPIPE || sockerrno == ECONNRESET) {
                the_next_event = sock_closed_event;
                return;
        }
    }

    tcb->numread += len;
    if (tcb->numread == tcb->numtoread) {
        the_next_event = finished_event;
        tcb->buffer[tcb->numread] = 0;
    }
}

/************************   SEND PEER DATA TO XITAMI   ***********************/

MODULE send_peer_data_to_xitami (THREAD *thread)
{
    static unsigned short   count = 0;
    char                    filename[16];
    int                     hdl;
    int                     err;


    tcb = thread->tcb;

    sprintf(filename, "@lrwp%05d.tmp", ++count);

    hdl = open(filename+1, O_CREAT | O_BINARY | O_RDWR, 0666);
    err = write(hdl, tcb->buffer, tcb->numread);
    err = close(hdl);
    send_wsx_ok(&tcb->replyQ, filename);
}


/***********************   CLEANUP AND INFORM ROUTER   ***********************/

MODULE cleanup_and_inform_router (THREAD *thread)
{
    tcb = thread-> tcb;
    tcb->ptcb->busy = FALSE;
    *tcb->sizebuff = 0;
    tcb->numtoread = 0;
    tcb->numread = 0;
    mem_free(tcb->buffer);

    SEND(&rtrThread->queue->qid, LRWP_JOBCOMP,
            makeTableKey(tcb->rtr->name, tcb->rtr->vhost));
}



/***************************   SIGNAL PEER CLOSED   **************************/

MODULE signal_peer_closed (THREAD *thread)
{
    tcb = thread->tcb;                 /*  Point to thread's context        */
    sendfmt(&operq, "INFO",
            LRWP_NAME ": App %s on %s terminating",
            tcb->rtr->name, tcb->rtr->vhost);
}

/***************************   SIGNAL STARTUP ERROR   *************************/

MODULE signal_startup_error (THREAD *thread)
{
    tcb = thread->tcb;                 /*  Point to thread's context        */
    sendfmt(&operq, "INFO",
            LRWP_NAME ": Peer failed to connect (%s)", scratch);
}

/****************************   SEND ERROR REPLY   ***************************/

MODULE send_error_reply (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */
    send_wsx_error(&tcb->replyQ, HTTP_RESPONSE_INTERNALERROR);
}


/**************************   FIND AVAILABLE PEER   **************************/

static Bool send_to_unbusy_peer(THREAD* thread, RTRData* rtr, REQ* req)
{
    PTCB*    ptcb;

    for (ptcb=rtr->peerList.next;
        (NODE*)ptcb != &rtr->peerList;
        ptcb=ptcb->next) {
        if (! ptcb->busy) {
            event_send(&ptcb->thread->queue->qid,   /* To           */
                &thread->queue->qid,                /* From         */
                LRWP_REQUEST,                       /* Event name   */
                (byte*)req, sizeof(REQ),            /* The message  */
                NULL, NULL, NULL, 0);

                /* free the req, but not its contents. */
            node_destroy(req);
            ptcb->busy = TRUE;
            return TRUE;
        }
    }
    return FALSE;
}


static RTRData* lrwp_match_peer(char* uri, char* vhost)
{
    SYMBOL*     symbol;
    RTRData*    rtr;

    if (*uri == '/')
        uri++;

    symbol = peerTable->symbols;
    while (symbol) {
        rtr = symbol->data;
        if ((streq(vhost, rtr->vhost) || streq(VHOST_ANY, rtr->vhost)) &&
             strprefixed(uri, rtr->name))
            return rtr;

        symbol = symbol->next;
    }
    return NULL;
}


MODULE find_available_peer (THREAD *thread)
{
    REQ*        req;
    WSXREQ*     wsxreq;
    RTRData*    rtr;


    tcb = thread->tcb;                 /*  Point to thread's context        */

    /*  Decode the WSX request          */
    get_smt_wsx_request(thread->event->body, (void **)&wsxreq);
    if (!wsxreq) {
        /*  The request can only be null if there is no memory left       */
        send_wsx_error(&thread->event->sender, HTTP_RESPONSE_OVERLOADED);
        the_next_event = exception_event;
        return;
    }

    req = node_create(NULL, sizeof(REQ));
    req->replyQ = thread->event->sender;
    req->wsxreq = wsxreq;

        /*
         * find peerData that matches the vhost and the leading portion of the
         * wsxreq->request_url
         */
    rtr = lrwp_match_peer(wsxreq->request_url, wsxreq->virtual_host);
    if (!rtr) {
        send_wsx_error(&req->replyQ, HTTP_RESPONSE_NOTFOUND);
        mem_free(req);
        free_smt_wsx_request((void**)&wsxreq);
        return;
    }


        /* send request to available peer */
    if (! send_to_unbusy_peer(thread, rtr, req)) {
            /* if none available, save the request for later */
        node_relink_before(req, &rtr->waitingRequests);
    }
}

/************************   CHECK FOR SAVED REQUESTS   ***********************/

MODULE check_for_saved_requests (THREAD *thread)
{
    REQ*        req;
    char*       key;
    SYMBOL*     rtrSym;
    RTRData*    rtr;

    tcb = thread->tcb;
    key = (char*)thread->event->body;
    rtrSym = sym_lookup_symbol(peerTable, key);
    if (rtrSym) {
        rtr = rtrSym->data;
        if (rtr->waitingRequests.next
        && (rtr->waitingRequests.next != (void*)&rtr->waitingRequests)) {
            req = rtr->waitingRequests.next;
            node_unlink(req);
                /* send request to available peer */
            if (! send_to_unbusy_peer(thread, rtr, req)) {
                /* if none available, put the request back at the
                   head of the list */
                node_relink_after(req, &rtr->waitingRequests);
            }
        }
    }
}


/***************************   CHECK IF ALL GONE   ***************************/

MODULE check_if_all_gone (THREAD *thread)
{
    REQ*        req;
    char*       key;
    SYMBOL*     rtrSym;
    RTRData*    rtr;

    tcb = thread->tcb;                 /*  Point to thread's context        */

    /* if the peer list is empty, cleanup. */
    key = (char*)thread->event->body;
    rtrSym = sym_lookup_symbol(peerTable, key);
    if (rtrSym) {
        rtr = rtrSym->data;
        if (rtr->peerList.next == (void*)&rtr->peerList) {
            /* first clean up the waitingRequests list. */
            while (rtr->waitingRequests.next != (void*)&rtr->waitingRequests) {
                req = rtr->waitingRequests.next;
                node_unlink(req);
                send_wsx_error(&req->replyQ, HTTP_RESPONSE_NOTIMPLEMENTED);
                mem_free(req);
                free_smt_wsx_request((void**)&req->wsxreq);
            }

            /* Remove this alias from Xitami */
            send_wsx_cancel(&http_thread->queue->qid,
                            (streq(rtr->vhost, VHOST_ANY) ? NULL : rtr->vhost),
                            rtr->name);

            /* Then remove this thing from the peerTable */
            mem_free(rtr->name);
            mem_free(rtr->vhost);
            mem_free(rtrSym->data);
            sym_delete_symbol(peerTable, rtrSym);

            /* if the table is empty, get rid of it. */
            if (peerTable->size == 0) {
                sym_delete_table(peerTable);
                peerTable = NULL;
                if (tcb->termWhenEmpty)
                    the_next_event = terminate_event;
            }
        }
    }
}


