/*  ----------------------------------------------------------------<Prolog>-
    Name:       smthttpl.c
    Title:      Hyper Text Transfer Protocol functions library
    Package:    Libero SMT 2.x

    Written:    96/04/26  iMatix SMT kernel team <smt@imatix.com>
    Revised:    98/01/31

    Copyright:  Copyright (c) 1991-98 iMatix
    License:    This is free software; you can redistribute it and/or modify
                it under the terms of the SMT License Agreement as provided
                in the file LICENSE.TXT.  This software is distributed in
                the hope that it will be useful, but without any warranty.
 ------------------------------------------------------------------</Prolog>-*/

#include "sfl.h"                        /*  SFL library header file          */

#define  DEFINE_HTTP_TABLES
#include "smthttpl.h"                   /*  Function prototypes              */


/*- Global variables used in this source only -------------------------------*/

static char
    server_message_text [LINE_MAX] = "";
static char
    *eoln = "\r\n";                     /*  Standard end-of-line             */
static char
    buffer [HTTP_URL_MAX + 1];          /*  General-use string buffer        */


/*- Global variables --------------------------------------------------------*/

char
    *server_name = "iMatix SMTHTTP Agent",
    *server_message = server_message_text;
qbyte
    connect_count = 0,                  /*  Number of connections made       */
    cur_connects  = 0,                  /*  Cur connections made             */
    max_connects  = 0,                  /*  Max connections made             */
    error_count   = 0,                  /*  Number of errors                 */
    transfer_size = 0;                  /*  Amount of data transfered        */
SYMTAB
    *config = NULL,                     /*  Global configuration table       */
    *defaults = NULL,                   /*  Defaults for config options      */
    *wsx_aliases = NULL;                /*  WSX aliases table                */
Bool
    server_killed = FALSE,              /*  If ended from control panel      */
    defaults_loaded = FALSE;            /*  Is defaults table accessible?    */


/*- Definitions and constants -----------------------------------------------*/

#define MAP_MAX_POINTS      100         /*  Max. points in one map line      */
#define MAX_DEFAULTS        32          /*  Max. defaults to look for        */

/*  These limits are a rough way of ensuring that user-defined error texts   */
/*  do not exceed the size of the header buffer (4096).                      */
#define ERROR_FOOTER_MAX    1000        /*  Max. size of error header text   */
#define ERROR_HEADER_MAX    1000        /*  Max. size of error footer text   */
#define ERROR_BODY_MAX      2000        /*  Max. size of error body text     */

#define HCONFIG(s)      http_config (p_http-> config, (s))


/*- Internal function prototypes --------------------------------------------*/

static void    load_defaults_table  (void);
static void    http_free_strings    (HTTP_CONTEXT *p_http);
static void    http_close_cgi       (HTTP_CONTEXT *p_http);
static char   *get_map_value        (char *map_file, char *request_point);
static void    point_in_map_element (char *buffer, char *rc_url, int x, int y,
                                     char *url_def);
static char   *http_get_file_type   (HTTP_CONTEXT *);
static char   *http_get_filter      (HTTP_CONTEXT *);
static Bool    set_keep_alive       (HTTP_CONTEXT *);
static void    parse_authorization  (HTTP_CONTEXT *);
static SYMTAB *add_cgi_environment  (HTTP_CONTEXT *);
static void    format_error_header  (HTTP_CONTEXT *, char *buffer);
static void    format_full_url      (HTTP_CONTEXT *, char *buffer,
                                     char *url, char *suffix);
static char   *find_eoln            (char *buffer);


/*  ---------------------------------------------------------------------[<]-
    Function: http_init

    Synopsis: Initialises the HTTP library.
    ---------------------------------------------------------------------[>]-*/

void
http_init (void)
{
    if (defaults == NULL)
        load_defaults_table ();

    if (wsx_aliases == NULL)
        wsx_aliases = sym_create_table ();

    if (*http_config (config, "server:debug") == '1')
      {
        enable_trace ();
        set_trace_file ("debug.log", 'a');
      }

    /*  These values must be calculated at runtime after initialisation     */
    sym_assume_symbol (defaults, "server:hostname", get_hostname ());
    http_capture_console ();
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_term

    Synopsis: Terminates the HTTP library.
    ---------------------------------------------------------------------[>]-*/

void
http_term (void)
{
    defaults_loaded = FALSE;
    if (defaults)
      {
        sym_delete_table (defaults);
        defaults = NULL;
      }
    if (wsx_aliases)
      {
        sym_delete_table (wsx_aliases);
        wsx_aliases = NULL;
      }
    disable_trace ();
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_config

    Synopsis: Returns config value for specified key.  If the value is not
    defined, returns the default value.  If no default is defined, returns
    an empty string ("").
    ---------------------------------------------------------------------[>]-*/

char *
http_config (SYMTAB *table, const char *key)
{
    if (defaults == NULL)
        load_defaults_table ();

    return (sym_get_value (table, key,
            sym_get_value (defaults, key, "")));
}


static void
load_defaults_table (void)
{
    static struct {
        char *name;
        char *value;
    } default_list [] = {

    { "server:", NULL                                                        },
    {   "autostart",        "1"                                              },
    {   "background",       "0"                                              },
    {   "base_host",        "basehost.cfg"                                   },
    {   "cache_defaults",   "1"                                              },
    {   "cgi_bin",          "cgi-bin"                                        },
    {   "cgi_url",          "/cgi-bin"                                       },
    {   "defaults",         "defaults.cfg"                                   },
    {   "debug",            "0"                                              },
    {   "default1",         "index.htm"                                      },
    {   "default2",         "index.html"                                     },
    {   "default3",         "default.htm"                                    },
    {   "default4",         "default.html"                                   },
    {   "error_footer",     "@errors/footer.htm"                             },
    {   "error_header",     "@errors/header.htm"                             },
    {   "ipaddress",        "*"                                              },
    {   "keep_alive",       "1"                                              },
    {   "keep_alive_max",   "50"                                             },
    {   "limit",            "0",                                             },
    {   "portbase",         "0"                                              },
    {   "priority",         "1"                                              },
    {   "refresh",          "60"                                             },
    {   "text_400",         "@errors/text-400.htm"                           },
    {   "text_401",         "@errors/text-401.htm"                           },
    {   "text_402",         "@errors/text-402.htm"                           },
    {   "text_403",         "@errors/text-403.htm"                           },
    {   "text_404",         "@errors/text-404.htm"                           },
    {   "text_413",         "@errors/text-413.htm"                           },
    {   "text_500",         "@errors/text-500.htm"                           },
    {   "text_501",         "@errors/text-501.htm"                           },
    {   "text_502",         "@errors/text-502.htm"                           },
    {   "timeout",          "30"                                             },
    {   "translate",        "1"                                              },
    {   "webpages",         "webpages"                                       },

    { "wsx:", NULL                                                           },
    {   "xiadmin",          "/admin"                                         },

    { "cgi:", NULL                                                           },
    {   "debug",            "0"                                              },
    {   "dirlist",          "0"                                              },
    {   "environment",      "1"                                              },
    {   "errlog",           "cgierr.log"                                     },
    {   "exit_ok",          "0"                                              },
    {   "form_fields",      "1"                                              },
    {   "form_prefix",      "FORM_"                                          },
    {   "form_query",       "0"                                              },
    {   "http_fields",      "1"                                              },
    {   "http_prefix",      "HTTP_"                                          },
    {   "mixed_url",        "1"                                              },
    {   "monitor",          "200"                                            },
    {   "msdos-style",      "0"                                              },
    {   "stdio",            "1"                                              },
    {   "timeout",          "60"                                             },
    {   "workdir",          "cgi-bin"                                        },

    { "security:", NULL                                                      },
    {   "dirlist",          "1"                                              },
    {   "encrypt",          "0"         /*  TO BE IMPLEMENTED            */  },
    {   "filename",         "xitami.aut"                                     },
    {   "password_case",    "1"                                              },
    {   "superuser",        ""                                               },
    {   "webmask",          "*"                                              },
    {   "admin",            "1"                                              },

    { "serverlog:", NULL                                                     },
    {   "cycle",            "daily"                                          },
    {   "enabled",          "1"                                              },
    {   "filename",         "xitami.log"                                     },

    { "accesslog:", NULL                                                     },
    {   "cycle",            "daily"                                          },
    {   "enabled",          "1"                                              },
    {   "filename",         "access.log"                                     },

    { "errorlog:", NULL                                                      },
    {   "cycle",            "daily"                                          },
    {   "enabled",          "1"                                              },
    {   "filename",         "errors.log"                                     },

    { "console:", NULL                                                       },
    {   "append",           "0"                                              },
    {   "capture",          "0"                                              },
    {   "filename",         "console.log"                                    },
    {   "rate",             "10",                                            },
    {   "refresh",          "1"                                              },
    {   "startup",          "1"                                              },

    { "ftp:", NULL                                                           },
    {   "email_check",      "0"                                              },
    {   "data_port",        "200"                                            },
    {   "enabled",          "1"                                              },
    {   "force_ip",         "0"                                              },
    {   "http_aliases",     "0",                                             },
    {   "ipaddress",        "127.0.0.1"                                      },
    {   "limit",            "25"                                             },
    {   "port",             "21"                                             },
    {   "root",             "ftproot"                                        },
    {   "timeout",          "300"                                            },
    {   "user_file",        "ftpusers.aut"                                   },
    {   "directory_file",   "ftpdirs.aut"                                    },
    {   "webmask",          "*"                                              },
    {   "welcome",          "@ftphello.txt"                                  },
    {   "login-text",       "@ftplogin.txt"                                  },

    { "ftplog:", NULL                                                        },
    {   "enabled",          "1"                                              },
    {   "filename",         "access.log"                                     },
    {   "cycle",            "daily"                                          },

    { "ftperrlog:", NULL                                                     },
    {   "enabled",          "1"                                              },
    {   "filename",         "errors.log"                                     },
    {   "cycle",            "daily"                                          },

    { "mime:", NULL                                                          },
    {   "doc",              "text/plain"                                     },
    {   "gif",              "image/gif"                                      },
    {   "htm",              "text/html"                                      },
    {   "html",             "text/html"                                      },
    {   "jpg",              "image/jpeg"                                     },
    {   "jpeg",             "image/jpeg"                                     },
    {   "txt",              "text/plain"                                     },

    { "win32:", NULL                                                         },
    {   "16bit_cgi",        "1"                                              },
    {   "secure",           "0"                                              },

    { "lrwp:", NULL                                                          },
    {   "enabled",          "1"                                              },
    {   "port",             "81"                                             },
    {   "webmask",          "local"                                          },

    { NULL, NULL }
    };
    int
        index;

    defaults = sym_create_table ();
    for (index = 0; default_list [index].name; index++)
      {
        /*  If new heading, get heading name                                 */
        if (default_list [index].value == NULL)
            strcpy (buffer, default_list [index].name);
        else
          {
            /*  Else reuse existing heading name in buffer                   */
            ASSERT (strchr (buffer, ':'));
            *strchr (buffer, ':') = '\0';
            strcat (buffer, ":");
            strcat (buffer, default_list [index].name);
            sym_assume_symbol (defaults, buffer, default_list [index].value);
          }
      }
    defaults_loaded = TRUE;
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_init_context

    Synopsis: Initialises a HTTP_CONTEXT structure; this structure is used
    in calls to the functions that provide HTTP support.  You must allocate
    the structure yourself.
    ---------------------------------------------------------------------[>]-*/

void
http_init_context (HTTP_CONTEXT *p_http)
{
    ASSERT (p_http);
    p_http-> method          = 0;
    p_http-> protocol        = 0;
    p_http-> response        = 0;
    p_http-> file_size       = 0;
    p_http-> file_date       = 0;
    p_http-> file_time       = 0;
    p_http-> file_offset     = 0;
    p_http-> socket          = 0;
    p_http-> port            = 0;
    p_http-> output_timeout  = 0;
    p_http-> keep_alive      = FALSE;
    p_http-> transaction     = 0;
    p_http-> error_type      = 0;
    p_http-> send_file       = FALSE;
    p_http-> temp_file       = FALSE;
    p_http-> file_name       = NULL;
    p_http-> url             = NULL;
    p_http-> url_param       = NULL;
    p_http-> table           = NULL;
    p_http-> username        = NULL;
    p_http-> password        = NULL;
    p_http-> realm           = NULL;
    p_http-> script_path     = NULL;
    p_http-> script_name     = NULL;
    p_http-> path_info       = NULL;
    p_http-> cgi_process_id  = NULL_PROCESS;
    p_http-> cgi_environ     = NULL;
    p_http-> post_data       = NULL;
    p_http-> cgi_stdin       = NULL;
    p_http-> cgi_status      = NULL;
    p_http-> vhost           = NULL;
    p_http-> wsx_agent       = NULL;
    p_http-> rootdir         = NULL;
    p_http-> cgidir          = NULL;
    p_http-> cgi_header      = FALSE;
    p_http-> default_page    = TRUE;
    http_refresh_context (p_http);
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_refresh_context

    Synopsis: Initialises the various fields in the HTTP structure that
    come from the config table.
    ---------------------------------------------------------------------[>]-*/

void
http_refresh_context (HTTP_CONTEXT *p_http)
{
    p_http-> input_timeout  = atoi (HCONFIG ("server:timeout"));
    p_http-> keep_alive_max = atoi (HCONFIG ("server:keep_alive_max"));
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_free_context

    Synopsis: Frees any strings allocated in the HTTP_CONTEXT block.  Use
    this function when you are finished using the HTTP support functions.
    Does not actually free the HTTP_CONTEXT block.
    ---------------------------------------------------------------------[>]-*/

void
http_free_context (HTTP_CONTEXT *p_http)
{
    ASSERT (p_http);

    http_close_cgi    (p_http);
    http_free_strings (p_http);
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_set_priority

    Synopsis: Initialises the HTTP server priority.
    ---------------------------------------------------------------------[>]-*/

void
http_set_priority (void)
{
    if (*http_config (config, "server:priority") == '2')
        process_priority (PRIORITY_LOW);
    else
    if (*http_config (config, "server:priority") == '3')
        process_priority (PRIORITY_HIGH);
    else
        process_priority (PRIORITY_NORMAL);
}


static void
http_close_cgi (HTTP_CONTEXT *p_http)
{
    if (p_http-> table)
      {
        sym_delete_table (p_http-> table);
        p_http-> table = NULL;
      }
    if (p_http-> cgi_environ)
      {
        strtfree (p_http-> cgi_environ);
        p_http-> cgi_environ = NULL;
      }
    http_close_process (p_http);
}


static void
http_free_strings (HTTP_CONTEXT *p_http)
{
    mem_strfree (&p_http-> url);
    mem_strfree (&p_http-> url_param);
    mem_strfree (&p_http-> username);
    mem_strfree (&p_http-> password);
    mem_strfree (&p_http-> realm);
    mem_strfree (&p_http-> script_path);
    mem_strfree (&p_http-> script_name);
    mem_strfree (&p_http-> path_info);
    mem_strfree (&p_http-> post_data);
    mem_strfree (&p_http-> cgi_status);
    mem_strfree (&p_http-> vhost);
    mem_strfree (&p_http-> wsx_agent);
    mem_strfree (&p_http-> rootdir);
    mem_strfree (&p_http-> cgidir);

    if (p_http-> cgi_stdin)
      {
        if (*(HCONFIG ("cgi:debug")) == '0')
            file_delete (p_http-> cgi_stdin);
        mem_strfree (&p_http-> cgi_stdin);
      }
    if (p_http-> file_name)
      {
        if (p_http-> temp_file
        && (p_http-> transaction == TRANSACTION_FILE
        || *(HCONFIG ("cgi:debug")) == '0'))
            file_delete (p_http-> file_name);
        mem_strfree (&p_http-> file_name);
        p_http-> temp_file = FALSE;
      }
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_parse_request

    Synopsis: Extracts the HTTP method, URL, HTTP version from the request,
    and parses all other HTTP headers into the http symbol table.  Returns
    the HTTP method type, or zero if there was an error.  Note: this
    function destroys the buffer.

    Examples:
    This is a typical request -
        POST /cgi-bin/process HTTP/1.0
        If-Modified-Since: Sunday, 05-Nov-95 20:23:42 GMT; length=683
        Connection: Keep-Alive
        User-Agent: Mozilla/2.0GoldB2 (Win95; I)
        Host: www.imatix.com
        Accept: image/gif, image/x-xbitmap, image/jpeg, image/pjpeg
        <blank line>
        ...post data...
---------------------------------------------------------------------[>]-*/

int
http_parse_request (HTTP_CONTEXT *p_http, char *request)
{
    char
        *protocol,
        *method,
        *url,
        *separator,
        *query_body,
        *p_param;
    int
        index;
    Bool
        found = FALSE;

    ASSERT (p_http);
    ASSERT (request);

    /*  Create a symbol table for MIME values in the request header         */
    if (!p_http-> table)
        p_http-> table = sym_create_table ();

    /*  Save the first line for later output to the log file.  We limit      */
    /*  this to LINE_MAX characters, which is the space allowed for it       */
    /*  in the HTTP_CONTEXT structure.                                       */
    separator = strpbrk (request, "\n\r");
    if (separator)
      {
        *separator = '\0';
        strncpy (p_http-> log_request, request, LINE_MAX);
        p_http-> log_request [LINE_MAX] = 0;
        *separator = '\r';
      }

    /*  Get the method in the request; delimit on space or tabs              */
    p_http-> response = HTTP_RESPONSE_OK;
    method = strtok (request, "\t ");
    if (method)
      {
        for (index = 0; strused (methods [index]); index++)
            if (!strcmp (method, methods [index]))
              {
                p_http-> method = index + 1;
                found = TRUE;
                if (p_http-> method == HTTP_METHOD_GET)
                    p_http-> send_file = TRUE;
                break;
              }
        if (!found)
          {
            p_http-> response = HTTP_RESPONSE_NOTIMPLEMENTED;
            return (0);
          }
      }
    else
      {
        p_http-> response = HTTP_RESPONSE_INTERNALERROR;
        return (0);
      }

    /*  Get the URL in the request; the URL is everything between the method
     *  and the HTTP protocol indicator...  It's a little messy, but we need
     *  to allow for URLs that contain spaces (e.g. Win95).  So, we look from
     *  the end of the line, take-off " HTTP/x.x" and use what remains as the
     *  URL.  If we don't find HTTP/x.x at the end, we default to HTTP/0.9.
     */
    url = strtok (NULL, "\n\r");        /*  Get token up to end of line      */
    if (url)
      {
        index = strlen (url);

        /*  Get the protocol in the request                                  */
        protocol = url + index - 8;     /*  Look for HTTP/x.x                */
        p_http-> protocol = HTTP_PROT_0_9;
        if (strprefixed (protocol, "HTTP/"))
          {
            for (index = 0; strused (protocols [index]); index++)
                if (streq (protocol, protocols [index]))
                  {
                    p_http-> protocol = index;
                    break;
                  }
            *protocol = '\0';           /*  Cut URL at protocol              */
            strcrop (url);              /*  Kill trailing spaces             */
          }
        /*  Check if URL has parameters (for CGI or image map)               */
        p_param = strchr (url, '?');
        if (p_param)
          {
            *p_param++ = '\0';
            p_http-> url_param = mem_strdup (p_param);
          }
        /*  Store the URL, resolving all . and .. in path                    */
        http_unescape (url, NULL);
        p_http-> url = mem_strdup (resolve_path (url));
      }

    /*  Parse HTTP header variables; point to line after header              */
    query_body = http_parse_header (p_http, strtok (NULL, "\n"));
    if (query_body)
      {
        query_body = strtok (NULL, "");
        if (query_body)                 /*  POST data                        */
          {
            /*  Browsers sometimes add CRLF...  Kill it                      */
            separator = query_body + strlen (query_body) - 1;
            if (*separator < ' ')
              {
                *separator-- = '\0';
                if (*separator < ' ')
                    *separator = '\0';
              }
            p_http-> post_data = mem_strdup (query_body);
          }
      }
    /*  Set the keep-alive connection flag                                   */
    set_keep_alive      (p_http);
    parse_authorization (p_http);

    return (p_http-> method);
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_parse_header

    Synopsis: Parses a HTTP header string, and stores the header values into
    the symbol table.  Each name is stored as lower-case; values are stored
    as-is.  The string should contain header values like 'Content-Length: 85'
    separated by CRLF.   A double CRLFCRLF, or a null byte, ends the header.
    Modifies header.  Returns a pointer to the blank line or null ending the
    header text.
    ---------------------------------------------------------------------[>]-*/

char *
http_parse_header (HTTP_CONTEXT *p_http, char *header)
{
    char
        *header_field,
        *separator;
    int
        fields = 0;

    ASSERT (p_http);

    /*  Insert "0" as default value for numeric fields                       */
    sym_assume_symbol (p_http-> table, "content-length", "0");
    if (header == NULL)
        return (NULL);

    header_field = header;
    while (header_field && header_field [0] != '\r')
      {
        if ((separator = strchr (header_field, ':')) != NULL)
          {
            *separator = '\0';
            separator++;
            while (*separator == ' ')
                separator++;

            strlwc (header_field);      /*  Put name into lowercase          */

            /*  Remove any trailing carriage-return or linefeed              */
            if ((strchr (separator, '\r')) != NULL)
                *strchr (separator, '\r') = '\0';

            /*  It's possible that the Referer: field contains an URL
             *  followed by ? and a whole long argument...  We don't want
             *  that stuff.  Get rid of it.
             */
            if ((streq (header_field, "referer"))
            &&  (strchr (separator, '?')) != NULL)
                *strchr (separator, '?') = '\0';

            sym_assume_symbol (p_http-> table, header_field, separator);
            fields++;
          }
        header_field = strtok (NULL, "\n");
      }
    return (header_field);
}


/*  ---------------------------------------------------------------------[<]-
    Function: set_keep_alive - internal

    Synopsis: check if keep alive connection is in the mime symbol table
    ---------------------------------------------------------------------[>]-*/

static Bool
set_keep_alive (HTTP_CONTEXT *p_http)
{
    SYMBOL
        *symptr;

    ASSERT (p_http);
    ASSERT (p_http-> table);

    p_http-> keep_alive = FALSE;
    if (*(HCONFIG ("server:keep_alive")) == '1')
      {
        symptr = sym_lookup_symbol (p_http-> table, "connection");
        if (symptr
        && !strncmp (symptr-> value, "Keep-Alive", 10)
        &&  p_http-> keep_alive_max > 1)
          {
            /*  We disable Nagle's algorithm; this makes keep-alive
             *  connections faster.  We don't do it for non-keep-alive
             *  connections, since it can slow-down the network.
             */
            socket_nodelay (p_http-> socket);
            p_http-> keep_alive = TRUE;
          }
      }
    return (p_http-> keep_alive);
}


/*  -------------------------------------------------------------------------
    Function: parse_authorization -- internal

    Synopsis: Parses the Basic Authorization field from the HTTP header.
    If the field is present, fills the HTTP_CONTEXT username and password
    values.  Otherwise these values are left unchanged.
    -------------------------------------------------------------------------*/

static void
parse_authorization (HTTP_CONTEXT *p_http)
{
    SYMBOL
        *symptr;                        /*  Authorization symbol             */
    char
        *auth,                          /*  Encoded Authorization value      */
        *delim;                         /*  ':' delimiter after name         */
    int
        buflen;                         /*  Length of decoded buffer         */

    ASSERT (p_http);
    ASSERT (p_http-> table);
    ASSERT (p_http-> username == NULL);
    ASSERT (p_http-> password == NULL);

    symptr = sym_lookup_symbol (p_http-> table, "authorization");
    if (symptr && memcmp (symptr-> value, "Basic ", 6) == 0)
      {
        auth   = symptr-> value + 6;    /*  Get & decode authorization       */
        buflen = decode_base64 ((byte *) auth, buffer, strlen (auth));
        buffer [buflen] = '\0';         /*  Terminate string nicely          */

        delim = strchr (buffer, ':');
        if (delim)
          {
            *delim++ = '\0';
            p_http-> username = mem_strdup (buffer);
            p_http-> password = mem_strdup (delim);
            /*  User names are always compared in lower-case                 */
            strlwc    (p_http-> username);
            strconvch (p_http-> username, '-', '_');
            if (*(HCONFIG ("security:password_case")) == '0')
                strlwc (p_http-> password);
          }
      }
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_header_value

    Synopsis: Returns the value for the specified HTTP header field.  If
    the field was not present in the header, returns an empty string, else
    returns the value.  The header field may be specified in mixed-case;
    the symbol table lookup disregards case.
    ---------------------------------------------------------------------[>]-*/

char *
http_header_value (HTTP_CONTEXT *p_http, char *field)
{
    SYMBOL
        *symptr;                        /*  Symbol to look for               */

    ASSERT (p_http);
    ASSERT (p_http-> table);

    strcpy (buffer, field);
    strlwc (buffer);
    symptr = sym_lookup_symbol (p_http-> table, buffer);
    if (symptr)
        return (symptr-> value);
    else
        return ("");
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_get_url_type

    Synopsis: Gets the URL type (used for CGI and image maps).  Algorithm
    used is as follows: if the URL contains '.map' it is a map file.  If
    the URL contains '/cgi-bin', or starts with a CGI alias, it is a CGI
    command.  If the URL matches a WSX pattern, it is a WSX plugin; if it
    matches a filter pattern, it's a filtered file.  Returns one of:
    <Table>
    HTTP_URL_HTML       Normal HTML text (the default)
    HTTP_URL_IMAP       Image map type
    HTTP_URL_CGI        CGI program
    HTTP_URL_WSX        WSX application
    HTTP_URL_FILTER     Filtered file
    </Table>
    ---------------------------------------------------------------------[>]-*/

int
http_get_url_type (HTTP_CONTEXT *p_http)
{
    int
        url_type;

    ASSERT (p_http);
    ASSERT (p_http-> url);

    /*  Default transaction is a normal file URL                             */
    p_http-> transaction = TRANSACTION_FILE;

    /*  For a start, let's kick-out any map files                            */
    if (strstr (p_http-> url, ".map"))
        url_type = HTTP_URL_IMAP;
    else
    if (strstr (p_http-> url, HCONFIG ("server:cgi_url"))
    ||  http_match_alias (p_http-> config, p_http-> url, "cgi_alias", NULL))
      {
        p_http-> transaction = TRANSACTION_CGI;
        url_type = HTTP_URL_CGI;
      }
    else
    if (http_match_wsx (p_http))
      {
        p_http-> transaction = TRANSACTION_WSX;
        url_type = HTTP_URL_WSX;
      }
    else
    if (http_get_filter (p_http))
      {
        p_http-> transaction = TRANSACTION_FILTER;
        url_type = HTTP_URL_FILTER;
      }
    else
        url_type = HTTP_URL_HTML;

    return (url_type);
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_match_alias

    Returns the alias symbol pointer if the URI matches an alias defined
    in the configuration table.  The alias type is the section name without
    a colon.  The alias may optionally start with '/'.  Aliases match only
    whole path components.  For example, the aliases 'demo' and '/demo' both
    match a URI '/demo/myprog/'.  If the 'base' argument is not null, it is
    set to point to the remainder of the URI after the alias.  Assumes that
    the config table is sorted.
    ---------------------------------------------------------------------[>]-*/

SYMBOL *
http_match_alias (
    SYMTAB *config, const char *uri, const char *type, char **base)
{
    SYMBOL
        *symbol;
    char
        *alias_name,                    /*  Name of alias, no leading '/'    */
        *base_char;

    /*  Find section header for aliases                                      */
    symbol = sym_lookup_symbol (config, type);
    if (symbol)
        symbol = symbol-> next;         /*  Start with first section entry   */

    /*  Copy URI to buffer, remove leading / if any, and normalise it        */
    if (*uri == '/')
        uri++;
    strcpy    (buffer, uri);
    strlwc    (buffer);                 /*  All comparisons in lowercase     */
    strconvch (buffer, '-', '_');       /*    and with underlines            */

    while (symbol && strprefixed (symbol-> name, type))
      {
        /*  Skip 'type:' and '/' at start of alias for matching              */
        alias_name = symbol-> name + strlen (type) + 1;
        if (*alias_name == '/')
            alias_name++;

        /*  Alias should match start of URI, following '/' or null           */
        if (strprefixed (buffer, alias_name))
          {
            base_char = (char *) uri + strlen (alias_name);
            if (*base_char == '/' || *base_char == '\0')
              {
                if (base)               /*  Return base if necessary         */
                    *base = base_char;
                return (symbol);
              }
          }
        symbol = symbol-> next;
      }
    return (NULL);
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_match_wsx

    Synopsis: Checks whether the URI matches a WSX agent pattern.  If so,
    returns the name of the WSX agent that should handle the request.  If
    not, returns NULL.  WSX agents must be linked with the main server.
    Static WSX agents are defined in the config file under [WSX] as
    "name=pattern", and can be limited to specific virtual hosts.  Dynamic
    WSX agents can send WSX_INSTALL messages to install WSX aliases.
    Static definitions take precedence over dynamic definitions.

    When a WSX agent is found, sets the p_http-> wsx_agent to point to the
    agent name, and path_info to contain the URL part following the alias,
    if any.
    ---------------------------------------------------------------------[>]-*/

char *
http_match_wsx (HTTP_CONTEXT *p_http)
{
    SYMBOL
        *symbol;
    char
        *info;                          /*  Anything following alias         */

    ASSERT (wsx_aliases);

    /*  Try static WSX aliases first                                         */
    symbol = http_match_alias (p_http-> config, p_http-> url, "wsx", &info);
    if (symbol == NULL)
      {
        /*  Now try dynamic WSX aliases for current virtual host             */
        symbol = http_match_alias (wsx_aliases, p_http-> url,
                                   p_http-> vhost, &info);
        if (symbol == NULL && p_http-> uses_vhost)
            /*  Now try dynamic WSX aliases for default virtual host         */
            symbol = http_match_alias (wsx_aliases, p_http-> url,
                                       VHOST_ANY, &info);
      }
    if (symbol)
      {
        /*  Set PATH_INFO variable to whatever followed alias                */
        mem_strfree (&p_http-> path_info);
        mem_strfree (&p_http-> wsx_agent);
        p_http-> path_info = mem_strdup (info);
        p_http-> wsx_agent = mem_strdup (symbol-> value);
        return (symbol-> value);        /*  Return name of WSX agent         */
      }
    else
        return (NULL);
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_wsx_symbols

    Synopsis: Prepares an environment table for a WSX agent and returns
    it as a DESCR block.  The environment table corresponds to the CGI
    environment block (see add_cgi_environment()).  The caller must use
    mem_free() to release the block when finished.  Returns NULL if there
    was insufficient memory to allocate the block, and sets the http
    error_type and response appropriately.
    ---------------------------------------------------------------------[>]-*/

DESCR *
http_wsx_symbols (HTTP_CONTEXT *p_http)
{
    SYMTAB
        *symtab;                        /*  Environment symbol table         */
    char
        **symenv;                       /*  Formatted as environment         */
    DESCR
        *symdesc;                       /*  Formatted as descriptor          */

    /*  Fill in members of p_http that add_cgi_environment() looks for       */
    mem_strfree (&p_http-> script_name);
    mem_strfree (&p_http-> script_path);
    p_http-> script_name = mem_strdup (p_http-> url);
    p_http-> script_path = mem_strdup ("");

    symtab = add_cgi_environment (p_http);
    if (symtab)
      {
        /*  Convert symbols names to uppercase, hyphens to underlines        */
        symenv  = symb2env   (symtab);
        symdesc = strt2descr (symenv);
        sym_delete_table (symtab);
        strtfree (symenv);
        if (symdesc)
            return (symdesc);
      }
    p_http-> error_type = ERROR_CGI_NORESOURCES;
    p_http-> response   = HTTP_RESPONSE_INTERNALERROR;
    return (NULL);
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_check_file

    Synopsis: Checks that a specified file exists and is readable.  Returns
    one of these values:
    <Table>
    HTTP_FILE_OK            File exists and is readable.
    HTTP_FILE_EXECUTABLE    File exists and is executable.
    HTTP_FILE_NOT_OK        File does not exist.
    HTTP_FILE_DIRECTORY     File is a directory, so is not readable.
    </TABLE>
    ---------------------------------------------------------------------[>]-*/

int
http_check_file (HTTP_CONTEXT *p_http)
{
    ASSERT (p_http);

    /*  Translate URL into local file name                                   */
    http_get_file_name (p_http);
    if (p_http-> file_name == NULL)
      {
        trace ("- insufficient memory");
        p_http-> response = HTTP_RESPONSE_OVERLOADED;
        return (HTTP_FILE_NOT_OK);
      }

    /*  First Handle CGI requests.  If the file is an executable program,
     *  we can handle it.  Note that this takes precedence over a directory
     *  with the same name.  If the file is a non-executable file, and the
     *  cgi:mixed_url allows it, we'll send that file.
     */
    if (p_http-> transaction == TRANSACTION_CGI)
      {
        if (file_is_executable (p_http-> file_name))
          {
            trace ("- CGI request refers to an executable file");
            p_http-> response = HTTP_RESPONSE_OK;
            return (HTTP_FILE_EXECUTABLE);
          }
        if (file_is_readable  (p_http-> file_name)
        && !file_is_directory (p_http-> file_name)
        && *HCONFIG ("cgi:mixed_url") == '1')
          {
            trace ("- CGI request refers to a resources file");
            p_http-> response    = HTTP_RESPONSE_OK;
            p_http-> transaction = TRANSACTION_FILE;
            return (HTTP_FILE_OK);
          }
      }

    /*  If the URL is a directory ending in '/' we can handle it.  If it's
     *  a directory without the '/', we send back a 302, so that the browser
     *  will ask for the full directory name...  We check to see whether
     *  there is a default page available, so it can ask for it specifically.
     */
    if (file_is_directory (p_http-> file_name))
      {
        if (strlast (p_http-> file_name) == '/')
          {
            trace ("- request refers to a directory name");
            p_http-> transaction = TRANSACTION_FILE;
            return (HTTP_FILE_DIRECTORY);
          }
        else
          {
            /*  Format directory name including trailing slash              */
            format_full_url (p_http, buffer, p_http-> url, "/");
            mem_strfree (&p_http-> url);
            p_http-> url = mem_strdup (buffer);

            /*  Now generate a 302 'error' response                          */
            trace ("- request redirected to %s", buffer);
            p_http-> response    = HTTP_RESPONSE_FOUND;
            p_http-> transaction = TRANSACTION_FILE;
            return (HTTP_FILE_NOT_OK);
          }
      }

    /*  Finally, check that the URL refers to a normal readable file         */
    if (file_is_readable (p_http-> file_name))
      {
        trace ("- request refers to a normal readable file");
        p_http-> response  = HTTP_RESPONSE_OK;
        return (HTTP_FILE_OK);
      }
    else
      {
        trace ("- requested file '%s' not found", p_http-> file_name);
        p_http-> response = HTTP_RESPONSE_NOTFOUND;
        return (HTTP_FILE_NOT_OK);
      }
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_get_file_name

    Synopsis: Builds a filename for the current URL, with full path
    information.  Places the filename in the HTTP_CONTEXT file_name field.
    Allocates a new string for the filename.  Returns a pointer to the
    filename string, or NULL if there was insufficient memory.
    ---------------------------------------------------------------------[>]-*/

char *
http_get_file_name (HTTP_CONTEXT *p_http)
{
    char
        *pathinfo,                      /*  Presumed path info data          */
        *execname;                      /*  Points to executable filename    */

    ASSERT (p_http);
    ASSERT (p_http-> url);

    /*  These strings are normally NULL already, but they may be set if we're
     *  handling a redirection from a CGI program.
     */
    mem_strfree (&p_http-> path_info);
    mem_strfree (&p_http-> script_path);
    mem_strfree (&p_http-> script_name);
    mem_strfree (&p_http-> file_name);

    p_http-> file_name = mem_strdup (
        http_map_uri (p_http-> config,
                      p_http-> rootdir,
                      p_http-> cgidir,
                      p_http-> url));

    /*  We now split the CGI URI into an executable filename and the path
     *  info that follows, if any.  We do this by removing components from
     *  the path until it is a valid directory.  The component that follows
     *  must be the executable filename, and anything else is the path info.
     */
    if (p_http-> transaction == TRANSACTION_CGI)
      {
        /*  Start with pathinfo set to the trailing null, and execname to
         *  the previous component.  Then work backwards through the path.
         */
        pathinfo = p_http-> file_name + strlen (p_http-> file_name);
        execname = strrchr (p_http-> file_name, '/');
        while (execname)
          {
            *execname = '\0';           /*  Cut path before filename         */
            if (file_is_directory (p_http-> file_name))
              {
                mem_strfree (&p_http-> path_info);
                p_http-> path_info   = mem_strdup (pathinfo);
                *pathinfo = '\0';       /*  Cut file_name at delimiter       */

                mem_strfree (&p_http-> script_path);
                p_http-> script_path = mem_strdup (p_http-> file_name);

                /*  Script name is original URI without path info            */
                mem_strfree (&p_http-> script_name);
                p_http-> script_name = mem_strdup (p_http-> url);
                p_http-> script_name [strlen (p_http-> url)
                                    - strlen (p_http-> path_info)] = '\0';
                *execname = '/';        /*  Restore path to original state   */
                break;
              }
            pathinfo = execname;
            execname = strrchr (p_http-> file_name, '/');
            *pathinfo = '/';            /*  Restore path to original state   */
          }
      }
    else
    if (p_http-> transaction == TRANSACTION_FILTER)
        p_http-> script_name = mem_strdup (http_get_filter (p_http));

    return (p_http-> file_name);
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_map_uri

    Synopsis: Maps an URL into an internal physical filename with full path
    information.  Returns a pointer to a static buffer containing the name,
    or "" if there was a problem.  The mapping algorithm is as follows:

    1.  If the URI is a reserved URI, it is returned unchanged.

    2.  Any single-word alias at the start of the URI is replaced by the
        value of the alias.  We limit aliases to single words to make the
        lookup process faster (no scanning is involved).  Aliases are not
        case-dependent.

    3.  If the URI starts with server:cgi_url (/cgi-bin) then we take the
        following text and append this to the current CGI directory name.
        This cannot be combined with aliasing: if we attach the CGI dir
        to the result of an alias we get garbage.

    4.  If the URI does not start with server:cgi_url then we prefix the
        root directory.  If an alias was used, the root directory is the
        alias.
    ---------------------------------------------------------------------[>]-*/

char *
http_map_uri (
    SYMTAB *config,
    const char *rootdir,
    const char *cgidir,
    const char *uri)
{
    static char
        full_name [LINE_MAX + 1];
    char
        *uri_base,                      /*  URI following alias, if any      */
        *local_root,                    /*  What root dir do we use?         */
        *cgi_url;                       /*  /cgi-bin prefix                  */
    SYMBOL
        *alias;                         /*  Alias symbol pointer             */

    ASSERT (config);
    ASSERT (uri);

    trace ("Resolving URI: %s", uri);

    /*  Check for reserved URIs - these are not mapped                       */
    if (http_match_alias (config, (char *) uri, "wsx", NULL)
    ||  streq (uri, http_config (config, "server:error_url")))
        return ((char *) uri);

    /*  Alias from Outer Space!
     *  Resolve any aliases used in the URI.  The current implementation
     *  only allows single-word aliases at the start of the URI.  So, we
     *  take the first URI path component and see if it is a known alias.
     *  If so, we replace that by the alias value.  Aliases are lowercase.
     */
    /*  We look for CGI aliases first; these take precedence                 */
    alias = http_match_alias (config, uri, "cgi_alias", &uri_base);

    /*  If that did not find anything, try an ordinary alias                 */
    if (alias == NULL)
        alias = http_match_alias (config, uri, "alias", &uri_base);

    if (alias)
      {
        /*  Get alias path, and clean it up                                  */
        trace ("- using alias: %s=%s", alias-> name, alias-> value);
        local_root = resolve_path (alias-> value);
        if (strlast (local_root) == '/')
            strlast (local_root) = '\0';
      }
    else                                /*  No alias: use normal rootdir     */
      {
        uri_base   = (char *) uri;
        local_root = (char *) rootdir;
      }
    trace ("- URI=%s, rootdir=%s", uri_base, local_root);

    /*  If URI matches the CGI directory, construct full filename thusly:
     *  1. If URI starts with "/cgi-bin", take text following "/cgi-bin"
     *     and append to specified cgidir path.
     *  2. If URI does not start with "/cgi-bin", prefix root directory.
     *     URI following alias is always in alias location, obviously.
     *  Note that cgidir and rootdir DO NOT end in '/'.
     */
    cgi_url = http_config (config, "server:cgi_url");
    if (!alias && strprefixed (uri_base, cgi_url))
      {
        /*  File is in specific CGI directory                                */
        uri_base += strlen (cgi_url);
        local_root = (char *) cgidir;
        trace ("- in specific CGI directory %s", cgidir);
      }
    else
        trace ("- URL is in webpage directory %s", local_root);

    if (strlen (local_root) + strlen (uri_base) >= LINE_MAX)
        return ("");                    /*  Too long - can't do nothing      */
    else
      {
        strcpy (full_name, local_root);
        /*  We'll stick-in a / if it looks like it's necessary               */
        if (*uri_base && *uri_base != '/')
            strcat (full_name, "/");
        strcat (full_name, uri_base);
        trace ("- translated file name: %s", full_name);
        return (full_name);
      }
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_format_full_url -- internal

    Synopsis: Takes an URL and returns a buffer formatted with the host
    name, port, and url.  If the URL already contains a hostname, it is
    returned as-is, otherwise the local host name, and the port number
    (if not equal to 80) are prefixed: http://hostname[:port]/url.  The
    suffix (if not NULL and not empty) is attached to the resulting
    buffer.  The hostname is formatted as a number if the config option
    server:translate is set to 0, else it is taken from the config option
    server:hostname, and if that is not defined, from the local hostname.
    ---------------------------------------------------------------------[>]-*/

static void
format_full_url (HTTP_CONTEXT *p_http, char *buffer, char *url, char *suffix)
{
    static char
        port_name [32];                 /*  HTTP port as text                */
    const char
        *local_address;                 /*  Local host address string        */

    ASSERT (p_http);
    ASSERT (buffer);
    ASSERT (url);

    if (is_full_url (url))
        strcpy (buffer, url);
    else
      {
        if (*HCONFIG ("server:translate") == '1')
          {
            local_address = http_header_value (p_http, "host");
            if (strnull (local_address))
                local_address = HCONFIG ("server:hostname");
          }
        else
            local_address = socket_localaddr (p_http-> socket);

        if (p_http-> port > 80 && strchr (local_address, ':') == NULL)
            sprintf (port_name, ":%d", p_http-> port);
        else
            strclr  (port_name);

        sprintf (buffer, "http://%s%s", local_address, port_name);
        if (url [0] != '/')
            strcat (buffer, "/");
        strcat (buffer, url);
        if (suffix && *suffix)
            strcat (buffer, suffix);    /*  Add suffix if necessary          */
      }
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_format_header

    Synopsis: Formats a HTTP response header.  Does not output MIME
    information.  Places the formatted header into the buffer argument
    which should be at least HTTP_BUFFERMIN characters long.  Returns
    buffer.
    ---------------------------------------------------------------------[>]-*/

char *
http_format_header (HTTP_CONTEXT *p_http, char *header)
{
    time_t
        file_time;
    SYMBOL
        *symptr;                        /*  Header field symbol              */

    ASSERT (p_http);
    ASSERT (header);

    /*  Handle the special cases first.  If we just ran a CGI that built     */
    /*  a complete header, our own header is empty.                          */
    if (p_http-> cgi_header)
      {
        strclr (header);
        return (header);
      }
    if (p_http-> cgi_status == NULL)
        sprintf (header, "HTTP/1.0 %s %s\r\n",
                          http_response [p_http-> response].code,
                          http_response [p_http-> response].name);
    else
        /*  CGI program provided Status: field                               */
        sprintf (header, "HTTP/1.0 %s\r\n", p_http-> cgi_status);

    switch (p_http-> response)
      {
        case HTTP_RESPONSE_OK:
            if (p_http-> file_name)
              {
                if (p_http-> file_size == 0)
                    p_http-> file_size = get_file_size (p_http-> file_name);
                if (p_http-> file_time == 0 || p_http-> file_date == 0)
                  {
                    file_time = get_file_time (p_http-> file_name);
                    p_http-> file_date = timer_to_gmdate (file_time);
                    p_http-> file_time = timer_to_gmtime (file_time);
                  }
              }
            sprintf (buffer, "%ld", p_http-> file_size);
            if (p_http-> transaction == TRANSACTION_CGI
            ||  p_http-> transaction == TRANSACTION_FILTER
            ||  p_http-> transaction == TRANSACTION_WSX)
              {
                for (symptr = p_http-> table-> symbols; symptr;
                     symptr = symptr-> next)
                    xstrcat (header, symptr-> name, ": ",
                                     symptr-> value, eoln, NULL);
                xstrcat (header,
                    "Content-Length: ",
                        buffer, eoln,
                    NULL);
              }
            else
              {
                xstrcat (header,
                    "Server: ",
                        server_name, eoln,
                    "Content-Type: ",
                        http_get_file_type (p_http), eoln,
                    "Content-Length: ",
                        buffer, eoln,
                    "Last-Modified: ",
                        encode_mime_time (p_http-> file_date,
                                          p_http-> file_time), eoln,
                    NULL);
              }
            if (p_http-> keep_alive)
              {
                sprintf (buffer, "timeout=%d, max=%d",
                         p_http-> input_timeout, p_http-> keep_alive_max);
                xstrcat (header,
                    "Connection: ",
                        "Keep-Alive", eoln,
                    "Keep-Alive: ",
                        buffer, eoln,
                    NULL);
              }
            strcat (header, eoln);      /*  End header with a blank line     */
            break;

        case HTTP_RESPONSE_FOUND:
            format_full_url (p_http, buffer, p_http-> url, NULL);
            xstrcat (header,
                "Server: ",
                    server_name, eoln,
                "Location: ",
                    buffer, eoln,
                "Content-Length: 0", eoln,
                eoln,                   /*  End header with a blank line     */
                NULL);
            break;

        case HTTP_RESPONSE_UNAUTHORIZED:
            xstrcat (header,            /*  If we set the realm, use that,   */
                "WWW-Authenticate: ",   /*    else use the URL               */
                    "Basic realm=\"",
                    p_http-> realm? p_http-> realm: p_http-> url,
                    "\"", eoln,
                    NULL);
            format_error_header (p_http, header);
            break;

        default:
            /*  Anything >= 400 is an error                                  */
            format_error_header (p_http, header);
            if (p_http-> response >= HTTP_RESPONSE_BADREQUEST)
                error_count++;          /*  Maintain statistics              */
      }
    return (header);
}


/*  Formats a error header                                                   */

static void
format_error_header (HTTP_CONTEXT *p_http, char *header)
{
    static char
        length [12];
    char
        *error_header,                  /*  We load error header/footer      */
        *error_footer,                  /*    when we initialise             */
        *error_body,                    /*  Get error message body if any    */
        *error_lookup;                  /*  Key for custom error message     */
    DESCR
        *error_header_descr = NULL,
        *error_footer_descr = NULL,
        *error_body_descr   = NULL;

    /*  Handle 3xx status codes                                              */
    if (p_http-> response < HTTP_RESPONSE_BADREQUEST)
      {
        if (p_http-> keep_alive)
          {
            sprintf (buffer, "timeout=%d, max=%d",
                     p_http-> input_timeout, p_http-> keep_alive_max);
            xstrcat (header,
                "Connection: ",
                    "Keep-Alive", eoln,
                "Keep-Alive: ",
                    buffer, eoln,
                NULL);
          }
        strclr (buffer);                /*  No body for 3xx messages         */
      }
    /*  Handle 4xx and 5xx status codes                                      */
    else
      {
        /*  Get header and footer from file if neccessary                    */
        error_header = HCONFIG ("server:error_header");
        if (error_header [0] == '@')
          {
            error_header_descr = file_slurp (error_header + 1);
            if (error_header_descr)
              {
                error_header = (char *) error_header_descr-> data;
                if (strlen (error_header) > ERROR_HEADER_MAX)
                    error_header [ERROR_HEADER_MAX] = 0;
              }
            else
                error_header = "<HTML><TITLE>Error</TITLE><BODY><H2>";
          }
        error_footer = HCONFIG ("server:error_footer");
        if (error_footer [0] == '@')
          {
            error_footer_descr = file_slurp (error_footer + 1);
            if (error_footer_descr)
              {
                error_footer = (char *) error_footer_descr-> data;
                if (strlen (error_footer) > ERROR_FOOTER_MAX)
                    error_footer [ERROR_FOOTER_MAX] = 0;
              }
            else
                error_footer = "</H2></BODY></HTML>";
          }

        /*  Now look-up "server:text_xxx" in config table                    */
        error_lookup = xstrcpy (NULL, "server:text_",
                                http_response [p_http-> response].code, NULL);
        error_body = HCONFIG (error_lookup);
        if (error_body [0] == '@')
          {
            error_body_descr = file_slurp (error_body + 1);
            if (error_body_descr)
              {
                error_body = (char *) error_body_descr-> data;
                if (strlen (error_body) > ERROR_BODY_MAX)
                    error_body [ERROR_BODY_MAX] = 0;
              }
            else
                error_body = "";
          }
        if (*error_body)
            sprintf (buffer, "%s\r\n%s\r\n", error_header, error_body);
        else
            sprintf (buffer, "%s\r\n%s %s: %s\r\n",
                     error_header,
                     protocols     [p_http-> protocol],
                     http_response [p_http-> response].code,
                     http_response [p_http-> response].name);

        if (p_http-> error_type)
          {
            strcat (buffer, "<BR>");
            strcat (buffer, error_msg [p_http-> error_type]);
            p_http-> error_type = 0;
          }
        strcat (buffer, error_footer);
        mem_free (error_lookup);
      }
    sprintf (length, "%d", (word) strlen (buffer));
    xstrcat (header,
        "Server: ",
            server_name, eoln,
        "Content-Length: ",
            length, eoln,
        "Content-Type: text/html",
            eoln,
        eoln, buffer,
        NULL);

    /*  Free and reset descriptors; this does no harm if they are NULL       */
    mem_free (error_header_descr);
    mem_free (error_footer_descr);
    mem_free (error_body_descr);
}


/*  -------------------------------------------------------------------------
    Function: http_get_file_type - internal

    Synopsis: Returns the MIME type corresponding to the file extension.
    -------------------------------------------------------------------------*/

static char *
http_get_file_type (HTTP_CONTEXT *p_http)
{
    char
        *extension,
        *mime_lookup,
        *mime_type;

    /*  Find file extension, if any                                          */
    extension = strrchr (p_http-> file_name, '.');
    if (extension)
        extension++;                    /*  Skip past dot if found           */

    /*  Now look-up "mime:xxx" in config table                               */
    mime_lookup = xstrcpy (NULL, "mime:", extension, NULL);
    strlwc    (mime_lookup);            /*  Always lower-case                */
    strconvch (mime_lookup, '-', '_');
    mime_type = HCONFIG (mime_lookup);
    mem_free (mime_lookup);

    /*  Default is  * / *, which allows the browser to choose                */
    if (strnull (mime_type))
        mime_type = "*/*";

    return (mime_type);
}


/*  -------------------------------------------------------------------------
    Function: http_get_filter - internal

    Synopsis: If the file should be handled by a filter, returns filter name.
    Else returns NULL.
    -------------------------------------------------------------------------*/

static char *
http_get_filter (HTTP_CONTEXT *p_http)
{
    char
        *extension,
        *filter_lookup,
        *filter_type;

    /*  Find file extension, if any                                          */
    extension = strrchr (p_http-> url, '.');
    if (extension)
        extension++;                    /*  Skip past dot if found           */

    /*  Now look-up "filter:xxx" in config table                             */
    filter_lookup = xstrcpy (NULL, "filter:", extension, NULL);
    strlwc    (filter_lookup);          /*  Always lower-case                */
    strconvch (filter_lookup, '-', '_');
    filter_type = HCONFIG (filter_lookup);
    mem_free (filter_lookup);

    return (strnull (filter_type)? NULL: filter_type);
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_get_default_page

    Synopsis: Gets the default page for the specified URL, as specified in
    the config file for the virtual host.  Returns TRUE if it finds a
    default page.  The default page is placed into the 'url' field, so
    must be reparsed as if it was a new request.
    ---------------------------------------------------------------------[>]-*/

Bool
http_get_default_page (HTTP_CONTEXT *p_http)
{
    static
        char default_key [20];          /*  server:defaultnnn                */
    char
        *default_name;                  /*  Default name from config file    */
    int
        default_nbr;                    /*  Read from config file?           */

    ASSERT (p_http);

    p_http-> default_page = FALSE;
    for (default_nbr = 1; default_nbr < MAX_DEFAULTS; default_nbr++)
      {
        sprintf (default_key, "server:default%d", default_nbr);
        default_name = HCONFIG (default_key);
        if (strnull (default_name))
            break;
        else
          {
            /*  file_name holds local directory name, maybe ending in '/'     */
            strcpy (buffer, p_http-> file_name);
            if (*buffer && strlast (buffer) != '/')
                strcat (buffer, "/");
            strcat (buffer, default_name);

            /*  If file exists, recreate new URL from previous URL plus
             *  default file name
             */
            if (file_is_readable (buffer))
              {
                strcpy (buffer, p_http-> url);
                if (*buffer && strlast (buffer) != '/')
                    strcat (buffer, "/");
                strcat (buffer, default_name);
                mem_strfree (&p_http-> url);
                p_http-> url = mem_strdup (buffer);
                p_http-> default_page = TRUE;
                break;
              }
          }
      }
    return (p_http-> default_page);
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_check_file_time

    Synopsis: Checks if the file was modified since the last request.  Sets
    the flag 'send_file' to TRUE or FALSE appropriately.
    ---------------------------------------------------------------------[>]-*/

void
http_check_file_time (HTTP_CONTEXT *p_http)
{
    SYMBOL
        *symptr;
    time_t
        time_secs;
    long
        mime_date = 0,
        mime_time = 0;

    ASSERT (p_http);
    ASSERT (p_http-> table);
    ASSERT (p_http-> file_name);

    /*  In some configurations, defaults page caching is a real pain -
     *  so we allow the user to disable this.
     */
    if (p_http-> default_page
    && *HCONFIG ("server:cache_defaults") == '0')
        return;

    if (p_http-> send_file)
      {
        symptr = sym_lookup_symbol (p_http-> table, "if-modified-since");
        if (symptr)
          {
            /*  Get file date and time                                       */
            if (!p_http-> file_date || !p_http->file_time)
              {
                time_secs   = get_file_time (p_http-> file_name);
                p_http-> file_date = timer_to_gmdate (time_secs);
                p_http-> file_time = timer_to_gmtime (time_secs);
              }
            decode_mime_time (symptr-> value, &mime_date, &mime_time);
            if (p_http-> file_date < mime_date)
                p_http-> send_file = FALSE;
            else
            if (p_http-> file_date == mime_date)
              {
                if (p_http-> file_time <= mime_time )
                  p_http-> send_file = FALSE;
              }
          }
        if (!p_http-> send_file)
            p_http-> response = HTTP_RESPONSE_NOTMODIFIED;
      }
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_get_url_in_map

    Synopsis: Gets the URL from the image map file.  If a valid URL was
    found, returns TRUE, else returns FALSE.
    ---------------------------------------------------------------------[>]-*/

Bool
http_get_url_in_map (HTTP_CONTEXT *p_http)
{
    char
        *url_map;                       /*  URL for image map                */

    ASSERT (p_http);

    http_get_file_name (p_http);
    url_map = get_map_value (p_http-> file_name, p_http-> url_param);
    if (url_map)
      {
        mem_strfree (&p_http-> url);
        mem_strfree (&p_http-> url_param);
        mem_strfree (&p_http-> file_name);

        /*  Return 302 to browser so that it specifically asks for the       */
        /*  URL; this means that relative addressing will work from there    */
        /*  on.  Also, the data in the log file will be correct.             */
        format_full_url (p_http, buffer, url_map, NULL);
        p_http-> response = HTTP_RESPONSE_FOUND;
        p_http-> url      = mem_strdup (buffer);
        return (TRUE);
      }
    else
      {
        p_http-> response = HTTP_RESPONSE_NOTFOUND;
        return (FALSE);
      }
}


/*  -------------------------------------------------------------------------
    Function: get_map_value - Internal

    Synopsis: Returns a URL in function of a point in the map.  Uses a
    map file in NCSA format.  Returns a static pointer to a URL name if
    one was found, else returns NULL.  The point is supplied as a string
    formatted like this: "X,Y" - e.g. "120,12".  The map file must be
    fully specified.
    -------------------------------------------------------------------------*/

static char *
get_map_value (
    char *map_file,                     /*  Name of map file                 */
    char *request_point                 /*  Point coordinates                */
)
{
    static char
        url         [LINE_MAX],         /*  Buffer to store URL              */
        default_url [LINE_MAX];         /*  Default URL value for the map    */
    FILE
        *file;
    char
        *delim;                         /*  Find delimiter in path           */
    int
        point_x = 0,                    /*  Used only for sscanf             */
        point_y = 0;

    ASSERT (map_file);

    strclr (url);
    strclr (default_url);
    file = file_open (map_file, 'r');
    if (file)
      {
        /*  Get the coodinates of point                                      */
        if (request_point)
            sscanf (request_point, "%d,%d", &point_x, &point_y);

        /*  Note: The first line of map sets the value url_def               */
        while (file_read (file, buffer))
          {
            /*  Ignore comments and blank lines                              */
            if ((buffer [0] == '#') || strnull (buffer))
                continue;
            point_in_map_element (buffer, url, point_x, point_y, default_url);
            if (strused (url))
                break;
           }
        /*  If not found a URL, set the return with the default value        */
        if (strnull (url))
            strcpy (url, default_url);

        file_close (file);
      }

    /*  Return final URL string, or NULL if we did not find anything         */
    if (strused (url))
      {
        /*  If URL looks like a relative filename - i.e. does not start
         *  with a scheme name or directory specifier - we add the path
         *  of the map file.
         */
        if (!is_full_url (url) && url [0] != '/')
          {
            delim = strchr (map_file, '/');
            if (delim)
              {
                strcpy (buffer, delim);
                delim  = strrchr (buffer, '/') + 1;
                *delim = '\0';
              }
            else
                strcpy (buffer, "/");

            strcat (buffer, url);
            strcpy (url, buffer);
          }
        return (url);
      }
    else
        return (NULL);
}


/*  -------------------------------------------------------------------------
    Function: point_in_map_element - Internal

    Synopsis: Check if the requested point is in the element.
              The syntax of each map line is:
                  <Type of element> <URL> <x,y>...
              Each item must be separated by 1 or more spaces.
              Element is point, rectangle, circle, polygon, or default.
              Example:
              poly /test.htm 49,6 93,16 88,50
    -------------------------------------------------------------------------*/

static void
point_in_map_element (char *line, char *return_url, int point_x, int point_y,
                      char *default_url)
{
    char
        **tokens,                       /*  Line split into tokens           */
        *map_type,                      /*  Element type name                */
        *map_url,                       /*  Element URL value                */
        *comma;                         /*  Split x,y at comma               */
    FPOINT
        click_point,                    /*  Request point                    */
        shape_points [MAP_MAX_POINTS];  /*  Array of element point           */
    int
        token_nbr,                      /*  Index into token array           */
        nbr_points;                     /*  Number of points in element      */

    /*  Store selected point in image                                        */
    click_point.x = (double) point_x;
    click_point.y = (double) point_y;

    /*  Now parse the map file line into space-delimited tokens              */
    tokens = tok_split (line);

    /*  First token is image element type; second is the URL                 */
    map_type = strlwc (tokens [0]);
    map_url  = tokens [1];

    /*  Third to n tokens are x,y pairs...                                   */
    token_nbr = 2;
    for (nbr_points = 0; nbr_points < MAP_MAX_POINTS; nbr_points++)
      {
        if (!tokens [token_nbr])
            break;                      /*  Null token => finished           */
        comma = strchr (tokens [token_nbr], ',');
        if (!comma)
            break;                      /*  Syntax error in line => quit     */
        *comma = '\0';                  /*  Else break into separate x & y   */
        shape_points [nbr_points].x = atof (tokens [token_nbr]);
        shape_points [nbr_points].y = atof (comma + 1);
        token_nbr++;                    /*  Bump to next token               */
      }

    /*  Check element type and test the request point                      */
    if (streq (map_type, "poly"))
      {
        if (point_in_poly (&click_point, shape_points, nbr_points))
            strcpy (return_url, map_url);
      }
    else
    if (streq (map_type, "circle"))
      {
        if (point_in_circle (&click_point, shape_points))
            strcpy (return_url, map_url);
      }
    else
    if (streq (map_type, "rect"))
      {
        if (point_in_rect (&click_point, shape_points))
            strcpy (return_url, map_url);
      }
    else
    if (streq (map_type, "point"))
      {
        double dist;
        dist = ((click_point.x - shape_points [0].x)
             *  (click_point.x - shape_points [0].x))
             + ((click_point.y - shape_points [0].y)
             *  (click_point.y - shape_points [0].y));
       if (dist < 10)
          strcpy (return_url, map_url);
      }
    else
    if (streq (map_type, "default"))
        strcpy (default_url, map_url);

    tok_free (tokens);                  /*  Release memory                   */
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_list_directory

    Synopsis: Make a temporary html file with directory listing.
    ---------------------------------------------------------------------[>]-*/

Bool
http_list_directory (
    HTTP_CONTEXT *p_http)
{
    static char
        filedesc [32];                  /*  File size or type                */
    FILE
        *output;
    NODE
        *file_list;
    FILEINFO
        *file_info;
    char
        *format;                        /*  Format for each line             */

    /*  Load directory listing                                               */
    if ((file_list = load_dir_list (p_http-> file_name, "n")) == NULL)
        return (FALSE);

    /*  Open output file for HTML directory listing                          */
    sprintf (buffer, "./tmp%05ld.htm", p_http-> socket);
    output = file_open (buffer, 'w');
    if (output == NULL)
      {
        free_dir_list (file_list);
        return (FALSE);
      }

    /*  Register the new file name in the context block                      */
    mem_strfree (&p_http-> file_name);
    p_http-> file_name = mem_strdup (buffer);
    p_http-> temp_file = TRUE;

    fprintf (output,
    "<HTML><BODY><H2>File list of %s directory</H2><PRE>", p_http-> url);
    fprintf (output,
    "<B>Name                     Size         Last Modified<BR><HR><BR>\r\n");

    if (strlast (p_http-> url) == '/')
        format = "<A HREF=\"%s%s\">%s</A>";
    else
        format = "<A HREF=\"%s/%s\">%s</A>";

    for (file_info  = file_list-> next;
         file_info != (FILEINFO *) file_list;
         file_info  = file_info-> next)
      {
        sprintf (buffer, format, p_http-> url,
                                 file_info-> dir.file_name,
                                 file_info-> dir.file_name);

        strpad (buffer, ' ', strlen (p_http-> url)
                           + strlen (file_info-> dir.file_name) + 35);

        if (file_info-> directory)
            strcpy (filedesc, "Directory         ");
        else
            sprintf (filedesc, "  %7ld bytes   ",
                     (long) file_info-> dir.file_size);

        xstrcat (buffer, filedesc, encode_mime_time
                 (timer_to_date (file_info-> dir.file_time),
                  timer_to_time (file_info-> dir.file_time)),
                  "\r\n", NULL);
        fprintf (output, buffer);
      }
    file_write (output, "</PRE></BODY></HTML>");
    file_close (output);
    free_dir_list (file_list);

    /*  Register the new URL in the context block                            */
    sprintf (buffer, "/tmp%05ld.htm", p_http-> socket);
    mem_strfree (&p_http-> url);
    p_http-> url = mem_strdup (buffer);

    return (TRUE);
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_create_cgi_process

    Synopsis: Create cgi environment and process.  Returns 0 if process is
    created, else returns -1.
    ---------------------------------------------------------------------[>]-*/

int
http_create_cgi_process (HTTP_CONTEXT *p_http)
{
    static char
        build_filename [13],            /*  tempNNNN.cgi                     */
        cgi_stdout [13];                /*  tempNNNN.cgo                     */
    static int
        build_counter = 0;              /*  CGI number, 0 to 9999            */
    char
        *curdir,                        /*  Current directory                */
        *escaped,                       /*  CGI filename, escaped            */
        *command,                       /*  CGI command to execute           */
        *workdir;                       /*  CGI working directory            */
    int
        feedback = 0;
    FILE
        *cgi_stdin_stream;
    SYMTAB
        *symtab;

    /*  Unless we know better, we don't have a precise error cause           */
    p_http-> error_type = ERROR_UNSPECIFIED;

    /*  Make stdin and stdout filenames                                      */
    if (!p_http-> cgi_stdin)            /*  May already be created           */
      {
        sprintf (build_filename, "temp%04d.cgi", ++build_counter);
        if (build_counter > 9999)
            build_counter = 0;
        p_http-> cgi_stdin = mem_strdup (build_filename);

        cgi_stdin_stream = file_open (p_http-> cgi_stdin, 'w');
        if (cgi_stdin_stream)
          {
            /*  Write POSTed data if present                                 */
            if (p_http-> post_data)
                file_write (cgi_stdin_stream, p_http-> post_data);
            file_close (cgi_stdin_stream);
          }
      }
    strcpy (cgi_stdout, p_http-> cgi_stdin);
    strlast (cgi_stdout) = 'o';

    /*  Remove old stdout file if any                                        */
    if (file_exists (cgi_stdout))
        file_delete (cgi_stdout);

    /*  Initialise process parameters                                        */
    if ((symtab = add_cgi_environment (p_http)) == NULL)
      {
        mem_strfree (&p_http-> cgi_stdin);
        p_http-> error_type = ERROR_CGI_NORESOURCES;
        return (-1);                    /*  Insufficient memory              */
      }

    /*  Set CGI input/output filename symbols                                */
    curdir = get_curdir ();

    if (*HCONFIG ("cgi:msdos_style") == '1')
        sprintf (buffer, "%s\\%s", curdir, p_http-> cgi_stdin);
    else
        sprintf (buffer, "%s/%s", curdir, p_http-> cgi_stdin);
    sym_create_symbol (symtab, "Cgi-Stdin",  buffer);

    if (*HCONFIG ("cgi:msdos_style") == '1')
        sprintf (buffer, "%s\\%s", curdir, cgi_stdout);
    else
        sprintf (buffer, "%s/%s", curdir, cgi_stdout);

    sym_create_symbol (symtab, "Cgi-Stdout", buffer);
    sym_create_symbol (symtab, "Cgi-Stderr", HCONFIG ("cgi:errlog"));
    mem_free (curdir);

    p_http-> cgi_environ = symb2env (symtab);
    sym_delete_table (symtab);          /*  Get rid of environment table     */
    if (p_http-> cgi_environ == NULL)
      {
        mem_strfree (&p_http-> cgi_stdin);
        p_http-> error_type = ERROR_CGI_NORESOURCES;
        return (-1);                    /*  Insufficient memory              */
      }

    /*  Create the process                                                   */
    process_delay = 0;                  /*  Run process and return at once   */
    escaped = process_esc (NULL, p_http-> file_name);
    command = xstrcpy (NULL, escaped, " ", p_http-> url_param, NULL);
    workdir = HCONFIG ("cgi:workdir");
    if (streq (workdir, "-"))           /*  '-' means use script directory   */
        workdir = p_http-> script_path;

    /*  Set sflproc flag for separate virtual machines                       */
    /*  This only affects 32-bit Windows systems                             */
    process_compatible = *HCONFIG ("win32:16bit_cgi") == '1';

    if (*HCONFIG ("cgi:stdio") == '1')
        p_http-> cgi_process_id = process_create (
            command,                    /*  Command to run                   */
            NULL,                       /*  Arguments for command            */
            workdir,                    /*  Working directory                */
            p_http-> cgi_stdin,         /*  Redirect stdin from here         */
            cgi_stdout,                 /*  Redirect stdout to here          */
            HCONFIG ("cgi:errlog"),     /*  Redirect stderr to here          */
            p_http-> cgi_environ,       /*  Environment table                */
            FALSE);                     /*  Wait until finished?             */
    else
        p_http-> cgi_process_id = process_create (
            command,                    /*  Command to run                   */
            NULL,                       /*  Arguments for command            */
            workdir,                    /*  Working directory                */
            NULL,                       /*  Redirect stdin from here         */
            NULL,                       /*  Redirect stdout to here          */
            NULL,                       /*  Redirect stderr to here          */
            p_http-> cgi_environ,       /*  Environment table                */
            FALSE);                     /*  Wait until finished?             */

    mem_free (command);
    mem_free (escaped);

    if (p_http-> cgi_process_id != NULL_PROCESS)
      {
        mem_strfree (&p_http-> file_name);
        p_http-> file_name = mem_strdup (cgi_stdout);
        p_http-> temp_file = TRUE;
      }
    else
      {
        p_http-> error_type = ERROR_CGI_PROCESSNF;
        feedback = -1;                  /*  Could not create the process     */
      }
    return (feedback);
}


/*  -------------------------------------------------------------------------
    Function: add_cgi_environment - Internal

    Synopsis: Loads CGI environment variables into a symbol table, and
    returns that symbol table.  Returns NULL if there was a problem, e.g.
    insufficient free memory.  Creates these symbols:

      - Unless cgi:environment=0, loads process environment.
      - Loads all values under [CGI-Environment] section
      - For each HTTP header field, inserts a symbol called HTTP_xxxx
        where xxxx is the header field name.  The field names are put
        into uppercase, and hyphens replaced by underlines.  This is
        controlled by the cgi:http_fields configuration option.
      - For each variable in the query string, inserts a symbol called
        FORM_xxxx where xxxx is the variable name.
      - SERVER_SOFTWARE is the name of the web server (the global variable
        server_name).
      - SERVER_NAME is the current host name.
      - SERVER_PORT is the port on which the web server is running.
      - SERVER_PROTOCOL is the HTTP version in use (currently "HTTP/1.0").
      - GATEWAY_INTERFACE is the CGI version in use (currently "CGI/1.1").
      - REQUEST_METHOD is the HTTP method used, either "GET" or "POST".
      - CONTENT_TYPE is the Content-Type: HTTP field.
      - CONTENT_LENGTH is the Content-Length: HTTP field.
      - REMOTE_USER is the authorised username, if any, else "-".
      - REMOTE_HOST is the remote host name, equal to REMOTE_ADDR.
      - REMOTE_ADDR is the remote host address, "xxx.xxx.xxx.xxx".
      - SCRIPT_PATH is the path of the script being executed.
      - SCRIPT_NAME is the URI of the script being executed.
      - QUERY_STRING is the GET query string, or POST data.
      - PATH_INFO is any path data following the CGI program name
      - PATH_TRANSLATED is the full translated URI path including PATH_INFO
      - CGI_ROOT is the CGI binary directory
      - CGI_URL is the server:cgi_url value
      - DOCUMENT_ROOT is the main webpages directory
      - Caller also adds CGI_STDOUT, CGI_STDIN, CGI_STDERR.

    The calling function must delete the symbol table when finished with it.
    -------------------------------------------------------------------------*/

#define SET(name,value) \
    sym_assume_symbol (symtab, (name), (value)? value: "")

static SYMTAB *
add_cgi_environment (HTTP_CONTEXT *p_http)
{
    static char
        port_name [20];                 /*  Enough for a 64-bit system       */
    SYMBOL
        *symbol;                        /*  Next symbol in table             */
    SYMTAB
        *symtab;                        /*  Symbol table we create           */
    char
        *work_name,                     /*  Working file name                */
        *path_translated;               /*  PATH_TRANSLATED symbol           */

    ASSERT (p_http);
    ASSERT (p_http-> table);

    /*  Initial symbol table is either from server environment, or empty     */
    if (*HCONFIG ("cgi:environment") == '1')
        symtab = env2symb ();
    else
        symtab = sym_create_table ();

    if (symtab == NULL)
        return (NULL);                  /*  Insufficient memory              */

    /*  Load all symbols from CGI-Environment section                        */
    for (symbol = p_http-> config-> symbols; symbol; symbol = symbol-> next)
        if (strprefixed (symbol-> name, "cgi_environment:"))
            sym_create_symbol (symtab,
                symbol-> name + strlen ("cgi_environment:"),
                symbol-> value);

    /*  Load HTTP header fields into the temporary symbol table.  This is
     *  only done if cgi:http_fields is 1.  Header fields are prefixed by
     *  cgi:http_prefix.
     */
    if (*HCONFIG ("cgi:http_fields") == '1')
        for (symbol = p_http-> table-> symbols; symbol; symbol = symbol-> next)
          {
            xstrcpy (buffer, HCONFIG ("cgi:http_prefix"), symbol-> name, NULL);
            sym_create_symbol (symtab, buffer, symbol-> value);
          }

    /*  Load CGI query into symbol table, if cgi:form_fields is 1.  The
     *  query fields are also passed on stdin, and may be too large to put
     *  into the environment.  Query fields are prefixed by the value of
     *  cgi:form_prefix.
     */
    if (p_http-> post_data
    && *HCONFIG ("cgi:form_fields") == '1')
        cgi_parse_query_vars (symtab, p_http-> post_data,
                                      HCONFIG ("cgi:form_prefix"));
    /*  Format port number as a string                                       */
    sprintf (port_name, "%d", p_http-> port);

    /*  Format PATH_TRANSLATED symbol
     *  This takes the path_info data and prefixes the document root
     */
    work_name = file_where ('r', NULL, p_http-> rootdir, "");
    if (work_name)
        work_name = resolve_path (work_name);
    path_translated = xstrcpy (NULL, work_name, p_http-> path_info, NULL);
    if (*HCONFIG ("cgi:msdos_style") == '1')
      {
        strconvch (path_translated,      '/', '\\');
        strconvch (p_http-> script_path, '/', '\\');
        strconvch (p_http-> script_name, '/', '\\');
      }
    SET ("Server-Software",   server_name);
    SET ("Server-Name",       HCONFIG ("server:hostname"));
    SET ("Server-Port",       port_name);
    SET ("Server-Protocol",   "HTTP/1.0");
    SET ("Gateway-Interface", "CGI/1.1");
    SET ("Request-Method",    methods [p_http-> method - 1]);
    SET ("Script-Path",       p_http-> script_path);
    SET ("Script-Name",       p_http-> script_name);
    SET ("Content-Type",      http_header_value (p_http, "content-type"));
    SET ("Content-Length",    http_header_value (p_http, "content-length"));
    SET ("Remote-User",       p_http-> username? p_http-> username: "-");
    SET ("Remote-Host",       socket_peeraddr (p_http-> socket));
    SET ("Remote-Addr",       socket_peeraddr (p_http-> socket));
    SET ("Path-Info",         p_http-> path_info);
    SET ("Path-Translated",   path_translated);
    SET ("Cgi-Root",          p_http-> cgidir);
    SET ("Cgi-Url",           HCONFIG ("server:cgi_url"));
    SET ("Document-Root",     p_http-> rootdir);

    /*  If the url_param is set, we put that into Query-String, otherwise we
     *  put-in the post data; this is switched-off by default, since it's not
     *  standard CGI, and can do strange things if the post data is large.
     */
    if (p_http-> url_param)
        SET ("Query-String",  p_http-> url_param);
    else
    if (p_http-> post_data
    && *HCONFIG ("cgi:form_query") == '1')
        SET ("Query-String",  p_http-> post_data);

    mem_free (path_translated);
    return (symtab);
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_create_filter_process

    Synopsis: Create filter environment and process.  Returns 0 if process
    is created, else returns -1.
    ---------------------------------------------------------------------[>]-*/

int
http_create_filter_process (HTTP_CONTEXT *p_http)
{
    static char
        filter_stdout [13];             /*  tempNNNN.flt                     */
    static int
        build_counter = 0;              /*  Temporary file number, 0-9999    */
    char
        *command;                       /*  Command to execute               */
    int
        feedback = 0;
    SYMTAB
        *symtab;

    /*  Unless we know better, we don't have a precise error cause           */
    p_http-> error_type = ERROR_UNSPECIFIED;

    /*  Make stdout filename                                                 */
    sprintf (filter_stdout, "temp%04d.flt", ++build_counter);
    if (build_counter > 9999)
        build_counter = 0;

    /*  Remove old stdout file if any                                        */
    if (file_exists (filter_stdout))
        file_delete (filter_stdout);

    /*  Build command to execute, then script_name is URI                    */
    command = xstrcpy (NULL, p_http-> script_name,
                        " ", p_http-> file_name, NULL);
    mem_strfree (&p_http-> script_name);
    p_http-> script_name = mem_strdup (p_http-> url);
    /*  Under Windows, some programs may expect paths to contain \ not /     */
    if (*HCONFIG ("cgi:msdos_style") == '1')
        strconvch (command, '/', '\\');

    /*  Initialise process parameters                                        */
    if ((symtab = add_cgi_environment (p_http)) == NULL)
      {
        mem_strfree (&p_http-> cgi_stdin);
        p_http-> error_type = ERROR_CGI_NORESOURCES;
        return (-1);                    /*  Insufficient memory              */
      }

    /*  Certain filter programs (e.g. PHP/FI) expect the HTML file name to
     *  be supplied in the PATH_INFO and PATH_TRANSLATED variables.  God
     *  knows why.  Anyhow, we do this, and furthermore convert it to a
     *  DOSish name if needed.
     */
    strcpy (buffer, p_http-> file_name);
    if (*HCONFIG ("cgi:msdos_style") == '1')
        strconvch (buffer, '/', '\\');
    SET ("Path-Info",       buffer);
    SET ("Path-Translated", buffer);

    p_http-> cgi_environ = symb2env (symtab);
    sym_delete_table (symtab);          /*  Get rid of environment table     */
    if (p_http-> cgi_environ == NULL)
      {
        mem_strfree (&p_http-> cgi_stdin);
        p_http-> error_type = ERROR_CGI_NORESOURCES;
        return (-1);                    /*  Insufficient memory              */
      }

    /*  Set sflproc flag for separate virtual machines                       */
    /*  This only affects 32-bit Windows systems                             */
    process_compatible = *HCONFIG ("win32:16bit_cgi") == '1';
    process_delay      = 0;             /*  Run process and return at once   */

    /*  Create the process                                                   */
    p_http-> cgi_process_id = process_create (
        command,                        /*  Command to run                   */
        NULL,                           /*  Arguments for command            */
        NULL,                           /*  Working directory                */
        p_http-> file_name,             /*  Redirect stdin from here         */
        filter_stdout,                  /*  Redirect stdout to here          */
        HCONFIG ("cgi:errlog"),         /*  Redirect stderr to here          */
        p_http-> cgi_environ,           /*  Environment table                */
        FALSE);                         /*  Wait until finished?             */

    mem_free (command);

    if (p_http-> cgi_process_id != NULL_PROCESS)
      {
        mem_strfree (&p_http-> file_name);
        p_http-> file_name = mem_strdup (filter_stdout);
        p_http-> temp_file = TRUE;
      }
    else
      {
        p_http-> error_type = ERROR_CGI_PROCESSNF;
        feedback = -1;                  /*  Could not create the process     */
      }
    return (feedback);
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_process_state

    Synopsis: Get the cgi process state; returns one of these values:
    <Table>
    CGI_STATE_RUNNING  Process is in progess
    CGI_STATE_END      Process ended OK
    CGI_STATE_ERROR    Process ended with error
    </Table>
    ---------------------------------------------------------------------[>]-*/

int
http_process_state (HTTP_CONTEXT *p_http)
{
    int
        feedback;

    ASSERT (p_http);

    if (p_http-> cgi_process_id == NULL_PROCESS)
      {
        feedback = CGI_STATE_ERROR;
        p_http-> error_type = ERROR_CGI_INTERNALERROR;
      }
    else
        switch (process_status (p_http-> cgi_process_id))
          {
            case PROCESS_RUNNING:
                feedback = CGI_STATE_RUNNING;
                break;
            case PROCESS_ENDED_OK:
                feedback = CGI_STATE_END;
                break;
            case PROCESS_ENDED_ERROR:
                if (*HCONFIG ("cgi:exit_ok") == '1')
                  {
                    feedback = CGI_STATE_ERROR;
                    p_http-> error_type = ERROR_CGI_PROCESSERROR;
                  }
                else                    /*  If exit_ok=0, ignore exit code   */
                    feedback = CGI_STATE_END;
                break;
            case PROCESS_INTERRUPTED:
                feedback = CGI_STATE_ERROR;
                p_http-> error_type = ERROR_CGI_INTERRUPTED;
                break;
            default:
                feedback = CGI_STATE_ERROR;
                p_http-> error_type = ERROR_CGI_INTERNALERROR;
          }
    return (feedback);
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_close_process

    Synopsis: Terminates the currently-running CGI process; if the process
    already ended normally, just cleans-up.
    ---------------------------------------------------------------------[>]-*/

void
http_close_process (HTTP_CONTEXT *p_http)
{
    ASSERT (p_http);

    if (p_http-> cgi_process_id != NULL_PROCESS)
      {
        if (process_status (p_http-> cgi_process_id) == PROCESS_RUNNING)
            process_kill  (p_http-> cgi_process_id);
        else
            process_close (p_http-> cgi_process_id);

        p_http-> cgi_process_id = NULL_PROCESS;
      }
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_reparse_output

    Synopsis: Scans the CGI output file and executes the server directives
    in the file.  Accepts the standard CGI directives:

        Location: filename
        Location: URL
        Content-Type: what/type
        Status: code

    Ignores the first line if it starts with 'HTTP/'.  If the first line
    does not look like a server directive, the output is returned as type
    text/html.  Returns 1 if the CGI output refers to a new URL, else
    returns 0.
    ---------------------------------------------------------------------[>]-*/

int
http_reparse_output (HTTP_CONTEXT *p_http)
{
    FILE
        *response;                      /*  CGI response file                */
    char
        *separator;                     /*  ':' character in header          */
    int
        feedback = 0;

    ASSERT (p_http);

    /*  Create a new, empty table for the mime values                        */
    http_close_cgi (p_http);
    p_http-> table = sym_create_table ();

    /*  By default we send the full CGI output, as-is                        */
    p_http-> file_offset = 0;
    p_http-> file_size   = get_file_size (p_http-> file_name);
    p_http-> cgi_header  = FALSE;       /*  Assume normal CGI output         */

    /*  Read the output from the CGI program                                 */
    response = file_open (p_http-> file_name, 'r');
    if (response == NULL)               /*  File not accessible?             */
      {
        trace ("Cannot reparse %s: %s", p_http-> file_name, strerror (errno));
        return (0);
      }
    sym_assume_symbol (p_http-> table, "content-type", "text/html");
    while (file_read (response, buffer))
      {
        /*  Header ends with a blank line or a line starting with <HTML>     */
        if (lexncmp (buffer, "<html>", 6) == 0 || *buffer == '\0')
            break;

        separator = strchr (buffer, ':');
        if (separator == NULL)          /*  And so does a non-header line    */
          {                             /*    in which case we send it all   */
            fseek (response, 0, SEEK_SET);
            p_http-> cgi_header = strprefixed (buffer, "HTTP/");
            break;
          }

        /*  Parse the header and place into the symbol table                 */
        *separator = '\0';              /*  Split at separator               */
        separator++;                    /*    and skip space after ':'       */
        while (*separator == ' ')
            separator++;
        strlwc (buffer);                /*  Lowercase the header name        */

        if (streq (buffer, "location"))
          {
            mem_strfree (&p_http-> cgi_status);
            mem_strfree (&p_http-> url);
            p_http-> url = mem_strdup (separator);

            /*  If 'Location:' is a full URL, return 302 + url to client     */
            if (is_full_url (separator))
                p_http-> response = HTTP_RESPONSE_FOUND;
            else
            if (*separator == '/')      /*  Local URL                        */
              {
                /*  If URL contains '?', split into URL + params             */
                separator = strchr (p_http-> url, '?');
                if (separator)
                  {
                    mem_strfree (&p_http-> url_param);
                    *separator++ = '\0';
                    p_http-> url_param = mem_strdup (separator);
                  }
                feedback = 1;           /*  We need to cycle the dialog...   */
              }
            else                        /*  Relative URLs are not allowed    */
                p_http-> response = HTTP_RESPONSE_NOTFOUND;
          }
        else
        if (streq (buffer, "status"))
          {
            mem_strfree (&p_http-> cgi_status);
            p_http-> cgi_status = mem_strdup (separator);
          }
        else
            sym_assume_symbol (p_http-> table, buffer, separator);
      }
    /*  We transmit the file from the current position onwards               */
    if (feedback == 0)
      {
        p_http-> file_offset = ftell (response);
        p_http-> file_size  -= p_http-> file_offset;
      }
    file_close (response);

    /*  If CGI is redirected, release file_name to refer to new URL          */
    if (feedback == 1)
      {
        file_delete (p_http-> file_name);
        mem_strfree (&p_http-> file_name);
        p_http-> temp_file = FALSE;
        p_http-> file_size = 0;
      }
    return (feedback);
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_reparse_buffer

    Synopsis: Scans a HTTP output buffer and executes the server directives
    in the file.  Accepts the standard CGI directives:

        Location: filename
        Location: URL
        Content-Type: what/type
        Status: code

    Ignores the first line if it starts with 'HTTP/'.  If the first line
    does not look like a server directive, the output is returned as type
    text/html.  Returns 1 if the HTTP buffer refers to a new URL, else
    returns 0.  If the buffer contained HTTP headers, removes these from
    the buffer.  Sets file_size to the resulting buffer size.
    ---------------------------------------------------------------------[>]-*/

int
http_reparse_buffer (HTTP_CONTEXT *p_http, char *body)
{
    char
        *this_line,                     /*  Start of line in buffer          */
        *eoln,                          /*  And end of line                  */
        *separator;                     /*  ':' character in header          */
    int
        line_size,
        feedback = 0;

    ASSERT (p_http);

    /*  Create a new, empty table for the mime values                        */
    http_close_cgi (p_http);
    p_http-> table = sym_create_table ();

    /*  By default we send the full body, as-is                              */
    p_http-> file_offset = 0;
    p_http-> file_size   = strlen (body);
    p_http-> cgi_header  = FALSE;       /*  Assume normal HTTP output        */

    sym_assume_symbol (p_http-> table, "content-type", "text/html");

    this_line = body;
    while ((eoln = find_eoln (this_line)) != NULL)
      {
        /*  Line starting with <HTML> indicates HTML body                    */
        if (lexncmp (this_line, "<html>", 6) == 0)
            break;

        line_size = min (HTTP_URL_MAX, (eoln - this_line));
        memcpy (buffer, this_line, line_size);
        buffer [line_size] = '\0';

        if (*buffer == '\0')            /*  Blank line ends header           */
          {
            this_line++;                /*  Find start of next line          */
            if (*this_line == '\n')     /*    skipping CRLF as needed        */
                this_line++;
            break;
          }
        separator = strchr (buffer, ':');
        if (separator == NULL)          /*  And so does a non-header line    */
          {                             /*    in which case we send it all   */
            this_line = body;
            p_http-> cgi_header = strprefixed (body, "HTTP/");
            break;
          }

        /*  Parse the header and place into the symbol table                 */
        *separator = '\0';              /*  Split at separator               */
        separator++;                    /*    and skip space after ':'       */
        while (*separator == ' ')
            separator++;
        strlwc (buffer);                /*  Lowercase the header name        */

        if (streq (buffer, "location"))
          {
            mem_strfree (&p_http-> cgi_status);
            mem_strfree (&p_http-> url);
            p_http-> url = mem_strdup (separator);

            /*  If 'Location:' is a full URL, return 302 + url to client     */
            if (is_full_url (separator))
                p_http-> response = HTTP_RESPONSE_FOUND;
            else
            if (*separator == '/')      /*  Local URL                        */
              {
                /*  If URL contains '?', split into URL + params             */
                separator = strchr (p_http-> url, '?');
                if (separator)
                  {
                    mem_strfree (&p_http-> url_param);
                    *separator++ = '\0';
                    p_http-> url_param = mem_strdup (separator);
                  }
                feedback = 1;           /*  We need to cycle the dialog...   */
              }
            else                        /*  Relative URLs are not allowed    */
                p_http-> response = HTTP_RESPONSE_NOTFOUND;
          }
        else
        if (streq (buffer, "status"))
          {
            mem_strfree (&p_http-> cgi_status);
            p_http-> cgi_status = mem_strdup (separator);
          }
        else
            sym_assume_symbol (p_http-> table, buffer, separator);

        this_line = eoln + 1;           /*  Bump up to the next line         */
        if (*this_line == '\n')         /*    skipping CRLF as needed        */
            this_line++;
      }
    /*  We send the body from the current position onwards                   */
    if (feedback == 0 && this_line > body)
      {
        p_http-> file_size -= (this_line - body);
        memcpy (body, this_line, p_http-> file_size);
        body [p_http-> file_size] = 0;
      }

    /*  If CGI is redirected, release file_name to refer to new URL          */
    if (feedback == 1)
      {
        file_delete (p_http-> file_name);
        mem_strfree (&p_http-> file_name);
        p_http-> temp_file = FALSE;
        p_http-> file_size = 0;
      }
    return (feedback);
}


/*  ---------------------------------------------------------------------[<]-
    Function: find_eoln -- internal

    Synopsis: Find the end of the line held in buffer.  The line ends in
    either LF or CRLF.  If no end-of-line is found, returns NULL, else
    returns the address of the first character of the end-of-line-sequence.
    ---------------------------------------------------------------------[>]-*/

static char *
find_eoln (char *buffer)
{
    char
        *eoln;

    eoln = strchr (buffer, '\n');       /*  Require a LF                     */
    if (eoln)
      {
        if (eoln > buffer
        &&  eoln [-1] == '\r')
            eoln--;                     /*    and possible a CR              */
      }
    return (eoln);
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_capture_console

    Synopsis: Prepares console capturing according to the current value of
    console:capture and console:filename.
    ---------------------------------------------------------------------[>]-*/

void
http_capture_console (void)
{
    char
        mode;

    if (*http_config (config, "console:capture") == '1')
      {
        if (*http_config (config, "console:append") == '1')
            mode = 'a';
        else
            mode = 'w';

        console_capture (http_config (config, "console:filename"), mode);
      }
    else
        console_capture (NULL, 0);
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_uri_protected

    Synopsis:
    Looks for the URI in the authorisation table.  The authorisation table
    always specifies a full URI or directory name.  Searches for the most
    specific URI first, by successively removing path segments from the
    right-hand side.  If asked to look for "/pub/mypages/file", will look
    for entries in this order:

        [pub/mypages/file]
        [pub/mypages]
        [pub]
        [/]

    Returns TRUE if the directory is defined in the authorisation table,
    and modifies the supplied url to match the entry keyword. The specified
    uri MUST start with '/'.
    ---------------------------------------------------------------------[>]-*/

Bool
http_uri_protected (SYMTAB *passwd, char *uri)
{
    char
        *slash;                         /*  Position of last '/' in name     */

    ASSERT (passwd);
    ASSERT (uri);
    ASSERT (*uri == '/');

    strlwc    (uri);                    /*  All searches in lowercase        */
    strconvch (uri, '-', '_');          /*    and underlines for hyphens     */
    FOREVER
      {
        if (sym_lookup_symbol (passwd, uri))
            return (TRUE);              /*  URI was found                    */
        else
        if (*uri == '/'                 /*  Remove leading slash if any      */
        &&  sym_lookup_symbol (passwd, uri + 1))
          {
            memmove (uri, uri + 1, strlen (uri));
            return (TRUE);
          }
        /*  Cut at slash, if any                                             */
        slash = strrchr (uri, '/');
        if (slash)
            *slash = '\0';
        else
          {
            strcpy (uri, "/");
            if (sym_lookup_symbol (passwd, uri))
                return (TRUE);          /*  "/" was found                    */
            else
                return (FALSE);         /*  Not found                        */
          }
      }
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_password_okay

    Synopsis: Returns TRUE if the response matches the challenge.  If the
    challenge is "" or "-", the response is never accepted.  Challenge and
    response may either or both be null.
    ---------------------------------------------------------------------[>]-*/

Bool
http_password_okay (char *challenge, char *response)
{
    if (!challenge || !response)
        return (FALSE);                 /*  Either argument not supplied     */
    else
    if (streq (challenge, "-") || strnull (challenge))
        return (FALSE);                 /*  Challenge is disabled or empty   */
    else
        return (streq (challenge, response));
}
