/*  ----------------------------------------------------------------<Prolog>-
    Name:       smthttp.c
    Title:      Hyper Text Transfer Protocol Agent
    Package:    Libero SMT 2.x

    Written:    96/07/22  iMatix SMT kernel team <smt@imatix.com>
    Revised:    98/01/20

    Copyright:  Copyright (c) 1991-98 iMatix
    License:    This is free software; you can redistribute it and/or modify
                it under the terms of the SMT License Agreement as provided
                in the file LICENSE.TXT.  This software is distributed in
                the hope that it will be useful, but without any warranty.
 ------------------------------------------------------------------</Prolog>-*/

#include "smtdefn.h"                    /*  SMT definitions                  */
#include "smthttpl.h"                   /*  HTTP library functions           */
#include "smtmsg.h"                     /*  SMT message functions            */


/*- Definitions -------------------------------------------------------------*/

#undef  AGENT_NAME
#define AGENT_NAME     SMT_HTTP         /*  Our public name                  */
#define BUFFER_SIZE    HTTP_HEADER_MAX  /*  Max. incoming HTTP header        */
#define TCONFIG(s)     http_config (tcb-> config, (s))


typedef struct {                        /*  Thread context block:            */
    HTTP_STAT
        stats;                          /*    Statistics structure           */
    sock_t
        handle;                         /*    Handle for i/o                 */
    int
        readsize;                       /*    Size of input message so far   */
    long
        post_size,                      /*    Size of POSTed data, if any    */
        post_read;                      /*    Amount of POSTed data read     */
    int
        portbase;                       /*    Index for portbase attempts    */
    Bool
        cgi_request;                    /*    Request for a CGI resource     */
    char
        *buffer,                        /*    Buffer for i/o                 */
        *http_body;                     /*    HTTP body, if any              */
    event_t
        thread_type;                    /*    Thread type indicator          */
    HTTP_CONTEXT
        http;                           /*    HTTP context                   */
    word
        input_timeout,                  /*    Timeout for input and output   */
        output_timeout;                 /*      waits on socket              */
    SYMTAB
        *config,                        /*    Configuration for host         */
        *passwd;                        /*    User authentication data       */
    QID
        alogq,                          /*    Access log queue               */
        elogq;                          /*    Error log queue                */
    FILE
        *post_file;                     /*    File captures POSTed data      */
} TCB;


typedef struct {                        /*  Virtual host resources           */
    SYMTAB                              /*                                   */
        *config,                        /*    Configuration for host         */
        *passwd;                        /*    User authentication data       */
    char
        *name,                          /*    Virtual host name              */
        *alogfile,                      /*    Access log filename            */
        *elogfile,                      /*    Error log filename             */
        *rootdir,                       /*    Web pages root directory       */
        *cgidir;                        /*    CGI root directory             */
    Bool
        alog_owner,                     /*    This VH created access log?    */
        elog_owner;                     /*    This VH created error log?     */
    QID
        alogq,                          /*    Access log queue               */
        elogq;                          /*    Error log queue                */
} VHOST;


/*- Function prototypes -----------------------------------------------------*/

static void    set_timer_refresh   (THREAD *thread);
static int     cycle_mode          (const char *key, Bool startup);
static void    load_config_file    (SYMTAB *table, char *filename);
static SYMTAB *load_password_file  (const char *filename);
static void    set_vhost_name      (SYMTAB *table, const char *name);
static void    create_log_threads  (VHOST *vhost);
static void    check_socket_rc     (int rc);
static void    close_post_file     (void);
static void    log_http_access     (THREAD *thread);
static char   *time_str            (void);
static void    count_transfer      (long size);
static char   *make_wsx_path       (char *virtual_host, char *path);


/*- Global variables used in this source file only --------------------------*/

static AGENT
    *this_agent;                        /*  Handle to ourselves              */
static THREAD
    *clog_thread;                       /*  Console logging agent thread     */
static TCB
    *tcb;                               /*  Address thread contect block     */
static QID
    operq,                              /*  Operator console event queue     */
    clogq,                              /*  Server log queue                 */
    sockq,                              /*  Socket agent event queue         */
    timeq,                              /*  Timer agent event queue          */
    tranq,                              /*  Transfer agent event queue       */
    mainq;                              /*  Master thread queue              */
static char
    *server_log = NULL,                 /*  Main server log file name        */
    *rootdir,                           /*  Root directory for documents     */
    *cgidir,                            /*  Directory for CGI programs       */
    *master_port;                       /*  Port for http service            */

static Bool
    server_logging;                     /*  True if server logging enabled   */
static SYMTAB
    *vhosts = NULL;                     /*  Virtual hosts table              */


/*  The largest message we send to another agent is when we transmit a       */
/*  HTTP header to the socket agent.  The header is at most BUFFER_SIZE      */
/*  bytes, and we add a little for the other fields in the event:            */

#define MSG_MAX     BUFFER_SIZE + 64

static byte
    msg_body [MSG_MAX];                 /*  Messages sent to other agents    */
static int
    msg_size;                           /*  Size of formatted msg_body       */
static DESCR                            /*  Descriptor for exdr_writed()     */
    msg = { MSG_MAX,  msg_body };


#include "smthttp.d"                    /*  Include dialog data              */

/********************   INITIALISE AGENT - ENTRY POINT   ********************/

/*  ---------------------------------------------------------------------[<]-
    Function: smthttp_init

    Synopsis: Initialises the HTTP agent.  Returns 0 if initialised okay,
    -1 if there was an error.  The HTTP agent manages a HTTP service on
    port 80 (+ ip_portbase).  Sends errors and information messages to the
    SMTOPER agent.
    ---------------------------------------------------------------------[>]-*/

int smthttp_init (char *p_rootdir,      /*  Document root directory          */
                  char *p_cgidir)       /*  CGI binary directory             */
{
    AGENT  *agent;                      /*  Handle for our agent             */
    THREAD *thread;                     /*  Handle to various threads        */
#   include "smthttp.i"                 /*  Include dialog interpreter       */

    /*                      Method name      Event value     Priority        */
    /*  Shutdown event comes from Kernel                                     */
    method_declare (agent, "SHUTDOWN",       shutdown_event, SMT_PRIORITY_MAX);

    /*  Reply events from socket agent                                       */
    method_declare (agent, "SOCK_INPUT_OK",  ok_event,            0);
    method_declare (agent, "SOCK_OUTPUT_OK", ok_event,            0);
    method_declare (agent, "SOCK_READ_OK",   ok_event,            0);
    method_declare (agent, "SOCK_WRITE_OK",  ok_event,            0);
    method_declare (agent, "SOCK_CLOSED",    sock_closed_event,   0);
    method_declare (agent, "SOCK_ERROR",     sock_error_event,    0);
    method_declare (agent, "SOCK_TIMEOUT",   sock_timeout_event,  0);

    /*  Reply events from transfer agent                                     */
    method_declare (agent, "TRAN_PUT_OK",    finished_event,      0);
    method_declare (agent, "TRAN_CLOSED",    sock_closed_event,   0);
    method_declare (agent, "TRAN_ERROR",     sock_error_event,    0);

    /*  Registration events from WSX agents                                  */
    method_declare (agent, "WSX_INSTALL",    wsx_install_event,   0);
    method_declare (agent, "WSX_CANCEL",     wsx_cancel_event,    0);

    /*  Reply events from WSX agents                                         */
    method_declare (agent, "WSX_OK",         ok_event,            0);
    method_declare (agent, "WSX_ERROR",      error_event,         0);
    method_declare (agent, "WSX_REDIRECT",   redirect_event,      0);
    method_declare (agent, "WSX_RESTART",    restart_event,       0);
    method_declare (agent, "WSX_KILL",       kill_event,          0);

    /*  Reply events from timer agent                                        */
    method_declare (agent, "TIME_ALARM",     timer_event, SMT_PRIORITY_LOW);

    /*  Private methods used to pass initial thread events                   */
    method_declare (agent, "_MASTER",        master_event,        0);
    method_declare (agent, "_CLIENT",        client_event,        0);

    /*  Private methods used between threads                                 */
    method_declare (agent, "_CANCEL",        cancel_event,        0);
    method_declare (agent, "_RESTART",       restart_event,       0);

    /*  Ensure that logging agent is running, else start it up               */
    smtlog_init ();

    /*  Ensure that operator console is running, else start it up            */
    smtoper_init ();
    if ((thread = thread_lookup (SMT_OPERATOR, "")) != NULL)
        operq = thread-> queue-> qid;
    else
        return (-1);

    /*  Ensure that socket i/o agent is running, else start it up            */
    smtsock_init ();
    if ((thread = thread_lookup (SMT_SOCKET, "")) != NULL)
        sockq = thread-> queue-> qid;
    else
        return (-1);

    /*  Ensure that timer agent is running, else start it up                 */
    smttime_init ();
    if ((thread = thread_lookup (SMT_TIMER, "")) != NULL)
        timeq = thread-> queue-> qid;
    else
        return (-1);

    /*  Ensure that transfer agent is running, else start it up              */
    smttran_init ();
    if ((thread = thread_lookup (SMT_TRANSFER, "")) != NULL)
        tranq = thread-> queue-> qid;
    else
        return (-1);

    /*  Create initial thread to manage master port                          */
    if ((thread = thread_create (AGENT_NAME, "main")) != NULL)
      {
        SEND (&thread-> queue-> qid, "_MASTER", "");
        ((TCB *) thread-> tcb)-> thread_type = master_event;
        ((TCB *) thread-> tcb)-> handle      = 0;
      }
    else
        return (-1);

    this_agent = agent;                 /*  Handle to ourselves              */

    /*  Use standard port number.  Caller can set ip_portbase if wanted      */
    /*  to place the port into a specific area.                              */
    master_port = SMT_HTTP_PORT;

    http_init ();                       /*  Initialise HTTP library          */

    /*  Get root and cgi directories, clean-up, and store.  Note that we     */
    /*  remove any trailing '/' from the directory name.                     */
    rootdir = mem_strdup (resolve_path (p_rootdir));
    cgidir  = mem_strdup (resolve_path (p_cgidir));
    if (*rootdir && strlast (rootdir) == '/')
        strlast (rootdir) = '\0';
    if (*cgidir &&  strlast (cgidir) == '/')
        strlast (cgidir)  = '\0';

    /*  Signal okay to caller that we initialised okay                       */
    return (0);
}


/*************************   INITIALISE THE THREAD   *************************/

MODULE initialise_the_thread (THREAD *thread)
{
    long
        max_threads;

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> buffer    = NULL;
    tcb-> portbase  = 0;                /*  Not serving any port             */
    tcb-> http_body = NULL;             /*  No HTTP body                     */
    tcb-> config    = config;           /*  Start with global config file    */
    tcb-> post_file = NULL;             /*  No POST file capture open        */
    tcb-> http.config = tcb-> config;
    http_init_context (&tcb-> http);

    /*  Take timeouts from HTTP context definition if child thread           */
    if (tcb-> thread_type == client_event)
      {
        tcb-> input_timeout  = tcb-> http.input_timeout;
        tcb-> output_timeout = tcb-> http.output_timeout;
        tcb-> http.port      = atoi (master_port) + ip_portbase;
        tcb-> http.socket    = tcb-> handle;
        tcb-> stats.client   = TRUE;
        tcb-> stats.socket   = tcb-> handle;
        tcb-> stats.username = NULL;
      }
    else
      {
        tcb-> input_timeout  = 0;
        tcb-> output_timeout = 0;
        tcb-> http.port      = 0;       /*  Not serving any port             */
        tcb-> stats.client   = FALSE;
        mainq = thread-> queue-> qid;

        /*  Get maximum connections                                          */
        max_threads = atoi (CONFIG ("server:limit"));
        if (max_threads > 0)
            this_agent-> max_threads = max_threads + 1;
      }
}


/**************************   OPEN SERVER LOG FILE   *************************/

MODULE open_server_log_file (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    mem_strfree (&server_log);
    server_log     = mem_strdup (CONFIG ("serverlog:filename"));
    server_logging = *CONFIG ("serverlog:enabled") == '1';
    if (strnull (server_log))           /*  Blank filename-> no logging      */
        server_logging = FALSE;

    /*  Open console log file: if none, tell console to be quiet also        */
    if (server_logging)
      {
        clog_thread = thread_create (SMT_LOGGING, server_log);
        clogq = clog_thread-> queue-> qid;

        /*  Cycle console log file if required                               */
        file_cycle (server_log, cycle_mode ("serverlog:cycle", 1));
        sendfmt (&clogq, "APPEND", server_log);
        sendfmt (&operq, "LOG",    server_log);
        sendfmt (&operq, "ENABLE", "");
      }
    else
        sendfmt (&operq, "DISABLE", "");

    /*  Set rate at which the timer agent sends us alarm events              */
    /*  We use these to cycle the log files                                  */
    set_timer_refresh (thread);
}


/*  -------------------------------------------------------------------------
 *  set_timer_refresh
 *
 *  Tell timer to send us an alarm every so often, so we can check for
 *  modified configuration files and/or log files that need to be
 *  cycled.  Config refresh rate is in seconds; we need centiseconds.
 */

static void
set_timer_refresh (THREAD *thread)
{
    qbyte
        refresh;                        /*  Server refresh rate              */

    refresh = atol (CONFIG ("server:refresh")) * 100;
    if (refresh == 0)
        return;                         /*  Then do nothing                  */

    if (refresh < 1000)
        refresh = 1000;                 /*  10 seconds minimum rate          */

    event_send (
        &timeq,                         /*  Send to specified queue          */
        &thread-> queue-> qid,          /*  Queue for reply                  */
        "FLUSH",                        /*  Name of event to send            */
        NULL, 0,                        /*  No event body                    */
        NULL, NULL, NULL,               /*  No response events               */
        0);                             /*  No timeout                       */

    msg_size = exdr_writed (&msg, SMT_TIME_CLOCK,
                           (qbyte) 0, refresh, (word) 0, 0, NULL);
    event_send (
        &timeq,                         /*  Send to specified queue          */
        &thread-> queue-> qid,          /*  Queue for reply                  */
        "CLOCK",                        /*  Name of event to send            */
        msg_body, msg_size,             /*  Event body and size              */
        NULL, NULL, NULL,               /*  No response events               */
        0);                             /*  No timeout                       */
}


/*  -------------------------------------------------------------------------
 *  cycle_mode
 *
 *  Searches the config table for a cycle mode for the specified logfile.
 *  Returns one of the sflfile.h CYCLE_ constants; default if no entry is
 *  defined is CYCLE_NEVER.  If startup argument is FALSE, treats the
 *  'startup' mode as 'never'.
 */

static int
cycle_mode (const char *key, Bool startup)
{
    char
        *cycle_mode;                    /*  How do we cycle the file?        */
    int
        cycle_how;                      /*  Cycle mode                       */

    cycle_mode = CONFIG (key);
    if (streq (cycle_mode, "startup") && startup)
        cycle_how = CYCLE_ALWAYS;
    else
    if (streq (cycle_mode, "hourly"))
        cycle_how = CYCLE_HOURLY;
    else
    if (streq (cycle_mode, "daily"))
        cycle_how = CYCLE_DAILY;
    else
    if (streq (cycle_mode, "weekly"))
        cycle_how = CYCLE_WEEKLY;
    else
    if (streq (cycle_mode, "monthly"))
        cycle_how = CYCLE_MONTHLY;
    else
        cycle_how = CYCLE_NEVER;

    return (cycle_how);
}


/*************************   REOPEN SERVER LOG FILE   ************************/

MODULE reopen_server_log_file (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (strneq (clog_thread-> name, CONFIG ("serverlog:filename")))
      {
        sendfmt (&clogq, "CLOSE", "");
        open_server_log_file (thread);
      }
}


/***************************   LOAD VIRTUAL HOSTS   **************************/

MODULE load_virtual_hosts (THREAD *thread)
{
    SYMBOL
        *symbol;                        /*  Virtual host name in config      */
    char
        *vhost_name,                    /*  Virtual host name, if any        */
        *vhost_file,                    /*  Virtual host config file         */
        *vpasswd_file;                  /*  Virtual host password file       */
    SYMBOL
        *vhsym;                         /*  Virtual host entry in table      */
    VHOST
        *vhost;                         /*  Virtual host resource block      */

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    /*  Find and process all virtual host definitions.  For each virtual
     *  host, we create an entry in the vhosts table, and create the various
     *  resources that a virtual host needs (config table, logger, etc.)
     */
    vhosts = sym_create_table ();

    for (symbol = config-> symbols; symbol; symbol = symbol-> next)
      {
        if (strprefixed (symbol-> name, "virtual_hosts:"))
          {
            /*  Get virtual host name, and required config file              */
            vhost_name = symbol-> name + strlen ("virtual_hosts:");
            vhost_file = symbol-> value;
            trace ("Loading VH data for %s from %s", vhost_name, vhost_file);

            /*  Host may not be defined more than once                       */
            if (sym_lookup_symbol (vhosts, vhost_name))
              {
                sprintf (server_message, "Duplicate host '%s'", vhost_name);
                sendfmt (&operq, "ERROR", server_message);
              }
            else
            if (file_is_readable (vhost_file))
              {
                /*  Create new symbol for virtual host                       */
                vhsym = sym_create_symbol (vhosts, vhost_name, NULL);
                vhost = mem_alloc (sizeof (VHOST));
                vhsym-> data = vhost;
                vhost-> name = mem_strdup (vhost_name);

                /*  Load config file, inheriting xitami.cfg and defaults.cfg */
                vhost-> config = sym_create_table ();
                load_config_file (vhost-> config, vhost_file);
                set_vhost_name   (vhost-> config, vhost-> name);
                trace ("- loading config data from %s", vhost_file);

                /*  Load password data for this virtual host                 */
                vpasswd_file = http_config (vhost-> config,
                                            "security:filename");
                vhost-> passwd = load_password_file (vpasswd_file);

                /*  Set rootdir and cgidir for this virtual host             */
                vhost-> rootdir = sym_get_value (vhost-> config,
                                  "server:webpages", rootdir);
                vhost-> cgidir  = sym_get_value (vhost-> config,
                                  "server:cgi_bin", cgidir);
                trace ("- webpages=%s cgi-dir=%s", vhost-> rootdir,
                                                   vhost-> cgidir);

                /*  Create logging threads as required                       */
                create_log_threads (vhost);
              }
            else
              {
                sprintf (server_message, "Config file '%s' not found - %s",
                          vhost_file, strerror (errno));
                sendfmt (&operq, "ERROR", server_message);
              }
          }
      }
    trace ("Loading VH data for base host");

    /*  Now load default host, and create resources as required              */
    vhsym = sym_create_symbol (vhosts, VHOST_ANY, NULL);
    vhost = mem_alloc (sizeof (VHOST));
    vhsym-> data = vhost;
    vhost-> name = mem_strdup (VHOST_ANY);

    /*  Load config file, inheriting xitami.cfg and defaults.cfg             */
    vhost-> config = sym_create_table ();
    load_config_file (vhost-> config, CONFIG ("server:base_host"));

    /*  Load password data for default host                                  */
    vhost-> passwd = load_password_file (CONFIG ("security:filename"));

    /*  Resolve options that can come both from command-line & config file:
     *  If these were specified on the command-line or control panel AND
     *  were not the same as the default values, we look in the config
     *  file.  If both the command-line and config file specify non-default
     *  values, the command-line wins.
     */
    if (streq (rootdir, DEFAULT ("server:webpages")))
      {
        mem_strfree (&rootdir);
        rootdir = mem_strdup (resolve_path (
            sym_get_value (vhost-> config, "server:webpages",
                           DEFAULT ("server:webpages"))));
      }
    if (streq (cgidir, DEFAULT ("server:cgi_bin")))
      {
        mem_strfree (&cgidir);
        cgidir = mem_strdup (resolve_path (
            sym_get_value (vhost-> config, "server:cgi_bin",
                           DEFAULT ("server:cgi_bin"))));
      }
    if (ip_portbase == 0)
        ip_portbase = atoi (
            sym_get_value (vhost-> config, "server:portbase", "0"));

    /*  Set rootdir and cgidir for default host                              */
    vhost-> rootdir = rootdir;
    vhost-> cgidir  = cgidir;
    trace ("- webpages=%s cgi-dir=%s", rootdir, cgidir);

    /*  Create logging threads as required                                   */
    create_log_threads (vhost);

    /*  Set TCB for master thread to use default resources                   */
    tcb-> config = vhost-> config;
    tcb-> passwd = vhost-> passwd;
    tcb-> alogq  = vhost-> alogq;
    tcb-> elogq  = vhost-> elogq;
}


/*  -------------------------------------------------------------------------
 *  load_config_file
 *
 *  Loads the default configuration into the specified table.
 */

static void
load_config_file (SYMTAB *table, char *filename)
{
    ini_dyn_load (table, "xitami.cfg");
    ini_dyn_load (table, CONFIG ("server:defaults"));
    ini_dyn_load (table, filename);
}


/*  -------------------------------------------------------------------------
 *  load_password_file
 *
 *  Loads the specified file into a symbol table; returns the symbol table.
 *  If the file could not be read, sends an error message to the console
 *  and halts the server with a fatal error exception.  The filename may be
 *  empty, in which case an empty table is loaded.
 */

static SYMTAB *
load_password_file (const char *filename)
{
    SYMTAB
        *table;

    table = sym_create_table ();
    if (strnull (filename))
        trace ("- no password data");
    else
    if (file_is_readable (filename))
      {
        trace ("- loading password data from: %s", filename);
        ini_dyn_load (table, filename);
      }
    else
      {
        sprintf (server_message, "'%s' not found - %s",
                 filename, strerror (errno));
        sendfmt (&operq, "ERROR", server_message);
        sendfmt (&operq, "INFO",
            "For security reasons, you must define a password file");
        sendfmt (&operq, "INFO",
            "Create a password file, or set 'security:filename=\"\"'");
        raise_exception (fatal_event);
      }
    return (table);
}


/*  -------------------------------------------------------------------------
 *  set_vhost_name
 *
 *  If the configuration does not define a server:hostname, inserts the
 *  specified hostname symbol.
 */

static void
set_vhost_name (SYMTAB *table, const char *name)
{
    /*  We do not do this if the name starts with a letter                   */
    if (isalpha (*name)
    &&  sym_lookup_symbol (table, "server:hostname") == NULL)
        sym_assume_symbol (table, "server:hostname", name);
}


/*  -------------------------------------------------------------------------
 *  create_log_threads
 *
 *  Creates an access log and error log, as specified for the virtual host.
 *  If the access log name is "NULL", will not create a log file.  Creates
 *  a logging agent thread when needed, and prepares the log file for
 *  writing.  Sets the vhost-> alogq and elogq entries.
 */

static void
create_log_threads (VHOST *vhost)
{
    static QID
        null_qid = { 0, 0 };
    THREAD
        *log_thread;                    /*  Access log thread                */

    vhost-> alog_owner = FALSE;
    vhost-> elog_owner = FALSE;
    vhost-> alogfile = mem_strdup (
                       http_config (vhost-> config, "accesslog:filename"));
    vhost-> elogfile = mem_strdup (
                       http_config (vhost-> config, "errorlog:filename"));

    if (*http_config (vhost-> config, "accesslog:enabled") == '1')
      {
        /*  Create access logging thread, if it does not already exist       */
        log_thread = thread_lookup (SMT_LOGGING, vhost-> alogfile);
        if (log_thread == NULL)
          {
            /*  Cycle log file if required                                   */
            file_cycle (vhost-> alogfile, cycle_mode ("accesslog:cycle", 1));

            /*  Open access log file                                         */
            log_thread = thread_create (SMT_LOGGING, vhost-> alogfile);
            sendfmt (&log_thread-> queue-> qid, "APPEND", "");
            sendfmt (&log_thread-> queue-> qid, "PLAIN",  "");
            vhost-> alog_owner = TRUE;
          }
        vhost-> alogq = log_thread-> queue-> qid;
      }
    else
        vhost-> alogq = null_qid;

    if (*http_config (vhost-> config, "errorlog:enabled") == '1')
      {
        /*  Create error logging thread, if it does not already exist        */
        log_thread = thread_lookup (SMT_LOGGING, vhost-> elogfile);
        if (log_thread == NULL)
          {
            file_cycle (vhost-> elogfile, cycle_mode ("errorlog:cycle", 1));
            log_thread = thread_create (SMT_LOGGING, vhost-> elogfile);
            sendfmt (&log_thread-> queue-> qid, "APPEND", "");
            sendfmt (&log_thread-> queue-> qid, "PLAIN",  "");
            vhost-> elog_owner = TRUE;
          }
        vhost-> elogq = log_thread-> queue-> qid;
      }
    else
        vhost-> elogq = null_qid;
}


/**************************   CHECK ROOTDIR EXISTS   *************************/

MODULE check_rootdir_exists (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    /*  Initialise the root directory for document and cgi files             */
    if (!file_is_directory (rootdir))
      {
        sprintf (server_message, "Webpages directory '%s' not found - %s",
                 rootdir, strerror (errno));
        sendfmt (&operq, "ERROR", server_message);
        raise_exception (fatal_event);
      }
}


/***************************   OPEN MASTER SOCKET   **************************/

MODULE open_master_socket (THREAD *thread)
{
    static char
        portbase_opt [20];              /*  "server:portbaseNN"              */

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    /*  We try the current ip_portbase first; if that does not work then     */
    /*  we try each server:portbaseN in turn.                                */
    if (*CONFIG ("server:ipaddress") != '*')
        ip_passive = inet_addr (CONFIG ("server:ipaddress"));
    tcb-> http.port = atoi (master_port) + ip_portbase;
    tcb-> handle    = passive_TCP (master_port, 5);

    sendfmt (&operq, "INFO",
                     "smthttp: preparing for connections on port %d",
                      tcb-> http.port);

    if (tcb-> handle != INVALID_SOCKET)
      {
        http_set_priority ();
        sendfmt (&operq, "INFO",
                 "smthttp: ready for HTTP connections on port %d",
                 tcb-> http.port);
        sprintf (server_message, "HTTP service started at %s", time_str ());
      }
    else
      {
        sendfmt (&operq, "ERROR",
                 "smthttp: could not open HTTP port %d", tcb-> http.port);
        sendfmt (&operq, "ERROR",
                 "smthttp: %s", connect_errlist [connect_error ()]);
        sendfmt (&operq, "ERROR",
                 "smthttp: %s", sockmsg ());
        raise_exception (fatal_event);
        sprintf (server_message, "Could not open HTTP port %d - %s (%s)",
                 tcb-> http.port, connect_errlist [connect_error ()],
                 sockmsg ());

        /*  Try the configuration options server:portbaseN if possible       */
        sprintf (portbase_opt, "server:portbase%d", ++tcb-> portbase);
        ip_portbase = atoi (CONFIG (portbase_opt));
        if (ip_portbase)
            raise_exception (sock_retry_event);
      }
}


/*************************   WAIT FOR SOCKET INPUT   *************************/

MODULE wait_for_socket_input (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    msg_size = exdr_writed (&msg, SMT_SOCK_INPUT, tcb-> input_timeout,
                            tcb-> handle, (qbyte) 0);
    event_send (
        &sockq,                         /*  Send to specified queue          */
        &thread-> queue-> qid,          /*  Queue for reply                  */
        "INPUT",                        /*  Name of event to send            */
        msg_body, msg_size,             /*  Event body and size              */
        NULL, NULL, NULL,               /*  No response events               */
        0);                             /*  No timeout                       */
}


/************************   RELOAD CONFIG IF NEEDED   ************************/

MODULE reload_config_if_needed (THREAD *thread)
{
    THREAD
        *ftp_thread;                    /*  FTP agent thread, if any         */
    SYMBOL
        *symbol;                        /*  Pointer to symbol                */
    VHOST
        *vhost;
    char
        *vhost_file;

    tcb = thread-> tcb;                 /*  Point to thread's context        */
    http_set_priority ();
    for (symbol = vhosts-> symbols; symbol; symbol = symbol-> next)
      {
        vhost = symbol-> data;
        if (ini_dyn_changed (vhost-> config))
          {
            vhost_file = mem_strdup (sym_get_value
                             (vhost-> config, "filename", NULL));
            sym_empty_table  (vhost-> config);
            load_config_file (vhost-> config, vhost_file);
            set_vhost_name   (vhost-> config, vhost-> name);
            mem_free (vhost_file);

            /*  Set rootdir and cgidir for this virtual host                 */
            vhost-> rootdir = sym_get_value (vhost-> config,
                                             "server:webpages", rootdir);
            vhost-> cgidir  = sym_get_value (vhost-> config,
                                             "server:cgi_bin", cgidir);
          }
        ini_dyn_refresh (vhost-> passwd);

        /*  Recycle access log file if we're the owner and it's time         */
        if (vhost-> alog_owner
        &&  file_cycle_needed (vhost-> alogfile,
                               cycle_mode ("accesslog:cycle", 0)))
            sendfmt (&vhost-> alogq, "CYCLE", vhost-> alogfile);

        /*  Recycle error log file if we're the owner and it's time          */
        if (vhost-> elog_owner
        &&  file_cycle_needed (vhost-> elogfile,
                               cycle_mode ("errorlog:cycle", 0)))
            sendfmt (&vhost-> elogq, "CYCLE", vhost-> elogfile);
      }
    /*  Also reload main config file                                         */
    if (ini_dyn_changed (config))
      {
        sym_empty_table  (config);
        load_config_file (config, CONFIG ("server:base_host"));
        set_timer_refresh (thread);     /*  Refresh the refresh              */
      }
    /*  Cycle server log file if required                                    */
    if (server_logging
    &&  file_cycle_needed (server_log, cycle_mode ("serverlog:cycle", 0)))
        sendfmt (&clogq, "CYCLE", server_log);

    /*  Look for FTP agent and refresh that too if possible                  */
    if ((ftp_thread = thread_lookup ("smtftpc", "main")) != NULL)
        SEND (&ftp_thread-> queue-> qid, "RELOAD", "");
}


/************************   ACCEPT CLIENT CONNECTION   ***********************/

MODULE accept_client_connection (THREAD *thread)
{
    sock_t
        slave_socket;                   /*  Connected socket                 */
    THREAD
        *child_thread;                  /*  Handle to child threads          */

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    slave_socket = accept_socket (tcb-> handle);
    if (slave_socket != INVALID_SOCKET)
      {
        child_thread = thread_create (AGENT_NAME, "");
        if (child_thread)
          {
            SEND (&child_thread-> queue-> qid, "_CLIENT", "");
            ((TCB *) child_thread-> tcb)-> thread_type = client_event;
            ((TCB *) child_thread-> tcb)-> handle      = slave_socket;
            connect_count++;            /*  Maintain statistics              */
            cur_connects++;             /*  One more active connection       */
            if (max_connects < cur_connects)
                max_connects = cur_connects;
          }
        else
            close_socket (slave_socket);
      }
    else
    if (sockerrno != EAGAIN && sockerrno != EWOULDBLOCK)
      {
        sprintf (server_message, "Network problem: %s", sockmsg ());
        sendfmt (&operq, "ERROR",
                 "smthttp: could not accept connection: %s", sockmsg ());
        raise_exception (exception_event);
      }
}


/*************************   PREPARE CLIENT THREAD   *************************/

MODULE prepare_client_thread (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> buffer = mem_alloc (BUFFER_SIZE + 1);
    if (tcb-> buffer == NULL)
      {
        sendfmt (&operq, "ERROR", "smthttp: out of memory");
        raise_exception (exception_event);
      }
    else
        memset (tcb-> buffer, 0, BUFFER_SIZE + 1);
}


/************************   PREPARE TO READ REQUEST   ************************/

MODULE prepare_to_read_request (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */
    tcb-> readsize = 0;                 /*  No data read yet                 */
}


/*************************   READ HTTP REQUEST NEXT   ************************/

MODULE read_http_request_next (THREAD *thread)
{
    int
        rc;                             /*  Return code from read            */

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    /*  tcb-> readsize is amount of data we've read so far.  If we already
     *  read a full buffer, then the buffer is too small.  We reply with a
     *  code 413 (request too large), which is really HTTP/1.1 but okay.
     *  We parse the request, so that whatever header fields we got can
     *  be stuck into the request symbol table (to log the request).
     */
    if (tcb-> readsize == BUFFER_SIZE)
      {
        http_parse_request (&tcb-> http, tcb-> buffer);
        raise_exception (error_event);
        tcb-> http.response = HTTP_RESPONSE_TOOLARGE;
      }
    else
      {
        /*  Read at most (BUFFER_SIZE - tcb-> readsize) bytes into the
         *  thread's buffer, at offset tcb-> readsize.  Note that we can
         *  only be here when the socket has something for us, so this
         *  read call will never block (if TCP/IP is behaving nicely).
         */
        rc = read_TCP (tcb-> handle,
                      &tcb-> buffer [tcb-> readsize],
                       BUFFER_SIZE - tcb-> readsize);
        if (rc > 0)
            tcb-> readsize += rc;       /*  We read something                */
        else
            check_socket_rc (rc);
      }
}


static void
check_socket_rc (int rc)
{
    if (rc == 0 || sockerrno == EPIPE || sockerrno == ECONNRESET)
        raise_exception (sock_closed_event);
    else
    if (sockerrno == EAGAIN || sockerrno == EWOULDBLOCK)
        raise_exception (sock_retry_event);
    else
      {
        trace ("Socket error: %s", sockmsg ());
        raise_exception (sock_error_event);
      }
}


/***********************   CHECK IF REQUEST COMPLETE   ***********************/

MODULE check_if_request_complete (THREAD *thread)
{
    static char
        length_field [] = "Content-Length: ",
        build_filename [13];            /*  For generated POST filenames     */
    static int
        build_counter = 0;              /*  Cyclic filename counter          */
    char
        *eoln,                          /*  End-of-line marker               */
        *blank_line,                    /*  End of request header            */
        *length_ptr,                    /*  Content-Length: field            */
        *post_data;                     /*  Start of body content            */
    int
        header_size,                    /*  Size of request header           */
        skip_spaces;                    /*  Whitespace at start of request   */
    FILE
        *log;                           /*  Debugging log file               */

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    /*  We examine the data we've read so far, to figure-out whether it looks
     *  complete or not.  Since the connection remains open in order for the
     *  server to reply, we can't use the end-of-connection as the end-of-data
     *  indicator.  A normal HTTP header ends in a blank line (CRLFCRLF or
     *  LFLF) followed by the body text.  We also accept a one-line header
     *  that ends in a newline (LF) with no body text.  If the header also
     *  has a 'Content-Length:' field, we expect a body of that size.  The
     *  body is stored in a temporary file that is later passed to the CGI
     *  program (generally posted data is handled by a CGI program).
     */

    /*  Some browsers throw-in a CRLF after posting data; on a keep-alive
     *  connection this has the pleasant side-effect of sticking a blank
     *  line at the start of the next request.  We flush any such junk, by
     *  skipping any white space at the start of the request.
     */
    skip_spaces = 0;
    while (skip_spaces < tcb-> readsize
    && isspace (tcb-> buffer [skip_spaces]))
        skip_spaces++;

    if (skip_spaces)
      {
        tcb-> readsize -= skip_spaces;
        memmove (tcb-> buffer, tcb-> buffer + skip_spaces, tcb-> readsize);
      }

    /*  Look for CRLFCRLF, which indicates that the header is complete       */
    tcb-> buffer [tcb-> readsize] = 0;

    /*  If server:debug is set, log the incoming request                     */
    if (*CONFIG ("server:debug") == '1')
      {
        log = fopen ("request.log", "a");
        if (log)
          {
            fprintf (log, "%s\n", tcb-> buffer);
            fclose (log);
          }
      }

    eoln = "\r\n\r\n";
    blank_line = txtfind (tcb-> buffer, eoln);
    if (!blank_line)
      {
        eoln = "\n\n";
        blank_line = txtfind (tcb-> buffer, eoln);
      }

    if (blank_line)                     /*  Header is complete               */
      {
        /*  Look for Content-Length: header field in request header.
         *  If we find it, then we're reading POSTed data.  If the amount
         *  of data is large, we create a temporary file, and save any
         *  posted data already read into that file.  If the data will fit
         *  into the buffer, we carry on reading.
         *  If there's no Content-Length: field, the request is complete.
         */
        the_next_event = finished_event;
        length_ptr = txtfind (tcb-> buffer, length_field);
        if (length_ptr)
          {
            post_data   = blank_line + sizeof (eoln);
            header_size = post_data - tcb-> buffer;

            tcb-> post_size = atol (length_ptr + sizeof (length_field) - 1);
            tcb-> post_read = tcb-> readsize - (post_data - tcb-> buffer);
            if ((header_size + tcb-> post_size) > BUFFER_SIZE)
              {
                /*  POSTed data is large, possibly an uploaded file.  So, we
                 *  will shunt it into a temporary file 'postnnnn.cgi'.
                 */
                sprintf (build_filename, "post%04d.cgi", ++build_counter);
                if (build_counter > 9999)
                    build_counter = 0;
                tcb-> http.cgi_stdin = mem_strdup (build_filename);
                tcb-> post_file = fopen (build_filename, FOPEN_WRITE_BINARY);
                if (tcb-> post_file)
                    fwrite (post_data, tcb-> post_read, 1, tcb-> post_file);

                /*  There is always more data waiting                        */
                the_next_event = post_event;
              }
            else
            if (tcb-> post_size > 0)
              {
                /*  Header + POSTed data will fit into the tcb-> buffer      */
                if (tcb-> post_read < tcb-> post_size)
                    the_next_event = post_event;
              }
          }
      }
    else
      {
        /*  Check if we got a 1-line header                                  */
        blank_line = strchr (tcb-> buffer, '\n');
        if (blank_line && blank_line [1] == '\0')
            the_next_event = finished_event;
        else
            the_next_event = more_event;
      }
}


/*************************   READ HTTP POSTED DATA   *************************/

MODULE read_http_posted_data (THREAD *thread)
{
    int
        rc;                             /*  Return code from read_TCP        */

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (tcb-> post_file)
      {
        /*  Read chunk of data from the socket and write to the POST capture
         *  file. Note that we can only be here when the socket has something
         *  for us, so this read call will never block (if TCP/IP is behaving
         *  nicely).
         */
        rc = read_TCP (tcb-> handle, msg_body, BUFFER_SIZE);
        if (rc > 0)
          {
            tcb-> post_read += rc;      /*  We read something                */
            fwrite (msg_body, rc, 1, tcb-> post_file);
          }
        else
            check_socket_rc (rc);
      }
    else
        read_http_request_next (thread);
}


/**********************   CHECK IF POST DATA COMPLETE   **********************/

MODULE check_if_post_data_complete (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (tcb-> post_read < tcb-> post_size)
        the_next_event = post_event;
    else
      {
        the_next_event = finished_event;
        close_post_file ();
      }
}


static void
close_post_file ()
{
    if (tcb-> post_file)
      {
        fprintf (tcb-> post_file, "\n");
        fclose  (tcb-> post_file);
      }
    tcb-> post_file = NULL;
}


/***************************   PARSE HTTP REQUEST   **************************/

MODULE parse_http_request (THREAD *thread)
{
    int
        method;

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    method = http_parse_request (&tcb-> http, tcb-> buffer);
    switch (method)
      {
        case HTTP_METHOD_GET:
        case HTTP_METHOD_HEADER:
            the_next_event = get_request_event;
            break;
        case HTTP_METHOD_POST:
            the_next_event = post_request_event;
            break;
        default:
            the_next_event = error_event;
            break;
      }
    tcb-> stats.username = tcb-> http.username;
}


/************************   RESOLVE VIRTUAL HOSTNAME   ***********************/

MODULE resolve_virtual_hostname (THREAD *thread)
{
    char
        *vhost_name;                    /*  Name of virtual host             */
    SYMBOL
        *vhost_sym;                     /*  Virtual host entry in table      */
    VHOST
        *vhost;                         /*  Virtual host resource block      */
    Bool
        uses_vhost = TRUE;              /*  Did we use a virtual host?       */

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    /*  Resolve virtual host name or multihoming, and prepare thread to use
     *  the required host resources.
     *  First, take local socket IP address and try that...
     */
    vhost_name = socket_localaddr  (tcb-> handle);
    vhost_sym  = sym_lookup_symbol (vhosts, vhost_name);
    trace ("Looking for VH using IP address: %s", vhost_name);

    /*  If that failed, try Host: header from HTTP request                   */
    if (!vhost_sym)
      {
        vhost_name = http_header_value (&tcb-> http, "Host");
        if (*vhost_name)                /*  If "", no such header field      */
          {
            vhost_name = mem_strdup (vhost_name);
            strlwc    (vhost_name);
            strconvch (vhost_name, '-', '_');
            vhost_sym = sym_lookup_symbol (vhosts, vhost_name);
            trace ("- looking for VH using Host: value: %s", vhost_name);
            mem_free  (vhost_name);
          }
       }
    /*  If that failed, use VHOST_ANY host name                              */
    if (!vhost_sym)
      {
        vhost_sym = sym_lookup_symbol (vhosts, VHOST_ANY);
        trace ("- using default virtual host values");
        uses_vhost = FALSE;
      }
    ASSERT (vhost_sym);
    vhost = vhost_sym-> data;

    /*  Use virtual host resources                                           */
    tcb-> config = vhost-> config;
    tcb-> passwd = vhost-> passwd;
    tcb-> alogq  = vhost-> alogq;
    tcb-> elogq  = vhost-> elogq;

    mem_strfree (&tcb-> http.vhost);
    mem_strfree (&tcb-> http.rootdir);
    mem_strfree (&tcb-> http.cgidir);
    tcb-> http.uses_vhost = uses_vhost;
    tcb-> http.vhost      = mem_strdup (vhost_sym-> name);
    tcb-> http.rootdir    = mem_strdup (vhost-> rootdir);
    tcb-> http.cgidir     = mem_strdup (vhost-> cgidir);
    tcb-> http.config     = vhost-> config;
    http_refresh_context (&tcb-> http);
}


/************************   CHECK CLIENT IP ALLOWED   ************************/

MODULE check_client_ip_allowed (THREAD *thread)
{
    char
        *webmask;                       /*  Permitted mask for connection    */

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    webmask = TCONFIG ("security:webmask");
    if (streq (webmask, "local"))
        webmask = socket_localaddr (tcb-> handle);

    if (!socket_is_permitted (socket_peeraddr (tcb-> handle), webmask))
      {
        tcb-> http.response = HTTP_RESPONSE_FORBIDDEN;
        raise_exception (error_event);
      }
}


/******************************   GET URL TYPE   *****************************/

MODULE get_url_type (THREAD *thread)
{
    int
       file_type;

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    file_type = http_get_url_type (&tcb-> http);
    switch (file_type)
      {
        case HTTP_URL_HTML:   the_next_event = html_event;   break;
        case HTTP_URL_IMAP:   the_next_event = imap_event;   break;
        case HTTP_URL_WSX:    the_next_event = wsx_event;    break;
        case HTTP_URL_CGI:    the_next_event = cgi_event;    break;
        case HTTP_URL_FILTER: the_next_event = filter_event; break;
        default:              the_next_event = error_event;
      }
    tcb-> cgi_request = (file_type == HTTP_URL_CGI);
}


/************************   CHECK FILE OR DIRECTORY   ************************/

MODULE check_file_or_directory (THREAD *thread)
{
    int
        file_status;

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    file_status = http_check_file (&tcb-> http);
    switch (file_status)
      {
        case HTTP_FILE_OK:
            the_next_event = file_event;
            break;
        case HTTP_FILE_DIRECTORY:
            the_next_event = directory_event;
            break;
        case HTTP_FILE_EXECUTABLE:
            the_next_event = executable_event;
            break;
        default:
            the_next_event = error_event;
            break;
      }
}


/****************************   GET URL FROM MAP   ***************************/

MODULE get_url_from_map (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (!http_get_url_in_map (&tcb-> http))
        raise_exception (error_event);
}


/***************************   CHECK IF PROTECTED   **************************/

MODULE check_if_protected (THREAD *thread)
{
    char
        *url = NULL;                    /*  Protected URL value              */
    char
        *url_password,                  /*  Required password for URL        */
        *url_webmask,                   /*  Permitted mask for URL           */
        *superuser;                     /*  Superuser password               */

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    url = mem_strdup (tcb-> http.url);

    /*  First check super-user password                                      */
    superuser = TCONFIG ("security:superuser");
    if (http_password_okay (superuser, tcb-> http.password))
        the_next_event = ok_event;      /*  Let the superuser through        */
    else
    if (http_uri_protected (tcb-> passwd, url))
      {
        /*  First check that client IP address is allowed                    */
        the_next_event = protected_event;

        /*  The Realm is an arbitrary name that the administrator can use to
         *  group resources.  If this is defined in the password file for a
         *  resource, using realm=, we use the realm.  Otherwise we use the
         *  protected resource name.
         */
        mem_strfree (&tcb-> http.realm);
        tcb-> http.realm = mem_strdup (ini_dyn_value (tcb-> passwd,
                                                      url, "realm", url));

        url_webmask = ini_dyn_value (tcb-> passwd, url, "webmask", "*");
        if (streq (url_webmask, "local"))
            url_webmask = socket_localaddr (tcb-> handle);

        if (socket_is_permitted (socket_peeraddr (tcb-> handle), url_webmask)
        &&  tcb-> http.username && *tcb-> http.username
        &&  tcb-> http.password && *tcb-> http.password)
          {
            /*  If browser sent authorization information, check it...       */
            url_password = ini_dyn_value (tcb-> passwd,
                                          url, tcb-> http.username, "");
            if (http_password_okay (url_password, tcb-> http.password))
                the_next_event = ok_event;
          }
      }
    else
        the_next_event = ok_event;      /*  URL is unprotected - no check    */

    mem_free (url);
}


/***************************   WRITE HTTP HEADER   ***************************/

MODULE write_http_header (THREAD *thread)
{
    FILE
        *log;

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    http_format_header (&tcb-> http, tcb-> buffer);
    log_http_access (thread);

    /*  If server:debug is set, log the transmitted header                   */
    if (*CONFIG ("server:debug") == '1')
      {
        log = fopen ("header.log", "a");
        if (log)
          {
            fprintf (log, "%s", tcb-> buffer);
            if (*tcb-> buffer && strlast (tcb-> buffer) != '\n')
                fprintf (log, "\n\n");
            fclose (log);
          }
      }
    msg_size = exdr_writed (&msg, SMT_SOCK_WRITE,
        tcb-> output_timeout,           /*  Timeout for request              */
        tcb-> handle,                   /*  Socket to write to               */
        strlen (tcb-> buffer),          /*  Amount of data to write          */
        tcb-> buffer,                   /*  Address of data block            */
        (qbyte) 0);                     /*  No request tag                   */

    event_send (
        &sockq,                         /*  Send to specified queue          */
        &thread-> queue-> qid,          /*  Queue for reply                  */
        "WRITE",                        /*  Name of event to send            */
        msg_body, msg_size,             /*  Event body and size              */
        NULL, NULL, NULL,               /*  No response events               */
        0);                             /*  No timeout                       */

    event_wait ();                      /*  Wait for socket agent reply      */
}


/*  Create log line in standard CERN/NCSA format:
 *
 *  host - user [DD/Mon/YYYY:hh:mm:ss] "request" status size "refer" "agent"
 */

local
log_http_access (THREAD *thread)
{
    tcb = thread-> tcb;

    sendfmt (&tcb-> alogq, "PUT", "%s - %s [%s] \"%s\" %s %ld \"%s\" \"%s\"",
        socket_hostaddr (tcb-> handle),
        tcb-> http.username? tcb-> http.username: "-",
        time_str (),
        tcb-> http.log_request,
        http_response [tcb-> http.response].code,
        tcb-> http.file_size,
        http_header_value (&tcb-> http, "Referer"),
        http_header_value (&tcb-> http, "User-Agent"));

    /*  Anything >= 400 is an error...                                       */
    if (tcb-> http.response >= HTTP_RESPONSE_BADREQUEST)
        sendfmt (&tcb-> elogq, "PUT", "%s - %s [%s] %s \"%s::%s\" \"%s\"",
            socket_hostaddr (tcb-> handle),
            tcb-> http.username? tcb-> http.username: "-",
            time_str (),
            http_response [tcb-> http.response].code,
            tcb-> http.vhost,
            tcb-> http.url,
            tcb-> http.file_name? tcb-> http.file_name: "(None)");
}


/*  -------------------------------------------------------------------------
 *  time_str
 *
 *  Returns the current date and time formatted as: "DD/Mon/YYYY:hh:mm:ss".
 *  The formatted time is in a static string that each call overwrites.
 */

static char *
time_str (void)
{
    static char
        formatted_time [30],
        *months = "Jan\0Feb\0Mar\0Apr\0May\0Jun\0Jul\0Aug\0Sep\0Oct\0Nov\0Dec";
    time_t
        time_secs;
    struct tm
        *time_struct;

    time_secs   = time (NULL);
    time_struct = localtime (&time_secs);

    sprintf (formatted_time, "%02d/%s/%04d:%02d:%02d:%02d %s",
                              time_struct-> tm_mday,
                              months + time_struct-> tm_mon * 4,
                              time_struct-> tm_year + 1900,
                              time_struct-> tm_hour,
                              time_struct-> tm_min,
                              time_struct-> tm_sec,
                              timezone_string ());
    return (formatted_time);
}


/****************************   WRITE HTTP BODY   ****************************/

MODULE write_http_body (THREAD *thread)
{
    int
        msg_size;
    byte
        *buffer;

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (tcb-> http_body)
      {
        count_transfer (strlen (tcb-> http_body));

        msg_size = exdr_write (NULL, SMT_SOCK_WRITE,
            tcb-> output_timeout,       /*  Timeout for request              */
            tcb-> handle,               /*  Socket to write to               */
            strlen (tcb-> http_body),   /*  Amount of data to write          */
            tcb-> http_body,            /*  Address of data block            */
            (qbyte) 0);                 /*  No request tag                   */

        buffer = mem_alloc (msg_size);
        msg_size = exdr_write (buffer, SMT_SOCK_WRITE,
            tcb-> output_timeout,       /*  Timeout for request              */
            tcb-> handle,               /*  Socket to write to               */
            strlen (tcb-> http_body),   /*  Amount of data to write          */
            tcb-> http_body,            /*  Address of data block            */
            (qbyte) 0);                 /*  No request tag                   */

        event_send (
            &sockq,                     /*  Send to specified queue          */
            &thread-> queue-> qid,      /*  Queue for reply                  */
            "WRITE",                    /*  Name of event to send            */
            buffer, msg_size,           /*  Event body and size              */
            NULL, NULL, NULL,           /*  No response events               */
            0);                         /*  No timeout                       */

        mem_free (buffer);
        mem_strfree (&tcb-> http_body);

        event_wait ();                  /*  Wait for socket agent reply      */
      }
}

static void
count_transfer (long size)
{
    static long
        transfer_bytes;                 /*  Count transfer in Kbytes         */

    /*  Keep track of Kbytes for display and individual bytes internally     */
    transfer_bytes += size;
    transfer_size  += transfer_bytes / 1024;
    transfer_bytes  = transfer_bytes % 1024;
}


/***************************   CHECK IF SEND FILE   **************************/

MODULE check_if_send_file (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    http_check_file_time (&tcb-> http);
    if (!tcb-> http.send_file)
        raise_exception (send_header_event);
}


/**************************   SEND FILE TO BROWSER   *************************/

MODULE send_file_to_browser (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    /*  If server:debug is set, log the transmission data                    */
    trace ("Transfer file=%s size=%ld offset=%ld",
            tcb-> http.file_name,
            tcb-> http.file_size,
            tcb-> http.file_offset);

    ASSERT (tcb-> http.file_name);

    msg_size = exdr_writed (&msg, SMT_TRAN_PUTS,
                            tcb-> handle,
                            tcb-> http.file_name,
                            tcb-> http.file_offset, 0);
    event_send (
        &tranq,                         /*  Send to specified queue          */
        &thread-> queue-> qid,          /*  Queue for reply                  */
        "PUT_SLICE",                    /*  Name of event to send            */
        msg_body, msg_size,             /*  Event body and size              */
        NULL, NULL, NULL,               /*  No response events               */
        0);                             /*  No timeout                       */

    /*  Clean-up file offset for any later work                              */
    tcb-> http.file_offset = 0;
    count_transfer (tcb-> http.file_size);
}


/**************************   GET DEFAULT WEB PAGE   *************************/

MODULE get_default_web_page (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (http_get_default_page (&tcb-> http))
        the_next_event = file_event;
    else
        the_next_event = not_found_event;
}


/*************************   MAKE DIRECTORY LISTING   ************************/

MODULE make_directory_listing (THREAD *thread)
{
    char
        *dirlist_key;                   /*  Key to use to check dirlists     */

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    /*  cgi/security:dirlist must be 1 to enable directory listings          */
    /*  Any system-level protection error is considered as a hard error      */
    if (tcb-> cgi_request)
        dirlist_key = "cgi:dirlist";
    else
        dirlist_key = "security:dirlist";

    if (strneq (TCONFIG (dirlist_key), "1")
    || !http_list_directory (&tcb-> http))
      {
        tcb-> http.response = HTTP_RESPONSE_FORBIDDEN;
        raise_exception (error_event);
      }
}


/**************************   CHECK IF KEEP ALIVE   **************************/

MODULE check_if_keep_alive (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (tcb-> http.keep_alive
    &&  socket_is_alive (tcb-> handle))
        the_next_event = ok_event;
    else
        the_next_event = close_event;
}


/***********************   PARTIAL CLEAR HTTP CONTEXT   **********************/

MODULE partial_clear_http_context (THREAD *thread)
{
    int
        max;

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    max = tcb-> http.keep_alive_max - 1;
    memset (tcb-> buffer, 0, BUFFER_SIZE);
    tcb-> readsize = 0;

    http_free_context (&tcb-> http);
    http_init_context (&tcb-> http);
    tcb-> http.port   = atoi (master_port) + ip_portbase;
    tcb-> http.socket = tcb-> handle;
    tcb-> http.keep_alive_max = max;

    if (tcb-> post_file)
        close_post_file ();
}


/*************************   CREATE FILTER PROCESS   *************************/

MODULE create_filter_process (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (http_create_filter_process (&tcb-> http) == 0)
        log_http_access (thread);
    else
      {
        tcb-> http.response = HTTP_RESPONSE_INTERNALERROR;
        raise_exception (error_event);
      }
    tcb-> http.timeout_date = date_now ();
    tcb-> http.timeout_time = time_now ();
    future_date (&tcb-> http.timeout_date, &tcb-> http.timeout_time,
                 0, atoi (TCONFIG ("cgi:timeout")) * 100);
}


/***************************   CREATE CGI PROCESS   **************************/

MODULE create_cgi_process (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (http_create_cgi_process (&tcb-> http) == 0)
        log_http_access (thread);
    else
      {
        tcb-> http.response = HTTP_RESPONSE_INTERNALERROR;
        raise_exception (error_event);
      }
    tcb-> http.timeout_date = date_now ();
    tcb-> http.timeout_time = time_now ();
    future_date (&tcb-> http.timeout_date, &tcb-> http.timeout_time,
                 0, atoi (TCONFIG ("cgi:timeout")) * 100);
}


/****************************   WAIT PROCESS END   ***************************/

MODULE wait_process_end (THREAD *thread)
{
    int
       state;

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    state = http_process_state (&tcb-> http);
    if (state == CGI_STATE_END)         /*  CGI finished okay                */
        the_next_event = finished_event;
    else
    if (state == CGI_STATE_ERROR)       /*  CGI aborted                      */
      {
        tcb-> http.response = HTTP_RESPONSE_INTERNALERROR;
        the_next_event = error_event;
      }
    else                                /*  CGI ran out of time              */
    if (date_is_past (tcb-> http.timeout_date, tcb-> http.timeout_time))
      {
        tcb-> http.response = HTTP_RESPONSE_OVERLOADED;
        the_next_event = error_event;
      }
    else                                /*  CGI is still running             */
      {
        /*  Ask timer to send us an event after the monitor timeout          */
        msg_size = exdr_writed (
            &msg, SMT_TIME_ALARM, (qbyte) 0, (qbyte)
            (atoi (TCONFIG ("cgi:monitor")) / 10),
            0, (void *) NULL);
        event_send (
            &timeq,                     /*  Send to timer queue              */
            &thread-> queue-> qid,      /*  Queue for reply                  */
            "ALARM",                    /*  Name of event to send            */
            msg_body, msg_size,         /*  Event body and size              */
            NULL, NULL, NULL,           /*  No response events               */
            0);                         /*  No timeout                       */
      }
}


/**************************   CLOSE CHILD PROCESS   **************************/

MODULE close_child_process (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    http_close_process (&tcb-> http);
}


/*************************   REPARSE PROCESS OUTPUT   ************************/

MODULE reparse_process_output (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (http_reparse_output (&tcb-> http))
        raise_exception (redirect_event);
}


/***************************   REPARSE HTTP BODY   ***************************/

MODULE reparse_http_body (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (http_reparse_buffer (&tcb-> http, tcb-> http_body))
        raise_exception (redirect_event);
}


/***************************   INSTALL WSX ALIAS   ***************************/

MODULE install_wsx_alias (THREAD *thread)
{
    /*  The wsx_symbol table holds dynamic WSX aliases.  Each symbol is
     *  stored in the form "vhost:alias=agent" as specified by the
     *  WSX_INSTALL message.  When no virtual host is specified, the
     *  symbol is stored with the VHOST_ANY virtual host name.
     *
     *  This table is always kept sorted.  Alias names are normalised to
     *  lowercase with hyphens replaced by underlines, to allow consistent
     *  lookups and comparisons.
     *
     *  For ease of searching, the symbol table is held in the same format
     *  as loaded ini files.  That is, for each different virtual host, a
     *  symbol is defined with the name "vhost" and no value.
     */

    struct_smt_wsx_install
        *request;
    char
        *virtual_host,
        *alias_path;                    /*  vhost:path                       */

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    get_smt_wsx_install (thread-> event-> body, (void **) &request);

    alias_path   = make_wsx_path (request-> virtual_host, request-> path);
    virtual_host = *request-> virtual_host? request-> virtual_host: VHOST_ANY;

    sym_assume_symbol (wsx_aliases, alias_path, request-> agent);
    sym_assume_symbol (wsx_aliases, virtual_host, "");
    sym_sort_table    (wsx_aliases, NULL);

    mem_free (alias_path);
    free_smt_wsx_install ((void **) &request);
}

static char *
make_wsx_path (char *virtual_host, char *path)
{
    if (virtual_host == NULL || strnull (virtual_host))
        virtual_host = VHOST_ANY;
    if (*path == '/')
        path++;                         /*  Skip '/' at start of path        */

    path = xstrcpy (NULL, virtual_host, ":", path, NULL);
    strlwc    (path);
    strconvch (path, '-', '_');
    return (path);
}


/****************************   CANCEL WSX ALIAS   ***************************/

MODULE cancel_wsx_alias (THREAD *thread)
{
    struct_smt_wsx_cancel
        *request;
    char
        *alias_path;                    /*  vhost:path                       */
    SYMBOL
        *symbol;

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    get_smt_wsx_cancel (thread-> event-> body, (void **) &request);
    alias_path = make_wsx_path (request-> virtual_host, request-> path);
    symbol = sym_lookup_symbol (wsx_aliases, alias_path);
    if (symbol)
        sym_delete_symbol (wsx_aliases, symbol);
    mem_free (alias_path);
    free_smt_wsx_cancel ((void **) &request);
}


/***********************   PASS REQUEST TO WSX AGENT   ***********************/

MODULE pass_request_to_wsx_agent (THREAD *thread)
{
    static char
        post_filename [14];             /*  @postxxxx.cgi                    */
    char
        *post_data,                     /*  Data to send to agent            */
        *virtual_host;                  /*  Name of virtual host, if any     */
    THREAD
        *wsx_thread;                    /*  Handle to WSX agent thread       */
    DESCR
        *symdesc;                       /*  HTTP symbol table                */

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    /*  Look for WSX agent thread                                            */
    if ((wsx_thread = thread_lookup (tcb-> http.wsx_agent, "main")) == NULL)
        raise_exception (not_supported_event);
    else
      {
        /*  POSTed data may be in a temporary file, or in the http block     */
        if (tcb-> http.cgi_stdin)
          {
            sprintf (post_filename, "@%s", tcb-> http.cgi_stdin);
            post_data = post_filename;
          }
        else
            post_data = tcb-> http.post_data;

        if (streq (tcb-> http.vhost, VHOST_ANY))
            virtual_host = NULL;
        else
            virtual_host = tcb-> http.vhost;

        symdesc = http_wsx_symbols (&tcb-> http);
        if (symdesc)
          {
            /*  Pass URL, url-param, and symbol table to WSX agent           */
            send_wsx_request (
                &wsx_thread-> queue-> qid,
                tcb-> http.url,         /*  Requested URL                    */
                tcb-> http.url_param,   /*  Arguments following URL          */
                virtual_host,           /*  Virtual host name, if any        */
                post_data,              /*  POSTed data, if any              */
                (word) symdesc-> size,  /*  Size of symbol table             */
                symdesc-> data);        /*  Symbol table contents            */
            mem_free (symdesc);
          }
        else
            raise_exception (error_event);
      }
}


/*************************   GET WSX RETURN FIELDS   *************************/

MODULE get_wsx_return_fields (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    /*  Pick up WSX reply body from event body, encoded as SMT_WSX_OK.  If
     *  the first character of the reply is '@', it refers to a temporary
     *  file containing the HTTP data.  Otherwise the event body is sent
     *  as-is.
     */
    exdr_read (thread-> event-> body, SMT_WSX_OK, &tcb-> http_body);
    if (tcb-> http_body [0] == '@')     /*  HTTP data is in a file           */
      {
        mem_strfree (&tcb-> http.file_name);
        tcb-> http.file_name = mem_strdup (tcb-> http_body + 1);
        mem_strfree (&tcb-> http_body);
        tcb-> http.temp_file = TRUE;
        the_next_event = redirect_event;
      }
    else
        the_next_event = ok_event;
}


/**************************   GET WSX ERROR FIELDS   *************************/

MODULE get_wsx_error_fields (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    /*  Pick up HTTP error code from event body, encoded as SMT_WSX_ERROR    */
    exdr_read (thread-> event-> body, SMT_WSX_ERROR, &tcb-> http.response);
}


/************************   GET WSX REDIRECT FIELDS   ************************/

MODULE get_wsx_redirect_fields (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    /*  Pick up redirected URL event body, encoded as SMT_WSX_REDIRECT       */
    mem_strfree (&tcb-> http.url);
    exdr_read (thread-> event-> body, SMT_WSX_REDIRECT, &tcb-> http.url);
    tcb-> http.response = HTTP_RESPONSE_FOUND;
}


/***********************   SEND MASTER THREAD RESTART   **********************/

MODULE send_master_thread_restart (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    SEND (&mainq, "_RESTART", "");
}


/**********************   SEND MASTER THREAD SHUTDOWN   **********************/

MODULE send_master_thread_shutdown (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    server_killed = TRUE;
    SEND (&mainq, "SHUTDOWN", "");
}


/**********************   KILL ALL ACTIVE CONNECTIONS   **********************/

MODULE kill_all_active_connections (THREAD *thread)
{
    QUEUE
        *queue;                         /*  Task control block               */

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    /*  We send a CANCEL event to all threads; the master thread ignores it  */
    for (queue = this_agent-> queues.next;
        (void *) queue != &(this_agent-> queues.next);
         queue = queue-> next)
        SEND (&queue-> qid, "_CANCEL", "");
}


/************************   SIGNAL URL UNAUTHORISED   ************************/

MODULE signal_url_unauthorised (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> http.response = HTTP_RESPONSE_UNAUTHORIZED;
    write_http_header (thread);
}


/**************************   SIGNAL NOT SUPPORTED   *************************/

MODULE signal_not_supported (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> http.response = HTTP_RESPONSE_NOTIMPLEMENTED;
    write_http_header (thread);
}


/***************************   SIGNAL BAD REQUEST   **************************/

MODULE signal_bad_request (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> http.response = HTTP_RESPONSE_BADREQUEST;
    write_http_header (thread);
}


/**************************   SIGNAL SOCKET ERROR   **************************/

MODULE signal_socket_error (THREAD *thread)
{
    char
        *message = NULL;

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    /*  Pick up message from event body, encoded as SMT_SOCK_ERROR           */
    exdr_read (thread-> event-> body, SMT_SOCK_ERROR, &message, NULL, NULL);

    sendfmt (&operq, "ERROR",
             "smthttp: error on socket %d: %s", (int) tcb-> handle, message);
    mem_free (message);
}


/***************************   CHECK SOCKET TYPE   ***************************/

MODULE check_socket_type (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    the_next_event = tcb-> thread_type;
}


/*************************   SIGNAL CLOSING SERVER   *************************/

MODULE signal_closing_server (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (tcb-> http.port)
        sendfmt (&operq, "INFO",
                 "smthttp: closing HTTP connections on port %d",
                  tcb-> http.port);
}


/***********************   FREE ALL SERVER RESOURCES   ***********************/

MODULE free_all_server_resources (THREAD *thread)
{
    SYMBOL
        *symbol;                        /*  Pointer to symbol                */
    VHOST
        *vhost;

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    /*  Free main resources used by the web server                           */
    if (vhosts)
      {
        for (symbol = vhosts-> symbols; symbol; symbol = symbol-> next)
          {
            vhost = symbol-> data;
            /*  Close any log files that we own                              */
            if (vhost-> alog_owner)
                SEND (&vhost-> alogq, "CLOSE", "");
            if (vhost-> elog_owner)
                SEND (&vhost-> elogq, "CLOSE", "");

            sym_delete_table (vhost-> config);
            sym_delete_table (vhost-> passwd);
            mem_free (vhost-> alogfile);
            mem_free (vhost-> elogfile);
            mem_free (vhost-> name);
            mem_free (vhost);           /*  Free symbol VHOSTS data          */
          }
        sym_delete_table (vhosts);      /*    and finally, entire table      */
        vhosts = NULL;
      }
}


/************************   SHUTDOWN THE APPLICATION   ***********************/

MODULE shutdown_the_application (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    /*  Free all resources used by the web server                            */
    mem_strfree (&rootdir);             /*  Log file name                    */
    mem_strfree (&cgidir);              /*  Directory for CGI programs       */
    mem_strfree (&server_log);          /*  Main server log filename         */
    http_term ();                       /*  Terminate HTTP library           */
    smt_shutdown ();                    /*  Halt the application             */
}


/*************************   TERMINATE THE THREAD   **************************/

MODULE terminate_the_thread (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (tcb-> buffer)
      {
        mem_free (tcb-> buffer);
        tcb-> buffer = NULL;
      }
    if (tcb-> handle)
      {
        close_socket (tcb-> handle);
        tcb-> handle = 0;
      }
    if (tcb-> post_file)
        close_post_file ();

    if (tcb-> thread_type == client_event)
        cur_connects--;                 /*  One less active connection       */

    http_free_context (&tcb-> http);
    the_next_event = terminate_event;
}
