/*  ----------------------------------------------------------------<Prolog>-
    Name:       smtftpl.h
    Title:      File Transfer Protocol functions
    Package:    Libero SMT 2.x

    Written:    96/08/20  iMatix SMT kernel team <smt@imatix.com>
    Revised:    98/01/04

    Copyright:  Copyright (c) 1991-98 iMatix
    License:    This is free software; you can redistribute it and/or modify
                it under the terms of the SMT License Agreement as provided
                in the file LICENSE.TXT.  This software is distributed in
                the hope that it will be useful, but without any warranty.
 ------------------------------------------------------------------</Prolog>-*/

#ifndef _SMTFTPL_INCLUDED              /*  Allow multiple inclusions         */
#define _SMTFTPL_INCLUDED

#define FTP_VERSION \
    "Xitami FTP " CUR_VERSION " (c) 1991-98 iMatix <http://www.imatix.com>"

#define FTP_HELP_MESSAGE                                                      \
    "214-The following commands are recognized (* =>'s unimplemented).\r\n"   \
    "    USER    PASV    STOU*   MAIL*   ALLO*   CWD     PWD     XMKD \r\n"   \
    "    PASS    TYPE    SYST    MSND*   REST*   CDUP    RMD     XRMD \r\n"   \
    "    ACCT*   STRU*   XSYS*   MSOM*   RNFR    XCWD    SITE*   XPWD \r\n"   \
    "    REIN*   MODE    PASV    MSAM*   RNTO    LIST    STAT*   XCUP*\r\n"   \
    "    QUIT    RETR    APPE*   MRSQ*   ABOR    MKD     HELP    XEXC*\r\n"   \
    "    PORT    STOR    MLFL*   MRCP*   DELE    NLST    NOOP    SIZE \r\n"   \
    "214 " FTP_VERSION "\r\n"


/*- FTP responses -----------------------------------------------------------*/

#define FTP_RC_RESTART_MARKER             0
#define FTP_RC_READY_IN_DELAY             1
#define FTP_RC_DATA_OPEN_START            2
#define FTP_RC_FILE_OK                    3
#define FTP_RC_COMMAND_OK                 4
#define FTP_RC_NOT_IMPLEMENTED            5
#define FTP_RC_SYS_STATUS                 6
#define FTP_RC_DIR_STATUS                 7
#define FTP_RC_FILE_STATUS                8
#define FTP_RC_HELP_MESSAGE               9
#define FTP_RC_SYSTEM_NAME                10
#define FTP_RC_SERVICE_READY              11
#define FTP_RC_SERVICE_CLOSING            12
#define FTP_RC_DATA_OPEN_NO_TRANSFER      13
#define FTP_RC_DATA_CLOSE                 14
#define FTP_RC_PASSIVE_MODE               15
#define FTP_RC_USER_LOGGED                16
#define FTP_RC_FILE_ACTION_OK             17
#define FTP_RC_PATHNAME_CREATED           18
#define FTP_RC_USER_NAME_OK               19
#define FTP_RC_NEED_ACCOUNT               20
#define FTP_RC_FILE_ACTION_PENDING        21
#define FTP_RC_SERVICE_NOT_AVAILABLE      22
#define FTP_RC_DATA_OPEN_FAIL             23
#define FTP_RC_CONNECTION_CLOSE           24
#define FTP_RC_FILE_ACTION_NOT_TAKEN      25
#define FTP_RC_ACTION_ABORTED             26
#define FTP_RC_ACTION_NOT_TAKEN           27
#define FTP_RC_SYNTAX_ERROR               28
#define FTP_RC_SYNTAX_ERROR_PARAM         29
#define FTP_RC_COMMAND_NOT_IMPLEMENTED    30
#define FTP_RC_BAD_SEQUENCE               31
#define FTP_RC_BAD_PARAMETER              32
#define FTP_RC_LOGIN_FAILED               33
#define FTP_RC_NEED_ACCOUNT_STORE         34
#define FTP_RC_NO_AUTHORISATION           35
#define FTP_RC_PAGE_TYPE_UKNOW            36
#define FTP_RC_BAD_STORAGE_ALLOC          37
#define FTP_RC_FILE_NAME_NOT_ALLOWED      38
#define FTP_RC_MAX                        39

typedef struct {
    char *code;                         /*  Response code, 3 digits          */
    char *name;                         /*  Response text                    */
} FTP_RESPONSE;

#if (defined (DEFINE_FTP_TABLES))
FTP_RESPONSE                            /*  Some strings are in smtftpl.c    */
ftp_response [] = {
    { "110", "Restart marker reply"                                          },
    { "120", "Service ready in %d minutes"                                   },
    { "125", "Data connection already open; transfer starting"               },
    { "150", "" /*  Prepare transfer    */                                   },
    { "200", "Command okay"                                                  },
    { "202", "Command not implemented"                                       },
    { "211", "System status, or system help reply"                           },
    { "212", "Directory status"                                              },
    { "213", "" /*  File status         */                                   },
    { "214", "" /*  Help message        */                                   },
    { "215", "" /*  System type         */                                   },
    { "220", "" /*  Ready for login     */                                   },
    { "221", "Closing connection"                                            },
    { "225", "Data connection open; no transfer in progress"                 },
    { "226", "Closing data connection"                                       },
    { "227", "" /*  Passive mode        */                                   },
    { "230", "" /*  User logged-in      */                                   },
    { "250", "Requested file action okay, completed"                         },
    { "257", "" /*  Pathname created    */                                   },
    { "331", "" /*  Password required   */                                   },
    { "332", "Need account for login"                                        },
    { "350", "Requested file action pending further information"             },
    { "421", "Service not available at present"                              },
    { "425", "Can't open data connection"                                    },
    { "426", "Connection closed; transfer aborted"                           },
    { "450", "Requested file action not taken"                               },
    { "451", "Requested action aborted: local error in processing"           },
    { "452", "Requested action not taken; insufficient storage space"        },
    { "500", "" /*  Syntax error        */                                   },
    { "501", "Syntax error in parameters or arguments"                       },
    { "502", "" /*  Not implemented     */                                   },
    { "503", "" /*  Bad sequence        */                                   },
    { "504", "Command not implemented for that parameter"                    },
    { "530", "" /*  Login failed    */                                       },
    { "532", "Need account for storing files"                                },
    { "550", "This command is not permitted"                                 },
    { "551", "Failed: page type unknown"                                     },
    { "552", "Failed: aborted"                                               },
    { "553", "Failed: filename is not allowed"                               },
    };
#else
extern FTP_RESPONSE ftp_response [];
#endif


/*- Permission bits ---------------------------------------------------------*/

#define FTP_AUTH_GET     0x0001
#define FTP_AUTH_PUT     0x0002
#define FTP_AUTH_DELETE  0x0004
#define FTP_AUTH_MKDIR   0x0008
#define FTP_AUTH_RMDIR   0x0010
#define FTP_AUTH_ALL     0xFFFF


/*- File types --------------------------------------------------------------*/

#define FTP_TYPE_ASCII  0x01
#define FTP_TYPE_EBCDIC 0x02
#define FTP_TYPE_IMAGE  0x04
#define FTP_TYPE_NONPRT 0x08
#define FTP_TYPE_TELNET 0x10
#define FTP_TYPE_CAR    0x20
#define FTP_TYPE_LOCBYT 0x40


/*- Structure definitions ---------------------------------------------------*/

typedef struct {
    SYMTAB
        *users,                         /*  Users table                      */
        *direct,                        /*  Directory access table           */
        *config;                        /*  Configuration values             */
    char
        *user,                          /*  User name                        */
        *password,                      /*  Required password                */
        *guestname,                     /*  Supplied anonymous password      */
        *user_root,                     /*  User root directory              */
        *real_root,                     /*  As physical file name            */
        *cur_dir,                       /*  Current directory                */
        *last_file,                     /*  Last protected resource          */
        *file_name,                     /*  File name                        */
        *parameters,                    /*  Command parameters               */
         command [5];                   /*  Command (3-4 characters)         */
    Bool
        root_user,                      /*  TRUE if root user                */
        login,                          /*  TRUE if login                    */
        passive_connect,                /*  TRUE if passive connection       */
        temp_file;                      /*  Temporary files                  */
    dbyte
        last_permissions,               /*  Last permissions                 */
        file_type,                      /*  File type                        */
        return_code,                    /*  Return message                   */
        port,                           /*  Control port                     */
        data_port;                      /*  Port number of data connection   */
    qbyte
        file_size,                      /*  File size                        */
        ip_address,                     /*  IP adress of control connection  */
        timeout,                        /*  Wait input timeout               */
        data_host;                      /*  Host address of data connection  */
} FTP_CONTROL_CTX;

typedef struct {
    char
        *file_name,                     /*  File name                        */
        *parameters;                    /*  Command parameters               */
    dbyte
        command,                        /*  Ftp command                      */
        file_type,                      /*  File type                        */
        timeout,                        /*  Wait input timeout               */
        data_port;                      /*  Port number of data connection   */
    Bool
        passive_connect;                /*  TRUE if passive connection       */
    qbyte
        id,                             /*  ID of control connection         */
        data_host;                      /*  Host address of data connection  */
} FTP_DATA_CTX;


/*- Function prototypes -----------------------------------------------------*/

#ifdef __cplusplus
extern "C" {
#endif

/*  Ftp control functions                                                    */

void  ftp_initialise       (void);
void  ftpc_init_connection (FTP_CONTROL_CTX *);
void  ftpc_free_connection (FTP_CONTROL_CTX *);
void  ftpc_set_type        (FTP_CONTROL_CTX *);
void  ftpc_get_port        (FTP_CONTROL_CTX *);
Bool  ftpc_get_user        (FTP_CONTROL_CTX *);
Bool  ftpc_get_password    (FTP_CONTROL_CTX *);
Bool  ftpc_verify_password (FTP_CONTROL_CTX *);
dbyte ftpc_permissions     (FTP_CONTROL_CTX *, char *filename);
void  ftpc_get_rootdir     (FTP_CONTROL_CTX *, char *rootdir);
void  ftpc_mkdir           (FTP_CONTROL_CTX *);
void  ftpc_rmdir           (FTP_CONTROL_CTX *);
void  ftpc_chdir           (FTP_CONTROL_CTX *);
void  ftpc_cdup            (FTP_CONTROL_CTX *);
void  ftpc_return_message  (FTP_CONTROL_CTX *, char *buffer);
void  ftpc_file_name       (FTP_CONTROL_CTX *, char *buffer);
Bool  ftpc_dir_name        (FTP_CONTROL_CTX *, char *buffer);
Bool  ftpc_make_listing    (FTP_CONTROL_CTX *, Bool full);
char *ftpc_map_name        (FTP_CONTROL_CTX *, char *external);

/*  Ftp data functions                                                       */

void ftpd_init (FTP_DATA_CTX *p_ftpd);
void ftpd_free (FTP_DATA_CTX *p_ftpd);

#ifdef __cplusplus
}

#endif

#endif
