/*  ----------------------------------------------------------------<Prolog>-
    Name:       smtftpc.c
    Title:      SMT FTP control agent
    Package:    Libero SMT 2.x

    Written:    96/07/22  iMatix SMT kernel team <smt@imatix.com>
    Revised:    98/01/25

    Copyright:  Copyright (c) 1991-98 iMatix
    License:    This is free software; you can redistribute it and/or modify
                it under the terms of the SMT License Agreement as provided
                in the file LICENSE.TXT.  This software is distributed in
                the hope that it will be useful, but without any warranty.
 ------------------------------------------------------------------</Prolog>-*/

#include "smtdefn.h"                    /*  SMT definitions                  */
#include "smthttpl.h"                   /*  SMT HTTP definitions             */
#include "smtftpl.h"                    /*  SMT FTP declarations             */
#include "smtmsg.h"                     /*  SMT message functions            */


/*- Definitions -------------------------------------------------------------*/

#ifdef AGENT_NAME
#undef AGENT_NAME
#endif

#define AGENT_NAME   SMT_FTPC           /*  Our public name                  */
#define BUFFER_SIZE  4096

#define FTPCONFIG(s)    http_config (tcb-> ftp_control.config, (s))

typedef struct                          /*  Thread context block:            */
{
    HTTP_STAT
        stats;                          /*    Statistics structure           */
    sock_t
        handle;                         /*    Handle for i/o                 */
    int
        readsize,                       /*    Size of input message          */
        portbase;                       /*    Index for portbase attempts    */
    char
        *buffer;                        /*    Buffer for i/o                 */
    event_t
        thread_type;                    /*    Thread type indicator          */
    FTP_CONTROL_CTX
        ftp_control;                    /*    FTP control context            */
    dbyte
        data_port;                      /*    Default data port              */
} TCB;


/*- Function prototypes -----------------------------------------------------*/

static void  open_server_access_log (THREAD *thread);
static void  open_server_error_log  (THREAD *thread);
static int   cycle_mode             (const char *key, Bool startup);
static char *time_str               (void);
static void  count_transfer         (long size);


/*- Global variables used in this source file only --------------------------*/

static TCB
    *tcb;                               /*  Address thread contect block     */
static QID
    alogq,                              /*  Access log queue, or null        */
    elogq,                              /*  Error log queue, or null         */
    operq,                              /*  Operator console event queue     */
    sockq,                              /*  Socket agent event queue         */
    dataq,                              /*  FTP data agent queue             */
    timeq;                              /*  Timer agent event queue          */
static AGENT
    *this_agent;                        /*  Handle to ourselves              */
static THREAD
    *alog_thread,                       /*  Access log thread                */
    *elog_thread;                       /*  Error log thread                 */
static char
    *arg_rootdir = NULL,                /*  Rootdir to use (not allocated)   */
    *cur_rootdir = NULL,                /*  Effective root directory         */
    *master_port = NULL;                /*  Port for http service            */
static Bool
    alogq_owner,                        /*  Did we create access log queue?  */
    elogq_owner;                        /*  Did we create error log queue?   */

#define MSG_MAX BUFFER_SIZE + 64

static byte
    msg_body [MSG_MAX];                 /*  Messages sent to other agents    */
static int
    msg_size;                           /*  Size of formatted msg_body       */
static DESCR                            /*  Descriptor for exdr_writed()     */
    msg = { MSG_MAX,  msg_body };
static char
    buffer [BUFFER_SIZE];               /*  General-use string buffer        */
extern SYMTAB
    *config;                            /*  Global config file               */
static SYMTAB
    *users = NULL,                      /*  Global users file                */
    *direct = NULL;                     /*  Global directory access file     */

#include "smtftpc.d"                    /*  Include dialog data              */


/********************   INITIALISE AGENT - ENTRY POINT   *********************/

int
smtftpc_init (char *p_rootdir)          /*  Server root directory            */
{
    AGENT  *agent;                      /*  Handle for our agent             */
    THREAD *thread;                     /*  Handle to various threads        */

#   include "smtftpc.i"                 /*  Include dialog interpreter       */

    /*                      Method name      Event value     Priority        */
    /*  Shutdown event comes from Kernel                                     */
    method_declare (agent, "SHUTDOWN",       shutdown_event,
                                             SMT_PRIORITY_MAX);

    /*  Restart can come from control panel and HTTP agent                   */
    method_declare (agent, "RESTART",        restart_event,   0);
    method_declare (agent, "RELOAD",         reload_event,    0);

    /*  Reply events from socket agent                                       */
    method_declare (agent, "SOCK_INPUT_OK",  input_ok_event,     0);
    method_declare (agent, "SOCK_OUTPUT_OK", ok_event,           0);
    method_declare (agent, "SOCK_READ_OK",   ok_event,           0);
    method_declare (agent, "SOCK_WRITE_OK",  ok_event,           0);
    method_declare (agent, "SOCK_CLOSED",    sock_closed_event,  0);
    method_declare (agent, "SOCK_ERROR",     sock_error_event,   0);
    method_declare (agent, "SOCK_TIMEOUT",   sock_timeout_event, 0);

    /*  Reply events from timer agent                                        */
    method_declare (agent, "TIME_ALARM",     timeout_event,
                                             SMT_PRIORITY_LOW);
    /*  Reply events from FTP data agent                                     */
    method_declare (agent, "FTPD_CONNECTED", connected_event,
                                             SMT_PRIORITY_LOW);
    method_declare (agent, "FTPD_ERROR",     error_event,
                                             SMT_PRIORITY_LOW);
    method_declare (agent, "FTPD_END",       end_event,
                                             SMT_PRIORITY_HIGH);
    method_declare (agent, "FTPD_PASS_OK",   ok_event,
                                             SMT_PRIORITY_LOW);
    method_declare (agent, "FTPD_ABORTED",   aborted_event,
                                             SMT_PRIORITY_HIGH);

    /*  Private methods used to pass initial thread events                   */
    method_declare (agent, "_MASTER",        master_event,    0);
    method_declare (agent, "_CLIENT",        client_event,    0);

    /*  Private methods used between threads                                 */
    method_declare (agent, "_CANCEL",        cancel_event,    0);

    /*  Ensure that operator console is running, else start it up            */
    smtoper_init ();
    if ((thread = thread_lookup (SMT_OPERATOR, "")) != NULL)
        operq = thread-> queue-> qid;
    else
        return (-1);

    /*  Ensure that socket i/o agent is running, else start it up            */
    smtsock_init ();
    if ((thread = thread_lookup (SMT_SOCKET, "")) != NULL)
        sockq = thread-> queue-> qid;
    else
        return (-1);

    /*  Ensure that timer agent is running, else start it up                 */
    smttime_init ();
    if ((thread = thread_lookup (SMT_TIMER, "")) != NULL)
        timeq = thread-> queue-> qid;
    else
        return (-1);

    /*  Ensure that FTP data agent is running, else start it up              */
    smtftpd_init ();
    if ((thread = thread_lookup (SMT_FTPD, "")) != NULL)
        dataq = thread-> queue-> qid;
    else
        return (-1);

    /*  Create initial thread to manage master port                          */
    if ((thread = thread_create (AGENT_NAME, "main")) != NULL)
      {
        SEND (&thread-> queue-> qid, "_MASTER", "");
        ((TCB *) thread-> tcb)-> thread_type = master_event;
        ((TCB *) thread-> tcb)-> handle      = 0;
      }
    else
        return (-1);

    this_agent  = agent;                /*  Handle to ourselves              */
    arg_rootdir = p_rootdir;            /*  Get root directory               */
    ftp_initialise ();

    /*  Signal to caller that we initialised okay                            */
    return (0);
}


/*************************   INITIALISE THE THREAD   *************************/

MODULE initialise_the_thread (THREAD *thread)
{
    char
        *config_ip;
    long
        max_threads;

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> buffer   = NULL;
    tcb-> portbase = 0;                 /*  Not serving any port             */
    ftpc_init_connection (&tcb-> ftp_control);

    tcb-> ftp_control.config  = config;
    tcb-> ftp_control.users   = users;
    tcb-> ftp_control.direct  = direct;
    tcb-> ftp_control.timeout = atol (FTPCONFIG ("ftp:timeout")) * 100;

    /*  Set timeout to minimum 10 sec                                        */
    if (tcb-> ftp_control.timeout > 0
    &&  tcb-> ftp_control.timeout < 1000)
        tcb-> ftp_control.timeout = 1000;

    /*  Set IP address of control thread                                     */
    if (*FTPCONFIG ("ftp:force_ip") == '1')
        config_ip = FTPCONFIG ("ftp:ipaddress");
    else
        config_ip = socket_localaddr (tcb-> handle);

    tcb-> ftp_control.ip_address = ntohl (inet_addr (config_ip));

    /*  Get default data port in passive connection mode                     */
    tcb-> data_port = (dbyte) atoi (FTPCONFIG ("ftp:data_port"));

    if (tcb-> thread_type == client_event)
      {
        tcb-> stats.client   = TRUE;
        tcb-> stats.socket   = tcb-> handle;
        tcb-> stats.username = "";
      }
    else
      {
        tcb-> stats.client = FALSE;

        /*  Get maximum connections                                          */
        max_threads = atoi (CONFIG ("ftp:limit"));
        if (max_threads > 0)
            this_agent-> max_threads = max_threads + 1;
      }
}


/**************************   LOAD FTP USERS FILE   **************************/

MODULE load_ftp_users_file (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (config)
      {
        /*  Get list of users and directory access permissions               */
        users  = ini_dyn_load (users,  CONFIG ("ftp:user_file"));
        direct = ini_dyn_load (direct, CONFIG ("ftp:directory_file"));
      }
}


/*************************   OPEN SERVER LOG FILES   *************************/

MODULE open_server_log_files (THREAD *thread)
{
    open_server_access_log (thread);
    open_server_error_log (thread);
}

static void
open_server_access_log (THREAD *thread)
{
    static QID
        null_qid = { 0, 0 };
    char
        *access_log;
    Bool
        access_logging;

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    access_log     =  CONFIG ("ftplog:filename");
    access_logging = *CONFIG ("ftplog:enabled") == '1';

    alogq_owner = FALSE;
    if (access_logging)
      {
        /*  Create access logging thread, if it does not already exist       */
        alog_thread = thread_lookup (SMT_LOGGING, access_log);
        if (alog_thread == NULL)
          {
            /*  Cycle log file if required                                   */
            file_cycle (access_log, cycle_mode ("ftplog:cycle", 1));

            /*  Open access log file                                         */
            alog_thread = thread_create (SMT_LOGGING, access_log);
            sendfmt (&alog_thread-> queue-> qid, "APPEND", "");
            sendfmt (&alog_thread-> queue-> qid, "PLAIN", "");
            alogq_owner = TRUE;
          }
        alogq = alog_thread-> queue-> qid;
      }
    else
        alogq = null_qid;
}

static void
open_server_error_log (THREAD *thread)
{
    static QID
        null_qid = { 0, 0 };
    char
        *error_log;
    Bool
        error_logging;

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    error_log      =  CONFIG ("ftperrlog:filename");
    error_logging  = *CONFIG ("ftperrlog:enabled") == '1';

    elogq_owner = FALSE;
    if (error_logging)
      {
        /*  Create error logging thread, if it does not already exist        */
        elog_thread = thread_lookup (SMT_LOGGING, error_log);
        if (elog_thread == NULL)
          {
            file_cycle (error_log, cycle_mode ("ftperrlog:cycle", 1));
            elog_thread = thread_create (SMT_LOGGING, error_log);
            sendfmt (&elog_thread-> queue-> qid, "APPEND", "");
            sendfmt (&elog_thread-> queue-> qid, "PLAIN", "");
            elogq_owner = TRUE;
          }
        elogq = elog_thread-> queue-> qid;
      }
    else
        elogq = null_qid;
}


/*  -------------------------------------------------------------------------
 *  cycle_mode
 *
 *  Searches the config table for a cycle mode for the specified logfile.
 *  Returns one of the sflfile.h CYCLE_ constants; default if no entry is
 *  defined is CYCLE_NEVER.  If startup argument is FALSE, treats the
 *  'startup' mode as 'never'.
 */

static int
cycle_mode (const char *key, Bool startup)
{
    char
        *cycle_mode;                    /*  How do we cycle the file?        */
    int
        cycle_how;                      /*  Cycle mode                       */

    cycle_mode = sym_get_value (config, key, "");
    if (streq (cycle_mode, "startup") && startup)
        cycle_how = CYCLE_ALWAYS;
    else
    if (streq (cycle_mode, "hourly"))
        cycle_how = CYCLE_HOURLY;
    else
    if (streq (cycle_mode, "daily"))
        cycle_how = CYCLE_DAILY;
    else
    if (streq (cycle_mode, "weekly"))
        cycle_how = CYCLE_WEEKLY;
    else
    if (streq (cycle_mode, "monthly"))
        cycle_how = CYCLE_MONTHLY;
    else
        cycle_how = CYCLE_NEVER;

    return (cycle_how);
}


/************************   RELOAD CONFIG IF NEEDED   ************************/

MODULE reload_config_if_needed (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (ini_dyn_changed (users))
      {
        sym_empty_table (users);
        users = ini_dyn_load (users, CONFIG ("ftp:user_file"));
      }
    if (ini_dyn_changed (direct))
      {
        sym_empty_table (direct);
        direct = ini_dyn_load (direct, CONFIG ("ftp:directory_file"));
      }
}


/************************   REOPEN SERVER LOG FILES   ************************/

MODULE reopen_server_log_files (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (alogq_owner
    &&  strneq (alog_thread-> name, CONFIG ("ftplog:filename")))
      {
        sendfmt (&alogq, "CLOSE", "");
        open_server_access_log (thread);
      }
    if (elogq_owner
    &&  strneq (elog_thread-> name, CONFIG ("ftperrlog:filename")))
      {
        sendfmt (&elogq, "CLOSE", "");
        open_server_error_log (thread);
      }
}


/**************************   CHECK ROOTDIR EXISTS   *************************/

MODULE check_rootdir_exists (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    /*  Resolve options that can come both from command-line & config file:
     *  If these were specified on the command-line or control panel AND
     *  were not the same as the default values, we look in the config
     *  file.  If both the command-line and config file specify non-default
     *  values, the command-line wins.
     */
    if (streq (arg_rootdir, DEFAULT ("ftp:root")))
        arg_rootdir = CONFIG ("ftp:root");

    mem_strfree (&cur_rootdir);
    cur_rootdir = mem_strdup (resolve_path (arg_rootdir));
    if (strlast (cur_rootdir) == '/')
        strlast (cur_rootdir) = '\0';

    if (!file_is_directory (cur_rootdir))
      {
        sprintf (server_message, "Root directory '%s' not found - %s",
                 cur_rootdir, strerror (errno));
        sendfmt (&operq, "ERROR", server_message);
        raise_exception (fatal_event);
      }
}


/***************************   OPEN MASTER SOCKET   **************************/

MODULE open_master_socket (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (*CONFIG ("server:ipaddress") != '*')
        ip_passive = inet_addr (CONFIG ("server:ipaddress"));

    if (ip_portbase == 0)
        ip_portbase = atoi (FTPCONFIG ("server:portbase"));

    /*  We try the main ip_portbase only                                     */
    master_port = FTPCONFIG ("ftp:port");
    tcb-> ftp_control.port = atoi (master_port) + ip_portbase;
    tcb-> handle    = passive_TCP (master_port, 5);
    sendfmt (&operq, "INFO",
                     "smtftpc: preparing for connections on port %d",
                      tcb-> ftp_control.port);

    if (tcb-> handle != INVALID_SOCKET)
        sendfmt (&operq, "INFO",
             "smtftpc: ready for FTP connections on port %d",
             tcb-> ftp_control.port);
    else
      {
        sendfmt (&operq, "ERROR",
                 "smtftpc: could not open FTP port %d",
                 tcb-> ftp_control.port);
        sendfmt (&operq, "ERROR",
                 "smtftpc: %s", connect_errlist [connect_error ()]);
        sendfmt (&operq, "ERROR",
                 "smtftpc: %s", sockmsg ());
        raise_exception (fatal_event);
        sprintf (server_message, "Could not open FTP port %d - %s (%s)",
                 tcb-> ftp_control.port, connect_errlist [connect_error ()],
                 sockmsg ());
      }
}


/************************   ACCEPT CLIENT CONNECTION   ***********************/

MODULE accept_client_connection (THREAD *thread)
{
    sock_t
        slave_socket;                   /*  Connected socket                 */
    THREAD
        *child_thread;                  /*  Handle to child threads          */

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    slave_socket = accept_socket (tcb-> handle);
    if (slave_socket != INVALID_SOCKET)
      {
        child_thread = thread_create (AGENT_NAME, "");
        if (child_thread)
          {
            SEND (&child_thread-> queue-> qid, "_CLIENT", "");
            ((TCB *) child_thread-> tcb)-> thread_type = client_event;
            ((TCB *) child_thread-> tcb)-> handle      = slave_socket;
            connect_count++;            /*  Maintain statistics              */
            cur_connects++;             /*  One more active connection       */
            if (max_connects < cur_connects)
                max_connects = cur_connects;
          }
        else
            close_socket (slave_socket);
      }
    else
    if (sockerrno != EAGAIN && sockerrno != EWOULDBLOCK)
      {
        sprintf (server_message, "Network problem: %s", sockmsg ());
        sendfmt (&operq, "ERROR",
                 "smtftpc: could not accept connection: %s", sockmsg ());
        raise_exception (exception_event);
      }
}


/************************   CHECK CLIENT IP ALLOWED   ************************/

MODULE check_client_ip_allowed (THREAD *thread)
{
    char
        *webmask;                       /*  Permitted mask for connection    */

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    webmask = FTPCONFIG ("ftp:webmask");
    if (streq (webmask, "local"))
        webmask = socket_localaddr (tcb-> handle);

    if (!socket_is_permitted (socket_peeraddr (tcb-> handle), webmask))
        raise_exception (unauthorised_event);
}


/*************************   PREPARE CLIENT THREAD   *************************/

MODULE prepare_client_thread (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> buffer = mem_alloc (BUFFER_SIZE + 1);
    if (tcb-> buffer == NULL)
      {
        sendfmt (&operq, "ERROR", "smtftpc: out of memory");
        raise_exception (exception_event);
      }
    else
        memset (tcb-> buffer, 0, BUFFER_SIZE + 1);
}


/*************************   WAIT FOR SOCKET INPUT   *************************/

MODULE wait_for_socket_input (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    msg_size = exdr_writed (&msg, SMT_SOCK_INPUT, (word) 0,
                            tcb-> handle, (qbyte) 0);
    event_send (
        &sockq,                         /*  Send to specified queue          */
        &thread-> queue-> qid,          /*  Queue for reply                  */
        "INPUT",                        /*  Name of event to send            */
        msg_body, msg_size,             /*  Event body and size              */
        NULL, NULL, NULL,               /*  No response events               */
        0);                             /*  No timeout                       */
}


/**************************   PREPARE TO GET INPUT   *************************/

MODULE prepare_to_get_input (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> readsize = 0;
}


/**************************   CHECK IF MORE INPUT   **************************/

MODULE check_if_more_input (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    /*  Want CRLF or LF in buffer                                            */
    if (strchr (tcb-> buffer, '\n') == NULL)
        raise_exception (sock_retry_event);
}


/****************************   READ FTP REQUEST   ***************************/

MODULE read_ftp_request (THREAD *thread)
{
    int
        rc;                             /*  Return code from read            */

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    rc = read_TCP (tcb-> handle, tcb-> buffer + tcb-> readsize,
                                 BUFFER_SIZE - tcb-> readsize);
    if (rc > 0)                     /*  We read something                    */
      {
        tcb-> readsize += rc;
        tcb-> buffer [tcb-> readsize] = '\0';
      }
    else
    if (rc == 0 || sockerrno == EPIPE || sockerrno == ECONNRESET)
        raise_exception (sock_closed_event);
    else
    if (sockerrno == EAGAIN || sockerrno == EWOULDBLOCK)
        raise_exception (sock_retry_event);
    else
        raise_exception (sock_error_event);
}


/****************************   GET FTP COMMAND   ****************************/

MODULE get_ftp_command (THREAD *thread)
{
    static struct {
        char    *name;
        event_t  event;
    } command [] = {
      { "ABOR", abort_event        },   /*  Abort                            */
      { "ACCT", unsupported_event  },   /*                                   */
      { "ALLO", unsupported_event  },   /*                                   */
      { "APPE", unsupported_event  },   /*                                   */
      { "CDUP", cdup_event         },   /*  Change to parent directory       */
      { "CWD",  cwd_event          },   /*  Change working directory         */
      { "DELE", delete_event       },   /*  Delete a file                    */
      { "HELP", help_event         },   /*  Display server help              */
      { "LIST", list_event         },   /*  Return directory listing         */
      { "MAIL", unsupported_event  },   /*                                   */
      { "MKD",  mkdir_event        },   /*  Make directory                   */
      { "MLFL", unsupported_event  },   /*                                   */
      { "MODE", mode_event         },   /*  Stream mode                      */
      { "MRCP", unsupported_event  },   /*                                   */
      { "MRSQ", unsupported_event  },   /*                                   */
      { "MSAM", unsupported_event  },   /*                                   */
      { "MSND", unsupported_event  },   /*                                   */
      { "MSOM", unsupported_event  },   /*                                   */
      { "NLST", names_event        },   /*  Name list                        */
      { "NOOP", noop_event         },   /*  Null operation                   */
      { "PASS", password_event     },   /*  User password                    */
      { "PASV", passive_event      },   /*  Request passive data connection  */
      { "PORT", port_event         },   /*  Data port                        */
      { "PWD",  pwd_event          },   /*  Print current directory          */
      { "QUIT", quit_event         },   /*  Logout                           */
      { "REIN", unsupported_event  },   /*                                   */
      { "REST", unsupported_event  },   /*                                   */
      { "RETR", retrieve_event     },   /*  Retrieve file                    */
      { "RMD",  rmdir_event        },   /*  Remove directory                 */
      { "RNFR", rename_from_event  },   /*  Rename, source filename          */
      { "RNTO", rename_to_event    },   /*  Rename, destination filename     */
      { "SITE", unsupported_event  },   /*                                   */
      { "SIZE", size_event         },   /*  Get size of file                 */
      { "STAT", unsupported_event  },   /*                                   */
      { "STOR", store_event        },   /*  Store a file                     */
      { "STOU", unsupported_event  },   /*                                   */
      { "STRU", unsupported_event  },   /*                                   */
      { "SYST", system_event       },   /*  Return the OS type               */
      { "TYPE", type_event         },   /*  Representation type              */
      { "USER", user_event         },   /*  User name login                  */
      { "XCUP", unsupported_event  },   /*                                   */
      { "XCWD", cwd_event          },   /*  X change working directory       */
      { "XEXC", unsupported_event  },   /*                                   */
      { "XMKD", mkdir_event        },   /*  X Make directory                 */
      { "XPWD", pwd_event          },   /*  X print current directory        */
      { "XRMD", rmdir_event        },   /*  X Remove directory               */
      { "XSYS", unsupported_event  }    /*                                   */
     };
    char
        *cur_command,
        *line_ptr;
    int
        lower_limit,
        upper_limit,
        index;

    tcb = thread-> tcb;                 /*  Point to thread's context        */
    /*  Remove CR/LF at end of buffer                                        */
    line_ptr = tcb-> buffer + tcb-> readsize - 1;
    if (*line_ptr == '\n')
        *line_ptr-- = '\0';
    if (*line_ptr == '\r')
        *line_ptr-- = '\0';

    /*  If debug mode, trace the start of whatever was in the buffer         */
    strncpy (buffer, tcb-> buffer, 80);
    buffer [80] = '\0';
    trace ("FTP input: %s", buffer);

    /*  Get 4-char command from buffer                                       */
    cur_command = tcb-> ftp_control.command;
    memcpy (cur_command, tcb-> buffer, 4);
    if (tcb-> buffer [3] == ' ')
        cur_command [3] = '\0';
    else
        cur_command [4] = '\0';

    /*  Do binary search through the table                                   */
    lower_limit  = 0;
    upper_limit  = tblsize (command) - 1;
    index = (lower_limit + upper_limit) / 2;

    strupc (cur_command);               /*  Compare in uppercase             */
    until (streq (command [index].name, cur_command)
    ||     lower_limit > upper_limit)
      {
        index = (lower_limit + upper_limit) / 2;
        if (strcmp (command [index].name, cur_command) > 0)
            upper_limit = index - 1;
        else
        if (strcmp (command [index].name, cur_command) < 0)
            lower_limit = index + 1;
      }
    if (lower_limit <= upper_limit)     /*  Found?                           */
      {
        /*  Get parameters off original command, if they're present          */
        strcrop (tcb-> buffer);         /*  Remove trailing spaces if any    */
        line_ptr = strchr (tcb-> buffer, ' ');
        if (line_ptr)
          {
            *line_ptr++ = '\0';
            strconvch (line_ptr, '\\', '/');
            strunique (line_ptr, '/');
          }
        tcb-> ftp_control.parameters  = line_ptr;
        tcb-> ftp_control.return_code = FTP_RC_COMMAND_OK;
        the_next_event = command [index].event;
      }
    else
      {
        /*  Point parameters field to original command, for error message    */
        tcb-> ftp_control.parameters = tcb-> buffer;
        the_next_event = invalid_event;
      }
}


/**************************   SIGNAL SOCKET ERROR   **************************/

MODULE signal_socket_error (THREAD *thread)
{
    char
        *message = NULL;

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    /*  Pick up message from event body, encoded as SMT_SOCK_ERROR           */
    exdr_read (thread-> event-> body, SMT_SOCK_ERROR, &message, NULL, NULL);

    sendfmt (&operq, "ERROR",
             "smtftpc: error on socket %d: %s", (int) tcb-> handle, message);
    mem_free (message);
}


/**************************   WRITE RETURN MESSAGE   *************************/

MODULE write_return_message (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    ftpc_return_message (&tcb-> ftp_control, buffer);
    msg_size = exdr_writed (&msg, SMT_SOCK_WRITE,
        0,                              /*  Timeout for request              */
        tcb-> handle,                   /*  Socket to write to               */
        strlen (buffer),                /*  Amount of data to write          */
        buffer,                         /*  Address of data block            */
        (qbyte) 0);                     /*  No request tag                   */

    event_send (
        &sockq,                         /*  Send to specified queue          */
        &thread-> queue-> qid,          /*  Queue for reply                  */
        "WRITE",                        /*  Name of event to send            */
        msg_body, msg_size,             /*  Event body and size              */
        NULL, NULL, NULL,               /*  No response events               */
        0);                             /*  No timeout                       */
}


/***************************   CHECK USER PROFILE   **************************/

MODULE check_user_profile (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (ftpc_get_user     (&tcb-> ftp_control)
    &&  ftpc_get_password (&tcb-> ftp_control))
      {
        /*  Empty password means we can consider the user logged-in          */
        if (strnull (tcb-> ftp_control.password))
            the_next_event = not_protected_event;
        else
            the_next_event = ok_event;
      }
    else
        the_next_event = error_event;   /*  Fatal error, e.g. in user file   */
}


/**************************   CHECK USER PASSWORD   **************************/

MODULE check_user_password (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (ftpc_verify_password (&tcb-> ftp_control))
        the_next_event = ok_event;
    else
        the_next_event = invalid_event;
}


/************************   INITIALISE USER SESSION   ************************/

MODULE initialise_user_session (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> stats.username = tcb-> ftp_control.user;
    tcb-> ftp_control.login = TRUE;
    ftpc_get_rootdir (&tcb-> ftp_control, cur_rootdir);
}


/************************   SET REPRESENTATION TYPE   ************************/

MODULE set_representation_type (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    ftpc_set_type (&tcb-> ftp_control);
}


/*****************************   GET DATA PORT   *****************************/

MODULE get_data_port (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    ftpc_get_port (&tcb-> ftp_control);
}


/**********************   MAKE FULL DIRECTORY LISTING   **********************/

MODULE make_full_directory_listing (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (!ftpc_make_listing (&tcb-> ftp_control, TRUE))
        raise_exception (error_event);
}


/**********************   MAKE BRIEF DIRECTORY LISTING   *********************/

MODULE make_brief_directory_listing (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (!ftpc_make_listing (&tcb-> ftp_control, FALSE))
        raise_exception (error_event);
}


/************************   CHECK IF TEMPORARY FILE   ************************/

MODULE check_if_temporary_file (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (tcb-> ftp_control.temp_file)
      {
        if (tcb-> ftp_control.file_name)
            file_delete (tcb-> ftp_control.file_name);
        tcb-> ftp_control.temp_file = FALSE;
      }
}


/*****************************   MAKE FILE NAME   ****************************/

MODULE make_file_name (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    ftpc_file_name (&tcb-> ftp_control, buffer);
    mem_strfree (&tcb-> ftp_control.file_name);
    tcb-> ftp_control.file_name
        = mem_strdup (ftpc_map_name (&tcb-> ftp_control, buffer));
}


/**************************   CHECK IF FILE EXISTS   *************************/

MODULE check_if_file_exists (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (file_exists       (tcb-> ftp_control.file_name)
    && !file_is_directory (tcb-> ftp_control.file_name))
      {
        the_next_event = ok_event;
        tcb-> ftp_control.file_size =
            get_file_size (tcb-> ftp_control.file_name);
      }
    else
        the_next_event = unauthorised_event;
}


/****************************   DELETE THE FILE   ****************************/

MODULE delete_the_file (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (tcb-> ftp_control.file_name == NULL)
        tcb-> ftp_control.return_code = FTP_RC_SYNTAX_ERROR_PARAM;
    else
    if ((ftpc_permissions (&tcb-> ftp_control, tcb-> ftp_control.file_name)
    &    FTP_AUTH_DELETE) == 0)
        raise_exception (unauthorised_event);
    else
      {
        if (file_delete (tcb-> ftp_control.file_name) == 0)
            tcb-> ftp_control.return_code = FTP_RC_FILE_ACTION_OK;
        else
            tcb-> ftp_control.return_code = FTP_RC_FILE_ACTION_NOT_TAKEN;
      }
}


/****************************   RENAME THE FILE   ****************************/

MODULE rename_the_file (THREAD *thread)
{
    char
        *destname = NULL;               /*  Destination file name            */

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    /*  Get destination file name into buffer                                */
    ftpc_file_name (&tcb-> ftp_control, buffer);
    destname = ftpc_map_name (&tcb-> ftp_control, buffer);

    if (tcb-> ftp_control.file_name == NULL)
        tcb-> ftp_control.return_code = FTP_RC_SYNTAX_ERROR_PARAM;
    else
    if ((ftpc_permissions (&tcb-> ftp_control, tcb-> ftp_control.file_name)
    &    FTP_AUTH_DELETE) == 0
    ||  (ftpc_permissions (&tcb-> ftp_control, buffer)
    &    FTP_AUTH_PUT) == 0)
        raise_exception (unauthorised_event);
    else
      {
        if (file_rename (tcb-> ftp_control.file_name, destname) == 0)
            tcb-> ftp_control.return_code = FTP_RC_FILE_ACTION_OK;
        else
            tcb-> ftp_control.return_code = FTP_RC_FILE_ACTION_NOT_TAKEN;
      }
}


/****************************   SET STREAM MODE   ****************************/

MODULE set_stream_mode (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */
    if (*tcb-> ftp_control.parameters == 'S'
    ||  *tcb-> ftp_control.parameters == 's')
        tcb-> ftp_control.return_code = FTP_RC_BAD_PARAMETER;
}


/*********************   CHECK READY FOR FILE TRANSFER   *********************/

MODULE check_ready_for_file_transfer (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (tcb-> ftp_control.passive_connect == FALSE
    &&  tcb-> ftp_control.data_host       == 0
    &&  tcb-> ftp_control.data_port       == 0)
        raise_exception (unexpected_event);
}


/***********************   SEND PUT DATA FILE REQUEST   **********************/

MODULE send_put_data_file_request (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (tcb-> ftp_control.temp_file == FALSE
    && (ftpc_permissions (&tcb-> ftp_control, tcb-> ftp_control.file_name)
    &    FTP_AUTH_GET) == 0)
      {
        raise_exception (unauthorised_event);
        return;
      }
    msg_size = exdr_writed (&msg, SMT_FTPD_PUTF,
        thread-> thread_id,                /*  id of connection thread       */
        tcb-> ftp_control.passive_connect, /*  passive or active connection  */
        tcb-> ftp_control.file_type,       /*  FTP file type                 */
        tcb-> ftp_control.file_name,       /*  File name                     */
        tcb-> ftp_control.data_host,       /*  Host value                    */
        tcb-> ftp_control.data_port);      /*  Port value                    */

    event_send (
        &dataq,                            /*  Send to specified queue       */
        &thread-> queue-> qid,             /*  Queue for reply               */
        "FTPD_PUT_FILE",                   /*  Name of event to send         */
        msg_body, msg_size,                /*  Event body and size           */
        NULL, NULL, NULL,                  /*  No response events            */
        0);                                /*  No timeout                    */
}


/***********************   SEND GET DATA FILE REQUEST   **********************/

MODULE send_get_data_file_request (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if ((ftpc_permissions (&tcb-> ftp_control, tcb-> ftp_control.file_name)
    &    FTP_AUTH_PUT) == 0)
      {
        raise_exception (unauthorised_event);
        return;
      }
    msg_size = exdr_writed (&msg, SMT_FTPD_GETF,
        thread-> thread_id,                /*  id of connection thread       */
        tcb-> ftp_control.passive_connect, /*  passive or active connection  */
        tcb-> ftp_control.file_type,       /*  FTP file type                 */
        tcb-> ftp_control.file_name,       /*  File name                     */
        tcb-> ftp_control.data_host,       /*  Host value                    */
        tcb-> ftp_control.data_port);      /*  Port value                    */
    event_send (
        &dataq,                            /*  Send to specified queue       */
        &thread-> queue-> qid,             /*  Queue for reply               */
        "FTPD_GET_FILE",                   /*  Name of event to send         */
        msg_body, msg_size,                /*  Event body and size           */
        NULL, NULL, NULL,                  /*  No response events            */
        0);                                /*  No timeout                    */
}


/***************************   SET TRANSFER ENDED   **************************/

MODULE set_transfer_ended (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> ftp_control.return_code     = FTP_RC_DATA_CLOSE;
    tcb-> ftp_control.passive_connect = FALSE;
    tcb-> ftp_control.data_host       = 0;
    tcb-> ftp_control.data_port       = 0;
}


/************************   CHANGE CURRENT DIRECTORY   ***********************/

MODULE change_current_directory (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    ftpc_chdir (&tcb-> ftp_control);
}


/***************************   DIRECTORY MOVE UP   ***************************/

MODULE directory_move_up (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    ftpc_cdup (&tcb-> ftp_control);
}


/***********************   MAKE OPEN PASSIVE MESSAGE   ***********************/

MODULE make_open_passive_message (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> ftp_control.passive_connect = TRUE;
    exdr_read (thread-> event-> body, SMT_FTPD_PASS_OK,
                &tcb-> ftp_control.data_host,
                &tcb-> ftp_control.data_port);
}


/**********************   OPEN PASSIVE DATA CONNECTION   *********************/

MODULE open_passive_data_connection (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    msg_size = exdr_writed (&msg, SMT_FTPD_PASSIVE, thread-> thread_id,
                            tcb-> data_port, tcb-> ftp_control.ip_address);
    event_send (
        &dataq,                            /*  Send to specified queue       */
        &thread-> queue-> qid,             /*  Queue for reply               */
        "FTPD_PASSIVE",                    /*  Name of event to send         */
        msg_body, msg_size,                /*  Event body and size           */
        NULL, NULL, NULL,                  /*  No response events            */
        0);                                /*  No timeout                    */
}


/*********************   SEND ABORT TO DATA CONNECTION   *********************/

MODULE send_abort_to_data_connection (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    msg_size = exdr_writed (&msg, SMT_FTPD_ABORT, thread-> thread_id);
    event_send (
        &dataq,                            /*  Send to specified queue       */
        &thread-> queue-> qid,             /*  Queue for reply               */
        "FTPD_ABORT",                      /*  Name of event to send         */
        msg_body, msg_size,                /*  Event body and size           */
        NULL, NULL, NULL,                  /*  No response events            */
        0);                                /*  No timeout                    */
}


/***************************   CANCEL DATA THREAD   **************************/

MODULE cancel_data_thread (THREAD *thread)
{
    THREAD
        *data_thread;

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    /* check if a ftp data thread is active for this connection thread
       if TRUE, send close control message to this thread                    */
    sprintf (buffer, "%ld", thread-> thread_id);
    if ((data_thread = thread_lookup (SMT_FTPD, buffer)) != NULL)
        SEND (&data_thread-> queue-> qid, "FTPD_CLOSECTRL", "");
}


/***************************   MAKE NEW DIRECTORY   **************************/

MODULE make_new_directory (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    ftpc_mkdir (&tcb-> ftp_control);
}


/****************************   REMOVE DIRECTORY   ***************************/

MODULE remove_directory (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    ftpc_rmdir (&tcb-> ftp_control);
}


/*****************************   FLUSH TIME OUT   ****************************/

MODULE flush_time_out (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (tcb-> ftp_control.timeout > 0)
        event_send (
            &timeq,                     /*  Send to specified queue          */
            &thread-> queue-> qid,      /*  Queue for reply                  */
            "FLUSH",                    /*  Name of event to send            */
            NULL, 0,                    /*  No event body                    */
            NULL, NULL, NULL,           /*  No response events               */
            0);                         /*  No timeout                       */
}


/******************************   SET TIME OUT   *****************************/

MODULE set_time_out (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (tcb-> ftp_control.timeout > 0)
      {
       /*  Ask timer to send us an event after the monitor timeout           */
        msg_size = exdr_writed (&msg, SMT_TIME_ALARM, (qbyte) 0, (qbyte)
                   tcb-> ftp_control.timeout, 0, (void *) NULL);
        event_send (
            &timeq,                     /*  Send to timer queue              */
            &thread-> queue-> qid,      /*  Queue for reply                  */
            "ALARM",                    /*  Name of event to send            */
            msg_body, msg_size,         /*  Event body and size              */
            NULL, NULL, NULL,           /*  No response events               */
            0);                         /*  No timeout                       */
      }
}


/**************************   WRITE SERVICE READY   **************************/

MODULE write_service_ready (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> ftp_control.return_code = FTP_RC_SERVICE_READY;
    write_return_message (thread);
}


/**************************   WRITE USER LOGGED IN   *************************/

MODULE write_user_logged_in (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> ftp_control.return_code = FTP_RC_USER_LOGGED;
    write_return_message (thread);
}


/*************************   WRITE INVALID PASSWORD   ************************/

MODULE write_invalid_password (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> ftp_control.return_code = FTP_RC_LOGIN_FAILED;
    write_return_message (thread);
}


/************************   WRITE OPEN PASSIVE OKAY   ************************/

MODULE write_open_passive_okay (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> ftp_control.return_code = FTP_RC_PASSIVE_MODE;
    write_return_message (thread);
}


/************************   WRITE CLOSING CONNECTION   ***********************/

MODULE write_closing_connection (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> ftp_control.return_code = FTP_RC_SERVICE_CLOSING;
    write_return_message (thread);
}


/***********************   WRITE USER NOT AUTHORISED   ***********************/

MODULE write_user_not_authorised (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> ftp_control.return_code = FTP_RC_NO_AUTHORISATION;
    write_return_message (thread);
}


/*************************   WRITE TIME OUT MESSAGE   ************************/

MODULE write_time_out_message (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> ftp_control.return_code = FTP_RC_SERVICE_NOT_AVAILABLE;
    write_return_message (thread);
}


/*************************   WRITE INVALID COMMAND   *************************/

MODULE write_invalid_command (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> ftp_control.return_code = FTP_RC_SYNTAX_ERROR;
    write_return_message (thread);
}


/***********************   WRITE UNSUPPORTED COMMAND   ***********************/

MODULE write_unsupported_command (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> ftp_control.return_code = FTP_RC_COMMAND_NOT_IMPLEMENTED;
    write_return_message (thread);
}


/************************   WRITE UNEXPECTED COMMAND   ***********************/

MODULE write_unexpected_command (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> ftp_control.return_code = FTP_RC_BAD_SEQUENCE;
    write_return_message (thread);
}


/**************************   WRITE PENDING ACTION   *************************/

MODULE write_pending_action (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> ftp_control.return_code = FTP_RC_FILE_ACTION_PENDING;
    write_return_message (thread);
}


/****************************   WRITE FILE SIZE   ****************************/

MODULE write_file_size (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> ftp_control.return_code = FTP_RC_FILE_STATUS;
    write_return_message (thread);
}


/**************************   WRITE NAME OF SYSTEM   *************************/

MODULE write_name_of_system (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> ftp_control.return_code = FTP_RC_SYSTEM_NAME;
    write_return_message (thread);
}


/************************   WRITE WORKING DIRECTORY   ************************/

MODULE write_working_directory (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> ftp_control.return_code = FTP_RC_PATHNAME_CREATED;
    write_return_message (thread);
}


/***************************   WRITE HELP MESSAGE   **************************/

MODULE write_help_message (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> ftp_control.return_code = FTP_RC_HELP_MESSAGE;
    write_return_message (thread);
}


/*************************   WRITE TRANSFER ABORTED   ************************/

MODULE write_transfer_aborted (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> ftp_control.return_code = FTP_RC_ACTION_ABORTED;
    write_return_message (thread);
}


/**************************   WRITE DATA CONNECTED   *************************/

MODULE write_data_connected (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> ftp_control.return_code = FTP_RC_FILE_OK;
    write_return_message (thread);
}


/***********************   WRITE OPEN PASSIVE FAILED   ***********************/

MODULE write_open_passive_failed (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> ftp_control.return_code = FTP_RC_DATA_OPEN_FAIL;
    write_return_message (thread);
}


/***********************   WRITE SERVICE UNAVAILABLE   ***********************/

MODULE write_service_unavailable (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    tcb-> ftp_control.return_code = FTP_RC_SERVICE_NOT_AVAILABLE;
    write_return_message (thread);
    event_wait ();                      /*  Before closing socket            */
}


/*******************************   LOG ACCESS   ******************************/

MODULE log_access (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    /*  Create log line in standard CERN/NCSA format:
     *
     *  host - user [date:time] "request" status size "refer" "password"
     */
    count_transfer (tcb-> ftp_control.file_size);
    strcpy (buffer, tcb-> ftp_control.command);
    if (tcb-> ftp_control.file_name)
        xstrcat (buffer, " ", tcb-> ftp_control.file_name, " ", NULL);

    strcat (buffer, "FTP");
    sendfmt (&alogq, "PUT",
        "%s - %s [%s] \"%s\" %s %ld \"ftp://%s/%s\" \"%s\"",
        socket_hostaddr (tcb-> handle),
        tcb-> ftp_control.user,
        time_str (),
        buffer,
        ftp_response [tcb-> ftp_control.return_code].code,
        tcb-> ftp_control.file_size,
        socket_hostaddr (tcb-> handle),
        tcb-> ftp_control.file_name? tcb-> ftp_control.file_name: "",
        tcb-> ftp_control.guestname? tcb-> ftp_control.guestname: "");

    /*  Anything >= 400 is an error...                                       */
    if (tcb-> ftp_control.return_code >= FTP_RC_SERVICE_NOT_AVAILABLE)
        sendfmt (&elogq, "PUT", "%s - %s [%s] %s \"%s::%s\" \"%s\"",
            socket_hostaddr (tcb-> handle),
            tcb-> ftp_control.user,
            time_str (),
            ftp_response [tcb-> ftp_control.return_code].code,
            "-",
            "ftp://",
            tcb-> ftp_control.file_name? tcb-> ftp_control.file_name: "");
}


/*  -------------------------------------------------------------------------
 *  time_str
 *
 *  Returns the current date and time formatted as: "DD/Mon/YYYY:hh:mm:ss".
 *  The formatted time is in a static string that each call overwrites.
 */

static char *
time_str (void)
{
    static char
        formatted_time [30],
        *months = "Jan\0Feb\0Mar\0Apr\0May\0Jun\0Jul\0Aug\0Sep\0Oct\0Nov\0Dec";
    time_t
        time_secs;
    struct tm
        *time_struct;

    time_secs   = time (NULL);
    time_struct = localtime (&time_secs);

    sprintf (formatted_time, "%02d/%s/%04d:%02d:%02d:%02d %s",
                              time_struct-> tm_mday,
                              months + time_struct-> tm_mon * 4,
                              time_struct-> tm_year + 1900,
                              time_struct-> tm_hour,
                              time_struct-> tm_min,
                              time_struct-> tm_sec,
                              timezone_string ());
    return (formatted_time);
}


static void
count_transfer (long size)
{
    static long
        transfer_bytes;                 /*  Count transfer in Kbytes         */

    /*  Keep track of Kbytes for display and individual bytes internally     */
    transfer_bytes += size;
    transfer_size  += transfer_bytes / 1024;
    transfer_bytes  = transfer_bytes % 1024;
}


/***************************   CHECK SOCKET TYPE   ***************************/

MODULE check_socket_type (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    the_next_event = tcb-> thread_type;
}


/*************************   CLOSE SERVER LOG FILES   ************************/

MODULE close_server_log_files (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    if (alogq_owner)
        SEND (&alogq, "CLOSE", "");
    if (elogq_owner)
        SEND (&elogq, "CLOSE", "");
}


/*************************   SIGNAL CLOSING SERVER   *************************/

MODULE signal_closing_server (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    sendfmt (&operq, "INFO", "smtftpc: closing FTP connections on port %d",
                              tcb-> ftp_control.port);
}


/***********************   FREE ALL SERVER RESOURCES   ***********************/

MODULE free_all_server_resources (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    /*  Free all resources used by the ftp server                            */
    if (users)
        sym_delete_table (users);
    users = NULL;

    if (direct)
        sym_delete_table (direct);
    direct = NULL;

    mem_strfree (&cur_rootdir);
    cur_rootdir = NULL;
}


/**********************   KILL ALL ACTIVE CONNECTIONS   **********************/

MODULE kill_all_active_connections (THREAD *thread)
{
    QUEUE
        *queue;                         /*  Task control block               */

    tcb = thread-> tcb;                 /*  Point to thread's context        */

    /*  We send a CANCEL event to all threads; the master thread ignores it  */
    for (queue = this_agent-> queues.next;
        (void *) queue != &(this_agent-> queues.next);
         queue = queue-> next)
        SEND (&queue-> qid, "_CANCEL", "");
}


/*************************   TERMINATE THE THREAD   **************************/

MODULE terminate_the_thread (THREAD *thread)
{
    tcb = thread-> tcb;                 /*  Point to thread's context        */

    ftpc_free_connection (&tcb-> ftp_control);
    mem_strfree (&tcb-> buffer);
    if (tcb-> handle)
      {
        close_socket (tcb-> handle);
        tcb-> handle = 0;
      }
    if (tcb-> thread_type == client_event)
        cur_connects--;                 /*  One less active connection       */

    the_next_event = terminate_event;
}
