/*
 *  Compile script for iMatix sources, for OS/2
 *
 *  This script will compile, link, or archive, one or more files.
 *  The syntax is a subset of the syntax of the c bourne shell script.
 *
 *  Written:    98/01/02  Ewen McNeill <ewen@imatix.com>
 *  Revised:    98/01/15  Ewen McNeill <ewen@imatix.com>
 *
 *  Syntax:     c filename...     Compile ANSI C program(s)
 *              c -l main         Compile and link main program(s)
 *              c -L main         Link main program(s), no compile
 *              c -r lib file     Replace file.o into library
 *              c -C              Print out full call to compiler
 *
 *  Bug reports & questions to <ewen@imatix.com>
 *  ---------------------------------------------------------------
 */
'@echo off'

/* Configuration options */
/* Use -Zexe so that we can use unix style makefiles. */
CC     = 'gcc'
CCOPTS = '-DDEBUG -O2 -Wall -pedantic -Zsysv-signals -Zexe'
OBJ    = '.o'
LIB    = '.a'
LIBRARIES = '-llibsfl -lsocket'
AR     = 'ar -rs' 

PARSE ARG commandline
IF commandline == "" THEN
DO
  SAY "Usage: c [-L] source"
  exit 1
END

first   = 1
compile = 1            /* Default is to compile, no link */
link    = 0
archive = 0

DO WHILE commandline \= ""
  PARSE VAR commandline thisarg commandline        /* Get next argument */
 
  IF (first = 1) & (compare(thisarg, "-l") = 0) THEN
  DO
    compile = 1
    link    = 1
    archive = 0
    PARSE VAR commandline thisarg commandline      /* Get next argument */
  END
  ELSE
    IF (first = 1) & (compare(thisarg, "-L") = 0) THEN
    DO 
      compile = 0
      link    = 1
      archive = 0
      PARSE VAR commandline thisarg commandline      /* Get next argument */
    END
    ELSE
      IF (first = 1) & (compare(thisarg, "-r") = 0) THEN
      DO 
        compile = 0
        link    = 0
        archive = 1
        PARSE VAR commandline library thisarg commandline  /* Get next arg */
      END
      ELSE
        IF (first = 1) & (compare(thisarg, "-C") = 0) THEN
        DO
          /* Report on the command line invocation, and exit        */
          /* We add the -D__EMX__ bit for the benefit of makedepend */
          /* And we pick up the C_INCLUDE_PATH item if defined and  */
          /* stick that on the end.                                 */
          includepath = value('C_INCLUDE_PATH',,'OS2ENVIRONMENT')
          if (includepath \= '') then includepath = "-I"includepath
          SAY CC "-c" CCOPTS "-D__EMX__" includepath
          exit 0
        END

  first = 0

  IF thisarg \= "" THEN
  DO
    IF compile = 1 THEN CALL CompileToObj thisarg
    IF link    = 1 THEN CALL LinkProgram  thisarg
    IF archive = 1 THEN CALL ArchiveObj   thisarg  library
  END
END

exit 0
/* End script */

/* Utility subroutines */

/* AssertFileExists <file>
 *
 * Checks to see if the specified file (argument 1) exists.  If it does the
 * subroutine returns.  If it doesn't, the subroutine displays an error
 * message, and exits the procedure.
 */

AssertFileExists:
fullpath = stream(ARG(1), 'C', 'QUERY EXISTS')
IF fullpath = "" THEN
DO
  SAY ARG(1) "missing -- aborting"
  exit 1
END

return

/* CompileToLib <source> <library>
 *
 * Compile C source file into object file, then put that object file into
 * the library.  If either of these steps fail, then abort the script.
 *
 * CC, CCOPTS, AR, OBJ, and LIB are REXX variables defined at the top
 * of the script.
 */

CompileToLib:

/* Check to make sure the symbols we require are defined  */
/* NOTE: commas at end of line to continue onto next line */
IF (symbol('CC') \= 'VAR') | (symbol('CCOPTS') \= 'VAR') |,
   (symbol('AR') \= 'VAR') | (symbol('OBJ') \= 'VAR') |,
   (symbol('LIB') \= 'VAR') THEN
DO
  SAY "One of the required symbols is not defined."
  SAY "CC is a " symbol('CC')
  SAY "CCOPTS is a " symbol('CCOPTS')
  SAY "AR is a " symbol('AR')
  SAY "OBJ is a " symbol('OBJ')
  SAY "LIB is a " symbol('LIB')
  EXIT 99
END

PARSE ARG SOURCE LIBRARY

/* Figure out the filename for the source file */
IF (right(SOURCE, 2) = '.c') | (right(SOURCE, 2 ) = '.C') THEN
DO
  sourcefile   = SOURCE
  basefilename = left(SOURCE, (length(SOURCE) - 2))
END
ELSE
DO
  sourcefile   = SOURCE'.c'
  basefilename = SOURCE
END

/* Join the strings together to get object filename */
objectfile   = basefilename''OBJ

IF (right(LIBRARY, length(LIB)) = LIB)
THEN libraryfile = LIBRARY
ELSE libraryfile = LIBRARY''LIB

CC CCOPTS '-c' sourcefile
IF (RC > 0) THEN exit 2
AR libraryfile objectfile
IF (RC > 0) then exit 3
'del' objectfile '>nul'
return

/* CompileToObj <source>
 *
 * Compile C source file into object file.  If this fails, abort the script.
 *
 * CC, CCOPTS, and OBJ are REXX variables defined at the top of the script.
 */

CompileToObj:

/* Check to make sure the symbols we require are defined */
/* NOTE: commas at end of line to continue onto next line */
IF (symbol('CC') \= 'VAR') | (symbol('CCOPTS') \= 'VAR') |,
   (symbol('OBJ') \= 'VAR') THEN
DO
  SAY "One of the required symbols is not defined."
  EXIT 99
END

/* Figure out the filename for the source file */
IF (right(ARG(1), 2) = '.c') | (right(ARG(1), 2 ) = '.C') THEN
DO
  sourcefile   = ARG(1)
  basefilename = left(ARG(1), (length(ARG(1)) - 2))
END
ELSE
DO
  sourcefile   = ARG(1)'.c'
  basefilename = ARG(1)
END

/* Join the strings together */
objectfile   = basefilename''OBJ

CC CCOPTS '-c' sourcefile
IF (RC > 0) THEN exit 2
return

/* ArchiveObj <object> <library>
 *
 * Archive the object specified into the library.
 *
 * AR, OBJ, and LIB are REXX variables defined at the top of the script.
 */

ArchiveObj:

/* Check to make sure the symbols we require are defined  */
/* NOTE: commas at end of line to continue onto next line */
IF (symbol('AR') \= 'VAR') | (symbol('OBJ') \= 'VAR') |,
   (symbol('LIB') \= 'VAR') THEN
DO
  SAY "One of the required symbols is not defined."
  SAY "AR is a " symbol('AR')
  SAY "OBJ is a " symbol('OBJ')
  SAY "LIB is a " symbol('LIB')
  EXIT 99
END

PARSE ARG object library

/* Figure out the filename for the source file */
IF (right(object, 2) = '.c') | (right(object, 2) = '.C') |,
   (right(object, 2) = '.o') | (right(object, 2) = '.O') THEN
  basefilename = left(object, (length(object) - 2))
ELSE
  basefilename = object

/* Join the strings together to get object filename */
objectfile   = basefilename''OBJ

IF (right(library, length(LIB)) = LIB)
THEN libraryfile = library
ELSE libraryfile = library''LIB

AR libraryfile objectfile
IF (RC > 0) then exit 3
return

/* LinkProgram <name>
 *
 * Link the object file name.OBJ to get name.EXE.  If the link fails, exit.
 *
 * CC, CCOPTS, OBJ and LIBRARIES are REXX variables defined at the top
 * of the script.
 */

LinkProgram:

/* Check to make sure the symbols we require are defined */
/* NOTE: commas at end of line to continue onto next one */
IF (symbol('CC') \= 'VAR') | (symbol('CCOPTS') \= 'VAR') |,
   (symbol('OBJ') \= 'VAR') | (symbol('LIBRARIES') \= 'VAR') THEN
DO
  SAY "One of the required symbols is not defined."
  EXIT 99
END

basefilename = ARG(1)
IF (right(basefilename, 4) = '.exe') | (right(object, 4) = '.EXE') THEN
  basefilename = left(basefilename, (length(basefilename) - 4))

/* Join the strings together */
/* If we are linking with -Zexe, then the output filename shouldn't have     */
/* ".exe" extension, because this will be added automatically.               */
objectfile   = basefilename''OBJ
IF (POS("-Zexe", CCOPTS) > 0) THEN
  executable = basefilename
ELSE
  executable = basefilename'.exe'

/* Figure out the libraries to link against */
locallibraries = '-L.'
'if exist library.xxx del library.xxx'
'for %%f in (*'LIB') do echo %%f >>library.xxx'
fullpath = stream('library.xxx', 'C', 'QUERY EXISTS')
IF (fullpath \= '') THEN
DO
  libfilename = 'xxx'
  DO WHILE libfilename \= ''
    libfilename = LINEIN('library.xxx')
    IF libfilename \= '' THEN
    DO
      libfilename = STRIP(libfilename)      /* CMD.EXE leaves trailing spaces*/
      IF (right(libfilename, length(LIB)) = LIB)
      THEN libraryfile = left(libfilename, (length(libfilename) - length(LIB)))
      ELSE libraryfile = libfilename

      locallibraries = locallibraries '-l'libraryfile
    END
  END
  result = stream('library.xxx','c','CLOSE')
  'if exist library.xxx del library.xxx >nul'
END

CC CCOPTS '-o' executable objectfile locallibraries LIBRARIES
IF (RC > 0) THEN exit 2
return

/* End of File */

