/*  ----------------------------------------------------------------<Prolog>-
    Name:       sfltree.c
    Title:      Linked-list functions
    Package:    Standard Function Library (SFL)

    Written:    97/11/18  Jonathan Schultz <jonathan@imatix.com>
    Revised:    98/02/03  Jonathan Schultz <jonathan@imatix.com>

    Copyright:  Copyright (c) 1991-98 iMatix
    License:    This is free software; you can redistribute it and/or modify
                it under the terms of the SFL License Agreement as provided
                in the file LICENSE.TXT.  This software is distributed in
                the hope that it will be useful, but without any warranty.
 ------------------------------------------------------------------</Prolog>-*/

#include "prelude.h"                    /*  Universal header file            */
#include "sfltree.h"                    /*  Prototypes for functions         */

/*  Constants                                                                */

TREE
    TREE_EMPTY = {TREE_NULL, TREE_NULL, NULL, BLACK};

/*  Internal function prototypes                                             */

static void insert_fixup (TREE **root, TREE *tree);
static void rotate_left  (TREE **root, TREE *tree);
static void rotate_right (TREE **root, TREE *tree);
static void delete_fixup (TREE **root, TREE *tree);


/*  ---------------------------------------------------------------------[<]-
    Function: tree_init

    Synopsis: Initialises an empty tree.
    ---------------------------------------------------------------------[>]-*/

void tree_init (TREE **root)
{
    *root = TREE_NULL;
}


/*  ---------------------------------------------------------------------[<]-
    Function: tree_insert

    Synopsis: Inserts a node into an existing tree.  Initialises node
    pointers and colour to correct values.  The data used by the compare
    functions must be filled in so that tree_insert can find the correct
    place in the tree to insert the node.
    ---------------------------------------------------------------------[>]-*/

int tree_insert (TREE **root, void *tree, TREE_COMPARE *comp,
                 Bool allow_duplicates)
{
    TREE
       *current,
       *parent;
    int
        last_comp = 0;

    /* find where node belongs */
    current = *root;
    parent  = NULL;
    while (current != TREE_NULL)
      {
        parent  = current;
        last_comp = (comp) (tree, current);
        switch (last_comp)
          {
            case -1: current = current-> left;  break;
            case  1: current = current-> right; break;
            default: if (allow_duplicates)
                         current = current-> left;
                     else
                         return TREE_DUPLICATE;

          }
      }

    /* setup new node */
    ((TREE *) tree)-> parent = parent;
    ((TREE *) tree)-> left   = TREE_NULL;
    ((TREE *) tree)-> right  = TREE_NULL;
    ((TREE *) tree)-> colour = RED;

    /* insert node in tree */
    if (parent)
        switch (last_comp)
          {
            case  1: parent-> right = tree;  break;
            default: parent-> left  = tree;
          }
    else
        *root = tree;

    insert_fixup (root, tree);
    return (TREE_OK);
}

/*  -------------------------------------------------------------------------
    Internal Function: insert_fixup

    Synopsis: Maintains the Red-Black tree balance after a node has been
    inserted.
    -------------------------------------------------------------------------*/

void insert_fixup (TREE **root, TREE *tree)
{
TREE *uncle;

    /* check red-black properties */
    while ((tree != *root)
       &&  (tree-> parent-> colour == RED))
      {
        /* we have a violation */
        if (tree-> parent == tree-> parent-> parent-> left)
          {
            uncle = tree-> parent-> parent-> right;
            if (uncle-> colour == RED)
              {
                /* uncle is RED */
                tree -> parent->          colour = BLACK;
                uncle->                   colour = BLACK;
                tree -> parent-> parent-> colour = RED;

                tree = tree-> parent-> parent;
              }
            else
              {
                /* uncle is BLACK */
                if (tree == tree-> parent-> right)
                  {
                    /* make tree a left child */
                    tree = tree-> parent;
                    rotate_left (root, tree);
                  }

                /* recolor and rotate */
                tree-> parent->          colour = BLACK;
                tree-> parent-> parent-> colour = RED;
                rotate_right (root, tree-> parent-> parent);
              }
          }
        else
          {
            /* mirror image of above code */
            uncle = tree-> parent-> parent-> left;
            if (uncle-> colour == RED)
              {
                /* uncle is RED */
                tree -> parent->          colour = BLACK;
                uncle->                   colour = BLACK;
                tree -> parent-> parent-> colour = RED;

                tree = tree-> parent-> parent;
              }
            else
              {
                /* uncle is BLACK */
                if (tree == tree-> parent-> left)
                  {
                    tree = tree-> parent;
                    rotate_right (root, tree);
                  }
                tree-> parent->          colour = BLACK;
                tree-> parent-> parent-> colour = RED;
                rotate_left (root, tree-> parent-> parent);
              }
          }
      }
    (*root)-> colour = BLACK;
}

/*  -------------------------------------------------------------------------
    Internal Function: rotate_left

    Synopsis: Rotates tree to left.
    -------------------------------------------------------------------------*/

void rotate_left (TREE **root, TREE *tree)
{
    TREE *other;

    other = tree-> right;

    /* establish tree-> right link */
    tree-> right = other-> left;
    if (other-> left != TREE_NULL)
        other-> left-> parent = tree;

    /* establish other-> parent link */
    if (other != TREE_NULL)
        other-> parent = tree-> parent;

    if (tree-> parent)
      {
        if (tree == tree-> parent-> left)
            tree-> parent-> left  = other;
        else
            tree-> parent-> right = other;
      }
    else
        *root = other;

    /* link tree and other */
    other-> left = tree;
    if (tree != TREE_NULL)
        tree-> parent = other;
}

/*  -------------------------------------------------------------------------
    Internal Function: rotate_right

    Synopsis: Rotates tree to right.
    -------------------------------------------------------------------------*/

void rotate_right (TREE **root, TREE *tree)
{
    TREE *other;

    other = tree-> left;

    /* establish tree-> left link */
    tree-> left = other-> right;
    if (other-> right != TREE_NULL)
        other-> right-> parent = tree;

    /* establish other-> parent link */
    if (other != TREE_NULL)
        other-> parent = tree-> parent;

    if (tree-> parent)
      {
        if (tree == tree-> parent-> right)
            tree-> parent-> right = other;
        else
            tree-> parent-> left  = other;
      }
    else
        *root = other;

    /* link tree and other */
    other-> right = tree;
    if (tree != TREE_NULL)
        tree-> parent = other;
}


/*  ---------------------------------------------------------------------[<]-
    Function: tree_delete

    Synopsis: Deletes a node from a tree.  Does not deallocate any memory.
    ---------------------------------------------------------------------[>]-*/

void tree_delete (TREE **root, void *tree)
{
    TREE
       *youngest, *elder;
    TREE_COLOUR
        colour;

    if ((!tree)
    ||  (tree == TREE_NULL))
        return;

    if ((((TREE *) tree)-> left  == TREE_NULL)
    ||  (((TREE *) tree)-> right == TREE_NULL))
        /* elder has a TREE_NULL node as a child */
        elder = tree;
    else
      {
        /* find tree successor with a TREE_NULL node as a child */
        elder = ((TREE *) tree)-> right;
        while (elder-> left != TREE_NULL)
            elder = elder-> left;
      }

    /* youngest is elder's only child */
    if (elder-> left != TREE_NULL)
        youngest = elder-> left;
    else
        youngest = elder-> right;

    /* remove elder from the parent chain */
    youngest-> parent = elder-> parent;
    if (elder-> parent)
        if (elder == elder-> parent-> left)
            elder-> parent-> left  = youngest;
        else
            elder-> parent-> right = youngest;
    else
        *root = youngest;

    colour = elder-> colour;

    if (elder != tree)
      {
/*
    JS 1997/11/18: This is from the original code.  I have changed it because
    our implementation knows nothing about the data, and should handle varying
    size nodes within a single tree, provided the part of the data used for
    comparison remains the same.  Plus just moving the data around will mess
    up any other pointers to it.

        tree-> Data = elder-> Data;
*/
        elder-> left   = ((TREE *) tree)-> left;
        elder-> right  = ((TREE *) tree)-> right;
        elder-> parent = ((TREE *) tree)-> parent;
        elder-> colour = ((TREE *) tree)-> colour;
        if (((TREE *) tree)-> parent)
            if (tree == ((TREE *) tree)-> parent-> left)
                ((TREE *) tree)-> parent-> left  = elder;
            else
                ((TREE *) tree)-> parent-> right = elder;
        else
            *root = elder;
      }

    if (colour == BLACK)
        delete_fixup (root, youngest);
}

/*  -------------------------------------------------------------------------
    Internal Function: delete_fixup

    Synopsis: Maintains Red-Black tree balance after deleting a node.
    -------------------------------------------------------------------------*/

void delete_fixup (TREE **root, TREE *tree)
{
    TREE
       *sibling;

    while (tree != *root && tree-> colour == BLACK)
      {
        if (tree == tree-> parent-> left)
          {
            sibling = tree-> parent-> right;
            if (sibling-> colour == RED)
              {
                sibling->       colour = BLACK;
                tree-> parent-> colour = RED;
                rotate_left (root, tree-> parent);
                sibling = tree-> parent-> right;
              }
            if ((sibling-> left->  colour == BLACK)
            &&  (sibling-> right-> colour == BLACK))
              {
                sibling-> colour = RED;
                tree = tree-> parent;
              }
            else
              {
                if (sibling-> right-> colour == BLACK)
                  {
                    sibling-> left-> colour = BLACK;
                    sibling->        colour = RED;
                    rotate_right (root, sibling);
                    sibling = tree-> parent-> right;
                  }
                sibling-> colour = tree-> parent-> colour;
                tree->    parent-> colour = BLACK;
                sibling-> right->  colour = BLACK;
                rotate_left (root, tree-> parent);
                tree = *root;
              }
          }
        else
          {
            sibling = tree-> parent-> left;
            if (sibling-> colour == RED)
              {
                sibling->       colour = BLACK;
                tree-> parent-> colour = RED;
                rotate_right (root, tree-> parent);
                sibling = tree-> parent-> left;
              }
            if ((sibling-> right-> colour == BLACK)
            &&  (sibling-> left->  colour == BLACK))
              {
                sibling-> colour = RED;
                tree = tree-> parent;
              }
            else
              {
                if (sibling-> left-> colour == BLACK)
                  {
                    sibling-> right-> colour = BLACK;
                    sibling->         colour = RED;
                    rotate_left (root, sibling);
                    sibling = tree-> parent-> left;
                  }
                sibling-> colour = tree-> parent-> colour;
                tree->    parent-> colour = BLACK;
                sibling-> left->   colour = BLACK;
                rotate_right (root, tree-> parent);
                tree = *root;
              }
          }
      }
    tree-> colour = BLACK;
}


/*  ---------------------------------------------------------------------[<]-
    Function: tree_find

    Synopsis: Finds a node with data matching that provided.
    ---------------------------------------------------------------------[>]-*/

void *tree_find (TREE **root, void *tree, TREE_COMPARE *comp)
{
    TREE
       *current = *root,
       *found;

    found = NULL;
    while (current != TREE_NULL)
        switch ((comp) (tree, current))
          {
            case -1: current = current-> left;  break;
            case  1: current = current-> right; break;
            default: found = current;           /*  In case of duplicates,   */
                     current = current-> left;  /*  get the first one.       */
          }

    return found;
}


/*  ---------------------------------------------------------------------[<]-
    Function: tree_traverse

    Synopsis: Traverse the tree, calling a processing function at each
    node.
    ---------------------------------------------------------------------[>]-*/

void tree_traverse (void *tree, TREE_PROCESS *process, int method)
{
    if ((!tree)
    ||  (tree == TREE_NULL))
        return;

    if (method == 1)
      {
        (process) (tree);
        tree_traverse (((TREE *) tree)-> left,  process, method);
        tree_traverse (((TREE *) tree)-> right, process, method);
      }
    else if (method == 2)
      {
        tree_traverse (((TREE *) tree)-> left,  process, method);
        tree_traverse (((TREE *) tree)-> right, process, method);
        (process) (tree);
      }
    else
      {
        tree_traverse (((TREE *) tree)-> left,  process, method);
        (process) (tree);
        tree_traverse (((TREE *) tree)-> right, process, method);
      }
}


/*  ---------------------------------------------------------------------[<]-
    Function: tree_first

    Synopsis: Finds and returns the first node in a (sub-)tree.
    ---------------------------------------------------------------------[>]-*/

void *tree_first (void *tree)
{
    TREE
       *current;

    if ((!tree)
    ||  (tree == TREE_NULL))
        return NULL;

    current = tree;
    while (current-> left != TREE_NULL)
        current = current-> left;

    return current;
}


/*  ---------------------------------------------------------------------[<]-
    Function: tree_last

    Synopsis: Finds and returns the last node in a (sub-)tree.
    ---------------------------------------------------------------------[>]-*/

void *tree_last (void *tree)
{
    TREE
       *current;

    if ((!tree)
    ||  (tree == TREE_NULL))
        return NULL;

    current = tree;
    while (current-> right != TREE_NULL)
        current = current-> right;

    return current;
}


/*  ---------------------------------------------------------------------[<]-
    Function: tree_next

    Synopsis: Finds and returns the next node in a tree.
    ---------------------------------------------------------------------[>]-*/

void *tree_next (void *tree)
{
    TREE
       *current,
       *child;

    if ((!tree)
    ||  (tree == TREE_NULL))
        return NULL;

    current = tree;
    if (current-> right != TREE_NULL)
        return tree_first (current-> right);
    else
      {
        current = tree;
        child   = TREE_NULL;
        while ((current-> parent)
           &&  (current-> right == child))
          {
            child = current;
            current = current-> parent;
          }
        if (current-> right != child)
            return current;
        else
            return NULL;
      }
}


/*  ---------------------------------------------------------------------[<]-
    Function: tree_prev

    Synopsis: Finds and returns the previous node in a tree.
    ---------------------------------------------------------------------[>]-*/

void *tree_prev (void *tree)
{
    TREE
       *current,
       *child;

    if ((!tree)
    ||  (tree == TREE_NULL))
        return NULL;

    current = tree;
    if (current-> left != TREE_NULL)
        return tree_last (current-> left);
    else
      {
        current = tree;
        child   = TREE_NULL;
        while ((current-> parent)
           &&  (current-> left == child))
          {
            child = current;
            current = current-> parent;
          }
        if (current-> left != child)
            return current;
        else
            return NULL;
      }
}
