/*  ----------------------------------------------------------------<Prolog>-
    Name:       sflproc.h
    Title:      Process control functions
    Package:    Standard Function Library (SFL)

    Written:    96/09/09  iMatix SFL project team <sfl@imatix.com>
    Revised:    98/01/30

    Synopsis:   Provides functions to create and manage processes.  The main
                set of functions lets you create, monitor, and end processes.
                A secondary function lets you run the current process as a
                background process.

    Copyright:  Copyright (c) 1991-98 iMatix
    License:    This is free software; you can redistribute it and/or modify
                it under the terms of the SFL License Agreement as provided
                in the file LICENSE.TXT.  This software is distributed in
                the hope that it will be useful, but without any warranty.
 ------------------------------------------------------------------</Prolog>-*/

#ifndef _SFLPROC_INCLUDED               /*  Allow multiple inclusions        */
#define _SFLPROC_INCLUDED


/*  Type definitions                                                         */

#if (defined (WIN32))
typedef struct {
    HANDLE process;
    HANDLE in;
    HANDLE out;
    HANDLE err;
    DESCR  *envd;                       /*  Environment data                 */
} PROC_HANDLE;
typedef PROC_HANDLE *PROCESS;           /*  Process ID type                  */
#define NULL_PROCESS NULL               /*    and null process               */

#elif (defined (__VMS__))
typedef struct {
    long id;
    long status;
} PROC_HANDLE;
typedef PROC_HANDLE *PROCESS;           /*  Process ID type                  */
#define NULL_PROCESS NULL               /*    and null process               */

#else
typedef qbyte        PROCESS;           /*  Process ID type                  */
#   define NULL_PROCESS 0               /*    and null process               */
#endif


/*  Macros:
 *  FILEHANDLE_MAX      Maximum possible number of open files
 */

/*  getdtablesize () is not available on all systems                         */
#if (defined (__UNIX__))
#   if (defined (__UTYPE_UNIXWARE))
#       define FILEHANDLE_MAX   sysconf (_SC_OPEN_MAX)
#   elif (defined (__UTYPE_HPUX))
#       define FILEHANDLE_MAX   FD_SETSIZE
#   elif (defined (__UTYPE_SINIX))
#       define FILEHANDLE_MAX   FD_SETSIZE
#   else
#       define FILEHANDLE_MAX   getdtablesize ()
#   endif
#elif (defined (FD_SETSIZE))
#   define FILEHANDLE_MAX       FD_SETSIZE
#else
#   define FILEHANDLE_MAX       32      /*  Arbitrary                        */
#endif


/*  Global variables                                                         */

extern int process_errno;               /*  Last process exit code           */
extern int process_delay;               /*  Wait for child to start          */
extern Bool process_compatible;         /*  Try to be compatible             */

#ifdef __cplusplus
extern "C" {
#endif

PROCESS process_create   (const char *file, char *argv [],
        const char *workdir, const char *std_in, const char *std_out,
        const char *std_err, char *envv [], Bool wait);
int     process_status   (PROCESS process_id);
int     process_kill     (PROCESS process_id);
void    process_close    (PROCESS process_id);
int     process_server   (const char *workdir, const char *lockfile);
Bool    process_alarm    (long delay);
char   *process_esc      (char *dest, const char *src);
char   *process_unesc    (char *dest, const char *src);
int     process_priority (int priority);

#ifdef __cplusplus
}
#endif

/*  Return values from process_status()                                      */

#define PROCESS_RUNNING         0
#define PROCESS_ENDED_OK        1
#define PROCESS_ENDED_ERROR     2
#define PROCESS_INTERRUPTED     3

/*  Values for process_priority()                                            */

#define PRIORITY_LOW            0
#define PRIORITY_NORMAL         1
#define PRIORITY_HIGH           2

#endif
