/*  ----------------------------------------------------------------<Prolog>-
    Name:       sflproc.c
    Title:      Process control functions
    Package:    Standard Function Library (SFL)

    Written:    96/09/09  iMatix SFL project team <sfl@imatix.com>
    Revised:    98/01/30

    Copyright:  Copyright (c) 1991-98 iMatix
    License:    This is free software; you can redistribute it and/or modify
                it under the terms of the SFL License Agreement as provided
                in the file LICENSE.TXT.  This software is distributed in
                the hope that it will be useful, but without any warranty.
 ------------------------------------------------------------------</Prolog>-*/

#include "prelude.h"                    /*  Universal header file            */
#include "sfltok.h"                     /*  Token-handling functions         */
#include "sfllist.h"                    /*  Linked-list functions            */
#include "sflmem.h"                     /*  Memory-handling functions        */
#include "sflstr.h"                     /*  String-handling functions        */
#include "sflfile.h"                    /*  File access functions            */
#include "sflnode.h"                    /*  Memory node functions            */
#include "sfldir.h"                     /*  Directory access functions       */
#include "sflproc.h"                    /*  Prototypes for functions         */
#include "sflcons.h"                    /*  Prototypes for functions         */
#include "sflsymb.h"                    /*  Symbol table handling            */
#include "sflenv.h"                     /*  Environment handling functions   */

/*  Global variables                                                         */

int  process_errno = 0;                 /*  Last process exit code           */
int  process_delay = 1000;              /*  Wait for child to start, msecs   */
Bool process_compatible = TRUE;         /*  Try to be compatible             */

#if (!defined (WIN32))
#define HANDLE int                      /*  Windows uses HANDLE for files    */
#endif


/*  Function prototypes                                                      */

#if (defined(__UNIX__))
static void   ignore_signal  (int sigvalue);
#endif
static HANDLE redirect_io    (const char *device, int fileno, int err_pipe,
                              Bool read_only);
#if (defined (__OS2__))
static void restore_redirection(HANDLE old_stdin, HANDLE old_stdout,
                                HANDLE old_stderr);
#endif
#if (defined (WIN32) || defined (__OS2__))
static char  *redirect_exec  (const char *filename);
#endif
#if (defined (__VMS__))
static void translate_to_vms (char *filename);
#endif


/*  ---------------------------------------------------------------------[<]-
    Function: process_create

    Synopsis: Creates a subprocess and returns a PROCESS identifying the new
    process.  Optionally directs standard input, output, and error streams
    to specified devices.  The caller can also specify environment symbols
    that the subprocess can access.  Accepts these arguments:
    <Table>
    filename    File to execute; if not fully specified, searches PATH.
    argv_[]     List of arguments; argv [0] is filename; ends in a NULL.
    workdir     Working directory; if NULL, remains in current directory.
    std_in      Device to use for standard input; NULL = no redirection.
    std_out     Device to use for standard output; NULL = no redirection.
    std_err     Device to use for standard error; NULL = no redirection.
    envs_[]     List of environment symbols to define, or NULL.
    </Table>
    If argv is NULL, parses the filename argument into words delimited by
    whitespace and builds the necessary argv table automatically.  Use this
    feature to execute a command with arguments, specified as one string.
    The envv list consists of strings in the form "name=value", ending in a
    NULL pointer.  If the envv argument is null, the environment of the
    current process is passed.  Otherwise the envv environment is used.
    If the process is started correctly, this function will sleep for
    'process_delay' milliseconds.  If the child command detects an error
    at startup, it may exit with an error status.  The sleep allows this
    error to be collected by calling process_status() after this call.  If
    process_delay is 0, any child error is ignored.
    Returns child process id, or 0 if there was an error.
    Under VMS, the filename must have been defined as a command before the
    calling process was started; the path is disregarded.
    Under OS/2 the filename can be the name of a CMD script, and this will
    be run with the interpreter specified in the first line (EXTPROC line,
    or "'/'*!" line; or failing that with the default command interpreter.

    Known bugs: when parsing filename argument into words, does not handle
    quotes in any special way; "this text" is 2 words, '"this' and 'text"'.
    You should have passed the filename through process_esc() before adding
    any optional arguments.
    ---------------------------------------------------------------------[>]-*/

PROCESS
process_create (
    const char *filename,               /*  Name of file to execute          */
    char *argv [],                      /*  Arguments for process, or NULL   */
    const char *workdir,                /*  Working directory, or NULL       */
    const char *std_in,                 /*  Stdin device, or NULL            */
    const char *std_out,                /*  Stdout device, or NULL           */
    const char *std_err,                /*  Stderr device, or NULL           */
    char *envv [],                      /*  Environment variables, or NULL   */
    Bool  wait                          /*  Wait for process to end          */
)
{
#if (defined (__UNIX__))
    /*************************************************************************
     **  UNIX  ***************************************************************
     *************************************************************************/

    pid_t
        fork_result;                    /*  Result from fork()               */
    int
        pipe_handle [2],                /*  Parent-to-child pipe             */
        pipe_readsize,                  /*  Amount of data read from pipe    */
        pipe_data;                      /*  Data read from pipe              */
    struct itimerval
        timeout;                        /*  Wait for response from child     */
    struct sigaction
        old_handler;                    /*  Old handler for SIGALRM          */
    const char
        *filename_only;                 /*  Filename, without arguments      */
    char
        *clean_filename,                /*  Unescaped filename               */
        *full_filename;                 /*  File to execute, with path       */

    /*  Create pipe for feedback from child to parent; quit if this fails    */
    if (pipe (pipe_handle))
        return (0);

    /*  Create subprocess - this returns 0 if we are the child, the pid if   */
    /*  we are the parent, or -1 if there was an error (not enough memory).  */
    fork_result = fork ();

    if (fork_result < 0)                /*  < 0 is an error                  */
      {
        close (pipe_handle [0]);        /*  Close the pipe                   */
        close (pipe_handle [1]);
        return (0);                     /*  Could not fork                   */
      }
    else
    if (fork_result > 0)                /*  > 0 is the parent process        */
      {
        /*  --- PARENT PROCESS HANDLING ------------------------------------ */
        /*  If the child process has a problem with the exec() call, it      */
        /*  sends us an errno value across the pipe.  If the exec() call     */
        /*  works okay, we get no feedback across the pipe.  We wait for a   */
        /*  small time (number of msecs specified by process_delay).  If     */
        /*  nothing comes across the pipe, we assume everything went okay.   */
        /*  The FD_CLOEXEC setting *should* cause the child pipe to close    */
        /*  after exec() but this does not seem to work; the read() still    */
        /*  blocks.  Bummer.                                                 */

        if (process_delay > 0)
          {
            timeout.it_interval.tv_sec  = 0;
            timeout.it_interval.tv_usec = 0;
            timeout.it_value.tv_sec     =  process_delay / 1000;
            timeout.it_value.tv_usec    = (process_delay % 1000) * 1000;

            /*  Save old signal handler to be polite to the calling program  */
            /*  then redirect the SIGALRM signal to our own (empty) handler  */
            sigaction (SIGALRM, NULL, &old_handler);
            signal    (SIGALRM, ignore_signal);
            setitimer (ITIMER_REAL, &timeout, 0);

            /*  Now read on the pipe until data arrives or the alarm goes    */
            pipe_readsize = read (pipe_handle [0], &pipe_data, sizeof (errno));

            /*  Restore old signal handler                                   */
            sigaction (SIGALRM, &old_handler, NULL);

          }
        else
            pipe_readsize = 0;

        close (pipe_handle [0]);        /*  Close the pipe                   */
        close (pipe_handle [1]);
        if (pipe_readsize == -1)
          {
            if (errno == EBADF || errno == EINTR)
              {
                /*  Normal - SIGALRM arrived or FD_CLOEXEC worked :)         */
                if (wait)
                    waitpid (fork_result, 0, 0);
                return ((PROCESS) fork_result);
              }
            else
                return (0);             /*  Error on read()                  */
          }
        else
        /*  We come here if process_delay was zero, or FD_CLOEXEC did its    */
        /*  job and the pipe was closed by the child process.                */
        if (pipe_readsize == 0)
          {
            if (wait)
                waitpid (fork_result, 0, 0);
            return ((PROCESS) fork_result);
          }
        else
          {
            /*  We read data from the pipe - this is an error feedback from  */
            /*  the child - i.e. file not found, or a permission problem.    */
            errno = pipe_data;          /*  Stuff the errno                  */
            return (0);
          }
      }
    /*  --- CHILD PROCESS HANDLING ----------------------------------------- */
    /*  Prepare the process environment and execute the file                 */

    /*  If argv[] array was not supplied, build it now from filename         */
    /*  And pull out the name of the file that we want to run.               */
    if (!argv)
      { /*  Split off the arguments, and pick out the filename to use        */
        argv = tok_split (filename);

        /*  The filename, and only the filename, is the 0th argument         */
        filename_only = argv[0];
      }
    else
      { /*  Already got our arguments, so the filename is just the filename  */
        filename_only = filename;
      }

    /*  If requested, close stdin, stdout, stderr, and redirect them         */
    redirect_io (std_in,  STDIN_FILENO,  pipe_handle [1], TRUE);
    redirect_io (std_out, STDOUT_FILENO, pipe_handle [1], FALSE);
    redirect_io (std_err, STDERR_FILENO, pipe_handle [1], FALSE);

    /*  Find file on path, make sure it is executable                        */
    /*  This is a good moment to unescape any spaces in the filename...      */

    clean_filename = process_unesc (NULL, filename_only);
    if (strchr (clean_filename, '/')     == NULL
    &&  strchr (clean_filename, PATHEND) == NULL)
        full_filename = file_where ('r', "PATH", clean_filename, NULL);
    else
        full_filename = file_where ('r',  NULL,  clean_filename, NULL);
    mem_free (clean_filename);

    if (full_filename == NULL)
      {
        errno = ENOENT;                 /*  No such file                     */
        write (pipe_handle [1], &errno, sizeof (errno));
        exit (EXIT_FAILURE);            /*  Kill the child process           */
      }
    if (!file_is_executable (full_filename))
      {
        errno = EACCES;                 /*  No permission to access file     */
        write (pipe_handle [1], &errno, sizeof (errno));
        exit (EXIT_FAILURE);            /*  Kill the child process           */
      }

    /*  Tell the system to close the pipe when we've done the exec()         */
    fcntl (pipe_handle [0], F_SETFD, FD_CLOEXEC);
    fcntl (pipe_handle [1], F_SETFD, FD_CLOEXEC);

    /*  If requested, change to working directory                            */
    if (workdir)
        chdir (workdir);

    /*  Execute the program - normally this call does not return, as it      */
    /*  replaces the current process image by the new one.  If we ever do    */
    /*  return, it is because there was an error.                            */
    if (envv)                           /*  If caller provided envv, use it  */
        execve (full_filename, argv, envv);
    else                                /*  Otherwise use current values     */
        execv  (full_filename, argv);

    write (pipe_handle [1], &errno, sizeof (errno));
    exit (EXIT_FAILURE);                /*  Kill the child process           */

#elif (defined (__OS2__))
    /*************************************************************************
     **  OS/2  ***************************************************************
     *************************************************************************/

    int
        process = 0;                    /*  Process number                   */
    HANDLE
        old_stdin = -1,                 /*  Dup'd handle for old stdin       */
        old_stdout = -1,                /*  Dup'd handle for old stdout      */
        old_stderr = -1;                /*  Dup'd handle for old stderr      */
    int parsedargs = 0,                 /*  argv() points at token array     */
        free_argv  = 0;                 /*  argv() points at handbuilt array */
    const char
        *filename_only = NULL,          /*  Filename, without arguments      */
        *actual_command = NULL;         /*  Actual command string to run     */
    char
        *clean_filename = NULL,         /*  Unescaped filename               */
        *full_filename = NULL,          /*  File to execute, with path       */
        *curdir = NULL,                 /*  Current working directory        */
        *strfree_this = NULL;           /*  strfree() this, if not NULL      */

    /*  NOTE: special care must be taken to ensure this code does not leak   */
    /*  memory, as the memory will be leaked in the main process which       */
    /*  potientally tries to run for a long period of time.  Token arrays    */
    /*  have a lot of potiental for leaks if care is not taken.  To avoid    */
    /*  these potiental problems strings are copied a little more than       */
    /*  otherwise would have been done, and then the original token arrays   */
    /*  are freed.                                                           */

    /*  If argv[] array was not supplied, build it now from filename         */
    /*  And pull out the name of the file that we want to run.               */
    if (!argv)
    {   /*  Split off the arguments, and pick out the filename to use        */
        argv = tok_split (filename);

        /*  The filename, and only the filename, is the 0th argument         */
        filename_only = argv[0];

        parsedargs = 1;                 /* Yes, we split off the arguments   */
    }
    else
    {   /*  Already got our arguments, so the filename is just the filename  */
        filename_only = filename;
    }

    /*  Under OS/2, we accept the magic file headers "#!", and "'/'*!".      */
    /*  We also special case running CMD scripts, so that we invoke the      */
    /*  default command interpreter, with a "/c" parameter, and the script   */
    /*  name.  The magic file headers are checked first so can be used to    */
    /*  override the default command interpreter.                            */

    actual_command = redirect_exec (filename_only);

    if (actual_command != NULL)
    {
        /*  At this point we have a string containing the name of the        */
        /*  program to run, followed by the arguments and the scriptname,    */
        /*  if it was a script that we were going to run.  So we tokenise the*/
        /*  string we got back and arrange for those bits to end up in the   */
        /*  arguments if required.                                           */
        char **newargs = NULL;
        int num_new = 0, num_existing = 0;
        int  free_newargs = 0;

        newargs = tok_split (actual_command);  /*  Split off the arguments   */

        actual_command = newargs[0];

        /*  Count the number of new arguments (should be at least 1)         */
        /*  And while we are here, eliminate any double quotes around the    */
        /*  arguments (especially the script name), since they'll only get   */
        /*  in the way later.                                                */
        for (num_new = 0; newargs[num_new] != NULL; num_new++)
            if (*newargs[num_new] == '"')
            {  char *pair = NULL;
               pair = strrchr(newargs[num_new], '"');
               if (pair != NULL)
               {  *pair = '\0';                /* Eliminate the last "       */
                  newargs[num_new]++;          /* Step over the first one    */
               }
            }

        ASSERT(num_new >= 1);

        /*  Count the number of existing arguments (from above), should be   */
        /*  at least 1.                                                      */
        for (num_existing = 0; argv[num_existing] != NULL; num_existing++)
            ;  /* EMPTY BODY */

        ASSERT(num_existing >= 1);

        /*  Handle .CMD script files where the redirection wasn't done above */
        if (num_new == 1)
        {   /*  Okay, it didn't expand there.  But possibly we have a CMD    */
            /*  script and need to invoke the command processor.             */
            char *extension = NULL;

            /*  Find file extension; if not found, set to NULL               */
            extension = strrchr (actual_command, '.');
            if (extension == NULL
            ||  strchr (extension, '/')         /*  Last '.' is part of path */
            ||  strchr (extension, '\\'))       /*  => filename has no ext   */
                extension = NULL;

            if (extension != NULL && (lexcmp(extension, ".CMD") == 0))
            {   /* This is a CMD script, and we need to invoke the command   */
                /* interpreter over it.                                      */
                char *command_processor = NULL;

                command_processor = strdupl (env_get_string ("COMSPEC", ""));

                if (*command_processor != '\0') /*  Not an empty string      */
                {   /*  Determine command processor arguments                */
                    char **cmdargs = NULL;
                    char **tmpargs = NULL;

                    cmdargs = tok_split (command_processor);

                    /*  Count the number of new arguments (at least 1)       */
                    for (num_new = 0; cmdargs[num_new] != NULL; num_new++)
                        ;    /* EMPTY BODY */

                    ASSERT(num_new >= 1);

                    /*  Now merge those arguments with script name           */
                    /*  Need: num_new + 1 for "/c", +1 for script name       */
                    /*        + 1 to terminate array                         */
                    tmpargs = mem_alloc((num_new+3) * sizeof(char *));

                    if (tmpargs != NULL)
                    {   /*  Okay, copy all the arguments into place          */
                        int i = 0;

                        for (i = 0; i < num_new; i++)
                           tmpargs[i] = strdupl (cmdargs[i]);

                        tmpargs[num_new++] = strdupl ("/c");
                        tmpargs[num_new++] = strdupl (actual_command);
                        tmpargs[num_new]   = NULL;

                        /*  Free the old arguments, and the old parse        */
                        tok_free(newargs);
                        tok_free(cmdargs);

                        /*  Now use that for our new arguments               */
                        newargs = tmpargs;
                        actual_command = newargs[0];
                        free_newargs = 1;       /*  Must free newargs later  */
                    }

                    /*  Free the command processor string                    */
                    strfree(&command_processor);
                }
            }   /* extension is .cmd                                         */
        }   /*  only one new argument (filename to run)                      */

        /*  Now collect all the arguments together into one array            */

        if (num_new >= 2 && num_existing >= 2)
        {   /*  Okay, we've got arguments to merge together, so we put the   */
            /*  new ones first followed by the old ones.                     */
            char **tmpargs;

            ASSERT(newargs != NULL);

            /*  Allocate space for the new arguments (at start), and the     */
            /*  existing arumgnents (at end), and a terminator.              */
            tmpargs = mem_alloc((num_new+num_existing+1) * sizeof(char *));
            if (tmpargs != NULL)
            {   /*  Okay, copy all the arguments into place                  */
                int i = 0;

                for (i = 0; i < num_new; i++)
                    tmpargs[i] = strdupl (newargs[i]);

                /*  NOTE: We skip the first argument here, since it is the  */
                /*  name of the script, and we've got one of those above.   */
                /*  BUT we've got to put next arg in next position, hence -1*/

                for (i = 1; i < num_existing; i++)
                    tmpargs[num_new + i - 1] = strdupl (argv[i]);

                /*  Terminate the array of arguments                        */
                tmpargs[num_new + num_existing - 1] = NULL;

                /*  Pick up a new pointer to the command to run             */
                actual_command = tmpargs[0];

                /*  Tidy up after ourselves                                  */
                if (free_newargs)
                  {
                    int j = 0;
                    for (j = 0; newargs[j] != NULL; j++)
                        strfree(&newargs[j]);

                    mem_free(newargs);
                  }
                else
                    tok_free (newargs);

                if (parsedargs)
                  {
                    tok_free(argv);
                    parsedargs = 0;
                  }

                /*  Change pointer to point at the new (combined) arguments  */
                argv = tmpargs;
                free_argv = 1;
            }
            else
            {   /*  We couldn't allocate the new memory required             */
                /*  Return failure.                                          */
                tok_free(newargs);
                if (parsedargs)
                   tok_free(argv);
                errno = ENOMEM;
                return ((PROCESS)0);
            }
        }
        else if (num_new >= 2 && num_existing <= 1)
        {   /*  There were no arguments before, there are now.  Use new ones */
            if (parsedargs)
            {  /* We parsed the arguments, free up some of the memory        */
               tok_free(argv);
               parsedargs = 0;
            }

            argv = newargs;
            if (free_newargs)               /* Make sure we free arguments   */
                free_argv = 1;
            else
                parsedargs = 1;
        }
        else /* (num_new <= 1) */   /* num_existing is 1 or more             */
        {   /*  No expansion of the string, we just use the existing args    */
            /*  But we do use the string as returned, because it may have    */
            /*  an extension on it.                                          */
            ASSERT(num_new <= 1);
            ASSERT(num_existing >= 1);

            /*  Copy the string as returned, so that we can use it below     */
            strfree_this = strdupl (actual_command);
            if (strfree_this != NULL)
            {
                actual_command = strfree_this;
                ASSERT(free_newargs == 0);
                tok_free(newargs);
            }
        }
    }   /*  Redirection found a filename to run */
    else
    {   /*  Redirection failed.  This means that it isn't executable, because*/
        /*  we should either have got a full name back, or a command string  */
        /*  to run.                                                          */

        if (parsedargs)
           tok_free(argv);
        errno = EACCES;                 /*  No permission to access file     */
        return (PROCESS)0;
    }

    /*  Find file on path, make sure it is executable                        */
    /*  This is a good moment to unescape any spaces in the filename...      */
    clean_filename = process_unesc (NULL, actual_command);
    if (strchr (clean_filename, '/')     == NULL
    &&  strchr (clean_filename, PATHEND) == NULL)
        full_filename = file_where ('r', "PATH", clean_filename, NULL);
    else
        full_filename = file_where ('r',  NULL,  clean_filename, NULL);
    mem_free (clean_filename);

    if (full_filename == NULL)
      {
        /*  Clear out the memory that we don't need any longer               */
        if (parsedargs)
            tok_free(argv);
        else if (free_argv)
          {
            int j = 0;
            for (j = 0; argv[j] != NULL; j++)
                strfree(&argv[j]);
            mem_free(argv);
          }
        if (strfree_this != NULL)
            strfree(&strfree_this);

        errno = ENOENT;                 /*  No such file                     */
        return (PROCESS)0;              /*  Failed to open                   */
      }
    if (!file_is_executable (full_filename))
      {
        /*  Clear out the memory that we don't need any longer               */
        if (parsedargs)
            tok_free(argv);
        else if (free_argv)
          {
            int j = 0;
            for (j = 0; argv[j] != NULL; j++)
                strfree(&argv[j]);
            mem_free(argv);
          }
        if (strfree_this != NULL)
           strfree(&strfree_this);

        errno = EACCES;                 /*  No permission to access file     */
        return (PROCESS)0;
      }

    /*  Redirect the IO streams, and save copies of the ones we redirect     */
    old_stdin  = redirect_io(std_in,  STDIN_FILENO,  0, TRUE);
    old_stdout = redirect_io(std_out, STDOUT_FILENO, 0, FALSE);
    old_stderr = redirect_io(std_err, STDERR_FILENO, 0, FALSE);

    if (old_stdin == -2 || old_stdout == -2 || old_stderr == -2)
    {   /* An error redirecting one of the file handles; restore them all    */
        /* and exit having failed our job.                                   */
        restore_redirection(old_stdin, old_stdout, old_stderr);

        /*  Clear out the memory that we don't need any longer               */
        if (parsedargs)
            tok_free(argv);
        else if (free_argv)
          {
            int j = 0;
            for (j = 0; argv[j] != NULL; j++)
                strfree(&argv[j]);
            mem_free(argv);
          }
        if (strfree_this != NULL)
           strfree(&strfree_this);

        return (PROCESS)0;
    }

    /*  If requested, change to working directory                            */
    if (workdir)
    {
        curdir = getcwd(NULL, 256);
        chdir (workdir);
    }
    else
        curdir = NULL;

    /*  Spawn the new program, and pick up its process ID.                   */
    if (envv)                           /*  If caller provided envv, use it  */
        process = spawnve (P_NOWAIT, full_filename, argv, envv);
    else                                /*  Otherwise use the current values */
        process = spawnv  (P_NOWAIT, full_filename, argv);

    /*  Put things back the way they were before                             */
    restore_redirection(old_stdin, old_stdout, old_stderr);
    if (curdir != NULL)                 /*  If directory changed, restore it */
    {
        chdir(curdir);
        free(curdir);
    }

    /*  Clear out the memory that we don't need any longer                   */
    if (parsedargs)
        tok_free(argv);
    else if (free_argv)
    {
        int j = 0;
        for (j = 0; argv[j] != NULL; j++)
            strfree(&argv[j]);

        mem_free(argv);
    }
    if (strfree_this != NULL)
        strfree(&strfree_this);

    if (process <= -1)
        return ((PROCESS)0);            /*  Error starting child process     */

    if (wait)
        waitpid (process, 0, 0);

    return ((PROCESS) process);

#elif (defined (WIN32))
    /*************************************************************************
     **  WINDOWS 32  *********************************************************
     *************************************************************************/

    PROCESS
        process;                        /*  Our created process handle       */
    STARTUPINFO
        newinfo = {0},                  /*  Specification for new process    */
        curinfo;                        /*  Specification of cur process     */
    PROCESS_INFORMATION
        procinfo;                       /*  Information about created proc   */
    DWORD
        dwCreateFlags = CREATE_NEW_CONSOLE;
    char
        *olddir,                        /*  Caller's working directory       */
        *fulldir,                       /*  Process' working directory       */
        *args,                          /*  Command arguments, if any        */
        *actual_command,                /*  Command, possibly qualified      */
        *buffer = NULL;                 /*  Working buffer                   */
    int
        argn;                           /*  Argument number                  */

    /*  Format full working directory, if specified                          */
    if (workdir)
      {
        olddir = get_curdir ();         /*  Just a lazy way to let the OS    */
        set_curdir (workdir);           /*  figure-out if the workdir is a   */
        fulldir = get_curdir ();        /*  relative or absolute directory.  */
        set_curdir (olddir);
        mem_free (olddir);
      }
    else
        fulldir = NULL;

    /*  Under Windows we accept the magic file header "#!".  If the          */
    /*  caller supplied an argument list, we attach this to the command.     */
    actual_command = redirect_exec (filename);
    strconvch (actual_command, '/', '\\');
    GetShortPathName (actual_command, actual_command,
                              strlen (actual_command) + 1);

    args = strchr (filename, ' ');      /*  Find arguments, if any           */
    if (argv)
      {                                 /*  Build full command buffer        */
        buffer = mem_alloc (tok_text_size ((char **) argv)
                                + strlen (actual_command) + 1);
        strcpy (buffer, actual_command);
        for (argn = 1; argv [argn]; argn++)
            xstrcat (buffer, " ", argv [argn], NULL);
            actual_command = buffer;
      }
    else
    if (args)
        buffer = xstrcpy (NULL, actual_command, args, NULL);
                actual_command = buffer;

    process = mem_alloc (sizeof (PROC_HANDLE));
    process-> process = NULL;
    process-> in  = redirect_io (std_in,  0, 0, TRUE);
    process-> out = redirect_io (std_out, 0, 0, FALSE);
    process-> err = redirect_io (std_err, 0, 0, FALSE);

    /*  Convert environment to a Windows-type packed block of strings        */
    /*  Use supplied environment, or parent environment if necessary.        */
    process-> envd = strt2descr (envv? (char **) envv: environ);

    GetStartupInfo (&curinfo);
    newinfo.cb          = sizeof (newinfo);
    newinfo.dwFlags     = STARTF_USESHOWWINDOW | STARTF_USESTDHANDLES;
    newinfo.wShowWindow = SW_HIDE;
    newinfo.hStdInput   = process-> in?  process-> in:  curinfo.hStdInput;
    newinfo.hStdOutput  = process-> out? process-> out: curinfo.hStdOutput;
    newinfo.hStdError   = process-> err? process-> err: curinfo.hStdError;
    newinfo.lpTitle     = NULL;

    /*  If necessary, run in separate VM, for 16-bit programs                */
    if (process_compatible)
        dwCreateFlags |= CREATE_SEPARATE_WOW_VDM;

    /*  CreateProcess returns errors sometimes, even when the process was    */
    /*  started correctly.  The cause is not evident.  For now: we detect    */
    /*  an error by checking the value of procinfo.hProcess after the call.  */
    procinfo.hProcess = NULL;
    CreateProcess (
        NULL,                           /*  Name of executable module        */
        actual_command,                 /*  Command line string              */
        NULL,                           /*  Process security attributes      */
        NULL,                           /*  Thread security attributes       */
        TRUE,                           /*  Handle inheritance flag          */
        dwCreateFlags,                  /*  Creation flags                   */
        process-> envd-> data,          /*  New environment block            */
        fulldir,                        /*  Current directory name           */
        &newinfo,                       /*  STARTUPINFO                      */
        &procinfo);                     /*  PROCESS_INFORMATION              */

    mem_strfree (&fulldir);
    mem_strfree (&buffer);              /*  Deallocate buffer, if used       */

    if (procinfo.hProcess == NULL)      /*  Error, we presume                */
      {
        process_close (process);
        return (NULL);
      }

    /*  Release our hold on the thread                                       */
    CloseHandle (procinfo.hThread);
    process-> process = procinfo.hProcess;

    /*  We do not need access to the files any longer in this process        */
    if (process-> in)
      {
        CloseHandle (process-> in);
        process-> in = NULL;
      }
    if (process-> out)
      {
        CloseHandle (process-> out);
        process-> out = NULL;
      }
    if (process-> err)
      {
        CloseHandle (process-> err);
        process-> err = NULL;
      }

    /*  Wait for the process to finish or be cancelled                       */
    if (wait)
      {
        WaitForSingleObject (procinfo.hProcess, INFINITE);
        process_close (process);
      }
    return (process);

#elif (defined (__VMS__))
    /*************************************************************************
     **  OPENVMS  ************************************************************
     *************************************************************************/

    PROCESS
        process;                        /*  Our created process handle       */
    char
        *curdir,                        /*  Current directory                */
        *clean_filename,                /*  Unescaped filename               */
        *full_filename = NULL,
        *full_std_in   = NULL,
        *full_std_out  = NULL;
    qbyte
        process_flags;                  /*  Process creation flags           */
    int
        argn,                           /*  Argument number                  */
        rc;                             /*  Return code from lib$spawn       */
    Bool
        rebuilt_argv = FALSE;           /*  Did we rebuild argv[]?           */

    VMS_STRING (command_dsc, "");       /*  Define string descriptors        */
    VMS_STRING (std_in_dsc,  "");
    VMS_STRING (std_out_dsc, "");

    /*  If argv[] array was not supplied, build it now from filename         */
    if (!argv)
      {
        argv = tok_split (filename);
        filename = argv [0];
        rebuilt_argv = TRUE;
      }
    /*  If filename contains a path or extension, disregard them             */
    clean_filename = strrchr (filename, '/');
    if (clean_filename)
        clean_filename++;
    else
        clean_filename = (char *) filename;
    if (strchr (clean_filename, '.'))
       *strchr (clean_filename, '.') = '\0';

    /*  Rebuild full command from filename and arguments                     */
    full_filename = mem_alloc (tok_text_size ((char **) argv)
                               + strlen (clean_filename) + 1);
    strcpy (full_filename, clean_filename);
    for (argn = 1; argv [argn]; argn++)
        xstrcat (full_filename, " ", argv [argn], NULL);

    /*  Free argument table if we allocated it dynamically here              */
    if (rebuilt_argv)
        tok_free (argv);

    command_dsc.value  = full_filename;
    command_dsc.length = strlen (full_filename);

    /*  Prepare full names for stdin and stdout                              */
    curdir = get_curdir ();
    if (std_in)
      {
        if (strchr (std_in, '/'))       /*  If already with path, use as is  */
            full_std_in = mem_strdup (std_in);
        else
            full_std_in = xstrcpy (NULL, curdir, "/", std_in, NULL);

        translate_to_vms  (full_std_in);
        std_in_dsc.value = full_std_in;
      }
    if (std_out)
      {
        if (strchr (std_out, '/'))      /*  If already with path, use as is  */
            full_std_out = mem_strdup (std_out);
        else
            full_std_out = xstrcpy (NULL, curdir, "/", std_out, NULL);

        translate_to_vms   (full_std_out);
        std_out_dsc.value = full_std_out;
      }
    std_in_dsc.length  = std_in?  strlen (std_in_dsc.value): 0;
    std_out_dsc.length = std_out? strlen (std_out_dsc.value): 0;

    /*  If requested, change to working directory                            */
    if (workdir)
        chdir (workdir);

    /*  Prepare process flags                                                */
    if (wait)
        process_flags = 0;
    else
        process_flags = 1;              /*  Bit 1 = don't wait for child     */

    process = mem_alloc (sizeof (PROC_HANDLE));
    process-> id     = 0;
    process-> status = 0;               /*  Completion status                */

/*  char *envv [],  */                  /*  Environment variables, or NULL   */

    rc = lib$spawn (
        &command_dsc,                   /*  Command to run                   */
        std_in?  &std_in_dsc: NULL,     /*  Stdin descriptor                 */
        std_out? &std_out_dsc: NULL,    /*  Stdout+stderr                    */
        &process_flags,                 /*  Options for new process          */
        &NULL,                          /*  Process name -- generated        */
        &process-> id,                  /*  Returned process ID              */
        &process-> status);

    if (workdir)                        /*  Switch back to original dir      */
        chdir (curdir);
    mem_free (curdir);

    mem_strfree (&full_filename);       /*  Deallocate various buffers,      */
    mem_strfree (&full_std_in);         /*    if they were used              */
    mem_strfree (&full_std_out);        /*                                   */

    /*  Return process ID.  If we waited for completion, the process id      */
    /*  is always NULL.                                                      */
    if (rc != 1)                        /*  Process failed with error        */
      {
        process_close (process);
        process = NULL;
      }
    else
    if (wait)                           /*  Finished with process            */
        process_close (process);

    return (process);

#else
    return ((PROCESS) 0);               /*  Not supported on this system     */
#endif
}


/*  --------------------------------------------------------------------------
 *  ignore_signal -- local
 *
 *  Handles a signal (we ignore it).
 */

#if (defined(__UNIX__))
local
ignore_signal (int sigvalue)
{
    ASSERT (sigvalue == SIGALRM);
}
#endif


/*  --------------------------------------------------------------------------
 *  redirect_io -- local
 *
 *  Redirects the specified file number to the specified device, if the
 *  device name is not null or empty.  If the redirection fails, leaves
 *  the original device unchanged, writes the errno to the error pipe and
 *  exits the current process.  Returns the file handle, or 0 if the
 *  device name was empty or NULL.  Existing files are not truncated.
 *
 *  Under OS/2: If the device name is not null or empty, this routine
 *  duplicates the existing file handle to another one, for safe keeping.
 *  It then opens the file requested, and duplicates that over the
 *  requested file handle.  It returns the file handle of the duplicated
 *  original, for use later to undo the redirection.  If errors are
 *  encountered -2 is returned. If no redirection is required -1 is returned.
 *
 */

static HANDLE
redirect_io (const char *device, int fileno, int err_pipe,
             Bool read_only)
{
    HANDLE
        file_handle = 0;                /*  Opened file handle               */

#if (defined (__OS2__))
    HANDLE
        dupe_file_handle = -1;          /*  Duplicate of old file handle     */
#endif

#if (!defined (WIN32))                  /*  See #define at top of file       */
    file_handle = -1;                   /*  No file handle open at present   */
#endif

    /*  If device name is not null, and not empty, redirect it               */
    if (device && *device)
      {
#if (defined (__UNIX__))
        /*  Open experimentally, to see if we have access to the device      */
        if (read_only)
            file_handle = open (device, O_RDONLY);
        else
            file_handle = open (device, O_WRONLY | O_CREAT, S_IREAD | S_IWRITE);
        if (file_handle == -1)
          {
            write (err_pipe, &errno, sizeof (errno));
            exit (EXIT_FAILURE);        /*  Kill the child process           */
          }
        else
          {
            /*  Okay, now close both devices and reallocate the original     */
            close (file_handle);
            close (fileno);
            if (read_only)
                file_handle = open (device, O_RDONLY);
            else
                file_handle = open (device, O_WRONLY | O_CREAT | O_APPEND,
                                            S_IREAD | S_IWRITE);

            /*  New file handle must be same as old one                      */
            ASSERT (file_handle == fileno);
          }
#elif (defined (__OS2__))
        /*  Open experimentally, to see if we have access to the device      */
        if (read_only)
            file_handle = open (device, O_RDONLY);
        else
            file_handle = open (device, O_WRONLY | O_CREAT, S_IREAD | S_IWRITE);
        if (file_handle <= -1)
          {
            return -2;          /*  Cannot acomplish redirection             */
          }
        else
          {
            /*  Okay, duplicate the existing one to a safe place, and then   */
            /*  duplicate the new file handle into place.  Close the old     */
            /*  copy of the new file handle.                                 */
            dupe_file_handle = dup(fileno);
            if (dupe_file_handle <= -1)
              {
                return -2;      /*  Cannot acomplish redirection             */
              }

            /*  Let dup2() close fileno (if open) if duplication suceeds     */
            if (dup2(file_handle, fileno) != fileno)
              {
                return -2;      /*  Cannot acomplish redirection             */
                close(dupe_file_handle);   /*  Close unneeded duplicate      */
              }

            if (file_handle != fileno)
              close(file_handle);          /*  Close unneeded duplicate      */

            /*  File handle to return is the duplicate of the old one        */
            file_handle = dupe_file_handle;
          }
#elif (defined (WIN32))
        SECURITY_ATTRIBUTES
            g_sa = { sizeof (SECURITY_ATTRIBUTES), NULL, TRUE };
        char
            *short_path;

        /*  Get the path without long file name                              */
        short_path = mem_strdup (device);
        if (short_path)
          {
            strconvch (short_path, '/', '\\');
            GetShortPathName (short_path, short_path, strlen (short_path) + 1);
          }
        if (read_only)
            file_handle = CreateFile (short_path, GENERIC_READ,
                                      FILE_SHARE_READ, &g_sa, OPEN_ALWAYS,
                                      FILE_ATTRIBUTE_NORMAL, NULL);
        else
          {
            /*  Open the file for appending, and move pointer to file end    */
            file_handle = CreateFile (short_path, GENERIC_READ | GENERIC_WRITE,
                                      FILE_SHARE_READ + FILE_SHARE_WRITE,
                                      &g_sa, OPEN_ALWAYS,
                                      FILE_ATTRIBUTE_NORMAL, NULL);
            SetFilePointer (file_handle, 0, NULL, FILE_END);
          }
        if (short_path != device)
            mem_strfree (&short_path);
#endif
      }
    return (file_handle);
}

#if (defined (__OS2__))
/*  --------------------------------------------------------------------------
 *  restore_redirection -- local
 *
 *  If the file handles for old_stdin, old_stdout, old_stderr, are greater
 *  than zero then duplicate those file handles over stdin, stdout, stderr
 *  respectively, and close the old file handles.  Each of the file handles
 *  is considered seperately.
 *
 *  This is primarily used to restore the file handles after IO redirection.
 */

static void
restore_redirection(HANDLE old_stdin, HANDLE old_stdout, HANDLE old_stderr)
{
    if (old_stdin >= 0)
    {
        dup2(old_stdin, STDIN_FILENO);
        if (old_stdin != STDIN_FILENO)
           close(old_stdin);
    }
    if (old_stdout >= 0)
    {
        dup2(old_stdout, STDOUT_FILENO);
        if (old_stdout != STDOUT_FILENO)
           close(old_stdout);
    }
    if (old_stderr >= 0)
    {
        close(STDERR_FILENO);
        dup2(old_stderr, STDERR_FILENO);
        if (old_stderr != STDERR_FILENO)
           close(old_stderr);
    }
}
#endif


#if (defined (WIN32) || defined (__OS2__))
/*  --------------------------------------------------------------------------
 *  redirect_exec -- local
 *
 *  If the specified file is an executable script (it starts with the bytes
 *  "#!"), then extracts the script interpreter name from the first line
 *  of the file, and prepares a new command consisting of the interpreter
 *  name followed by the command.  This emulates the UNIX action, e.g. to
 *  run a Perl script, the script starts with a line "#! /usr/bin/perl".
 *  If the combined command is too long it is truncated, brutally.  If the
 *  filename does not contain '/', searches the PATH for the file.
 *  Does not return any arguments following the command.  The characters
 *  '/''*''!' are treated as equivalent to '#!'.  If the interpreter name
 *  contains a path (e.g. '/usr/bin') but does not exist, the path is
 *  removed so that the executable can be located on the PATH.
 *
 *  Under OS/2 the EXTPROC line will be scanned for in the same way as the
 *  "#!" and "'/'*!" lines above.  The EXTPROC line should _not_ be used to
 *  invoke a command processor (but the "'/'*!" line can be), since some
 *  command processors (eg, 4OS/2, see page 92 of version 4 manual) will
 *  scan the EXTPROC line and invoke the program in it (ie, the command
 *  processor) again.  Repeat until the heat death of the universe, or
 *  running out of resources, whichever comes first.
 */

#define COMMAND_MAX         1024        /*  Program name with path           */

static char *
redirect_exec (const char *command)
{
    static char
        buffer [COMMAND_MAX + 1];       /*  Working buffer                   */
    char
        *first_space,                   /*  Cut buffer into filename + args  */
        *full_filename,                 /*  Filename with path               */
        *arguments = NULL,              /*  Program arguments if any         */
        input_char;                     /*  Character read from file stream  */
    int
        buffer_index;                   /*  Index into buffer                */
    FILE
        *stream;                        /*  File input stream                */
    Bool
        redirected;                     /*  Did we redirect the filename?    */

    /*  Copy command to buffer, but don't let it overflow                    */
    strncpy (buffer, command, COMMAND_MAX);
    buffer [COMMAND_MAX] = '\0';
    strconvch (buffer, '\\', '/');

    /*  Now truncate at first space to separate filename from arguments.     */
    /*  Caller should have used process_esc() somewhere to escape spaces...  */
    first_space = strchr (buffer, ' ');
    if (first_space)
        *first_space = '\0';

    process_unesc (buffer, buffer);     /*  Unescape any spaces in filename  */
    redirected = FALSE;

    /*  Now look for the file on the PATH, unless it already has a path      */
    if (strchr (buffer, '/') == NULL)
    {
        full_filename = file_where ('r', "PATH", buffer, NULL);
#if (defined (__OS2__))
        /*  Special case for .CMD scripts, so we don't have to specify the   */
        /*  whole name, and thus they run like they would in a shell.        */
        if (full_filename == NULL)
           full_filename = file_where ('r', "PATH", buffer, ".cmd");
#endif
    }
    else
    {
        full_filename = file_where ('r',  NULL,  buffer, NULL);
#if (defined (__OS2__))
        /*  Special case for .CMD scripts, so we don't have to specify the   */
        /*  whole name, and thus they run like they would in a shell.        */
        if (full_filename == NULL)
           full_filename = file_where ('r', NULL, buffer, ".cmd");
#endif
    }

    if (full_filename)
      {
        /*  Open the file, and check whether it matches the "#!" rule        */
        full_filename = mem_strdup (full_filename);
        stream = file_open (full_filename, 'r');
        if (stream)
          {
            input_char = fgetc (stream);
            if ((input_char == '#' && fgetc (stream) == '!')
            ||  (input_char == '/' && fgetc (stream) == '*'
                                   && fgetc (stream) == '!')
#if (defined (__OS2__))
            /*  Look for EXTPROC line, in both capitals and lower case       */
            /*  NOTE: If the EXTPROC line happens to specify a command       */
            /*  processor that understands EXTPROC lines then if it is       */
            /*  poorly written (eg 4OS/2) it may attempt to run itself       */
            /*  over the script repeatedly until running out of memory       */

            /*  NOTE: we work in a round about way because toupper may       */
            /*  be a macro, and we don't want to call fgetc() too much       */

            ||  (                                toupper(input_char) == 'E' &&
                 (input_char = fgetc(stream)) && toupper(input_char) == 'X' &&
                 (input_char = fgetc(stream)) && toupper(input_char) == 'T' &&
                 (input_char = fgetc(stream)) && toupper(input_char) == 'P' &&
                 (input_char = fgetc(stream)) && toupper(input_char) == 'R' &&
                 (input_char = fgetc(stream)) && toupper(input_char) == 'O' &&
                 (input_char = fgetc(stream)) && toupper(input_char) == 'C')
#endif
            )
              {
                /*  It matches, so skip 0 or more spaces and pick-up the     */
                /*  interpreter name                                         */
                do
                    input_char = fgetc (stream);
                until (input_char != ' ');

                /*  Read up to COMMAND_MAX chars; stop at end of line        */
                for (buffer_index = 0;
                     buffer_index < COMMAND_MAX;
                     buffer_index++)
                  {
                    buffer [buffer_index] = input_char;
                    input_char = fgetc (stream);
                    if (input_char == '\n'
                    ||  input_char == '\r')
                        break;
                  }
                /*  Terminate the buffer cleanly                             */
                redirected = (buffer_index > 0);
                buffer [++buffer_index] = '\0';

                /*  Strip trailing spaces and OS/2 end of comment            */
                strcrop (buffer);
                buffer_index = strlen (buffer);
                if (buffer [buffer_index - 1] == '/'
                &&  buffer [buffer_index - 2] == '*')
                  {
                    buffer [buffer_index - 2] = '\0';
                    strcrop (buffer);
                  }
                /*  Remove and save arguments after program name, if any     */
                first_space = strchr (buffer, ' ');
                if (first_space)
                  {
                    arguments = mem_strdup (first_space);
                    *first_space = '\0';
                  }
                /*  If buffer contains path, but can't be found, kill path   */
                if (!file_is_program (buffer))
                    strip_file_path (buffer);
              }
            file_close (stream);
          }
        /*  Prepare full command name in buffer                              */
        if (redirected)
          {                             /*  Put quotes around argument       */
            if (file_is_executable (buffer))
                strcpy (buffer, file_exec_name (buffer));
            if (arguments)
                strcat (buffer, arguments);
            strncat (buffer, " \"",         COMMAND_MAX - strlen (buffer));
            strncat (buffer, full_filename, COMMAND_MAX - strlen (buffer));
            strncat (buffer, "\"",          COMMAND_MAX - strlen (buffer));
          }
        else
            strcpy (buffer, full_filename);

        mem_strfree (&arguments);
        mem_strfree (&full_filename);
      }
    /*  If we did not find the file as specified, but the file is directly   */
    /*  executable, format and return a name with an .exe, .com, .cmd, or    */
    /*  .bat extension (as is appropriate).  Otherwise the filename is in    */
    /*  buffer.                                                              */
    if (!redirected && file_is_executable (buffer))
        return (file_exec_name (buffer));
    else
        return (buffer);
}
#endif


#if (defined (__VMS__))
/*  --------------------------------------------------------------------------
 *  translate_to_vms -- local
 *
 *  Translates POSIX style filename /top/path2/path2/filename into OpenVMS
 *  style filename top:[path1.path2]filename, which is always the same size.
 *  Does nothing if the filename is not valid, i.e. with at least a top,
 *  one path component, and a filename.
 */

static void
translate_to_vms (char *filename)
{
    char
        *path_start,
        *path_end;

    /*  Filename must start with '/'                                         */
    if (*filename != '/')
        return;

    /*  Find start and end of file path                                      */
    path_start = strchr  (filename + 1, '/');
    path_end   = strrchr (filename, '/');
    if (path_start == NULL || path_start == path_end)
        return;                         /*  Badly-formed filename            */

    path_start--;
    memmove (filename, filename + 1, path_start - filename);
    *path_start++ = ':';
    *path_start++ = '[';
    *path_end     = '\0';               /*  Cut string before filename       */
    strconvch (path_start, '/', '.');   /*    and replace slashes by dots    */
    *path_end     = ']';                /*  Finally, add ']' after path      */
}
#endif


/*  ---------------------------------------------------------------------[<]-
    Function: process_status

    Synopsis: Returns status of process specified by process ID.  Returns
    one of these values, or -1 if there was an error:
    <Table>
    PROCESS_RUNNING       Process is still running.
    PROCESS_ENDED_OK      Process ended normally.
    PROCESS_ENDED_ERROR   Process ended with an error status.
    PROCESS_INTERRUPTED   Process was interrupted (killed).
    </Table>
    In the case of PROCESS_ENDED_ERROR, the global variable process_errno is
    set to the exit code returned by the process.
    ---------------------------------------------------------------------[>]-*/

int
process_status (
    PROCESS process)
{
#if (defined (__UNIX__) || defined (__OS2__))
    int
        status;
    pid_t
        return_pid;

    /*  waitpid() returns 0 if the child process is still running, or the    */
    /*  process id if it stopped.  It can also return -1 in case of error.   */
    /*  No other return value is possible.                                   */

    return_pid = waitpid (process, &status, WNOHANG | WUNTRACED);
    if (return_pid == 0)
        return (PROCESS_RUNNING);
    else
    if (return_pid == process)
      {
        if (WIFEXITED (status))        /*  Program called exit()             */
          {
            process_errno = WEXITSTATUS (status);
            if (process_errno)         /*  Treat exit (0) as normal end      */
                return (PROCESS_ENDED_ERROR);
            else
                return (PROCESS_ENDED_OK);
          }
        else
        if (WIFSIGNALED (status))       /*  Process was interrupted          */
            return (PROCESS_INTERRUPTED);
        else
            return (PROCESS_ENDED_OK);
      }
    else
        return (-1);

#elif (defined (WIN32))
    DWORD
         status;

    ASSERT (process);
    status = WaitForSingleObject (process-> process, 0);

    if (status == WAIT_TIMEOUT)
        return (PROCESS_RUNNING);
    else
    if (status == WAIT_OBJECT_0)
        return (PROCESS_ENDED_OK);
    else
    if (status == WAIT_ABANDONED)
        return (PROCESS_ENDED_ERROR);
    else
        return (-1);

#elif (defined (__VMS__))
    ASSERT (process);
    if (process-> status == 0)
        return (PROCESS_RUNNING);
    else
        return (PROCESS_ENDED_OK);

#else
    return (-1);                        /*  Not supported on this system     */
#endif
}


/*  ---------------------------------------------------------------------[<]-
    Function: process_kill

    Synopsis: Ends a process specified by a process id.  The current process
    must have the appropriate authority to stop the specified process.
    Returns zero if the process was killed, -1 if there was an error.
    ---------------------------------------------------------------------[>]-*/

int
process_kill (
    PROCESS process)
{
#if (defined (__UNIX__) || defined (__OS2__))
    int count = 5;

    /*  First give it a chance to gracefully exit...                         */
    kill (process, SIGTERM);
    while (process_status (process) == PROCESS_RUNNING && count--)
        sleep (1);

    /*  Then get brutal if neccessary.                                       */
    if (process_status (process) == PROCESS_RUNNING)
      {
        kill (process, SIGKILL);
        while (process_status (process) == PROCESS_RUNNING)
            sleep (1);
      }
    return (0);

#elif (defined (WIN32))
    ASSERT (process);
    TerminateProcess (process-> process, 1);
    while (process_status (process) == PROCESS_RUNNING)
        Sleep (100);

    process_close (process);
    return (0);

#elif (defined (__VMS__))
    ASSERT (process);
    sys$delprc (process-> id);
    process_close (process);
    return (0);

#else
    return (-1);                        /*  Not supported on this system     */
#endif
}


/*  ---------------------------------------------------------------------[<]-
    Function: process_close

    Synopsis: You should call this function when a process has ended
    normally, if you did not specify the wait option when calling the
    process_create() function.  On some systems, each created process
    uses some memory.  process_close() guarantees that this memory is
    correctly freed.  Does nothing if the process handle is NULL.
    ---------------------------------------------------------------------[>]-*/

void
process_close (
    PROCESS process)
{
#if (defined (WIN32))
    if (process)
      {
        if (process-> process)
            CloseHandle (process-> process);
        if (process-> in)
            CloseHandle (process-> in);
        if (process-> out)
            CloseHandle (process-> out);
        if (process-> err)
            CloseHandle (process-> err);

        mem_free (process-> envd);
        mem_free (process);
      }

#elif (defined (__VMS__))
    mem_free (process);

#endif
}


/*  ---------------------------------------------------------------------[<]-
    Function: process_server

    Synopsis: Converts the process from an interactive foreground process
    into a background process.  The precise effect of this depends on the
    system.  On UNIX, does this:
    <LIST>
    Switches the process to run as a background job, under init;
    closes all open files;
    moves to a safe, known working directory, if required;
    sets the umask for new files;
    opens stdin, stdout, and stderr to the null device;
    enforces exclusive execution through a lock file, if required;
    logs the process id in the lock file;
    ignores the hangup unwanted signals.
    </LIST>
    On other systems, may do nothing.
    ---------------------------------------------------------------------[>]-*/

int
process_server (
    const char *workdir,                /*  Where server runs, or NULL/""    */
    const char *lockfile)               /*  For exclusive execution          */
{
#if (defined (__UNIX__))
    int
        fork_result,
        file_handle;
    char
        pid_buffer [10];
    struct flock
        lock_file;                      /*  flock() argument block           */

    /*  We recreate our process as a child of init.  The process continues   */
    /*  to exit in the background.  UNIX calls wait() for all children of    */
    /*  the init process, so the server will exit cleanly.                   */

    fork_result = fork ();
    if (fork_result < 0)                /*  < 0 is an error                  */
        return (-1);                    /*  Could not fork                   */
    else
    if (fork_result > 0)                /*  > 0 is the parent process        */
        exit (EXIT_SUCCESS);            /*  End parent process               */

    /*  We close all open file descriptors that may have been inherited      */
    /*  from the parent process.  This is to reduce the resources we use.    */

    for (file_handle = FILEHANDLE_MAX - 1; file_handle >= 0; file_handle--)
        close (file_handle);            /*  Ignore errors                    */

    /*  We move to a safe and known directory, which is supplied as an       */
    /*  argument to this function (or not, if workdir is NULL or empty).     */

    if (workdir && strused (workdir))
        chdir (workdir);

    /*  We set the umask so that new files are given mode 750 octal          */

    umask (027);                        /*  Complement of 0750               */

    /*  We set standard input and output to the null device so that any      */
    /*  functions that assume that these files are open can still work.      */

    file_handle = open ("/dev/null", O_RDWR);    /*  stdin = handle 0        */
    dup (file_handle);                           /*  stdout = handle 1       */
    dup (file_handle);                           /*  stderr = handle 2       */

    /*  We enforce a lock on the lockfile, if specified, so that only one    */
    /*  copy of the server can run at once.  We return -1 if the lock fails. */
    /*  This locking code might be better isolated into a separate package,  */
    /*  since it is not very portable between unices.                        */

    if (lockfile && strused (lockfile))
      {
        file_handle = open (lockfile, O_RDWR | O_CREAT, 0640);
        if (file_handle < 0)
            return (-1);                /*  We could not open lock file      */
        else
          {
            lock_file.l_type = F_WRLCK;
            if (fcntl (file_handle, F_SETLK, &lock_file))
                return (-1);            /*  We could not obtain a lock       */
          }
        /*  We record the server's process id in the lock file               */
        sprintf (pid_buffer, "%6d\n", getpid ());
        write   (file_handle, pid_buffer, strlen (pid_buffer));
      }

    /*  We ignore any hangup signal from the controlling TTY                 */
    signal (SIGHUP, SIG_IGN);

    return (0);                         /*  Initialisation completed ok      */
#else
    return (0);                         /*  Nothing to do on this system     */
#endif
}

#if (defined (__WINDOWS__))

/*  handle_timer -- internal
 *
 *  This function is called by Windows when the timer goes off.  We use this
 *  to send a SIGALRM to whoever is handling signals.  (SIGALRM is actually
 *  SIGFPE, since MSVC does not know SIGALRM, and its raise() function
 *  only works with the pathetic little group of standard signals.)
 *  We call WSACancelBlockingCall(), which *should* unblock any select() or
 *  other blocking winsock call that is in progress.  If you are waiting in
 *  a select() loop, and a timer goes off, you want to go handle it right
 *  away.  Sadly, this does not work with all (any?) winsocks.  So, a word
 *  to the wise: if you use select() and timers, under Windows you should
 *  use a select() timeout that matches the level of responsiveness that
 *  you need.  (Typically 100-500ms.)
 */
static UINT                             /*  We only want a single timer to   */
    last_timer = 0;                     /*    be active at once.             */
#   if (defined (WIN32))
void FAR PASCAL handle_timer (UINT idTimer, UINT msg,
                              DWORD dw1, DWORD dw2, DWORD dw3)
#   else
void CALLBACK handle_timer (HWND hwnd, UINT msg, UINT timerid, DWORD time)
#   endif
{
    WSACancelBlockingCall ();           /*  Force "interrupted system call"  */
    raise (SIGALRM);                    /*  Simulate UNIX signal             */
    last_timer = 0;                     /*    and note that timer has gone   */
}
#endif

/*  ---------------------------------------------------------------------[<]-
    Function: process_alarm

    Synopsis: Sets a system timer to raise a SIGALRM after a specified
    interval in milliseconds.  Returns TRUE if the timer could be created
    and FALSE if there were insufficient resources, or if the system does
    not support timers.  Permits a single alarm for the current process:
    any alarm that was still pending when you called this function is
    annulled.  The implementation is system-dependent and highly
    non-portable.

    Under UNIX we use the setitimer() system function, which is clean and
    simple.

    Under 16-bit Windows we use the SetTimer() call.  This does not work
    in 32-bit console applications.  Under 32-bit Windows we use the
    'multimedia' timer, which provides better resolution and does work
    in console applications.  In both these cases we cache the id of the
    last-created alarm (and kill it before each new request), to avoid
    multiple active alarms.  It is not a good idea to create too many
    concurrent timers; after 16 or so the alarms start to fail.  This is
    not supposed to happen with MM timers, but does anyway.  Under
    Windows, SIGALRM does not exist.  Since signal() only accepts one of
    a small set of fixed signals, we hijack the SIGFPE signal...  It's a
    compromise and requires that any code which expects a SIGALRM does
    not use SIGFPE.  This can be tweaked in prelude.h.

    Under OS/2 we use the alarm() function which is accurate to one
    second only.  The required accuracy of timing is not easily
    achieved, so process_alarm() rounds down to whole seconds (except if
    rounding down would give 0, in which case it will delay 1 second).
    This will probably cause problems in code applications that depends
    on sub-second timing resolution.

    Under OpenVMS 7 and later we use the setitimer() function as for
    UNIX.  Under OpenVMS 6 and earlier we use the alarm() function as
    for OS/2.  This code may be tuned to use native VMS system calls.
    ---------------------------------------------------------------------[>]-*/

Bool
process_alarm (long delay)
{
#if (defined (__UNIX__) || defined (__VMS_XOPEN))
    struct itimerval
        timeout;                        /*  Timeout for setitimer            */

    /*  If the system supports interval timers, ask for a signal             */
    timeout.it_interval.tv_sec  = 0;
    timeout.it_interval.tv_usec = 0;
    timeout.it_value.tv_sec     = delay / 1000;
    timeout.it_value.tv_usec    = delay % 1000 * 1000L;
    setitimer (ITIMER_REAL, &timeout, 0);
    return (TRUE);

#elif (defined (__OS2__) || defined (__VMS__))
    /*  Since we use alarm() for our timeout, we can only time to            */
    /*  the nearest second, and alarm(0) turns off the alarm.                */
    /*  NOTE: we also have only one timer -- if alarm() is called while      */
    /*  the alarm is active, then it will be reset to the new value, and     */
    /*  only a single SIGALRM will be generated.                             */
    delay = (delay < 1000) ? 1 : (delay / 1000);
    alarm (delay);
    return (TRUE);

#elif (defined (__WINDOWS__))
#   if (defined (WIN32))
#   pragma comment (lib, "winmm")       /*  Link-in multimedia library       */
    /*  The multimedia timer gives the best accuracy, and works in console   */
    /*  applications                                                         */
    int rc;
    if (last_timer)
        rc = timeKillEvent (last_timer);

    last_timer = timeSetEvent (delay, 50, handle_timer, 0, TIME_ONESHOT);
    return (TRUE);

#   else
    /*  But the normal Windows timer will do if we're in 16 bits             */
    if (last_timer)
        KillTimer ((HWND) NULL, last_timer);

    last_timer = SetTimer ((HWND) NULL, 0, (UINT) delay, handle_timer);
    return (TRUE);
#   endif

#else
    return (FALSE);                     /*  No timers - function failed      */
#endif
}


/*  ---------------------------------------------------------------------[<]-
    Function: process_esc

    Synopsis: Escapes a directory string so that process_create() can handle
    it correctly.  If you pass a command to process_create with a directory
    name that contains spaces, it will assume that the spaces delimit the
    command from its arguments.  For instance, under Windows 95, the filename
    "C:\Program Files\Myprog.exe" will be incorrectly treated as a program
    called "C:\Program" with arguments "Files\Myprog.exe".  This function
    replaces spaces by the escape character (0x1B).  You cannot use this
    value in a filename and expect process_create() to work correctly.  On
    an EBCDIC system, the escape character (0x27) is also used.  If the
    dest argument is NULL, allocates a string using mem_alloc() and returns
    that.  Otherwise copies into the dest string and returns that.  If the
    src string is NULL, returns an empty string.
    ---------------------------------------------------------------------[>]-*/

char *
process_esc (char *dest, const char *src)
{
#if (defined (__EBCDIC__))
#   define ESC_CHAR   0x27
#else
#   define ESC_CHAR   0x1B
#endif
    /*  Copy to dest, allocate if necessary                                  */
    if (dest != src)
        dest = xstrcpy (dest, src, NULL);
    strconvch (dest, ' ', ESC_CHAR);
    return (dest);
}


/*  ---------------------------------------------------------------------[<]-
    Function: process_unesc

    Synopsis: Does the reverse translaction to process_esc().
    ---------------------------------------------------------------------[>]-*/

char *
process_unesc (char *dest, const char *src)
{
    /*  Copy to dest, allocate if necessary                                  */
    if (dest != src)
        dest = xstrcpy (dest, src, NULL);
    strconvch (dest, ESC_CHAR, ' ');
    return (dest);
}


/*  ---------------------------------------------------------------------[<]-
    Function: process_priority

    Synopsis: Sets process priority as specified, to one of PRIORITY_LOW,
    PRIORITY_NORMAL, or PRIORITY_HIGH.  Currently has an effect only under
    Windows NT/95.  Returns 0 if okay, -1 if there was an error.
    ---------------------------------------------------------------------[>]-*/

int
process_priority (int priority)
{
#if (defined (WIN32))
    int
        class;

    if (priority == PRIORITY_HIGH)
        class = HIGH_PRIORITY_CLASS;
    else
    if (priority == PRIORITY_LOW)
        class = IDLE_PRIORITY_CLASS;
    else
        class = NORMAL_PRIORITY_CLASS;

    return (SetPriorityClass (GetCurrentProcess (), HIGH_PRIORITY_CLASS));
#else
    return (0);
#endif
}
